// itsywnd.cpp : implementation file
//

#include "stdafx.h"
#include <windowsx.h>
#include "itsywnd.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


// CAPTIONXY is the default size of the system menu icon.  This
// determines the height/width of the caption.
//
// The value that results from the following formula works out
// nicely for the veritcal caption on VGA, 8514 (Large Fonts),
// 8514 (Small Fonts), XGA (Small Fonts), XGA (Large Fonts),
// and TIGA (Small Fonts).  It may not be good on other displays.
//
// The problem is that TT fonts turn into bitmap fonts when they
// are sized below a certain threshold.  The idea is to make the
// size of the caption just big enough to get the smallest TT
// (scalable) font to fit.
//
#define CAPTIONXY (GetSystemMetrics( SM_CYCAPTION ) / 2 + 1)

#define TestWinStyle(dwStyleBit ) \
        (((DWORD)GetWindowLong( m_hWnd, GWL_STYLE ) & (DWORD)dwStyleBit) ? TRUE : FALSE )
#define HASCAPTION()    (TestWinStyle(IBS_VERTCAPTION ) ||\
                               TestWinStyle(IBS_HORZCAPTION ))
                               
#define SETCAPTIONSIZE(i)   (UINT)SetProp(m_hWnd,"ibSize",(HANDLE)i)
#define GETCAPTIONSIZE()     (UINT)GetProp(m_hWnd,"ibSize")
#define FREECAPTIONSIZE()    RemoveProp(m_hWnd,"ibSize")

#define SETMENUWASUPFLAG(i) (UINT)SetProp(m_hWnd,"ibFlag",(HANDLE)i)
#define GETMENUWASUPFLAG()   (UINT)GetProp(m_hWnd,"ibFlag")
#define FREEMENUWASUPFLAG()  RemoveProp(m_hWnd,"ibFlag")

#define DRAWFASTRECT(pDC,lprc) pDC->ExtTextOut(0,0,ETO_OPAQUE,lprc,NULL,0,NULL)

// The DrawArrow function takes the following to indicate what
// kind of arrow to draw.
//
#define ARROW_UP        0
#define ARROW_DOWN      1
#define ARROW_RESTORE   2


/////////////////////////////////////////////////////////////////////////////
// CItsyWnd

CItsyWnd::CItsyWnd()
{
}

CItsyWnd::~CItsyWnd()
{
}

BEGIN_MESSAGE_MAP(CItsyWnd, CWnd)
    //{{AFX_MSG_MAP(CItsyWnd)
    ON_WM_SYSCHAR()
    ON_WM_SYSKEYDOWN()
    ON_WM_SYSKEYUP()
    ON_WM_KEYDOWN()
    ON_WM_KEYUP()
    ON_WM_GETMINMAXINFO()
    ON_WM_NCCREATE()
    ON_WM_NCDESTROY()
    ON_WM_NCCALCSIZE()
    ON_WM_NCHITTEST()
    ON_WM_NCLBUTTONDBLCLK()
    ON_WM_NCLBUTTONDOWN()
    ON_WM_NCPAINT()
    ON_WM_NCACTIVATE()
    //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CItsyWnd message handlers

///////////////////////////////////////////////////////////////////////////
// Function:    Create
// Purpose:     create isty window
// Parameters:  see CWnd:Create, I'm not typing them all in...
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::Create(LPCSTR lpszClassName, LPCSTR lpszWindowName, DWORD dwStyle,           
                        int x, int y, int nWidth, int nHeight,HWND hwndParent, HMENU nIDorHMenu, 
                        LPSTR lpParam /* = NULL */)
{


 BOOL bRet = CreateEx(0,
                      lpszClassName,
                      lpszWindowName,
                      dwStyle,
                      x, y, nWidth, nHeight,
                      hwndParent,
                      nIDorHMenu, 
                      lpParam
                     );
  
  if(bRet)                    
    SetCaptionSize(GetCaptionSize()) ;                                    
    
 return bRet;   
}                                    
///////////////////////////////////////////////////////////////////////////
// Function:    GetCaptionSize
// Purpose:     gets caption size, height if horz, width if vert
// Parameters:  none
//
// Returns:     size of caption in pixels
///////////////////////////////////////////////////////////////////////////
UINT CItsyWnd::GetCaptionSize(void)
{
    return GETCAPTIONSIZE() + 1;
}

///////////////////////////////////////////////////////////////////////////
// Function:    SetCaptionSize
// Purpose:     sets caption size, height if horz, width if vert
// Parameters:  nSize   -   size of caption in pixels
//
// Returns:     non-zero if successfull, zero if failed
///////////////////////////////////////////////////////////////////////////
UINT CItsyWnd::SetCaptionSize(UINT nSize )
{
    UINT ui ;

    if (nSize <= 0)
        return 0 ;

    nSize-- ;
    ui = SETCAPTIONSIZE(nSize) ;

    // Once we change the window style, we need a WM_NCCALCRECT
    // to be generated.
    //
    // SWP_FRAMECHANGED is not documented in the 3.1 SDK docs,
    // but *is* in WINDOWS.H.
    //
    SetWindowPos(NULL, 0, 0, 0, 0, SWP_FRAMECHANGED | 
                        SWP_NOSIZE | SWP_NOMOVE | 
                        SWP_NOACTIVATE | SWP_NOZORDER) ;

    return ui ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    AdjustWindowRect
// Purpose:     Does the same thing as the USER function AdjustWindowRect(),
//              but knows about itsybitsy windows.  AdjustWindowRect() is
//              bogus for stuff like this.
//
// Parameters:  lprc    -    new window rect
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::AdjustWindowRect(LPRECT lprc )
{
    short    cx = 0, cy = 0 ;
    UINT     nCapSize ;
    
    nCapSize = GETCAPTIONSIZE() ;
    
    // First check Windows's styles, then our own.
    //
    if (TestWinStyle(WS_THICKFRAME ))
    {
        cx = GetSystemMetrics( SM_CXFRAME ) ;
        cy = GetSystemMetrics( SM_CYFRAME ) ;
    }
    else
        if (TestWinStyle(DS_MODALFRAME ))
        {
            cx = GetSystemMetrics( SM_CXDLGFRAME ) + GetSystemMetrics( SM_CXBORDER ) ;
            cy = GetSystemMetrics( SM_CYDLGFRAME ) + GetSystemMetrics( SM_CYBORDER ) ;
        }
        else
            if (TestWinStyle(WS_BORDER ))
            {
                cx = GetSystemMetrics( SM_CXBORDER ) ;
                cy = GetSystemMetrics( SM_CYBORDER ) ;
            }
    
    InflateRect( lprc, cx, cy ) ;
    
    if (TestWinStyle(IBS_VERTCAPTION ))
        lprc->left -= nCapSize ;
    else
        if (TestWinStyle(IBS_HORZCAPTION ))
            lprc->top -= nCapSize ;

}


///////////////////////////////////////////////////////////////////////////
// Function:    
// Purpose:
// This function is called when the user has pressed either the min or
// max button (i.e. WM_NCLBUTTONDOWN).  We go into a Peekmessage() loop,
// waiting for the mouse to come back up.  This allows us to make the
// button change up/down state like a real button does.
//
// lprc points to the rectangle that describes the button the
// user has clicked on.
     
// Parameters:  
//
// Returns:     
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::DepressMinMaxButton(UINT uiHT, LPRECT lprc) 
{
    BOOL    fDepressed = TRUE ;
    MSG     msg ;
                      
    // Draw button in down state
    DrawButton( NULL, uiHT == HTMINBUTTON, fDepressed ) ;
    SetCapture() ;
    
    while (TRUE)
    {
        if (PeekMessage((LPMSG)&msg, NULL, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE))
        {
            switch (msg.message)
            {
                case WM_LBUTTONUP:
                    if (fDepressed)
                        DrawButton( NULL, uiHT == HTMINBUTTON, !fDepressed ) ;
                    ReleaseCapture();
                return PtInRect( lprc, msg.pt ) ;
                
                case WM_MOUSEMOVE:
                    if (PtInRect( lprc, msg.pt ))
                    {
                        if (!fDepressed)
                            DrawButton( NULL,uiHT == HTMINBUTTON, fDepressed = TRUE ) ;
                    }
                    else
                    {
                        if (fDepressed)
                            DrawButton( NULL,uiHT == HTMINBUTTON, fDepressed = FALSE ) ;
                    }
                break;
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////
// Function:    DoMenu
// Purpose:     handles system menu
// Parameters:  none
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::DoMenu(void) 
{
    CDC    *pDC;
    RECT   rcIcon ;
    RECT   rc ;
    POINT  pt ;
    CMenu  *pMenu ;
    DWORD  dw ;

    if (!TestWinStyle(WS_SYSMENU))
        return FALSE ;
    
    if (pDC = GetWindowDC())
    {
        // Invert the icon
        //
        DrawSysMenu( pDC, TRUE ) ;
        
        // Pop up the menu
        //
        if (TestWinStyle(IBS_VERTCAPTION ))
        {
            pt.x = -1 ;
            pt.y = 0 ;
        }
        else
        {
            pt.x = 0 ;
            pt.y = -1 ;
        }
        
        GetIconRect(&rcIcon ) ;
        GetWindowRect(&rc ) ;
        ::OffsetRect(&rcIcon, -rc.left, -rc.top ) ;
        
        ClientToScreen(&pt) ;
        ClientToScreen((LPPOINT)&rc.right ) ;
        
        dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;
        SetWindowLong( m_hWnd, GWL_STYLE, dw | WS_SYSMENU ) ;
        
        pMenu = GetSystemMenu(FALSE ) ;
        SetupSystemMenu(pMenu) ;
        
        SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;

        pMenu->TrackPopupMenu(0, //TPM_LEFTALIGN,
                    pt.x,
                    pt.y,
                    this,
                    &rc ) ;
    
        DrawSysMenu( pDC, FALSE ) ;
        ReleaseDC(pDC ) ;
    }
    return TRUE ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    SetupSystemMenu
// Purpose:     enable appropriate entries in system menu based upon window
//              state
// Parameters:  pMenu - pointer to menu
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::SetupSystemMenu(CMenu *pMenu) 
{
    UINT     wMove ;        
    UINT     wSize ;
    UINT     wMinBox ;
    UINT     wMaxBox ;
    UINT     wRestore ;
    
    // Assume all should be grayed.
    //
    wSize = wMove = wMinBox = wMaxBox = wRestore = MF_GRAYED ;
    
    if (TestWinStyle(WS_MAXIMIZEBOX ) || IsIconic())
        wMaxBox = MF_ENABLED ;
    
    if (TestWinStyle(WS_MINIMIZEBOX ))
        wMinBox = MF_ENABLED ;
    
    if (IsZoomed())
        wRestore = MF_ENABLED ;
    
    if (TestWinStyle(WS_THICKFRAME ) &&
        !(IsIconic() || IsZoomed()))
        wSize = MF_ENABLED ;
    
    if (!IsZoomed() &&
        !IsIconic() &&
        TestWinStyle(WS_CAPTION ) )
        wMove = MF_ENABLED ;
    
    pMenu->EnableMenuItem(SC_MOVE,     wMove ) ;
    pMenu->EnableMenuItem(SC_SIZE,     wSize ) ;
    pMenu->EnableMenuItem(SC_MINIMIZE, wMinBox ) ;
    pMenu->EnableMenuItem(SC_MAXIMIZE, wMaxBox ) ;
    pMenu->EnableMenuItem(SC_RESTORE,  wRestore ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    GetCaptionRect
// Purpose:     gets bounding rect of caption
// Parameters:  lprc - pointer to rectangle object to fill
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::GetCaptionRect(LPRECT lprc ) 
{
    UINT     nCapSize ;
    
    nCapSize = GETCAPTIONSIZE() ;
    
    if (!HASCAPTION())
    {
        ::SetRectEmpty( lprc ) ;
        return FALSE ;
    }
    
    GetWindowRect(lprc ) ;
    
    // the window might have other non-client components like
    // borders 
    //
    if (TestWinStyle(WS_THICKFRAME ))
    {
        lprc->left += GetSystemMetrics( SM_CXFRAME ) ;
        lprc->top  += GetSystemMetrics( SM_CYFRAME ) ;
        lprc->right -= GetSystemMetrics( SM_CXFRAME ) ;
        lprc->bottom -= GetSystemMetrics( SM_CYFRAME ) ;
    }
    else
        if (TestWinStyle(DS_MODALFRAME ))  // if it's a dialog box
        {   
            lprc->left += GetSystemMetrics( SM_CXDLGFRAME ) + GetSystemMetrics( SM_CXBORDER ) ;
            lprc->top  += GetSystemMetrics( SM_CYDLGFRAME ) + GetSystemMetrics( SM_CYBORDER ) ;
            lprc->right -= GetSystemMetrics( SM_CXDLGFRAME ) + GetSystemMetrics( SM_CXBORDER ) ;
            lprc->bottom -= GetSystemMetrics( SM_CYDLGFRAME ) + GetSystemMetrics( SM_CYBORDER ) ;
        }
        else
            if (TestWinStyle(WS_BORDER ))
            {   
                lprc->left += GetSystemMetrics( SM_CXBORDER ) ;
                lprc->top  += GetSystemMetrics( SM_CYBORDER ) ;
                lprc->right -= GetSystemMetrics( SM_CXBORDER ) ;
                lprc->bottom -= GetSystemMetrics( SM_CYBORDER ) ;
            }
    
    if (TestWinStyle(IBS_VERTCAPTION ))
        lprc->right = lprc->left + nCapSize ;
    else
        lprc->bottom = lprc->top + nCapSize ;
    
    return TRUE ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    GetIconRect
// Purpose:     calcs rect of icon in screen coords
// Parameters:  lprc - rectangle to fill in
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::GetIconRect(LPRECT lprc ) 
{
    UINT    nCapSize ;
    BOOL    fMenu, fVert ;

    fMenu= TestWinStyle(WS_SYSMENU ) ;
    fVert = TestWinStyle(IBS_VERTCAPTION ) ;

    if (!GetCaptionRect(lprc ))   // window coords
      return FALSE ;

    if (!fMenu)
    {
        ::SetRectEmpty( lprc ) ;
        return FALSE ;
    }

    nCapSize = GETCAPTIONSIZE() ;

    if (fVert)
        lprc->bottom = lprc->top + nCapSize ;
    else
        lprc->right = lprc->left + nCapSize ;
    
    lprc->bottom-- ;
    lprc->right-- ;
    
    return TRUE ;

}

///////////////////////////////////////////////////////////////////////////
// Function:    GetButtonRect
// Purpose:     get rectangle of a button at nPos
// Parameters:  nPos - x position
//              lprc - rectangle to fill in
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::GetButtonRect(UINT nPos, LPRECT lprc ) 
{
    UINT    nSize = 0 ;

    if (!GetCaptionRect(lprc))   //window coords
      return FALSE ;

    nSize = GETCAPTIONSIZE() ;

    if (TestWinStyle(IBS_VERTCAPTION ))
    {
        lprc->bottom -= nSize * (nPos-1) ;
        lprc->top = lprc->bottom - nSize + 1 ;
    }
    else
    {
        lprc->right -= nSize * (nPos-1) ;
        lprc->left = lprc->right - nSize + 1 ;
    }

    return TRUE ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    GetMinButtonRect
// Purpose:     gets rectangle of minimize button
// Parameters:  lprc - rectangle to fill in
//
// Returns:     sucess or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::GetMinButtonRect(LPRECT lprc ) 
{
    if (!TestWinStyle(WS_MINIMIZEBOX )) 
    {
        ::SetRectEmpty( lprc ) ;
        return FALSE ;
    }

    // The minimize button can be in either position 1 or 2.  If there
    // is a maximize button, it's in position 2.
    //
    if (TestWinStyle(WS_MAXIMIZEBOX ))
        return GetButtonRect(2, lprc ) ;
    else
        return GetButtonRect(1, lprc ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    GetMaxButtonRect
// Purpose:     get rectangle of maximize button
// Parameters:  lprc - rectangle to fill in
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::GetMaxButtonRect(LPRECT lprc ) 
{
    //The maximize button can only be in position 1.
    //
    if (TestWinStyle(WS_MAXIMIZEBOX ))
        return GetButtonRect(1, lprc ) ;
    else
    {
        ::SetRectEmpty( lprc ) ;
        return FALSE ;
    }
}

///////////////////////////////////////////////////////////////////////////
// Function:    DrawCaption
// Purpose:     code to paint the small caption
// Parameters:  pDC - device context to draw on
//              lprc - caption rect
//              fVert - TRUE if vertical
//              fSysMenu - TRUE of system menu present
//              fMin    - TRUE if minimize button present
//              fMax    - TRUE if maximize button present
//              fActive - TRUE if window currently active
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::DrawCaption(CDC *pDC, LPRECT lprc,
                              BOOL fVert, BOOL fSysMenu, 
                              BOOL fMin, BOOL fMax, BOOL fActive ) 
{
    RECT        rc ;
    RECT        rcCap ;
    COLORREF    rgbCaptionBG ;
    COLORREF    rgbText ;
    COLORREF    rgbWindowFrame ;
    CBrush      brCaption ;
    UINT        ui ;
    UINT        nCapSize ;
    
    nCapSize = GETCAPTIONSIZE() ;
    
    // Get the colors.
    //
    rgbWindowFrame = GetSysColor( COLOR_WINDOWFRAME ) ;
    
    // if we have focus use the active caption color
    // otherwise use the inactive caption color
    //
    if (fActive)
    {
        rgbText = GetSysColor( COLOR_CAPTIONTEXT ) ;
        rgbCaptionBG = GetSysColor( COLOR_ACTIVECAPTION ) ;
    }
    else
    {
        if (fWin31)
            rgbText = GetSysColor( COLOR_INACTIVECAPTIONTEXT ) ;
        else
            rgbText = GetSysColor( COLOR_CAPTIONTEXT ) ;

        rgbCaptionBG = GetSysColor( COLOR_INACTIVECAPTION )  ;
    }

    pDC->SetBkMode(TRANSPARENT ) ;
    pDC->SelectObject(GetStockObject( NULL_BRUSH ) ) ;
    pDC->SelectObject(GetStockObject( NULL_PEN ) ) ;
    
    rcCap = *lprc ;
    
    if (fSysMenu)
    {
        if (fVert)
            rcCap.top += nCapSize ;
        else
            rcCap.left += nCapSize ;
    }
    
    if (fMax)
    {
        if (fVert)
            rcCap.bottom -= nCapSize ;
        else
            rcCap.right -= nCapSize ;
    }
    
    if (fMin)
    {
        if (fVert)
            rcCap.bottom -= nCapSize ;
        else
            rcCap.right -= nCapSize ;
    }
    
    if (fVert)
    {
        rc.left = lprc->right - 1 ;
        rc.right = lprc->right ;
        rc.top = lprc->top ;
        rc.bottom = lprc->bottom ;
    }
    else
    {
        rc.left = lprc->left ;
        rc.right = lprc->right ;
        rc.bottom = lprc->bottom ;
        rc.top = rc.bottom - 1 ;
    }
    
    pDC->SetBkColor(rgbWindowFrame ) ;
    DRAWFASTRECT( pDC, &rc ) ;

    brCaption.CreateSolidBrush( rgbCaptionBG ) ;
    CBrush *pOldBrush = pDC->SelectObject(&brCaption ) ;
    
    CPen *pOldPen = (CPen *)pDC->SelectStockObject(NULL_PEN) ;
    
    if (fVert)
        pDC->Rectangle(rcCap.left, rcCap.top, rcCap.right, rcCap.bottom + 1 ) ;
    else
        pDC->Rectangle(rcCap.left, rcCap.top, rcCap.right+1, rcCap.bottom ) ;
    
    pDC->SelectObject(pOldBrush) ;
    pDC->SelectObject(pOldPen) ;
    
    // Draw caption text here.  Only do it in 3.1 'cause 3.1 gives
    // us 'small fonts'.
    //
    ui = GetWindowTextLength() ;
    
    if (fWin31)
    {
        CFont          Font;
        LPSTR          lpsz ;
        LOGFONT        lf ;
        TEXTMETRIC     tm ;
        int            cx ;
        int            cy ;
        CSize          Size ;
        
        if (lpsz = (LPSTR)GlobalAllocPtr( GHND, ui + 2 ))
        {
            UINT    nBkMode ;

            GetWindowText(lpsz, ui + 1 ) ;
            nBkMode = pDC->SetBkMode(TRANSPARENT ) ;
            rgbText = pDC->SetTextColor(rgbText ) ;
            
            memset( &lf, '\0', sizeof(LOGFONT) ) ;

            lf.lfHeight = -(int)(nCapSize - 3) ;
            lf.lfCharSet = ANSI_CHARSET ;
            lf.lfQuality = DEFAULT_QUALITY ;
            lf.lfClipPrecision = CLIP_LH_ANGLES | CLIP_STROKE_PRECIS ;
            if (nCapSize >= 20)
            {
                lf.lfWeight = FW_BOLD ;
            }
            
            if (fVert)
            {
                // Can only rotate true type fonts (well, ok, we could
                // try and use "modern").
                lstrcpy( lf.lfFaceName, "Arial" ) ;
                lf.lfPitchAndFamily = FF_SWISS | 0x04;
                lf.lfEscapement = 900 ;

                // Note:  The Win 3.1 documentation for CreateFont() say's
                // that the lfOrientation member is ignored.  It appears,
                // that on Windows 16 3.1 this is true, but when running
                // as a 16 bit WinApp on WindowsNT 3.1 the lfOrientation
                // must be set or the text does not rotate!
                //
                lf.lfOrientation = 900 ;
                
                Font.CreateFontIndirect( &lf ) ;
                CFont *pOldFont = (CFont *)pDC->SelectObject(&Font) ;
                
                Size = pDC->GetTextExtent(lpsz, ui) ;
                
                cx = rcCap.bottom - ((rcCap.bottom - rcCap.top - Size.cx) / 2) ;
                cy = rcCap.left - 1 + ((rcCap.right - rcCap.left - Size.cy) / 2) ;

                // Make sure we got a rotatable font back.
                //
                pDC->GetTextMetrics(&tm ) ;
                if (tm.tmPitchAndFamily & TMPF_VECTOR     ||
                    tm.tmPitchAndFamily & TMPF_TRUETYPE)
                {
                    pDC->ExtTextOut(cy,
                                min( cx, rcCap.bottom),
                                ETO_CLIPPED, &rcCap,
                                lpsz, ui, NULL ) ;
                }
                
                pDC->SelectObject(pOldFont) ;                
            }
            else
            {
                // Use small fonts always for the horizontal. Cause it looks
                // more like "System" than Arial.
                //
                lf.lfPitchAndFamily = FF_SWISS ;
                
                Font.CreateFontIndirect( &lf ) ;
                CFont *pOldFont = (CFont *)pDC->SelectObject(&Font) ;
                
                Size = pDC->GetTextExtent(lpsz, ui) ;
                cx = rcCap.left + ((rcCap.right - rcCap.left - Size.cx) / 2) ;
                cy = rcCap.top + ((rcCap.bottom - rcCap.top - Size.cy) / 2) ;
                
                // Figger out how big the string is
                //
                pDC->ExtTextOut(
                            max( cx, rcCap.left ),
                            cy,
                            ETO_CLIPPED, &rcCap,
                            lpsz, ui, NULL ) ;
                
                pDC->SelectObject(&Font) ;
            }

            // Unsetup the DC
            //
            pDC->SetTextColor(rgbText ) ;
            pDC->SetBkMode(nBkMode ) ;
            
            GlobalFreePtr( lpsz ) ;
        }
    }
    
    if (fSysMenu)
        DrawSysMenu(pDC,FALSE ) ;
    
    if (fMin)
        DrawButton(pDC,TRUE, FALSE ) ;

    if (fMax)
        DrawButton(pDC,FALSE, FALSE ) ;
    
    return TRUE ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    DrawSysMenu
// Purpose:     draws system menu in caption bar
// Parameters:  pDC - device context to draw on
//              fInvert - TRUE if system menu is inverted ie. active
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::DrawSysMenu(CDC *pDC, BOOL fInvert ) 
{
    RECT         rcIcon ;
    RECT         rcTemp ;
    RECT         rc ;
    COLORREF     rgbIconFace ;
    COLORREF     rgbWindowFrame ;
    BOOL         fDC ;
    UINT         nCapSize ;
        
    nCapSize = GETCAPTIONSIZE() ;
    
    if (!pDC)
    {
        fDC = TRUE ;
        pDC = GetWindowDC() ;
    }
    else
        fDC = FALSE ;
    
    if (pDC)
    {
        rgbIconFace    = pDC->GetNearestColor(RGBLTGRAY ) ;
        rgbWindowFrame = GetSysColor( COLOR_WINDOWFRAME ) ;
        
        GetIconRect(&rcIcon ) ;
        GetWindowRect(&rc ) ;
           
        ::OffsetRect( &rcIcon, -rc.left, -rc.top ) ;
        
        rcTemp = rcIcon ;
        
        if (TestWinStyle(IBS_VERTCAPTION ))
        {
            rc = rcIcon ;      // separator line
            rc.top = ++rc.bottom - 1 ;
        }
        else
        {
            rc = rcIcon ;      // separator line
            rc.left = ++rc.right - 1 ;
        }
        
        // Fill
        pDC->SetBkColor(rgbIconFace ) ;
        DRAWFASTRECT(pDC,&rcTemp ) ;
        
        // Draw separator line
        pDC->SetBkColor(rgbWindowFrame ) ;
        DRAWFASTRECT(pDC, &rc ) ;
        
        if (nCapSize > 4)
        {
            // Draw the little horizontal doo-hickey
            //
            rcTemp.top = rcIcon.top + ((nCapSize-1) / 2) ;
            rcTemp.bottom = rcTemp.top + 3 ;
            rcTemp.left  = rcTemp.left + 3 ; 
            rcTemp.right = rcTemp.right - 1 ;
                
            pDC->SetBkColor(RGBGRAY ) ;
            DRAWFASTRECT( pDC, &rcTemp ) ;
                
            rc = rcTemp ;
            ::OffsetRect( &rc, -1, -1 ) ;
            pDC->SetBkColor(RGBBLACK ) ;
            DRAWFASTRECT( pDC, &rc ) ;
                
            InflateRect( &rc, -1, -1 ) ;
            pDC->SetBkColor(RGBWHITE ) ;
            DRAWFASTRECT( pDC, &rc ) ;
        }
        
        if (fInvert)
            pDC->InvertRect(&rcIcon ) ;
    
        if (fDC)
            ReleaseDC(pDC ) ;
    }
}

///////////////////////////////////////////////////////////////////////////
// Function:    DrawButton
// Purpose:     draws a button on the caption bar
// Parameters:  pDC - device context to draw on
//              fMin - TRUE if button is a minimize button
//              fDepressed - TRUE if button is depressed
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::DrawButton(CDC *pDC, BOOL fMin, BOOL fDepressed ) 
{
    RECT            rcButton ;
    RECT            rc ;
    COLORREF        rgbWindowFrame ;
    BOOL            fDC ;
    UINT            nCapSize ;
    UINT            nOffset ;
    int             n ;

    nCapSize = GETCAPTIONSIZE() ;

    // If you look at the standard Windows' min/max buttons, you will notice
    // that they have two pixels of 'shadow' to the bottom and right.  Since
    // our buttons can be really, really small, we only want one pixel of 
    // shadow when they are small.  I arbitrarily decided that if the 
    // caption size is greater than or equal to 20 we will use two
    // pixels.  That's what this THREASHOLD stuff does.
    //
    #define THRESHOLD   20
    nOffset = (nCapSize >= THRESHOLD) ? 2 : 1 ;    

    if (!pDC)
    {
        fDC = TRUE ;
        pDC = GetWindowDC() ;
    }
    else
        fDC = FALSE ;
    
    if (pDC)
    {
        rgbWindowFrame = GetSysColor( COLOR_WINDOWFRAME ) ;
        
        if (fMin)
            GetMinButtonRect(&rcButton ) ;
        else
            GetMaxButtonRect(&rcButton ) ;
        
        GetWindowRect(&rc ) ;
        ::OffsetRect( &rcButton, -rc.left, -rc.top ) ;
        
        rc = rcButton ;
        if (TestWinStyle(IBS_VERTCAPTION ))
        {
            rc = rcButton ;     //separator line
            rc.bottom = --rc.top + 1 ;
            rcButton.right-- ;
        }
        else
        {
            rc = rcButton ;     //separator line
            rc.right = --rc.left + 1 ;
            rcButton.bottom-- ;
        }
        
        //Draw separator line
        pDC->SetBkColor(rgbWindowFrame ) ;
        DRAWFASTRECT( pDC, &rc ) ;
        
        //Fill
        pDC->SetBkColor(RGBLTGRAY ) ;
        DRAWFASTRECT( pDC, &rcButton ) ;
        
        if (!fDepressed)
        {
            //The normal min/max buttons have one pixel on the top and left
            //sides for the highlight, and two pixels on the bottom and
            //right side for the shadow.
            //
            //When our caption is 'small' we only use one pixel on all
            //sides.
            //
            pDC->SetBkColor(RGBWHITE ) ;
            //Draw left side
            rc = rcButton ;
            rc.right = rc.left + 1 ;
            DRAWFASTRECT( pDC, &rc ) ;

            //Draw Top
            rc = rcButton ;
            rc.bottom = rc.top + 1 ;
            DRAWFASTRECT( pDC, &rc ) ;

            pDC->SetBkColor(RGBGRAY ) ;
            //Draw right side
            rc = rcButton ;
            rc.left = rc.right - 1 ;
            DRAWFASTRECT( pDC, &rc ) ;
            if (nCapSize > THRESHOLD)
            {
                rc.left-- ;
                rc.top++ ;
                DRAWFASTRECT( pDC, &rc ) ;
            }

            //Draw bottom
            rc = rcButton ;
            rc.top = rc.bottom - 1 ;
            DRAWFASTRECT( pDC, &rc ) ;
            if (nCapSize > THRESHOLD)
            {
                rc.top-- ;
                rc.left++ ;
                DRAWFASTRECT( pDC, &rc ) ;
            }

            rcButton.left++ ;
            rcButton.top++ ;
            rcButton.right -= nOffset ;
            rcButton.bottom -= nOffset ;
        }
        else
        {
            //Draw depressed state

            pDC->SetBkColor(RGBGRAY ) ;
            //Draw left side
            rc = rcButton ;
            rc.right = rc.left + nOffset ;
            DRAWFASTRECT( pDC, &rc ) ;

            //Draw Top
            rc = rcButton ;
            rc.bottom = rc.top + nOffset ;
            DRAWFASTRECT( pDC, &rc ) ;

            rcButton.left += 2 * nOffset ;
            rcButton.top += 2 * nOffset ;
        }

        // Now draw the arrows.  We do not want the
        // arrows to grow too large when we have a bigger than
        // normal caption, so we restrict their size.
        //
        // rcButton now represents where we can place our
        // arrows.  
        //
        // The maximum size of our arrows (i.e. the width of rcButton)
        // has been empirically determined to be SM_CYCAPTION / 2
        //
        n = ((GetSystemMetrics( SM_CYCAPTION )) / 2) - 
            (rcButton.right - rcButton.left) ;
        if (n < 1)
            InflateRect( &rcButton, n/2-1, n/2-1 ) ;

        if (fMin)
            DrawArrow( pDC, &rcButton, ARROW_DOWN ) ;
        else 
            if (IsZoomed())
            {
                DrawArrow( pDC, &rcButton, ARROW_RESTORE ) ;
            }
            else
                DrawArrow( pDC, &rcButton, ARROW_UP ) ;
        
        if (fDC)
            ReleaseDC(pDC ) ;
    }
}

///////////////////////////////////////////////////////////////////////////
// Function:    DrawArrow
// Purpose:     draw an arrow used by min/max/restore buttons
// Parameters:  pDC - device context to draw on
//              lprc - bounding rectangle
//              uiStyle - type of arrow
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::DrawArrow(CDC *pDC, LPRECT lprc, UINT uiStyle ) 
{
    int     row ;
    int     xTip ;
    int     yTip ;
    RECT    rc ;
    int     nMax = (lprc->bottom - lprc->top) >> 1 ;

    pDC->SetBkColor(RGBBLACK ) ;

    // We draw the arrow by drawing a series of horizontal lines
    //
    xTip = lprc->left + ((lprc->right - lprc->left+1) >> 1) ;
    switch (uiStyle)
    {
        case ARROW_UP:
            yTip = lprc->top + ((lprc->bottom - lprc->top-1) >> 2) ;
            for (row = 1 ; row <= nMax ; row++ )
            {
                rc.left = xTip - row ;
                rc.right = xTip + row - 1 ;
                rc.top = yTip + row ;
                rc.bottom = rc.top + 1 ;
                DRAWFASTRECT( pDC, &rc ) ;
            }
        break ;

        case ARROW_DOWN:
            yTip = lprc->bottom - ((lprc->bottom - lprc->top-1) >> 2) ;
            for ( row = nMax ; row > 0 ; row-- )
            {
                rc.left = xTip - row ;
                rc.right = xTip + row - 1 ;
                rc.top = yTip - row ;
                rc.bottom = rc.top + 1 ;
                DRAWFASTRECT( pDC, &rc ) ;
            }
        break ;

        case ARROW_RESTORE:
        default:
            yTip = lprc->top + ((lprc->bottom - lprc->top-1) >> 3) - 2;
            for (row = 1 ; row <= nMax ; row++ )
            {
                rc.left = xTip - row ;
                rc.right = xTip + row - 1 ;
                rc.top = yTip + row ;
                rc.bottom = rc.top + 1 ;
                DRAWFASTRECT( pDC, &rc ) ;
            }

            yTip += (nMax+1) * 2 ;
            for ( row = nMax ; row > 0 ; row-- )
            {
                rc.left = xTip - row ;
                rc.right = xTip + row - 1 ;
                rc.top = yTip - row ;
                rc.bottom = rc.top + 1 ;
                DRAWFASTRECT( pDC, &rc ) ;
            }
        break ;
    }

}


///////////////////////////////////////////////////////////////////////////
// Function:    OnSysChar
// Purpose:     responds to WM_SYSCHAR
// Parameters:  nChar - character
//              nRepCnt- repeat count
//              nFlags - state flags
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnSysChar(UINT nChar, UINT nRepCnt, UINT nFlags)
{
            // If ALT-SPACE 
            // was hit then pop up the menu
            // 
            if (HASCAPTION() && (nChar == VK_SPACE))
                DoMenu() ;

    DWORD dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;

    // Fool DefWindowProc into thinking we do not have
    // a system menu.  Otherwise it will try to
    // pop up its own.
    // 
    SetWindowLong( m_hWnd, GWL_STYLE, dw &~WS_SYSMENU ) ;
    
    CWnd::OnSysChar(nChar, nRepCnt, nFlags);
    SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnSysKeyDown
// Purpose:     responds to WM_SYSKEYDOWN
// Parameters:  nChar - character
//              nRepCnt- repeat count
//              nFlags - state flags
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnSysKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    
    
    DWORD dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;

    // Fool DefWindowProc into thinking we do not have
    // a system menu.  Otherwise it will try to
    // pop up its own.
    // 
    SetWindowLong( m_hWnd, GWL_STYLE, dw &~WS_SYSMENU ) ;
    
    CWnd::OnSysKeyDown(nChar, nRepCnt, nFlags);
    SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnSysKeyUp
// Purpose:     responds to WM_SYSKEYUP
// Parameters:  nChar - character
//              nRepCnt- repeat count
//              nFlags - state flags
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnSysKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    
    
    DWORD dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;

    // Fool DefWindowProc into thinking we do not have
    // a system menu.  Otherwise it will try to
    // pop up its own.
    // 
    SetWindowLong( m_hWnd, GWL_STYLE, dw &~WS_SYSMENU ) ;
    
    CWnd::OnSysKeyUp(nChar, nRepCnt, nFlags);
    SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnKeyDown
// Purpose:     responds to WM_KEYDOWN
// Parameters:  nChar - character
//              nRepCnt- repeat count
//              nFlags - state flags
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    
    
    DWORD dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;

    // Fool DefWindowProc into thinking we do not have
    // a system menu.  Otherwise it will try to
    // pop up its own.
    // 
    SetWindowLong( m_hWnd, GWL_STYLE, dw &~WS_SYSMENU ) ;
    
    CWnd::OnKeyDown(nChar, nRepCnt, nFlags);
    SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnKeyUp 
// Purpose:     responds to WM_KEYUP
// Parameters:  nChar - character
//              nRepCnt- repeat count
//              nFlags - state flags
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    
    DWORD dw = GetWindowLong( m_hWnd, GWL_STYLE ) ;

    // Fool DefWindowProc into thinking we do not have
    // a system menu.  Otherwise it will try to
    // pop up its own.
    // 
    SetWindowLong( m_hWnd, GWL_STYLE, dw &~WS_SYSMENU ) ;
    
    CWnd::OnKeyUp(nChar, nRepCnt, nFlags);
    
    SetWindowLong( m_hWnd, GWL_STYLE, dw ) ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnCommand
// Purpose:     reponds to WM_COMMAND
// Parameters:  std parameters
//
// Returns:     parent return value
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::OnCommand( WPARAM wParam, LPARAM lParam )
{
            // The menu that is poped up for the system menu with
            // TrackPopupMenu() sends it's notifications as WM_COMMAND
            // messages.  We need to translate these into
            // WM_SYSCOMMAND messages.  All standard WM_SYSCOMMAND
            // ids are greater than 0xF000.
            //
            // This could be a possible cause of confusion if the
            // itsybitsy window had children that used SC_MOVE or SC_CLOSE
            // as their IDs.  Take note and be careful.
            //
            // Also, because ibDefWindowProc looks at WM_COMMAND messages,
            // you will need to be careful to call ibDefWindowProc() for
            // any wm_command messages that you would normally ignore.
            // Otherwise the system menu won't work.
            //
            if (wParam >= 0xF000)
                // Call PostMessage() here instead of SendMessage!
                // Here's why:
                //      Our menu was created by TrackPopupMenu().  TPM() does
                //      not return until after the menu has been destroyed 
                //      (and thus the command associated with the menu selection
                //      sent).  Therefore when we get here, we are still
                //      *inside* TPM().  If we Send WM_SYSCOMMAND, SC_CLOSE
                //      to the window, the window will be destroyed before
                //      TPM() returns to our code within DoMenu() below.  Wel...
                //      We do stuff with the window handle after DoMenu()
                //      returns (namely GetProp()).  Since the window has
                //      been destroyed,this is bad.  
                PostMessage(WM_SYSCOMMAND, wParam, lParam ) ;


    return CWnd::OnCommand(wParam,lParam);
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnGetMinMaxInfo
// Purpose:     repsonds to WM_GETMINMAXINFO
// Parameters:  lpMMI - min/max info structure
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnGetMinMaxInfo(MINMAXINFO FAR* lpMMI)
{
    UINT  nCapSize ;
            nCapSize = GETCAPTIONSIZE() ;
            if (HASCAPTION() && TestWinStyle(WS_THICKFRAME ))
            {
                LPPOINT lppt = (LPPOINT)lpMMI ;
                RECT    rcMenu ;
                RECT    rcMin ;
                RECT    rcMax ;
                int     nX ;
                int     cx, cy ;    // window frame/border width

                if (TestWinStyle(WS_THICKFRAME ))
                {
                    cx = GetSystemMetrics( SM_CXFRAME ) ;
                    cy = GetSystemMetrics( SM_CYFRAME ) ;
                }
                else
                    if (TestWinStyle(WS_BORDER ))
                    {
                        cx = GetSystemMetrics( SM_CXBORDER ) ;
                        cy = GetSystemMetrics( SM_CYBORDER ) ;
                    }
                                
                GetIconRect(&rcMenu ) ;
                GetMinButtonRect(&rcMin ) ;
                GetMaxButtonRect(&rcMax ) ;
                nX = (rcMenu.right-rcMenu.left) + 
                    (rcMin.right-rcMin.left) +
                    (rcMin.right-rcMin.left) ;


                if (TestWinStyle(IBS_VERTCAPTION ) )
                {
                    lppt[3].x = nCapSize + cx * 2 - 1 ;
                    lppt[3].y = nX + (2* nCapSize) ;
                }
                else
                {
                    lppt[3].x = nX + (2* nCapSize) ;
                    lppt[3].y = nCapSize + cy * 2 - 1 ;
                }
            }
                
    CWnd::OnGetMinMaxInfo(lpMMI);
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcCreate
// Purpose:     responds to WM_NCCREATE
// Parameters:  lpCreateStruct - creation information
//
// Returns:     success or failure
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::OnNcCreate(LPCREATESTRUCT lpCreateStruct)
{
            DWORD dwStyle ;
            
            // We have two things that we need to store somewhere:
            //     1) The caption height (width).
            // and 2) A flag indicating whether the sysmenu was
            //        just up or not.
            // 
            // CAPTIONXY is a macro that calls GetSystemMetrics.
            //
            SETCAPTIONSIZE(CAPTIONXY ) ;

            // Set our flag that tells us whether the system menu was
            // 'just up'. 
            //
            SETMENUWASUPFLAG(FALSE ) ;

            // Are we in 3.1?  If so we have some neat features
            // we can use like rotated TrueType fonts.
            //
            fWin31 = (BOOL)(LOWORD( GetVersion() ) >= 0x030A) ;

            // If IBS_????CAPTION was specified and the WS_DLGFRAME (or
            // WS_DLGFRAME 'cause it == WS_CAPTION | WS_BORDER)
            // was specified the creator made a mistake.  Things get really
            // ugly if DefWindowProc sees WS_DLGFRAME, so we strip
            // the WS_DLGFRAME part off!
            //
            dwStyle = GetWindowLong( m_hWnd, GWL_STYLE ) ;
            if ((dwStyle & IBS_VERTCAPTION || dwStyle & IBS_HORZCAPTION) &&
                 dwStyle & WS_DLGFRAME)
            {
                dwStyle &= (DWORD)~WS_DLGFRAME ;
                SetWindowLong( m_hWnd, GWL_STYLE, dwStyle ) ;
            }

    if (!CWnd::OnNcCreate(lpCreateStruct))
        return FALSE;
    
    
    return TRUE;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcDestroy
// Purpose:     responds to WM_NCCDESTROY
// Parameters:  none
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnNcDestroy()
{
            // We store the caption size in a window prop. so we
            // must remove props.
            //
            FREECAPTIONSIZE() ;
            FREEMENUWASUPFLAG() ;

    CWnd::OnNcDestroy();    
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcCalcSize
// Purpose:     resonds to WM_NCCALCSIZE
// Parameters:  bCalcValidRects - Specifies whether the application should 
//                  specify which part of the client area contains valid 
//                  information. Windows will copy the valid information to the 
//                  specified area within the new client area. 
//                  If this parameter is TRUE, the application should specify 
//                  which part of the client area is valid.
//              lpncsp - contains info needed to calc area
//
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnNcCalcSize(BOOL bCalcValidRects, NCCALCSIZE_PARAMS FAR* lpncsp)
{
    UINT  nCapSize ;
            // This is sent when the window manager wants to find out
            // how big our client area is to be.  If we have a mini-caption
            // then we trap this message and calculate the cleint area rect,
            // which is the client area rect calculated by DefWindowProc()
            // minus the width/height of the mini-caption bar
            //
            CWnd::OnNcCalcSize(bCalcValidRects, lpncsp);
            
            if (!IsIconic() && HASCAPTION())
            {
                nCapSize = GETCAPTIONSIZE() ;

                if (TestWinStyle(IBS_VERTCAPTION ) )
                    ((LPRECT)lpncsp)->left += nCapSize ;
                else
                    ((LPRECT)lpncsp)->top += nCapSize ;
            }    
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcHitTest
// Purpose:     reponds to WM_NCHITTEST
// Parameters:  point - position of cursor in screen coords
//
// Returns:     hit code
///////////////////////////////////////////////////////////////////////////
UINT CItsyWnd::OnNcHitTest(CPoint point)
{
            // This message is sent whenever the mouse moves over us.
            // We will depend on DefWindowProc for everything unless
            // there is a mini-caption, in which case we will
            // return HTCAPTION or HTSYSMENU.  When the user clicks
            // or double clicks, NC_LBUTTON/ message are sent with
            // wParam equal to what we return here.
            // This means that this is an ideal place to figure out
            // where we are!
            //
            // let defwindowproc handle the standard borders etc...
            //
            UINT lRet = CWnd::OnNcHitTest(point);
            
            if (!IsIconic() && HASCAPTION() && lRet == HTNOWHERE)
            {
                RECT  rc ;
                RECT  rcMenu ;
                RECT  rcMinButton ;
                RECT  rcMaxButton ;

                UINT nCapSize = GETCAPTIONSIZE() ;

                // if DefWindowProc returned HTCAPTION then we have to
                // refine the area and return HTSYSMENU if appropriate
                //

                GetCaptionRect(&rc ) ;  // window coords
                if (PtInRect( &rc, point ))
                {
                     lRet = HTCAPTION ;

                     // rely on the fact that Get???Rect() return an invalid 
                     // (empty) rectangle if the menu/buttons don't exist
                     //
                     GetIconRect(&rcMenu ) ;
                     GetMinButtonRect(&rcMinButton ) ;
                     GetMaxButtonRect(&rcMaxButton ) ;
     
                     if (PtInRect( &rcMenu, point ))
                          lRet = HTSYSMENU ;

                     if (PtInRect( &rcMinButton, point ))
                          lRet = HTMINBUTTON ;
                     else
                          if (PtInRect( &rcMaxButton, point ))
                                lRet = HTMAXBUTTON ;
                }
            }
            if (lRet != HTSYSMENU)
                SETMENUWASUPFLAG(FALSE ) ;
        return lRet ;
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcLButtonDblClk
// Purpose:     responds to WM_NCLBUTTONDBLCLK
// Parameters:  nHitTest - code returned by WM_NCHITTEST
//              point    - position of cursor in screen coords
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnNcLButtonDblClk(UINT nHitTest, CPoint point)
{
            // Windows recieve WM_NC?BUTTONDBLCLK messages whether they
            // have CS_DBLCLKS or not.  We watch for one of these
            // to see if the user double clicked on the system menu (to
            // close the window) or on the caption (to maximize the window).
            //
            if (!IsIconic() && HASCAPTION() && nHitTest == HTSYSMENU)
                SendMessage(WM_CLOSE, 0, 0L ) ;
    
    CWnd::OnNcLButtonDblClk(nHitTest, point);
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcLButtonDown
// Purpose:     responds to WM_NCLBUTTONDOWN
// Parameters:  nHitTest - code returned by WM_NCHITTEST
//              point    - position of cursor in screen coords
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnNcLButtonDown(UINT nHitTest, CPoint point)
{
            RECT rc ;

            // If we're iconic or we don't have a caption then 
            // DefWindowProc will do the job just fine.
            //            
            if (IsIconic() || !HASCAPTION())
            {
             CWnd::OnNcLButtonDown(nHitTest, point);
             return;
            }
            // Here's were we handle the system menu, the min and max buttons.
            // If you wanted to change the behavior of the min/max buttons
            // do something like swap tool palettes or something, you 
            // would change the SendMessage() calls below.
            //
            switch (nHitTest)
            {
                case HTSYSMENU:
                    if (GETMENUWASUPFLAG() == FALSE && DoMenu())
                        SETMENUWASUPFLAG(TRUE ) ;
                    else
                        SETMENUWASUPFLAG(FALSE ) ;
                break ;
                
                case HTMINBUTTON:
                    GetMinButtonRect(&rc ) ;
                    // Note that DepressMinMaxButton() goes into
                    // a PeekMessage() loop waiting for the mouse
                    // to come back up.
                    //
                    if (DepressMinMaxButton(nHitTest, &rc ))
                        SendMessage(WM_SYSCOMMAND, SC_MINIMIZE, MAKELONG(point.x,point.y)) ;
                break ;
                
                case HTMAXBUTTON:
                    GetMaxButtonRect(&rc ) ;
                    // Note that DepressMinMaxButton() goes into
                    // a PeekMessage() loop waiting for the mouse
                    // to come back up.
                    //
                    if (DepressMinMaxButton(nHitTest, &rc ))
                    {
                        if (IsZoomed())
                            SendMessage(WM_SYSCOMMAND, SC_RESTORE, MAKELONG(point.x,point.y) ) ;
                        else
                            SendMessage(WM_SYSCOMMAND, SC_MAXIMIZE, MAKELONG(point.x,point.y) ) ;
                    }
                break ;
                
                default:
                    // Well, it appears as though the user clicked somewhere other
                    // than the buttons.  We let DefWindowProc do it's magic.
                    // This is where things like dragging and sizing the
                    // window happen.
                    //
                    CWnd::OnNcLButtonDown(nHitTest, point);
            }
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcPaint
// Purpose:     responds to WM_NCPAINT
// Parameters:  none
//
// Returns:     n/a
///////////////////////////////////////////////////////////////////////////
void CItsyWnd::OnNcPaint()
{
 if (IsIconic())
    return;

            if (HASCAPTION())
            {
                RECT  rcCap ;
                RECT  rc ;
                CDC   *pDC = GetWindowDC() ;
                BOOL  fActive ;

                GetCaptionRect(&rcCap ) ;    // Convert to window coords
                GetWindowRect(&rc ) ;
                OffsetRect( &rcCap, -rc.left, -rc.top ) ;

                
                fActive = (this == GetActiveWindow());

                DrawCaption( pDC, &rcCap,
                                        TestWinStyle( IBS_VERTCAPTION),
                                        TestWinStyle( WS_SYSMENU),
                                        TestWinStyle( WS_MINIMIZEBOX),
                                        TestWinStyle( WS_MAXIMIZEBOX),
                                        fActive ) ;

                ReleaseDC(pDC ) ;
            }
            CWnd::OnNcPaint();
}

///////////////////////////////////////////////////////////////////////////
// Function:    OnNcActivate
// Purpose:     responds to WM_NCACTIVATE
// Parameters:  bActive - TRUE if window active, FALSE if not
//
// Returns:     BOOL - whether or not to perform default processing
///////////////////////////////////////////////////////////////////////////
BOOL CItsyWnd::OnNcActivate(BOOL bActive)
{
  if (IsIconic())  
    return CWnd::OnNcActivate(bActive);
    
  BOOL lRet = CWnd::OnNcActivate(bActive);
  
            if (HASCAPTION())
            {
                RECT  rcCap ;
                RECT  rc ;
                CDC   *pDC = GetWindowDC() ;
                BOOL  fActive ;

                GetCaptionRect(&rcCap ) ;    // Convert to window coords
                GetWindowRect(&rc ) ;
                OffsetRect( &rcCap, -rc.left, -rc.top ) ;

                
                fActive = bActive ;

                DrawCaption( pDC, &rcCap,
                                        TestWinStyle( IBS_VERTCAPTION),
                                        TestWinStyle( WS_SYSMENU),
                                        TestWinStyle( WS_MINIMIZEBOX),
                                        TestWinStyle( WS_MAXIMIZEBOX),
                                        fActive ) ;

                ReleaseDC(pDC ) ;
            }
            return lRet;
}

