/* This program exports to a .DBF file from a SQL database, using ODBC 
version 1.0, with host language = Borland C 3.1 small model.

   This program is mentioned in chapter 5 of OPTIMIZING SQL, by Peter
   Gulutzan + Trudy Pelzer; R&D Publications 1994. */

/* We start with the usual list of C #include files. If you use a different C 
compiler, the names may differ. These #include files contain prototypes for 
the standard-library C functions: close(), exit(), free(), gets(), lseek(), 
malloc(), memset(), open(), printf(), strchr(), strcpy(), write(). */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include <dos.h>
#include <fcntl.h>
#include <mem.h>
#include <sys\stat.h>
#include <ctype.h>     

/* The main difference between the MS-DOS and Windows version of this program 
is that the Windows version contains '#include "windows.h"' as the next line, 
and does not contain the following five lines, since FALSE and TRUE and FAR 
and PASCAL are defined in "windows.h", and since the stack checking differs 
between the MS-DOS and Windows versions of the ODBC driver. */
#include "windows.h"     
/* #define FALSE 0 */
/* #define TRUE 1 */
/* #define FAR far */
/* #define PASCAL pascal */
/* extern unsigned _stklen = 30000; */	/* a big stack, Borland-specific */

/* THE INCLUDE FILE FOR THE ODBC CORE FUNCTIONS
SQL.H is the include file that comes with the Microsoft Kit, and generally 
the following line of code would be:  #include "sql.h".  The program on the 
disk that comes with this book will contain a different name here because it 
is not a Microsoft product. */

#include "ocelot.h"
     
/* Because __SQL is already defined in sql.h, all the lines between "##ifndef 
__SQL" and "#endif" are comments. But we want to expose all the sql.h 
definitions and equations we actually use in this program. So the following 
#define's and typedef's and function prototypes are all taken from sql.h. 
About 200 lines in all. */
     
#ifndef __SQL

#define __SQL

#define SQL_NTS			-3	/* In most ODBC functions, when
                       			strings are passed, a length parameter
                         		is passed too. This program uses a
                         		special value for length parameters:
                         		SQL_NTS "Null Terminated String", to
					indicate that strings end with \0. */
#define SQL_SQLSTATE_SIZE	5	/* The "state" of the last SQL
                                	command is how it came out: with
                                	an Error, a Warning, or OK. The
                                	specific state, SQLSTATE, is a
                                	5-byte alphanumeric string (\0 not
                                	included); the size required by ANSI.
                                	Microsoft defines 57 possible values;
                                	the driver maker can define more. */
#define SQL_MAX_MESSAGE_LENGTH	512    	/* Messages must have a fixed
                                        maximum size. 512 is a bit small but
                                        it's what comes with the package.*/
#define SQL_MAX_DSN_LENGTH	32	/* Maximum length of the "Data
                                        Source Name". Again a bit small (it
                                        would be nice if it equalled the
                                        maximum length of a path name), but
                                        it's what comes with the package. */

/* THE POSSIBLE RETURN CODES FROM CALLING SQL FUNCTIONS.
These are RETCODEs, which we'll define later with "typedef int". As in 
embedded SQL, errors are negative, warnings are positive, and OK is 0 -- but 
that' s all you find out from a RETCODE. The details about an error are in 
SQLSTATE. */
#define SQL_ERROR		-1	/* e.g SQL syntax is bad */
#define SQL_INVALID_HANDLE	-2	/* forgot SQLAllocEnv etc.? */
#define SQL_NO_DATA_FOUND	100	/* e.g. if a FETCH fails */
#define SQL_SUCCESS		0	/* "OK" */
#define SQL_SUCCESS_WITH_INFO	1	/* "OK but there's a warning" */

/* SQLFreeStmt has other options, SQL_DROP is the only one we use here. */
#define SQL_DROP	1

/* Following are all the "core" SQL data types (dates, timestamps and bits 
are "non-core" data types, and so are not supported by all ODBC drivers). We 
haven't considered the possibility that there might be a double-byte 
character set. The numbers are defined by ANSI. */

#define SQL_CHAR	1	/* ANSI definition: "fixed length string" */
#define SQL_NUMERIC	2	/* ANSI definition: "exact numeric, true
				precision = defined precision" */
#define SQL_DECIMAL	3	/* ANSI definition: "exact numeric, true
				precision >= defined precision" */
#define SQL_INTEGER	4	/* 32-bit signed integer */
#define SQL_SMALLINT	5	/* 16-bit signed integer */
#define SQL_FLOAT	6	/* ANSI definition: "approximate numeric,
				binary precision >= defined precision" */
#define SQL_REAL	7	/* ANSI definition: "approximate numeric,
				implementor defined precision" */
#define SQL_DOUBLE	8	/* ANSI definition: "approximate numeric,
				implementor defined precision > REAL" */
#define SQL_VARCHAR	12	/* ANSI definition: "varying length string */
     
/* When we describe how data can be fetched using the SQLBindCol() function, 
we can specify what C data type the ODBC driver should automatically convert 
to. In this program, we convert all SQL data types to C strings so the only 
type we have to know is SQL_C_CHAR, meaning "ASCII string ending with \0". 
Actually, there is no other kind of CHAR. */
     
#define SQL_C_CHAR    SQL_CHAR   /* CHAR, VARCHAR, DECIMAL, NUMERIC */
     
/* If we "fetch" a NULL value, then the length of the returned variable 
equals SQL_NULL_DATA. Compare the use of indicator variables in embedded SQL. 
*/
#define SQL_NULL_DATA      -1
     
/* Suppose we attempt to allocate an Environment Handle using the function 
henv=SQLAllocEnv(). If we fail, the handle's NULL innit? Frankly we'd just as 
soon check "if (henv==NULL)" but "if (henv==SQL_NULL_HENV)" must be what 
Microsoft recommends, since it's in sql.h. We'll go along with that. */
#define SQL_NULL_HENV      0     /* null Handle of ENVironment */
#define SQL_NULL_HDBC      0     /* null Handle of Database Connection */
#define SQL_NULL_HSTMT     0     /* null Handle of STateMenT */
     
/* Following are the "SQL portable types". Define your variables with these 
names, instead of the original C type, if the variable will be passed to/from 
ODBC, e.g. 'UWORD u;' instead of 'unsigned short int u;'. Portable typing's 
advantage is that when you switch from a 16-bit to a 32-bit C compiler the 
meaning of the word 'int' changes, but you can ensure that the meaning of 
SWORD remains the same by adjusting only one line here. */

/*      C TYPE                  PORTABLE TYPE   STANDS FOR
                                NAME */
typedef unsigned char		UCHAR;		/* "Unsigned CHAR" */
typedef signed char		SCHAR;		/* "Signed CHAR" */
typedef long int 		SDWORD;		/* "Signed Double Word" */
typedef short int		SWORD;		/* "Signed Word" */
typedef unsigned long int	UDWORD;		/* "Unsigned Double Word" */
typedef unsigned short int	UWORD;		/* "Unsigned Word" */
typedef double			SDOUBLE;	/* "Signed Double Real" */
typedef long double		LDOUBLE;	/* "Long Double Real" */
typedef float			SFLOAT;		/* "Signed Float" */

typedef void FAR *         PTR;
     
/* The environment, the database connection and the statement are three items 
that all working ODBC programs have handles to. */
typedef void FAR *         HENV;      /* handle of environment */
typedef void FAR *         HDBC;      /* handle of database connection */
typedef void FAR *         HSTMT;     /* handle of statement */
     
/* All ODBC functions are called the same way and return the same way:
int pascal far SQLAllocConnect (...) */

typedef int     RETCODE;

/* environment specific definitions */
     
#define SQL_API PASCAL FAR
     
/* THE PROTOTYPES OF THE ODBC CORE FUNCTIONS WE USE

In this program we use 15 ODBC functions, called "core" functions because 
they're fundamental and will be supported by all ODBC drivers. They fall into 
four quite separate groups, depending what phase we're in. In order of their 
appearance in the programme, the phases and their functions are:

Phase I -- initialization. The 4 functions are SQLAllocEnv(), 
SQLAllocConnect(), SQLConnect(), and SQLAllocStmt().

Phase II -- prepare, and check preparation results. The word "prepare"  is 
used to combine the two ordinary terms "parse and bind". The 3 functions are 
SQLPrepare(), SQLNumResultCols(), and  SQLDescribeCol().

Phase III-- execution, with ancillary functions telling the ODBC driver where 
to put things. The 3 functions are SQLBindCol(),  SQLExecute() and SQLFetch().

Phase IV -- reverse the initialization done in Phase I. The 4 functions are 
SQLFreeStmt(), SQLDisconnect(), SQLFreeConnect(), and  SQLFreeEnv().

The 15th function appears throughout: SQLError() gets error information. */
     
RETCODE SQL_API SQLAllocEnv(
HENV FAR *phenv);	 		/* point to environment handle */

RETCODE SQL_API SQLAllocConnect(
HENV henv,				/* environment handle */
HDBC FAR *phdbc);			/* point to connection handle */

RETCODE SQL_API SQLConnect(
HDBC hdbc,				/* connection handle */
UCHAR FAR *szDSN,			/* pointer to data source name */
SWORD cbDSN,      			/* length of data source name */
UCHAR FAR *szUID,	         	/* pointer to user identifier */
SWORD cbUID,				/* length of user identifier */
UCHAR FAR  *szAuthStr,			/* "authentication" e.g. password */
SWORD cbAuthStr);			/* length of authentication */

RETCODE SQL_API SQLAllocStmt(
HDBC hdbc,				/* connection handle */
HSTMT FAR *phstmt);			/* point to statement handle */

RETCODE SQL_API SQLPrepare(
HSTMT hstmt,				/* statement handle */
UCHAR FAR *szSqlStr,			/* SQL text string */
SDWORD cbSqlStr);			/* length of SQL text string */

RETCODE SQL_API SQLNumResultCols(
HSTMT hstmt,				/* statement handle */
SWORD FAR *pccol);			/* number of columns in result set*/

RETCODE SQL_API SQLDescribeCol(
HSTMT hstmt,				/* statement handle */
UWORD icol,				/* column number, base 1 */
UCHAR FAR *szColName,			/* column name */
SWORD cbColNameMax,			/* max length of column name */
SWORD FAR *pcbColName,			/* returned length of column name*/
SWORD FAR *pfSqlType,			/* SQL data type, e.g. SQL_CHAR*/
UDWORD FAR *pcbColDef,			/* precision */
SWORD FAR *pibScale,			/* scale */
SWORD FAR *pfNullable);			/*e.g. SQL_NULLABLE nulls allowed*/

RETCODE SQL_API SQLBindCol(
HSTMT hstmt,				/* statement handle */
UWORD icol,				/* column number, base 1 */
SWORD fCType,				/* C data type, e.g. SQL_C_CHAR*/
PTR rgbValue,				/* point to data */
SDWORD cbValueMax,			/* max length of rgbValue */
SDWORD FAR *pcbValue);			/* point to length of data */

RETCODE SQL_API SQLExecute(
HSTMT hstmt);				/* statement handle */

RETCODE SQL_API SQLFetch(
HSTMT hstmt);				/* statement handle */

RETCODE SQL_API SQLFreeStmt(
HSTMT hstmt,				/* statement handle */
UWORD fOption);				/* e.g. SQL_DROP */

RETCODE SQL_API SQLDisconnect(
HDBC hdbc);				/* connection handle */

RETCODE SQL_API SQLFreeConnect(
HDBC hdbc);				/* connection handle */

RETCODE SQL_API SQLFreeEnv(
HENV henv);				/* environment handle */

RETCODE SQL_API SQLError(
HENV henv,				/* environment handle */
HDBC hdbc,				/* connection handle */
HSTMT hstmt,				/* statement handle */
UCHAR FAR *szSqlState,			/* state ret'd by last SQL command*/
SDWORD FAR *pfNativeError,		/* error code ret'd by data source */
UCHAR FAR *szErrorMsg,			/* error message text */
SWORD cbErrorMsgMax,			/* max length of error message */
SWORD FAR *pcbErrorMsg);		/* ret'd length of error message */

#endif	/* #ifndef __SQL */

/* End of definitions taken from "sql.h" */
     
/* Prototypes for all three routines in the body of this program */
     
void main ();
int export (UCHAR *database, UCHAR *sql_statement, char *dbf_filename);
int errcheck (UCHAR *msg,RETCODE rc,HENV henv,HDBC hdbc,HSTMT hstmt);
     
/* Field names in .DBF files are 11 bytes long, SQL column names are 
generally 18 characters long. */
     
#define DBFNAMESIZE 11
#define SQLNAMESIZE 18
     
/* THE .DBF HEADER

The main type of dBASE-compatible file, called a .DBF file because the normal 
extension is "*.DBF", has a reasonably well-established form:
-- First comes the "file header" (dbf_header).
-- Then come the "field headers" (field_header), one per field.
-- Then comes the actual data, in fixed-field format.
The following 'struct header' defines the file header and field headers 
together.

Explanation of struct dbf_header: dbf_header_version has various flags 
marking the type of file (we'll skip the complications and say that a value 
of 3 means "dBASE-III-like with no ; dbf_header_update is a year-month-day 
field with the last time the file was changed (we'll put in a fixed value 
here); dbf_header_recordcount is the number of records in the file; 
dbf_header_size is the length of the header plus the length of all the field 
headers; dbf_header_record_size is the length of each record; 
dbf_header_reserve and dbf_header_pending etc. don't really matter for our 
purposes.

Explanation of struct field_header: fieldheader_name contains the name of the 
field with trailing \0s and the last (eleventh) byte must be \0; 
dbf_fieldheader_type can be 'C' for Character, 'N' for Number as an ASCII 
number including digits or . or - (other possibilities are 'D' date, 'F' 
float, 'L' Logical, and 'M' Memo but this program only produces 'C' or 'N'); 
fieldheader_address will be used for storing the address but we write a fake 
value in the file; dbf_fieldheader_size is the displayable size; and 
dbf_fieldheader_decimal is the number of positions after the decimal point, 
the rest doesn't matter here (we use some reserve fields as scratch space, 
though). */
     
     struct dbf_header {
       char dbf_header_version;
       char dbf_header_last_update[3];
       unsigned long dbf_header_recordcount;
       unsigned dbf_header_size;
       unsigned dbf_header_record_size;
       char dbf_header_reserve1[2];
       char dbf_header_pending;
       char dbf_header_reserve2[13];
       char dbf_header_mdx;
       char dbf_header_reserve3[3];
       struct field_header {
         char dbf_fieldheader_name[DBFNAMESIZE];
         char dbf_fieldheader_type;
         long dbf_fieldheader_address;
         unsigned char dbf_fieldheader_size;
         unsigned char dbf_fieldheader_decimal;
         unsigned char dbf_fieldheader_reserve1[2];
         unsigned char dbf_fieldheader_flag;
         unsigned char dbf_fieldheader_reserve2[10];
         char dbf_fieldheader_mdx; }
         ff[1]; };

/* THE main() FUNCTION
main() gets information typed in by the user, calls export(), returns. 
Although we use 'printf()' and 'gets()', all the major C compilers have 
libraries which allow the running of such programs under Windows without 
modification. */

void main ()
{
char sql_statement[25];
char database[SQL_MAX_DSN_LENGTH+1];	/*i.e. 32 according to sql.h*/
char dbf_filename[25];			/* maximum path  size? */
     
  printf("Type in the name of the database you are importing from\n");
  gets(database);
     
  printf("Type in a SQL 'SELECT' statement for the rows you want exported\n");
  gets(sql_statement);
     
  printf("Type in the name of the .DBF file you are exporting to\n");
  gets(dbf_filename);
     
  exit(export(database,sql_statement,dbf_filename)); }

/* export() does all the work, it's called from main(), it calls errcheck() */
     
int export (UCHAR *database, UCHAR *sql_statement, char *dbf_filename)
{
struct dbf_header *Head = NULL;		/* address of in-RAM .DBF header */
int file_fd = -1;			/* set by create() */
unsigned int dbf_header_size;		/* total size of field header table */
int i,j,k;
SWORD ccol;				/* passed back by SQLNumResultCols */
UWORD icol;				/* 'current col #', used in loops */
char c;
RETCODE         rc;
/* There are always at least 3 handles: one for the environment, one for the 
database connection, and one for the statement connection. ODBC allows 
multiple database connections and multiple (possibly synchronously running) 
statements, but this program doesn't need that. */
HENV henv = SQL_NULL_HENV;		/* env handle, initialized to 0 */
HDBC hdbc = SQL_NULL_HDBC;		/* dbc handle, initialized to 0 */
HSTMT hstmt = SQL_NULL_HSTMT;		/* stmt handle, initialized to 0 */

UCHAR *dbf_record = NULL;		/* points to malloc'd record buffer*/
unsigned int dbf_recordsize;
UCHAR szColName[SQLNAMESIZE+1];		/* returned from SQLDescribeCol*/
SWORD cbColName;			/* returned from SQLDescribeCol*/
SWORD fSqlType;				/* returned from SQLDescribeCol*/
SWORD colscale;				/* returned from SQLDescribeCol*/
SWORD colnullable;			/*ret'd from SQLDescribeCol, unused*/
UDWORD collen;				/* ret'd from SQLDescribeCol etc.*/
UCHAR *colvalue;
UCHAR *p;
int _offset;
int success_fail = FALSE;		/* what we return to main() */
     
/* PHASE I: INITIALIZATION */
     
  /* STANDARD ODBC INITIALIZATION CODE

  All functional ODBC programs always contain these statements in this order: 
  SQLAllocEnv(), SQLAllocConnect(), SQLConnect(), SQLAllocStmt(). So, unless 
  you need multiple connections and multiple statements at the same time, the 
  following code is boilerplate.

  Essentially, the SQLAlloc... functions merely cause the ODBC driver to 
  allocate internal space for storing semi-permanent information. Probably 
  the ODBC driver malloc's a bit of RAM for storing pointers to the other 
  handles and the most recent error messages, and (for the connection handle) 
  the name of the database, or (for the statement handle) the text of the 
  last SQL statement, the position within a cursor, etc.

  Take note: The three SQLAlloc... functions all return 32-bit "handles". 
  Some ODBC drivers are actually returning pointers to the ODBC driver's 
  semi-permanent information. To test for this, after initialization try 
  casting hstmt to a PTR ("p = (PTR) hstmt") and then see if accessing *p 
  results in a General Protection Fault with Windows 3.1. If not, you can 
  peek at the structure that the ODBC driver dynamically allocates when you 
  call SQLAllocStmt().

  The SQLConnect() function establishes a connection to a data source. A 
  "data source" is probably just a "database", as we can see from the fact 
  that "hdbc", according to Microsoft, stands for "handle to database 
  connection". So our SQLConnect(...,"SYSADM",..."",...) call just means 
  "open up <database> with userid='SYSADM', no password".

  Notice that we always follow ODBC function calls with a call to an 
  error-checking routine. In this phase, errors would most likely be due to 
  insufficient memory, failure to find a file, or network failure. */

  rc=SQLAllocEnv(&henv);
  if (errcheck("SQLAllocEnv",rc,henv,hdbc,hstmt)) goto cleanup;
     
  rc=SQLAllocConnect(henv,&hdbc);
  if (errcheck("SQLAllocConnect",rc,henv,hdbc,hstmt)) goto cleanup;
     
  rc=SQLConnect(hdbc,database,SQL_NTS,"SYSADM",SQL_NTS,"",SQL_NTS);
  if (errcheck("SQLConnect",rc,henv,hdbc,hstmt)) goto cleanup;
     
  rc=SQLAllocStmt(hdbc,&hstmt);
  if (errcheck("SQLAllocStmt",rc,henv,hdbc,hstmt)) goto cleanup;
     
  /* PHASE II: PREPARATION */
     
  /* PREPARE THE SELECT STATEMENT.

  Initialization is complete, we've done the SQLAlloc...() calls, and we've 
  opened the database. Now it's time to look at the SELECT statement that the 
  user typed in. We don't actually execute the SELECT statement yet -- first 
  we have to check if it's a valid statement, and then we'll do some 
  preparatory work based on the description of the columns being selected. */

  rc=SQLPrepare(hstmt,sql_statement,SQL_NTS);
  if (errcheck("SQLPrepare",rc,henv,hdbc,hstmt)) goto cleanup;

  /* GET THE NUMBER OF COLUMNS.

  We've initialized and we've run the user's SELECT statement through 
  SQLPrepare(). Now some information is available: specifically, how many 
  columns were in the SELECT (which we can now retrieve with the 
  SQLNumResultCols() function), and what the datatype, size, and name of each 
  column is (which we will soon retrieve with the SQLDescribeCol() function).

  Find out how many columns were in the SELECT that we just SQLPrepared. If 
  it wasn't a SELECT, then SQLNumResultCols should set ccol=0. If the number 
  is greater than the maximum number of fields allowed by some packages, 
  display a warning but continue. */
     
  /* "Let ccol = the number of columns in the SELECT that we SQLPrepared()" */
     
  rc = SQLNumResultCols(hstmt, &ccol);
  if (errcheck("SQLNumResultCols",rc,henv,hdbc,hstmt)) goto cleanup;
  if (ccol == 0) {
    printf("Error: there were no result columns in the SQL statement\n");
    printf("you typed. Either the statement was not legal, or it wasn't a\n");
    printf("SELECT statement (this program won't process INSERT,\n");
    printf("DELETE, etc.).\n");
    goto cleanup; }

  if (ccol>128) {
    printf("Warning: Some variants of dBASE and its clones cannot handle\n");
    printf("databases with more than 128 fields. We will be setting up a\n");
    printf("database with %d fields.\n",ccol); }
     
  /* ALLOCATE SPACE FOR THE HEADER OF THE .DBF FILE.

  We need enough for the file header, each of the field headers, and one 
  extra byte at the end. (Some packages put TWO bytes at the end; 
  consistency's tough when there's no standard)*/
     
  dbf_header_size=
  sizeof(struct dbf_header)+(ccol-1)*sizeof(struct field_header)+1;
  Head = malloc(dbf_header_size);
  if (Head == NULL) {
    printf("The malloc() function failed, the file header can't be set up\n");
    goto cleanup; }
  memset(Head,0,dbf_header_size);       /* initialize header to all \0s */
     
  /* SET VALUES IN THE .DBF FILE HEADER.

  The dbf_header_size value will be the number of bytes in the file header 
  plus all the field headers; the version will be 3, meaning "dBASE III made 
  this"; the date field will be October 1 1993 (we're avoiding Borland's 
  "getdate(&cdate)" because it's vendor-specific, the date here really 
  doesn't matter); dbf_header_recordcount will be filled in later; 
  dbf_header_reserve1, dbf_header_pending, dbf_header_reserve2, 
  dbf_header_mdx and on the diskette that comes withdbf_header_reserve3 all 
  remain equal to 0. */
     
  Head->dbf_header_size=dbf_header_size;
  Head->dbf_header_version=0x3;
  Head->dbf_header_last_update[0]=93;	/* year */
  Head->dbf_header_last_update[1]=10;	/* mon */
  Head->dbf_header_last_update[2]=01;	/* day */
     
  /* SET VALUES IN EACH OF THE .DBF FIELD HEADERS.

  This is done in a loop, ccol times (remember that ccol is the number of 
  columns, as we found out from our call to the SQLNumResultCols() function). 
  The key ODBC function that we need for finding information about columns is 
  the SQLDescribeCol() function. We have a few trivial problems to solve 
  afterwards, because the SQL description which we retrieve with 
  SQLDescribeCol() has names and types that don't have exact equivalents in a 
  .DBF description.

  In a general way, the loop is equivalent to the "exec sql DESCRIBE" 
  statement found in embedded SQL. */

  /* Head->dbf_header_record_size=1; */

  for (icol=0,_offset=0; icol<ccol; ++icol) {
    rc = SQLDescribeCol(hstmt,
    icol + 1,		/* "Given the column number, base 1, */
    szColName,		/* return the column name */
    SQLNAMESIZE,	/* (noting that the maximum name length is 18), */
    &cbColName,		/* return the size of the column name, */
    &fSqlType,		/*return the type, e.g. SQL_CHAR, SQL_INTEGER*/
    &collen,		/* return the precision ('length' if SqlType=CHAR),*/
    &colscale,		/* return the scale (0 if scale is not applicable), */
    &colnullable);	/* and return "nullability", which we'll ignore." */
    if (errcheck("SQLDescribeCol",rc,henv,hdbc,hstmt)) goto cleanup;

    /* SQLDescribeCol put the length of the SQL column name in cbColName and 
    the actual column name in szColName. Put the SQL column name in the .DBF 
    field header item [icol].dbf_fieldheader. */
     
    if (cbColName > DBFNAMESIZE - 1) {
      /* Alternative: check if rc=SQL_SUCCESS_WITH_INFO && sqlstate="01004" */
      printf("Warning: the SQL column name %s contains %d\n",cbColName);
      printf("characters. The maximum width of a colum name in a name in\n");
      printf("a .DBF field header is %d.\n",DBFNAMESIZE-1);
      cbColName=DBFNAMESIZE - 1; }
    for (i=0; i<cbColName; ++i) {
      if (!isalnum(szColName[i])) {
        printf("Warning: the SQL column name %s contains\n");
        printf("one or\n",szColName);
        printf("more non-alphanumeric characters. Such names may not\n");
        printf("be accepted in field headers by all dBASE-like products.\n");
        break; } }
    strncpy(Head->ff[icol].dbf_fieldheader_name,szColName,cbColName);

    /* SQLDescribeCol put the type of the SQL column in fSqlType, and the 
    length (or numeric precision) of the SQL column in collen. If the SQL 
    type is CHAR or VARCHAR, then the .DBF field header type is 'C' and the 
    .DBF size is the same as the SQL size. Otherwise, the .DBF fieldheader 
    type is 'N' and the size is the maximum number of displayable characters. 
    For instance, SQL_INTEGERs are 32 bits and signed, so the largest 
    possible number is -2147483648, which is 11 bytes wide. With SQL_DECIMAL, 
    "." and "-" are not included in SQL's precision+scale, but are included 
    for a .DBF, since the .DBF's size is the count of displayable characters. 
    */

    if (fSqlType==SQL_CHAR || fSqlType==SQL_VARCHAR) {
      Head->ff[icol].dbf_fieldheader_type='C'; }/* SQL char --> dbf 'C'*/
    else {
      if (fSqlType==SQL_INTEGER) collen=12;	/* max=-2147483648 */
      else if (fSqlType==SQL_SMALLINT) collen=6;/* max=-32767 */
      else if (fSqlType==SQL_DECIMAL) collen+=2;/* include for "-" and "."*/
      else if (fSqlType==SQL_FLOAT) ;
      else {
        printf("Column %s has a 'non-core' data type.\n",szColName);
        printf("The only data types we can process are INTEGER,\n");
        printf("SMALLINT, DECIMAL, FLOAT, CHAR and\n");
        printf("VARCHAR.\n"); goto cleanup; }
      Head->ff[icol].dbf_fieldheader_type='N'; }

    if (collen>255) {
      printf("Warning: Column %s is %d bytes wide.\n",szColName,collen);
      printf("Many dBASE-compatible products cannot handle a column\n");
      printf("wider than 255. But some can, so we'll continue without\n");
      printf("truncating.\n"); }

    Head->ff[icol].dbf_fieldheader_size=collen;
    Head->ff[icol].dbf_fieldheader_decimal=colscale;
    Head->dbf_header_record_size+=collen;
    Head->ff[icol].dbf_fieldheader_address=0x516b000d+_offset;
    _offset+=Head->ff[icol].dbf_fieldheader_size;
    Head->ff[icol].dbf_fieldheader_flag=1;
     
    /* dbf_fieldheader_reserve1 and dbf_fieldheader_reserve2 and 
    dbf_fieldheader_mdx all remain 0 */ }
     
  /* CREATE THE .DBF FILE AND WRITE THE HEADER.
  As a result of our call to SQLNumResultCols() and the loop that came after 
  it with the SQLDescribeCol() function, we now have all the information we 
  need for a .DBF header -- dbf_header has the general data, field_headers 
  have the dope on each field (name, size, type, etc.). We don't yet know the 
  number of rows, but we intend to come back later and fill in the 
  number-of-rows field when we've finished. */
     
  file_fd=open(dbf_filename,O_CREAT|O_RDWR|O_BINARY|O_TRUNC,S_IREAD|S_IWRITE);
  if (file_fd<0) {
    printf("The open() function failed, the .DBF file can't be created\n");
    goto cleanup; }
     
  if (write(file_fd,Head,dbf_header_size)!=dbf_header_size) {
    printf("The write() function failed, the .DBF header can't be written\n");
    goto cleanup; }
     
  /* PHASE III: EXECUTION */

  /* ALLOCATE A RECORD BUFFER TO FETCH DATA INTO AND WRITE DATA FROM.
  
  The record buffer is what we write to the .DBF file. Within the record 
  buffer is the space for each field. The size of each field was figured out 
  earlier, it's still in the field header (dbf_fieldheader_size). We believe 
  it's slightly more efficient to have just one record buffer, rather than a 
  separate buffer for each field.

  The key ODBC function call in the following loop is SQLBindCol(), which 
  tells the driver what address to fetch column values into and what 
  conversions to perform. In this case, we tell it to convert to SQL_C_CHAR 
  i.e. \0-terminated strings regardless of original type; that's because a 
  .DBF file contains nothing but ASCII data. */

  dbf_recordsize = Head->dbf_header_record_size + 1;
  dbf_record = (void *) malloc(dbf_recordsize + 1);
  if (dbf_record == NULL) {
    printf("A malloc() function failed, the record buffer can't be set up\n");
    goto cleanup; }
  for (icol=0,colvalue=dbf_record+1; icol<ccol; ++icol) {
    Head->ff[icol].dbf_fieldheader_address=(long)colvalue;
    collen=Head->ff[icol].dbf_fieldheader_size;
    /* colvalue points to where we'll fetch data for this field, i.e. for 
    field number [icol+1], when we call the SQLFetch() function later; collen 
    is the maximum amount of data that we can fetch for the field (which 
    should be OK since it equals the defined size); the final parameter for 
    SQLBindCol() tells it where to put a column's "indicator" value. */
    rc = SQLBindCol(hstmt,icol+1,SQL_C_CHAR,colvalue,collen,
    (SDWORD*)Head->ff[icol].dbf_fieldheader_reserve2);
    if (errcheck("BindCol",rc,henv,hdbc,hstmt)) goto cleanup;
    colvalue+=collen; }

  /* Set the first byte of the record buffer to blank to indicate it's not 
  deleted (a 'deleted' record in a .DBF file is represented by 0x2a in the 
  first byte). */
  *dbf_record=' ';

  /* ACTUALLY DO THE SELECT STATEMENT.

  This is "part II" of SQL-statement processing. In part I we call the 
  SQLPrepare() function to check the statement out and find out about its 
  characteristics and effects; in part II we call SQLExecute() to actually 
  perform the statement. (The two steps can be combined using the ODBC 
  function SQLExecDirect(), which we don't illustrate in this program.)

  We could have called SQLExecute() immediately after SQLPrepare(), and then 
  set up the .DBF header. Instead, we called SQLPrepare(), then we set up the 
  .DBF header, and now we're going to call SQLExecute(). Our idea is that 
  executing a SQL statement is a use of precious system (or network) 
  resources; therefore if anything can go wrong with the .DBF setup then we 
  may as well find out in advance, so that we can skip processing the SELECT. 
  It's a minor design point.
  
  Anyway, the ODBC driver stored the prepared SELECT statement in the handle 
  referenced by hstmt, so now we just have to invoke it. */

  rc = SQLExecute(hstmt);
  if (errcheck("SQLExecute",rc,henv,hdbc,hstmt)) goto cleanup;

  /* THE MAIN LOOP -- FETCH SQL ROWS, WRITE .DBF RECORDS.

  We've initialized the ODBC interface (including connecting to the 
  database), we've opened the .DBF file for output and set up the .DBF header 
  information based on information we got about the SQL statement's columns, 
  and we've actually executed the SQL statement, so now all the rows we've 
  asked for are waiting for us in what's called a "result set". All we have 
  to do is get data from the result set using the ODBC SQLFetch() function, 
  and dump the data into the .DBF file using good old write(), one row at a 
  time until SQLFetch() tells us there are no more rows. */
     
  for (;;) {
    /* We're ready to get the next row of the result set with SQLFetch(). */
    rc = SQLFetch(hstmt);
    if (errcheck("SQLFetch",rc,henv,hdbc,hstmt)) goto cleanup;
    if (rc==SQL_NO_DATA_FOUND) break;
  
    /* The SQLFetch() succeeded, so now the record buffer is filled with 
    data. Remember our call to the SqlNumResultCols() function? That's when 
    we set ccol to the number of columns in each row of the result set. 
    Remember our calls to SQLBindCol()? They established the address of each 
    column's data (pointed to by ...[icol].dbf_fieldheader_address in the 
    field header) and we use other items in the field header -- [icol] 
    .dbf_fieldheader_size, [icol].dbf_fieldheader_type -- to do the final 
    massaging of that data: space filling, null substitution, and right 
    justification. */
     
    for (icol=0; icol<ccol; ++icol) {
      memmove(&collen,Head->ff[icol].dbf_fieldheader_reserve2,sizeof(SDWORD));
      colvalue=(UCHAR *)Head->ff[icol].dbf_fieldheader_address;
      if (collen==SQL_NULL_DATA) {
        collen=0;

        /* collen == -1, meaning the SQL column had a NULL at this location. 
        Since dBASE doesn't support NULLs, there's no official way to store 
        them in .DBF files. We'll act as if the field is blank. That's 
        consistent with the way that e.g. XDB's export-to-DBF utility does 
        it.*/ }

      /* The data *colvalue is SQL_C_CHAR, so it's \0-terminated. That's not 
      how we want to store it: a .DBF file is always fixed field. If it's If 
      it's a Character field ('C'), space fill on the right. If it's a 
      Numeric field ('N'), right-justify and space fill on the left. */
     
      i=Head->ff[icol].dbf_fieldheader_size; /* i="defined size" */
      if (Head->ff[icol].dbf_fieldheader_type=='C') {
        memset(colvalue+collen,' ',i-collen); }
      else {
        strncpy(colvalue+i-collen,colvalue,collen);
        memset(colvalue,' ',i-collen); }
     
      /* Fix the number of zeros after the decimal point, if any. */
     
      k=Head->ff[icol].dbf_fieldheader_decimal;
      if (k>0) {
        *(colvalue+j)='\0';
        p=strchr(colvalue,'.');
        if (p==0) {
          strcpy(colvalue,colvalue+1);
          p=colvalue+j-1;
          *p='.'; }
        k-=(colvalue+j)-(p+1);
        while (k>0) {
          strcpy(colvalue,colvalue+1);
          *(colvalue+j-1)='0';
          --k; } } }
    
    /* We've massaged all the fields in the record buffer. Now dump the 
    record buffer to the .DBF file and increment the record count in the file 
    header buffer. This is the end of the main loop. */
     
    if (write(file_fd,dbf_record,dbf_recordsize)!=dbf_recordsize) {
      printf("The write() function failed, the record can't be written\n");
      goto cleanup; }
    ++Head->dbf_header_recordcount; }
     
  /* There's only one way to get here: we broke out of the SQLFetch() loop 
  because "if (rc==SQL_NO_DATA_FOUND)" is true, meaning all rows have been 
  written. At this point some packages write 0x1a (control-Z) as an "eof" 
  marker. */
     
  /* WRITE THE RECORD COUNT IN THE HEADER.
  
  While we were in the loop we were incrementing Head->dbf_header_recordcount 
  so now we know how many rows there are in the file, total. Seek back to 
  near the start of the .DBF file and rewrite that. The other header 
  information is unchanged. */
     
  if (lseek(file_fd,4L,0)==-1) {
    printf("The seek() function failed, the record count can't be written\n");
    goto cleanup; }
  if (write(file_fd,&Head->dbf_header_recordcount,4)!=4) {
    printf("The write() function failed, the recordcount can't be written\n");
    goto cleanup; }
     
  /* Mark that we're a big success, getting here means we did all the rows. */
     
  success_fail=TRUE;
     
  /* PHASE IV: DE-INITIALIZATION */
     
cleanup: ;
     
  /* CLEAN UP.

  We're done. Now it's time to close the .DBF file, free the buffers that we 
  malloc'd for the header and record buffer, disconnect from the database, 
  free the SQL environment and database connection and statement handles, and 
  return to the caller. Cleanup is necessary because we are taking up 
  system-wide resources. For example, an ODBC driver might call the Windows 
  function GlobalAlloc() when we called the SQLAllocStmt() function at the 
  start of the program. So now we have to call the SQLFreeStmt() function, 
  which will make the ODBC driver call GlobalFree().

  All handles and pointers were initialized to null values. Since we 'goto 
  cleanup' if any error happens, they might still be. For instance, hstmt 
  will still equal SQL_NULL_HSTMT if the SQLConnect() function failed. 
  Anyway: destroy all remaining global objects, in the reverse order that 
  they were created in. */
     
  if (dbf_record!=NULL) free(dbf_record);	/*reverse malloc(dbf_record)*/
  if (Head!=NULL) free(Head);			/* reverse malloc(Head) */
  if (file_fd>=0) close(file_fd);		/* reverse open(filename) */
     
  /* Experienced SQL programmers might execute a "ROLLBACK" statement here, 
  as part of the cleanup procedure. If that's your habit, break it. First, 
  you're supposed to avoid "ROLLBACK" even though it's ANSI; the 
  "recommended" ODBC equivalent is "SQLTransact(henv,hdbc,SQL_ROLLBACK);". 
  Second, there is no guarantee that all cursors will be closed by ROLLBACK 
  (even though that's ANSI too); it depends on a setting named 
  SQL_CURSOR_ROLLBACK_BEHAVIOR which can't be controlled. The correct way to 
  to close a cursor is with the SQLFreeStmt() function. Here, we'll use 
  SQLFreeStmt() with the SQL_DROP option. */
     
  if (hstmt!=SQL_NULL_HSTMT) {
    rc = SQLFreeStmt(hstmt, SQL_DROP);		/* reverse SQLAllocStmt() */

    /* It's very unlikely that SQLFreeStmt() will fail, given that it's 
    probably merely freeing memory.  All the more reason to check for 
    failure, then: if strange things are happening, the user should be warned 
    of the fact. */

    if (errcheck("SQLFreeStmt",rc,henv,hdbc,hstmt)) {
      printf("Continuing.\n"); }
      hstmt=SQL_NULL_HSTMT; }
     
    if (hdbc!=SQL_NULL_HDBC) {
      rc = SQLDisconnect(hdbc);		/* reverse SQLConnect() */
      rc = SQLFreeConnect(hdbc);	/* reverse SQLAllocConnect() */
      if (errcheck("SQLFreeConnect",rc,henv,hdbc,hstmt)) {
        printf("Continuing.\n"); }
      hdbc=SQL_NULL_HDBC; }

    if (henv!=SQL_NULL_HENV) {
      rc = SQLFreeEnv(henv);		/* reverse SQLAllocEnv() */
      if (errcheck("SQLExecute",rc,henv,hdbc,hstmt)) {
      printf("Continuing.\n"); }
      henv=SQL_NULL_HENV; }
     
  /* end of export() function, return TRUE or FALSE to main(). */
     
  return (success_fail); }

/* We call errcheck() after every ODBC function call. If there's any problem, 
we want to display whatever information we can get about it. ODBC's 
SQLError() function is available for this purpose. It should tell us a 
standardized error code -- sqlstate -- and it should give us an error 
message, which is probably driver-specific. There's a bunch of problems here: 
what if the call to SQLError() itself returns an error? what if the 
error-message buffer isn't big enough to hold what SQLError() returns? what 
if the error code isn't something we expect? We've tried to be thorough about 
our diagnosis, so this error checking routine is almost as long as the main 
program! But we didn't think it's a good idea to skimp on the error checking 
in a "model" ODBC program. If all goes well, errcheck() returns FALSE. 
Otherwise it returns TRUE, which is a signal to the caller that it's a good 
idea to smoothly abort. */
     
int errcheck (UCHAR *msg,RETCODE rc,HENV henv,HDBC hdbc,HSTMT hstmt)
{
UCHAR szSqlState[SQL_SQLSTATE_SIZE+1];	/* defined by SAG as = 5 */
SDWORD pfNativeError;
SWORD pcbErrorMsg;			/* gets actual error message size */
UCHAR *szErrorMsg;			/* points to error message buffer */
SWORD cbErrorMsgMax = SQL_MAX_MESSAGE_LENGTH;
RETCODE sqlerror_rc;

  /* Test for three general possibilities: the last function returned "okay" 
  (i.e. SQL_SUCCESS, SQL_SUCCESS_WITH_INFO, or SQL_NO_DATA_FOUND); or it 
  returned SQL_ERROR, or it returned something unexpected. */
     
  switch (rc) {
    case SQL_SUCCESS:
    case SQL_SUCCESS_WITH_INFO:
    case SQL_NO_DATA_FOUND:
    return (FALSE);
     
    case SQL_ERROR:
    printf("The call to function %s() returned SQL_ERROR.\n",msg);
    printf("Calling the SQLError() function to get more information.\n");

nogo:szErrorMsg=malloc(cbErrorMsgMax);	/* make a 512-byte message buffer*/
    if (szErrorMsg==NULL) {
      printf("A malloc() failed, can't set up szErrorMsg\n");
      return (TRUE); }
     
    sqlerror_rc = SQLError(     /* "I want more info about the error: */
    henv,			/* From the environment handle, or */
    hdbc,			/* from the database-connection handle, or*/
    hstmt,			/* from the statement handle (if not null) */
    szSqlState,			/* return a \0-terminated 5-byte sqlstate, */
    &pfNativeError,		/* return the driver's native error code, */
    szErrorMsg,			/* return the error message text */
    cbErrorMsgMax,		/* (whose maximum size is 512 bytes), */
    &pcbErrorMsg);		/* and return how big the message is." */

    switch (sqlerror_rc) {
      case SQL_SUCCESS:
      case SQL_SUCCESS_WITH_INFO:
      printf("A call to SQLError() gives more\n");
      printf("detailed information:\n");
      printf("SqlState = %s.\n",szSqlState);
      printf("Native error code = %ld.\n",pfNativeError);
      printf("Error Message = %s.\n",szErrorMsg);
      if (pcbErrorMsg>cbErrorMsgMax) {

	/* The message text in szErrorMsg can be as large as 
	SQL_MAX_MESSAGE_LENGTH, which is defined in SQL.H as 512. But the 
	actual text that could have been returned is larger than that, as we 
	see by comparing to pcbErrorMsg. You might think "let's increase the 
	message size and try again", thus: free(szErrorMsg); 
	cbErrorMsgMax=pcbErrorMsg; goto nogo; BUT! The second call to 
	SQLError() will probably return SQL_NO_DATA_FOUND; once you call 
	SQLError(), it "removes the error from the list of available errors". 
	So you have to have a big enough buffer the first time. */

        printf("Warning: the above error message was truncated\n");
        break;
        default:
     
	/* The call to SQLError() itself resulted in an error. Perhaps it's 
	SQL_INVALID_HANDLE because henv+hdbc+hstmt are all null -- which 
	would be the case if SQLAllocEnv() failed. Perhaps it's 
	SQL_NO_DATA_FOUND or SQL_ERROR, in which case heaven knows why */
     
        printf("No further information is available.\n");
        break; }
      free(szErrorMsg);
      return (TRUE); }
    default:
    
    /* In this program, it's quite unlikely that any of the SQL function 
    calls will return anything other than SQL_SUCCESS, SQL_SUCCESS_WITH_INFO, 
    SQL_NO_DATA_FOUND, or SQL_ERROR. But there are other theoretical 
    possibilities (SQL_INVALID_HANDLE, SQL_NEED_DATA, SQL_STILL_EXECUTING) so 
    we have a 'default:' case for completeness.*/
     
    printf("The call to function %s() returned an unexpected\n");
    printf("code.\n",msg);
    printf("This is probably due to a programming error.\n");
    return (TRUE); } }
