%{
/*
 *  Program: extract -- a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@ukc.ac.uk
 *
 */
%}

%{
#include <stdio.h>
#include <string.h>
#include "bool.h"
#include "defs.h"
#include "typedef.h"
#include "protos.h"
#include "taglist.h"
#include "map.h"
#include "lists.h"

	/* This version of the grammar contains 2 shift/reduce conflicts. */
%}

%token TERMINATING_RESULT
%token TAG STRING MISC_TAG
%token MOVE CHECK MOVE_NUMBER
%token RAV_START RAV_END
%token NAG
%token COMMENT

%union{
    /* This string is used to retain tag and result information. */
    char *token_string;
    /* Define a type for keeping track of a move list.
     * This is needed to make it possible to make the move_list
     * rules left recursive and effectively manage.
     * the resulting list.
     */
    struct{
	void *head, *tail;
    } head_and_tail;
    /* Move information. */
    Move *move_details;
    unsigned move_number;
    StringList *nags;
    Variation *variation_details;
    /* An index into the Details array for tag strings. */
    unsigned tag_index;
}

%{

/* The maximum length of an output line.  This is conservatively
 * slightly smaller than the PGN standard of 80.
 */
#define MAX_LINE_LENGTH 75

StateInfo GlobalState = {
    FALSE,		/* check_only */
    TRUE,		/* verbose */
    TRUE,		/* keep_NAGs */
    TRUE,		/* keep_comments */
    TRUE,		/* keep_variations */
    FALSE,		/* seven_tag_roster */
    FALSE,		/* match_permutations */
    FALSE,		/* positional_variations */
    FALSE,		/* use_soundex */
    FALSE,		/* suppress_duplicates */
    MAX_LINE_LENGTH,	/* max_line_length */
    (char *)NULL,	/* current_input_file */
    stdout,		/* outputfile */
    stderr,		/* logfile */
    (FILE *)NULL, 	/* duplicate_file */
};
    

    /* Retain the tag values. */
static char *Details[NUM_TAGS];

/* How much text we have output on the current line. */
static unsigned line_length = 0;

static void print_separator(FILE *fp);
static void print_str(FILE *fp, const char *str);
static void print_comment(FILE *fp, StringList *comment);
static Boolean print_move(FILE *outputfile, unsigned move_number,
		unsigned print_move_number, unsigned white_to_move,
		const Move *move_details);
static void print_move_list(FILE *outputfile, unsigned move_number,
		unsigned white_to_move, const Move *move_details);
static void add_check(char *move);
static void free_details(void);
static void show_tags(FILE *outfp);
static void output_STR(FILE *outfp);
static void setup_for_new_game(void);
int yylex(void);
int yyparse(void);

%}

%type <token_string> STRING TERMINATING_RESULT opt_result
%type <move_details> MOVE move move_unit move_and_variants
%type <head_and_tail> move_list variant_list
%type <variation_details> variant
%type <move_number> MOVE_NUMBER
%type <move_number> opt_move_number
%type <nags> opt_NAG_list NAG_list
%type <token_string> NAG
%type <tag_index> TAG
%type <token_string> MISC_TAG
%type <move_details> COMMENT

%%

list : opt_game_list
     ;

opt_game_list : /* Empty. */
	      | game_list
	      ;

game_list : game
	  | game_list game
	  ;

game : opt_tag_list move_list TERMINATING_RESULT
          {   Game current_game;

	      /* Append the result to the score. */
	      ((Move *) $2.tail)->terminating_result = $3;
	      /* Fill in the information currently known. */
	      current_game.tags = Details;
	      current_game.moves = $2.head;
	      /* Check out the moves.  This will also fill in
	       * current_game.final_hash_value and current_game.cumulative_hash_value .
	       */
	      if(apply_move_list(&current_game) && 
			 check_textual_variations(current_game) &&
			 CheckDetails(current_game.tags)){
		  Boolean duplicate = game_is_a_duplicate(current_game);

		  if(!duplicate || !GlobalState.suppress_duplicates){
		      if(!GlobalState.check_only){
			  FILE *outputfile = GlobalState.outputfile;

			  /* See if we wish to separate out duplicates. */
			  if(duplicate && (GlobalState.duplicate_file != NULL)){
			      outputfile = GlobalState.duplicate_file;
			  }
			  /* Report details on the output. */
			  if(GlobalState.seven_tag_roster){
			      output_STR(outputfile);
			  }
			  else{
			      show_tags(outputfile);
			  }
			  print_move_list(outputfile,1,1,current_game.moves);
			  if(line_length > 0){
			      putc('\n',outputfile);
			  }
			  putc('\n',outputfile);
		      }
		      if(GlobalState.verbose){
			  /* Report progress on logfile. */
			  report_details(GlobalState.logfile);
		      }
		  }
	      }
	      free_details();
	      free_move_list(current_game.moves);
	      setup_for_new_game();
	  }
      | tag_list TERMINATING_RESULT
	  {
	      if(GlobalState.verbose){
		  fprintf(GlobalState.logfile,"Empty game: ");
		  report_details(GlobalState.logfile);
	      }
	      free_details();
	      (void) free((void *)$2);
	      setup_for_new_game();
	  }
      | COMMENT
	  { 
	    free_move_list($1);
	  }
      | opt_tag_list move_list error TERMINATING_RESULT
	  {
	      /* An error was found, skip to the end of the game. */
	      yyerrok;
	      yyclearin;
	      free_details();
	      free_move_list($2.head);
	      (void) free((void *)$4);
	      setup_for_new_game();
	  }
       | error TERMINATING_RESULT
	  {
	      /* An error was found, skip to the end of the game. */
	      yyerrok;
	      yyclearin;
	      free_details();
	      (void) free((void *)$2);
	      setup_for_new_game();
	  }
      ;

opt_tag_list : /* Empty. */
	     | tag_list
	     ;

tag_list     : tag
	     | tag_list tag
	     ;

tag	: TAG STRING
		{ /* A tag which might be worth keeping. */
		  if($1 < NUM_TAGS){
		      Details[$1] = $2;
		  }
		  else{
		      fprintf(GlobalState.logfile,"Internal error: Illegal tag index for %s\n",$2);
		  }
		}
	| MISC_TAG STRING
		{ /* We don't wish to use these strings. */
		  (void) free($1);
		  (void) free($2);
		}
	;

move_list : move_and_variants
		{ /* The end of a new list. */
		  $1->next = NULL;
		  $$.head = $1;
		  $$.tail = $1;
		}
	  | move_list move_and_variants
		{ 
		  $2->next = NULL;
		  /* Add this move to the tail of the list. */
		  ((Move *) $1.tail)->next = $2;
		  /* Retain the old head. */
		  $$.head = $1.head;
		  /* Pick up the new tail. */
		  $$.tail = $2;
		}
	  ;

move_and_variants : move
		      { $1->next = NULL;
			$1->Variants = NULL;
			$$ = $1;
		      }
		  | move variant_list
		      { $1->Variants = $2.head;
			$$ = $1;
		      }
		  ;


move : opt_move_number move_unit opt_NAG_list
	{ $2->Nags = $3;
	  $$ = $2;
	}
     ;

move_unit : MOVE
	  | MOVE CHECK
	     { add_check($1->move);
	       $$ = $1;
	     }
	  | COMMENT
	     { /* Represent a comment with a NULL move string. */
	       $1->move[0] = '\0';
	       $$ = $1;
	     }
	  ;

opt_move_number : /* Empty. */
		  { /* We don't know what the number is. */
		    $$ = 0;
		  }
		| MOVE_NUMBER
		  {
		    /* Pass up the move number. */
		    $$ = $1;
		  }
		;

opt_NAG_list : /* Empty. */
		{
		    $$ = (StringList *)NULL;
		}
	     | NAG_list
		{ $$ = $1;
		}
	     ;

NAG_list : NAG
	    { if(GlobalState.keep_NAGs){
		$$ = SaveStringListItem((StringList *)NULL,$1);
	      }
	      else{
		(void) free((void *)$1);
		$$ = (StringList *)NULL;
	      }
	    }
	 | NAG_list NAG
	    { if(GlobalState.keep_NAGs){
		$$ = SaveStringListItem($1,$2);
	      }
	      else{
		(void) free((void *)$2);
		$$ = (StringList *)NULL;
	      }
	    }
	 ;

variant_list : variant
		{ 
		  $1->next = NULL;
		  $$.head = $1;
		  $$.tail = $1;
		}
	     | variant_list variant
		{ $2->next = NULL;
		  ((Variation *) $1.tail)->next = $2;
		  $$.head = $1.head;
		  $$.tail = $2;
		}
	     ;

variant : RAV_START move_list opt_result RAV_END
		{   Move *last_move = (Move *) $2.tail;

		    $$ = (Variation *) MallocOrDie(sizeof(Variation));
		    last_move->terminating_result = $3;
		    $$->moves = $2.head;
		    $$->next = NULL;
		}
	| RAV_START move_list TERMINATING_RESULT COMMENT RAV_END
		{   Move *last_move = (Move *) $2.tail;

		    $$ = (Variation *) MallocOrDie(sizeof(Variation));
		    last_move->terminating_result = $3;
		    if($4 != NULL){
			/* Add in a terminating comment. */
			last_move->next = $4;
		    }
		    $$->moves = $2.head;
		    $$->next = NULL;
		}
	 ;

opt_result : { /* Empty. */
		$$ = (char *)NULL;
	     }
	   | TERMINATING_RESULT
	      { $$ = $1;
	      }
	   ;

%%

	/* Ensure that there is room for len more characters on the
	 * current line.
	 */
static void
check_line_length(FILE *fp,unsigned len)
{
    if((line_length + len) > GlobalState.max_line_length){
	putc('\n',fp);
	line_length = 0;
    }
}

	/* Print ch to fp and update how much of the line
	 * has been printed on.
	 */
static void
single_char(FILE *fp, char ch)
{
    check_line_length(fp,1);
    putc(ch,fp);
    line_length++;
}

	/* Print a space, unless at the beginning of a line. */
static void
print_separator(FILE *fp)
{
    /* There is no printing a trailing space so allow for at least
     * one more character on the line.
     */
    check_line_length(fp,2);
    if(line_length != 0){
	putc(' ',fp);
	line_length++;
    }
}

	/* Print str to fp and update how much of the line
	 * has been printed on.
	 */
static void
print_str(FILE *fp, const char *str)
{   unsigned len = strlen(str);

    check_line_length(fp,len);
    fprintf(fp,"%s",str);
    line_length += len;
}

static void
print_comment(FILE *fp, StringList *comment)
{
    if(comment != NULL){
	/* We will use strtok to break up the comment string,
	 * with chunk to point to each bit in turn.
	 */
	char *chunk;
	
	single_char(fp,'{');
	for( ; comment != NULL; comment = comment->next){
	    chunk = strtok(comment->str," ");
	    while(chunk != NULL){
		print_separator(fp);
		print_str(fp,chunk);
		chunk = strtok((char *)NULL," ");
	    }
	}
	print_separator(fp);
	single_char(fp,'}');
    }
}
	/* Output the current move along with associated information.
	 * Return TRUE if a variation was printed, FALSE otherwise.
	 * This is needed to determine whether a new move number
	 * is to be printed after a variation.
	 */
static Boolean
print_move(FILE *outputfile, unsigned move_number, unsigned print_move_number,
		unsigned white_to_move, const Move *move_details)
{   Boolean variation_printed = FALSE;

    if(move_details == NULL){
	/* Shouldn't happen. */
    }
    else{
	StringList *nags = move_details->Nags;
	Variation *variants = move_details->Variants;
        if(!GlobalState.check_only){
	if(move_details->move[0] != '\0'){
	    if(white_to_move || print_move_number){
		static char small_number[] = "99999...";

		sprintf(small_number,"%u.%s",move_number, white_to_move?"":"..");
		print_str(outputfile,small_number);
		print_separator(outputfile);
	    }
	    print_str(outputfile,move_details->move);
	    /* Update white_to_move and move_number, in case there
	     * are variations to print.  
	     */
	    if(!white_to_move){
		move_number++;
	    }
	    white_to_move = !white_to_move;
	}
	else{
	    /* An empty move should be a comment. */
	    if(GlobalState.keep_comments){
		if(move_details->Comment != NULL){
		    print_comment(outputfile,move_details->Comment);
		}
	    }
	    /* No need to modify white_to_move and move_number as this
	     * will have been done by print_move_list.
	     */
	}
	/* Print further information, that may be attached to moves
	 * and comments.
	 */
	if(GlobalState.keep_NAGs){
	    while(nags != NULL){
		print_separator(outputfile);
		print_str(outputfile,nags->str);
		nags = nags->next;
	    }
	}
	if((GlobalState.keep_variations) && (variants != NULL)){
	    /* Step back the current move status for the variation, because
	     * these will already havve been advanced by printing of the
	     * previous move.
	     */
	    if(white_to_move){
		move_number--;
	    }
	    white_to_move = !white_to_move;
	    while(variants != NULL){
		print_separator(outputfile);
		single_char(outputfile,'(');
		/* Always start with a move number. */
		print_move_list(outputfile,move_number,white_to_move,variants->moves);
		single_char(outputfile,')');
		variants = variants->next;
	    }
	    variation_printed = TRUE;
	}
      }
    }
    /* See if there is a result attached.  This may be attached either
     * to a move or a comment.
     */
    if(!GlobalState.check_only && (move_details != NULL) &&
		(move_details->terminating_result != NULL)){
       print_separator(outputfile);
       print_str(outputfile,move_details->terminating_result);
    }
    return variation_printed;
}

static void
print_move_list(FILE *outputfile, unsigned move_number, unsigned white_to_move,
		const Move *move_details)
{   unsigned print_move_number = 1;
    const Move *move = move_details;

    while(move != NULL){
        /* Reset print_move number if a variation was printed. */
        if(print_move(outputfile,move_number,print_move_number,white_to_move,move)){
 	   print_move_number = 1;
        }
        else{
	   print_move_number = 0;
        }
	if(move->move[0] != '\0'){
	    /* A genuine move was just printed, rather than a comment. */
	    if(white_to_move){
	      white_to_move = 0;
	    }
	    else{
	       move_number++;
	       white_to_move = 1;
	    }
	}
        move = move->next;
        if(move != NULL){
 	   print_separator(outputfile);
        }
    }
}

	/* Append a check-designator against the current move. */
static void
add_check(char *move)
{
    strcat(move,"+");
}

	/* Print out on outfp the current details.
	 * These can be used in the case of an error.
	 */
void
report_details(FILE *outfp)
{
      if(Details[WHITE_TAG] != NULL){
	  fprintf(outfp,"%s - ",Details[WHITE_TAG]);
      }
      if(Details[BLACK_TAG] != NULL){
	  fprintf(outfp,"%s ",Details[BLACK_TAG]);
      }
      fflush(outfp);
      if(Details[EVENT_TAG] != NULL){
	  fprintf(outfp,"%s ",Details[EVENT_TAG]);
      }
      if(Details[SITE_TAG] != NULL){
	  fprintf(outfp,"%s ",Details[SITE_TAG]);
      }
      fflush(outfp);
      if(Details[DATE_TAG] != NULL){
	  fprintf(outfp,"%s ",Details[DATE_TAG]);
      }
      putc('\n',outfp);
      fflush(outfp);
}


	/* Output the tags held in the Details structure.
	 * At least the full Seven Tag Roster is printed.
	 */
static void
output_tag(unsigned tag_index, FILE *outfp)
{   const char *tag_string;

    switch(tag_index){
	case EVENT_TAG:
	    tag_string = "Event";
	    break;
	case SITE_TAG:
	    tag_string = "Site";
	    break;
	case DATE_TAG:
	    tag_string = "Date";
	    break;
	case ROUND_TAG:
	    tag_string = "Round";
	    break;
	case WHITE_TAG:
	    tag_string = "White";
	    break;
	case BLACK_TAG:
	    tag_string = "Black";
	    break;
	case RESULT_TAG:
	    tag_string = "Result";
	    break;
	case ECO_TAG:
	    tag_string = "ECO";
	    break;
	case WHITE_TITLE_TAG:
	    tag_string = "WhiteTitle";
	    break;
	case BLACK_TITLE_TAG:
	    tag_string = "BlackTitle";
	    break;
	case WHITE_ELO_TAG:
	    tag_string = "WhiteElo";
	    break;
	case BLACK_ELO_TAG:
	    tag_string = "BlackElo";
	    break;
	case WHITE_USCF_TAG:
	    tag_string = "WhiteUSCF";
	    break;
	case BLACK_USCF_TAG:
	    tag_string = "BlackUSCF";
	    break;
	case WHITE_NA_TAG:
	    tag_string = "WhiteNA";
	    break;
	case BLACK_NA_TAG:
	    tag_string = "BlackNA";
	    break;
	case WHITE_TYPE_TAG:
	    tag_string = "WhiteType";
	    break;
	case BLACK_TYPE_TAG:
	    tag_string = "BlackType";
	    break;
	case ANNOTATOR_TAG:
	    tag_string = "Annotator";
	    break;
	case PSEUDO_PLAYER_TAG:
	    /* A Pseudo-tag, not to be output. */
	    tag_string = NULL;
	    break;
	default:
	    fprintf(GlobalState.logfile,"Internal error: unknown index %d in output_tag.\n",
			tag_index);
	    tag_string = NULL;
    }
    if(tag_string != NULL){
	/* Must print STR elements. */
	if((tag_index < 7) || (Details[tag_index] != NULL)){
	    fprintf(outfp,"[%s \"",tag_string);
	    if(Details[tag_index] != NULL){
		fwrite(Details[tag_index],sizeof(char),
				strlen(Details[tag_index]),outfp);
	    }
	    else{
		if(tag_index == DATE_TAG){
		    fprintf(outfp,"????.??.??");
		}
		else{
		    fprintf(outfp,"?");
		}
	    }
	    fprintf(outfp,"\"]\n");
	}
    }
}

	/* Output the Seven Tag Roster. */
static void
output_STR(FILE *outfp)
{   unsigned tag_index;

    for(tag_index = 0; tag_index < 7; tag_index++){
	output_tag(tag_index,outfp);
    }
    putc('\n',outfp);
}

	/* Print out on outfp the current details.
	 * These can be used in the case of an error.
	 */
static void
show_tags(FILE *outfp)
{   unsigned tag_index;

    for(tag_index = 0; tag_index < NUM_TAGS; tag_index++){
	output_tag(tag_index,outfp);
    }
    putc('\n',outfp);
}

	/* Discard any data held in the Details structure. */
static void
free_details(void)
{ int tag;

  for(tag = 0; tag < NUM_TAGS; tag++){
      if(Details[tag] != NULL){
	  free(Details[tag]);
	  Details[tag] = NULL;
      }
  }
}

	/* Discard data from a gathered game. */
static void
free_string_list(StringList *list)
{   StringList *next;

    while(list != NULL){
	next = list;
	list = list->next;
	if(next->str != NULL){
	    (void) free((void *)next->str);
	}
	(void) free((void *)next);
    }
}

static void
free_variation(Variation *variation)
{   Variation *next;

    while(variation != NULL){
	next = variation;
	variation = variation->next;
	if(next->moves != NULL){
	    (void) free_move_list((void *)next->moves);
	}
	(void) free((void *)next);
    }
}

void
free_move_list(Move *move_list)
{   Move *next;

    while(move_list != NULL){
	next = move_list;
	move_list = move_list->next;
	if(next->Nags != NULL){
	    free_string_list(next->Nags);
	}
	if(next->Comment != NULL){
	    free_string_list(next->Comment);
	}
	if(next->Variants != NULL){
	    free_variation(next->Variants);
	}
	if(next->terminating_result != NULL){
	    (void) free((void *)next->terminating_result);
	}
	(void) free((void *)next);
    }
}

	/* Add str onto the tail of list and
	 * return the head of the resulting list.
	 */
StringList *
SaveStringListItem(StringList *list,const char *str)
{
    if(str != NULL){
      unsigned len = strlen(str);
      StringList *new_item;

      new_item = (StringList *)MallocOrDie(sizeof(*new_item));
      new_item->str = (char *)MallocOrDie(len+1);
      strcpy(new_item->str,str);
      new_item->next = NULL;
      if(list == NULL){
	  list = new_item;
      }
      else{
	  StringList *tail = list;

	  while(tail->next != NULL){
	      tail = tail->next;
	  }
	  tail->next = new_item;
      }
    }
    return list;
}

	/* Print a usage message, and exit. */
static void
usage(const char *progname)
{
    fprintf(GlobalState.logfile,
	"extract -- a Portable Game Notation (PGN) extractor.\n");
    fprintf(GlobalState.logfile, "Usage: %s [arguments] [file.pgn ...]\n",progname);
    fprintf(GlobalState.logfile, "Arguments:\n");
    fprintf(GlobalState.logfile,
	"  -aoutputfile -- the file to which extracted games are to be appended.\n");
    fprintf(GlobalState.logfile,
	"        See -o flag for overwriting an existing file.\n");
    fprintf(GlobalState.logfile,
	"  -C -- don't include comments in the output. Ordinarily these are retained.\n");
    fprintf(GlobalState.logfile,
	"  -dduplicates -- write duplicate extracted games to the file duplicates.\n");
    fprintf(GlobalState.logfile,
	"  -D -- don't output duplicate extracted game scores.\n");
    fprintf(GlobalState.logfile,
	"  -ffile_list  -- file_list contains the list of PGN files to be\n");
    fprintf(GlobalState.logfile,
	"        searched - one per line. If this argument is used, there\n");
    fprintf(GlobalState.logfile,
	"        should be no additional file arguments on the command line.\n");
    fprintf(GlobalState.logfile, "  -h -- print this message.\n");
    fprintf(GlobalState.logfile, "  -? -- print this message.\n");
    fprintf(GlobalState.logfile,
	"  -llogfile  -- Write all diagnostics to logfile rather than stderr.\n");
    fprintf(GlobalState.logfile,
	"  -N -- don't include NAGs in the output. Ordinarily these are retained.\n");
    fprintf(GlobalState.logfile,
	"  -ooutputfile -- the file to which extracted games are to be written.\n");
    fprintf(GlobalState.logfile,
	"        Any existing contents of outputfile are lost (see -a flag).\n");
    fprintf(GlobalState.logfile, "  -p -- match permutations of the textual variations (-v).\n");
    fprintf(GlobalState.logfile, "  -r -- report any errors but don't extract.\n");
    fprintf(GlobalState.logfile,
	"  -s -- silent mode: don't report each game as it is extracted.\n");
    fprintf(GlobalState.logfile,
	"        Errors and empty games are still reported.\n");
    fprintf(GlobalState.logfile,
	"  -S -- Use a simple soundex algorithm for some tag matches. If used\n");
    fprintf(GlobalState.logfile,
	"        this option must precede the -t or -T options.\n");
    fprintf(GlobalState.logfile,
	"  -ttagfile -- file of tag extraction criteria.\n");
    fprintf(GlobalState.logfile,
	"  -Tcriterion -- player, date, or result extraction criterion.\n");
    fprintf(GlobalState.logfile,
	"  -vvariations -- the file variations contains the textual lines of interest.\n");
    fprintf(GlobalState.logfile,
	"  -V -- don't include variations in the output. Ordinarily these are retained.\n");
    fprintf(GlobalState.logfile,
	"  -wwidth -- set width as an approximate line width for output (default %u).\n",
			    MAX_LINE_LENGTH);
    fprintf(GlobalState.logfile,
	"  -xvariations -- the file variations contains the lines resulting in\n");
    fprintf(GlobalState.logfile,
	"        positions of interest.\n");
    exit(1);
}

int
main(int argc, char *argv[])
{   int argnum;
    /* Whether we have met the -f flag or not. */
    Boolean using_file_list = FALSE;
    /* Keep track of the list of PGN files.  These will either be the
     * remaining arguments once flags have been dealt with, or
     * those read from a -f argument.
     */
    char **file_list;

    /* Prepare the hash tables for transposition detection. */
    init_hashtab();
    /* Allow for some arguments. */
    for(argnum = 1; (argnum < argc) && (argv[argnum][0] == '-'); argnum++){
	switch(argv[argnum][1]){
	    case '7':
		GlobalState.seven_tag_roster = TRUE;
		break;
	    case 'a':
		{ /* We expect an output file. */
		  char *filename = &(argv[argnum][2]);

		  if(*filename != '\0'){
		      GlobalState.outputfile = fopen(filename,"a");
		      if(GlobalState.outputfile == NULL){
			  fprintf(GlobalState.logfile,
					"Unable to open %s for writing.\n",filename);
			  exit(1);
		      }
		  }
		  else{
		      fprintf(GlobalState.logfile,"usage: %sfilename.\n",argv[argnum]);
		      exit(1);
		  }
		}
		break;
	    case 'C':
		GlobalState.keep_comments = FALSE;
		break;
	    case 'd':
		{ /* We expect an output file for duplicate games. */
		  char *filename = &(argv[argnum][2]);
		  Boolean Ok = FALSE;

		  if(*filename != '\0'){
		      if(!GlobalState.suppress_duplicates){
			  GlobalState.duplicate_file = fopen(filename,"w");
			  if(GlobalState.duplicate_file != NULL){
			      Ok = TRUE;
			  }
			  else{
			      fprintf(GlobalState.logfile,
					    "Unable to open %s for writing.\n",filename);
			  }
			}
			else{
			    fprintf(GlobalState.logfile,
				    "%s clashes with -D flag.\n",argv[argnum]);
			}
		  }
		  else{
		      fprintf(GlobalState.logfile,"usage: %sfilename.\n",argv[argnum]);
		  }
		  if(!Ok){
		      exit(1);
		  }
		}
		break;
	    case 'D':
		/* Make sure that this doesn't clash with -d. */
		if(GlobalState.duplicate_file == NULL){
		    GlobalState.suppress_duplicates = TRUE;
		}
		else{
		    fprintf(GlobalState.logfile,
				"%s clashes with -d flag.\n",argv[argnum]);
		    exit(1);
		}
		break;
	    case 'f':
		{ /* We expect a list of files. */
		  char *filename = &(argv[argnum][2]);

		  if(*filename != '\0'){
		      FILE *fp;

		      fp = fopen(filename,"r");
		      if(fp != NULL){
			  file_list = compose_file_list(fp);
			  (void) fclose(fp);
			  if((file_list != NULL) && (file_list[0] != NULL)){
			      using_file_list = TRUE;
			  }
			  else{
			      fprintf(GlobalState.logfile,
					"Unable to read the list of files.\n");
			  }
		      }
		      else{
			  fprintf(GlobalState.logfile,"Unable to open %s for reading.\n",
				filename);
			  exit(1);
		      }
		  }
		}
		break;
	    case 'h':
	    case '?':
		usage(argv[0]);
		break;
	    case 'l':
		/* Take precautions against multiple log files. */
		if((GlobalState.logfile != stderr) && (GlobalState.logfile != NULL)){
		    (void) fclose(GlobalState.logfile);
		}
		GlobalState.logfile = fopen(&(argv[argnum][2]),"w");
		if(GlobalState.logfile == NULL){
		    fprintf(stderr,"Unable to open %s for writing.\n",
				&(argv[argnum][2]));
		    GlobalState.logfile = stderr;
		}
		break;
	    case 'N':
		GlobalState.keep_NAGs = FALSE;
		break;
	    case 'o':
		{ /* We expect an output file. */
		  char *filename = &(argv[argnum][2]);

		  if(*filename != '\0'){
		      GlobalState.outputfile = fopen(filename,"w");
		      if(GlobalState.outputfile == NULL){
			  fprintf(GlobalState.logfile,
					"Unable to open %s for writing.\n",filename);
			  exit(1);
		      }
		  }
		  else{
		      fprintf(GlobalState.logfile,"usage: %sfilename.\n",argv[argnum]);
		      exit(1);
		  }
		}
		break;
	    case 'p':
		GlobalState.match_permutations = TRUE;
		break;
	    case 'r':
		/* Report errors, but don't convert. */
		GlobalState.check_only = TRUE;
		break;
	    case 's':
		/* Silent. */
		GlobalState.verbose = FALSE;
		break;
	    case 'S':
		GlobalState.use_soundex = TRUE;
		break;
	    case 't':
		{   /* Allow a file to hold Tag strings to be used as
		     * extraction criteria.
		     */
	            char *TagFile;

		    TagFile =  &(argv[argnum][2]);
		    if(*TagFile != '\0'){
			read_tag_file(TagFile);
		    }
		}
		break;
	    case 'T':
		/* A single extraction criterion. */
		extract_argument(&(argv[argnum][2]));
		break;
	    case 'v':
		{ /* We expect a variation file. */
		  char *filename = &(argv[argnum][2]);
		  /* Where the list of variations of interest are kept. */
		  FILE *variation_file;

		  if(*filename != '\0'){
		      variation_file = fopen(filename,"r");
		      if(variation_file != NULL){
			  /* We wish to search for particular variations. */
			  init_textual_variations(variation_file);
			  fclose(variation_file);
		      }
		      else{
			  fprintf(GlobalState.logfile,
					"Unable to open %s for reading.\n",filename);
			  exit(1);
		      }
		  }
		}
		break;
	    case 'V':
		GlobalState.keep_variations = FALSE;
		break;
	    case 'w':
		{ unsigned length;

		  if(sscanf(argv[argnum],"-w%u",&length) > 0){
		      GlobalState.max_line_length = length;
		  }
		  else{
		      fprintf(GlobalState.logfile,"Unable to set a new line length.\n");
		  }
		}
		break;
	    case 'x':
		{ /* We expect a variation file. */
		  char *filename = &(argv[argnum][2]);

		  if(*filename != '\0'){
		      FILE *variation_file;

		      variation_file = fopen(filename,"r");
		      if(variation_file != NULL){
			  /* We wish to search for positional variations. */
			  init_positional_variations(variation_file);
			  fclose(variation_file);
			  GlobalState.positional_variations = TRUE;
		      }
		      else{
			  fprintf(GlobalState.logfile,"Unable to open %s for reading.\n",
						filename);
			  exit(1);
		      }
		  }
		}
		break;
	    default:
		usage(argv[0]);
		break;
	}
    }
    if(using_file_list){
	if(argnum != argc){
	    fprintf(GlobalState.logfile,"Superfluous arguments on the command line.\n");
	    usage(argv[0]);
	}
    }
    else if(argnum == argc){
	/* Missing filenames. */
	fprintf(GlobalState.logfile,"You must provide a list of PGN files to be read.\n");
	usage(argv[0]);
    }
    else{
	/* Pick up the remaining arguments as the files to be processed. */
	file_list = &argv[argnum];
    }
    /* Register the list of files so that yywrap has access to them. */
    register_file_list(file_list);
    /* Open up the first file to act as Lex's source of input. */
    if(!open_input(file_list[0])){
	fprintf(GlobalState.logfile,"Unable to open the PGN file: %s\n",file_list[0]);
	exit(1);
    }
    setup_for_new_game();
    yyparse();
    if((GlobalState.logfile != stderr) && (GlobalState.logfile != NULL)){
	(void) fclose(GlobalState.logfile);
    }
    return 0;
}

static void
setup_for_new_game(void)
{
    restart_lex_for_new_game();
    line_length = 0;
}
