%{
/*
 *  Program: extract -- a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@ukc.ac.uk
 *
 */

#include <stdlib.h>
#include <string.h>
#include "bool.h"
#include "defs.h"
#include "typedef.h"
/* Gain access the YACC's token values. */
#include "pgn_tab.h"

#include "protos.h"
#include "taglist.h"
#include "lists.h"

	/* Prototypes for the functions in this file. */
static void save_tag_string(const char *tag);
static void save_string(const char *result);
/* When a move is saved, what is known of its source and destination coordinates
 * should also be saved.
 */
static void save_move(const char *move);
static void save_q_castle(void);
static void save_k_castle(void);
static void terminate_input(void);

static unsigned line_number = 1;
/* Keep track of the Recursive Annotation Variation level. */
static unsigned RAV_level = 0;
/* Keep track of the last move found. */
static char last_move[MAX_MOVE_LEN+1];
/* Keep track of the text of the current comment. */
StringList *current_comment = NULL;
/* How many games we have extracted from this file. */
static unsigned games_in_file = 0;

#ifdef yywrap
#undef yywrap
#endif

%}
%s Tag Commentary
%%
"["		    { BEGIN Tag;
		    }
<Tag>"]"	    { BEGIN 0;
		    }
<Tag>Event	    { yylval.tag_index = EVENT_TAG;
		      return TAG;
		    }
<Tag>Site	    { yylval.tag_index = SITE_TAG;
		      return TAG;
		    }
<Tag>Date	    { yylval.tag_index = DATE_TAG;
		      return TAG;
		    }
<Tag>White	    { yylval.tag_index = WHITE_TAG;
		      return TAG;
		    }
<Tag>Black	    { yylval.tag_index = BLACK_TAG;
		      return TAG;
		    }
<Tag>Result         { yylval.tag_index = RESULT_TAG;
		      return TAG;
		    }
<Tag>Round	    { yylval.tag_index = ROUND_TAG;
		      return TAG;
		    }
<Tag>ECO            { yylval.tag_index = ECO_TAG;
		      return TAG;
                    }
<Tag>WhiteTitle     { yylval.tag_index = WHITE_TITLE_TAG;
		      return TAG;
		    }
<Tag>BlackTitle     { yylval.tag_index = BLACK_TITLE_TAG;
		      return TAG;
		    }
<Tag>WhiteElo       { yylval.tag_index = WHITE_ELO_TAG;
		      return TAG;
		    }
<Tag>BlackElo       { yylval.tag_index = BLACK_ELO_TAG;
		      return TAG;
		    }
<Tag>WhiteUSCF      { yylval.tag_index = WHITE_USCF_TAG;
		      return TAG;
		    }
<Tag>BlackUSCF      { yylval.tag_index = BLACK_USCF_TAG;
		      return TAG;
		    }
<Tag>WhiteNA        { yylval.tag_index = WHITE_NA_TAG;
		      return TAG;
		    }
<Tag>BlackNA        { yylval.tag_index = BLACK_NA_TAG;
		      return TAG;
		    }
<Tag>WhiteType      { yylval.tag_index = WHITE_TYPE_TAG;
		      return TAG;
		    }
<Tag>BlackType      { yylval.tag_index = BLACK_TYPE_TAG;
		      return TAG;
		    }
<Tag>Annotator      { yylval.tag_index = ANNOTATOR_TAG;
		      return TAG;
		    }
<Tag>Player         { /* Note: this is not a proper PGN tag.
		       * It is used with the -t flag to specify a
		       * White or Black player to be searched for.
		       */
		      yylval.tag_index = PSEUDO_PLAYER_TAG;
		      return TAG;
		    }
<Tag>[A-Za-z0-9_]+  { /* Pick up any other general tags. */
		      save_string(yytext);
		      return MISC_TAG;
		    }
<Tag>"\""[^"\n]*"\""	{ /* Save the string for this tag. */
			  save_tag_string(yytext);
			  return STRING;
			}
<Tag>"\""[^"\n]*\n	{ /* A non-terminated string. */
			  fprintf(GlobalState.logfile,"Improperly terminated tag: %s",yytext);
			  /* Save the string for this tag. */
			  save_tag_string(yytext);
			  line_number++;
			  /* Terminate the <Tag> state. */
			  BEGIN 0;
			  return STRING;
			}
"{"		{ /* Save comments. */
		  BEGIN Commentary;
		  current_comment = NULL;
		}
<Commentary>"}"	{ /* End of the comment. */
		  /* Return a fake Move * structure holding the comment text. */
		  Move *dummy_move = new_move_structure();

		  /* Use a NULL move to signify a comment. */
		  dummy_move->move[0] = '\0';
		  dummy_move->Comment = current_comment;
		  yylval.move_details = dummy_move;
		  current_comment = NULL;
		  BEGIN 0;
		  return COMMENT;
		}
<Commentary>[^}\n]+	{ /* Save the text of this comment portion. */
			  current_comment = SaveStringListItem(current_comment,yytext);
			}
<Commentary>\n		{ /* End of line in a comment. */
			  line_number++;
			}
"("		{ RAV_level++;
		  return RAV_START;
		}
")"		{ if(RAV_level > 0){
		     RAV_level--;
		  }
		  else{
		     fprintf(GlobalState.logfile,
			"Internal error: Failed to close an RAV.\n");
		  }
		  return RAV_END;
		}
;[^\n]*			{ /* Strip line comments. */
			}
("+""+"?)|"#"	{ return CHECK;
		}
[KQRNB][a-h]?[1-8]?x[a-h][1-8]	{ 
	      /* A capture. */
	      save_move(yytext);
	      return MOVE;
	    }
[KQRNB][a-h]?[1-8]?[a-h][1-8]	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h]x[a-h][1-8](ep)?	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h]x[a-h](ep)?	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][1-8][a-h][1-8](ep)?	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][a-h][1-8](ep)?	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][a-h](ep)?	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][1-8]	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h]x[a-h][18](=?[QRNB])	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h]x[a-h](=?[QRNB])	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][a-h][18](=?[QRNB])	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][a-h](=?[QRNB])	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[a-h][18](=?[QRNB])	{ 
	      save_move(yytext);
	      return MOVE;
	    }
[O0o]-[O0o]	{ 
		  save_k_castle();
		  return MOVE;
		}
[O0o]-[O0o]-[O0o]	{ 
			  save_q_castle();
			  return MOVE;
			}
!!+		{ save_string("$3");
		  return NAG;
		}
"!?"[!?]*		{ save_string("$5");
			  return NAG;
			}
!		{ save_string("$1");
		  return NAG;
		}
"??"+		{ save_string("$4");
		  return NAG;
		}
"?!"[!?]*	{ save_string("$6");
		  return NAG;
		}
"?"		{ save_string("$2");
		  return NAG;
		}
"$"[0-9]+		{ /* Pickup a NAG */
			  save_string(yytext);
			  return NAG;
			}
"1-0"|"0-1"|"1/2-1/2"|"*"	{
				    save_string(yytext);
				    return TERMINATING_RESULT;
				}
[0-9]+"."*		{ yylval.move_number = 0;
			  (void) sscanf(yytext,"%u",&yylval.move_number);
			  return MOVE_NUMBER;
			}
^"%"[^\n]*		{ /* Strip escape mechanism. */
			}
[ \t]+		{ /* Ignore whitespace. */
		}
\n		{ line_number++;
		}
.	{ /* We missed matching a character.  Tell the reader what
	   * it was, and the context in which it occurred.
	   */
	  print_error_context(GlobalState.logfile);
	  fprintf(GlobalState.logfile,"Missed character <%s>",yytext);
	  if(last_move[0] != '\0'){
	      fprintf(GlobalState.logfile," context <%s>",last_move);
	  }
	  putc('\n',GlobalState.logfile);
	  report_details(GlobalState.logfile);
	}
%%

	/* Save castling moves in a standard way. */
static void
save_q_castle(void)
{
    save_move("O-O-O");
}

	/* Save castling moves in a standard way. */
static void
save_k_castle(void)
{
    save_move("O-O");
}

	/* Make a copy of the matched text of the move. */
static void
save_move(const char *move)
{
    /* Decode the move into its components. */
    yylval.move_details = decode_move(move);
    /* Remember the last move. */
    strcpy(last_move,move);
}

void
restart_lex_for_new_game(void)
{
    *last_move = '\0';
    current_comment = NULL;
    BEGIN 0;
}

	/* Make it possible to read multiple input files.
	 * This is done by having main() call register_file_list
	 * with what is left of argv[] once the arguments have
	 * been processed.
	 */
static int next_file_num = 0;
static char **file_list = NULL;

void
register_file_list(char *files_to_be_read[])
{
    file_list = files_to_be_read;
}

	/* Use infile as the input source.
	 * Return 1 on a successful open, 0 otherwise.
	 */
int
open_input(char *infile)
{   extern FILE *yyin;

    yyin = fopen(infile,"r");
    if(yyin != NULL){
	GlobalState.current_input_file = infile;
	if(GlobalState.verbose){
	    fprintf(GlobalState.logfile,"Processing %s\n",infile);
	}
    }
    return yyin != NULL;
}

void
print_error_context(FILE *fp)
{
    if(GlobalState.current_input_file != NULL){
        fprintf(fp,"File %s: ",GlobalState.current_input_file);
    }
    fprintf(fp,"Line number: %u\n",line_number);
}

	/* Standard YACC function. */
void
yyerror(const char *s)
{
    print_error_context(GlobalState.logfile);
    fprintf(GlobalState.logfile, " a syntactic error was found.\n");
    report_details(GlobalState.logfile);
    if(*last_move != '\0'){
	fprintf(GlobalState.logfile,"The context of the last matched move was <%s>\n",
			last_move);
    }
}


	/* Save the given tag accessible to YACC.
	 * Strip off any surrounding quotes in the process.
	 */
static void
save_tag_string(const char *str)
{   const int len = strlen(str);
    char *token;

    /* Strip any quotes. */
    if(len > 2){
	if(*str == '"'){
	    token = MallocOrDie(len+1);
	    /* Skip the opening quote. */
	    strcpy(token,&str[1]);
	    /* Remove the closing quote. */
	    token[len-2] = '\0';
	    yylval.token_string = token;
	}
	else{
	    token = MallocOrDie(len+1);
	    strcpy(token,str);
	    yylval.token_string = token;
	}
    }
    else{
	yylval.token_string = NULL;
    }
}

	/* Save the given str accessible to YACC. */
static void
save_string(const char *str)
{   const int len = strlen(str);
    char *token;

    token = MallocOrDie(len+1);
    strcpy(token,str);
    yylval.token_string = token;
}

	/* Standard Lex function. */
int
yywrap(void)
{   int time_to_exit;

    /* Beware of this being called in inappropriate circumstances. */
    if(file_list == NULL){
	/* There are no files. */
	time_to_exit = 1;
    }
    else if(file_list[next_file_num] == NULL){
	/* There was no last file! */
	time_to_exit = 1;
    }
    else{
	/* Close the input files.  */
	terminate_input();
	/* See if there is another. */
	next_file_num++;
	if(file_list[next_file_num] == NULL){
	    /* We have processed the last file. */
	    time_to_exit = 1;
	}
	else if(!open_input(file_list[next_file_num])){
	    fprintf(GlobalState.logfile,"Unable to open the PGN file: %s\n",
					file_list[next_file_num]);
	    time_to_exit = 1;
	}
	else{
	    /* Ok, we opened it. */
	    time_to_exit = 0;
	    /* Set everything up for a new file. */
	    restart_lex_for_new_game();
	    games_in_file = 0;
	    line_number = 1;
	}
    }
    return time_to_exit;
}

static void
terminate_input(void)
{
    if((yyin != stdin) && (yyin != NULL)){
	(void) fclose(yyin);
	yyin = NULL;
    }
}

	/* Read the list of extraction criteria from TagFile.
	 * This uses the normal lexical analyser before the
	 * PGN files are processed.
	 * Be careful to leave lex in the right state.
	 */
void
read_tag_file(const char *TagFile)
{   
    yyin = fopen(TagFile,"r");
    if(yyin != NULL){
	int tag;
	
	BEGIN Tag;
	tag = yylex();

	/* Keep going until we hit a non tag. */
	while(tag == TAG){
	    int tag_index = yylval.tag_index;

	    if(yylex() == STRING){
		add_tag_to_list(tag_index,yylval.token_string);
		(void) free((void *)yylval.token_string);
	    }
	    else{
		fprintf(stderr,"File %s: ",TagFile);
		fprintf(stderr,"Expected a tag string.\n");
	    }
	    BEGIN Tag;
	    tag = yylex();
	}
	(void) fclose(yyin);
	yyin = stdin;
	restart_lex_for_new_game();
	line_number = 1;
    }
    else{
	fprintf(stderr,"Unable to open %s for reading.\n",TagFile);
	exit(1);
    }
}
void *
MallocOrDie(unsigned nbytes)
{   void *result;

    result = malloc(nbytes);
    if(result == NULL){
	perror("");
	abort();
    }
    return result;
}
