/*
 *  Program: extract -- a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@ukc.ac.uk
 *
 */

	/***
	 ***
	 * These routines are concerned with gathering moves of
	 * the various sorts of variations specified by the -v
	 * and -x flags.  In the former case, there are also functions
	 * for checking the moves of a game against the variation
	 * lists that are wanted.  Checking of the variations specified
	 * by the -x flag is handled elsewhere by apply_move_list().
	 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "bool.h"
#include "defs.h"
#include "typedef.h"
#include "protos.h"
#include "map.h"
#include "lists.h"

/* Define a character that can be used in the variations file to
 * mean that we don't mind what move was played at this point.
 * So: 
 *	* b6
 * means look for all games in which Black playes 1...b6, regardless
 * of White's first move.
 */
#define ANY_MOVE '*'
/* Define a character that can be used in the variations file to
 * mean that we do not wish to match a particular move.
 * So:
 *	e4 c5 !Nf3
 * means look for games in which Black does not play 2. Nf3 against
 * the Sicilian defence.
 */
#define DISALLOWED_MOVE '!'

    /* Hold details of a single move within a variation. */
typedef struct{
    /* Characters of the move.
     * Alternative notations for the same move are separated by
     * a non-move character, e.g.:
     *	cxd|cxd4|c5xd4
     * could all be alternatives for the same basic pawn capture.
     */
    char *move;
    /* If we are interested in matching the moves in any order,
     * then we need to record whether or not the current move has
     * been matched or not.
     */
    Boolean matched;
} variant_move;

    /* Hold details of a single variation, with a pointer to
     * an alternative variation.
     */
typedef struct variation_list {
    /* The list of moves. */
    variant_move *moves;
    /* Keep a count of how many ANY_MOVE moves there are in the move
     * list for each colour.  If these are non-zero then one is used
     * when a match fails when looking for permutations.
     */
    unsigned num_white_any_moves;
    unsigned num_black_any_moves;
    /* Keep a count of how many DISALLOWED_MOVE moves there are in the move
     * list for each colour.  If these are non-zero then the game
     * must be searched for them when looking for permutations before
     * the full search is made.
     */
    unsigned num_white_disallowed_moves;
    unsigned num_black_disallowed_moves;
    /* How many half-moves in the variation? */
    unsigned length;
    struct variation_list *next;
} variation_list;

    /* The head of the variations-of-interest list. */
static variation_list *games_to_keep = NULL;

static Boolean non_blank_line(const char *line);

	/*** Functions concerned with reading details of the variations
	 *** of interest.
	 ***/


    /* Read a single line of input. */
#define INIT_LINE_LENGTH 80
#define LINE_INCREMENT 20

static char *
read_line(FILE *fpin)
{   char *line = NULL;
    unsigned len = 0;
    unsigned max_length;
    int ch;

     ch = getc(fpin);
     if(ch != EOF){
	line = (char *) malloc(INIT_LINE_LENGTH + 1);
	if(line != NULL){
	    max_length = INIT_LINE_LENGTH;
	    while((ch != '\n') && (ch != EOF)){
		/* Another character to add. */
		if(len == max_length){
		     line = (char *)realloc((void *)line,max_length+LINE_INCREMENT+1);
		     if(line == NULL){
			 return NULL;
		     }
		     max_length += LINE_INCREMENT;
		}
		line[len] = ch;
		len++;
		ch = getc(fpin);
	    }
	    line[len] = '\0';
	}
     }
     return line;
}

/* Define values for malloc/realloc allocation. */
#define INIT_MOVE_NUMBER 10
#define MOVE_INCREMENT 5

	/* Break up a single line of moves into a list of moves
	 * comprising a variation.
	 */
static variation_list *
compose_variation(char *line)
{   variation_list *variation;
    variant_move *move_list;
    /* Keep track of the number of moves extracted from line. */
    unsigned num_moves = 0;
    unsigned max_moves = 0;
    char *move;

    variation = (variation_list *)malloc(sizeof(variation_list));
    if(variation != NULL){
	/* We don't yet know how many ANY_MOVEs or DISALLOWED_MOVES there
	 * will be in this variation.
	 */
        variation->num_white_any_moves = 0;
        variation->num_black_any_moves = 0;
        variation->num_white_disallowed_moves = 0;
        variation->num_black_disallowed_moves = 0;
	/* Allocate an initial number of pointers for the moves of the variation. */
	move_list = (variant_move *) malloc(INIT_MOVE_NUMBER * sizeof(*move_list));
	if(move_list != NULL){
	    max_moves = INIT_MOVE_NUMBER;
	    /* Find the first move. */
	    move = strtok(line," ");
	    while(move != NULL){
		/* See if we need some more space. */
		if(num_moves == max_moves){
		     move_list = (variant_move *)realloc((void *)move_list,
				(max_moves+MOVE_INCREMENT) * sizeof(*move_list));
		     if(move_list == NULL){
			 /* Catastrophic failure. */
			 free((void *)variation);
			 return NULL;
		     }
		     else{
			 max_moves += MOVE_INCREMENT;
		     }
		}
		/* Keep the move and initialise the matched field for
		 * when we start matching games.
		 */
		move_list[num_moves].move = move;
		move_list[num_moves].matched = FALSE;
		/* Keep track of moves that will match anything. */
		if(*move == ANY_MOVE){
		    /* Odd numbered half-moves in the variant list are Black. */
		    if(num_moves & 0x01){
			variation->num_black_any_moves++;
		    }
		    else{
			variation->num_white_any_moves++;
		    }
		    /* Beware of the potential for false matches. */
		    if(strlen(move) > 1){
			fprintf(stderr,
			    "Warning: %c in %s should not be followed by additional move text.\n",
				*move,move);
			fprintf(stderr,"It could give false matches.\n");
		    }
		}
		else if(*move == DISALLOWED_MOVE){
		    /* Odd numbered half-moves in the variant list are Black. */
		    if(num_moves & 0x01){
			variation->num_black_disallowed_moves++;
		    }
		    else{
			variation->num_white_disallowed_moves++;
		    }
		}
		else{
		    /* Unadorned move. */
		}
		num_moves++;
		move = strtok((char *)NULL," ");
	    }
	    variation->moves = move_list;
	    variation->length = num_moves;
	    variation->next = NULL;
	}
	else{
	    free((void *)variation);
	    variation = NULL;
	}
    }
    return variation;
}

	/* Read each line of input and decompose it into a variation
	 * to be placed in the games_to_keep list.
	 */
void
init_textual_variations(FILE *fpin)
{   char *line;
    variation_list *next_variation;

    games_to_keep = NULL;
    while((line = read_line(fpin)) != NULL){
	if(non_blank_line(line)){
	    next_variation = compose_variation(line);
	    if(next_variation != NULL){
		next_variation->next = games_to_keep;
		games_to_keep = next_variation;
	    }
	}
    }
}

	/*** Functions concerned with reading details of the positional variations
	 *** of interest.
	 ***/

	/* Break up a single line of moves into a list of moves
	 * comprising a positional variation.
	 */
static Move *
compose_positional_variation(char *line)
{   char *move;
    /* Build a linked list of the moves of the variation. */
    Move *head = NULL, *tail = NULL;
    Boolean Ok = TRUE;

    move = strtok(line," ");
    while(Ok && (move != NULL) && (*move != '*')){
	Move *next = decode_move(move);

	if(next == NULL){
	    fprintf(GlobalState.logfile,"Failed to identify %s\n",move);
	    Ok = FALSE;
	}
	else{
	    /* Chain it on to the list. */
	    if(tail == NULL){
		head = next;
		tail = next;
	    }
	    else{
		tail->next = next;
		tail = next;
	    }
	    next->next = NULL;
	}
	move = strtok(NULL," ");
    }
    if(Ok){
	/* Generate a hash code for the variation. */
    }
    else{
	if(head != NULL){
	    free_move_list(head);
	}
	head = NULL;
    }
    return head;
}

	/* Read each line of input and decompose it into a positional variation
	 * to be placed in the list of required hash values.
	 */
void
init_positional_variations(FILE *fpin)
{   char *line;
    Move *next_variation;

    while((line = read_line(fpin)) != NULL){
	if(non_blank_line(line)){
	    next_variation = compose_positional_variation(line);
	    if(next_variation != NULL){
		store_hash_value(next_variation);
		free_move_list(next_variation);
	    }
	}
    }
}

	/*** Functions concerned with matching the moves of the current game
	 *** against the variations of interest.
	 ***/

	/* Do the moves of the current game match the given variation?
	 * Go for a straight 1-1 match in the ordering, without considering
	 * permutations.
	 * Return TRUE if so, FALSE otherwise.
	 */
static Boolean
straight_match(Move *current_game_head, variation_list variation)
{
    variant_move *moves_of_the_variation;
    /* Which is the next move that we wish to match. */
    Move *next_move;
    unsigned move_index = 0;
    /* Assume that it matches. */
    Boolean matches = TRUE;

    /* Access the head of the current game. */
    next_move = current_game_head;
    /* Go for a straight move-by-move match in the order in which
     * the variation is listed.
     * Point at the head of the moves list.
     */
    moves_of_the_variation = variation.moves;
    move_index = 0;
    while(matches && (next_move != NULL) && (move_index < variation.length)){
	Boolean this_move_matches;

	if(*(moves_of_the_variation[move_index].move) == ANY_MOVE){
	    /* Still matching as we don't care what the actual move is. */
	}
	else{
	    this_move_matches = strstr(moves_of_the_variation[move_index].move,
					    next_move->move) != NULL;
	    if(this_move_matches){
		/* We found a match, check that it isn't disallowed. */
		if(*moves_of_the_variation[move_index].move == DISALLOWED_MOVE){
		    /* This move is disallowed and implies failure. */
		    matches = FALSE;
		}
	    }
	    else{
		if(*moves_of_the_variation[move_index].move != DISALLOWED_MOVE){
		    /* No match found for this move. */
		    matches = FALSE;
		}
		else{
		    /* This is ok, because we didn't want a match. */
		}
	    }
		
	}
	/* If we are still matching, go on to the next move. */
	if(matches){
	    move_index++;
	    next_move = next_move->next;
	}
    }
    /* The game could be shorter than the variation, so don't rely
     * on the fact that matches is still true.
     */
    matches = (move_index == variation.length);
    return matches;
}

    /* Do the moves of the current game match the given variation?
     * Try all possible orderings for the moves, within the
     * constraint of proper WHITE/BLACK moves.
     * The parameter variation is passed as a copy because we modify
     * the num_ fields within it.
     * Note that there is a possibility of a false match in this
     * function if a variant move is specified in a form such as:
     *		*|c4
     * This is because the num_ field is set from this on the basis of the
     * ANY_MOVE character at the start.  However, this could also match a
     * normal move with its c4 component.  If it is used for the latter
     * purpose then it should not count as an any_ move.  There is a warning
     * issued about this when variations are read in.
     * Return TRUE if we match, FALSE otherwise.
     *
     * The DISALLOWED_MOVE presents some problems with permutation matches
     * because an ANY_MOVE could match an otherwise disallowed move. The
     * approach that has been taken is to cause matching of a single disallowed
     * move to result in complete failure of the current match.
     */
static Boolean
permutation_match(Move *current_game_head, variation_list variation)
{   variant_move *moves_of_the_variation;
    /* Which is the next move that we wish to match? */
    Move *next_move;
    unsigned variant_index = 0;
    /* Assume that it matches. */
    Boolean matches = TRUE;
    /* How many moves have we matched?
     * When this reaches variation.length we have a full match.
     */
    unsigned matched_moves = 0;
    Boolean white_to_move = TRUE;

    moves_of_the_variation = variation.moves;
    /* Clear all of the matched fields of the variation. */
    for(variant_index = 0; variant_index < variation.length; variant_index++){
	moves_of_the_variation[variant_index].matched = FALSE;
    }
    /* Access the head of the current game. */
    next_move = current_game_head;

    /*** Stage One.
     * The first task is to ensure that there are no DISALLOWED_MOVEs in
     * the current game.
     */
    if((variation.num_white_disallowed_moves > 0) ||
	    (variation.num_black_disallowed_moves > 0)){
	unsigned tested_moves = 0;
	Boolean disallowed_move_found = FALSE;

	/* Keep going as long as we still have not found a diallowed move,
	 * we haven't matched the whole variation, and we haven't reached the end of
	 * the game.
	 */
	while((!disallowed_move_found) && (tested_moves < variation.length) &&
					(next_move != NULL)){
	    /* We want to see if next_move is a disallowed move of the variation. */
	    if(white_to_move){
		/* White; start with the first move. */
		variant_index = 0;
	    }
	    else{
		/* For a Black move, start at the second half-move in the list, if any. */
		variant_index = 1;
	    }
	    /* Try each move of the variation in turn, until a match is found. */
	    while((!disallowed_move_found) && (variant_index < variation.length)){
		if((*moves_of_the_variation[variant_index].move ==
			    DISALLOWED_MOVE) &&
		    (strstr(moves_of_the_variation[variant_index].move,
			    next_move->move) != NULL)){
		    /* Found one. */
		    disallowed_move_found = TRUE;
		}
		if(!disallowed_move_found){
		    /* Move on to the next available move -- 2 half moves along. */
		    variant_index += 2;
		}
	    }
	    if(!disallowed_move_found){
		/* Ok so far, so move on. */
		tested_moves++;
		white_to_move = !white_to_move;
		next_move = next_move->next;
	    }
	}
	if(disallowed_move_found){
	    /* This rules out the whole match. */
	    matches = FALSE;
	}
	else{
	    /* In effect, each DISALLOWED_MOVE now becomes an ANY_MOVE. */
	    for(variant_index = 0; variant_index < variation.length; variant_index++){
		if(*moves_of_the_variation[variant_index].move == DISALLOWED_MOVE){
		    moves_of_the_variation[variant_index].matched = TRUE;
		    if((variant_index & 1) == 0){
			variation.num_white_any_moves++;
		    }
		    else{
			variation.num_black_any_moves++;
		    }
		}
	    }
	}
    }

    /*** Stage Two.
     * Having eliminated moves which have been disallowed, try permutations
     * of the variation against the moves of the current game.
     */
    /* Access the head of the current game. */
    next_move = current_game_head;
    white_to_move = TRUE;
    /* Keep going as long as we still have matches, we haven't
     * matched the whole variation, and we haven't reached the end of
     * the game.
     */
    while(matches && (matched_moves < variation.length) && (next_move != NULL)){
	/* Assume failure. */
	matches = FALSE;
	/* We want to find next_move in an unmatched move of the variation. */
	if(white_to_move){
	    /* Start with the first move. */
	    variant_index = 0;
	}
	else{
	    /* For a Black move, start at the second half-move in the list, if any. */
	    variant_index = 1;
	}
	/* Try each move of the variation in turn, until a match is found. */
	while((!matches) && (variant_index < variation.length)){
	    if(moves_of_the_variation[variant_index].matched){
		/* We can't try this. */
	    }
	    else{
		Boolean this_move_matches = strstr(
					moves_of_the_variation[variant_index].move,
					next_move->move) != NULL;
		if(this_move_matches){
		    /* Found it. */
		    moves_of_the_variation[variant_index].matched = TRUE;
		    matches = TRUE;
		}
	    }
	    if(!matches){
		/* Move on to the next available move -- 2 half moves along. */
		variant_index += 2;
	    }
	}
	/* See if we made a straight match. */
	if(!matches){
	    /* See if we have some ANY_MOVEs available. */
	    if(white_to_move && (variation.num_white_any_moves > 0)){
		matches = TRUE;
		variation.num_white_any_moves--;
	    }
	    else if(!white_to_move && (variation.num_black_any_moves > 0)){
		matches = TRUE;
		variation.num_black_any_moves--;
	    }
	    else{
		/* No slack. */
	    }
	}
	/* We have tried everything, did we succeed? */
	if(matches){
	    /* Yes, so move on. */
	    matched_moves++;
	    next_move = next_move->next;
	    white_to_move = !white_to_move;
	}
    }
    if(matches){
	/* Ensure that we completed the variation. */
	matches = matched_moves == (variation.length);
    }
    return matches;
}

	/* Determine whether or not the current game is wanted.
	 * It will be if it matches one of the current variations
	 * and its tag details match those that we are interested in.
	 */
Boolean
check_textual_variations(Game game_details)
{   Boolean wanted = FALSE;
    variation_list *variation;

    if(games_to_keep != NULL){
	for(variation = games_to_keep; (variation != NULL) && !wanted;
		    variation = variation->next){
	    if(GlobalState.match_permutations){
		wanted = permutation_match(game_details.moves,*variation);
	    }
	    else{
		wanted = straight_match(game_details.moves,*variation);
	    }
	}
    }
    else{
        /* There are no variations, assume that selection is done
         * on the basis of the Details.
         */
        wanted = TRUE;
    }
    return wanted;
}

	/* Return TRUE if line contains a non-space character. */
static Boolean
non_blank_line(const char *line)
{   Boolean blank = TRUE;

    if(line != NULL){
	while(blank && (*line != '\0')){
	    if(!isspace(*line)){
		blank = FALSE;
	    }
	    else{
		line++;
	    }
	}
    }
    return !blank;
}

	/* Read a list of lines from fp.
	 * The list should have a (char *)NULL on the end.
	 */
char **
compose_file_list(FILE *fp)
{    char **file_list;
     char *line;
     /* Keep track of the number of lines. */
     unsigned num_lines = 0;
     unsigned max_lines = 0;

     /* Allocate an initial number of pointers for the lines. */
     file_list = (char **) malloc((INIT_LIST_SPACE+1) * sizeof(char *));
     if(file_list != NULL){
	 max_lines = INIT_LIST_SPACE;
	 /* Find the first line. */
         line = read_line(fp);
	 while(line != NULL){
	     if(non_blank_line(line)){
		 if(num_lines == max_lines){
		     file_list = (char **)realloc((void *)file_list,
				(max_lines+MORE_LIST_SPACE+1) * sizeof(char *));
		     if(file_list != NULL){
			 max_lines += MORE_LIST_SPACE;
		     }
		     else{
			 return NULL;
		     }
		 }
		 file_list[num_lines] = line;
		 num_lines++;
	     }
	     else{
	         (void) free((void *)line);
	     }
	     line = read_line(fp);
	 }
	 file_list[num_lines] = NULL;
     }
     return file_list;
}
