/*
 *  Program: extract -- a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@ukc.ac.uk
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "bool.h"
#include "defs.h"
#include "typedef.h"
#include "pgn_tab.h"
#include "protos.h"
#include "map.h"

static Boolean position_matches(HashCode current_hash_value);

	/* Structures to hold the x,y displacements of the various
	 * piece movements.
	 */

/* Define a list of possible Knight moves. */
#define NUM_KNIGHT_MOVES 8
static short Knight_moves[2*NUM_KNIGHT_MOVES] =
	{  1, 2,
	   1,-2,
	   2, 1,
	   2,-1,
	  -1, 2,
	  -1,-2,
	  -2, 1,
	  -2,-1,
	};

/* Define a list of possible Bishop moves. */
#define NUM_BISHOP_MOVES 4
static short Bishop_moves[2*NUM_BISHOP_MOVES] =
    {  1, 1,
       1,-1,
      -1, 1,
      -1,-1
    };

/* Define a list of possible Rook moves. */
#define NUM_ROOK_MOVES 4
static short Rook_moves[2*NUM_ROOK_MOVES] =
    {  1, 0,
       0, 1,
      -1, 0,
       0,-1
    };

/* Define a list of possible King moves. */
#define NUM_KING_MOVES 8
static short King_moves[2*NUM_KING_MOVES] =
    {  1, 0,
       1, 1,
       1,-1,
       0, 1,
       0,-1,
      -1, 0,
      -1, 1,
      -1,-1,
    };

/* Define a list of possible Queen moves. */
#define NUM_QUEEN_MOVES 8
static short Queen_moves[2*NUM_QUEEN_MOVES] =
    {  1, 0,
       1, 1,
       1,-1,
       0, 1,
       0,-1,
      -1, 0,
      -1, 1,
      -1,-1,
    };


	/* A table of hash values for square/piece/colour combinations. */
#define NUMBER_OF_PIECES 6
static HashCode HashTab[BOARDSIZE][BOARDSIZE][NUMBER_OF_PIECES][2];

	/* Code to allocate and free MovePair structures.  New moves are
	 * allocated from the move_pool, if it isn't empty.  Old moves
	 * are freed to this pool.
	 */
/* Keep a pool of free move structures. */
static MovePair *move_pool = NULL;

static MovePair *
malloc_move(void)
{   MovePair *move;

    if(move_pool != NULL){
	move = move_pool;
	move_pool = move_pool->next;
    }
    else{
	move = (MovePair *)MallocOrDie(sizeof(MovePair));
    }
    move->next = NULL;
    return move;
}

	/* Simply add the move to the free move pool. */
static void
free_move(MovePair *move)
{
    move->next = move_pool;
    move_pool = move;
}

	/* Free a whole list of moves to the move pool. */
static void
free_move_pair_list(MovePair *move_list)
{
    if(move_list != NULL){
	free_move_pair_list(move_list->next);
	free_move(move_list);
    }
}

	/* Look up the hash value for this combination. */
static HashCode
HashLookup(Col col, Rank rank, Piece piece, Colour colour)
{
    return HashTab[col-FIRSTCOL][rank-FIRSTRANK][piece-PAWN][colour-BLACK];
}

	/* Produce a hash value for each piece, square, colour combination.
	 * This code is a modified version of that to be found in
	 * Steven J. Edwards' SAN kit.
	 */
#define SHIFT_LENGTH 7

void
init_hashtab(void)
{   Piece piece;
    Colour colour;
    Rank rank;
    Col col;
    static HashCode seed = 0;

    for(col = FIRSTCOL; col <= LASTCOL; col++){
	for(rank = FIRSTRANK; rank <= LASTRANK; rank++){
	    for(piece = PAWN; piece <= KING; piece++){
		for(colour = BLACK; colour <= WHITE; colour++){
		    HashCode code;

		    seed = (seed * 1103515245L) + 12345L;
		    code = (seed >> SHIFT_LENGTH);
		    HashTab[col-FIRSTCOL][rank-FIRSTRANK][piece-PAWN][colour-BLACK] =
					code;
		}
	    }
	}
    }
}

	/* Print out the current occupant of the given square. */
static void
print_square(Col col, Rank rank, const Board *board,FILE *outfp)
{   short r = RankConvert(rank);
    short c = ColConvert(col);

    switch(board->board[r][c]){
	case W(PAWN): putc('P',outfp); break;
	case W(KNIGHT): putc('N',outfp); break;
	case W(BISHOP): putc('B',outfp); break;
	case W(ROOK): putc('R',outfp); break;
	case W(QUEEN): putc('Q',outfp); break;
	case W(KING): putc('K',outfp); break;
	case B(PAWN): putc('p',outfp); break;
	case B(KNIGHT): putc('n',outfp); break;
	case B(BISHOP): putc('b',outfp); break;
	case B(ROOK): putc('r',outfp); break;
	case B(QUEEN): putc('q',outfp); break;
	case B(KING): putc('k',outfp); break;
	case EMPTY: putc('.',outfp); break;
	case OFF: putc('?',outfp); break;
	default:
	    fprintf(GlobalState.logfile,
		"Attempt to print illegal square %c,%c in print_square.\n",
			col,rank);
	    break;
    }
}

	/* Print out the contents of the given board. */
static void
print_board(const Board *board,FILE *outfp)
{  Rank rank;
   Col col;

   for(rank = LASTRANK; rank >= FIRSTRANK; rank--){
       for(col = FIRSTCOL; col <= LASTCOL; col++){
	   print_square(col,rank,board,outfp);
       }
       putc('\n',outfp);
   }
   putc('\n',outfp);
}

	/* Is the given piece of the named colour? */
static Boolean
piece_is_colour(Piece coloured_piece, Colour colour)
{   
    return EXTRACT_COLOUR(coloured_piece) == colour;
}

	/* Make the given move.  This is assumed to have been thoroughly
	 * checked beforehand.  Update the board structure to reflect
	 * the full set of changes implied by it.
	 */
static void
make_move(Col from_col, Rank from_rank, Col to_col, Rank to_rank,
		Piece piece, Colour colour,Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    short from_r = RankConvert(from_rank);
    short from_c = ColConvert(from_col);

    if(piece == KING){
	board->EnPassant = FALSE;
	if(colour == WHITE){
	    board->WKingCol = to_col;
	    board->WKingRank = to_rank;
	    board->WKingCastle =
		    board->WQueenCastle = FALSE;
	}
	else{
	    board->BKingCol = to_col;
	    board->BKingRank = to_rank;
	    board->BKingCastle =
		    board->BQueenCastle = FALSE;
	}
    }
    if(piece != PAWN){
	board->EnPassant = FALSE;
    }
    else{
	if(colour == WHITE){
	    if((from_rank == '2') && (to_rank == '4')){
		/* This move permits an en-passant capture on the following move. */
		board->EnPassant = TRUE;
		board->ep_rank = to_rank-1;
		board->ep_col = to_col;
	    }
	    else if((board->EnPassant) && (board->ep_rank == to_rank) &&
			(board->ep_col == to_col)){
		/* This is an ep capture. Remove the intermediate pawn. */
	        board->board[RankConvert(to_rank)-1][ColConvert(to_col)] = EMPTY;
	        board->EnPassant = FALSE;
		board->hash_value ^= HashLookup(to_col,to_rank-1,PAWN,BLACK);
	    }
	    else{
		board->EnPassant = FALSE;
	    }
	}
	else{
	    if((from_rank == '7') && (to_rank == '5')){
		board->EnPassant = TRUE;
		board->ep_rank = to_rank+1;
		board->ep_col = to_col;
	    }
	    else if((board->EnPassant) && (board->ep_rank == to_rank) &&
			(board->ep_col == to_col)){
		/* This is an ep capture. Remove the intermediate pawn. */
		board->board[RankConvert(to_rank)+1][ColConvert(to_col)] = EMPTY;
		board->EnPassant = FALSE;
		board->hash_value ^= HashLookup(to_col,to_rank+1,PAWN,WHITE);
	    }
	    else{
		board->EnPassant = FALSE;
	    }
	}
    }
    board->board[from_r][from_c] = EMPTY;
    board->hash_value ^= HashLookup(from_col,from_rank,piece,colour);
    if(board->board[to_r][to_c] != EMPTY){
	/* Delete the captured piece from the hash value. */ Piece
	coloured_piece = board->board[to_r][to_c]; Piece removed_piece;
	Colour removed_colour;

	removed_piece = EXTRACT_PIECE(coloured_piece);
	removed_colour = EXTRACT_COLOUR(coloured_piece);
	board->hash_value ^= HashLookup(to_col,to_rank,removed_piece,removed_colour);
    }
    /* Insert the moved piece into the hash value. */
    board->board[to_r][to_c] = MAKE_COLOURED_PIECE(colour,piece);
    board->hash_value ^= HashLookup(to_col,to_rank,piece,colour);
}

	/* Find pawn moves matching the to_ and from_ information. */
static MovePair *
find_pawn_moves(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    short from_r = 0, from_c = 0;
    MovePair *move_list = NULL;
    MovePair *move = NULL;
    /* White pawn moves are offset by +1, Black by -1. */
    short offset = (colour == WHITE)? 1 : -1;
    Piece piece_to_move = MAKE_COLOURED_PIECE(colour,PAWN);

    if((to_col != 0) && (to_rank != 0)){
	/* We know the complete destination. */
	if(board->board[to_r][to_c] == EMPTY){
	    /* Destination must be empty for this form. */
	    if(board->board[to_r-offset][to_c] == piece_to_move){
		/* MovePair of one square. */
		move = malloc_move();
		move->from_rank = ToRank(to_r-offset);
		move->from_col = ToCol(to_c);
		move->to_rank = to_rank;
		move->to_col = to_col;
	    }
	    else if((board->board[to_r-offset][to_c] == EMPTY) &&
			    (to_rank == (colour == WHITE?'4':'5'))){
		/* Special case of initial two square move. */
		if(board->board[to_r-2*offset][to_c] == piece_to_move){
		    move = malloc_move();
		    move->from_rank = ToRank(to_r-2*offset);
		    move->from_col = ToCol(to_c);
		    move->to_rank = to_rank;
		    move->to_col = to_col;
		}
	    }
	    else if(board->EnPassant &&
			    (board->ep_rank == to_rank) && (board->ep_col == to_col)){
		/* Make sure that there is a valid pawn in position. */
		if(from_col != 0){
		    from_c = ColConvert(from_col);
		    from_r = to_r-offset;
		    if(board->board[from_r][from_c] == piece_to_move){
			move = malloc_move();
			move->from_rank = ToRank(from_r);
			move->from_col = ToCol(from_c);
			move->to_col = to_col;
			move->to_rank = ToRank(to_r);
		    }
		}
	    }
	}
	else if(piece_is_colour(board->board[to_r][to_c],OPPOSITE_COLOUR(colour))){
	    /* Capture on the destination square. */
	    if(from_col != 0){
		/* We know the from column. */
		from_c = ColConvert(from_col);
		from_r = to_r-offset;
		if(board->board[from_r][from_c] == piece_to_move){
		    move = malloc_move();
		    move->from_rank = ToRank(from_r);
		    move->from_col = ToCol(from_c);
		    move->to_rank = to_rank;
		    move->to_col = to_col;
		}
	    }
	}
	move_list = move;
    }
    else if((from_col != 0) && (to_col != 0)){
	/* Should be a diagonal capture. */
	if(((from_col + 1) != to_col) && ((from_col - 1) != to_col)){
	    /* Inconsistent information. */
	}
	else{
	    to_c = ColConvert(to_col);
	    if(from_rank != 0){
		/* We have complete source information. Check its
		 * veracity.
		 */
		from_r = RankConvert(from_rank);
		from_c = ColConvert(from_col);
		to_r = from_r - offset;
		if(board->board[from_r][from_c] == piece_to_move){
		    if(piece_is_colour(board->board[to_r][to_c],
						OPPOSITE_COLOUR(colour))){
			move = malloc_move();
			move->from_rank = ToRank(from_r);
			move->from_col = ToCol(from_c);
			move->to_col = to_col;
			move->to_rank = ToRank(to_r);
		    }
		    else if(board->EnPassant && (board->ep_rank == ToRank(to_r)) &&
				    (board->ep_col == ToCol(to_c))){
			move = malloc_move();
			move->from_rank = ToRank(from_r);
			move->from_col = ToCol(from_c);
			move->to_col = to_col;
			move->to_rank = ToRank(to_r);
		    }
		}
	    }
	    else{
		/* We must search the from_col and to_col for appropriate
		 * combinations.  There may be more than one.
		 */
		short start_rank, end_rank;

		from_c = ColConvert(from_col);
		/* Work out from where to start and end. */
		if(colour == WHITE){
		    start_rank = RankConvert(FIRSTRANK+1);
		    end_rank = RankConvert(LASTRANK);
		}
		else{
		    start_rank = RankConvert(LASTRANK-1);
		    end_rank = RankConvert(FIRSTRANK);
		}
		for(from_r = start_rank; from_r != end_rank; from_r += offset){
		    to_r = from_r+offset;
		    if(board->board[from_r][from_c] == piece_to_move){
			if(piece_is_colour(board->board[to_r][to_c],
							OPPOSITE_COLOUR(colour))){
			    move = malloc_move();
			    move->from_rank = ToRank(from_r);
			    move->from_col = ToCol(from_c);
			    move->to_col = to_col;
			    move->to_rank = ToRank(to_r);
			    move->next = move_list;
			    move_list = move;
			}
			else if(board->EnPassant && (board->ep_rank == ToRank(to_r)) &&
					(board->ep_col == ToCol(to_c))){
			    move = malloc_move();
			    move->from_rank = ToRank(from_r);
			    move->from_col = ToCol(from_c);
			    move->to_col = to_col;
			    move->to_rank = ToRank(to_r);
			    move->next = move_list;
			    move_list = move;
			}
		    }
		}
	    }
	}
    }
    return move_list;
}

	/* Find knight moves to the given square. */
static MovePair *
find_knight_moves(Col to_col,Rank to_rank, Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    unsigned index;
    MovePair *move_list = NULL;

    /* Pick up pairs of offsets from to_r,to_c to look for a Knight of
     * the right colour.
     */
    for(index = 0; index < 2*NUM_KNIGHT_MOVES; index += 2){
	short r = Knight_moves[index]+to_r;
	short c = Knight_moves[index+1]+to_c;

	if(board->board[r][c] == MAKE_COLOURED_PIECE(colour,KNIGHT)){
	    MovePair *move = malloc_move();

	    move->from_rank = ToRank(r);
	    move->from_col = ToCol(c);
	    move->to_rank = to_rank;
	    move->to_col = to_col;
	    move->next = move_list;
	    move_list = move;
	}
    }
    return move_list;
}

	/* Find bishop moves to the given square. */
static MovePair *
find_bishop_moves(Col to_col,Rank to_rank, Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    unsigned index;
    MovePair *move_list = NULL;

    /* Pick up pairs of offsets from to_r,to_c to look for a Knight of
     * the right colour.
     */
    for(index = 0; index < 2*NUM_BISHOP_MOVES; index += 2){
	short r = to_r, c = to_c;
	
	/* Work backwards from the destination to find a bishop of
	 * the right colour.
	 */
	do{
	    r += Bishop_moves[index];
	    c += Bishop_moves[index+1];
	} while(board->board[r][c] == EMPTY);

	if(board->board[r][c] == MAKE_COLOURED_PIECE(colour,BISHOP)){
	    MovePair *move = malloc_move();

	    move->from_rank = ToRank(r);
	    move->from_col = ToCol(c);
	    move->to_rank = to_rank;
	    move->to_col = to_col;
	    move->next = move_list;
	    move_list = move;
	}
    }
    return move_list;
}

	/* Find rook moves to the given square. */
static MovePair *
find_rook_moves(Col to_col,Rank to_rank, Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    unsigned index;
    MovePair *move_list = NULL;

    /* Pick up pairs of offsets from to_r,to_c to look for a Knight of
     * the right colour.
     */
    for(index = 0; index < 2*NUM_ROOK_MOVES; index += 2){
	short r = to_r, c = to_c;
	
	/* Work backwards from the destination to find a rook of
	 * the right colour.
	 */
	do{
	    r += Rook_moves[index];
	    c += Rook_moves[index+1];
	} while(board->board[r][c] == EMPTY);

	if(board->board[r][c] == MAKE_COLOURED_PIECE(colour,ROOK)){
	    MovePair *move = malloc_move();

	    move->from_rank = ToRank(r);
	    move->from_col = ToCol(c);
	    move->to_rank = to_rank;
	    move->to_col = to_col;
	    move->next = move_list;
	    move_list = move;
	}
    }
    return move_list;
}

	/* Find queen moves to the given square. */
static MovePair *
find_queen_moves(Col to_col,Rank to_rank, Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    unsigned index;
    MovePair *move_list = NULL;

    /* Pick up pairs of offsets from to_r,to_c to look for a Knight of
     * the right colour.
     */
    for(index = 0; index < 2*NUM_QUEEN_MOVES; index += 2){
	short r = to_r, c = to_c;
	
	/* Work backwards from the destination to find a rook of
	 * the right colour.
	 */
	do{
	    r += Queen_moves[index];
	    c += Queen_moves[index+1];
	} while(board->board[r][c] == EMPTY);

	if(board->board[r][c] == MAKE_COLOURED_PIECE(colour,QUEEN)){
	    MovePair *move = malloc_move();

	    move->from_rank = ToRank(r);
	    move->from_col = ToCol(c);
	    move->to_rank = to_rank;
	    move->to_col = to_col;
	    move->next = move_list;
	    move_list = move;
	}
    }
    return move_list;
}

	/* Find King moves to the given square. */
static MovePair *
find_king_moves(Col to_col,Rank to_rank, Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    unsigned index;
    MovePair *move_list = NULL;

    /* Pick up pairs of offsets from to_r,to_c to look for a Knight of
     * the right colour.
     */
    for(index = 0; index < 2*NUM_KING_MOVES; index += 2){
	short r = King_moves[index]+to_r;
	short c = King_moves[index+1]+to_c;
	
	if(board->board[r][c] == MAKE_COLOURED_PIECE(colour,KING)){
	    MovePair *move = malloc_move();

	    move->from_rank = ToRank(r);
	    move->from_col = ToCol(c);
	    move->to_rank = to_rank;
	    move->to_col = to_col;
	    move->next = move_list;
	    move_list = move;
	}
    }
    return move_list;
}

	/* possibles contains a list of possible moves of piece.
	 * This function should exclude all of those moves of this piece
	 * which leave its own king in check.  The list of remaining legal
	 * moves is returned as result.
	 * This function operates by looking for at least one reply by the
	 * opponent that could capture the king of the given colour.
	 * Only one such move needs to be found to invalidate one of the
	 * possible moves.
	 */
static MovePair *
exclude_checks(Piece piece, Colour colour,MovePair *possibles, const Board *board)
{   Board copy_board;
    MovePair *move_list = NULL;
    MovePair *move;
    Col king_col;
    Rank king_rank;
    Colour opponent_colour = OPPOSITE_COLOUR(colour);

    /* For each possible move, make the move and see if it leaves the king
     * in check.
     */
    for(move = possibles; move != NULL; ){
	MovePair *replies;

	/* Take a copy of the board before playing this next move. */
	copy_board = *board;
	make_move(move->from_col,move->from_rank,
		  move->to_col,move->to_rank,piece,colour,&copy_board);
	/* Find out where the king is now. */
	if(colour == WHITE){
	    king_col = copy_board.WKingCol;
	    king_rank = copy_board.WKingRank;
	}
	else{
	    king_col = copy_board.BKingCol;
	    king_rank = copy_board.BKingRank;
	}
	/* Try and find one move that leaves this king in check.
	 * There is probably an optimal order for these tests, but I haven't
	 * tried to find it.
	 */
	if((replies =
	    find_pawn_moves(0,0,king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a pawn. */
	}
	else if((replies =
	    find_knight_moves(king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a knight. */
	}
	else if((replies =
	    find_bishop_moves(king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a bishop. */
	}
	else if((replies =
	    find_rook_moves(king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a rook. */
	}
	else if((replies =
	    find_queen_moves(king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a queen. */
	}
	else if((replies =
	    find_king_moves(king_col,king_rank,
					opponent_colour,&copy_board)) != NULL){
	    /* King is in check from a king. */
	}
	else{
	    /* King is safe. */
	}
	if(replies == NULL){
	    /* King is safe and the move may be kept. */
	    MovePair *legal_move = move;

	    move = move->next;
	    legal_move->next = move_list;
	    move_list = legal_move;
	}
	else{
	    MovePair *illegal_move;

	    /* Free the list of moves which killed this move. */
	    free_move_pair_list(replies);
	    illegal_move = move;
	    move = move->next;
	    /* Free the illegal move. */
	    free_move(illegal_move);
	}
    }
    return move_list;
}

    /* We must exclude the possibility of the king passing
     * through check.
     */
static Boolean
exclude_castling_across_checks(MoveClass class, Colour colour, const Board *board)
{   Boolean Ok = TRUE;
    MovePair *move = malloc_move();
    Rank rank = (colour == WHITE)? FIRSTRANK : LASTRANK;
    Col to_col;

    if(class == KINGSIDE_CASTLE){
	/* Start where we are, because you can't castle out of check. */
	for(to_col = 'e'; (to_col <= 'g') && Ok; to_col++){
	    move->from_col = 'e';
	    move->from_rank = rank;
	    move->to_col = to_col;
	    move->to_rank = rank;
	    move = exclude_checks(KING,colour,move,board);
	    if(move == NULL){
		Ok = FALSE;
	    }
	}
    }
    else{
	/* Start where we are, because you can't castle out of check. */
	for(to_col = 'e'; (to_col <= 'c') && Ok; to_col--){
	    move->from_col = 'e';
	    move->from_rank = rank;
	    move->to_col = to_col;
	    move->to_rank = rank;
	    move = exclude_checks(KING,colour,move,board);
	    if(move == NULL){
		Ok = FALSE;
	    }
	}
    }
    if(move != NULL){
	free_move(move);
    }
    return Ok;
}

	/* Possibles is a list of possible moves of piece.
	 * Exclude all of those that either leave the king in check
	 * or those excluded by non-null information in from_col or from_rank.
	 */
static MovePair *
exclude_moves(Piece piece, Colour colour,Col from_col, Rank from_rank,
	MovePair *possibles, const Board *board)
{   MovePair *move_list = NULL;

    /* See if we have disambiguating from_ information. */
    if((from_col != 0) || (from_rank != 0)){
	MovePair *move, *temp;

	for(move = possibles; move != NULL; ){
	    Boolean excluded = FALSE;

	    if(from_col != 0){
		/* The from_col is specified. */
		if(move->from_col != from_col){
		    excluded = TRUE;
		}
	    }
	    if((from_rank != 0) && !excluded){
		if(move->from_rank != from_rank){
		    excluded = TRUE;
		}
	    }
	    temp = move;
	    move = move->next;
	    if(!excluded){
		/* Add it to the list of possibles. */
		temp->next = move_list;
		move_list = temp;
	    }
	    else{
		/* Discard it. */
		free_move(temp);
	    }
	}
    }
    else{
	/* Everything is still possible. */
	move_list = possibles;
    }
    if(move_list != NULL){
	move_list = exclude_checks(piece,colour,move_list,board);
    }
    return move_list;
}

	/* Make a pawn move.
	 * En-passant information in the original move text is not currently used
	 * to disambiguate pawn moves.  E.g. with Black pawns on c4 and c5 after
	 * White move 1. d4 a reply 1... cdep will be rejected as ambiguous.
	 */
static Boolean
pawn_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Board *board)
{   MovePair *move_list = find_pawn_moves(from_col,from_rank,to_col,to_rank,
		colour,board);
    Boolean Ok = TRUE;

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No pawn move available to %c%c\n",to_col,to_rank);
	Ok = FALSE;
    }
    else{
	move_list = exclude_moves(PAWN,colour,from_col,from_rank,move_list,board);
	if(move_list != NULL){
	    if(move_list->next == NULL){
		MovePair *move = move_list;

		/* Unambiguous move. */
		make_move(move->from_col,move->from_rank,
			    move->to_col,move->to_rank,PAWN,colour,board);
		/* Detect implicit promotion. */
		if((move->to_rank == LASTRANK) && (colour == WHITE)){
		    /* Make the promotion. */
		    make_move(move->to_col,move->to_rank,
				move->to_col,move->to_rank,W(QUEEN),colour,board);
		}
		else if((move->to_rank == FIRSTRANK) && (colour == BLACK)){
		    /* Make the promotion. */
		    make_move(move->to_col,move->to_rank,
				move->to_col,move->to_rank,B(QUEEN),colour,board);
		}
	    }
	    else{
		fprintf(GlobalState.logfile,
				"Ambiguous pawn move to %c%c\n",to_col,to_rank);
	    }
	    free_move_pair_list(move_list);
	}
	else{
	    fprintf(GlobalState.logfile,
				"Pawn moves to %c%c are excluded.\n",to_col,to_rank);
	    Ok = FALSE;
	}
    }
    return Ok;
}

	/* Make a pawn move that involves an explicit promotion to new_piece. */
static Boolean
promote(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Piece new_piece, Board *board)
{   MovePair *move_list;
    Boolean Ok = TRUE;

    if(colour == WHITE){
	to_rank = LASTRANK;
    }
    else{
	to_rank = FIRSTRANK;
    }
    move_list = find_pawn_moves(from_col,from_rank,to_col,to_rank,colour,board);
    if(move_list != NULL){
	if(move_list->next == NULL){
	    /* Unambiguous move. */
	    make_move(move_list->from_col,move_list->from_rank,
			move_list->to_col,move_list->to_rank,PAWN,colour,board);
	    /* Now make the promotion. */
	    make_move(move_list->to_col,move_list->to_rank,
			move_list->to_col,move_list->to_rank,new_piece,colour,board);
	}
	else{
	    fprintf(GlobalState.logfile,"Ambiguous pawn move to %c%c\n",to_col,to_rank);
	    Ok = FALSE;
	}
	free_move_pair_list(move_list);
    }
    else{
	fprintf(GlobalState.logfile,"Illegal pawn promotion to %c%c\n",to_col,to_rank);
	Ok = FALSE;
    }
    return Ok;
}

	/* Make a knight move. */
static Boolean
knight_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    MovePair *move_list = find_knight_moves(to_col,to_rank,colour,board);
    /* Assume everything will be ok. */
    Boolean Ok = TRUE;

    move_list = exclude_moves(KNIGHT,colour,from_col,from_rank,move_list,board);

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No knight move possible to %c%c.\n",to_col,to_rank);
	Ok = FALSE;
    }
    else if(move_list->next == NULL){
	/* Only one possible.  Check for legality. */
	Piece occupant = board->board[to_r][to_c];

	if((occupant == EMPTY) || piece_is_colour(occupant,OPPOSITE_COLOUR(colour))){
	    make_move(move_list->from_col,move_list->from_rank,
				to_col,to_rank,KNIGHT,colour,board);
	}
	else{
	    fprintf(GlobalState.logfile,"Knight destination square %c%c is illegal.\n",
				to_col,to_rank);
	    Ok = FALSE;
	}
	free_move(move_list);
    }
    else{
	fprintf(GlobalState.logfile,"Ambiguous knight move to %c%c.\n",to_col,to_rank);
	free_move_pair_list(move_list);
	Ok = FALSE;
    }
    return Ok;
}

	/* Make a bishop move. */
static Boolean
bishop_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank, Colour colour,
			Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    MovePair *move_list = find_bishop_moves(to_col,to_rank,colour,board);
    /* Assume that it is ok. */
    Boolean Ok = TRUE;

    move_list = exclude_moves(BISHOP,colour,from_col,from_rank,move_list,board);

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No bishop move possible to %c%c.\n",to_col,to_rank);
	Ok = FALSE;
    }
    else if(move_list->next == NULL){
	/* Only one possible.  Check for legality. */
	Piece occupant = board->board[to_r][to_c];

	if((occupant == EMPTY) || piece_is_colour(occupant,OPPOSITE_COLOUR(colour))){
	    make_move(move_list->from_col,move_list->from_rank,
					to_col,to_rank,BISHOP,colour,board);
	}
	else{
	    fprintf(GlobalState.logfile,"Bishop's destination square %c%c is illegal.\n",
				to_col,to_rank);
	    Ok = FALSE;
	}
	free_move(move_list);
    }
    else{
	fprintf(GlobalState.logfile,"Ambiguous bishop move to %c%c.\n",to_col,to_rank);
	free_move_pair_list(move_list);
	Ok = FALSE;
    }
    return Ok;
}

	/* Make a rook move. */
static Boolean
rook_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    MovePair *move_list = find_rook_moves(to_col,to_rank,colour,board);
    /* Assume that it is ok. */
    Boolean Ok = TRUE;

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No rook move possible to %c%c.\n",to_col,to_rank);
	Ok = FALSE;
    }
    else{
	move_list = exclude_moves(ROOK,colour,from_col,from_rank,move_list,board);

	if(move_list == NULL){
	    fprintf(GlobalState.logfile,"Indicated rook move is excluded.\n");
	    Ok = FALSE;
	}
	else if(move_list->next == NULL){
	    /* Only one possible.  Check for legality. */
	    Piece occupant = board->board[to_r][to_c];

	    if((occupant == EMPTY) || piece_is_colour(occupant,OPPOSITE_COLOUR(colour))){
		make_move(move_list->from_col,move_list->from_rank,
					to_col,to_rank,ROOK,colour,board);
		if(colour == WHITE){
		    if(move_list->from_rank == FIRSTRANK){
			/* Consider disallowing castling. */
			if(board->WKingCastle && (move_list->from_col == LASTCOL)){
			    board->WKingCastle = FALSE;
			}
			else if(board->WQueenCastle && (move_list->from_col == FIRSTCOL)){
			    board->WQueenCastle = FALSE;
			}
		    }
		}
		else{
		    if(move_list->from_rank == LASTRANK){
			/* Consider disallowing castling. */
			if(board->BKingCastle && (move_list->from_col == LASTCOL)){
			    board->BKingCastle = FALSE;
			}
			else if(board->BQueenCastle && (move_list->from_col == FIRSTCOL)){
			    board->BQueenCastle = FALSE;
			}
		    }
		}
	    }
	    else{
		fprintf(GlobalState.logfile,
				    "Rook's destination square %c%c is illegal.\n",
				    to_col,to_rank);
		Ok = FALSE;
	    }
	    free_move(move_list);
	}
	else{
	    fprintf(GlobalState.logfile,"Ambiguous rook move to %c%c.\n",to_col,to_rank);
	    free_move_pair_list(move_list);
	    Ok = FALSE;
	}
    }
    return Ok;
}

	/* Make a queen move. */
static Boolean
queen_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
		Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    MovePair *move_list = find_queen_moves(to_col,to_rank,colour,board);
    /* Assume that it is ok. */
    Boolean Ok = TRUE;

    move_list = exclude_moves(QUEEN,colour,from_col,from_rank,move_list,board);

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No queen move possible to %c%c.\n",to_col,to_rank);
	Ok = FALSE;
    }
    else if(move_list->next == NULL){
	/* Only one possible.  Check for legality. */
	Piece occupant = board->board[to_r][to_c];

	if((occupant == EMPTY) || piece_is_colour(occupant,OPPOSITE_COLOUR(colour))){
	    make_move(move_list->from_col,move_list->from_rank,
					to_col,to_rank,QUEEN,colour,board);
	}
	else{
	    fprintf(GlobalState.logfile,"Queen's destination square %c%c is illegal.\n",
					to_col,to_rank);
	    Ok = FALSE;
	}
	free_move(move_list);
    }
    else{
	fprintf(GlobalState.logfile,"Ambiguous queen move to %c%c.\n",to_col,to_rank);
	free_move_pair_list(move_list);
	Ok = FALSE;
    }
    return Ok;
}

	/* Castle king side. */
static Boolean
kingside_castle(Colour colour, Board *board)
{   /* Assume failure. */
    Boolean Ok = FALSE;

    if(colour == WHITE){
	short king_r = RankConvert(FIRSTRANK);
	short king_c = ColConvert('e');

	if(board->WKingCastle){
	    /* It is permitted. */
	    if((board->board[king_r][king_c] == W(KING)) &&
		    (board->board[king_r][king_c+1] == EMPTY) &&
		    (board->board[king_r][king_c+2] == EMPTY) &&
		    (board->board[king_r][king_c+3] == W(ROOK))){
		if(exclude_castling_across_checks(KINGSIDE_CASTLE,colour,board)){
		    make_move('e',FIRSTRANK,'g',FIRSTRANK,KING,WHITE,board);
		    make_move(LASTCOL,FIRSTRANK,'f',FIRSTRANK,ROOK,WHITE,board);
		    Ok = TRUE;
		}
		else{
		    fprintf(GlobalState.logfile,
				"Castling across or out of check is not allowed.\n");
		}
		    
	    }
	    else{
		fprintf(GlobalState.logfile,"Kingside castling is blocked for White.\n");
	    }
	}
	else{
	    fprintf(GlobalState.logfile,"Kingside castling is forbidden to White.\n");
	}
    }
    else{
	short king_r = RankConvert(LASTRANK);
	short king_c = ColConvert('e');

	if(board->BKingCastle){
	    /* It is permitted. */
	    if((board->board[king_r][king_c] == B(KING)) &&
		    (board->board[king_r][king_c+1] == EMPTY) &&
		    (board->board[king_r][king_c+2] == EMPTY) &&
		    (board->board[king_r][king_c+3] == B(ROOK))){
		if(exclude_castling_across_checks(KINGSIDE_CASTLE,colour,board)){
		    make_move('e',LASTRANK,'g',LASTRANK,KING,BLACK,board);
		    make_move(LASTCOL,LASTRANK,'f',LASTRANK,ROOK,BLACK,board);
		    Ok = TRUE;
		}
		else{
		    fprintf(GlobalState.logfile,
				"Castling across or out of check is not allowed.\n");
		}
	    }
	    else{
		fprintf(GlobalState.logfile,"Kingside castling is blocked for Black.\n");
	    }
	}
	else{
	    fprintf(GlobalState.logfile,"Kingside castling is forbidden to Black.\n");
	}
    }
    return Ok;
}

static Boolean
queenside_castle(Colour colour, Board *board)
{   /* Assume failure. */
    Boolean Ok = FALSE;

    if(colour == WHITE){
	short king_r = RankConvert(FIRSTRANK);
	short king_c = ColConvert('e');

	if(board->WQueenCastle){
	    /* It is permitted. */
	    if((board->board[king_r][king_c] == W(KING)) &&
		    (board->board[king_r][king_c-1] == EMPTY) &&
		    (board->board[king_r][king_c-2] == EMPTY) &&
		    (board->board[king_r][king_c-3] == EMPTY) &&
		    (board->board[king_r][king_c-4] == W(ROOK))){
		if(exclude_castling_across_checks(QUEENSIDE_CASTLE,colour,board)){
		    make_move('e',FIRSTRANK,'c',FIRSTRANK,KING,WHITE,board);
		    make_move(FIRSTCOL,FIRSTRANK,'d',FIRSTRANK,ROOK,WHITE,board);
		    Ok = TRUE;
		}
		else{
		    fprintf(GlobalState.logfile,
				"Castling across or out of check is not allowed.\n");
		}
	    }
	    else{
		fprintf(GlobalState.logfile,
				"Queenside castling is blocked for White.\n");
	    }
	}
	else{
	    fprintf(GlobalState.logfile,"Queenside castling is forbidden to White.\n");
	}
    }
    else{
	short king_r = RankConvert(LASTRANK);
	short king_c = ColConvert('e');

	if(board->BQueenCastle){
	    /* It is permitted. */
	    if((board->board[king_r][king_c] == B(KING)) &&
		    (board->board[king_r][king_c-1] == EMPTY) &&
		    (board->board[king_r][king_c-2] == EMPTY) &&
		    (board->board[king_r][king_c-3] == EMPTY) &&
		    (board->board[king_r][king_c-4] == B(ROOK))){
		if(exclude_castling_across_checks(QUEENSIDE_CASTLE,colour,board)){
		    make_move('e',LASTRANK,'c',LASTRANK,KING,BLACK,board);
		    make_move(FIRSTCOL,LASTRANK,'d',LASTRANK,ROOK,BLACK,board);
		    Ok = TRUE;
		}
		else{
		    fprintf(GlobalState.logfile,
				"Castling across or out of check is not allowed.\n");
		}
	    }
	    else{
		fprintf(GlobalState.logfile,
				"Queenside castling is blocked for White.\n");
	    }
	}
	else{
	    fprintf(GlobalState.logfile,
			"Queenside castling is forbidden to Black.\n");
	}
    }
    return Ok;
}

static Boolean
king_move(Col from_col, Rank from_rank, Col to_col,Rank to_rank,
			Colour colour, Board *board)
{   short to_r = RankConvert(to_rank);
    short to_c = ColConvert(to_col);
    /* Find all possible king moves to the destination squares. */
    MovePair *move_list = find_king_moves(to_col,to_rank,colour,board);
    /* Assume that it is ok. */
    Boolean Ok = TRUE;

    /* Exclude disambiguated and illegal moves. */
    move_list = exclude_moves(KING,colour,from_col,from_rank,move_list,board);

    if(move_list == NULL){
	fprintf(GlobalState.logfile,"No king move possible to %c%c.\n",to_col,to_rank);
	Ok = FALSE;
    }
    else if(move_list->next == NULL){
	/* Only one possible.  Check for legality. */
	Piece occupant = board->board[to_r][to_c];

	if((occupant == EMPTY) || piece_is_colour(occupant,OPPOSITE_COLOUR(colour))){
	    make_move(move_list->from_col,move_list->from_rank,
			to_col,to_rank,KING,colour,board);
	    if(colour == WHITE){
		board->WKingCastle =
			board->WQueenCastle = FALSE;
	    }
	    else{
		board->BKingCastle =
			board->BQueenCastle = FALSE;
	    }
	}
	else{
	    fprintf(GlobalState.logfile,"King's destination square %c%c is illegal.\n",
				to_col,to_rank);
	    Ok = FALSE;
	}
	free_move(move_list);
    }
    else{
	fprintf(GlobalState.logfile,
			"Ambiguous king move possible to %c%c.\n",to_col,to_rank);
	fprintf(GlobalState.logfile,"Multiple kings?!\n");
	free_move_pair_list(move_list);
	Ok = FALSE;
    }
    return Ok;
}

static Board *
new_game(void)
{   Board *new_board = (Board *)MallocOrDie(sizeof(Board));
    static Board initial_board =
      {
	{ { OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF},
	  { OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF},
	  { OFF, OFF, W(ROOK), W(KNIGHT), W(BISHOP), W(QUEEN),
			    W(KING), W(BISHOP), W(KNIGHT), W(ROOK), OFF, OFF},
	  { OFF, OFF, W(PAWN), W(PAWN), W(PAWN), W(PAWN),
			    W(PAWN), W(PAWN), W(PAWN), W(PAWN), OFF, OFF},
	  { OFF, OFF, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, OFF, OFF},
	  { OFF, OFF, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, OFF, OFF},
	  { OFF, OFF, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, OFF, OFF},
	  { OFF, OFF, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, OFF, OFF},
	  { OFF, OFF, B(PAWN), B(PAWN), B(PAWN), B(PAWN),
			    B(PAWN), B(PAWN), B(PAWN), B(PAWN), OFF, OFF},
	  { OFF, OFF, B(ROOK), B(KNIGHT), B(BISHOP), B(QUEEN),
			    B(KING), B(BISHOP), B(KNIGHT), B(ROOK), OFF, OFF},
	  { OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF},
	  { OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF, OFF}
	},
	/* Castling rights. */
	TRUE, TRUE, TRUE, TRUE,
	/* Initial king positions. */
	'e',FIRSTRANK,
	'e',LASTRANK,
	/* En Passant rights. */
	FALSE, 0, 0,
	/* Initial hash value. */
	0ul,
      };
    Piece piece;
    Colour colour;
    Rank rank;
    Col col;

    /* Reset the contents of the new board. */
    *new_board = initial_board;

    /* Generate the hash value for the initial position. */
    for(col = FIRSTCOL; col <= LASTCOL; col++){
        for(rank = FIRSTRANK; rank <= LASTRANK; rank++){
	    Piece coloured_piece = new_board->board[RankConvert(rank)][ColConvert(col)];

	    piece = EXTRACT_PIECE(coloured_piece);
	    colour = EXTRACT_COLOUR(coloured_piece);
	    new_board->hash_value ^= HashLookup(col,rank,piece,colour);
	}
    }
    return new_board;
}

	/* Output the current move along with associated information.
	 * Return 1 if a variation was printed, 0 otherwise.
	 */
static Boolean
apply_move(Colour colour,const Move *move_details, Board *board)
{   Boolean Ok = FALSE;

    if(move_details == NULL){
	/* Shouldn't happen. */
	fprintf(GlobalState.logfile,"Empty move details in apply_move.\n");
    }
    else if(move_details->move[0] == '\0'){
	/* Shouldn't happen. */
	fprintf(GlobalState.logfile,"Null move in apply_move.\n");
    }
    else{
	const char *move = move_details->move;
	Piece new_piece = 0;

	/* We have something. */
	switch(move_details->class){
	    case PAWN_MOVE:
		Ok = pawn_move(move_details->from_col,move_details->from_rank,
			       move_details->to_col,move_details->to_rank,
			       colour,board);
		break;
	    case PAWN_MOVE_WITH_PROMOTION:
		{ unsigned last_char = strlen(move)-1;

		  /* Skip any check character. */
		  if(move[last_char] == '+'){
		      last_char--;
		  }
		    switch(move[last_char]){
			case 'Q':
			    new_piece = QUEEN;
			    break;
			case 'R':
			    new_piece = ROOK;
			    break;
			case 'B':
			    new_piece = BISHOP;
			    break;
			case 'N':
			    new_piece = KNIGHT;
			    break;
			default:
			    fprintf(GlobalState.logfile,
					"Unknown piece in promotion %s\n",move);
			    break;
		    }
		}
		if(new_piece != 0){
		    Ok = promote(move_details->from_col,move_details->from_rank,
			       move_details->to_col,move_details->to_rank,
			       colour, new_piece,board);
		}
		break;
	    case PIECE_MOVE:
		switch(*move){
		    case 'K':
			Ok = king_move(move_details->from_col,move_details->from_rank,
				       move_details->to_col,move_details->to_rank,
				       colour,board);
			break;
		      break;
		    case 'Q':
			Ok = queen_move(move_details->from_col,move_details->from_rank,
				       move_details->to_col,move_details->to_rank,
				       colour,board);
			break;
		      break;
		    case 'R':
			Ok = rook_move(move_details->from_col,move_details->from_rank,
				       move_details->to_col,move_details->to_rank,
				       colour,board);
			break;
		      break;
		    case 'N':
			Ok = knight_move(move_details->from_col,move_details->from_rank,
				       move_details->to_col,move_details->to_rank,
				       colour,board);
			break;
		      break;
		    case 'B':
			Ok = bishop_move(move_details->from_col,move_details->from_rank,
				       move_details->to_col,move_details->to_rank,
				       colour,board);
			break;
		      break;
		    default:
			fprintf(GlobalState.logfile,"Unknown piece move %s\n",move);
			break;
		}
		break;
	    case KINGSIDE_CASTLE:
		Ok = kingside_castle(colour,board);
		break;
	    case QUEENSIDE_CASTLE:
		Ok = queenside_castle(colour,board);
		break;
	    case UNKNOWN_MOVE:
		Ok = FALSE;
		break;
	    default:
		fprintf(GlobalState.logfile,"Unknown move class in apply_move(%d).\n",
				move_details->class);
		break;
	}
    }
    return Ok;
}

	/* move_details contains a complete move score.
	 * try to apply each move on a new board.
	 */
Boolean
apply_move_list(Game *game_details)
{   const Move *move = game_details->moves;
    Board *board = new_game();
    Colour colour = WHITE;
    unsigned move_number = 1;
    /* Force a match if we aren't looking for positional variations. */
    Boolean game_matches = GlobalState.positional_variations?FALSE:TRUE;
    Boolean game_ok = TRUE;

    /* Start off the cumulative hash value. */
    game_details->cumulative_hash_value = 0;
    while(game_ok && (move != NULL)){
        /* Reset print_move number if a variation was printed. */
	if(*(move->move) == '\0'){
	    /* A comment node, not a proper move. */
	   move = move->next;
	}
        else if(apply_move(colour,move,board)){
	   /* Don't try for a positional match if we already have one. */
	   if(!game_matches && position_matches(board->hash_value)){
	       game_matches = TRUE;
	   }
	   /* Add in this hash value to the cumulative one. */
	   game_details->cumulative_hash_value += board->hash_value;
	   colour = OPPOSITE_COLOUR(colour);
	   move = move->next;
	   if(colour == WHITE){
	       move_number++;
	   }
	}
	else{
	    fprintf(GlobalState.logfile,"Failed to make move %u%s %s in the game:\n",
				move_number,(colour == WHITE)?".":"...",
			move->move);
	    report_details(GlobalState.logfile);
	    print_board(board,GlobalState.logfile);
	    game_ok = FALSE;
	}
    }
    /* Fill in the hash value of the final position reached. */
    game_details->final_hash_value = board->hash_value;
    (void) free((void *)board);
    return game_matches;
}

	/* Define a type to hold hash values of interest.
	 * This is used both to aid in duplicate detection
	 * and in finding positional variations.
	 */
typedef struct HashLog {
    HashCode required_hash_value;
    /* cumulative_hash_value is used to disambiguate clashing
     * final hash values in duplicate detection.
     */
    HashCode cumulative_hash_value;
    struct HashLog *next;
} HashLog;

	/* Define a table to hold the positional hash codes of interest. */
#define MAX_CODE 53
static HashLog *codes_of_interest[MAX_CODE];

	/* move_details is a variation in which we are interested.
	 * Generate and store the hash value in codes_of_interest.
	 */
void
store_hash_value(const Move *move_details)
{   const Move *move = move_details;
    Board *board = new_game();
    Colour colour = WHITE;
    unsigned move_number = 1;
    Boolean Ok = TRUE;

    while((move != NULL) && Ok){
        /* Reset print_move number if a variation was printed. */
	if(*(move->move) == '\0'){
	    /* A comment node, not a proper move. */
	   move = move->next;
	}
        else if(apply_move(colour,move,board)){
	   colour = OPPOSITE_COLOUR(colour);
	   move = move->next;
	   if(colour == WHITE){
	       move_number++;
	   }
	}
	else{
	    fprintf(GlobalState.logfile,"Failed to make move %u%s %s\n",
				move_number,(colour == WHITE)?".":"...",
			move->move);
	    Ok = FALSE;
	}
    }
    if(!Ok){
	exit(1);
    }
    else{
	HashLog *entry = (HashLog *)MallocOrDie(sizeof(*entry));
	unsigned index = board->hash_value % MAX_CODE;

	entry->required_hash_value = board->hash_value;
	entry->cumulative_hash_value = 0;
	/* Link it into the head at this index. */
	entry->next =  codes_of_interest[index];
	codes_of_interest[index] = entry;
    }
    (void) free((void *)board);
}

	/* Look in codes_of_interest for current_hash_value. */
static Boolean
position_matches(HashCode current_hash_value)
{   unsigned index = current_hash_value % MAX_CODE;
    Boolean found = FALSE;
    HashLog *entry;

    for(entry = codes_of_interest[index]; !found && (entry != NULL);
			entry = entry->next){
	if(entry->required_hash_value == current_hash_value){
	    found = TRUE;
	}
    }
    return found;
}

/* Define a table to hold hash values of the extracted games.
 * This is used to enable duplicate detection.
 */
#define LOG_TABLE_SIZE 1027
static HashLog *LogTable[LOG_TABLE_SIZE];

	/* Return TRUE if it looks like we have met game_details before. */
Boolean
game_is_a_duplicate(Game game_details)
{   unsigned index = game_details.final_hash_value % LOG_TABLE_SIZE;
    HashLog *entry = NULL;
    Boolean duplicate = FALSE;

    for(entry = LogTable[index]; !duplicate && (entry != NULL); entry = entry->next){
	if((entry->required_hash_value == game_details.final_hash_value) &&
		(entry->cumulative_hash_value == game_details.cumulative_hash_value)){
	    duplicate = TRUE;
	}
    }

    if(!duplicate){
	/* First occurrence. */
	entry = (HashLog *)MallocOrDie(sizeof(*entry));

	entry->required_hash_value = game_details.final_hash_value;
	entry->cumulative_hash_value = game_details.cumulative_hash_value;
	/* Link it into the head at this index. */
	entry->next =  LogTable[index];
	LogTable[index] = entry;
    }
    return duplicate;
}
