/*
 *  Program: extract -- a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@ukc.ac.uk
 *
 */

	    /* This file contains function concerned with decoding
	     * the original text of a move in order to determine
	     * 		which MoveClass it is in,
	     *		any start and end square information.
	     * It extracts this information purely from the move text
	     * rather than analysing the move within the context of
	     * a board position.
	     */

#include <stdio.h>
#include <string.h>
#include "bool.h"
#include "defs.h"
#include "typedef.h"
#include "protos.h"

	/* Does the character represent a column of the board? */
static Boolean
is_col(char c)
{
    return (FIRSTCOL <= c) && (c <= LASTCOL);
}

	/* Does the character represent a rank of the board? */
static Boolean
is_rank(char c)
{
    return (FIRSTRANK <= c) && (c <= LASTRANK);
}

static Boolean
is_piece(char c)
{   Boolean piece = TRUE;

    switch(c){
	case 'K': case 'k':
	case 'Q': case 'q':
	case 'R': case 'r':
	case 'N': case 'n':
	case 'B': 
	     /* Lower case 'b' is most likely to be a pawn reference. */
	     break;
	default:
	    piece = FALSE;
	    break;
    }
    return piece;
}

static Boolean
is_capture(char c)
{
    return (c == 'x') || (c == 'X');
}

static Boolean
is_castling_character(char c)
{
    return (c == 'O') || (c == '0') || (c == 'o');
}

static Boolean
is_check(char c)
{
    return (c == '+') || (c == '#');
}

	/* Allocate space in which to return the information that
	 * has been gleaned from the move.
	 */
Move *
new_move_structure(void)
{   Move *move = (Move *)MallocOrDie(sizeof(Move));

    move->terminating_result = NULL;
    move->Nags = NULL;
    move->Comment = NULL;
    move->Variants = NULL;
    move->next = NULL;
    return move;
}

	/* Work out whatever can be gleaned from move_string of
	 * the starting and ending points of the given move.
	 * The move may be any legal string.
	 * The scanning here is libertarian, so it relies heavily on
	 * illegal moves having already been filtered out.
	 */
Move *
decode_move(const char *move_string)
{   /* The four components of the co-ordinates when known. */
    Rank from_rank = 0, to_rank = 0;
    Col from_col = 0, to_col = 0;
    MoveClass class;
    Boolean Ok = TRUE;
    /* Temporary locations until known whether they are from_ or to_. */
    Col col = 0;
    Rank rank = 0;
    /* A pointer to move along the move string. */
    const char *move = move_string;
    /* A pointer to the structure containing the details to be returned. */
    Move *move_details;

    if(is_col(*move)){
        /* Pawn move. */
	class = PAWN_MOVE;
	col = *move;
	move++;
	if(is_rank(*move)){
	    /* e4, e2e4 */
	    rank = *move;
	    move++;
	    if(is_capture(*move)){
		move++;
	    }
	    if(is_col(*move)){
		from_col = col;
		from_rank = rank;
		to_col = *move;
		move++;
		if(is_rank(*move)){
		    to_rank = *move;
		    move++;
		}
	    }
	    else{
		to_col = col;
		to_rank = rank;
	    }
	}
	else{
	    if(is_capture(*move)){
		/* axb */
		move++;
	    }
	    if(is_col(*move)){
		/* ab */
		from_col = col;
		to_col = *move;
		move++;
		if(is_rank(*move)){
		    to_rank = *move;
		    move++;
		}
	    }
	    else{
		print_error_context(GlobalState.logfile);
		fprintf(GlobalState.logfile,"Unknown pawn move %s.\n",move_string);
		Ok = FALSE;
	    }
	}
	if(Ok){
	    /* Look for promotions. */
	    if(*move == '='){
		move++;
	    }
	    if(is_piece(*move)){
		class = PAWN_MOVE_WITH_PROMOTION;
		switch(*move){
		    case 'Q': case 'q':
			 move++;
			 break;
		    case 'R': case 'r':
			 move++;
			 break;
		    case 'N': case 'n':
			 move++;
			 break;
		    case 'B': 
			 /* Lower case 'b' is most likely to be a pawn reference. */
			 move++;
			 break;
		    default:
			 print_error_context(GlobalState.logfile);
			 fprintf(GlobalState.logfile,"Illegal promoted piece %s.\n",move_string);
			 Ok = FALSE;
			 break;
		}
	    }
	}
    }
    else if(is_piece(*move)){
	switch(*move){
	    case 'K': case 'k':
		 class = PIECE_MOVE;
		 break;
	    case 'Q': case 'q':
		 class = PIECE_MOVE;
		 break;
	    case 'R': case 'r':
		 class = PIECE_MOVE;
		 break;
	    case 'N': case 'n':
		 class = PIECE_MOVE;
		 break;
	    case 'B': 
		 /* Lower case 'b' is most likely to be a pawn reference. */
		 class = PIECE_MOVE;
		 break;
	}
	move++;
	if(is_rank(*move)){
	    /* A disambiguating rank.
	     * R1e1, R1xe3.
	     */
	    from_rank = *move;
	    move++;
	    if(is_capture(*move)){
		move++;
	    }
	    if(is_col(*move)){
		to_col = *move;
		move++;
		if(is_rank(*move)){
		    to_rank = *move;
		    move++;
		}
	    }
	    else{
		Ok = FALSE;
		print_error_context(GlobalState.logfile);
		fprintf(GlobalState.logfile,"Unknown piece move %s.\n",move_string);
	    }
	}
	else{
	    if(is_capture(*move)){
		/* Rxe1 */
		move++;
		if(is_col(*move)){
		    to_col = *move;
		    move++;
		    if(is_rank(*move)){
			to_rank = *move;
			move++;
		    }
		    else{
			Ok = FALSE;
			print_error_context(GlobalState.logfile);
			fprintf(GlobalState.logfile,"Unknown piece move %s.\n",move_string);
		    }
		}
		else{
		    Ok = FALSE;
		    print_error_context(GlobalState.logfile);
		    fprintf(GlobalState.logfile,"Unknown piece move %s.\n",move_string);
		}
	    }
	    else if(is_col(*move)){
		col = *move;
		move++;
		if(is_capture(*move)){
		    move++;
		}
		if(is_rank(*move)){
		    /* Re1, Re1d1, Re1xd1 */
		    rank = *move;
		    move++;
		    if(is_capture(*move)){
			move++;
		    }
		    if(is_col(*move)){
			/* Re1d1 */
			from_col = col;
			from_rank = rank;
			to_col = *move;
			move++;
			if(is_rank(*move)){
			    to_rank = *move;
			    move++;
			}
			else{
			    Ok = FALSE;
			    print_error_context(GlobalState.logfile);
			    fprintf(GlobalState.logfile,
					"Unknown piece move %s.\n",move_string);
			}
		    }
		    else{
			to_col = col;
			to_rank = rank;
		    }
		}
		else if(is_col(*move)){
		    /* Rae1 */
		    from_col = col;
		    to_col = *move;
		    move++;
		    if(is_rank(*move)){
			to_rank = *move;
			move++;
		    }
		}
		else{
		    Ok = FALSE;
		    print_error_context(GlobalState.logfile);
		    fprintf(GlobalState.logfile,"Unknown piece move %s.\n",move_string);
		}
	    }
	    else{
		Ok = FALSE;
		print_error_context(GlobalState.logfile);
		fprintf(GlobalState.logfile,"Unknown piece move %s.\n",move_string);
	    }
	}
    }
    else if(is_castling_character(*move)){
	 /* Some form of castling. */
	 move++;
	 /* Allow separators to be optional. */
	 if(*move == '-'){
	     move++;
	 }
	 if(is_castling_character(*move)){
	     move++;
	     if(*move == '-'){
		 move++;
	     }
	     if(is_castling_character(*move)){
		 class = QUEENSIDE_CASTLE;
		 move++;
	     }
	     else{
		 class = KINGSIDE_CASTLE;
	     }
	 }
	 else{
	     print_error_context(GlobalState.logfile);
	     fprintf(GlobalState.logfile,"Unknown castling move %s.\n",move_string);
	     Ok = FALSE;
	 }
    }
    else{
	print_error_context(GlobalState.logfile);
	fprintf(GlobalState.logfile,"Unknown move %s.\n",move_string);
	Ok = FALSE;
    }
    if(Ok){
	/* Allow trailing checks. */
	while(is_check(*move)){
	    move++;
	}
	if((*move == '\0') || (strcmp(move,"ep") == 0) || (strcmp(move,"e.p.") == 0)){
	    /* These are ok. */
	}
	else{
	    Ok = FALSE;
	    print_error_context(GlobalState.logfile);
	    fprintf(GlobalState.logfile,
		"Unknown text trailing move %s <%s>.\n",move_string,move);
	}
    }
    /* Store all of the details gathered, even if the move is illegal. */
    if(!Ok){
	class = UNKNOWN_MOVE;
    }
    move_details = new_move_structure();
    strcpy(move_details->move,move_string);
    move_details->class = class;
    move_details->from_col = from_col;
    move_details->from_rank = from_rank;
    move_details->to_col = to_col;
    move_details->to_rank = to_rank;
    return move_details;
}
