
 Application create: #AuctionDataAccess with:
         (#( Kernel)
             collect: [:each | Smalltalk at: each ifAbsent: [
                 self error: 'Not all of the prerequisites are loaded']])!

 AuctionDataAccess becomeDefault!

 Object subclass: #Address
     instanceVariableNames: 'street postalLocation '
     classVariableNames: ''
     poolDictionaries: ''
 !

 Address comment: 'Keeps track of an address; mainly street and PostalLocation. PostalLocation is a separate
 class since it is more efficient to manage city and state with one instance of PostalLocation than
 require every address to include city and state as well as post code. However, it is a design
 choice to have Address still return the city and state since this then isolates other classes
 from having to look to post code for this information.'!

 Object subclass: #Person
     instanceVariableNames: 'name address '
     classVariableNames: 'SocialSecurityNumbers '
     poolDictionaries: ''
 !

 Person comment: 'Keeps track of general information about people. This is mainly name and address. A
 person can be an auctioneer, a bidder, a seller, or any combination of these. To avoid
 creating duplicate instances for the same person, every person must have a social security
 number specified.
 Keeps track of all instances of Person, indexed by social security number.
 '!

 Object subclass: #PostalLocation
     instanceVariableNames: 'city state postCode '
     classVariableNames: 'PostalLocations '
     poolDictionaries: ''
 !

 PostalLocation comment: 'Manages the post code, state, and city combination. This class returns the state and
 city for a particular post code. A fast indexing between post code and postal location
 information is provided.

 '!

 Application subclass: #AuctionDataAccess
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
 !


 !AuctionDataAccess class publicMethods !

 loaded
         "Set up the PostalLocations."

     PostalLocation
         addPostCode: '27511' forCity: 'Cary' forState: 'NC';
         addPostCode: '78746' forCity: 'Austin' forState: 'TX';
         addPostCode: '49255' forCity: 'Montgomery' forState: 'MI'.!

 removing
         "Remove the PostalLocation instances."

     PostalLocation flushInstances.
 ! !


 !Address class publicMethods !

 street: aString postCode: aPostCodeString
         "Returned an initialized instance of the receiver."

     ^super new
         street: aString;
         postalLocation: (PostalLocation postalLocationFor: aPostCodeString);
         yourself! !


 !Address publicMethods !

 city
         "Return the city for the receiver."

     ^self postalLocation city!

 postalLocation
         "Return the postalLocation for the receiver."

     ^postalLocation!

 postalLocation: aPostalLocation
         "Set the postalLocation for the receiver."

     postalLocation := aPostalLocation!

 postCode
         "Return the post code for the receiver."

     ^self postalLocation postCode!

 postCode: aString
         "Set the post code for the receiver."

     self postalLocation postCode: aString!

 state
         "Return the state for the receiver."

     ^self postalLocation state!

 street
         "Return the street for the receiver."

     ^street!

 street: aStreet
         "Set the street for the receiver."

     street := aStreet! !


 !Person class publicMethods !

 addPerson: aPerson withSocialSecurityNumber: socialSecurityNumber
         "Add a new social security number for the specified person.
          If the socialSecurityNumber already exists, do nothing.
          Parameter expected:
             socialSecurityNumber : a number."

     self socialSecurityNumbers at: socialSecurityNumber ifAbsentPut: [aPerson]
 !

 personFor: socialSecurityNumber
         "Return the person with this socialSecurityNumber.
          Return the Person if socialSecurityNumber is found.
          Return nil if socialSecurityNumber is not found.
          Parameter expected:
             socialSecurityNumber : a number."

     ^self socialSecurityNumbers at: socialSecurityNumber ifAbsent: [nil]
 !

 removePersonWith: socialSecurityNumber
         "Remove the person for this social security number.
          Return the removed person.
          Return nil if this socialSecurityNumber is not found.
          Parameter expected:
             socialSecurityNumber : a number."

     ^self socialSecurityNumbers removeKey: socialSecurityNumber ifAbsent: [nil]
 !

 socialSecurityNumberFor: aPerson
         "Return the social security number for this person.
          Return nil the person does not have a social security number."

     ^self socialSecurityNumbers keyAtValue: aPerson ifAbsent: [nil]
 !

 socialSecurityNumbers
         "Return the social security numbers.
          If SocialSecurityNumbers is nil, set it to a new dictionary."

     SocialSecurityNumbers isNil ifTrue: [SocialSecurityNumbers := Dictionary new].
     ^SocialSecurityNumbers! !


 !Person publicMethods !

 address
         "Return the address for the receiver."

     ^address!

 address: anAddress
         "Set the address for the receiver."

     address := anAddress!

 name
         "Return the name for the receiver."

     ^name!

 name: aString
         "Set the name for the receiver."

     name := aString!

 socialSecurityNumber
         "Return the social security number for the receiver. This information
          is found by querying the SocialSecurityNumbers dictionary.
          The value is an instance of Person and the key is the person's
          social security number."

     ^self class socialSecurityNumbers keyAtValue: self! !


 !PostalLocation class publicMethods !

 addPostCode: postCode forCity: city forState: state
         "Create a new PostalLocation and add it to the list.
          Parameters expected:
                 postCode : a String
                 city : a String
                 state : a String.
          Return the new PostalLocation."

     ^self postalLocations add:
         (PostalLocation new
                 city: city;
                 state: state;
                 postCode: postCode;
                 yourself)!

 postalLocationFor: postCode
         "Return a PostalLocation for the specified postCode.
          If postCode is invalid, return a new PostalLocation.
          Parameters expected:
                 postCode : a String."

     ^self postalLocations at: postCode ifAbsent: [PostalLocation new]!

 postalLocations
         "Return the postal locations.
          If nil, initialize to a new dictionary."

     PostalLocations isNil ifTrue: [PostalLocations := Dictionary new].
     ^PostalLocations! !

 !PostalLocation class privateMethods !

 flushInstances
         "Private - Initialize PostalLocations so that all instances
          of PostalLocation are removed."

     PostalLocations := Dictionary new! !


 !PostalLocation publicMethods !

 = aPostalLocation
     "Return true if the receiver is the same kind of object
     and has the same key as aPostalLocation."

     ^self species == aPostalLocation species and:
         [self key = aPostalLocation key]!

 city
         "Return the city for the receiver."

     ^city!

 hash
     "Return a hash value that is based upon the same
     information used to test equality."

     ^self key hash!

 key
         "Return the information that uniquely identifies
         the receiver."

     ^postCode!

 postCode
         "Return the post code for the receiver."

     ^postCode!

 state
         "Return the state for the receiver."

     ^state!

 value
         "Return the entire receiver."! !

 !PostalLocation privateMethods !

 city: aCity
         "Set the city for the receiver."

     city := aCity!

 postCode: aString
         "Set the post code for the receiver."

     postCode := aString!

 state: aState
         "Set the state for the receiver."

     state := aState! !

 AuctionDataAccess loaded.!

 Application create: #AuctionDomain with:
         (#( AuctionDataAccess)
             collect: [:each | Smalltalk at: each ifAbsent: [
                 self error: 'Not all of the prerequisites are loaded']])!

 AuctionDomain becomeDefault!

 Object subclass: #Auction
     instanceVariableNames: 'contracts date address salesTaxRate auctioneer bidders bidderID auctionItemID '
     classVariableNames: 'Auctions '
     poolDictionaries: ''
 !

 Auction comment: 'Keeps track of the information of an auction. This includes such information as the
 auctioneer, location, date, sales tax rate, bidders, and a list of contracts.

 Also responsible for assigning a unique ID to each item and each bidder
 for each auction.
 '!

 Object subclass: #AuctionItem
     instanceVariableNames: 'description id buyer sellingPrice advancedBids reservedBid '
     classVariableNames: ''
     poolDictionaries: ''
 !

 AuctionItem comment: 'Keeps track of information about items to be sold in an auction. This includes its
 description, unique ID, selling price, and bidder that bought the item.
 '!

 Object subclass: #Bidder
     instanceVariableNames: 'person number itemsBought '
     classVariableNames: ''
     poolDictionaries: ''
 !

 Bidder comment: 'Keeps track of name and address (kept by an instance of Person), the bidder''s card number, the list of items bought,
 and the calculation of total costs associated with the bought items.
 A Person can be a bidder and a seller and an auctioneer.
 '!

 Object subclass: #Contract
     instanceVariableNames: 'seller items commission '
     classVariableNames: ''
     poolDictionaries: ''
 !

 Contract comment: 'Manages the agreement between an auctioneer and a seller. There is an instance per
 auction per seller. It keeps track of the seller, the commission, and a list of items to be
 sold.
 '!

 Application subclass: #AuctionDomain
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
 !


 !Auction class publicMethods !

 addAuction
         "Create a new auction and add it to the list of auctions.
          Return the new auction."

     ^self auctions addLast: self new
 !

 auctions
         "Return the list of auctions. Initialize the list if it is nil."

     Auctions isNil ifTrue: [Auctions := OrderedCollection new].
     ^Auctions!

 auctionsFor: aPerson
         "Return a collection of auctions that are for a particular auctioneer."

     ^self auctions select: [:anAuction | anAuction auctioneer == aPerson]
 !

 new
         "Returned an initialized instance of the receiver."

     ^super new initialize!

 removeAuction: anAuction
         "Remove an auction from the list of auctions.
          Return the auction that was removed, or
          return nil if the auction is not in the list"

     ^self auctions remove: anAuction ifAbsent: [nil]! !


 !Auction publicMethods !

 addBidder: aPerson
         "Add a bidder to the list. A bidder cannot exist without belonging to
          an auction. This method creates a new instance of bidder for aPerson.

          Return the new bidder."

     ^self bidders addLast:
         (Bidder new
             person: aPerson;
             number: self assignBidderID;
             yourself)
         !

 addContract: aContract
         "Add a contract to the list."

     self contracts addLast: aContract!

 address
         "Return the address for the receiver."

     ^address!

 address: anAddress
         "Set the address for the receiver."

     address := anAddress!

 allItems
         "Return the list of all items to be sold in the receiver,
          regardless of the seller. "

     | theList |
     theList := OrderedCollection new.
     self contracts do: [:aContract | theList addAll: aContract items].
     ^theList!

 allItemsNotSold
         "Return a collection of all items that were not
          sold in the receiver for all of the sellers."

     ^self contracts collect: [:aContract |
         aContract items reject: [:item | item isSold]
      ]!

 allItemsSold
         "Return a collection of all items that were not
          sold in the receiver for all of the sellers."

     ^self contracts collect: [:aContract |
         aContract items select: [:item | item isSold]
      ]
 !

 assignAuctionItemID
         "Increment the auctionItemID by one and return the new value."

     self auctionItemID: self auctionItemID + 1.
     ^self auctionItemID
         !

 assignBidderID
         "Increment the bidderID by one and return the new value."

     self bidderID: self bidderID + 1.
     ^self bidderID
         !

 auctioneer
         "Return the auctioneer (an instance of Person) for the receiver."

     ^auctioneer!

 auctioneer: aPerson
         "Set the auctioneer for the receiver."

     auctioneer := aPerson!

 auctionItemID
         "Return the current auctionItemID for the receiver.
          Each auction item must have a unique ID in the receiver."

     ^auctionItemID!

 bidderID
         "Return the current bidderID for the receiver.
          Each bidder must have a unique ID in the receiver."

     ^bidderID!

 bidders
         "Return the list of bidders for the receiver.
          Initialize if bidders is nil."

     bidders isNil ifTrue: [bidders := OrderedCollection new].
     ^bidders!

 contracts
         "Return the list of contracts for the receiver.
          Initialize if contracts is nil."

     contracts isNil ifTrue: [contracts := OrderedCollection new].
     ^contracts!

 date
         "Return the date for the receiver."

     ^date!

 date: aDate
         "Set the date for the receiver."

     date := aDate!

 initialize
         "Initialize the data for the receiver."

     self date: Date today.
     self address: Address new.
     self salesTaxRate: 0.0.
     self auctioneer: Person new.
     self bidderID: 0.
     self auctionItemID: 0.  !

 removeBidder: aBidder
         "Remove a bidder from the list of bidders.
          Return nil if the bidder is not in the list.
          Otherwise, return the bidder that was removed."

     ^self bidders remove: aBidder ifAbsent: [nil]!

 removeContract: aContract
         "Remove a contract from the list of contracts.
          Return nil if the contract is not in the list.
          Otherwise, return the contract that was removed."

     ^self contracts remove: aContract ifAbsent: [nil]!

 salesTaxForBidder: aBidder
         "Compute and return the sales tax for aBidder."

     ^aBidder costOfItems * self salesTaxRate!

 salesTaxRate
         "Return the sales tax rate for the receiver."

     ^salesTaxRate!

 salesTaxRate: aRate
         "Set the sales tax rate for the receiver.
          Parameter expected:
             aRate : a number between 0 and 1."

     salesTaxRate := aRate!

 totalCostForBidder: aBidder
         "Return the total cost for aBidder including sales tax.
          The sales tax rate is specified between 0 and 1, not as a percentage.

          The total is computed by adding the cost of the items to the
          amount of sales tax on the cost of items. This equation is factored
          to avoid having to retrieve the cost of items twice."

     ^aBidder costOfItems * (1 + self salesTaxRate)! !

 !Auction privateMethods !

 auctionItemID: aNumber
         "Private - Set the current auctionItemID for the receiver."

     auctionItemID := aNumber!

 bidderID: aNumber
         "Private - Set the current bidderID for the receiver."

     bidderID := aNumber! !


 !AuctionItem class publicMethods !

 new
         "Returned an initialized instance of the receiver."

     ^super new initialize! !


 !AuctionItem publicMethods !

 addAdvancedBid: anAmount for: aBidder
         "Add an advanced bid to the Dictionary with aBidder being the key
          and anAmount being the value. The bidder is the key since keys have
          to be unique and a bidder can have only one advanced bid for an item.

          Parameter expected:
             anAmount : a number."

     self advancedBids at: aBidder put: anAmount!

 advancedBidForBidder: aBidder
         "Return the amount of the advanced bid for aBidder.
          If aBidder does not have an advanced bid for the receiver, return nil."

     ^self advancedBids at: aBidder ifAbsent: [nil].!

 advancedBidGreaterThan: anAmount
         "Return the association for the highest advanced bid if
          the amount is greater than anAmount. Otherwise return nil.
          Parameter expected:
             anAmount : a number."

     | advancedBid |
     advancedBid := self highestAdvancedBid.
     ^advancedBid value > anAmount
         ifTrue: [advancedBid]
         ifFalse: [nil].!

 advancedBids
         "Return the advanced bids for the receiver. This allows multiple advanced
          bids even though the highest advanced bid is the only valid advanced bid."

     advancedBids isNil ifTrue: [advancedBids := Dictionary new].
     ^advancedBids!

 buyer
         "Return the buyer (an instance of Bidder) for the receiver."

     ^buyer!

 description
         "Return the description for the receiver."

     ^description!

 description: aString
         "Set the description for the receiver."

     description := aString!

 highestAdvancedBid
         "Return the highest advanced bid in the list. In the association returned,
          the key is the bidder and the value is the amount.
          If the key is nil or the value is zero, then no advanced bids were bid."

     | highestBid |
     highestBid := Association key: nil value: 0.
     self advancedBids associationDo: [:anAssociation |
         anAssociation value > highestBid value
             ifTrue: [highestBid := anAssociation]
     ].
     ^highestBid!

 id
         "Return the id for the receiver."

     ^id!

 id: anID
         "Set the id for the receiver.
          Parameter expected:
             anID : a number."

     id := anID!

 initialize
         "Initialize the data for the receiver."

     self description: String new.!

 isAdvancedBidGreaterThan: anAmount
         "Return true if the advanced bid is greater than the amount.
          Otherwise return false. Defaults to false if there is no
          advanced bid.
          Parameter expected:
             anAmount : a number."

     ^self highestAdvancedBid value > anAmount
 !

 isReserveBidHigherThan: anAmount
         "Return true if the reserve bid is greater than a specified amount.
          Otherwise return false.
          Parameter expected:
             anAmount : a number."

     ^self reservedBid > anAmount
 !

 isSold
         "Return true if the receiver has been sold.
          Otherwise return false."

     ^self sellingPrice notNil!

 removeAdvancedBidForBidder: aBidder
         "Remove an advanced bid from the list for aBidder."

     ^self advancedBids remove: aBidder ifAbsent: [nil]!

 reservedBid
         "Return the reserved bid for the receiver."

     ^reservedBid!

 reservedBid: anAmount
         "Set the reserved bid for the receiver.
          Parameter expected:
             anAmount : a number."

     reservedBid := anAmount!

 sellingPrice
         "Return the selling price for the receiver. If this value is nil,
          the receiver has not been sold."

     ^sellingPrice!

 sellingPrice: anAmount buyer: aBidder
         "Set the selling price and the buyer for the receiver.
          These two items should always be set together.
          Parameter expected:
             anAmount : a number."

     self
         sellingPrice: anAmount;
         buyer: aBidder! !

 !AuctionItem privateMethods !

 buyer: aBidder
         "Private - Set the buyer for the receiver."

     buyer := aBidder!

 sellingPrice: anAmount
         "Private - Set the selling price for the receiver.
          Parameter expected:
             anAmount : a number."

     sellingPrice := anAmount! !


 !Bidder class publicMethods !

 new
         "Returned an initialized instance of the receiver."

     ^super new initialize! !


 !Bidder publicMethods !

 addBoughtItem: anAuctionItem
         "Add anAuctionItem that was bought by the receiver to the list."

     self itemsBought add: anAuctionItem!

 address
         "Return the address for the receiver."

     ^self person address!

 costOfItems
         "Return the total cost of the bought items without any sales
          tax added. This method assumes that if an item is added to this list,
          it has a valid selling price."

     | totalCost |
     totalCost := 0.
     self itemsBought do: [ :anItem |
         totalCost := totalCost + anItem sellingPrice].
     ^totalCost!

 initialize
         "Initialize the data for the receiver."

     self person: Person new.
     self number: 0!

 itemsBought
         "Return the list of items bought.
          If the list is nil, initialize it to an OrderedCollection."

     itemsBought isNil
         ifTrue: [itemsBought := OrderedCollection new].
     ^itemsBought!

 name
         "Return the name for the receiver."

     ^self person name!

 number
         "Return the number for the receiver."

     ^number!

 number: aNumber
         "Set the number for the receiver. It is up to the sender of this message
          to make sure the number is unique for the auction."

     number := aNumber!

 person
         "Return the person for the receiver."

     ^person!

 person: aPerson
         "Set the person for the receiver."

     person := aPerson!

 removeBoughtItem: anAuctionItem
         "Remove anAuctionItem from the list. If the item is not in the list,
          return nil. Otherwise, return anAuctionItem."

     ^self itemsBought remove: anAuctionItem ifAbsent: [nil].! !


 !Contract publicMethods !

 addItem: anAuctionItem
         "Add anAuctionItem to the list of items for the receiver.
          If anAuctionItem is already in the list, do nothing."

     (self items includes: anAuctionItem)
         ifFalse: [self items addLast: anAuctionItem]!

 commission
         "Return the commission for the receiver."

     ^commission!

 commission: aCommission
         "Set the commission for the receiver.
          Parameter expected:
             aCommission : a number between 0 and 1."

     commission := aCommission!

 items
         "Return the list of items for the receiver."

     items isNil ifTrue: [items := OrderedCollection new].
     ^items!

 items: aCollection
         "Set the list of items for the receiver to a collection."

     items := aCollection!

 removeItem: anAuctionItem
         "Remove an item from the list of items. Return nil if the item is not
          in the list. Otherwise return anAuctionItem."

     ^self items remove: anAuctionItem ifAbsent: [nil]!

 seller
         "Return the seller for the receiver."

     ^seller!

 seller: aSeller
         "Set the seller for the receiver."

     seller := aSeller! !


 Application create: #AuctionTest with:
         (#( AuctionDomain)
             collect: [:each | Smalltalk at: each ifAbsent: [
                 self error: 'Not all of the prerequisites are loaded']])!

 AuctionTest becomeDefault!

 Application subclass: #AuctionTest
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
 !

 AuctionTest comment: 'Test methods for the Auction class and application.'!


 !AuctionTest class publicMethods !

 runAuction1
         "Set up Auction #1"
         "AuctionTest runAuction1"

     | auction |
     "Create the auction"
     auction := Auction new
         address: Address address1;
         auctioneer:
             (Person new
                 name: 'Robin Smith';
                 address: Address address2);
         salesTaxRate: 7.0;
         yourself.

     "Add the bidders"
     auction
         addBidder: Person person1;
         addBidder: Person person2;
         addBidder: Person person3;
         addBidder: Person person4.

     "Add the items for sale"

     ^auction!

 runAuction2
         "Set up Auction #2"
         "AuctionTest runAuction2"

     | auction contract |
     "Create the auction"
     auction := Auction new
         address: Address address2;
         auctioneer:
             (Person new
                 name: 'T. Jones';
                 address: Address address3);
         salesTaxRate: 8.0;
         yourself.

     "Add the bidders"
     auction
         addBidder: Person person1;
         addBidder: Person person2;
         addBidder: Person person3;
         addBidder: Person person4.

     "Add the contract with the items for sale"
     auction addContract: (contract := Contract new).
     contract
         commission: 3.0;
         seller: 'H. Ball';
         addItem: (AuctionItem item1 id: auction assignAuctionItemID);
         addItem: (AuctionItem item2 id: auction assignAuctionItemID);
         addItem: (AuctionItem item3 id: auction assignAuctionItemID);
         addItem: (AuctionItem item4 id: auction assignAuctionItemID).

     ^auction! !


 !Address class publicMethods !

 address1
         "Return an instance of the receiver."
         "Address address1"

     ^Address
         street: 'Fisher Ave.'
         postCode: '49255'!

 address2
         "Return an instance of the receiver."
         "Address address2"

     ^Address
         street: 'Baseline Ave.'
         postCode: '49255'!

 address3
         "Return an instance of the receiver."
         "Address address3"

     ^Address
         street: 'Knoxdale Ave.'
         postCode: '78746'  !

 address4
         "Return an instance of the receiver."
         "Address address4"

     ^Address
         street: 'Exeter Blvd.'
         postCode: '78746'!

 address5
         "Return an instance of the receiver."
         "Address address5"

     ^Address
         street: 'Merivale Ave.'
         postCode: '78746'!

 address6
         "Return an instance of the receiver."
         "Address address6"

     ^Address
         street: 'Clyde Ave.'
         postCode: '49255'! !


 !AuctionItem class publicMethods !

 item1
         "Return an instance of the receiver."
         "AuctionItem item1"

     ^AuctionItem new
         description: 'Good old fashion rocking chair'!

 item2
         "Return an instance of the receiver."
         "AuctionItem item2"

     ^AuctionItem new
         description: 'Solid oak pantry'!

 item3
         "Return an instance of the receiver."
         "AuctionItem item3"

     ^AuctionItem new
         description: '18th century lantern'!

 item4
         "Return an instance of the receiver."
         "AuctionItem item4"

     ^AuctionItem new
         description: '1920 radio'!

 item5
         "Return an instance of the receiver."
         "AuctionItem item5"

     ^AuctionItem new
         description: 'Lime green french provincial couch'!

 item6
         "Return an instance of the receiver."
         "AuctionItem item6"

     ^AuctionItem new
         description: '10 piece mahogany dining room set'! !


 !Person class publicMethods !

 person1
         "Return an instance of the receiver."
         "Person person1"

     ^Person new
         name: 'Robin Smith';
         address: (Address
                     street: 'Holmwood Ave'
                     postCode: '27511');
         yourself
 !

 person2
         "Return an instance of the receiver."
         "Person person2"

     ^Person new
         name: 'Carmen LeDuke';
         address: (Address
                     street: 'Bank St.'
                     postCode: '27511');
         yourself
 !

 person3
         "Return an instance of the receiver."
         "Person person3"

     ^Person new
         name: 'I. StandTall';
         address: (Address
                     street: 'Bronson Ave.'
                     postCode: '27511');
         yourself
 !

 person4
         "Return an instance of the receiver."
         "Person person4"

     ^Person new
         name: 'Joey Jones';
         address: Address address5;
         yourself! !


 !PostalLocation class publicMethods !

 addInvalidPostalLocation
         "Add an invalid post code."
         "PostalLocation addInvalidPostalLocation"

     ^PostalLocation addPostCode: '78ui7ji' forCity: 'xx' forState: 'anycity' !


 addPostCode27511
         "Add a post code for 27511."

     ^PostalLocation addPostCode: '27511' forCity: 'Cary' forState: 'NC' !

 addPostCode49255
         "Add a post code for 49255."

     ^PostalLocation addPostCode: '49255' forCity: 'Montogomery' forState: 'MI' !

 addPostCode78746
         "Add a post code for 78746."

     ^PostalLocation addPostCode: '78746' forCity: 'Austin' forState: 'TX' !

 addPostCodes
         "Add and return all of the postal locations."
         "PostalLocation addPostCodes"

     ^self
         addPostCode27511;
         addPostCode49255;
         addPostCode78746;
         postCodes
                 !

 invalidPostalLocationFor11111
         "Get the PostalLocation for post code 11111 which is
          known to be invalid."
         "PostalLocation invalidPostalLocationFor11111"

     ^PostalLocation postalLocationFor: '11111'!

 listOfLocations
     "Return a Collection of printable representations of all the known post codes."

     ^self postalLocations asArray collect: [:postalLocation |
         postalLocation printString]!

 postCodes
         "Return all of the current post codes."

     ^PostalLocation postalLocations asArray collect: [:postalLocation |
         postalLocation postCode]!

 postLocationFor27511
         "Get the PostalLocation for 27511."

     ^PostalLocation postalLocationFor: '27511'!

 postLocationFor49255
         "Get the PostalLocation for 49255."

     ^PostalLocation postalLocationFor: '49255'!

 postLocationFor78746
         "Get the PostalLocation for 78746."

     ^PostalLocation postalLocationFor: '78746'!

 randomPostalLocation
     "Return a random post code."

     ^PostalLocations values asArray
         at: (Random new next * PostalLocations size) truncated + 1! !


 Application create: #AuctionUI with:
         (#( AuctionDomain)
             collect: [:each | Smalltalk at: each ifAbsent: [
                 self error: 'Not all of the prerequisites are loaded']])!

 AuctionUI becomeDefault!

 Application subclass: #AuctionUI
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
 !


 !Address publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         space;
         nextPutAll: self street;
         cr.

     self postalLocation printOn: aStream ! !


 !AuctionItem publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         nextPutAll: 'ID: ';
         nextPutAll: self id printString;
         cr;
         nextPutAll: self description;
         cr.
     self buyer notNil
         ifTrue: [
             aStream
                 nextPutAll: 'Buyer: ';
                 cr.
             self buyer printOn: aStream.
             aStream
                 nextPutAll: 'Selling price: $.';
                 nextPutAll: self sellingPriced printString
         ].

     ! !


 !Bidder publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         nextPutAll: 'Bidder(';
         cr;
         nextPutAll: self person printString;
         nextPutAll: 'ID: ';
         nextPutAll: self number printString;
         cr;
         nextPutAll: 'Items bought:';
         cr.

     self itemsBought asArray printOn: aStream.

     aStream
         nextPut: $);
         cr! !


 !Contract publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         nextPutAll: 'Seller: ';
         cr;
         nextPutAll: self seller;
         nextPutAll: ' with commission: ';
         nextPutAll: self commission printString;
         cr;
         nextPutAll: 'Items:';
         cr.

     self items asArray printOn: aStream.! !


 !Person publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         nextPut: $(;
         nextPutAll: self name;
         cr.

     self address printOn: aStream.

     aStream
         nextPut: $);
         cr! !


 !PostalLocation publicMethods !

 printOn: aStream
         "Print the receiver onto a Stream."

     aStream
         space;
         nextPutAll: self city;
         space;
         nextPutAll: self state;
         space;
         nextPutAll: self postCode           ! !
