/*
	This is a DMapEdit source code module.  Though it is copyrighted, you
	may modify it and use it for your own personal use, meaning that new
	modified code and anything derived from it (such as exe files) doesn't
	get distributed to anyone, unless you get my permission first.  Code
	from this file, or code based on ideas from this file may be used with
	other programs, provided that you give credit for it in the source code,
	documentation, and 'about' windows or screens, if one exists, for the
	programs using it.  Giving credit means something like this:

	Code from DMapEdit was used in this program

							  or

	Some code for this program was based on ideas presented in DMapEdit

	Whatever.  Just be sure to mention "DMapEdit" in such a way that it's
	self-evident how it was useful to the new program, and be sure to have
	"DMapEdit is a trademark of Jason Hoffoss" in the docs.  That's all..
*/

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <graphics.h>
#include <string.h>
#include <math.h>
#include <alloc.h>
#include <limits.h>
#include <time.h>

#include "dme.h"
#include "dme2.h"

#define VL_MAX 2048

void check_vertexes(void);
void check_things(void);
void add_node_list(int far *list, int *index, int line, int v1, int v2);
void remove_vertexes(void);
uint nodize(uint size, int far *list);
int resolve_space(int line, int size, int far *list,
	int far *left_list, int far *right_list, int *lx1, int *ly1,
	int *lx2, int *ly2, int *rx1, int *ry1, int *rx2, int *ry2, int test);
void add_node_list2(int far *list, int index, int line, int v1, int v2,
	int *x1, int *y1, int *x2, int *y2);
int make_seg(int line, int v1, int v2);

int b_max=0;
int optimize; /* method of optimizing node generation */
int v_size_backup;
int seg_orig;
int i, j; /* general loop indexing variables */
int nodeline; /* current nodeline */
int convex; /* set if polygon is convex */
int err; /* error code */
uint lcount; /* # of lines to the left of nodeline */
uint rcount; /* # of lines to the right of nodeline */
int bal; /* difference between lcount and rcount (0 = perfect balance) */
int splits; /* # of lines nodeline splits */
int left_size; /* lcount of best case */
int right_size; /* rcount of best case */
int balance; /* best case balance */
int splits_min; /* best case of minimum splits */
int simple_splits; /* total number of simple splits */
int complex_splits; /* total number of complex splits */
int *slivers; /* list to track complex splits (cause of slivers) */
uint link2; /* node/ssec left side link */
char test_msg[80];
time_t start, end;

void misc_options(void)
{
	int button;

	while (1)
	{
		window_text1("Misc. functions\t\n"
			" @ Error-check Vertexes\n"
			" @ Error-check Lines/Sidedefs \n"
			" @ Error-check Things\n"
			" @ Error-check Sectors\n\n"
			" @ Fix lines\n"
			" @ Generate Sectors\n"
			" @ Fix Sectors\n"
			" @ Generate Nodes\n"
			" @ Generate Blockmap\n\n", 1, 0, 0);
		b.pos[3].status = -1;
		b.pos[6].status = -1;
		draw_buttons();
		set_cancel_bar();

		if ((button = window_check()) < 0)
			return;

		switch(button)
		{
			case 0:
				check_vertexes();
				break;
			case 1:
				check_lines();
				break;
			case 2:
				check_things();
				break;
			case 4:
				fixup_lines();
				break;
			case 5:
				generate_sectors();
				break;
			case 6:
				fixup_sectors();
				break;
			case 7:
				generate_nodes();
				break;
			case 8:
				generate_blockmap();
				break;
		}
		draw_map();
	}
}

void check_things(void)
{
	char msg[1024], temp[80];
	int i, j, k, xx1, xx2, err1=15, err2=0, dm_entry=0;
	ulong total, count, count2;

	if (!t_size)
	{
		error("There are no Things to check!");
		return;
	}

	window_text("Checking for Thing errors..\n\n\n", 1);
	xx1 = draw_time_graph(win.top + 20);

	count = count2 = 0;
	total = t_size;
	for (i=0; i<t_size; i++)
	{
		if (things[i].type == 1)
		{
			if (!(err1 & 1))
				err1 |= 16;
			err1 &= ~1;
		}

		if (things[i].type == 2)
		{
			if (!(err1 & 2))
				err1 |= 32;
			err1 &= ~2;
		}

		if (things[i].type == 3)
		{
			if (!(err1 & 4))
				err1 |= 64;
			err1 &= ~4;
		}

		if (things[i].type == 4)
		{
			if (!(err1 & 8))
				err1 |= 128;
			err1 &= ~8;
		}

		if (things[i].type == 11)
			dm_entry++;
	}

	if (err1 || dm_entry < 4)
	{
		strcpy(msg, "Results\t\n");
		if (err1 & 1)
			strcat(msg, "Missing player #1 starting point\n");
		if (err1 & 2)
			strcat(msg, "Missing player #2 starting point\n");
		if (err1 & 4)
			strcat(msg, "Missing player #3 starting point\n");
		if (err1 & 8)
			strcat(msg, "Missing player #4 starting point\n");
		if (err1 & 16)
			strcat(msg, "Too many player #1 starting points\n");
		if (err1 & 32)
			strcat(msg, "Too many player #2 starting points\n");
		if (err1 & 64)
			strcat(msg, "Too many player #3 starting points\n");
		if (err1 & 128)
			strcat(msg, "Too many player #4 starting points\n");
		if (dm_entry < 4)
		{
			sprintf(temp, "Only %d of 4 deathmatch entry points found\n",
				dm_entry);
			strcat(msg, temp);
		}
		window_text(msg, 1);

	} else
		window_text("Results\t\nNo errors found\n", 1);

	while (mouse_check());
	while (!mouse_check())
		if (keypress)
			break;

	await_release();
	return;
}

void check_vertexes(void)
{
	char msg[256], temp[60];
	int i, j, k, xx1, xx2, err1=0, err2=0;
	ulong total, count, count2;

	if (!max_vertex)
	{
		error("There are no Vertexes to check!");
		return;
	}

	window_text("Checking for Vertex errors..\n\n\n", 1);
	xx1 = draw_time_graph(win.top + 20);

	count = count2 = 0;
	total = summation(max_vertex) + (ulong) max_vertex * l_size;
	for (i=0; i<max_vertex; i++)
	{
		for (j=i+1; j<max_vertex; j++)
		{
			if (vertexes[i].x == vertexes[j].x &&
				vertexes[i].y == vertexes[j].y)
			{
				err1++;
				for (k=0; k<l_size; k++)
				{
					if (linedefs[k].v1 == j)
						linedefs[k].v1 = i;
					if (linedefs[k].v2 == j)
						linedefs[k].v2 = i;
				}
				del_vertex(j);
			}
			count++;
		}

		if ((count - count2) > 1024)
		{
			time_graph(count, total, xx1, win.top + 20);
			count2 = count;
		}

		for (j=k=0; j<l_size; j++)
		{
			if (linedefs[j].v1 == i)
				k++;
			if (linedefs[j].v2 == i)
				k++;
			count++;
		}
		if (!k)
		{
			err2++;
			del_vertex(i--);
		}

		if ((count - count2) > 1024)
		{
			time_graph(count, total, xx1, win.top + 20);
			count2 = count;
		}
	}

	if (err1 || err2)
	{
		strcpy(msg, "Results\t\n");
		if (err1)
		{
			sprintf(temp, "%d overlapping vertexes (fixed)\n", err1);
			strcat(msg, temp);
		}

		if (err2)
		{
			sprintf(temp, "%d unused vertexes (removed)\n", err2);
			strcat(msg, temp);
		}
		window_text(msg, 1);

	} else
		window_text("Results\t\nNo errors found\n", 1);

	while (!mouse_check())
		if (keypress)
			break;

	await_release();
	return;
}

void del_vertex2(int num) /* delete unused vertex */
{
	int i;

	for (i=num; i<v_size-1; i++)
		vertexes[i] = vertexes[i+1];

	for (i=0; i<l_size; i++)
	{
		if (linedefs[i].v1 > num)
			linedefs[i].v1--;
		if (linedefs[i].v2 > num)
			linedefs[i].v2--;
	}

	for (i=0; i<seg_size; i++)
	{
		if (segs[i].v1 > num)
			segs[i].v1--;
		if (segs[i].v2 > num)
			segs[i].v2--;
	}

	v_size--;
	max_vertex--;
	return;
}

void check_lines(void)
{
	char msg[4096], temp[60];
	int i, j, xx1, xx2, xx3, num, v1, v2, v3, v4, sd1, sd2, total1, total2;
	uint err;
	ulong count, count2, total, err1, err2, err3, err4, err5;

	if (!l_size)
	{
		error("There are no Lines to check!");
		s_size = 0; /* shouldn't be any sidedefs if no lines */
		return;
	}
	if (!s_size)
		error("No Sidedefs!");

	window_text("Checking for Line/Sidedef errors..\n\n\n\n", 1);
	xx1 = draw_time_graph(win.top + 20);
	xx2 = draw_time_graph(win.top + 29); /*
	xx3 = draw_time_graph(win.top + 38); */
	err1 = err2 = err3 = err4 = err5 = 0;

re_check1:
	err = count = count2 = 0;
	total = summation(l_size);
	/* first check is overlapping lines */
	for (i=0; i<l_size; i++)
	{
		v1 = linedefs[i].v1;
		v2 = linedefs[i].v2;
		if (v1 == v2)
		{
			err5++;
			if (err == LL_MAX) /* too many errors to track */
			{
				for (i=0; i<err; i++)
					del_line(llist[i]); /* fix and try again */
				reset_time_graph(count, total, xx1, win.top + 20);
				goto re_check1;
			}
			count += l_size - i - 1;
			continue; /* no need to check further */
		}

		for (j=i+1; j<l_size; j++) /* i and lower already checked */
		{
			v3 = linedefs[j].v1;
			v4 = linedefs[j].v2;
			if ((v1 == v3 && v2 == v4) || (v1 == v4 && v2 == v3))
			{
				err1++;
				if (err == LL_MAX) /* too many errors to track */
				{
					for (i=0; i<err; i++)
						del_line(llist[i]); /* fix and try again */
					reset_time_graph(count, total, xx1, win.top + 20);
					goto re_check1;
				}
				llist[err++] = i;
			}
			count++;
		}

		if ((count - count2) > 1024)
		{
			time_graph(count, total, xx1, win.top + 20);
			count2 = count;
		}

		if ((sd2 = linedefs[i].sd2) != -1 && linedefs[i].sd1 == -1)
		{ /* if only 1 sidedef, must be on right */
			v1 = linedefs[i].v1;
			linedefs[i].v1 = linedefs[i].v2;
			linedefs[i].v2 = v1;

			linedefs[i].sd1 = sd2;
			linedefs[i].sd2 = -1;
			err4++;
		}

		if ((sd1 = linedefs[i].sd1) != -1 && sd2 != -1)
		{ /* sidedef with most textures must also be on the right */
			total1 = total2 = 0;

			if (sidedefs[sd1].top[0] != '-')
				total1++;
			if (sidedefs[sd1].middle[0] != '-')
				total1++;
			if (sidedefs[sd1].bottom[0] != '-')
				total1++;

			if (sidedefs[sd2].top[0] != '-')
				total2++;
			if (sidedefs[sd2].middle[0] != '-')
				total2++;
			if (sidedefs[sd2].bottom[0] != '-')
				total2++;

			if (total2 > total1)
			{
				v1 = linedefs[i].v1;
				linedefs[i].v1 = linedefs[i].v2;
				linedefs[i].v2 = v1;

				linedefs[i].sd1 = sd2;
				linedefs[i].sd2 = sd1;
				err4++;
			}
		}
	}
	time_graph(count, total, xx1, win.top + 20);
	for (i=0; i<err; i++)
		del_line(llist[i]);

	if (!s_size) /* skip sidedef checks */
		goto re_check2;

	err = 0;
	count = count2 = 0;
	/* second check is for sidedef used only once. */
	for (i=0; i<s_size; i++)
	{
		num = 0;
		for (j=0; j<l_size; j++, count++)
		{
			if (linedefs[j].sd1 == i)
			{
				if (num)
				{
					err2++; /* sidedef already used */
					linedefs[j].sd1 = add_sidedef(i);
				}
				num = 1;
			}
			if (linedefs[j].sd2 == i)
			{
				if (num)
				{
					err2++; /* sidedef already used */
					linedefs[j].sd2 = add_sidedef(i);
				}
				num = 1;
			}
		}
		if (!num)
		{
			err3++; /* sidedef wasn't used */
			del_sidedef(i--);
			count -= l_size;
			count2 -= l_size;
		}
		if ((count - count2) > 1024)
		{
			time_graph(count, (ulong) s_size * l_size, xx2, win.top + 29);
			count2 = count;
		}
	}

re_check2:

	if (err1 || err2 || err3 || err4 || err5)
	{
		strcpy(msg, "Results\t\n");
		if (err1)
		{
			sprintf(temp, "%d overlapping lines (fixed)\n", err1);
			strcat(msg, temp);
		}

		if (err2)
		{
			sprintf(temp, "%d overused sidedefs (fixed)\n", err2);
			strcat(msg, temp);
		}

		if (err3)
		{
			sprintf(temp, "%d unused sidedefs (removed)\n", err3);
			strcat(msg, temp);
		}

		if (err4)
		{
			sprintf(temp, "%d lines needed inverting (fixed)\n", err4);
			strcat(msg, temp);
		}

		if (err5)
		{
			sprintf(temp, "%d line singularities (removed)\n", err5);
			strcat(msg, temp);
		}

		window_text(msg, 1);
	} else
		window_text("Results\t\nNo errors found\n", 1);

	while (!mouse_check())
		if (keypress)
			break;

	await_release();
	return;
}

long summation(uint num)
{
	int i;
	long total=0L;

	if (num < 3)
		return 1L;
	for (i=1; i<num; i++)
		total += i;
	return total;
}

void fixup_lines(void)
{
	int i, j, vertex, v, orig_line, side, dir, line, temp;
	uint angle;

	draw_map();
	if ((!max_vertex) || (!l_size))
		return;
	vertex = 0;
	for (i=1; i<max_vertex; i++)
		if (vertexes[i].y < vertexes[vertex].y)
			vertex = i; /* find the bottom most vertex on map */
	for (i=0; i<l_size; i++)
		if (linedefs[i].v1 == vertex || linedefs[i].v2 == vertex)
			if (vertexes[linedefs[i].v1].x != vertexes[linedefs[i].v2].x)
				break; /* get a non-vertical line from this vertex */
	if ((line = i) == l_size)
		fatal_error("Line not found");

	orig_line = line; /* this is our starting line */
	v = linedefs[line].v1;
	vertex = linedefs[line].v2;
	side = dir = 0;
	if (vertexes[v].x < vertexes[vertex].x)
		side = 1;
	angle = calc_angle(vertexes[v].x, vertexes[v].y,
		vertexes[vertex].x, vertexes[vertex].y);

	do /* go through the entire outside border and make it all walls */
	{
		if (dir == side) /* which sidedef is which? */
		{
			if ((i = linedefs[line].sd2) != -1)
			{
				del_sidedef(i);
				linedefs[line].sd2 = -1;
			}
			if (linedefs[line].sd1 == -1)
				linedefs[line].sd1 = add_sidedef(-1);
		} else {
			if ((i = linedefs[line].sd1) != -1)
			{
				del_sidedef(i);
				linedefs[line].sd1 = -1;
			}
			if (linedefs[line].sd2 == -1)
				linedefs[line].sd2 = add_sidedef(-1);
		}
		linedefs[line].attrib |= 0x1; /* line is a wall */
		linedefs[line].attrib &= 0xfffb; /* can't be transparent */

		color2wall(line, 96, 32);
		dir = find_next_line(&vertex, &angle, &line, side);
		if (dir == -1)
			return;
	} while (line != orig_line);

	for (i=0; i<l_size; i++)
		if (linedefs[i].sd1 == -1 && linedefs[i].sd2 == -1)
		{ /* new line, not initialized yet */
			linedefs[i].sd1 = add_sidedef(-1);
			linedefs[i].sd2 = add_sidedef(-1);
			linedefs[i].attrib &= 0xfffe; /* not a wall */
			linedefs[i].attrib |= 0x4; /* probably will be transparent */

			sidedefs[linedefs[i].sd1].middle[0] = '-';
			sidedefs[linedefs[i].sd2].middle[0] = '-';
			for (j=1; j<8; j++)
			{
				sidedefs[linedefs[i].sd1].middle[j] = 0;
				sidedefs[linedefs[i].sd2].middle[j] = 0;
			}
			continue;
		}

	do
	{
		v = 0;
		for (i=0; i<l_size; i++)
		{
			if (linedefs[i].sd1 == -1 && (dir = wall_check(i, 1)))
			{
				if (dir == -1)
					return;
				linedefs[i].sd1 = add_sidedef(-1);
				v = 1;
				color2wall(i, 254, 96);
			}
			if (linedefs[i].sd2 == -1 && (dir = wall_check(i, 0)))
			{
				if (dir == -1)
					return;
				linedefs[i].sd2 = add_sidedef(-1);
				v = 1;
				color2wall(i, 254, 32);
			}
		}
	} while (v);

	for (i=0; i<l_size; i++)
	{
		if (linedefs[i].sd1 == -1 || linedefs[i].sd2 == -1)
		{
			linedefs[i].attrib |= 0x1;
			linedefs[i].attrib &= 0xfffb;
		} else {
			linedefs[i].attrib &= 0xfffe;
			linedefs[i].attrib |= 0x4;
		}

		flip_line(i);
	}
	return;
}

int wall_check(int line, int side)
{
	int line2, v1, v2, dir, vertex;
	uint angle, angle2;

	v1 = linedefs[line].v1;
	v2 = linedefs[line].v2;

	line2 = line;
	vertex = v2;
	angle = angle2 = calc_angle(vertexes[v1].x, vertexes[v1].y,
		vertexes[v2].x, vertexes[v2].y);
	if ((dir = find_next_line(&vertex, &angle, &line, side)) == -1)
		return -1;
	if (dir == side)
	{
		if (linedefs[line].sd2 != -1)
			return 1;
	} else {
		if (linedefs[line].sd1 != -1)
			return 1;
	}

	line = line2;
	vertex = v1;
	angle = angle2 ^ 0x8000;
	if ((dir = find_next_line(&vertex, &angle, &line, (side^1))) == -1)
		return -1;
	if (dir == side)
	{
		if (linedefs[line].sd1 != -1)
			return 1;
	} else {
		if (linedefs[line].sd2 != -1)
			return 1;
	}
	return 0;
}

void generate_sectors(void)
{
	char msg[256], msg2[60];
	int x, y, x1, y1, x2, y2, sector1, sector2, new_line;
	int i, line, xx, vertex, orig_line, side, dir, temp, re_loop;
	int min_vertex, min_vertex_y, sidedef, err_count=0, err_count2=0;
	int l_index, v_index, angle, dist;
	int vlist[VL_MAX];
	uint uangle, dist_min;

	struct sec_struct sec_info = { 0, 72, "FLOOR4_8", "CEIL3_5", 255, 0, 0 };

	for (i=0; i<s_size; i++)
		sidedefs[i].sector = -1; /* clear out sector references */

	window_text("Generating Sectors, please wait..\n\n\n", 1);
	xx = draw_time_graph(win.top + 20);
	draw_map();

	if (sec_max)
		free_farmem(sectors, "Sectors");
	if (r_size)
		free_farmem(reject, "Reject bitmap");
	l_index = v_index = sec_size = sec_max = r_size = 0;

	for (sidedef=0; sidedef<s_size; sidedef++)
	{
		if (sidedefs[sidedef].sector != -1)
			continue;
		for (line=0; line<l_size; line++)
		{
			if (linedefs[line].sd1 == sidedef)
			{
				side = 1;
				break;
			}
			if (linedefs[line].sd2 == sidedef)
			{
				side = 0;
				break;
			}
		}
		if (line == l_size)
		{
			err_count++;
			continue;
		}
		min_vertex_y = 32767;

		re_loop = 1;
		while (re_loop)
		{
			orig_line = line;
			temp = linedefs[line].v1;
			vertex = linedefs[line].v2;
			dir = 0;
			uangle = calc_angle(vertexes[temp].x, vertexes[temp].y,
				vertexes[vertex].x, vertexes[vertex].y);

			do
			{
				if (testmode > 1)
				{
					if (side)
					{
						tx1 = vertexes[temp].x;
						tx2 = vertexes[vertex].x;
						ty1 = vertexes[temp].y;
						ty2 = vertexes[vertex].y;
					} else {
						tx2 = vertexes[temp].x;
						tx1 = vertexes[vertex].x;
						ty2 = vertexes[temp].y;
						ty1 = vertexes[vertex].y;
					}
					setcolor(60);
					draw_side(tx1, ty1, tx2, ty2);

					sprintf(msg, "Sidedef=%d Line=%d Sector=%d Angle=%u "
						"Line list=%d group list=%d",
						sidedef, line, sec_size, uangle, l_index, v_index);
					toptext(msg);
					for (i=0; i<(maxy/10); i++)
					{
						sprintf(msg, "  %d", llist[i]);
						text(-1, i*10, msg);
					}
					cursored_get(0, 9);
					setcolor(0);
					draw_side(tx1, ty1, tx2, ty2);
				}
				color2wall(line, 96, 96);
				if (dir == side)
				{
					sidedefs[linedefs[line].sd2].sector = sec_size;
					temp = linedefs[line].sd1;
				} else {
					sidedefs[linedefs[line].sd1].sector = sec_size;
					temp = linedefs[line].sd2;
				}
				if (temp != -1)
					if (sidedefs[temp].sector == -1)
					{
						if (l_index == LL_MAX)
						{
							error("Line list full!");
							return;
						}
						llist[l_index++] = line;
						llist[l_index] = -1;
					}

				if ((temp = vertexes[vertex].y) < min_vertex_y)
				{
					min_vertex = vertex;
					min_vertex_y = temp;
				}
				temp = vertex;

				dir = find_next_line(&vertex, &uangle, &line, side);
				if (dir == -1)
				{
					deadend_error();
					return;
				}
			} while (line != orig_line);

			if (testmode == 1)
			{
				sprintf(msg, "Sector #%d", sec_size);
				toptext(msg);
				cursored_get(0, 9);
			}

			re_loop = 0;
			sec_size++;
			while (l_index)
			{
				llist[l_index] = -1;
				l_index--;
				line = llist[l_index]; /* get next line */
				if ((temp = linedefs[line].sd1) != -1 &&
					sidedefs[temp].sector == -1) /* find the sidedef */
				{
					side = 1;
					re_loop = 1;
					break;
				}
				if ((temp = linedefs[line].sd2) != -1 &&
					sidedefs[temp].sector == -1)
				{
					side = 0;
					re_loop = 1;
					break;
				}
			}
		}
		if (v_index == VL_MAX)
		{
			error("Vertex list full!");
			return;
		}
		vlist[v_index++] = min_vertex;
	}

	for (i=0; i<v_index; i++) /* now put sectors inside each other */
	{
		line = downward_line(vertex = vlist[i], &side, testmode);
		if (side)
			sidedef = linedefs[line].sd1;
		else
			sidedef = linedefs[line].sd2;
		x = vertexes[vertex].x;
		y = vertexes[vertex].y;

		line = inside_poly(x, y, &side, testmode);
		if (!side)
			side = linedefs[line].sd2;
		else if (side == 1)
			side = linedefs[line].sd1;

		if (sidedef == -1 || side == -1)
		{
			if (testmode > 1)
			{
				setcolor(0);
				draw_side(tx3, ty3, tx4, ty4);
				setcolor(96);
				draw_line2(adjustx(tx3), adjusty(ty3),
					adjustx(tx4), adjusty(ty4));
			}
			if (sidedef != -1 || side != -1)
				err_count2++; /* interior mismatch */
			continue;
		}

		if (testmode > 1)
		{
			sprintf(msg, "Detected sidedefs: %d and %d", sidedef, side);
			toptext(msg);
			cursored_get(0, 9);
			setcolor(0);
			draw_side(tx1, ty1, tx2, ty2);
			draw_side(tx3, ty3, tx4, ty4);
			setcolor(96);
			draw_line2(adjustx(tx1), adjusty(ty1),
				adjustx(tx2), adjusty(ty2));
			draw_line2(adjustx(tx3), adjusty(ty3),
				adjustx(tx4), adjusty(ty4));
		}

		sector1 = sidedefs[sidedef].sector;
		sector2 = sidedefs[side].sector;
		if (sector1 == sector2)
			fatal_error("Groups already match");

		if (testmode)
		{
			for (line=0; line<l_size; line++)
			{
				setcolor(255);
				test_draw_sector(line, sector1);
				setcolor(80);
				test_draw_sector(line, sector2);
			}
			sprintf(msg, "Sector %d merged with %d", sector2, sector1);
			toptext(msg);
			cursored_get(0, 9);
		}

		for (sidedef=0; sidedef<s_size; sidedef++)
		{
			if (sidedefs[sidedef].sector == sector2)
				sidedefs[sidedef].sector = sector1; /* make 1 sector */
			if (sidedefs[sidedef].sector > sector2)
				sidedefs[sidedef].sector--; /* close gap */
		}
		sec_size--;

		if (testmode)
		{
			setcolor(96);
			for (line=0; line<l_size; line++)
				test_draw_sector(line, sector1);
			toptext("");
		}
	}

	sec_max = sec_size;
	sectors = get_farmem(sec_size * sizeof(struct sec_struct), "Sectors");
	if (!sectors)
	{
		farmem_error("Sectors", sec_size);
		sec_size = sec_max = 0;
		return;
	}
	for (i=0; i<sec_size; i++) /* initialize the sectors */
		sectors[i] = sec_info;
	r_size = ((ulong) sec_size * sec_size + 7) / 8;
	reject = get_farmem(r_size, "Reject bitmap");
	for (i=0; i<r_size; i++)
		reject[i] = 0;

	*msg = 0;
	if (err_count)
	{
		sprintf(msg, "%d unused sidedefs detected", err_count);
		if (err_count2)
			strcat(msg, "\n");
	}

	if (err_count2)
	{
		sprintf(msg2, "%d group mismatches detected", err_count2);
		strcat(msg, msg2);
	}

	if (err_count || err_count2)
		error(msg);
	return;
}

void fixup_sectors(void)
{
}

void generate_nodes(void)
{
	char msg[256];
	int index, v1, v2, far *list;
	int i, button;
	long j;
	struct ss_struct ssec;

	optimize = 2;
	window_text("Generate Nodes\t\n"
		"Optimize for:\n\n"
		"@ Balance\n"
		"@     :\n"
		"@     :\n"
		"@ Least splits\n\n", 1);

	set_window_bars();
	b.pos[optimize].on = 1;
	while ((button = window_check()) > -1)
	{
		b.pos[optimize].on = 0;
		b.pos[button].on = 1;
		optimize = button;
	}

	if (button == -99)
		return;

	start = time(NULL);
	if (n_max)
		free_farmem(nodes, "Nodes");
	if (ss_max)
		free_farmem(ssectors, "Sub Sectors");
	if (seg_max)
		free_farmem(segs, "Segments");
	n_size = ss_size = seg_size = simple_splits = complex_splits = 0;
	n_max = 2;
	nodes = get_farmem(sizeof(struct n_struct) * n_max, "Nodes");
	ss_max = 1;
	ssectors = get_farmem(sizeof(struct ss_struct) * ss_max, "Sub Sectors");
	seg_max = l_size;
	segs = get_farmem(sizeof(struct seg_struct) * seg_max, "Segments");

	v_size = max_vertex;
	slivers = get_mem(4096, "Sliver tracking list");
	list = get_farmem(s_size * 6, "Base node list");
	{
		index = err = 0;
		clearviewport();
		setcolor(252);
		for (i=0; i<l_size; i++)
		{
			v1 = linedefs[i].v1;
			v2 = linedefs[i].v2;
			draw_line(i, SOLID_LINE);

			if (linedefs[i].sd1 != -1)
				add_node_list(list, &index, i, v1, v2);
			if (linedefs[i].sd2 != -1)
				add_node_list(list, &index, i, v2, v1);
		}
		if (nodize(index/3, list) != 65535)
		{
			free_farmem(list, "Base node list");
			remove_vertexes();
			free_mem(slivers, "Sliver tracking list");

			end = time(NULL);
			i = difftime(end, start);
			sprintf(msg, "Node generation completed\t\n"
				"Took %d seconds\n"
				"%d Nodes created\n"
				"%d Sub Sectors created\n"
				"%d Segments created\n"
				"%d extra Vertexes created\n"
				"%d simple splits occured\n"
				"%d complex splits occured\n",
				i, n_size, ss_size, seg_size, v_size - max_vertex,
				simple_splits, complex_splits);

			window_text(msg, 1);
			while (!mouse_check())
				if (keypress)
					break;

			return;
		}
		free_farmem(list, "Base node list");
		free_mem(slivers, "Sliver tracking list");
	}
	if (!err)
		error("Out of far memory!");
	if (err == 1)
		error("Too many Nodes!");
	if (err == 2)
		error("Too many Segments!");
	if (err == 3)
		error("Too many Sub Sectors!");
	if (err == 4)
		error("Too many Vertexes!");
	n_size = ss_size = seg_size = 0;
	v_size = max_vertex;
	return;
}

/* used by generate_nodes() to built a list of starting segs */

void add_node_list(int far *list, int *index, int line, int v1, int v2)
{
	list[(*index)++] = line;
	list[(*index)++] = v1;
	list[(*index)++] = v2;
	return;
}

void remove_vertexes(void)
{
	int i, j, v;

	j = complex_splits;
	while (j--) /* resolve slivers by merging vertexes, since it's safe now */
	{
		v = slivers[j] + 1;
		for (i=0; i<seg_size; i++)
		{
			if (segs[i].v1 >= v)
				segs[i].v1--;
			if (segs[i].v2 >= v)
				segs[i].v2--;
		}

		for (i=v; i<v_size; i++)
			vertexes[i-1] = vertexes[i];
		v_size--;
	}
	return;
}

/*  recursive node generator  */

uint nodize(uint size, int far *list)
{
	int x1, y1, x2, y2, lx1, ly1, lx2, ly2, rx1, ry1, rx2, ry2;
	int num, far *left_list, far *right_list;
	uint link1;

	convex = 1;
	balance = splits_min = size;
	nodeline = -1;
	for (i=0, j=1; i<size; i++, j +=3)
	{
		x1 = vertexes[list[j]].x;
		x2 = vertexes[list[j+1]].x;
		y1 = vertexes[list[j]].y;
		y2 = vertexes[list[j+1]].y;
		setcolor(80);
		draw_side(x1, y1, x2, y2);

		if (resolve_space(i, size, list, left_list, right_list,
			&lx1, &ly1, &lx2, &ly2, &rx1, &ry1, &rx2, &ry2, 1))
				return 65535;
		if (lcount && rcount) /* lines on both sides of nodelist */
			convex = 0; /* if ever so, can't be convex polygon */

		bal = abs(lcount - rcount);
		num = 0;
		switch (optimize)
		{
			case 0: /* optimize for balance */
				if (bal < balance)
					num++;
				if (bal == balance && splits < splits_min)
					num++;
				break;
			case 3: /* optimize for least splits */
				if (splits < splits_min)
					num++;
				if (splits == splits_min && bal < balance)
					num++;
				break;
			case 1: /* optimize equaly between both */
				if ((bal + splits) <= (balance + splits_min))
					if (bal < balance)
						num++;
				break;
			case 2: /* optimize both, mainly least splits, though */
				if ((bal + splits*3) <= (balance + splits_min*3))
					if (bal < balance)
						num++;
		}
		if (!lcount || !rcount) /* all lines on one side */
			num = 0; /* ineligible, because we can't split anymore */

		if (num)
		{
			balance = bal;
			splits_min = splits;
			left_size = lcount;
			right_size = rcount;
			nodeline = i; /* best case for nodeline so far */
			setcolor(60);
			draw_side(x1, y1, x2, y2);
		} else {
			setcolor(0);
			draw_side(x1, y1, x2, y2);
			setcolor(146);
			draw_line2(adjustx(x1), adjusty(y1), adjustx(x2), adjusty(y2));
		}
	}

	if (convex) /* is a convex polygon, so generate an ssec */
	{
		if (ss_size == ss_max)
		{
			void far *ptr;

			ptr = resize_farmem(ssectors, (ss_max+20) *
				sizeof(struct ss_struct), "Sub Sectors");
			if (!ptr)
			{
				err = 3;
				return 65535;
			}
			ssectors = ptr;
			ss_max += 20;
		}
		ssectors[ss_size].segptr = seg_size;
		ssectors[ss_size++].size = size;

		setcolor(255);
		for (i=0; i<size; i++)
		{
			num = i * 3;
			if (make_seg(list[num], list[num+1], list[num+2]))
			{
				err = 2;
				return 65535;
			}
		}

		if (testmode)
		{
			sprintf(test_msg, "Nodes=%u Sub Sectors=%u Segments=%u ",
				n_size, ss_size, seg_size);
			toptext(test_msg);
			test_map();
		}
		return (32767 + ss_size); /* ssector link value */
	}
	if (nodeline == -1)
		fatal_error("No nodeline selected");

new_nodeline:
	num = nodeline * 3;
	x1 = vertexes[list[num+1]].x;
	x2 = vertexes[list[num+2]].x;
	y1 = vertexes[list[num+1]].y;
	y2 = vertexes[list[num+2]].y;
	if (left_size)
	{
		left_list = get_farmem(left_size * 6, "left node list");
		if (!left_list)
			return 65535;
	}
	if (right_size)
	{
		right_list = get_farmem(right_size * 6, "right node list");
		if (!right_list)
		{
			free_farmem(left_list, "left node list");
			return 65535;
		}
	}

re_resolve:
	lx1 = rx1 = ly1 = ry1 = INT_MAX; /* reset limits */
	lx2 = rx2 = ly2 = ry2 = INT_MIN;
	if (resolve_space(nodeline, size, list, left_list, right_list,
		&lx1, &ly1, &lx2, &ly2, &rx1, &ry1, &rx2, &ry2, 0))
	{
		free_farmem(left_list, "left node list");
		free_farmem(right_list, "right node list");
		return 65535;
	}
	if (!lcount || !rcount)
		fatal_error("convex polygon not detected");

	do
	{
		setcolor(32);
		setlinestyle(DOTTED_LINE, 0, 1);
		draw_box(lx1, ly1, lx2, ly2);
		setlinestyle(SOLID_LINE, 0, 1);
		plot_list(left_size, left_list);
		setcolor(96);
		setlinestyle(DOTTED_LINE, 0, 1);
		draw_box(rx1, ry1, rx2, ry2);
		setlinestyle(SOLID_LINE, 0, 1);
		plot_list(right_size, right_list);
		setcolor(60);
		if (nodeline < 0)
			draw_side(x2, y2, x1, y2);
		else
			draw_side(x1, y1, x2, y2);
		num = test_map2(left_size, right_size, size, left_list,
			right_list, list);
		if (num == -1)
		{
			i = x1;
			x1 = x2;
			x2 = i;
			i = y1;
			y1 = y2;
			y2 = i;
			goto re_resolve;
		}
		if (num > 0)
		{
			nodeline = num - 1;
			free_farmem(left_list, "left node list");
			free_farmem(right_list, "right node list");
			resolve_space(nodeline, size, list, left_list, right_list,
				&lx1, &ly1, &lx2, &ly2, &rx1, &ry1, &rx2, &ry2, 1);
			goto new_nodeline;
		}
	} while (num);

	num = left_size; /* global var we need, and nodize recursion will scramble */
	link1 = nodize(right_size, right_list);
	free_farmem(right_list, "right node list"); /* don't need it anymore */
	if (link1 < 65535)
	{
		link2 = nodize(num, left_list);
		if (link2 < 65535)
		{
			free_farmem(left_list, "left node list");
			if (n_size == n_max)
			{
				void far *ptr;

				ptr = resize_farmem(nodes, (n_max+20) *
					sizeof(struct n_struct), "Nodes");
				if (!ptr)
				{
					err = 1;
					return 65535;
				}
				nodes = ptr;
				n_max += 20;
			}

			nodes[n_size].x = x1;
			nodes[n_size].y = y1;
			nodes[n_size].dx = x2 - x1;
			nodes[n_size].dy = y2 - y1;
			nodes[n_size].node1.x1 = rx1;
			nodes[n_size].node1.y2 = ry1;
			nodes[n_size].node1.x2 = rx2;
			nodes[n_size].node1.y1 = ry2;
			nodes[n_size].node2.x1 = lx1;
			nodes[n_size].node2.y2 = ly1;
			nodes[n_size].node2.x2 = lx2;
			nodes[n_size].node2.y1 = ly2;
			nodes[n_size].link1 = link1;
			nodes[n_size].link2 = link2;
			return n_size++; /* link to this node */
		}
	}
	free_farmem(left_list, "left node list");
	return 65535; /* error, return back through recursions */
}

/*  make left and right lists from master list  */

int resolve_space(int line, int size, int far *list,
	int far *left_list, int far *right_list, int *lx1, int *ly1,
	int *lx2, int *ly2, int *rx1, int *ry1, int *rx2, int *ry2, int test)
{
	int i, v1, v2, v3, v4, vv1, vv2, x, y, xx, yy, num;
	int angle1, angle2, left, right, line2;
	uint angle;

	lcount = rcount = splits = 0; /* reset counts */

	num = line * 3;
	vv1 = list[num + 1];
	vv2 = list[num + 2];
	x = vertexes[vv1].x;
	y = vertexes[vv1].y;
	angle = calc_angle(x, y, vertexes[vv2].x, vertexes[vv2].y);

	for (i=0; i<size; i++)
	{
		left = right = 0;
		num = i * 3;
		line2 = list[num];
		v1 = v3 = list[num + 1];
		v2 = v4 = list[num + 2];

		if (v1 == vv1)
			angle1 = 0;
		else {
			angle1 = adjusted_angle(angle, x, y, vertexes[v1].x, vertexes[v1].y);
			if (angle1 == INT_MIN) /* on line, but behind vertex 1 */
				angle1 = 0;
		}

		if (v2 == vv1)
			angle2 = 0;
		else {
			angle2 = adjusted_angle(angle, x, y, vertexes[v2].x, vertexes[v2].y);
			if (angle2 == INT_MIN)
				angle2 = 0;
		}

		if ((!angle1) && (!angle2)) /* on line */
		{
			if (adjusted_angle(angle, vertexes[v1].x, vertexes[v1].y,
				vertexes[v2].x, vertexes[v2].y))
				left = 1;
			else
				right = 1;
		} else if (angle1 >= 0 && angle2 >= 0) /* totally right of line */
			right = 1;
		else if (angle1 < 1 && angle2 < 1) /* totally left of line */
			left = 1;
		else { /* nodeline must split this line */
			left = right = 1; /* line on each side */
			splits++; /* inc split count */
			if (!test) /* if in test mode, don't need to make new vertexes */
			{
				if ((xx = calc_line_cross(&yy, x, y, vv2, v1, v2, angle)) == -1)
				{
					err = 4;
					return -1;
				}

				if (angle1 < 0 && angle2 > 0) /* which direction is line? */
				{
					v1 = xx;
					v4 = yy;
				} else {
					v2 = xx;
					v3 = yy;
				}
			}
		}

		if (!test) /* if not in test mode, build the lists */
		{
			if (right)
				add_node_list2(right_list, rcount++, line2, v1, v2,
					rx1, ry1, rx2, ry2);
			if (left)
				add_node_list2(left_list, lcount++, line2, v3, v4,
					lx1, ly1, lx2, ly2);
		} else {
			if (right) /* still need to track counts, though */
				rcount++;
			if (left)
				lcount++;
		}
	}
	return 0;
}

void add_node_list2(int far *list, int index, int line, int v1, int v2,
	int *x1, int *y1, int *x2, int *y2)
{
	uint idx;

	adjust_limit(vertexes[v1].x, vertexes[v1].y, x1, y1, x2, y2);
	adjust_limit(vertexes[v2].x, vertexes[v2].y, x1, y1, x2, y2);

	idx = index * 3;
	list[idx] = line;
	list[idx+1] = v1;
	list[idx+2] = v2;
	return;
}

void draw_side(int x1, int y1, int x2, int y2)
{
	int cx, cy, tx, ty;
	double angle;

	angle = ((double) calc_angle(x1, y1, x2, y2) - 16384.0) * M_PI / 32768.0;
	cx = (x1 + x2) / 2;
	cy = (y1 + y2) / 2;
	tx = cx + 16.0 * cos(angle);
	ty = cy + 16.0 * sin(angle);
	draw_line2(adjustx(x1), adjusty(y1), adjustx(x2), adjusty(y2));
	draw_line2(adjustx(cx), adjusty(cy), adjustx(tx), adjusty(ty));
	return;
}

int make_seg(int line, int v1, int v2)
{
	int v, vv, x, y, x1, y1, x2, y2, dir, dist;
	uint angle1, angle2;

	if (seg_size == seg_max)
	{
		void far *ptr;

		ptr = resize_farmem(segs, (seg_max+20) * sizeof(struct seg_struct),
			"Segments");
		if (!ptr)
			return -1;
		segs = ptr;
		seg_max += 20;
	}

	x1 = vertexes[v1].x;
	x2 = vertexes[v2].x;
	y1 = vertexes[v1].y;
	y2 = vertexes[v2].y;
	draw_side(x1, y1, x2, y2);
	v = linedefs[line].v1;
	vv = linedefs[line].v2;
	x = vertexes[v].x;
	y = vertexes[v].y;

	dir = dist = 0;
	angle1 = calc_angle(x1, y1, x2, y2);
	angle2 = calc_angle(x, y, vertexes[vv].x, vertexes[vv].y);
	if (labs((long int) angle2 - angle1) > 16384) /* opposite direction */
	{
		dir = 1;
		v = linedefs[line].v2;
		x = vertexes[v].x;
		y = vertexes[v].y;
	}
	if (v1 != v)
	{
		x = abs(x - x1);
		y = abs(y - y1);
		dist = sqrt((double) x * x + (double) y * y);
	}

	segs[seg_size].v1 = v1;
	segs[seg_size].v2 = v2;
	segs[seg_size].linedef = line;
	segs[seg_size].angle = angle1;
	segs[seg_size].dir = dir;
	segs[seg_size++].dist = dist;
	return 0;
}

void plot_list(int size, int far *list)
{
	int i, j, x1, y1, x2, y2;

	for (i=0, j=1; i<size; i++, j += 3)
	{
		x1 = vertexes[list[j]].x;
		y1 = vertexes[list[j]].y;
		x2 = vertexes[list[j+1]].x;
		y2 = vertexes[list[j+1]].y;

		draw_side(x1, y1, x2, y2);
	}
	return;
}

void calc_map_data(void)
{
	int i;

	for (i=0; i<max_vertex; i++)
	{
		adjvx[i] = adjustx(vertexes[i].x);
		adjvy[i] = adjusty(vertexes[i].y);
	}

	ll_size = 0;
	for (i=0; i<l_size; i++)
		if (line_visible(i))
			llines[ll_size++] = i;

	return;
}

int test_map(void)
{
	int i, key, rekey=0;

	if (!testmode)
		return 0;

	if (!(key = cursored_get(0, 9)))
		key = getch() + 1000;
	if (key >= 'A' && key <= 'Z')
		key += 0x20; /* remap uppercase to lowercase */
	if (map_keys(key))
		calc_map_data();

	clearviewport();
	setcolor(252);
	for (i=0; i<l_size; i++)
		draw_line(i, SOLID_LINE);
	return rekey;
}

int test_map2(int left, int right, int cur, int far *left_list,
	int far *right_list, int far *cur_list)
{
	int i, key, rekey=0, far *ptr;

	if (!testmode)
		return 0;

	sprintf(test_msg, "Current list=%d  Right list=%d  Left list=%d",
		cur, right, left);
	toptext(test_msg);
	if (!(key = cursored_get(0, 9)))
		key = getch() + 1000;
	if (key >= 'A' && key <= 'Z')
		key += 0x20; /* remap uppercase to lowercase */
	if (map_keys(key))
	{
		calc_map_data();
		rekey = -9;
	}

	switch (key)
	{
		case 'l':
			rekey = -9;
			showlist("Left", left, left_list);
			break;
		case 'r':
			rekey = -9;
			showlist("Right", right, right_list);
			break;
		case 'c':
			rekey = showlist("Current", cur, cur_list);
			break;
		case 'f':
			ptr = left_list;
			left_list = right_list;
			right_list = ptr;
			i = left;
			left = right;
			right = i;
			rekey = -1;
			break;
	}

	clearviewport();
	setcolor(252);
	for (i=0; i<l_size; i++)
		draw_line(i, SOLID_LINE);
	return rekey;
}

int showlist(char *name, int size, char far *list)
{
	int i;

	return 0;
}

/*  make polygon's sidedefs on both sided of <line> the same as <line>'s
	 sidedefs
*/

void fix_sidedefs(int line)
{
	fix_sidedef(line, 0);
	fix_sidedef(line, 1);
	return;
}

/*   make all sidedefs of polygon the same as <line>'s sidedef  */

void fix_sidedef(int line, int side)
{
	int orig_line, vertex, v1, dir;
	uint angle;
	struct s_struct sidedef_info;

	orig_line = line;
	v1 = linedefs[line].v1;
	vertex = linedefs[line].v2;
	angle = calc_angle(vertexes[v1].x, vertexes[v1].y,
		vertexes[vertex].x, vertexes[vertex].y);
	if (side)
		sidedef_info = sidedefs[linedefs[line].sd1];
	else
		sidedef_info = sidedefs[linedefs[line].sd2];
	dir = 0;

	do
	{
		if (dir == side)
			sidedefs[linedefs[line].sd2] = sidedef_info;
		else
			sidedefs[linedefs[line].sd1] = sidedef_info;

		dir = find_next_line(&vertex, &angle, &line, side);
		if (dir == -1)
		{
			deadend_error();
			return;
		}
	} while (line != orig_line);
	return;
}

void generate_blockmap(void)
{
	int i, xmin, ymin, xmax, ymax, x, y, x1, y1, x2, y2, xx, yy, block;
	int blockmax, xorigin, yorigin, xsize, ysize, point1, point2;

	window_text("Generating Blockmap, please wait..\n\n\n", 1);
	yy = win.top + 20;
	xx = draw_time_graph(yy);

	if (b_size)
		free_farmem(blockmap, "Blockmap");
	b_max = 1000;
	blockmap = get_farmem(b_max, "Blockmap");

	xmin = xmax = ymin = ymax = 0;
	if (v_size)
	{
		xmin = xmax = vertexes[0].x;
		ymin = ymax = vertexes[0].y;
		for (i=1; i<v_size; i++) /* figure out the size of the map */
			adjust_limit(vertexes[i].x, vertexes[i].y, &xmin, &ymin,
				&xmax, &ymax);
	}

	blockmap->xorigin = xorigin = xmin & 0xfff8;
	blockmap->yorigin = yorigin = ymin & 0xfff8;
	blockmap->xsize = xsize = (xmax - xorigin) / 128 + 1;
	blockmap->ysize = ysize = (ymax - yorigin) / 128 + 1;
	block = 0;
	blockmax = b_size = xsize * ysize;

	for (y=0; y<ysize; y++)
		for (x=0; x<xsize; x++)
		{
			xmin = xorigin + (x * 128);
			ymin = yorigin + (y * 128);
			xmax = xmin + 127;
			ymax = ymin + 127; /* setup testing cubical */
			blockmap->offsets[block++] = b_size + 4;

			if (blockmap_add(0)) /* a header I guess */
				return;

			for (i=0; i<l_size; i++)
			{
				point1 = linedefs[i].v1;
				point2 = linedefs[i].v2;
				x1 = vertexes[point1].x;
				x2 = vertexes[point2].x;
				y1 = vertexes[point1].y;
				y2 = vertexes[point2].y;

				if (line_in_rect(x1, y1, x2, y2, xmin, ymin, xmax, ymax))
					if (blockmap_add(i))
						return;
			}

			if (blockmap_add(-1)) /* end-of-list */
				return;
			time_graph(block, blockmax, xx, yy);
		}

	b_size = (b_size + 4) * 2;
	blockmap = resize_farmem(blockmap, b_size, "Blockmap");
	return;
}

int blockmap_add(int num)
{
	if ((b_size + 5) * 2 > b_max)
	{
		void far *ptr;

		ptr = resize_farmem(blockmap, b_max + 100, "Blockmap");
		if (!ptr)
		{
			error("Can't generate blockmap.  Map too big.");
			b_size = 0;
			return -1;
		}
		blockmap = ptr;
		b_max += 100;
	}

	blockmap->offsets[b_size++] = num;
	return 0;
}
