/***************************************************************************
 *                                                                         *
 *   GALFILUT.C                                                            *
 *                                                                         *
 *   Copyright (C) 1993-1994 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   File Libraries - Utility Routines                                     *
 *                                                                         *
 *                                                 - D. Pitchford  9/28/93 *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "galfil.h"
#include "filexfer.h"
#include "fsdbbs.h"
#include "galfilh.h"

char *
checkdir(                          /* modify path to proper format         */
char *path)                             /* ... i.e., no trailing \ or /    */
{
     int len;
     char chr,*sptr;

     if ((sptr=strchr(path,':')) != NULL) {
          if (sptr-path != 1) {
               *path='\0';
          }
          else {
               if (!isalpha(*path)) {
                    *path='\0';
               }
               else if ((sptr=strchr(++sptr,':')) != NULL) {
                    *sptr='\0';
               }
          }
     }
     len=strlen(path);
     while (len > 0) {
          chr=path[len-1];
          if (chr == '\\' || chr == '/') {
               if (path[len-2] == ':') {
                    strcat(path,".");
                    break;
               }
               else {
                    path[--len]='\0';
               }
          }
          else {
               if (chr == ':') {
                    strcat(path,".");
               }
               break;
          }
     }
     return(path);
}

char *
fixpath(                           /* return a properly formatted path     */
char *path)
{
     static char retval[4][PATHSIZE];
     static int cycle=0;

     return(checkdir(stzcpy(retval[cycle=((cycle+1)&3)],path,PATHSIZE)));
}

char *
compkey(                           /* build a composite key (struct key1)  */
char *libname,
char *filname)
{
     static char key[FLNAMESZ+FLFILENM];

     stzcpy(key,libname,FLNAMESZ);
     stzcpy(&key[FLNAMESZ],filname,FLFILENM);
     return(key);
}

char *
rcompkey(                          /* build a composite key (struct key4)  */
char *libname,
char *filname)
{
     static char key[FLNAMESZ+FLFILENM];

     stzcpy(key,filname,FLFILENM);
     stzcpy(&key[FLFILENM],libname,FLNAMESZ);
     return(key);
}

char *
cutpunc(                           /* cuts words at punctuation            */
char *stg)
{
     char *cptr;

     for (cptr=stg ; cptr != '\0' ; cptr++) {
          if (!isalnum(*cptr) && *cptr < 128) {  /* keep high-ASCII foreign */
               *cptr='\0';
               if ((++cptr)[0] != '\0' && (dargc < (MAXKWDS-1))) {
                    dargv[dargc++]=cptr;
               }
               break;
          }
     }
     return(stg);
}

char *
dat2srt(                           /* convert "10/24/93" to "1070207"      */
char *dt)                          /* 200-93==107, 12-10==02, 31-24==07    */
{
     static char retval[4][8];
     static int cycle=0;
     int yr,mnth,dy;

     setmem(retval[cycle=((cycle+1)&3)],8,0);
     if (strlen(dt) == 8 && dt[2] == '/' && dt[5] == '/') {
          yr=atoi(&dt[6]);
          mnth=atoi(dt);
          dy=atoi(&dt[3]);
          if (yr < 80) {
               yr+=100;
          }
          sprintf(retval[cycle],"%03d%02d%02d",200-yr,12-mnth,31-dy);
     }
     if (retval[cycle][0] == '\0') {
          return(NOTAPPED);
     }
     return(retval[cycle]);
}

long
dnlchg(                            /* approx d/l charge                    */
long siz,
char *libname)
{
     struct fllib *libptr;
     long estcreds=0L,price;

     if (libname[0] == '\0' || ((libptr=libfind(libname)) == NULL)) {
          return(0L);
     }
     if (!(fndcls(usaptr->curcls)->flags&CRDXMT)) {
          estcreds=((dnlsecl(siz,libptr)/15L)+1)*(lincst(usrnum)+
                    (long)(usrptr->crdrat/4));
     }
     price=dnlprc(siz,libptr);
     if (price == 0L) {
          estcreds=0L;
     }
     return(price+estcreds);
}

char *
dnlmin(                            /* approx mins. to d/l                  */
long siz)
{
     static char c[11];
     long x;

     setmem(c,11,0);
     x=dnlsec(siz);
     if (x < 45L) {
          return("< 1");
     }
     else {
          sprintf(c,"%ld",(x+30L)/60L);
     }
     return(c);
}

char *
fzer(                              /* returns l2as(number) or "< 1" if 0   */
long num)
{
     return(num == 0L ? "< 1" : l2as(num));
}

char *
protocol(void)                     /* return protocol to use               */
{
     return(flu->proto[0] != '\0'
        ? ((flo->flags&ABOUPRF) ? "?" : flu->proto) : defproto);
}

char *
srt2dat(                           /* convert "1070207" to "10/24/93"      */
char *dt)
{
     static char retval[4][9];
     static int cycle=0;
     char yr[4],mnth[3],dy[3];

     setmem(retval[cycle=((cycle+1)&3)],9,0);
     setmem(yr,4,0);
     setmem(mnth,3,0);
     setmem(dy,3,0);
     if (strlen(dt) == 7) {
          yr[0]=dt[0];
          yr[1]=dt[1];
          yr[2]=dt[2];
          mnth[0]=dt[3];
          mnth[1]=dt[4];
          dy[0]=dt[5];
          dy[1]=dt[6];
          sprintf(retval[cycle],"%02d/%02d/%02d",
             12-atoi(mnth),31-atoi(dy),(200-atoi(yr))%100);
     }
     return(retval[cycle]);
}

char *
tvcomeff(void)                     /* returns communications efficiency    */
{
     static char retval[4];

     sprintf(retval,"%d",comeff);
     return(retval);
}

char *
tvctim(void)                       /* return creation time & date          */
{
     return(curlib == NULL ? "" : spr("%s %s",
        ncdate(curlib->day),nctime(curlib->time)));
}

char *
tvdesc(void)                       /* return current Library description   */
{
     return(curlib == NULL ? "" : curlib->libdesc);
}

char *
tvdesc0(void)                      /* return part of long description      */
{
     static char rv[164];

     loadldsc(NULL);
     if (libupd->ldesc[0][0] == '\0') {
          return("");
     }
     sprintf(rv,"%s\r",libupd->ldesc[0]);
     return(rv);
}

char *
tvdesc1(void)                      /* return part of long description      */
{
     static char rv[162];

     if (libupd->ldesc[1][0] == '\0') {
          return("");
     }
     sprintf(rv,"%s\r",libupd->ldesc[1]);
     return(rv);
}

char *
tvdesc2(void)                      /* return part of long description      */
{
     static char rv[162];

     if (libupd->ldesc[2][0] == '\0') {
          return("");
     }
     sprintf(rv,"%s\r",libupd->ldesc[2]);
     return(rv);
}

char *
tvdesc3(void)                      /* return part of long description      */
{
     static char rv[162];

     if (libupd->ldesc[3][0] == '\0') {
          return("");
     }
     sprintf(rv,"%s\r",libupd->ldesc[3]);
     return(rv);
}

char *
tvdesc4(void)                      /* return part of long description      */
{
     static char rv[162];

     if (libupd->ldesc[4][0] == '\0') {
          return("");
     }
     sprintf(rv,"%s\r",libupd->ldesc[4]);
     return(rv);
}

char *
tvdnlmin(void)                     /* return est. minutes to d/l           */
{
     return(skpwht(dnlmin(flf->siz)));
}

char *
tvfapp(void)                       /* return current Library files to app  */
{
     static char retval[7];

     if (curlib == NULL) {
          return("");
     }
     sprintf(retval,"%d",(curlib->appwait < 0) ? 0 : curlib->appwait);
     return(retval);
}

char *
tvfdate(void)                      /* return dos date or "UNAPPROVED"      */
{
     return(notapped(flf->udate) ? "UNAPPRVD" : srt2dat(flf->fdate));
}

char *
tvfdates(void)                     /* return short dos date or "UNAPP"     */
{
     static char retval[6];
     char *stg;

     if (notapped(flf->udate)) {
          strcpy(retval,"UNAPP");
     }
     else {
          stg=srt2dat(flf->fdate);
          setmem(retval,6,0);
          retval[0]=stg[0];
          retval[1]=stg[1];
          retval[2]='/';
          retval[3]=stg[6];
          retval[4]=stg[7];
     }
     return(retval);
}

char *
tvfname(void)                      /* return filename                      */
{
     return(flf->filname);
}

char *
tvflib(void)                       /* return libname                       */
{
     struct fllib *libptr;

     libptr=libfind(flf->libname);
     return(libptr != NULL ? visxlib(libptr,"") : "");
}

char *
tvfnum(void)                       /* return current Library # of files    */
{
     static char retval[15];
     long nfiles;

     if (curlib == NULL) {
          return("");
     }
     if (curlib->flags&FLGDOS) {
          return("DOS");
     }
     if (curlib->libs > 0) {       /* if a top-level lib, get the totals   */
          scanlib(1,curlib);
          nfiles=curlib->totfiles;
          scanlib(0,curlib);
     }
     else {
          nfiles=curlib->numfiles-curlib->appwait;
     }
     if (nfiles < 0L || nfiles > 999999L) {
          nfiles=0L;
     }
     sprintf(retval,"%ld",nfiles);
     return(retval);
}

char *
tvfsize(void)                      /* return file size                     */
{
     return(l2as(flf->siz));
}

char *
filsize(                           /* return passed file size 5-byte fmt.  */
long size,                              /* size for the file               */
long kval)                              /* value for "K" and "M", 1000/1024*/
{
     static char rsize[6];
     long ksiz;

     if (kval != 1000L) {
          kval=1024L;
     }
     if (size < kval) {
          sprintf(rsize,"%5ld",size);
     }
     else {
          ksiz=size/kval;
          if (ksiz >= kval) {
               if (ksiz >= 100L*kval) {
                    sprintf(rsize,"%4ldM",ksiz/kval);
               }
               else {
                    sprintf(rsize,"%2ld.%1ldM",ksiz/kval,10L*(ksiz%kval)/kval);
               }
          }
          else {
               if (ksiz >= 10L) {
                    sprintf(rsize,"%4ldK",ksiz);
               }
               else {
                    sprintf(rsize,"%2ld.%1ldK",ksiz,10L*(size%kval)/kval);
               }
          }
     }
     return(rsize);
}

char *
tvfsizek(void)                     /* return file size                     */
{
     return(filsize(flf->siz,1024L));
}

char *
tvftimes(void)                     /* return # of times downloaded         */
{
     return(l2as(atol(flf->numdls)));
}

char *
tvfulby(void)                      /* return name of uploader              */
{
     return(flf->ulby);
}

char *
tvinit(void)                       /* if using FL_ tvars on a menu page,   */
{                                  /* call this one first to initialize    */
     fluoff(usrnum);               /* for the user                         */
     setftu();
     return("");
}

char *
tvmisc(void)                       /* handy misc variable, instead of a %s */
{
     return(flmisc);
}

char *
tvname(void)                       /* return name of current Library       */
{
     return(curlib == NULL ? "" : curlib->libname);
}

char *
tvtagk(void)                       /* return current tag file character    */
{
     static char retval[2];

     retval[0]=flo->tagk;
     retval[1]='\0';
     return(retval);
}

char *
tagk(void)                         /* return tag letter reference          */
{
     static char retval[200];
     char *stg;

     setmem(retval,200,0);
     setmbk(flmsg);
     if (strlen(stg=getmsg(flo->tagk == 'A' ? TAGKONE : TAGKMANY)) < 200) {
          sprintf(retval,stg,flo->tagk);
     }
     rstmbk();
     return(retval);
}

char *
tvtags(void)                       /* return # of tags (-1)                */
{
     static char nt[5];

     setftu();
     sprintf(nt,"%d",totags);
     return(nt);
}

char *
tvtshmsg(void)                     /* return file xfer facility's message  */
{
     return(tshmsg);
}

char *
tvudate(void)                      /* return u/l date or "UNAPPROVED"      */
{
     return(notapped(flf->udate) ? "UNAPPRVD" : srt2dat(flf->udate));
}

char *
tvbytu(void)                       /* return max upload size               */
{
     return(curlib == NULL ? "" : l2as(curlib->maxbup));
}

char *
tvbytt(void)                       /* return max Library size              */
{
     return(curlib == NULL ? "" : l2as(curlib->maxbyt));
}

char *
tvbyto(void)                       /* return max u/l size                  */
{
     if (curlib == NULL) {
          return("");
     }
     libroom(curlib);
     return(l2as(byteroom > 0L ? byteroom : 0L));
}

char *
tvfilt(void)                       /* return max Library # files           */
{
     return(curlib == NULL ? "" : l2as(curlib->maxfil));
}

int
alrtag(                            /* user already tagged file??           */
char *libname,
char *filname)
{
     struct taglib *tag;
     int loop,loop2;

     for (loop=0 ; loop < systags ; loop++) { /* see if usr already tagged */
          tag=tagoff(loop);
          if (tag->status > TGEMPTY && sameas(tag->filname,filname)
            && tag->usrnum == usrnum && sameas(tag->libname,libname)) {
               setftu();
               for (loop2=0 ; loop2 < ftuptr->numftg ; loop2++) {
                    ftgptr=&ftgusr[loop2];
                    if (spc == tag) {
                         return(loop+1);
                    }
               }
          }
     }
     return(0);
}

int
countc(                            /* count # of certain characters in stg */
char *stg,
char chr)
{
     int retval=0;

     while (*stg != '\0') {
          if (*stg++ == chr) {
               retval++;
          }
     }
     return(retval);
}

int
deparse(                           /* reduces and/or/not primitive to y/n  */
char *p)                           /* *p is the primitive, (!0&(0|!(1^0))) */
{
     char image[FLKEYLST];

     while (strlen(p) > 1) {
          stzcpy(image,p,FLKEYLST);
          do {
               do {
                    do {
                         do {
                              do {
                              } while (strsrep(p,"(1)","1")
                                 || strsrep(p,"(0)","0"));
                         } while (strsrep(p,"!1","0") || strsrep(p,"!0","1"));
                    } while (strsrep(p,"1&1","1") || strsrep(p,"1&0","0")
                       || strsrep(p,"0&1","0") || strsrep(p,"0&0","0"));
               } while (strsrep(p,"1^1","0") || strsrep(p,"1^0","1")
                  || strsrep(p,"0^1","1") || strsrep(p,"0^0","0"));
          } while (strsrep(p,"1|1","1") || strsrep(p,"1|0","1")
             || strsrep(p,"0|1","1") || strsrep(p,"0|0","0"));
          if (!countc(p,'1') && !countc(p,'0') || sameas(image,p)) {
               return(0);
          }
     }
     return(sameas(p,"1") ? 1 : -1);
}

int
dotags(                            /* decide what files a user has tagged  */
int quiet)
{
     int loop,tot=0;

     for (loop=0 ; loop < NLISTER ; loop++) {
          if (flo->tdline[loop] < 0) {
               tot+=submit(flo->tags[loop].libname,flo->tags[loop].filname,
                  flo->lrange,NOTVIEW,quiet);
          }
     }
     flo->tagcnt=0;
     setfull(totags >= (maxtags-1));
     return(tot);
}

void
setfull(                           /* set users' full tag flag on or off   */
int isfull)
{
     if (isfull) {
          flo->flags|=FULLTAG;
     }
     else {
          flo->flags&=~FULLTAG;
     }
}

int
findbest(                          /* find oldest task of type stt         */
int stt)
{
     int retval=0,loop;
     struct taglib *filetag;

     ftag=NULL;
     for (loop=0 ; loop < systags ; loop++) {
          filetag=tagoff(loop);
          if (filetag->status == stt) {
               if (!retval || ftag->tagtime > filetag->tagtime) {
                    retval=1;
                    ftag=filetag;
               }
          }
     }
     return(retval);
}

int
visilib(                           /* is libptr visible to current user?   */
struct fllib *libptr)                   /* always visible to libops/sysops */
{
     return(haskey(libptr->keyreq) && !(libptr->flags&FLGHID)
        || isflop(libptr));
}

char *
visxlib(                           /* is libptr visible even indirectly?   */
struct fllib *libptr,
char *libmatch)                         /* ... if specified, also joined   */
{
     int loop;
     struct fllib *libcat;

     if (visilib(libptr)) {
          if (libmatch[0] == '\0' || sameas(libmatch,libptr->libname)) {
               return(libptr->libname);
          }
     }
     if (!haslibkey(libptr,libptr->keyreq)) {
          return("");
     }
     for (loop=0 ; loop < FLNMCATS ; loop++) {
          if (libptr->cat[loop][0] != '\0'
             && (libcat=libfind(libptr->cat[loop])) != NULL
             && visilib(libcat)) {
               if (libmatch[0] == '\0' || sameas(libmatch,libcat->libname)) {
                    return(libcat->libname);
               }
          }
     }
     return("");
}

int
keyfig(void)                       /* apply primitive template to file     */
{
     int loop;
     char blist[FLKEYLST];

     if (!makwdlst(flf->desc,flf->filname)) {
          return(-1);
     }
     stzcpy(blist,flo->keylist,FLKEYLST);
     for (loop=0 ; loop < flo->kcount ; loop++) {
          if (!strsrep(blist,"?",keyin(&vdaptr[loop*FLKEYSIZ]) ? "1" : "0" )) {
               return(-1);
          }
     }
     return(deparse(blist));
}

int
keyin(                             /* is keyword in keylist?               */
char *key)
{
     int loop,len;
     char *filname;

     filname=spr("%s",flf->filname);
     strcrep(filname,'.','\0');
     len=strlen(flf->desc)-strlen(key);
     for (loop=0 ; loop <= len ; loop++) {
          if (sameto(key,&flf->desc[loop])) {
               return(1);
          }
     }
     len=strlen(filname)-strlen(key);
     for (loop=0 ; loop <= len ; loop++) {
          if (sameto(key,&filname[loop])) {
               return(1);
          }
     }
     return(0);
}

int
keysin(                            /* make key list and determine keyword  */
int start)
{
     int loop;

     if (!makwdlst(flf->desc,flf->filname)) {
          return(-1);
     }
     for (loop=0 ; loop < flo->kcount ; loop++) {
          if ((loop != start) && !keyin(&vdaptr[loop*FLKEYSIZ])) {
               return(-1);
          }
     }
     return(1);
}

int
kwdout(void)                       /* make kwd list and ready for insert   */
{
     if (!makwdlst(flf->desc,flf->filname)) {
          return(0);
     }
     prfmsg(usrptr->substt=KEYWORDY);
     stzcpy(flo->kwdy.filname,flf->filname,FLFILENM);
     stzcpy(flo->kwdy.libkey,flf->libname,FLNAMESZ);
     flo->miscflag=-1;
     btuinj(usrnum,CYCLE);
     return(1);
}

int
nlibtagd(void)                     /* # of Library files tagged            */
{
     int loop,retval=0;

     for (loop=0 ; loop < ftuptr->numftg ; loop++) {
          ftgptr=&ftgusr[loop];
          fltagf=0;
          ftgptr->tshndl(TSHDSC);
          if (fltagf) {
               retval++;
          }
     }
     return(retval);
}

int
okfname(                           /* make sure passed file name is ok     */
char *filename)
{
     int loop=0,len,dot=0;

     if (!(len=strlen(filename)) || len > 12) {
          return(0);
     }
     if (len > 8) {
          do {
               if (filename[loop] == '.') {
                    if (loop == 0 || loop < (len-4)) {
                         return(0);
                    }
                    break;
               }
               if (++loop > 8) {
                    return(0);
               }
          } while (1);
     }
     if (rsvnam(filename) || profane(filename)) {
          return(0);
     }
     for (loop=0 ; loop < len ; loop++,filename++) {
          switch (*filename) {
          case '.':
               if ((!dot++) && (loop >= (len-4)) && loop) {
                    break;
               }
          case '\\':
          case ':':
          case '/':
          case '*':
          case '?':
          case '+':
          case '<':
          case '>':
               return(0);
          default:
               if (*filename <= ' ') {
                    return(0);
               }
          }
     }
     return(1);
}

int
readline(                          /* read a line from a text file         */
FILE *ptr)
{
     static int last=0;
     int chr=0,retval=1;
     char *pos;

     setmem(input,INPSIZ,0);
     pos=input;
     while (chr != 13 && chr != 10 && ptr != NULL) {
          chr=fgetc(ptr);
          if (feof(ptr) || ferror(ptr)) {
               last=0;
               retval=0;
               rewind(ptr);
               chr=13;
          }
          else {
               switch (chr) {
               case 10:
               case 13:
                    if (!last) {
                         last=1;
                    }
                    else {
                         if (input[0] == '\0') {
                              chr=0;
                         }
                         last=0;
                    }
                    break;
               case 0:
                    chr=13;
                    break;
               case 26:
               case EOF:
                    break;
               default:
                    last=0;
                    *pos=chr;
                    pos++;
                    if (strlen(input) == INPSIZ-1) {
                         chr=13;
                    }
               }
          }
     }
     return(retval);
}

int
srch1(void)                        /* sysop search cycle for file checks   */
{
     struct fllib *libptr;

     setbtv(flfdat);
     if (agtbtv(NULL,&flo->key1,COMPLF)) {
          if (flo->misclib[0] != '\0' && !sameas(flf->libname,flo->misclib)) {
               return(0);
          }
          stzcpy(flo->key1.libname,flf->libname,FLNAMESZ);
          stzcpy(flo->key1.filname,flf->filname,FLFILENM);
          if ((libptr=libfind(flf->libname)) != NULL
             && !(libptr->flags&FLGDOS)
             && (!(libptr->flags&FLGCBD)
             || libptr->path[0] != '\0')) {
               if (fnd1st(&flo->fb,spr("%s\\%s",libpath(libptr),flf->filname),
                  0)) {
                    if (flo->fb.size != flf->siz) {
                         if (acqbtv(NULL,&flo->key1,COMPLF)) {
                              flf->siz=flo->fb.size;
                              upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
                         }
                         prfmsg(NOSIZEM,flf->filname,libptr->libname);
                         outprf(usrnum);
                    }
               }
               else {
                    if (flo->miscflag < 0 && acqbtv(NULL,&flo->key1,COMPLF)) {
                         if (!notapped(flf->udate)) {
                              libptr->appwait++;
                         }
                         chuldate(libptr,flf->udate,0);
                         libptr->flags|=LIBCHN;
                         stzcpy(flf->udate,NOTAPPED,DATESZ);
                         upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
                    }
                    prfmsg(NOFILEPR,flf->filname,libptr->libname,
                      (flo->miscflag < 0) ? "; unapproved" : "");
                    outprf(usrnum);
               }
          }
          return(1);
     }
     return(0);
}

int
srch2(void)                        /* sysop search cycle for file checks   */
{
     struct fllib *libptr;
     char *lib;

     setbtv(flfdat);
     lib=flo->key1.libname;
     if (lib[0] != '\0' && (libptr=libfind(lib)) != NULL) {
          if (fndnxt(&flo->fb)) {
               stzcpy(flo->destpath,spr("%s\\%s",libpath(libptr),
                  flo->fb.name),PATHSIZE);
               setbtv(flfdat);
               if (!acqbtv(NULL,compkey(flo->key1.libname,
                  flo->fb.name),COMPLF)
                  && flo->miscflag < 0
                  && strlen(flo->destpath) > 4
                  && sameas(".ZIP",&flo->destpath[strlen(flo->destpath)-4])
                  && openzip()) {
                    flo->retstt=SEARCHEM;
                    prfmsg(usrptr->substt=DIZZYING);
                    outprf(usrnum);
                    prf("");
               }
               else {
                    dofiler("");
               }
               rstbtv();
               return(1);
          }
          if (flo->misclib[0] != '\0') {
               return(0);
          }
     }
     while (1) {
          if ((libptr=(flo->misclib[0] != '\0' ? libfind(flo->misclib)
             : nexthigh(flo->key1.libname))) != NULL) {
               stzcpy(flo->key1.libname,libptr->libname,FLNAMESZ);
               if (!(libptr->flags&FLGDOS)
                  && (!(libptr->flags&FLGCBD) || libptr->path[0] != '\0')) {
                    if (fnd1st(&flo->fb,spr("%s\\*.*",libpath(libptr)),0)) {
                         stzcpy(flo->destpath,spr("%s\\%s",libpath(libptr),
                            flo->fb.name),PATHSIZE);
                         setbtv(flfdat);
                         if (!acqbtv(NULL,compkey(flo->key1.libname,
                            flo->fb.name),COMPLF)
                            && flo->miscflag < 0
                            && strlen(flo->destpath) > 4
                            && sameas(".ZIP",&flo->destpath[strlen(flo->destpath)-4])
                            && openzip()) {
                              flo->retstt=SEARCHEM;
                              prfmsg(usrptr->substt=DIZZYING);
                         }
                         else {
                              dofiler("");
                         }
                         rstbtv();
                         return(1);
                    }
                    else if (flo->misclib[0] != '\0') {
                         return(0);
                    }
               }
               else if (flo->misclib[0] != '\0') {
                    return(0);
               }
          }
          else {
               return(0);
          }
     }
}

int
strsrep(                           /* replace a stg in a stg w/ a stg      */
char *stg,                              /* ...the string to search         */
char *from,                             /* ...the string to be replaced    */
char *to)                               /* ...the string to replace with   */
{
     static char rs[400];
     int loop,len;
     char *o;

     setmem(rs,400,0);
     len=strlen(o=stg);
     if (len >= 200) {
          return(0);
     }
     for (loop=0 ; loop < len ; loop++) {
          if (sameto(from,o)) {
               strcat(rs,to);
               strcat(rs,&o[strlen(from)]);
               strcpy(stg,rs);
               return(1);
          }
          rs[loop]=*o++;
     }
     return(0);
}

int
ulfileq(                           /* user uploaded file query             */
int flag)
{
     struct fndblk fb;
     int retval=0;

     if (fnd1st(&fb,spr("%s\\*.*",userdir(1)),0)) {
          switch (flag) {
          case 2:   /* save file for resume */
               unlink(spr("%s\\..\\%s",userdir(1),fb.name));
               rename(spr("%s\\%s",userdir(1),fb.name),
                      spr("%s\\..\\%s",userdir(1),fb.name));
          case 1:   /* delete file */
               unlink(spr("%s\\%s",userdir(1),fb.name));
          }
          retval++;
          setmisc(fb.name);
     }
     else {
          setmisc("");
     }
     return(retval);
}

long
dnlsec(                            /* time it takes to dl file of size siz */
long siz)
{
     long cps;

     return(siz/((cps=(long)usrptr->baud/10L*(long)comeff/100L) > 0L
        ? cps : 100L));
}

long
dnlsecl(                           /* dnlsec() or 0 if Library free time   */
long siz,
struct fllib *libptr)
{
     return((libptr->flags&FLGFDL) ? 0L : dnlsec(siz));
}

struct fllib *
nexthigh(                          /* get next Library in order            */
char *oldlib)
{
     struct fllib *current,*best=NULL;
     int loop;

     for (loop=0 ; loop < numoflib ; loop++) {
          if ((current=liboff(loop)) != NULL && strlen(current->libname)
            && stricmp(current->libname,oldlib) > 0) {
               if ((best == NULL)
                  || (strcmp(best->libname,current->libname) > 0)) {
                    best=current;
               }
          }
     }
     return(best);
}

void
bubkwds(void)                      /* edit and sort keywords               */
{
     char *current,*new;
     int loop,loop2;

     for (loop=0 ; loop < dargc ; loop++) {
          cutpunc(dargv[loop]);              /* cut symbols                */
          loop2=strlen(dargv[loop]);
          if (loop2 > FLKEYSIZ-1) {          /* shorten long words         */
               dargv[loop][FLKEYSIZ-1]='\0';
          }
          else if (loop2 < 3) {              /* cut really short words     */
               dargc--;
               if (loop != dargc) {
                    dargv[loop--]=dargv[dargc];
               }
          }
     }
     for (loop=0 ; loop < (dargc-1) ; loop++) { /* sort them now           */
          current=dargv[loop];
          for (loop2=(loop+1) ; loop2 < dargc ; loop2++) {
               new=dargv[loop2];
               if (stricmp(current,new) > 0) {
                    dargv[loop2]=current;
                    current=dargv[loop]=new;
               }
          }
     }
     for (loop=0 ; loop < dargc-1 ; loop++) { /* eliminate duplicates      */
          if (stricmp(dargv[loop],dargv[loop+1]) == 0) {
               for (loop2=loop+1 ; loop2 < dargc-1 ; loop2++) {
                    dargv[loop2]=dargv[loop2+1];
               }
               dargc--;
               loop--;
          }
     }
}

void
sortlibs(void)                     /* sort Libraries (check for numbers)   */
{
     static struct fllib swapmem;
     struct fllib *current,*new;
     int loop,loop2;
     long curnum,newnum;

     for (loop=0 ; loop < numoflib ; loop++) { /* eliminate null libs      */
          if (liboff(loop)->libname[0] == '\0') {
               for (loop2=(loop+1) ; loop2 < numoflib ; loop2++) {
                    new=liboff(loop2);
                    movmem(new,liboff(loop2-1),sizeof(struct fllib));
                    setmem(new,sizeof(struct fllib),0);
               }
               numoflib--;
          }
     }
     for (loop=0 ; loop < (numoflib-1) ; loop++) { /* sort them now        */
          current=liboff(loop);
          for (loop2=(loop+1) ; loop2 < numoflib ; loop2++) {
               new=liboff(loop2);
               if ((curnum=atol(current->libname)) == 0L
                  || (newnum=atol(new->libname)) == 0L) {
                    curnum=newnum=0L;
               }
               if (curnum > newnum
                  || stricmp(current->libname,new->libname) > 0) {
                    movmem(current,&swapmem,sizeof(struct fllib));
                    movmem(new,current,sizeof(struct fllib));
                    movmem(&swapmem,new,sizeof(struct fllib));
               }
          }
     }
     chkjoin();
}

void
darsdesc(                          /* parsin() on descedit                 */
int limit,
char chr)
{
     char *sptr;
     int loop;

     sptr=descedit;
     dargc=0;
     for (loop=0 ; loop < limit ; loop++) {
          dargv[loop]=sptr;
          if (sptr[0] != '\0') {
               sptr=strchr(sptr,chr);
               if (sptr == NULL) {
                    sptr=&descedit[strlen(descedit)];
               }
               else {
                    *sptr='\0';
                    sptr++;
               }
               dargc++;
          }
     }
}

char *
lrange(                            /* return range of Library list         */
char *libname)
{
     static char retval[17];

     setmem(retval,17,0);
     if (libname[0] != '\0' && (strlen(libname) < 9)) {
          sprintf(retval,"%s Library",libname);
     }
     else {
          strcpy(retval,"All Libraries");
     }
     return(retval);
}

void
dispfhdr(                          /* display file header                  */
int resetcur)  /* -1=cls but no curs, 0=home no curs, 1=cls, 2=partial cls */
{
     char *sptr;

     setmbk(flmsg);
     if (abs(resetcur) == 1) {
          if (resetcur == 1) {
               btuclo(usrnum);
          }
          prfmsg(CLRSCN);
     }
     else if (resetcur == 2) {
          if (flo->flags&USRANSI) {
               prfmsg(CLRSUB);
               prfmsg(CLRLIST);
               prf("[4f");
          }
          else {
               resetcur=1;
          }
     }
     else {
          btuclo(usrnum);
          prf(" [f");
     }
     if (!(flo->flags&USRANSI)) {
          prf("\r");
     }
     usrptr->substt=(flo->flags&USRANSI) ? FILELST0 : FILELST2;
     if (resetcur > 0) {
          flo->cursor=(flo->flags&REVSRCH) ? NLISTER-1 : 0;
     }
     flo->prop=0;
     if (resetcur != 2) {
          prfmsg(FILELIST);
     }
     else {
          return;
     }
     switch (flo->keymeth) {
     case 0:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST0);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL0 : FILELSX0),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 1:
     case 6:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST1);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL1 : FILELSX1),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 2:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST2);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL2 : FILELSX2),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 3:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST3);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL3 : FILELSX3),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 4:
     case 7:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST4);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL4 : FILELSX4),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 5:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST5);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL5 : FILELSX5),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 8:
          setmisc(lrange(flo->lrange));
          prfmsg(FILELST6);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL6 : FILELSX6),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
          break;
     case 9:
          setmisc(spr("%s (DOS-Only)",flo->lrange));
          prfmsg(FILELST7);
          strsrep(sptr=getmsg((flo->flags&USRANSI) ? FILEHFL7 : FILELSX7),
             "\r","[20f\r");
          prf(sptr);
          prf("[22f");
     }
     setmisc("");
     if (flo->flags&USRANSI) {
          prfmsg(LISTFT);
     }
}

void
dlcharge(                          /* charge for d/l, pay royalty          */
long charge,
struct fllib *l)
{
     if (charge != 0L) {
          charge=hdedcrd(usaptr,charge,0,1);
          if (charge > 0L && l != NULL && l->royal > 0) {
               setbtv(flfdat);
               if (acqbtv(NULL,spc->libname,COMPLF)
                  && !sameas(flf->ulby,"Sysop")) {
                    crdusr(flf->ulby,l2as(charge*(long)l->royal/100L),0,0);
               }
               rstbtv();
          }
     }
}

void
dnlcount(                          /* modify # of times a file was d/l'ed  */
char *libname,
char *filname,
long ct)       /* if negative, adds abs(ct) to it.  if positive, sets it   */
{
     long times;

     setbtv(flfdat);
     if (acqbtv(NULL,compkey(libname,filname),COMPLF)) {
          times=(ct < 0 ? atol(flf->numdls)-ct : ct);
          if (times > 9999999L) {
               times=9999999L;
          }
          sprintf(flf->numdls,NUMSZP,times);
          upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
     }
     rstbtv();
}

void
dofiler(                           /* sysop search cycle warning           */
char *desc)
{
     struct fllib *libptr;
     int loop;

     setbtv(flfdat);
     if (!acqbtv(NULL,compkey(flo->key1.libname,flo->fb.name),COMPLF)) {
          if (flo->miscflag < 0) {
               setmem(flf,FLFILREC+DESCSIZ,0);
               stzcpy(flf->libname,flo->key1.libname,FLNAMESZ);
               stzcpy(flf->filname,flo->fb.name,FLFILENM);
               stzcpy(flf->udate,NOTAPPED,DATESZ);
               flf->utime=now();
               flf->siz=flo->fb.size;
               stzcpy(flf->fdate,dat2srt(ncdate(flo->fb.date)),DATESZ);
               flf->tim=flo->fb.time;
               stzcpy(flf->numdls,spr(NUMSZP,0L),NUMSZ);
               stzcpy(flf->ulby,usaptr->userid,UIDSIZ);
               if (flo->flags&APPTHEM) {
                    stzcpy(flf->udate,dat2srt(ncdate(today())),DATESZ);
                    flf->utime=now();
               }
               stzcpy(flf->desc,desc,DESCSIZ);
               invbtv(NULL,FLFILREC+strlen(flf->desc)+1);
               if ((libptr=libfind(flf->libname)) != NULL) {
                    libptr->numfiles++;
                    if (flo->flags&APPTHEM) {
                         chuldate(libptr,flf->udate,1);
                    }
                    else {
                         libptr->appwait++;
                    }
                    libptr->totbytes+=clfit(flf->siz,libptr->cluster);
                    libptr->flags|=LIBONG;
               }
               if (!longsrch) {
                    if (makwdlst(flf->desc,flf->filname)) {
                         for (loop=0 ; loop < dargc ; loop++) {
                              addkw(dargv[loop],flf->filname,flf->libname);
                         }
                    }
               }
          }
          prfmsg(NORECPRA,flo->fb.name,flo->key1.libname,
                      (flo->miscflag < 0) ? "; added" : "");
          outprf(usrnum);
          prf("");
     }
}

void
gather(void)                            /* read sdi & diz files in         */
{
     struct fndblk fb;
     int lines=0;
     char *path;
     FILE *fh=NULL;

     setmem(descedit,DESCSIZ,0);
     if (fnd1st(&fb,(path=userdir(2)),0)) {
          if ((fh=fopen(path,FOPRB)) != NULL) {
               lines=1;
               readin(1,fh,descedit);
               fclose(fh);
               fh=NULL;
          }
          unlink(path);
     }
     if (fnd1st(&fb,path=userdir(3),0)) {
          if ((fh=fopen(path,FOPRB)) != NULL) {
               readin(NDESCLN-lines,fh,descedit);
               fclose(fh);
               fh=NULL;
          }
          unlink(path);
     }
}

void
loadlibs(void)                     /* creates default lib, loads libs up.  */
{
     char key[FLNAMESZ],*sptr;

     setbtv(flldat);
     if (!valname(deflname)) {
          deflname="MAIN";
     }
     if (!acqbtv(NULL,deflname,0)) {
          curlib=&libedt->lib;
          stzcpy(curlib->libname,deflname,FLNAMESZ);
          stzcpy(curlib->libdesc,defdesc,FLDESCSZ);
          stzcpy(curlib->primary,defplop,UIDSIZ);
          curlib->day=today();
          curlib->time=now();
          setmbk(flmsg);
          stzcpy(curlib->libop,getmsg(DEFLOP),KEYSIZ);
          stzcpy(curlib->autoap,getmsg(DEFAPP),KEYSIZ);
          stzcpy(curlib->keyreq,getmsg(FLVKEY),KEYSIZ);
          stzcpy(curlib->dlkey,getmsg(FLDKEY),KEYSIZ);
          stzcpy(curlib->ulkey,getmsg(DEFUL),KEYSIZ);
          stzcpy(curlib->overw,getmsg(DEFOVR),KEYSIZ);
          curlib->maxfil=10000L;
          curlib->maxbyt=10000000L;
          curlib->maxbup=200000L;
          curlib->cluster=clsize(drvnum(libpath(curlib)));
          setmem(libedt->ldesc,FLLDESCS*NLDLIN,0);
          insbtv(libedt);
          setmem(libedt,sizeof(struct libdisk),0);
          mkdir(spr("%s",deflname));
          shocst(spr("LIBRARY CREATED: %s",deflname),
             "The \"%s\" Library was created",deflname);
     }
     numoflib=0;
     setmem(key,FLNAMESZ,0);
     while (agtbtv(libupd,key,0)) {
          curlib=liboff(numoflib++);
          if (curlib == NULL) {
               catastro("LIBRARY MEMORY ERROR - REBUILD GALFILL.DAT");
          }
          movmem(&libupd->lib,curlib,sizeof(struct fllib));
          if (sameto("_LIBOP",curlib->libop)) {
               curlib->libop[0]='\0';
          }
          if (curlib->path[0] != '\0'
            && !sameas(sptr=fixpath(curlib->path),curlib->path)) {
               stzcpy(curlib->path,sptr,PATHSIZE);
               curlib->flags|=LIBCHN;
          }
          stzcpy(key,curlib->libname,FLNAMESZ);
          mkdir(key);
     }
     setmem(libupd,sizeof(struct libdisk),0);
     sortlibs();
     scanlibs();
}

void
mopitup(void)                      /* clear spec, files, etc.              */
{
     int loop,savusn;
     struct taglib *tag;

     switch (-ftag->status) {
     case TGCOPY:
          if (cdst != NULL) {
               fclose(cdst);
               cdst=NULL;
          }
          if (csrc != NULL) {
               fclose(csrc);
               csrc=NULL;
          }
          break;
     case TGOTHER:
          if (onsysn(uacoff(ftag->usrnum)->userid,1)
              && !(othusp->flags&NOINJO)) {
               setmbk(flmsg);
               prfmlt(BEEP);
               if (othusp->state == flstt && othusp->substt == PLSWAIT) {
                    othusp->substt=MMENU;
               }
               prfmlt(NOTYFILE,ftag->filname,ftag->reflib);
               rstmbk();
               injoth();
               clrmlt();
          }
          break;
     case TGLOCAL:
          if (sameas(ftag->filname,"*")) {
               for (loop=0 ; loop < systags ; loop++) {
                    if (ftag == tagoff(loop)) {
                         unlink(spr("%s\\TAGS%d.FIL",copydir,loop));
                         break;
                    }
               }
          }
          break;
     case TGDLTMP:
          for (loop=0 ; loop < systags ; loop++) {
               if (ftag != (tag=tagoff(loop)) && tag->status > TGEMPTY) {
                    if (sameas(ftag->filname,tag->filname)
                     && sameas(ftag->libname,tag->libname)) {
                         break;
                    }
               }
               if (loop == systags-1) {
                    unlink(spr("%s\\%s\\%s",copydir,ftag->libname,
                       ftag->filname));
               }
          }
     }
     savusn=usrnum;
     if (ftag->usrnum >= 0 && ftag->usrnum < nterms
        && onsysn(uacoff(ftag->usrnum)->userid,1)) {
          usrnum=othusn;
          setftu();
          for (loop=0 ; loop < ftuptr->numftg ; loop++) {
               ftgptr=&ftgusr[loop];
               if (ftag == spc) {
                    removetg(loop--);
               }
          }
     }
     if (savusn >= 0 && savusn < nterms) {
          curusr(savusn);
     }
     setmem(ftag,sizeof(struct taglib),0);
}

void
moploop(void)                      /* mop up all tag entries               */
{
     int loop;

     for (loop=systags-1 ; loop >= 0 ; loop--) {
          ftag=tagoff(loop);
          if (ftag->status < TGEMPTY) {
               mopitup();
          }
     }
}

int
notify(                            /* notify to operators                  */
int logon)                              /* called during logon             */
{
     int doit,loop,beeped=0;
     struct fllib *libptr;

     setmbk(flmsg);
     clrprf();
     prfmsg(NOTIFTOP);
     for (loop=0 ; loop < numoflib ; loop++) {
          doit=0;
          if ((libptr=liboff(loop))->appwait > 0
             && valname(libptr->libname)
             && haskey(libptr->keyreq)) {
               if (!logon || (haskey(flsysop) && notifsys)
                          || (isflop(libptr) && notiflop)) {
                    doit=1;
                    curlib=libptr;
               }
          }
          if (doit) {
               if (!beeped++) {
                    prfmsg(BEEP);
               }
               if (!logon || (beeped < 10)) {
                    if (curlib->appwait == 1) {
                         prfmsg(NOTIFY);
                    }
                    else {
                         prfmsg(NOTIFYM);
                    }
               }
               else {
                    prfmsg(MORENOT);
                    break;
               }
          }
     }
     if (beeped) {
          outprf(usrnum);
     }
     clrprf();
     return(beeped);
}

void
readin(                           /* get contents of a txt file to buffer */
int lines,
FILE *fptr,
char *stg)
{
     int loop,gotline=1;

     for (loop=0 ; (loop < lines) && gotline ; loop++) {
          gotline=readline(fptr);
          stzcpy(&stg[strlen(stg)],input,SDESCLN);
          strcat(stg,"\r");
     }
}

void
regtvars(void)                     /* register all text variables here     */
{
     register_textvar("FL_NAME",tvname);     /* Library name               */
     register_textvar("FL_DESC",tvdesc);     /* Library description        */
     register_textvar("FL_DESC0",tvdesc0);   /* Library long desc. line 1  */
     register_textvar("FL_DESC1",tvdesc1);   /* Library long desc. line 2  */
     register_textvar("FL_DESC2",tvdesc2);   /* Library long desc. line 3  */
     register_textvar("FL_DESC3",tvdesc3);   /* Library long desc. line 4  */
     register_textvar("FL_DESC4",tvdesc4);   /* Library long desc. line 5  */
     register_textvar("FL_FNUM",tvfnum);     /* # of files in Library      */
     register_textvar("FL_FAPP",tvfapp);     /* # of unapp files in lib.   */
     register_textvar("FL_CTIM",tvctim);     /* lib. creation date & time  */
     register_textvar("FL_TAGS",tvtags);     /* # of tagged files open     */
     register_textvar("FL_TSHMSG",tvtshmsg); /* tagged file string         */
     register_textvar("FL_MISC",tvmisc);     /* misc string                */
     register_textvar("FL_INIT",tvinit);     /* init tvars for use outside */
     register_textvar("FL_BYTU",tvbytu);     /* max u/l size               */
     register_textvar("FL_BYTT",tvbytt);     /* max Library size           */
     register_textvar("FL_BYTO",tvbyto);     /* bytes open for upload      */
     register_textvar("FL_FILT",tvfilt);     /* max Library files          */
     register_textvar("FLF_DMIN",tvdnlmin);  /* est. minutes to d/l        */
     register_textvar("FLF_FDATE",tvfdate);  /* DOS date MM/DD/YY or UNAPP */
     register_textvar("FLF_FDATES",tvfdates);/* short MM/YY or "UNAPP"     */
     register_textvar("FLF_NAME",tvfname);   /* filename                   */
     register_textvar("FLF_LIB",tvflib);     /* Library name               */
     register_textvar("FLF_SIZEK",tvfsizek); /* file size in K             */
     register_textvar("FLF_SIZE",tvfsize);   /* file size in bytes         */
     register_textvar("FLF_TAGK",tvtagk);    /* file tag A-? or just A     */
     register_textvar("FLF_TIMES",tvftimes); /* # of times downloaded      */
     register_textvar("FLF_UDATE",tvudate);  /* appdate or "UNAPPROVED"    */
     register_textvar("FLF_ULBY",tvfulby);   /* file uploaded by           */
     register_textvar("COMEFF",tvcomeff);    /* communications efficiency  */
}

void
removetg(                          /* untag a file                         */
int n)
{
     setftu();
     if ((n >= 0) && (n < ftuptr->numftg)) {
          if ((n+1) < ftuptr->numftg) {
               movmem(ftgusr+n+1,ftgusr+n,
                      (ftuptr->numftg-(n+1))*sizeof(struct ftg));
          }
          ftuptr->numftg--;
     }
}

void
strcrep(                           /* replace a char in a string w/ a char */
char *stg,
char from,
char to)
{
     while (*stg != '\0') {
          if (*stg == from) {
               *stg=to;
          }
          stg++;
     }
}

void
unbreak(void)                      /* remove n,q,c message                 */
{
     int savbrk;

     savbrk=usaptr->scnbrk;
     usaptr->scnbrk=CTNUOS;
     rstrxf();
     usaptr->scnbrk=savbrk;
}

void
iflstyp(                           /* ask for type of list user wants      */
char c) {

     int flag=0;

     switch (c) {
     case 'X':
          retmenu(0);
          break;
     case 'D':
     case '\0':
     case '.':
          flag=2;
          break;
     case 'Q':
          flag=7;
          break;
     case 'F':
          flag=3;
          break;
     case 'W':
          flag=6;
          break;
     case 'K':
          flag=4;
          break;
     case 'N':
          flag=5;
     }
     if (flag) {
          dsearch(0,flag);
     }
}

void
iflslib(                           /* current or all Libraries?            */
char c) {

     int flag=0;

     switch (c) {
     case 'X':
          retmenu(0);
          break;
     case 'C':
     case '\0':
     case '.':
          flag=2;
          break;
     case 'A':
          flag=3;
     }
     if (flag) {
          dsearch(1,flag);
     }
}

void
iaskword(                          /* what keywords to search on?          */
char *stg) {

     if (stg[0] == '\0' || sameas(stg,"x")) {
          retmenu(0);
     }
     else if (sameas(stg,"?")) {
          prfmsg(usrptr->substt=ASKWHLPA);
     }
     else if (parsrch(stg)) {
          btumil(usrnum,DFTIMX);
          dsearch(1,((flo->flags&ABOUPRF) ? 1 : flu->mylib));
     }
}

void
iaskname(                          /* where to begin search in filenames   */
char *stg) {

     if (sameas(stg,"x")) {
          retmenu(0);
     }
     if (sameas(stg,".")) {
          *stg='\0';
     }
     stzcpy(flo->miscfil,stg,FLFILENM);
     dsearch(1,((flo->flags&ABOUPRF) ? 1 : flu->mylib));
}

int
ifdesced(                          /* edit file, enter filename            */
char *stg)
{
     stzcpy(flo->key1.libname,curlib->libname,FLNAMESZ);
     stzcpy(flo->key1.filname,stg,FLFILENM);
     setbtv(flfdat);
     if (sameas(stg,"x")) {
          retmenu(1);
     }
     else if (acqbtv(NULL,&flo->key1,COMPLF)) {
          flo->retstt=OPMENU;
          usrptr->substt=PRESENTR;
          bgnfiled(0);
          outprf(usrnum);
          return(1);
     }
     else {
          locsysfl(stg);
     }
     return(0);
}

void
iprefers(                          /* preference menu                      */
char c)
{
     switch (c) {
     case '\0':
     case 'X':
          retmenu(0);
          break;
     case 'P':
          usrptr->substt=TRANSPRT;
          break;
     case 'T':
          usrptr->substt=ASKTYPA;
          break;
     case 'R':
          usrptr->substt=ASKLIB;
          break;
     case 'C':
          flu->askme=flu->mylib=flu->mytyp=0;
          flu->proto[0]='\0';
          updusr(flu);
          prfmsg(OKPCLEAR);
     }
}

void
imanmnu(                           /* Library management menu              */
char c)
{
     switch (c) {
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case '1':
          flo->miscflag=1;
          usrptr->substt=WARNMARK;
          break;
     case '2':
          flo->miscflag=2;
          usrptr->substt=WARNMARK;
          break;
     case '3':
          flo->miscflag=3;
          usrptr->substt=WARNMARK;
          break;
     case '4':
          flo->miscflag=4;
          usrptr->substt=WARNMARK;
     }
}

void
ibigcopy(                          /* begin massive file copy              */
char *stg)
{
     if (stg[0] != '\0') {
          if (sameas(stg,"x")) {
               retmenu(1);
               return;
          }
          else if (sameas(stg,"*")) {
               setmem(&flo->key1,sizeof(struct key1),0);
               flo->miscflag=-2;
               flo->flags|=APPTHEM;
               stzcpy(flo->misclib,curlib->libname,FLNAMESZ);
               prfmsg(usrptr->substt=SEARCHEM);
               unbreak();
               btuinj(usrnum,CYCLE);
               return;
          }
          else if (fnd1st(&flo->fb,stg,0)) {
               while ((flo->fb.attr&FAMDIR) && fndnxt(&flo->fb)) {
               }
               if (!(flo->fb.attr&FAMDIR)) {
                    flo->flags|=AXSUSER;
                    mkdir(userdir(1));
                    unbreak();
                    while (stg[0] != '\0'
                       && stg[strlen(stg)-1] != '\\'
                       && stg[strlen(stg)-1] != ':') {
                         stg[strlen(stg)-1]='\0';
                    }
                    stzcpy(flo->keylist,stg,FLKEYLST);
                    unbreak();
                    btuinj(usrnum,CYCLE);
                    prfmsg(usrptr->substt=BIGCPY);
                    flo->fsrc=flo->fdst=NULL;
                    return;
               }
          }
          prfmsg(NOFILPTH);
     }
}

void
iwarnmrk(                          /* just warn sysop / actually do stuff  */
char *stg)
{

     if (stg[0] != '\0') {
          if (sameas(stg,"x")) {
               usrptr->substt=MANAGMNU;
          }
          else {
               setmem(&flo->key1,sizeof(struct key1),0);
               setmem(flo->misclib,FLNAMESZ,0);
               if (flo->miscflag < 3) {
                    stzcpy(flo->misclib,curlib->libname,FLNAMESZ);
                    if (flo->miscflag == 1) {
                         stzcpy(flo->key1.libname,flo->misclib,FLNAMESZ);
                    }
               }
               if (lingyn(*stg) == 'Y') {
                    flo->miscflag=-flo->miscflag;
               }
               prfmsg(usrptr->substt=SEARCHEM);
               btuinj(usrnum,CYCLE);
          }
     }
}

void
idelfil(                           /* delete files from Library            */
char *stg)
{
     if (stg[0] != '\0') {
          if (sameas(stg,"x") || (nlibtagd() == 0)) {
               setmbk(flmsg);
               retmenu(1);
          }
          else {
               if (lingyn(*stg) == 'Y') {
                    flo->flags|=DELTHEM;
               }
               else {
                    flo->flags&=~DELTHEM;
               }
               usrptr->substt=FILDELA;
               btuinj(usrnum,CYCLE);
          }
     }
}

int
isellib(                           /* select a Library                     */
char *stg)
{
     int retval=1;
     struct fllib *libptr;

     if (stg[0] != '\0') {
          if (sameas(stg,"?")) {
               flo->index=0;
               flo->retstt=usrptr->substt;
               usrptr->substt=LIBLST;
               prfmsg(LIBLST);
               btuinj(usrnum,CYCLE);
          }
          else if (sameas(stg,"x") || sameas(stg,".")) {
               if (sameas(stg,".") && usrptr->substt == SELLIB) {
                    flo->flags|=LONGDSC;
               }
               retmenu(usrptr->substt == SELLIB ? 0 : 1);
          }
          else {
               if ((libptr=libfind(stg)) == NULL || !visilib(libptr)) {
                    prfmsg(INVALIB);
               }
               else if (usrptr->substt == SELLIB3 && (libptr->flags&FLGDOS)) {
                    prfmsg(NODOSJN);
               }
               else if (usrptr->substt == SELLIB2 && !isflop(libptr)) {
                    prfmsg(NOSELOP);
               }
               else {
                    stzcpy(flu->lib,libptr->libname,FLNAMESZ);
                    curlib=setuaxs();
                    if (usrptr->substt == SELLIB) {
                         usrptr->substt=MMENU;
                         flo->flags|=LONGDSC;
                         while (morcnc() == ' ') {
                              cncchr();
                         }
                         if (!morcnc()) {
                              condex();
                         }
                    }
                    else if (usrptr->substt == SELLIB2) {
                         retmenu(1);
                    }
                    else {
                         usrptr->substt=CHAINMNU;
                    }
                    if (curlib == NULL) {
                         retval=0;
                         clrprf();
                    }
               }
          }
     }
     return(retval);
}

int
othrsin(                           /* are others in this lib?              */
char *libname)
{
     int retval=0,saveu;

     saveu=usrnum;
     for (othusn=0 ; othusn < nterms ; othusn++) {
          if (saveu != othusn) {
               if ((othusp=&user[othusn])->class == ACTUSR) {
                    fluoff(othusn);
                    if ((othusp->state == flstt)
                       && sameas(libname,flu->lib)) {
                         othuap=uacoff(othusn);
                         retval=1;
                         break;
                    }
               }
          }
     }
     fluoff(saveu);
     return(retval);
}

int
idellibn(                          /* delete a Library                     */
char *stg)
{
     struct fllib *libptr;

     if (stg[0] == '\0' || sameas(stg,"x")) {
          retmenu(1);
     }
     else if (sameas(stg,"?")) {
          prfmsg(HELPDLIB);
     }
     else if (sameas(stg,deflname)) {
          prfmsg(CANTDEFL);
     }
     else if (valname(stg)) {
          if ((libptr=libfind(stg)) == NULL) {
               prfmsg(LIBNOEXT);
          }
          else if (othrsin(stg)) {
               prfmsg(LIBNODEL,othuap->userid);
          }
          else {
               stzcpy(flo->misclib,libptr->libname,FLNAMESZ);
               untagall(libptr->libname,"");
               setmem(libptr,sizeof(struct fllib),0);
               sortlibs();
               scanlibs();
               curlib=setuaxs();
               flo->miscflag=0;
               prfmsg(usrptr->substt=DELLIB);
               outprf(usrnum);
               btuinj(usrnum,CYCLE);
               return(1);
          }
     }
     return(0);
}

int
itagman(                           /* manage tagged files                  */
char *stg)
{
     int flag,loop;

     flo->retstt=TAGMAN;
     if (stg[0] == '\0' || sameas(stg,"?")) {
          if (ftuptr->numftg > 0) {
               prfmsg(TAGMAN);
          }
          return(0);
     }
     else if (sameas(stg,"X")) {
          if (flo->flags&OFTGMAN) {
               usrptr->substt=PRESENTR;
               flo->retstt=FILELST0;
               inpfunc();
               return(1);
          }
          else {
               retmenu(0);
               return(0);
          }
     }
     if (toupper(*stg) == 'D' || sameas(stg,"ALL")) {
          flo->retstt=usrptr->substt;
          flo->pflg=0;
          if (flu->askme) {
               usrptr->substt=LOGOFF;
               return(0);
          }
          if (!sameas(stg,"ALL") && strlen(stg) > 1) {
               stg++;
          }
          else if (morcnc()) {
               stg=cncall();
          }
          else {
               stg=protocol();
          }
          if (*stg == '?' || *stg == '\0') {
               prfmsg(PROTHDR);
          }
          ftgdnl(sameas(stg,"?") ? "" : stg,retoin);
          outprf(usrnum);
          return(0);
     }
     flag=0;
     if (*stg == '-') {
          stg++;
          flag=1;
     }
     if (sameas(stg,"ALL")) {
          flo->retstt=usrptr->substt;
          clrprf();
          prfmsg(ALLUNTAG);
          ftgdnl("-ALL",retoinx);
          setmbk(flmsg);
          prf("");
          outprf(usrnum);
          prf("");
          if (flo->flags&OFTGMAN) {
               usrptr->substt=PRESENTR;
               flo->retstt=FILELST0;
               inpfunc();
               return(1);
          }
          retmenu(0);
          injacr();
          return(1);
     }
     loop=atoi(stg);
     if (strlen(stg) < 4 && loop && (loop <= ftuptr->numftg)) {
          if (!flag) {
               if (!ftgnew()) {
                    prfmsg(UNTAG1A);
               }
               else {
                    flo->retstt=usrptr->substt;
                    flo->pflg=loop;
                    if (flu->askme) {
                         usrptr->substt=LOGOFF;
                         return(0);
                    }
                    dnlstart(0);
                    outprf(usrnum);
                    return(1);
               }
          }
          else {
               stg--;
               ftgdnl(stg,retoin);
               setmbk(flmsg);
               clrprf();
               if (ftuptr->numftg > 0 || (flo->flags&OFTGMAN)) {
                    retoin();
                    prf("");
               }
               return(1);
          }
     }
     return(0);
}

int
ientdl(                            /* enter fn to download                 */
char *stg)
{
     struct fllib *libptr;
     int flag,curall;
     char *prtcl,*filname;

     curall=defallib;
     setmem(flo->miscfil,FLFILENM,0);
     if (stg[0] == '\0' || sameas(stg,"?")) {
          prfmsg(defallib ? ENTDLHL1 : ENTDLHL2);
          return(0);
     }
     else if (sameas(stg,"x")) {
          retmenu(0);
          return(0);
     }
     while (morcnc() == ' ') {
          cncchr();
     }
     strcpy(flo->savprot,strlen(prtcl=cncall()) < 3 ? prtcl : "");
     if ((filname=strchr(stg,'/')) != NULL
        || (filname=strchr(stg,'\\')) != NULL) {
          *filname++='\0';
          if ((libptr=libfind(stg)) != NULL && visilib(libptr)) {
               stzcpy(flu->lib,libptr->libname,FLNAMESZ);
               curlib=setuaxs();
               if (curlib == NULL) {
                    condex();
                    return(1);
               }
               curall=1;
          }
          stg=filname;
          if (stg[0] == '\0') {
               prfmsg(defallib ? ENTDLHL1 : ENTDLHL2);
               return(0);
          }
     }
     flag=fspex(stg,curall ? "" : curlib->libname,0);
     if (flag != 1) {
          wipetag();
          setmem(flo->miscfil,FLFILENM,0);
          if ((strchr(stg,'*') != NULL) || (strchr(stg,'?') != NULL)) {
               if ((curlib->flags&FLGDOS) && fnd1st(&flo->fb,
                  spr("%s\\FILES",curlib->libname),0)) {
                    prfmsg(NOWLDTAG);
                    return(0);
               }
               stzcpy(flo->miscfil,strupr(stg),FLFILENM);
               flo->fsrc=fopen(spr("%s\\TEMPTAGS.%d",copydir,usrnum),FOPWB);
               if (flo->fsrc != NULL) {
                    setmem(&flo->u,sizeof(union combo),0);
                    stzcpy(flo->u.key4.libname,curlib->libname,FLNAMESZ);
                    flo->keymeth=7;
                    flo->flags&=~REVSRCH;
                    flo->wtnum=0;
                    flo->wtsiz=0L;
                    if (!initsrc(WLDSRCH)) {
                         setmisc(flo->miscfil);
                         prfmsg(usrptr->substt=WLDSRCH);
                         setmisc("");
                         btuinj(usrnum,CYCLE);
                    }
                    if (!(curlib->flags&FLGDOS)) {
                         wldstuff(flo->u.key4.filname);
                    }
                    return(0);
               }
               flo->styp=7;
               srcinit(6,curall ? "" : curlib->libname);
               return(0);
          }
          if (curlib->flags&FLGDOS) {
               prfmsg(NOSFL,stg);
               return(0);
          }
          setbtv(flfdat);
          setmem(flo->miscfil,FLFILENM,0);
          stzcpy(flo->u.key4.filname,stg,FLFILENM);
          if (curall) {
               stzcpy(flo->u.key4.libkey,"",FLNAMESZ);
               if ((agtbtv(NULL,&flo->u.key4,COMPFL)
                  && sameas(flf->filname,stg))
                  || altbtv(NULL,&flo->u.key4,COMPFL)) {
                    stzcpy(flo->miscfil,flf->filname,FLFILENM);
               }
          }
          else {
               stzcpy(flo->u.key1.libname,curlib->libname,FLNAMESZ);
               stzcpy(flo->u.key1.filname,stg,FLFILENM);
               if (altbtv(NULL,&flo->u.key1,COMPLF)) {
                    stzcpy(flo->miscfil,flf->filname,FLFILENM);
               }
          }
          flo->styp=curall && (flag > 1) ? 8 : 3;
          stzcpy(flo->lrange,curall ? "" : curlib->libname,FLNAMESZ);
          srcinit(2,flo->lrange);
     }
     else {
          flag=fspex(stg,curall ? "" : curlib->libname,1);
          if (usrptr->state == flstt) {
               if (flag == -1 && !sameto("T",flo->savprot)) {
                    usrptr->substt=PLSWAIT;
               }
               else if (usrptr->substt != LOGOFF) {
                    prfmsg(ATAGGED,strupr(stg));
                    outprf(usrnum);
                    prf("");
                    retmenu(0);
               }
          }
     }
     return(0);
}

int
fspex(                             /* count up # of exact matches          */
char *stg,
char *lib,
int sub)
{
     static struct key4 key;
     int retval=0,loop;
     struct fllib *libptr;
     char fin[FLFILENM];
     long charge;

     setmem(&key,sizeof(struct key4),0);
     stzcpy(key.filname,stg,FLFILENM);
     stzcpy(fin,stg,FLFILENM);
     if (curlib->flags&FLGDOS) {
          if (!fnd1st(&flo->fb,spr("%s\\%s",libpath(curlib),fin),0)) {
               if (sameas(lib,curlib->libname)) {
                    return(0);
               }
          }
          else {
               stzcpy(fin,flo->fb.name,FLFILENM);
               if (!fndnxt(&flo->fb)) {
                    if (sub) {
                         if (!sameto("T",flo->savprot)
                            && candlnow(curlib->libname,fin) > 0) {
                              flo->retstt=usrptr->substt;
                              if (!submit(curlib->libname,fin,flo->lrange,
                                 DOWNLOD,0)) {
                                   return(0);
                              }
                              if (usrptr->state != flstt) {
                                   rstrxf();
                                   return(1);
                              }
                         }
                         if (submit(curlib->libname,fin,flo->lrange,
                            NOTVIEW,1)) {
                              return((curlib->flags&FLGCBD) ? -1 : 1);
                         }
                         return(0);
                    }
                    if ((strchr(stg,'*') != NULL)
                       || (strchr(stg,'?') != NULL)) {
                         if (fnd1st(&flo->fb,spr("%s\\FILES",curlib->libname),
                            0)) {
                              return(0);
                         }
                    }
                    return(1);
               }
               return(2);
          }
     }
     setbtv(flfdat);
     while (agtbtv(NULL,&key,COMPFL) && sameas(fin,flf->filname)
        && (lib[0] != '\0' || retval < 2)) {
          stzcpy(key.libkey,flf->libname,FLNAMESZ);
          libptr=libfind(flf->libname);
          if (haslibkey(libptr,libptr->dlkey)) {
               if (lib[0] == '\0' || sameas(lib,libptr->libname)) {
                    if (sub) {
                         if (!sameto("T",flo->savprot)
                            && candlnow(flf->libname,flf->filname) > 0) {
                              prfmsg(DLDESC);
                              charge=dnlchg(flf->siz,flf->libname);
                              if (charge != 0L) {
                                   prfmsg(DLDESCC,l2as(charge));
                              }
                              stzcpy(descedit,flf->desc,DESCSIZ);
                              darsdesc(NDESCLN,'\r');
                              for (loop=0 ; loop < NDESCLN ; loop++) {
                                   if (dargv[loop][0] != '\0') {
                                        if (loop == 0) {
                                             prfmsg(DLDESCD);
                                        }
                                        prf("   %s\r",dargv[loop]);
                                   }
                              }
                              if (isripu()) {
                                   prf("\r%c",20);
                              }
                              flo->retstt=usrptr->substt;
                              if (!submit(flf->libname,flf->filname,
                                 flo->lrange,DOWNLOD,0)) {
                                   return(0);
                              }
                              if (usrptr->state != flstt) {
                                   rstrxf();
                                   return(1);
                              }
                         }
                         if (submit(flf->libname,flf->filname,flo->lrange,
                            NOTVIEW,1)) {
                              return((libptr->flags&FLGCBD) ? -1 : 1);
                         }
                         return(0);
                    }
                    retval++;
               }
          }
     }
     rstbtv();
     return(retval);
}

int
ioverw(                            /* overwrite existing file?             */
char *stg)
{
     flo->flags&=~REUPLOD;
     switch (lingyn(*stg)) {
     case '?':
          prfmsg(REUPHLP);
          break;
     case 'Y':
          flo->flags|=REUPLOD;
          flo->retstt=ENTULFIL;
          fileup(flo->miscfil,"?",fupfil);
          outprf(usrnum);
          clrprf();
          return(1);
     default:
          usrptr->substt=ENTULFIL;
     case '\0':
          break;
     }
     return(0);
}

int
ientul(                            /* enter fn to upload                   */
char *stg)
{
     char *wrd;

     setmem(flo->miscfil,FLFILENM,0);
     if (*stg == '\0' || sameas(stg,"x")) {
          retmenu(0);
     }
     else if (sameas(stg,"?")) {
          prfmsg(ULHELP);
     }
     else if (sameas("M",stg)) {
          if (curlib->flags&FLGDOS) {
               prfmsg(NOFILDOS);
          }
          else {
               usrptr->substt=ULMOD;
          }
     }
     else {
          if (okfname(stg)) {
               strcpy(flo->miscfil,stg);
               switch (usrupd(curlib,flo->miscfil)) {
               case 2:   /* file exists, may re-upload, but user has tagged */
                    prf("\rSorry, a user has that file tagged, so it cannot "
                        "be re-uploaded now.\r");
                    return(0);
               case 1:   /* file exists, may re-upload */
                    usrptr->substt=OVERW;
                    return(0);
               case 0:   /* file exists, may not re-upload */
                    prfmsg(CANTOVER);
                    return(0);
               case -1:  /* file does not exist */
                    break;
               }
          }
          if (sameas(stg,"*") || flo->miscfil[0] != '\0') {
               flo->flags&=~REUPLOD;
               flo->retstt=usrptr->substt;
               while (morcnc() == ' ') {
                    cncchr();
               }
               wrd=cncwrd();
               fileup(flo->miscfil,wrd[0] != '\0' ? wrd : "?",fupfil);
               outprf(usrnum);
               clrprf();
               return(1);
          }
     }
     return(0);
}

int
iulmod(                            /* enter name of file to modify         */
char *stg)
{
     stzcpy(flo->key1.libname,curlib->libname,FLNAMESZ);
     stzcpy(flo->key1.filname,stg,FLFILENM);
     setbtv(flfdat);
     if (*stg == '\0') {
          return(0);
     }
     if (sameas(stg,"x")) {
          retmenu(0);
     }
     else if (acqbtv(NULL,&flo->key1,COMPLF)) {
          if (ucanmod()) {
               flo->retstt=usrptr->substt;
               usrptr->substt=PRESENTR;
               bgnfiled(0);
               outprf(usrnum);
               return(1);
          }
     }
     else {
          prfmsg(NOSFL,flo->key1.filname);
     }
     return(0);
}

int
ucanmod(void)                      /* user can modify current file         */
{
     if (isflop(curlib)) {
          return(1);
     }
     if (!sameas(flf->ulby,usaptr->userid)) {
          prfmsg(DIDNU);
          return(0);
     }
     if (!notapped(flf->udate)
        && !haslibkey(curlib,curlib->autoap)
        && !haslibkey(curlib,curlib->overw)) {
          prfmsg(MFAPP);
          return(0);
     }
     return(1);
}

void
exitview(void)
{
     if (flo->atend == LSTNONE) {
          deresume();
     }
     btuclo(usrnum);
     btupbc(usrnum,0);
     prfmsg(CLRSCN);
     rstrxf();
     btutsw(usrnum,usaptr->scnwid);
     dotags(0);
     echonu(usrnum);
     btutrg(usrnum,0);
     if ((flo->flags&USRANSI) && !(flo->flags&UNOINJO)) {
          usrptr->flags&=~NOINJO;
     }
     setftu();
     notidl();
     outprf(usrnum);
     retmenu(0);
}

void
dcurs(                             /* display cursor                       */
int und)
{
     if ((flo->flags&USRANSI) && flo->tdline[flo->cursor] != 0) {
          setmbk(flmsg);
          prfmsg(und ? UNCURSED : CURSED,abs(flo->tdline[flo->cursor])+3,
                  flo->tags[flo->cursor].filname);
          rstmbk();
     }
}

int
curspos(                           /* display (set) cursor                 */
int set)                                /* set = -1 then just display      */
{
     int hi;

     for (hi=NLISTER-1 ; hi >= 0 ; hi--) {
          if (flo->tdline[hi] != 0) {
               break;
          }
     }
     if (set > hi) {
          set=0;
     }
     else if (set < 0) {
          set=(hi >= 0 ? hi : 0);
     }
     if (flo->tdline[set] == 0) {
          for (hi=0 ; hi < NLISTER ; hi++) {
               if (flo->tdline[hi] != 0) {
                    set=hi;
                    break;
               }
          }
     }
     if (flo->cursor != set) {
          dcurs(1);
          flo->cursor=set;
          dcurs(0);
          return(1);
     }
     return(0);
}

void
clstags(void)                      /* clear the on-screen tags             */
{
     int loop;

     for (loop=0 ; loop < NLISTER ; loop++) {
          if (flo->tdline[loop] < 0) {
               flo->tdline[loop]=-flo->tdline[loop];
          }
     }
}

int
ientret(                           /* other file view handler              */
char *stg)
{
     int tagd,loop;

     if (stg[0] == '\0') {
          return(0);
     }
     if (sameas(stg,"&")) {
          if (!dlfrom(flo->tags[flo->hold].libname)) { /* dont let view */
               dlcant();
          }
          else if ((tagd=candlnow(flo->tags[flo->hold].libname
                      ,flo->tags[flo->hold].filname)) > DOWNLOD) {
               dotags(1);
               clstags();
               flo->retstt=usrptr->substt;
               btuclo(usrnum);
               if (!isripu()) {
                    setmbk(flmsg);
                    prfmsg(CLRSCN);
                    rstmbk();
               }
               rstrxf();
               echonu(usrnum);
               btutrg(usrnum,0);
               submit(flo->tags[flo->hold].libname,
                      flo->tags[flo->hold].filname,flo->lrange,tagd,0);
               if (usrptr->state == flstt) {
                    unbreak();
                    btutrg(usrnum,255);
               }
               else {
                    return(1);
               }
          }
          else {
               if (tagd == DOWNLOD) {
                    flo->retstt=usrptr->substt;
                    prfmsg(usrptr->substt=POPWIN3);
               }
               else {
                    flo->retstt=usrptr->substt;
                    prfmsg(usrptr->substt=POPWIN2);
               }
          }
     }
     else if (sameas(stg,"%")) {
          flo->retstt=usrptr->substt;
          if (!dlfrom(flo->tags[flo->hold].libname)) {
               dlcant();
          }
          else if (candlnow(flo->tags[flo->hold].libname
                      ,flo->tags[flo->hold].filname) > 0) {
               dotags(1);
               clstags();
               flo->retstt=usrptr->substt;
               btuclo(usrnum);
               prfmsg(CLRSCN);
               rstrxf();
               echonu(usrnum);
               btutrg(usrnum,0);
               submit(flo->tags[flo->hold].libname,
                      flo->tags[flo->hold].filname,flo->lrange,DOWNLOD,0);
               if (usrptr->state == flstt && usrptr->substt != LOGOFF2) {
                    unbreak();
                    btutrg(usrnum,255);
               }
               return(1);
          }
          else {
               prfmsg(usrptr->substt=POPWIN1);
          }
     }
     else if (sameas(stg,"@")) {
          setfull(totags >= (maxtags-1));
          if (flo->tdline[flo->hold] > 0) {
               if (!(flo->flags&FULLTAG)) {
                    if (dlfrom(flo->tags[flo->hold].libname)) {
                         if (!alrtag(flo->tags[flo->hold].libname,
                            flo->tags[flo->hold].filname)) {
                              flo->tagcnt++;
                         }
                         prfmsg(ENTRETRU);
                         flo->tdline[flo->hold]=-flo->tdline[flo->hold];
                    }
                    else {
                         dlcant();
                    }
               }
          }
          else {
               flo->tdline[flo->hold]=-flo->tdline[flo->hold];
               tagd=alrtag(flo->tags[flo->hold].libname,
                  flo->tags[flo->hold].filname);
               if (tagd == 0) {
                    flo->tagcnt--;
               }
               else {
                    ftag=tagoff(--tagd);
                    for (loop=0 ; loop < ftuptr->numftg ; loop++) {
                         ftgptr=&ftgusr[loop];
                         if (ftag == spc) {
                              removetg(loop--);
                         }
                    }
                    ftag->status=-ftag->status;
               }
               prfmsg(ENTRETRN);
          }
          setfull(totags >= (maxtags-1));
     }
     else if (sameas(stg,"x") || sameas(stg,"!")) {
          btupbc(usrnum,0);
          rebuild(1);
          unbreak();
          usrptr->substt=FILELST0;
          btuinj(usrnum,CYCLE);
     }
     else {
          prfmsg(BEEP);
     }
     return(0);
}

int
iviewasc(                          /* view a file info, ascii              */
char chr)
{
     int loop,flag;

     flag=flo->hold;
     cncall();
     switch (chr) {
     case 'X':
     case '\0':
          usrptr->substt=FILELST1;
          break;
     case 'T':
          if (flo->tdline[flag] != 0) {
               setfull(totags >= (maxtags-1));
               if (!dlfrom(flo->tags[flag].libname)) {
                    prfmsg(NOAXSFIL);
               }
               else if (flo->tdline[flag] > 0) {
                    if (!(flo->flags&FULLTAG)) {
                         if (!alrtag(flo->tags[flag].libname,
                            flo->tags[flag].filname)) {
                              flo->tagcnt++;
                         }
                         prfmsg(ATAGGED,flo->tags[flag].filname);
                         flo->tdline[flag]=-flo->tdline[flag];
                    }
               }
               else {
                    flo->tdline[flag]=-flo->tdline[flag];
                    loop=alrtag(flo->tags[flag].libname,
                       flo->tags[flag].filname);
                    if (loop == 0) {
                         flo->tagcnt--;
                    }
                    else {
                         ftag=tagoff(--loop);
                         for (loop=0 ; loop < ftuptr->numftg ; loop++) {
                              ftgptr=&ftgusr[loop];
                              if (ftag == spc) {
                                   removetg(loop--);
                              }
                         }
                    }
                    setfull(0);
                    prfmsg(AUNTAGED,flo->tags[flag].filname);
               }
               setfull(totags >= (maxtags-1));
          }
          usrptr->substt=FILELST1;
          break;
     case 'D':
          if (!dlfrom(flo->tags[flag].libname)) {
               prfmsg(NOAXSFIL);
               usrptr->substt=FILELST1;
          }
          else if (candlnow(flo->tags[flag].libname,
             flo->tags[flag].filname) > 0) {
               flo->retstt=usrptr->substt;
               btuclo(usrnum);
               submit(flo->tags[flag].libname,flo->tags[flag].filname,
                  flo->lrange,DOWNLOD,0);
               if (usrptr->state != flstt) {
                    rstrxf();
               }
               return(1);
          }
          break;
     case 'V':
          if (!dlfrom(flo->tags[flag].libname)) {
               prfmsg(NOAXSFIL);
               usrptr->substt=FILELST1;
          }
          else if ((loop=candlnow(flo->tags[flag].libname,
             flo->tags[flag].filname)) > DOWNLOD) {
               dotags(1);
               clstags();
               flo->retstt=usrptr->substt;
               btuclo(usrnum);
               prfmsg(CLRSCN);
               outprf(usrnum);
               submit(flo->tags[flag].libname,flo->tags[flag].filname,
                  flo->lrange,loop,0);
               if (usrptr->state != flstt) {
                    rstrxf();
               }
               return(1);
          }
     }
     return(0);
}

void
ifilistx(                          /* main search input handler, non-ansi  */
char *stg)
{
     char chr;
     int loop;
     struct fllib *libptr;

     chr=toupper(*stg);
     for (loop=0 ; loop < NLISTER ; loop++) {
          if (sameas(stg,flo->tags[loop].filname)) {
               chr='A'+loop;
          }
     }
     if (flo->atend&LSTNONE) {
          chr='X';
     }
     switch (chr) {
     case '\0':
          if (!(flo->atend&LSTBACK)) {
               if (!srcincx()) {
                    dotags(0);
                    dispfhdr(2);
                    flo->tagk='A'-1;
                    wipetag();
                    flo->flags&=~STOPLST;
                    return;
               }
               else {
                    usrptr->substt=FILELST2;
               }
          }
          else {
               prfmsg(BTWEND);
               return;
          }
          break;
     case 'X':
          exitview();
          cncall();
          return;
     case 'R':
          rebuild(0);
          usrptr->substt=FILELST1;
          break;
     case '?':
          prfmsg(FILEHELA);
          break;
     }
     if (chr >= 'A' && chr < ('A'+NLISTER)) {
          flo->hold=chr-'A';
          if (flo->tdline[flo->hold] != 0) {
               libptr=libfind(flo->tags[flo->hold].libname);
               if (libptr == NULL || !(libptr->flags&FLGDOS)) {
                    buildview(flo->hold);
               }
               usrptr->substt=VIEWASC;
               return;
          }
     }
}

int
ifilist(                           /* main search input handler            */
char *stg)
{
     int flag,loop,stags;
     char chr;

     if (*stg == '\0') {
          return(0);
     }
     chr=toupper(*stg);
     if (chr >= 'A' && chr < ('A'+NLISTER)) {
          if (!isripu() || flo->cursor != chr-'A') {
               if (flo->tdline[chr-'A'] != 0) {
                    curspos(chr-'A');
               }
               else {
                    prfmsg(BEEP);
               }
               flo->flags|=DBLCLCK;
               return(0);
          }
          else if (flo->flags&DBLCLCK) {
               chr='@';
          }
          else {
               flo->flags|=DBLCLCK;
               return(0);
          }
     }
     flo->flags&=~DBLCLCK;
     if (chr == '@') {
          stags=totags;
          setfull(totags >= (maxtags-1));
          if (flo->tdline[flo->cursor] > 0) {
               if (!(flo->flags&FULLTAG)) {
                    if (dlfrom(flo->tags[flo->cursor].libname)) {
                         if (!alrtag(flo->tags[flo->cursor].libname,
                            flo->tags[flo->cursor].filname)) {
                              flo->tagcnt++;
                         }
                         if (stags == 0) {
                              prfmsg(CTRLT);
                         }
                         prfmsg(TAGGED,flo->tdline[flo->cursor]+3,
                            'A'+flo->cursor);
                         flo->tdline[flo->cursor]=-flo->tdline[flo->cursor];
                    }
                    else {
                         dlcant();
                    }
               }
               else {
                    prfmsg(BEEP);
                    if (flo->flags&USRANSI) {
                         prfmsg(usrptr->substt=POPWIN7);
                    }
               }
          }
          else if (flo->tdline[flo->cursor] < 0) {
               flo->tdline[flo->cursor]=-flo->tdline[flo->cursor];
               loop=alrtag(flo->tags[flo->cursor].libname,
                  flo->tags[flo->cursor].filname);
               if (loop == 0) {
                    flo->tagcnt--;
               }
               else {
                    ftag=tagoff(--loop);
                    for (loop=0 ; loop < ftuptr->numftg ; loop++) {
                         ftgptr=&ftgusr[loop];
                         if (ftag == spc) {
                              removetg(loop--);
                         }
                    }
                    ftag->status=-ftag->status;
               }
               setfull(0);
               prfmsg(UNTAGGED,flo->tdline[flo->cursor]+3,'A'+flo->cursor);
               if (totags == 0) {
                    prfmsg(CLRCTRLT);
               }
          }
          setfull(totags >= (maxtags-1));
          return(0);
     }
     if (chr == '?') {
          prfmsg((flo->flags&STOPLST) ? usrptr->substt=FILHELP : BEEP);
          return(0);
     }
     if (chr == '!') {
          chr='A'+flo->cursor;
          if (chr >= 'A' && chr < ('A'+NLISTER)) {
               flag=chr-'A';
               if (flo->tdline[flag] != 0) {
                    if (flo->keymeth == 9) {
                         flo->retstt=usrptr->substt;
                         prfmsg((flo->flags&STOPLST) ? usrptr->substt=POPWIN4
                            : BEEP);
                    }
                    else {
                         btupbc(usrnum,0);
                         buildview(flag);
                         unbreak();
                         flo->hold=flag;
                    }
               }
               else {
                    prfmsg(BEEP);
               }
          }
          return(0);
     }
     if (chr == '<') {
          curspos(flo->cursor-1);
          return(0);
     }
     if (chr == '>') {
          curspos(flo->cursor+1);
          return(0);
     }
     if (chr == ')') {
          curspos(-1);
          return(0);
     }
     if (chr == '(') {
          curspos(0);
          return(0);
     }
     if (chr == '-' && (flo->atend == LSTFILS || flo->atend == LSTBACK)) {
          if (!srcincx()) {
               flo->flags|=REVSRCH;
               flo->flags&=~STOPLST;
               rset();
               dotags(1);
               dispfhdr(2);
               flo->tagk='A'-1;
               wipetag();
               newpage();
          }
          return(0);
     }
     if (chr == '+' && (flo->atend == LSTFILS || flo->atend == LSTFORE)) {
          if (!srcincx()) {
               flo->flags&=~REVSRCH;
               flo->flags&=~STOPLST;
               rset();
               dotags(1);
               dispfhdr(2);
               flo->tagk='A'-1;
               wipetag();
               newpage();
          }
          return(0);
     }
     flo->hold=flo->cursor;
     if (sameas(stg,"&")) {
          if (flo->tdline[flo->hold] == 0) {
               prfmsg(BEEP);
               return(0);
          }
          if (!dlfrom(flo->tags[flo->hold].libname)) {
               dlcant();
          }
          else if ((flag=candlnow(flo->tags[flo->hold].libname
                      ,flo->tags[flo->hold].filname)) > DOWNLOD) {
               dotags(1);
               clstags();
               flo->retstt=usrptr->substt;
               btuclo(usrnum);
               if (!isripu()) {
                    setmbk(flmsg);
                    prfmsg(CLRSCN);
                    rstmbk();
               }
               rstrxf();
               echonu(usrnum);
               btutrg(usrnum,0);
               submit(flo->tags[flo->hold].libname,
                      flo->tags[flo->hold].filname,flo->lrange,flag,0);
               if (usrptr->state == flstt) {
                    unbreak();
                    btutrg(usrnum,255);
               }
               return(1);
          }
          else {
               if (flag == 1) {
                    flo->retstt=usrptr->substt;
                    prfmsg((flo->flags&STOPLST) ? usrptr->substt=POPWIN3
                       : BEEP);
               }
               else {
                    flo->retstt=usrptr->substt;
                    prfmsg((flo->flags&STOPLST) ? usrptr->substt=POPWIN2
                       : BEEP);
               }
          }
     }
     else if (sameas(stg,"%")) {
          flo->retstt=usrptr->substt;
          dotags(1);
          clstags();
          if (flo->tdline[flo->hold] == 0) {
               prfmsg(BEEP);
               return(0);
          }
          if (!dlfrom(flo->tags[flo->hold].libname)) {
               dlcant();
          }
          else if (candlnow(flo->tags[flo->hold].libname
                      ,flo->tags[flo->hold].filname) > 0) {
               btuclo(usrnum);
               prfmsg(CLRSCN);
               outprf(usrnum);
               rstrxf();
               echonu(usrnum);
               btutrg(usrnum,0);
               submit(flo->tags[flo->hold].libname,
                      flo->tags[flo->hold].filname,flo->lrange,DOWNLOD,0);
               if (usrptr->state == flstt && usrptr->substt != LOGOFF3) {
                    unbreak();
                    btutrg(usrnum,255);
               }
               return(1);
          }
          else {
               prfmsg((flo->flags&STOPLST) ? usrptr->substt=POPWIN1 : BEEP);
          }
     }
     else if (sameas(stg,"#")) {
          dotags(1);
          clstags();
          if (ftuptr->numftg > 0) {
               flo->flags|=OFTGMAN;
               flo->retstt=usrptr->substt;
               usrptr->substt=TAGMAN;
               echonu(usrnum);
               btutrg(usrnum,0);
               btuclo(usrnum);
               prfmsg(CLRSCN);
               prfmsg(DEFRIPM);
               outprf(usrnum);
               prf("");
          }
          else if (isripu()) {
               prfmsg(BEEP);
          }
          return(0);
     }
     else if (sameas(stg,"x")) {
          exitview();
     }
     else {
          prfmsg(BEEP);
     }
     return(0);
}

void
itranspr(                          /* default transfer protocol            */
char chr)
{
     if ((chr == '\0') || (chr == '?') || (chr == '.')) {
          strcpy(flu->proto,"?");
     }
     else if (chr == 'X') {
          cncall();
          usrptr->substt=PREFERS;
          return;
     }
     else if (valdpc(spr("%c",chr)) && chr != 'T') {
          stzcpy(flu->proto,spr("%c%c",chr,
                 ((chr == 'Z') && (morcnc() == 'R')) ? 'R' : '\0'),3);
          if ((chr == 'Z') && (morcnc() == 'R')) {
               cncchr();
          }
          if ((chr == 'M') || (chr == 'C') || (chr == '1') || (chr == 'F')) {
               prfmsg(PROTNOTE);
          }
     }
     else {
          cncall();
          prfmsg(INVALPRT);
          return;
     }
     if (sameas(flu->proto,defproto)) {
          setmem(flu->proto,3,0);
     }
     if (flu->proto[0] == '\0' && sameas(defproto,"?")
        || sameas(flu->proto,"?")) {
          flu->askme=0;
          usrptr->substt=PREFERS;
          cncall();
     }
     else {
          usrptr->substt=ASKOFF;
     }
     updusr(flu);
}

void
irename(                           /* rename or maybe overwrite            */
char *stg)
{
     struct fndblk fb;
     struct taglib *tag;
     int loop;

     if (sameas(stg,"x")) {
          unlink(spr("%s\\%s",userdir(1),flo->miscfil));
          if (flo->dodesc != NEVRASK) {
               flo->dodesc=ASKDESC;
          }
          if (ulfileq(0)) {
               usrptr->substt=AUTOLOG;
               btuinj(usrnum,CYCLE);
          }
          else {
               usrptr->substt=ENTULFIL;
          }
          return;
     }
     if (sameas(stg,"*")) {
          for (loop=0 ; loop < systags ; loop++) {
               tag=tagoff(loop);
               if (tag->status > TGEMPTY && tag->usrnum != usrnum
                  && sameas(tag->filname,flo->miscfil)
                  && sameas(tag->libname,curlib->libname)) {
                    prf("\rAnother user "
                        "has this file, %s, tagged for download!\r",
                        flo->miscfil);
                    return;
               }
          }
          setbtv(flfdat);
          if (acqbtv(NULL,compkey(curlib->libname,flo->miscfil),COMPLF)
             && (haslibkey(curlib,curlib->overw) || (sameas(flf->ulby,
             usaptr->userid) && (notapped(flf->udate) || haslibkey(curlib,
             curlib->autoap))))) {
               stzcpy(flo->key1.libname,curlib->libname,FLNAMESZ);
               stzcpy(flo->key1.filname,flo->miscfil,FLFILENM);
               stzcpy(flo->srcpath,spr("%s\\%s",userdir(1),flo->miscfil),
                  PATHSIZE);
               stzcpy(flo->destpath,spr("%s\\%s",libpath(curlib),flo->miscfil),
                  PATHSIZE);
               flo->fsrc=fopen(flo->srcpath,FOPRB);
               flo->fdst=fopen(flo->destpath,FOPWB);
               if (flo->fsrc != NULL) {
                    if (flo->fdst != NULL) {
                         if (fnd1st(&fb,flo->srcpath,0)) {
                              curlib->totbytes-=clfit(flf->siz,
                                 curlib->cluster);
                              flf->siz=fb.size;
                              curlib->totbytes+=clfit(flf->siz,
                                 curlib->cluster);
                              if (notapped(flf->udate)) {
                                   curlib->appwait--;
                              }
                              chuldate(curlib,flf->udate,0);
                              if (!notapped(flf->udate)
                                 || haslibkey(curlib,curlib->autoap)) {
                                   stzcpy(flf->udate,dat2srt(ncdate(today())),
                                      DATESZ);
                              }
                              chuldate(curlib,flf->udate,1);
                              if (notapped(flf->udate)) {
                                   curlib->appwait++;
                              }
                              flf->utime=now();
                              stzcpy(flf->fdate,dat2srt(ncdate(fb.date)),
                                 DATESZ);
                              flf->tim=fb.time;
                              stzcpy(flf->ulby,usaptr->userid,UIDSIZ);
                              upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
                              prfmsg(usrptr->substt=COPYING);
                              btuinj(usrnum,CYCLE);
                              flo->retstt=AUTOLOG;
                              curlib->flags|=LIBCHN;
                              return;
                         }
                    }
                    fclose(flo->fsrc);
                    flo->fsrc=NULL;
               }
               if (flo->fdst != NULL) {
                    fclose(flo->fdst);
                    flo->fdst=NULL;
               }
          }
          prfmsg(CANTOVER);
     }
     else if (okfname(stg) && !rename(spr("%s\\%s",userdir(1),flo->miscfil),
                                      spr("%s\\%s",userdir(1),stg))) {
          usrptr->substt=AUTOLOG;
          flo->miscfil[0]='\0';
          btuinj(usrnum,CYCLE);
     }
     else if (stg[0] != '\0') {
          prfmsg(CANTRNAM);
     }
}

void
ichain(                            /* manage joining libs                  */
char chr)
{
     switch (chr) {
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case '?':
          prfmsg(CHAINHLP);
          flo->retstt=usrptr->substt;
          usrptr->substt=PRESENTR;
          break;
     case 'L':
          prfmsg(CHAINLST);
          listjoin();
          flo->retstt=usrptr->substt;
          usrptr->substt=PRESENTR;
          break;
     case 'J':
          flo->index=0;
          usrptr->substt=CHUNJ;
          break;
     case 'U':
          scanlib(0,curlib);
          if (chjoined(curlib,curlib->libname)) {
               curlib->libs--;
          }
          if (curlib->libs == 0) {
               prfmsg(CHAINO);
          }
          else {
               flo->index=1;
               usrptr->substt=CHUNJ;
          }
          break;
     case 'S':
          if (flo->nlibaxs > 1) {
               usrptr->substt=SELLIB3;
          }
          else {
               prfmsg(NOOLIB);
          }
     }
}

void
ijoin(                             /* join inputted Library up             */
char *stg)
{
     struct fllib *libptr;

     if (stg[0] == '\0' || sameas(stg,"x")) {
          usrptr->substt=CHAINMNU;
     }
     else {
          libptr=libfind(stg);
          if (libptr == NULL) {
               prfmsg(CHNOS);
          }
          else if (libptr == curlib) {
               prfmsg(NOJNSELF);
          }
          else if (libptr->flags&FLGDOS) {
               prfmsg(NOJNDOS);
          }
          else {
               if (chjoin(curlib,libptr,flo->index)) {
                    prfmsg(flo->index ? CHUJOIN : CHJOIN);
                    scanlibs();
               }
               else {
                    prfmsg(flo->index ? CHUJOINX : CHJOINX);
               }
          }
     }
}

void
iflblist(                          /* create a list of libs menu           */
char chr)
{
     int all,multi=0;

     if (chr == 'X' || chr == '\0') {
          retmenu(0);
     }
     if (chr == 'A') {
          all=1;
     }
     else if (chr == 'C') {
          all=0;
     }
     else {
          return;
     }
     while (morcnc() == ' ') {
          cncchr();
     }
     stzcpy(flo->savprot,cncall(),3);
     flo->flags|=AXSUSER;
     flo->fsrc=fopen(spr("%s\\%s.x%02X",copydir,(all ? "FILES!" : curlib->libname),
        channel[usrnum]),FOPWA);
     if (flo->fsrc == NULL) {
          prfmsg(FLBERR);
          return;
     }
     wipetag();
     flo->styp=3;
     setmem(flo->keylist,FLKEYLST,0);
     setmem(&flo->u,sizeof(union combo),0);
     if (!all) {
          if (scanlib(1,curlib)) {
               multi=1;
          }
          scanlib(0,curlib);
     }
     clrprf();
     flo->flags&=~REVSRCH;
     if (all || multi) {
          flo->keymeth=4;
          stzcpy(flo->u.key4.libname,all ? "" : curlib->libname,FLNAMESZ);
          setmem(flo->u.key4.filname,FLFILENM,0);
     }
     else {
          flo->keymeth=1;
          stzcpy(flo->u.key1.libname,curlib->libname,FLNAMESZ);
          setmem(flo->u.key1.filname,FLFILENM,0);
     }
     prfmsg(all ? LSTHDRA : LSTHDRB ,all ? "" : curlib->libname);
     if (all) {
          flo->flags|=LISTALL;
     }
     else {
          flo->flags&=~LISTALL;
     }
     if (isansiu()) {
          flo->flags|=USRANSI;
     }
     else {
          flo->flags&=~USRANSI;
     }
     if (!(flo->flags&USRANSI) || stplist) {
          stpans(prfbuf);
     }
     strcrep(prfbuf,13,10);
     fprintf(flo->fsrc,"%s",prfbuf);
     clrprf();
     if (!initsrc(LSTCYC)) {
          prfmsg(usrptr->substt=LSTCYC);
          btuinj(usrnum,CYCLE);
     }
}

int
inpfunc(void)                      /* the big status3-input function       */
{
     int retval=1;
     char *stg,chr,*path;

     switch (usrptr->substt) {
     case CYCIDX:
          break;
     case PLSWAIT:
          if (sameas(cncall(),"x")) {
               retmenu(0);
          }
          break;
     case AUTOGONE:
          prfmsg(BANGAVT);
          retmenu(0);
          break;
     case TAGSTAT:
          prfmsg(CLRSCN);
          retmenu(1);
          break;
     case LSTCYC:
          if (flo->fsrc != NULL) {
               fclose(flo->fsrc);
               flo->fsrc=NULL;
          }
          prfmsg(LSTABT);
          retmenu(0);
          break;
     case FLBLIST:
          iflblist(cncchr());
          break;
     case WLDSYS:
     case WLDSRCH:
          flo->wtnum=-(flo->wtnum+1);
          break;
     case INDDELK:
          break;
     case RESOLVE:
          break;
     case CLIBNAM:
          iclibnam(cncall());
          break;
     case DOSO:
          idoso(cncchr());
          break;
     case VIEWASC:
          if (morcnc() == ' ') {
               cncchr();
          }
          if (iviewasc(cncchr())) {
               return(2);
          }
          break;
     case POPWIN6:
          exitview();
          break;
     case POPWIN1:
     case POPWIN2:
     case POPWIN3:
     case POPWIN4:
     case POPWIN5:
     case POPWIN7:
     case FILHELP:
          if (cncall()[0] != '\0') {
               if (usrptr->substt == FILHELP) {
                    prtbuild(1,14);
                    usrptr->substt=FILELST0;
               }
               else if (flo->retstt == ENTRETT) {
                    usrptr->substt=ENTRETT;
                    buildview(flo->hold);
               }
               else {
                    prtbuild(5,6);
                    usrptr->substt=FILELST0;
               }
          }
          break;
     case RENNAME:
          if ((stg=cncall())[0] == '\0' || sameas(stg,"x")) {
               usrptr->substt=DELMENU;
          }
          else if (sameas(stg,flo->miscfil)) {
               prfmsg(INVALFN);
          }
          else if (okfname(strupr(stg))) {
               path=(strlen(curlib->path) ? curlib->path : curlib->libname);
               if (rename(spr("%s\\%s",path,flo->miscfil),spr("%s\\%s",path,
                  stg))) {
                    prfmsg(INVALFN);
               }
               else {
                    setbtv(flfdat);
                    if (acqbtv(NULL,compkey(curlib->libname,flo->miscfil),
                       COMPLF)) {
                         stzcpy(flf->filname,stg,FLFILENM);
                         stzcpy(flo->destpath,stg,FLFILENM);
                         upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
                    }
                    prfmsg(RENAMED);
                    if (!longsrch) {
                         usrptr->substt=RENCYC;
                         btuinj(usrnum,CYCLE);
                    }
                    else {
                         usrptr->substt=DELMENU;
                    }
               }
          }
          else {
               prfmsg(INVALFN);
          }
          break;
     case RENCYC:
          break;
     case UNAPPING:
     case APPING:
          retmenu(1);
          break;
     case DELLIB:
          break;
     case FILMOVA:
          break;
     case CHAINMNU:
          ichain(cncchr());
          break;
     case CHUNJ:
          ijoin(cncall());
          break;
     case ENTUADNM:
          ienuadnm(cncwrd());
          break;
     case FLSTYPA:
          iflstyp(cncchr());
          break;
     case FLSLIB:
          iflslib(cncchr());
          break;
     case ASKWORD:
          iaskword(cncall());
          break;
     case ASKNAME:
          iaskname(cncwrd());
          break;
     case FDESCED:
          if (ifdesced(cncall())) {
               return(2);
          }
          break;
     case PREFERS:
          iprefers(cncchr());
          break;
     case RENAME:
          irename(cncall());
          break;
     case MMENU:
          if (strlen(nxtcmd) > 1) {
               switch (toupper(*nxtcmd)) {
               case 'S':
                    if (sameto("s?",nxtcmd)) {
                         prfmsg(DEFRIPM);
                         outprf(usrnum);
                         prf("");
                    }
                    else if (!sameto("s ",nxtcmd)) {
                         usrptr->flags&=~CONCEX;
                    }
                    else if (sameto("s ?",nxtcmd)) {
                         cncall();
                         flo->index=0;
                         flo->retstt=usrptr->substt;
                         usrptr->substt=LIBLST;
                         prfmsg(DEFRIPM);
                         prfmsg(LIBLST);
                         outprf(usrnum);
                         prf("");
                         btuinj(usrnum,CYCLE);
                         return(1);
                    }
                    break;
               }
          }
          retval=lsmenu();
          break;
     case MANAGMNU:
          imanmnu(cncchr());
          break;
     case BIGCPY:
          break;
     case AUTOLOG:
          break;
     case BIGCOPY:
          ibigcopy(cncall());
          break;
     case WARNMARK:
          iwarnmrk(cncall());
          break;
     case SEARCHEM:
          prfmsg(SEARCHAB);
          retmenu(1);
          break;
     case DELFIL:
          idelfil(cncall());
          break;
     case FILDELA:
     case FILDELB:
          setmisc("0");
          flo->retstt=DELMENU;
          ftgdnl("-ALL",retoinl);
          setmisc("");
          return(2);
     case APPMENU:
          iappmenu(cncchr());
          break;
     case DELMENU:
          idelmenu(cncchr());
          break;
     case MOVMENU:
          imovmenu(cncchr());
          break;
     case MOVPATH:
          imovpath(cncall());
          break;
     case UNAPPMNU:
          iunapp(cncchr());
          break;
     case OPMENU:
          switch (retval=oprmenu()) {
          case 2:
               return(2);
          case 0:
               retmenu(0);
          default:
               retval=1;
          }
          break;
     case LIBLST:
          usrptr->substt=flo->retstt;
     case SELLIB:
     case SELLIB2:
     case SELLIB3:
          retval=isellib(cnclib());
          break;
     case LIBEDITN:
          break;
     case DELLIBNM:
          if (idellibn(cnclib())) {
               return(2);
          }
          break;
     case PRESENTR:
          switch (usrptr->substt=flo->retstt) {
          case FILELST0:
               rebuild(1);
               btutrg(usrnum,255);
               btuinj(usrnum,CYCLE);
               break;
          case ENTRETT:
               unbreak();
               buildview(flo->hold);
               btutrg(usrnum,255);
               break;
          case PRESENTR:
          case MMENU:
               retmenu(0);
               break;
          case OPMENU:
               retmenu(1);
               break;
          case AUTOLOG:
               btuinj(usrnum,CYCLE);
          }
          break;
     case ENTDLFIL:
          if (ientdl(cncwrd())) {
               return(2);
          }
          break;
     case TAGMAN:
          if (itagman(cncwrd())) {
               return(2);
          }
          break;
     case ASKWHLPA:
     case ASKWHLPB:
     case ASKWHLPC:
          if (sameas(cncall(),"x")) {
               usrptr->substt=ASKWORD;
          }
          else {
               switch (usrptr->substt) {
               case ASKWHLPA:
                    prfmsg(usrptr->substt=ASKWHLPB);
                    break;
               case ASKWHLPB:
                    prfmsg(usrptr->substt=ASKWHLPC);
                    break;
               case ASKWHLPC:
                    usrptr->substt=ASKWORD;
               }
          }
          break;
     case LOGOFF:
          if (sameas((stg=cncall()),"x")) {
               usrptr->substt=TAGMAN;
               break;
          }
          if (sameas(stg,"?")) {
               break;
          }
          dnlstart(lingyn(*stg) == 'Y');
          outprf(usrnum);
          return(2);
     case LOGOFF2:
     case LOGOFF3:
          if (sameas((stg=cncall()),"x")) {
               usrptr->state=0;
               retoin();
               return(2);
          }
          if (sameas(stg,"?")) {
               break;
          }
          dnlstart(lingyn(*stg) == 'Y');
          outprf(usrnum);
          return(2);
     case OVERW:
          if (ioverw(cncall())) {
               return(2);
          }
          break;
     case ENTULFIL:
          if (ientul(cncwrd())) {
               return(2);
          }
          break;
     case ULMOD:
          if (iulmod(cncall())) {
               return(2);
          }
          break;
     case COPYING:
          break;
     case DIZZYING:
          break;
     case KEYWORDY:
          break;
     case ENTRETT:
          if (ientret(cncall())) {
               return(2);
          }
          break;
     case FILELST2:
     case FILELST1:
          ifilistx(cncwrd());
          break;
     case FILELST0:
          if (ifilist(cncall())) {
               return(2);
          }
          break;
     case TRANSPRT:
          itranspr(cncchr());
          break;
     case ASKOFF:
          chr=cncyesno();
          cncall();
          if (lingyn(chr) == 'Y') {
               flu->askme=1;
               updusr(flu);
          }
          else if (chr != 'X') {
               flu->askme=0;
               updusr(flu);
          }
          usrptr->substt=PREFERS;
          break;
     case ASKTYPA:
          chr=cncchr();
          cncall();
          if (chr == 'X' || chr == 'X') {
               usrptr->substt=PREFERS;
               break;
          }
          switch (chr) {
          case 'P':
               flu->mytyp=1;
               break;
          case 'D':
               flu->mytyp=2;
               break;
          case 'F':
               flu->mytyp=3;
               break;
          case 'K':
               flu->mytyp=4;
               break;
          case 'N':
               flu->mytyp=5;
               break;
          case 'W':
               flu->mytyp=6;
               break;
          case 'Q':
               flu->mytyp=7;
               break;
          default:
               return(1);
          }
          usrptr->substt=PREFERS;
          updusr(flu);
          break;
     case ASKLIB:
          chr=cncchr();
          cncall();
          if (chr == '\0' || chr == 'X') {
               usrptr->substt=PREFERS;
               break;
          }
          switch (chr) {
          case 'P':
               flu->mylib=1;
               usrptr->substt=PREFERS;
               break;
          case 'C':
               flu->mylib=2;
               usrptr->substt=PREFERS;
               break;
          case 'A':
               flu->mylib=3;
               usrptr->substt=PREFERS;
          }
          if (usrptr->substt == PREFERS) {
               updusr(flu);
          }
          break;
     case DODESC:
          chr=cncchr();
          cncall();
          if (chr != '\0') {
               if (chr == '?') {
                    prfmsg(DESCHELP);
               }
               else {
                    flo->dodesc=(lingyn(chr) == 'Y' ? DOADESC :
                       (chr == 'X' ? NEVRASK : NIXDESC));
                    unbreak();
                    usrptr->substt=AUTOLOG;
                    btuinj(usrnum,CYCLE);
               }
          }
          break;
     default:
          retmenu(0);
     }
     return(retval);
}

int
editgo(                            /* begin editing ul file descrp.        */
char *filen)
{
     struct fndblk fb;
     char *dir,*udate;

     dir=userdir(1);
     if (filen[0] == '\0') {
          if (!fnd1st(&fb,spr("%s\\*.*",dir),0)) {
               return(0);
          }
     }
     else if (!fnd1st(&fb,spr("%s\\%s",dir,filen),0)) {
          return(0);
     }
     udate=(haslibkey(curlib,curlib->autoap) ? dat2srt(ncdate(today()))
          : NOTAPPED);
     return(insfile(curlib,dir,&fb,udate,usaptr->userid,""));
}

int
writable(                          /* is Library writable?                 */
struct fllib *libptr)
{
     return(!(libptr->flags&FLGRDO)
        && !(libptr->flags&FLGCBD));
}

int
lsmenu(void)                      /* main menu choice handler             */
{
     flo->flags&=~OFTGMAN;
     switch (cncchr()) {
     case 'O':
          if (isflop(curlib)) {
               deresume();
               prfmsg(DEFRIPM);
               flo->flags|=RETUOPM;
               usrptr->substt=OPMENU;
               return(1);
          }
          break;
     case 'P':
          prfmsg(DEFRIPM);
          usrptr->substt=PREFERS;
          return(1);
     case 'D':
          if (dlfroml(curlib)) {
               prfmsg(DEFRIPM);
               usrptr->substt=ENTDLFIL;
               return(1);
          }
          break;
     case 'T':
          if (ftuptr->numftg > 0) {
               prfmsg(DEFRIPM);
               usrptr->substt=TAGMAN;
               flo->retstt=TAGMAN;
               return(1);
          }
          break;
     case 'L':
          if (haskey(listkey) && !(curlib->flags&FLGDOS)) {
               deresume();
               usrptr->substt=FLBLIST;
               return(1);
          }
          break;
     case 'U':
          if (haslibkey(curlib,curlib->ulkey) && writable(curlib)) {
               deresume();
               prfmsg(DEFRIPM);
               flo->flags&=~RETUOPM;
               flo->miscflag=0;
               if (ulfileq(0)) {
                    flo->miscfil[0]='\0';
                    flo->dodesc=ASKDESC;
                    usrptr->substt=AUTOLOG;
                    btuinj(usrnum,CYCLE);
               }
               else {
                    usrptr->substt=ENTULFIL;
               }
               return(1);
          }
          break;
     case 'S':
          if (flo->nlibaxs > 1) {
               prfmsg(DEFRIPM);
               usrptr->substt=SELLIB;
               return(1);
          }
          break;
     case 'F':
          if (curlib->flags&FLGDOS) {
               sprintf(flo->srcpath,"%s\\FILES",curlib->libname);
               if (fnd1st(&flo->fb,flo->srcpath,0)) {
                    flo->retstt=PRESENTR;
                    listing(flo->srcpath,retoinl);
                    return(2);
               }
               else {
                    srcinit(-1,curlib->libname);
                    return(1);
               }
          }
          prfmsg(DEFRIPM);
          dsearch(0,((flo->flags&ABOUPRF) ? 1 : flu->mytyp));
          return(1);
     case 'R':
          if (flo->tagk >= 'A' && flo->styp != 4 && flo->styp != 6
             && flo->styp != -1) {
               clstags();
               if (!isansiu()) {
                    flo->flags&=~USRANSI;
                    clstags();
                    rebuild(0);
                    usrptr->substt=FILELST1;
               }
               else {
                    flo->flags|=USRANSI;
                    setbusy();
                    btutrg(usrnum,255);
                    rebuild(1);
               }
               btuinj(usrnum,CYCLE);
               return(1);
          }
          break;
     case 'X':
          return(0);
     case '\0':
     case '?':
          prfmsg(MMHLP);
          return(1);
     }
     prfmsg(NOTCMD);
     retmenu(0);
     return(1);
}

int
fupfil(                            /* file upload handler                  */
int code)
{
     int retval=0;
     char *stg,*new;
     struct fllib *libptr=NULL;

     fluoff(usrnum);
     flo->flags|=AXSUSER;
     mkdir(stg=userdir(1));
     switch (code) {
     case FUPPTH:
          if (flo->miscfil[0] != '\0') {
               strcpy(ftfscb->fname,flo->miscfil);
          }
          sprintf(ftfbuf,"%s\\%s",stg,ftfscb->fname);
          retval=rename(spr("%s\\..\\%s",stg,ftfscb->fname),ftfbuf) ? 1 : 2;
          break;
     case FUPBEG:
          libptr=libfind(flu->lib);
          if (libptr != NULL) {
               if (libroom(libptr) <= 0L && !haskey(flsysop)) {
                    strcpy(ftfbuf,"Too many files already in this Library.");
                    break;
               }
               if (!haskey(flsysop)) {
                    ftfscb->maxbyt=byteroom;
               }
          }
          if (flo->miscfil[0] != '\0') {
               strcpy(ftfscb->fname,flo->miscfil);
          }
          sprintf(ftfbuf,"%s\\%s",stg,ftfscb->fname);
          if (libptr != NULL && writable(libptr)) {
               zapcrd((libptr->flags&FLGFUL) || freeuls,0L,NULL);
               retval=1;
               if (ftuptr->flags&FTBANG) {   /* replace FTF autologoff     */
                    ftuptr->flags&=~FTBANG;
                    flo->flags|=OFFUPLD;
               }
          }
          break;
     case FUPEND:
          sprintf(ftfbuf,"%s\\%s",stg,ftfscb->fname);
          if (!okfname(ftfscb->fname)) {
               if (!rename(ftfbuf,new=spr("%s\\TMP.%03d",stg,
                  ftfscb->actfil))) {
                    strcpy(ftfscb->fname,spr("TMP.%03d",ftfscb->actfil));
                    strcpy(ftfbuf,new);
               }
          }
          rstcrd();
          if (flu->lib[0] != '\0') {
               libptr=libfind(flu->lib);
          }
          if (libptr != NULL && (auditall || (libptr->flags&FLGAUL))) {
               shocst("LIBRARY FILE UPLOAD","User %s uploaded %s\\%s",
                      usaptr->userid,flu->lib,ftfscb->fname);
          }
          sv.uplds++;
          break;
     case FUPSKP:
          if (flu->lib[0] != '\0' && (libptr=libfind(flu->lib)) != NULL
             && (auditall || (libptr->flags&FLGAUL))) {
               shocst("LIBRARY FILE UPLOAD ABORTED",
                      "User %s aborted upld of %s\\%s",
                      usaptr->userid,flu->lib,ftfscb->fname);
          }
          flo->flags&=~OFFUPLD;
          unlink(stg=spr("%s\\%s",userdir(0),ftfscb->fname));
          rename(spr("%s\\%s",userdir(1),ftfscb->fname),stg);
          break;
     case FUPFIN:
          retval=1;
     case FUPHUP:
          usrptr->state=flstt;
          usrptr->substt=flo->retstt;
          if (code != FUPFIN) {
               lshang();
          }
          else {
               if (ulfileq(0)) {
                    flo->miscfil[0]='\0';
                    if (flo->flags&(REUPLOD|OFFUPLD)) {
                         flo->dodesc=NEVRASK;
                    }
                    else {
                         flo->dodesc=ASKDESC;
                    }
                    usrptr->substt=AUTOLOG;
                    btuinj(usrnum,CYCLE);
               }
               else {
                    setmbk(flmsg);
                    unprop();
                    prfmsg(usrptr->substt);
                    outprf(usrnum);
                    clrprf();
                    rstmbk();
               }
          }
          btulok(usrnum,0);
          rstcrd();
     }
     return(retval);
}

int
parsrch(                 /* parse search string, primitive to flo->keylist */
char *stg)
{
     char *current,*new;
     int loop,loop2;

     while (strsrep(stg,"?","")) {
     }
     while (samend(stg," OR")) {
          stg[strlen(stg)-3]='\0';
     }
     while (samend(stg," AND")) {
          stg[strlen(stg)-4]='\0';
     }
     while (samend(stg," XOR")) {
          stg[strlen(stg)-4]='\0';
     }
     while (samend(stg," NOT")) {
          stg[strlen(stg)-4]='\0';
     }
     while (strsrep(stg," AND "," & ")) {
     }
     while (strsrep(stg," OR "," | ")) {
     }
     while (strsrep(stg," XOR "," ^ ")) {
     }
     while (strsrep(stg," NOT "," ! ")) {
     }
     if (sameto("NOT ",stg)) {
          strsrep(stg,"NOT ","!");
     }
     stzcpy(descedit,stg,DESCSIZ);
     strupr(descedit);
     strcrep(descedit,'&',' ');
     strcrep(descedit,'|',' ');
     strcrep(descedit,'^',' ');
     strcrep(descedit,'!',' ');
     strcrep(descedit,'(',' ');
     strcrep(descedit,')',' ');
     while (strsrep(descedit,"  "," ")) {
     }
     while (*descedit == ' ') {
          strcpy(descedit,&descedit[1]);
     }
     darsdesc(FLKEYLST/2,' ');
     if (dargc <= 0) {
          return(0);
     }
     for (loop=0 ; loop < dargc ; loop++) {
          stzcpy(&vdaptr[loop*FLKEYSIZ],dargv[loop],FLKEYSIZ);
     }
     flo->kcount=dargc;
     if (longsrch || countc(stg,'|') || countc(stg,'^') || countc(stg,'!')
        || countc(stg,'(')) {
          flo->styp=6;
     }
     if (flo->styp == 4) {
          setbtv(flkdat);
          for (loop=0 ; loop < dargc ; loop++) {
               stzcpy(flo->kwdy.keyword,dargv[loop],sizeof(struct key2));
               if (!agebtv(NULL,&flo->kwdy,COMPKFL)
                || !sameto(dargv[loop],kyd->keyword)) {
                    loop=1000;
               }
          }
          if (loop >= 1000) {
               setmisc(flo->kwdy.keyword);
               prfmsg(NOWORD);
               setmisc("");
               return(0);
          }
     }
     for (loop=0 ; loop < (dargc-1) ; loop++) { /* sort by length          */
          current=dargv[loop];
          for (loop2=(loop+1) ; loop2 < dargc ; loop2++) {
               new=dargv[loop2];
               if (strlen(new) > strlen(current)) {
                    dargv[loop2]=current;
                    current=dargv[loop]=new;
               }
          }
     }
     stzcpy(new=flo->keylist,stg,FLKEYLST);
     for (loop=0 ; loop < dargc ; loop++) {
          current=dargv[loop];
          if (current[0] != '\0') {
               while (strsrep(new,current,"?")) {
               }
          }
     }
     do {
          loop2=0;
          while (strsrep(new," ","")) {
               loop2=1;
          }
          while (strsrep(new,"??","?&?")) {
               loop2=1;
          }
          while (strsrep(new,"?!","?&!")) {
               loop2=1;
          }
          while (strsrep(new,"&&","&")) {
               loop2=1;
          }
          while (strsrep(new,"!&","!")) {
               loop2=1;
          }
          while (strsrep(new,"&|","|")) {
               loop2=1;
          }
          while (strsrep(new,"|&","|")) {
               loop2=1;
          }
          while (strsrep(new,"&^","^")) {
               loop2=1;
          }
          while (strsrep(new,"^&","^")) {
               loop2=1;
          }
          while (strsrep(new,"!!","")) {
               loop2=1;
          }
          while (strsrep(new,"()","")) {
               loop2=1;
          }
          while (strsrep(new,"(&","(")) {
               loop2=1;
          }
          while (strsrep(new,"(|","(")) {
               loop2=1;
          }
          while (strsrep(new,"(^","(")) {
               loop2=1;
          }
          while (strsrep(new,"&)",")")) {
               loop2=1;
          }
          while (strsrep(new,"|)",")")) {
               loop2=1;
          }
          while (strsrep(new,"||","|")) {
               loop2=1;
          }
          while (strsrep(new,"^)",")")) {
               loop2=1;
          }
          while (strsrep(new,"^^","^")) {
               loop2=1;
          }
     } while (loop2);
     loop2=0;
     for (stg=new ; *stg != '\0' ; stg++) {
          if (*stg == '(') {
               loop2++;
          }
          else if (*stg == ')') {
               loop2--;
          }
          if (loop2 < 0) {
               break;
          }
     }
     if (loop2 != 0) {
          prfmsg(PARENS);
          return(0);
     }
     while (new[0] == '|') {
          strcpy(&new[0],&new[1]);
     }
     while (new[0] == '^') {
          strcpy(&new[0],&new[1]);
     }
     while (new[0] == '&') {
          strcpy(&new[0],&new[1]);
     }
     while (new[strlen(new)-1] == '!') {
          new[strlen(new)-1]='\0';
     }
     return(1);
}

void
dnlstart(                          /* start d/l                            */
int logoff)
{
     char *stg,*prot;
     struct ftg tmpftg;

     stg=spr("%s%s",protocol(),logoff ? "!" : "");
     if (flo->pflg != 0 && flo->pflg <= ftuptr->numftg) {
          movmem(&ftgusr[flo->pflg-1],&tmpftg,sizeof(struct ftg));
          removetg(flo->pflg-1);
          if (ftgnew()) {
               movmem(&tmpftg,ftgptr,sizeof(struct ftg));
               ftgptr->flags&=~FTGABL;
               if (*stg == '?' || *stg == '\0') {
                    prfmsg(PROTHDR);
               }
               ftgdn1(stg,retoin);
          }
     }
     else {
          prot=protocol();
          if (*prot == '?' || *prot == '\0') {
               prfmsg(PROTHDR);
          }
          ftgdnl(sameas(prot,"?") ? "" : stg,retoin);
     }
}

void
rebuild(                           /* reconstruct screen display           */
int cls)
{
     int loop;

     if (cls == 1) {
          unbreak();
     }
     dispfhdr((flo->flags&USRANSI) ? (cls ? -1 : 0) : 2);
     setbtv(flfdat);
     for (loop=0 ; loop < NLISTER ; loop++) {
          flo->tagk='A'-1+loop;
          if (flo->tdline[loop] != 0 && (flo->keymeth == 9
             || acqbtv(NULL,compkey(flo->tags[loop].libname,
             flo->tags[loop].filname),COMPLF))) {
               srcdrec(1);
          }
     }
     if (cls != 0) {
          srcpaus(1);
     }
     prf("[3f");
     outprf(usrnum);
     prf("");
}

void                               /* partial reconstruct screen display   */
prtbuild(
int start,
int run)
{
     int loop;

     dispfhdr(0);
     clrprf();
     setbtv(flfdat);
     flo->tagk='A'-1;
     for (loop=0 ; loop < NLISTER ; loop++) {
          if (loop >= start) {
               prf("[%df[K",loop+4);
          }
          if (flo->tdline[loop] != 0 && (flo->keymeth == 9
             || acqbtv(NULL,compkey(flo->tags[loop].libname,
             flo->tags[loop].filname),COMPLF))) {
               srcdrec(1);
               if (loop < start) {
                    clrprf();
               }
          }
          else {
               flo->tagk++;
          }
          if (loop >= (start+run-1)) {
               break;
          }
     }
     prf("[3f");
     outprf(usrnum);
}

void
iappmenu(                          /* approval of files menu               */
char chr)
{
     switch (chr) {
     case 'T':
          if (getaufil(0,0)) {
               flo->autol=0;
               prfmsg(usrptr->substt=APPING);
               btuinj(usrnum,CYCLE);
          }
          else {
               prfmsg(NONAPP);
          }
          break;
     case 'A':
          stzcpy(flo->u.key0.libname,curlib->libname,FLNAMESZ);
          setmem(flo->u.key0.udate,DATESZ,0);
          setmem(flo->u.key0.filname,FLFILENM,0);
          flo->autol=1;
          prfmsg(usrptr->substt=APPING);
          btuinj(usrnum,CYCLE);
          break;
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case 'N':
          flo->retstt=usrptr->substt;
          usrptr->substt=ENTUADNM;
     }
}

void
iunapp(                          /* unapproval of files menu               */
char chr)
{
     switch (chr) {
     case 'T':
          if (getaufil(0,1)) {
               prfmsg(usrptr->substt=UNAPPING);
               btuinj(usrnum,CYCLE);
          }
          else {
               prfmsg(NONUAPP);
          }
          break;
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case 'N':
          flo->retstt=usrptr->substt;
          usrptr->substt=ENTUADNM;
     }
}

void
idelmenu(                          /* deletion of files menu               */
char chr)
{
     switch (chr) {
     case 'T':
          if (nlibtagd()) {
               setmbk(flmsg);
               usrptr->substt=DELFIL;
          }
          else {
               prfmsg(CANTNUKE);
          }
          break;
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case 'N':
          flo->retstt=usrptr->substt;
          usrptr->substt=ENTUADNM;
          break;
     case 'R':
          if (!writable(curlib)) {
               prfmsg(CANTRNML);
          }
          else {
               flo->retstt=-usrptr->substt;
               usrptr->substt=ENTUADNM;
          }
          break;
     case 'A':
          if (!(curlib->flags&FLGDOS)) {
               stzcpy(flo->u.key0.libname,curlib->libname,FLNAMESZ);
               setmem(flo->u.key0.udate,DATESZ,0);
               setmem(flo->u.key0.filname,FLFILENM,0);
               flo->autol=2;
               usrptr->substt=APPING;
               prfmsg(DELING);
               btuinj(usrnum,CYCLE);
          }
     }
}

void
imovmenu(                          /* move files menu                      */
char chr)
{
     switch (chr) {
     case 'T':
          if (nlibtagd()) {
               setmbk(flmsg);
               usrptr->substt=MOVPATH;
               setmem(flo->miscfil,FLFILENM,0);
          }
          else {
               prfmsg(CANTNUKE);
          }
          flo->flags&=~MOVFLAG;
          break;
     case '\0':
     case 'X':
          retmenu(1);
          break;
     case 'N':
          flo->flags&=~MOVFLAG;
          flo->retstt=usrptr->substt;
          usrptr->substt=ENTUADNM;
          break;
     case 'C':
          flo->flags|=MOVFLAG;
          flo->retstt=usrptr->substt;
          usrptr->substt=ENTUADNM;
          break;
     case 'A':
          if (nlibtagd()) {
               setmbk(flmsg);
               usrptr->substt=MOVPATH;
               setmem(flo->miscfil,FLFILENM,0);
          }
          else {
               prfmsg(CANTNUKE);
          }
          flo->flags|=MOVFLAG;
     }
}

void
imovpath(                          /* move files to what Library           */
char *stg)
{
     struct fllib *libptr;

     if (stg[0] == '\0' || sameas(stg,"x") || (nlibtagd() == 0
        && flo->miscfil[0] == '\0')) {
          setmbk(flmsg);
          retmenu(1);
          return;
     }
     if ((libptr=libfind(stg)) == NULL) {
          prfmsg(CHNOS);
          return;
     }
     if (libptr->flags&FLGDOS) {
          prfmsg(NOTODOS);
          return;
     }
     if (!isflop(libptr)) {
          prfmsg(MOVAXS);
          return;
     }
     if (libptr == curlib) {
          prfmsg(NOCANMOV);
          return;
     }
     if (!writable(libptr)) {
          prfmsg(MOVRDO);
          return;
     }
     stzcpy(flo->misclib,libptr->libname,FLNAMESZ);
     if ((strchr(flo->miscfil,'*') != NULL)
        || (strchr(flo->miscfil,'?') != NULL)) {
          flo->fsrc=fopen(spr("%s\\TEMPTAGS.%d",copydir,usrnum),
             FOPWB);
          if (flo->fsrc != NULL) {
               setmem(&flo->u,sizeof(union combo),0);
               stzcpy(flo->u.key4.libname,curlib->libname,FLNAMESZ);
               flo->keymeth=7;
               flo->flags&=~REVSRCH;
               flo->wtnum=0;
               wldstuff(flo->u.key4.filname);
               if (!initsrc(WLDSYS)) {
                    setmisc(flo->miscfil);
                    prfmsg(usrptr->substt=WLDSYS);
                    setmisc("");
               }
               else {
                    return;
               }
          }
          else {
               usrptr->substt=ENTUADNM;
               return;
          }
     }
     else {
          flo->retstt=usrptr->substt;
          usrptr->substt=FILMOVA;
     }
     unbreak();
     btuinj(usrnum,CYCLE);
}

void
ienuadnm(                          /* enter name of file for operations    */
char *stg)
{
     struct fllib *libptr;
     int update=0;

     if (stg[0] == '\0' || sameas(stg,"x")) {
          usrptr->substt=abs(flo->retstt);
          return;
     }
     if ((strchr(stg,'*') != NULL) || (strchr(stg,'?') != NULL)) {
          if (flo->retstt == -DELMENU) {
               prfmsg(CANTWLDS);
               return;
          }
          stzcpy(flo->miscfil,strupr(stg),FLFILENM);
          if (flo->retstt == MOVMENU) {
               usrptr->substt=MOVPATH;
               return;
          }
          flo->fsrc=fopen(spr("%s\\TEMPTAGS.%d",copydir,usrnum),FOPWB);
          if (flo->fsrc != NULL) {
               setmem(&flo->u,sizeof(union combo),0);
               stzcpy(flo->u.key4.libname,curlib->libname,FLNAMESZ);
               flo->keymeth=7;
               flo->flags&=~REVSRCH;
               flo->wtnum=0;
               if (!initsrc(WLDSYS)) {
                    setmisc(flo->miscfil);
                    prfmsg(usrptr->substt=WLDSYS);
                    setmisc("");
                    unbreak();
                    btuinj(usrnum,CYCLE);
               }
               wldstuff(flo->u.key4.filname);
               return;
          }
     }
     setbtv(flfdat);
     if (!acqbtv(NULL,compkey(curlib->libname,stg),COMPLF)) {
          locsysfl(stg);
          return;
     }
     switch (flo->retstt) {
          case APPMENU:
               if (!notapped(flf->udate)) {
                    prfmsg(NOTNEED,"");
               }
               else {
                    stzcpy(flf->udate,dat2srt(ncdate(today())),DATESZ);
                    flf->utime=now();
                    update=1;
                    curlib->appwait--;
                    curlib->flags|=LIBCHN;
                    chuldate(curlib,flf->udate,1);
               }
               break;
          case UNAPPMNU:
               if (notapped(flf->udate)) {
                    prfmsg(NOTNEED,"un");
               }
               else {
                    chuldate(curlib,flf->udate,0);
                    stzcpy(flf->udate,NOTAPPED,DATESZ);
                    flf->utime=now();
                    update=2;
                    curlib->appwait++;
                    curlib->flags|=LIBCHN;
               }
               break;
          case -DELMENU: /* renaming file    */
               stzcpy(flo->miscfil,flf->filname,FLFILENM);
               usrptr->substt=RENNAME;
               break;
          case DELMENU:
               stzcpy(flo->miscfil,flf->filname,FLFILENM);
               libptr=curlib;
               if (notapped(flf->udate) && (libptr->appwait > 0)) {
                    libptr->appwait--;
               }
               chuldate(libptr,flf->udate,0);
               libptr->totbytes-=clfit(flf->siz,libptr->cluster);
               libptr->numfiles--;
               libptr->flags|=LIBCHN;
               delbtv();
               untagall(flf->libname,flf->filname);
               prfmsg(DELED);
               if (!longsrch) {
                    prfmsg(usrptr->substt=INDDELK);
                    flo->flags&=~WLDRSLV;
                    btuinj(usrnum,CYCLE);
               }
               else {
                    usrptr->substt=flo->retstt;
               }
               break;
          case MOVMENU:
               stzcpy(flo->miscfil,flf->filname,FLFILENM);
               usrptr->substt=MOVPATH;
               break;
          default:
               usrptr->substt=flo->retstt;
               return;
     }
     if (update) {
          upvbtv(NULL,FLFILREC+strlen(flf->desc)+1);
          prfmsg((update == 2) ? UNAPPED : APPED);
          if (status != CYCLE) {
               prf("\r");
          }
          usrptr->substt=flo->retstt;
     }
}

void
idoso(                             /* want this to be a DOS-only Library?  */
char chr)
{
     cncall();
     curlib=&libedt->lib;
     if (lingyn(chr) == 'X') {
          setmem(curlib->libname,FLNAMESZ,0);
          retmenu(1);
          return;
     }
     if (lingyn(chr) == 'Y') {
          curlib->flags|=FLGDOS;
     }
     setmem(curlib->libdesc,FLDESCSZ,0);
     curlib->day=today();
     curlib->time=now();
     stzcpy(curlib->libop,getmsg(DEFLOP),KEYSIZ);
     stzcpy(curlib->autoap,getmsg(DEFAPP),KEYSIZ);
     stzcpy(curlib->keyreq,getmsg(FLVKEY),KEYSIZ);
     stzcpy(curlib->overw,getmsg(DEFOVR),KEYSIZ);
     stzcpy(curlib->dlkey,getmsg(FLDKEY),KEYSIZ);
     stzcpy(curlib->ulkey,getmsg(DEFUL),KEYSIZ);
     curlib->maxfil=10000L;
     curlib->maxbyt=10000000L;
     curlib->maxbup=200000L;
     cncall();
     bgnlibed();
}

void
iclibnam(                          /* enter Library name to create         */
char *stg)
{
     if (sameas(stg,"X")) {
          retmenu(1);
          return;
     }
     if (valname(stg)) {
          if (libfind(stg) == NULL) {
               curlib=&libedt->lib;
               stzcpy(curlib->libname,strupr(stg),FLNAMESZ);
               usrptr->substt=DOSO;
          }
          else {
               prfmsg(LIBEXT);
          }
     }
     else {
          prfmsg(NAMEOLIB);
     }
}

int
oprmenu(void)                     /* operator menu choice handler         */
{
     char chr,*uid;

     if (!isflop(curlib)) {
          return(0);
     }
     switch (chr=cncchr()) {
     case 'X':
          return(0);
     case '\0':
     case '?':
          cncall();
          prfmsg(OPMHLP);
          return(1);
     case 'I':
          if (!notify(0)) {
               prfmsg(NONEAPP);
          }
          flo->retstt=OPMENU;
          usrptr->substt=PRESENTR;
          return(1);
     case 'S':
          usrptr->substt=SELLIB2;
          return(1);
     case 'C':
          if (haskey(flsysop)) {
               if (numoflib == numlib+numladd) {
                    prfmsg(NOMEMLIB);
               }
               else {
                    uid=isedtlib();
                    if (uid == NULL) {
                         usrptr->substt=CLIBNAM;
                    }
                    else {
                         prfmsg(LIBNOEDT,uid);
                    }
               }
               return(1);
          }
          break;
     case 'U':
     case 'A':
          if (!(curlib->flags&FLGDOS)) {
               deresume();
               usrptr->substt=(chr == 'U' ? UNAPPMNU : APPMENU);
               return(1);
          }
          break;
     case 'R':
          if (!(curlib->flags&FLGDOS)) {
               deresume();
               usrptr->substt=DELMENU;
               return(1);
          }
          break;
     case 'V':
          if (!(curlib->flags&FLGDOS)) {
               if ((uid=inmvcpy()) == NULL) {
                    usrptr->substt=MOVMENU;
                    flo->flags|=INMVCPY;
               }
               else {
                    prfmsg(SYSMVCPY,uid);
               }
               return(1);
          }
          break;
     case 'F':
          if (!(curlib->flags&FLGDOS)) {
               usrptr->substt=FDESCED;
               return(1);
          }
          break;
     case 'T':
          if (haskey(flsysop)) {
               usrptr->substt=TAGSTAT;
               flo->index=0;
               flo->flags&=~TLSTDSP;
               btuinj(usrnum,CYCLE);
               return(1);
          }
          break;
     case 'D':
          if (haskey(flsysop)) {
               usrptr->substt=DELLIBNM;
               return(1);
          }
          break;
     case 'L':
          if (haskey(flsysop) && !(curlib->flags&FLGDOS)) {
               if (writable(curlib)) {
                    flo->miscflag=1;
                    usrptr->substt=BIGCOPY;
               }
               else {
                    prfmsg(CANTULR);
               }
               return(1);
          }
          break;
     case 'M':
          if (haskey(flsysop) && !(curlib->flags&FLGDOS)) {
               usrptr->substt=MANAGMNU;
               flo->flags&=~APPTHEM;
               return(1);
          }
          break;
     case 'E':
          cncall();
          if (haskey(flsysop)) {
               uid=isedtlib();
               if (uid == NULL) {
                    bgnlibed();
                    return(2);
               }
               else {
                    prfmsg(LIBNOEDT,uid);
                    return(1);
               }
          }
          break;
     case 'J':
          if (!(curlib->flags&FLGDOS)) {
               usrptr->substt=CHAINMNU;
               return(1);
          }
          break;
     }
     prfmsg(NOTCMD);
     retmenu(1);
     return(1);
}

char *
inmvcpy(void)                      /* is a sysop moving/copying files      */
{
     for (othusn=0 ; othusn < nterms ; othusn++) {
          if (othusn != usrnum) {
               othusp=&user[othusn];
               if (othusp->class == ACTUSR) {
                    if (othusp->state == flstt) {
                         if (((struct fluson *)ptrblok(flomem,
                            othusn))->flags&INMVCPY) {
                              return(uacoff(othusn)->userid);
                         }
                    }
               }
          }
     }
     return(NULL);
}

void
loadldsc(                          /* load description up for lib          */
struct libdisk *libdptr)           /* null = use curlib and libupd         */
{
     struct fllib *lib;
     struct libdisk *libact;
     int loop;

     if (libdptr == NULL) {
          if (curlib == NULL) {
               setmem(libupd,sizeof(struct libdisk),0);
               return;
          }
     }
     setbtv(flldat);
     lib=(libdptr != NULL ? &libdptr->lib : curlib);
     libact=(libdptr != NULL ? libdptr : libupd);
     if (acqbtv(libupd,lib->libname,0)) {
          if (libdptr != NULL) {
               for (loop=0 ; loop < NLDLIN ; loop++) {
                    stzcpy(libact->ldesc[loop],libupd->ldesc[loop],FLLDESCS);
               }
          }
     }
     else {
          for (loop=0 ; loop < NLDLIN ; loop++) {
               libact->ldesc[loop][0]='\0';
          }
     }
     rstbtv();
}

void
bgnlibed(void)                     /* begin edit or creation of Library    */
{
     char *stg;

     if (curlib != &libedt->lib) {
          movmem(curlib,&libedt->lib,sizeof(struct fllib));
          curlib=&libedt->lib;
     }
     loadldsc(libedt);
     if (isansiu()) {
          fsdrhd(spr("%s",getmsg(EDITLIBO)));
          outprf(usrnum);
          libeditp(usrptr->substt=LIBEDITA,1);
          stg=getasc(fsdusr->tmpmsg);
          if (curlib->flags&FLGDOS) {
               fsdscb->flddat[FLDHID].flags|=FFFAVD;
               tpwipe(stg,FLDHID,3,3);
               fsdscb->flddat[FLDCBD].flags|=FFFAVD;
               tpwipe(stg,FLDCBD,3,3);
               fsdscb->flddat[FLDROY].flags|=FFFAVD;
               tpwipe(stg,FLDROY,3,1);
               fsdscb->flddat[FLDOVW].flags|=FFFAVD;
               tpwipe(stg,FLDOVW,4,1);
               fsdscb->flddat[FLDAPU].flags|=FFFAVD;
               tpwipe(stg,FLDAPU,4,1);
          }
          fsdbkg(stg);
          fsdscb->flddat[FLDNAME].flags|=FFFAVD;
          fsdscb->flddat[FLDDATE].flags|=FFFAVD;
          fsdscb->flddat[FLDTIME].flags|=FFFAVD;
          fsdscb->flddat[FLDPATH].flags|=FFFNSP;
          fsdscb->flddat[FLDVIS].flags|=FFFNSP;
          fsdscb->flddat[FLDDLD].flags|=FFFNSP;
          fsdscb->flddat[FLDULD].flags|=FFFNSP;
          fsdscb->flddat[FLDOVW].flags|=FFFNSP;
          fsdscb->flddat[FLDAPU].flags|=FFFNSP;
          fsdscb->flddat[FLDLOP].flags|=FFFNSP;
          if (auditall) {
               fsdscb->flddat[FLDAUL].flags|=FFFAVD;
               fsdscb->flddat[FLDADL].flags|=FFFAVD;
          }
          if (freeuls) {
               fsdscb->flddat[FLDFUL].flags|=FFFAVD;
          }
          fsdscb->flddat[FLDDOSO].flags|=FFFAVD;
          fsdego(libeditv,libeditd);
     }
     else {
          libeditp(LIBEDITN,-1);
          btuoes(usrnum,1);
          usrptr->substt=LIBEDITN;
     }
     outprf(usrnum);
     flo->flags|=ISEDLIB;
}

void
cdsptags(void)                     /* display files a user has tagged      */
{
     while (btuoba(usrnum) > (outbsz-1024)) {
          if (flo->index == 0) {
               prfmsg(TAGSTAT);
          }
          ftag=tagoff(flo->index);
          if (ftag != NULL && ftag->status != TGEMPTY
             && ftag->usrnum >= 0 && ftag->usrnum < nterms) {
               prf(" %-8.8s %-12.12s %-29.29s [%d] ",ftag->libname,
                  (sameas("*",ftag->filname) ? ftag->wldspc : ftag->filname),
                  uacoff(ftag->usrnum)->userid,ftag->status);
               if (sameas("*",ftag->filname)) {
                    prf("files: %d, bytes: %s",ftag->nwld,l2as(ftag->size));
               }
               else {
                    switch (ftag->status) {
                    case TGLOCAL:
                         prf("DL ready, local drive");
                         break;
                    case TGINWAIT:
                         prf("Wait for local copy");
                         break;
                    case TGCOPY:
                         prf("Being copied locally");
                         break;
                    case TGOTHER:
                         prf("Waiting for copy");
                         break;
                    case TGSERVER:
                         prf("Being copied");
                         break;
                    case TGDLTMP:
                         prf("DL ready, temp area");
                         break;
                    default:
                         if (ftag->status < 0) {
                              prf("D/L over, clean up");
                         }
                         else {
                              prf("???");
                         }
                    }
               }
               prf("\r");
               flo->flags|=TLSTDSP;
               outprf(usrnum);
          }
          if (++flo->index >= systags) {
               if (!(flo->flags&TLSTDSP)) {
                    clrprf();
                    prfmsg(NONETAGD);
               }
               else {
                    prfmsg(TAGSTATF);
               }
               retmenu(1);
               opmenu();
               prfmsg(OPMENU);
               outprf(usrnum);
               prf("");
               return;
          }
     }
     btuinj(usrnum,CYCLE);
     prf("");
     outprf(usrnum);
     prf("");
}

void
wipetag(void)                      /* clear tag flags                      */
{
     int loop=0;

     while (loop < NLISTER) {
          flo->tdline[loop++]=0;
     }
     flo->atend=LSTFILS;
}

int
libchg(void)                       /* update changed LIBs                  */
{
     struct fllib *libptr;
     int loop,mark=-1,retval=0;

     for (loop=0 ; loop < numoflib ; loop++) {
          libptr=liboff(loop);
          if (libptr->flags&LIBONG) {
               libptr->flags|=LIBCHN;
               libptr->flags&=~LIBONG;
          }
          else if (libptr->flags&LIBCHN) {
               mark=loop;
               break;
          }
     }
     if (mark >= 0) {
          libptr=liboff(mark);
          libptr->flags&=~LIBCHN;
          setbtv(flldat);
          if (acqbtv(libupd,libptr->libname,0)) {
               movmem(libptr,&libupd->lib,sizeof(struct fllib));
               updbtv(libupd);
          }
          setmem(libupd,sizeof(struct libdisk),0);
          rstbtv();
          retval=1;
     }
     return(retval);
}

void
spinner(void)                      /* display a little spinner if busy     */
{
     static char *blade=" /-\\|";
     static unsigned numstep,oldtck=0;
     int tmr;

     if ((tmr=ticker) != oldtck) {
          numstep=0;
          oldtck=tmr;
     }
     if (numstep++ < 10) {
          if (flo->prop++ > 0) {
               prf("");
          }
          if (flo->prop >= strlen(blade)) {
               flo->prop=1;
          }
          prf("%c",blade[flo->prop]);
          outprf(usrnum);
          prf("");
     }
}

void
locsysfl(                          /* locate a file for the sysop or libop */
char *filname)
{
     char libname[FLNAMESZ];
     struct fllib *libptr;
     int ctr=0;

     setmem(libname,FLNAMESZ,0);
     setbtv(flfdat);
     setmbk(flmsg);
     prfmsg(SYSFILES);
     while (agtbtv(NULL,rcompkey(libname,filname),COMPFL)
        && sameas(flf->filname,filname)) {
          stzcpy(libname,flf->libname,FLNAMESZ);
          if ((libptr=libfind(libname)) != NULL && isflop(libptr)) {
               prf("%s ",libname);
               ctr++;
          }
     }
     if (ctr == 0) {
          clrprf();
          prfmsg(NOFILIB);
     }
     else {
          prfmsg(SUMFILES);
     }
     rstmbk();
}

void
notidl(void)                       /* notify user of files to d/l          */
{
     switch (uwaitcpy(usrnum)) {
     case 0:
          if (totags > 0) {
               prfmsg(NOTDLNW);
          }
          break;
     case 1:
          prfmsg(NOTDLWT);
     }
}

int
notapped(                           /* is file unapproved?                 */
char *udate)
{
     return(udate[0] == '\0' || sameas(udate,NOTAPPED));
}

long
libroom(                           /* how much room for an upload?         */
struct fllib *libptr)
{
     if (libptr->flags&FLGDOS) {
          cntdir(spr("%s\\*.*",libpath(libptr)));
     }
     else {
          numfils=libptr->numfiles;
          numbytp=libptr->totbytes;
     }
     byteroom=min(libptr->maxbup,libptr->maxbyt-numbytp);
     return(libptr->maxfil-numfils);
}

char *
libpath(                           /* return path to lib's files           */
struct fllib *libptr)
{
     return(libptr->path[0] != '\0' ? fixpath(libptr->path) : libptr->libname);
}

int
haslibkey(                         /* user has certain access to Library   */
struct fllib *libptr,                   /* ... Library concerned           */
char *keyname)                          /* ... key name to check for       */
{
     return(libptr == NULL ? 0 : (isflop(libptr) || haskey(keyname)));
}

int
usrupd(                            /* user has access to modify u/l'd file */
struct fllib *libptr,
char *filname)
{
     struct taglib *tag;
     int retval=0,loop;

     setbtv(flfdat);
     if (acqbtv(NULL,compkey(libptr->libname,filname),COMPLF)) {
          if (isflop(libptr) || sameas(usaptr->userid,flf->ulby)) {
               retval=1;
               for (loop=0 ; loop < systags ; loop++) {
                    tag=tagoff(loop);
                    if (tag->status > TGEMPTY && tag->usrnum != usrnum
                       && sameas(tag->filname,filname)
                       && sameas(tag->libname,libptr->libname)) {
                         retval=2;
                    }
               }
          }
     }
     else {
          retval=-1;
     }
     rstbtv();
     return(retval);
}

void
chkjoin(void)                      /* check all Libraries' joined status   */
{
     struct fllib *libptr;
     int loop,loop2;
     char *lib;

     for (loop=0 ; loop < numoflib ; loop++) {
          libptr=liboff(loop);
          if (libptr != NULL) {
               for (loop2=0 ; loop2 < FLNMCATS ; loop2++) {
                    lib=libptr->cat[loop2];
                    if (lib[0] != '\0' && libfind(lib) == NULL) {
                         lib[0]='\0';
                         movmem(libptr->cat[loop2+1],lib,
                            (FLNMCATS-loop2-1)*FLNAMESZ);
                         libptr->cat[FLNMCATS-1][0]='\0';
                         libptr->flags|=LIBCHN;
                    }
               }
          }
     }
}

int
totnlibs(                          /* return # of libraries joined+1, or   */
char *libname,                     /* the index of the library number      */
int index)                              /* == -1 if total # is wanted      */
{
     int loop,libs=0;
     struct fllib *libptr;

     for (loop=0 ; loop < numoflib ; loop++) {
          libptr=liboff(loop);
          if (strlen(visxlib(libptr,libname)) != 0) {
               if (index == libs++) {
                    return(loop);
               }
          }
     }
     return(libs);
}

int
initsrc(                           /* init. join or multi. lib search      */
int retstt)                             /* return substt                   */
{                                       /* returns 1 if speed search is it */
     int tl;
     struct fllib *libptr=NULL;
     char *libname="*";

     srcu->nlibs=0;
     switch (flo->keymeth) {
     case 3:
          libname=flo->u.key3.libname;
          break;
     case 4:
     case 7:
     case 8:
          libname=flo->u.key4.libname;
          break;
     }
     if (libname[0] != '*') {
          if (libname[0] != '\0') {
               libptr=libfind(libname);
          }
          if ((libptr != NULL || libname[0] == '\0')
             && (tl=totnlibs(libname,-1)) <= maxjoin && tl > 1) {
               flo->retstt=retstt;
               prfmsg(usrptr->substt=CYCIDX);
               btuinj(usrnum,CYCLE);
               srcu->nlibs=tl;
               srcu->nloaded=0;
               flo->flags&=~REVSRCH;
               return(1);
          }
     }
     return(0);
}

int
ccycidx(void)                      /* cycle index building                 */
{
     union srch *next,*prev;
     struct fllib *libptr=NULL;

     next=&srcu->tn[srcu->nloaded];
     setmem(next,sizeof(union srch),0);
     prev=&srcu->tp[srcu->nloaded];
     setmem(prev,sizeof(union srch),0);
     setbtv(flfdat);
     switch (flo->keymeth) {
     case 3:
          libptr=liboff(totnlibs(flo->u.key3.libname,srcu->nloaded));
          if (libptr != NULL) {
               stzcpy(next->key0.libname,libptr->libname,FLNAMESZ);
               stzcpy(next->key0.filname,flo->u.key3.filname,FLFILENM);
               stzcpy(next->key0.udate,flo->u.key3.udate,DATESZ);
               stzcpy(prev->key0.libname,libptr->libname,FLNAMESZ);
               stzcpy(prev->key0.filname,flo->u.key3.filname,FLFILENM);
               stzcpy(prev->key0.udate,flo->u.key3.udate,DATESZ);
               if (agebtv(NULL,&next->key0,COMPLUF)
                  && sameas(flf->libname,next->key0.libname)) {
                    stzcpy(next->key0.filname,flf->filname,FLFILENM);
                    stzcpy(next->key0.udate,flf->udate,DATESZ);
               }
               else {
                    next->key0.udate[0]=0xFF;
               }
               if (prev->key0.udate[0] != 0x00
                  && altbtv(NULL,&prev->key0,COMPLUF)
                  && sameas(flf->libname,prev->key0.libname)) {
                    stzcpy(prev->key0.filname,flf->filname,FLFILENM);
                    stzcpy(prev->key0.udate,flf->udate,DATESZ);
               }
               else {
                    prev->key0.udate[0]=0x00;
               }
               prev=&srcu->fp[srcu->nloaded];
               movmem(next,prev,sizeof(union srch));
          }
          break;
     case 4:
     case 7:
     case 8:
          libptr=liboff(totnlibs(flo->u.key4.libname,srcu->nloaded));
          if (libptr != NULL) {
               stzcpy(next->key1.libname,libptr->libname,FLNAMESZ);
               stzcpy(next->key1.filname,flo->u.key4.filname,FLFILENM);
               stzcpy(prev->key1.libname,libptr->libname,FLNAMESZ);
               stzcpy(prev->key1.filname,flo->u.key4.filname,FLFILENM);
               if (agebtv(NULL,&next->key1,COMPLF)
                  && sameas(flf->libname,next->key1.libname)) {
                    stzcpy(next->key1.filname,flf->filname,FLFILENM);
               }
               else {
                    next->key1.filname[0]=0xFF;
               }
               if (prev->key1.filname[0] != 0x00
                  && altbtv(NULL,&prev->key1,COMPLF)
                  && sameas(flf->libname,prev->key1.libname)) {
                    stzcpy(prev->key1.filname,flf->filname,FLFILENM);
               }
               else {
                    prev->key1.filname[0]=0x00;
               }
               prev=&srcu->fp[srcu->nloaded];
               movmem(next,prev,sizeof(union srch));
          }
          break;
     }
     if (libptr == NULL) {
          srcu->nlibs=srcu->nloaded;
          return(0);
     }
     if (++srcu->nloaded == srcu->nlibs) {
          return(0);
     }
     return(1);
}

void
newpage(void)                      /* ready the keysets for a new page     */
{
     if (!(flo->flags&REVSRCH)) {
          movmem(srcu->fn,srcu->tp,srcu->nlibs*sizeof(union srch));
          movmem(srcu->tn,srcu->fp,srcu->nlibs*sizeof(union srch));
     }
     else {
          movmem(srcu->tp,srcu->fn,srcu->nlibs*sizeof(union srch));
          movmem(srcu->fp,srcu->tn,srcu->nlibs*sizeof(union srch));
     }
}
