{*      S3M.PAS
 *
 * Scream Tracker 3 Module Player, v1.10
 *
 * Copyright 1994 Petteri Kangaslampi and Jarno Paananen
 *
 * This file is part of the MIDAS Sound System, and may only be
 * used, modified and distributed under the terms of the MIDAS
 * Sound System license, LICENSE.TXT. By continuing to use,
 * modify or distribute this file you indicate that you have
 * read the license and understand and accept it fully.
*}

unit S3M;


interface



{****************************************************************************\
*       struct s3mHeader
*       ----------------
* Description:  S3M module file header
\****************************************************************************}

type
    s3mHeader = Record
        name : array[0..27] of char;    { song name, ASCIIZ }
        num1A : byte;                   { $1A }
        ftype : byte;                   { file type }
        unused1 : word;
        songLength : word;              { number of orders }
        numInsts : word;                { number of instruments }
        numPatts : word;                { number of patterns }
        flags : word;                   { S3M file flags }
        trackerVer : word;              { tracker version }
        formatVer : word;               { file format version }
        SCRM : array[0..3] of char;     { "SCRM" }
        masterVol : byte;               { master volume }
        speed : byte;                   { initial speed }
        tempo : byte;                   { initial tempo }
        masterMult : byte;              { master multiplier (bits 0-3),
                                          stereo (bit 4) }
        unused2 : array[0..11] of byte;
        chanSettings : array[0..31] of byte;    { channel settings }
    end;



{****************************************************************************\
*       struct s3mInstHdr
*       -----------------
* Description:  S3M instrument header
\****************************************************************************}

type
    s3mInstHdr = Record
        itype : byte;                   { instrument type }
        dosName : array[0..11] of char; { DOS filename (8+3), ASCIIZ }
        zero : byte;                    { 0 }
        samplePtr : word;               { paragraph ptr to sample data }
        length : longint;               { sample length }
        loopStart : longint;            { sample loop start }
        loopEnd : longint;              { sample loop end }
        volume : byte;                  { volume }
        disk : byte;                    { instrument disk number }
        pack : byte;                    { sample packing info (0 = raw,
                                          1 = DP30ADPCM1) }
        flags : byte;                   { bit0 = loop, bit1 = stereo,
                                          bit2 = 16-bit }
        c2Rate : longint;               { C2 sampling rate }
        unused : longint;
        gusPos : word;                  { position in GUS memory / 32 }
        int512 : word;
        intLastUsed : longint;
        iname : array[0..27] of char;   { instrument name, ASCIIZ }
        SCRS : array[0..3] of char;     { "SCRS" }
    end;




{****************************************************************************\
*       struct s3mChannel
*       -----------------
* Description:  S3M player internal channel structure
\****************************************************************************}

type
    s3mChannel = Record
        note : byte;
        int : byte;
        vol : byte;
        cmd : byte;
        info : byte;
        flags : byte;

        sample : byte;
        volume : byte;

        period : word;
        snote : byte;
        preinfo : byte;
        toperi : word;
        notepsp : byte;
        retrigc : byte;

        status : byte;

        vibcmd : byte;
        vibpos : byte;

        volbar : byte;

        trefl : byte;
        trecnt : byte;
    end;




type
    Ppointer = ^pointer;
    Pword = ^word;
    Pinteger = ^integer;



{****************************************************************************\
*
* Function:     s3mLoadModule(fileName : string; SD : pointer;
*                   module : Ppointer) : integer;
*
* Description:  Loads a Scream Tracker 3 module into memory
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to the Sound Device which will
*                                       store the samples
*               module : Ppointer       pointer to variable which will store
*                                       the module pointer.
*
* Returns:      MIDAS error code.
*               Pointer to module structure is stored in module^.
*
\****************************************************************************}

function s3mLoadModule(fileName : string; SD : pointer; module : Ppointer) :
    integer;



{****************************************************************************\
*
* Function:     s3mFreeModule(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates a Scream Tracker 3 module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      MIDAS error code
*
\****************************************************************************}

function s3mFreeModule(module : pointer; SD : pointer) : integer;




{****************************************************************************\
*
* Function:     s3mDetectChannels(module : pointer; numChns : Pword) :
*                   integer;
*
* Description:  Detects the number of channels in a Scream Tracker 3 module
*
* Input:        module : pointer        pointer to module structure
*               numChns : Pword         pointer to channel number variable
*
* Returns:      MIDAS error code.
*               Number of channels in module stored in numChns^.
*
\****************************************************************************}

function s3mDetectChannels(module : pointer; numChns : Pword) : integer;




{***************************************************************************\
*
* Function:     s3mFindUsedInsts(module : pointer; used : pointer) : integer;
*
* Description:  Finds which instruments are used in a Scream Tracker 3 module.
*
* Input:        module : pointer        Pointer to module structure. At least
*                                       the patterns must have been loaded
*               used : pointer          Pointer to instruments used array -
*                                       one byte per instrument. Set to 1 if
*                                       instrument is used, 0 if not.
*
* Returns:      MIDAS error code.
*
\***************************************************************************}

function s3mFindUsedInsts(module : pointer; used : pointer) : integer;


function s3mIdentify(header : pointer; recognized : Pinteger) : integer;
function s3mInit(SD : pointer) : integer;
function s3mClose : integer;
function s3mPlayModule(module : pointer; firstSDChannel, numSDChannels,
    loopStart, loopEnd : word) : integer;
function s3mStopModule : integer;
function s3mSetInterrupt : integer;
function s3mRemoveInterrupt : integer;
function s3mPlay : integer;
function s3mSetPosition(npos : word) : integer;
function s3mGetInformation(info : pointer) : integer;



procedure mpS3M;                        { S3M Module Player structure. }



implementation


USES  Errors, mGlobals, mMem, EMS, MPlayer, SDevice, Timer
{$IFDEF REALVUMETERS}
    ,VU
{$ENDIF}
    ;


function s3mDetectChannels(module : pointer; numChns : Pword) : integer;
    external;
function s3mFindUsedInsts(module : pointer; used : pointer) : integer;
    external;
function s3mIdentify(header : pointer; recognized : Pinteger) : integer;
    external;
function s3mInit(SD : pointer) : integer; external;
function s3mClose : integer; external;
function s3mPlayModule(module : pointer; firstSDChannel, numSDChannels,
    loopStart, loopEnd : word) : integer; external;
function s3mStopModule : integer; external;
function s3mSetInterrupt : integer; external;
function s3mRemoveInterrupt : integer; external;
function s3mPlay : integer; external;
function s3mSetPosition(npos : word) : integer; external;
function s3mGetInformation(info : pointer) : integer; external;
procedure mpS3M; external;
{$L S3M.OBJ}


{****************************************************************************\
*       enum s3mFunctIDs
*       ----------------
* Description:  ID numbers for Scream Tracker 3 Module Player functions
\****************************************************************************}

const
    ID_s3mIdentify = ID_s3m;
    ID_s3mInit = ID_s3m + 1;
    ID_s3mClose = ID_s3m + 2;
    ID_s3mLoadModule = ID_s3m + 3;
    ID_s3mFreeModule = ID_s3m + 4;
    ID_s3mPlayModule = ID_s3m + 5;
    ID_s3mStopModule = ID_s3m + 6;
    ID_s3mSetInterrupt = ID_s3m + 7;
    ID_s3mRemoveInterrupt = ID_s3m + 8;
    ID_s3mPlay = ID_s3m + 9;
    ID_s3mSetPosition = ID_s3m + 10;
    ID_s3mGetInformation = ID_s3m + 11;
    ID_s3mDetectChannels = ID_s3m + 12;
    ID_s3mFindUsedInsts = ID_s3m + 13;



{$I-}

{ Size of temporary memory area used for avoiding memory fragmentation
  if EMS is used }
const
    TEMPSIZE = 8192;

type
    wordArray = array[0..8192] of word;



{***************************************************************************\
*       Module loader buffers and file pointer. These variables are static
*       instead of local so that a separate deallocation can be used which
*       will be called before exiting in error situations
\***************************************************************************}

var
    f : file;
    fileOpened : boolean;
    ms3m : PmpModule;
    instPtrs : ^wordArray;
    pattPtrs : ^wordArray;
    smpBuf : pointer;
    tempMem : pointer;




{****************************************************************************\
*
* Function:     s3mFreeModule(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates a Scream Tracker 3 module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      MIDAS error code
*
\****************************************************************************}

function s3mFreeModule(module : pointer; SD : pointer) : integer;
var
    i, error : integer;
    sdev : ^SoundDevice;
begin
    ms3m := module;
    sdev := SD;

    if ms3m = NIL then                 { valid module? }
    begin
        mError(errUndefined, ID_s3mFreeModule);
        s3mFreeModule := errUndefined;
        exit;
    end;

    { deallocate pattern orders if allocated: }
    if ms3m^.orders <> NIL then
    begin
        error := memFree(ms3m^.orders);
        if error <> OK then
        begin
            mError(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    { deallocate sample used flags: }
    if ms3m^.instsUsed <> NIL then
    begin
        error := memFree(ms3m^.instsUsed);
        if error <> OK then
        begin
            mError(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;


    if ms3m^.insts <> NIL then       { instruments? }
    begin
        for i := 0 to (ms3m^.numInsts-1) do
        begin
            { If the instrument has been added to Sound Device, remove
               it, otherwise just deallocate the sample if allocated }
            if ms3m^.insts^[i].sdInstHandle <> 0 then
            begin
                error := sdev^.RemInstrument(ms3m^.insts^[i].sdInstHandle);
                if error <> OK then
                begin
                    mError(error, ID_s3mFreeModule);
                    s3mFreeModule := error;
                    exit;
                end;
            end
            else
            begin
                if ms3m^.insts^[i].sample <> NIL then
                begin
                    error := memFree(ms3m^.insts^[i].sample);
                    if error <> OK then
                    begin
                        mError(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;

{$IFDEF REALVUMETERS}
            { remove VU meter information if used: }
            if realVU = 1 then
            begin
                if ms3m^.insts^[i].sdInstHandle <> 0 then
                begin
                    error := vuRemove(ms3m^.insts^[i].sdInstHandle);
                    if error <> OK then
                    begin
                        mError(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;
{$ENDIF}
        end;

        { deallocate instrument structures: }
        error := memFree(ms3m^.insts);
        if error <> OK then
        begin
            mError(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    if (ms3m^.patterns <> NIL) and (ms3m^.pattEMS <> NIL) then
    begin
        for i := 0 to (ms3m^.numPatts-1) do
        begin
            { if the pattern has been allocated, deallocate it - either
                from conventional memory or from EMS }
            if ms3m^.patterns^[i] <> NIL then
            begin
                if ms3m^.pattEMS^[i] = 1 then
                begin
                    error := emsFree(ms3m^.patterns^[i]);
                    if error <> OK then
                    begin
                        mError(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end
                else
                begin
                    error := memFree(ms3m^.patterns^[i]);
                    if error <> OK then
                    begin
                        mError(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;
        end;

        { deallocate pattern pointers: }
        error := memFree(ms3m^.patterns);
        if error <> OK then
        begin
            mError(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;

        { deallocate pattern EMS flags: }
        error := memFree(ms3m^.pattEMS);
        if error <> OK then
        begin
            mError(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    { deallocate the module: }
    error := memFree(ms3m);
    if error <> OK then
    begin
        mError(error, ID_s3mFreeModule);
        s3mFreeModule := error;
        exit;
    end;

    s3mFreeModule := OK;
end;




{***************************************************************************\
*
* Function:     procedure s3mLoadError(SD : PSoundDevice);
*
* Description:  Stops loading the module, deallocates all buffers and closes
*               the file.
*
* Input:        SD : PSoundDevice       Sound Device that has been used for
*                                       loading.
*
\***************************************************************************}

procedure s3mLoadError(SD : PSoundDevice);
begin
    if fileOpened then                  { close file if opened }
        Close(f);

    { Attempt to deallocate module if allocated. Do not process errors. }
    if ms3m <> NIL then
        if s3mFreeModule(ms3m, SD) <> OK then
            exit;

    { Deallocate buffers if allocated. Do not process errors. }
    if smpBuf <> NIL then
        if memFree(smpBuf) <> OK then
            exit;
    if tempmem <> NIL then
        if memFree(tempmem) <> OK then
            exit;
    if instPtrs <> NIL then
        if memFree(instPtrs) <> OK then
            exit;
    if pattPtrs <> NIL then
        if memFree(pattPtrs) <> OK then
            exit;
end;




{****************************************************************************\
*
* Function:     s3mLoadModule(fileName : string; SD : pointer;
*                   module : Ppointer) : integer;
*
* Description:  Loads a Scream Tracker 3 module into memory
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to the Sound Device which will
*                                       store the samples
*               module : Ppointer       pointer to variable which will store
*                                       the module pointer.
*
* Returns:      MIDAS error code.
*               Pointer to module structure is stored in module^.
*
\****************************************************************************}

function s3mLoadModule(fileName : string; SD : pointer; module : Ppointer) :
    integer;

    { memcpy - copies bytes from src to dest }
    procedure memcpy(var dest; var src; bytes : word); assembler;
    asm
            push    ds
            les     di,dest
            lds     si,src
            mov     cx,bytes
            cld
            rep     movsb
            pop     ds
    end;

    { cmpstr - compares memory area m against string s. Returns 0 if
      identical. }
    function cmpstr(var m1; s : string) : integer;
    var
        b : ^byteArray;
        pos : byte;
    begin
        b := @m1;
        cmpstr := 0;
        for pos := 1 to ord(s[0]) do
            if b^[pos-1] <> ord(s[pos]) then
                cmpStr := 1;
    end;


var
    s3mh : s3mHeader;
    s3mi : s3mInstHdr;
    i : integer;
    inst : PmpInstrument;
    pattSize : word;
    pattData : PmpPattern;
    lend : word;
    maxSmpLength : longint;
    error : integer;
    ordersize : word;
    p : pointer;
    numRead : word;
    sdev : PSoundDevice;

begin
    sdev := SD;

    { point file ptr and buffers to NIL so that s3mLoadError() can be
       called at any point }
    fileOpened := False;
    ms3m := NIL;
    instPtrs := NIL;
    pattPtrs := NIL;
    smpBuf := NIL;
    tempmem := NIL;


    { Open module file: }
    Assign(f, fileName);
    Reset(f, 1);
    if IOResult <> 0 then
    begin
        s3mLoadError(SD);
        mError(errFileOpen, ID_s3mLoadModule);
        s3mLoadModule := errFileOpen;
        exit;
    end;
    fileOpened := True;

    { Allocate memory for the module structure: }
    error := memAlloc(SizeOf(mpModule), @ms3m);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    ms3m^.orders := NIL;                 { clear module structure so that }
    ms3m^.insts := NIL;                  { it can be deallocated with }
    ms3m^.patterns := NIL;               { s3mFree() at any point }
    ms3m^.pattEMS := NIL;
    ms3m^.instsUsed := NIL;

    { Read .S3M file header: }
    BlockRead(f, s3mh, SizeOf(s3mHeader), numRead);
    if numRead <> SizeOf(s3mHeader) then
    begin
        s3mLoadError(SD);
        mError(errFileRead, ID_s3mLoadModule);
        s3mLoadModule := errFileRead;
        exit;
    end;

    { Check the "SCRM" signature in header: }
    if cmpstr(s3mh.SCRM, 'SCRM') <> 0 then
    begin
        s3mLoadError(SD);
        mError(errInvalidModule, ID_s3mLoadModule);
        s3mLoadModule := errInvalidModule;
        exit;
    end;

    memcpy(ms3m^.ID, s3mh.SCRM, 4);     { copy ID }
    ms3m^.IDnum := idS3M;               { S3M module ID }

    memcpy(ms3m^.songName, s3mh.name, 28);  { copy song name }
    ms3m^.songLength := s3mh.songLength;  { copy song length }
    ms3m^.numInsts := s3mh.numInsts;    { copy number of instruments }
    ms3m^.numPatts := s3mh.numPatts;    { copy number of patterns }
    ms3m^.flags := s3mh.flags;          { copy S3M flags }
    ms3m^.masterVol := s3mh.masterVol;  { copy master volume }
    ms3m^.speed := s3mh.speed;          { copy initial speed }
    ms3m^.tempo := s3mh.tempo;          { copy initial BPM tempo }
    ms3m^.masterMult := s3mh.masterMult and 15;     { copy master multiplier }
    ms3m^.stereo := (s3mh.masterMult shr 4) and 1;  { copy stereo flag }
    { copy channel settings: }
    memcpy(ms3m^.chanSettings, s3mh.chanSettings, 32);

    { Allocate memory for pattern orders: (length of pattern orders must be
       even) }
    ordersize := 2 * ((ms3m^.songLength+1) div 2);
    error := memAlloc(orderSize, @ms3m^.orders);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read pattern orders from file: }
    BlockRead(f, ms3m^.orders^, ordersize, numRead);
    if numRead <> ordersize then
    begin
        s3mLoadError(SD);
        mError(errFileRead, ID_s3mLoadModule);
        s3mLoadModule := errFileRead;
        exit;
    end;

    { Calculate real song length: (exclude 0xFF bytes from end) }
    i := ms3m^.songLength - 1;
    while ms3m^.orders^[i] = $FF do
        i := i - 1;
    ms3m^.songLength := i + 1;

    { Allocate memory for instrument structures: }
    error := memAlloc(ms3m^.numInsts * sizeof(mpInstrument), @ms3m^.insts);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Clear all instruments: }
    for i := 0 to (ms3m^.numInsts-1) do
    begin
        ms3m^.insts^[i].sample := NIL;
        ms3m^.insts^[i].sdInstHandle := 0;
    end;

    { Allocate memory for instrument paragraph pointers: }
    error := memAlloc(2 * ms3m^.numInsts, @instPtrs);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read instrument pointers: }
    BlockRead(f, instPtrs^, 2 * ms3m^.numInsts, numRead);
    if numRead <> (2 * ms3m^.numInsts) then
    begin
        s3mLoadError(SD);
        mError(errFileRead, ID_s3mLoadModule);
        s3mLoadModule := errFileRead;
        exit;
    end;

    { Allocate memory for S3M file pattern pointers: }
    error := memAlloc(2 * ms3m^.numPatts, @pattPtrs);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read pattern pointers: }
    BlockRead(f, pattPtrs^, 2 * ms3m^.numPatts, numRead);
    if numRead <> (2 * ms3m^.numPatts) then
    begin
        s3mLoadError(SD);
        mError(errFileRead, ID_s3mLoadModule);
        s3mLoadModule := errFileRead;
        exit;
    end;

    { Allocate memory for pattern pointers: }
    error := memAlloc(4 * ms3m^.numPatts, @ms3m^.patterns);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Allocate memory for pattern EMS flags: }
    error := memAlloc(ms3m^.numPatts, @ms3m^.pattEMS);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Point all unallocated patterns to NIL for safety: }
    for i := 0 to (ms3m^.numPatts-1) do
        ms3m^.patterns^[i] := NIL;

    { Read all patterns to memory: }
    for i := 0 to (ms3m^.numPatts-1) do
    begin
        { Seek to pattern beginning in file: }
        Seek(f, longint(16) * longint(pattPtrs^[i]));
        if IOResult <> OK then
        begin
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            s3mLoadModule := errFileRead;
            exit;
        end;

        { Read pattern length from file: }
        BlockRead(f, pattSize, 2, numRead);
        if numRead <> 2 then
        begin
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            s3mLoadModule := errFileRead;
            exit;
        end;

        if useEMS = 1 then
        begin
            { Try to allocate EMS memory for pattern: }
            error := emsAlloc(pattSize+2, @p);
            if error <> OK then
            begin
                { failed - if only EMS memory should be used, or the
                    error is other than out of EMS memory, pass the error
                    on }
                if (forceEMS = 1) or (error <> errOutOfEMS) then
                begin
                    s3mLoadError(SD);
                    mError(error, ID_s3mLoadModule);
                    s3mLoadModule := error;
                    exit;
                end
                else
                begin
                    { pattern not in EMS: }
                    ms3m^.pattEMS^[i] := 0;

                    { try to allocate conventional memory instead: }
                    error := memAlloc(pattSize+2, @p);
                    if error <> OK then
                    begin
                        s3mLoadError(SD);
                        mError(error, ID_s3mLoadModule);
                        s3mLoadModule := error;
                        exit;
                    end;
                end;
            end
            else
            begin
                { Pattern is in EMS - map pattern EMS block to conventional
                   memory and point pattData to it }
                ms3m^.pattEMS^[i] := 1;

                { map EMS block to conventional memory and point pattData
                    to the memory area: }
                error := emsMap(p, @pattData);
                if error <> OK then
                begin
                    s3mLoadError(SD);
                    mError(error, ID_s3mLoadModule);
                    s3mLoadModule := error;
                    exit;
                end;
            end;
        end
        else
        begin
            { No EMS memory used - allocate conventional memory for
               pattern: }
            ms3m^.pattEMS^[i] := 0;

            error := memAlloc(pattSize+2, @p);
            if error <> OK then
            begin
                s3mLoadError(SD);
                mError(error, ID_s3mLoadModule);
                s3mLoadModule := error;
                exit;
            end;

            pattData := p;
        end;

        ms3m^.patterns^[i] := p;

        pattData^.length := pattSize;   { save pattern length }

        { Read pattern data from file: }
        BlockRead(f, pattData^.data, pattSize, numRead);
        if numRead <> pattSize then
        begin
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            s3mLoadModule := errFileRead;
            exit;
        end;
    end;

    { deallocate pattern file pointers: }
    error := memFree(pattPtrs);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    pattPtrs := NIL;

    { detect number of channels: }
    error := s3mDetectChannels(ms3m, @ms3m^.numChans);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { allocate memory for instrument used flags: }
    error := memAlloc(ms3m^.numInsts, @ms3m^.instsUsed);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { find which instruments are used: }
    error := s3mFindUsedInsts(ms3m, ms3m^.instsUsed);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Find maximum sample length: }
    maxSmpLength := 0;
    for i := 0 to (ms3m^.numInsts-1) do
    begin
        { Seek to instrument header in file: }
        Seek(f, longint(instPtrs^[i]) * longint(16));
        if IOResult <> 0 then
        begin
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            s3mLoadModule := errFileRead;
            exit;
        end;

        { Read instrument header from file: }
        BlockRead(f, s3mi, SizeOf(s3mInstHdr), numRead);
        if numRead <> SizeOf(s3mInstHdr) then
        begin
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            s3mLoadModule := errFileRead;
            exit;
        end;

        if maxSmpLength < s3mi.length then
            maxSmpLength := s3mi.length;
    end;

    { Check that no instrument is too long: }
    if maxSmpLength > SMPMAX then
    begin
        mError(errInvalidInst, ID_s3mLoadModule);
        s3mLoadError(SD);
        s3mLoadModule := errInvalidInst;
        exit;
    end;

    { If EMS is used, allocate TEMPSIZE bytes of memory before the sample
       buffer and deallocate it after allocating all temporary loading
       buffers to minimize memory fragmentation }
    if useEMS = 1 then
    begin
        error := memAlloc(TEMPSIZE, @tempmem);
        if error <> OK then
        begin
            s3mLoadError(SD);
            mError(error, ID_s3mLoadModule);
            s3mLoadModule := error;
            exit;
        end;
    end;

    { allocate memory for sample loading buffer: }
    error := memAlloc(maxSmpLength, @smpBuf);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    if useEMS = 1 then
    begin
        error := memFree(tempmem);
        if error <> OK then
        begin
            s3mLoadError(SD);
            mError(error, ID_s3mLoadModule);
            s3mLoadModule := error;
            exit;
        end;
        tempmem := NIL;
    end;


    for i := 0 to (ms3m^.numInsts-1) do
    begin

        { point inst to current instrument structure }
        inst := @ms3m^.insts^[i];

        { Seek to instrument header in file: }
        Seek(f, longint(instPtrs^[i]) * longint(16));
        if IOResult <> 0 then
        begin
            s3mLoadError(SD);
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadModule := errFileRead;
            exit;
        end;

        { Read instrument header from file: }
        BlockRead(f, s3mi, SizeOf(s3mInstHdr), numRead);
        if numRead <> SizeOf(s3mInstHdr) then
        begin
            s3mLoadError(SD);
            mError(errFileRead, ID_s3mLoadModule);
            s3mLoadModule := errFileRead;
            exit;
        end;

        { Check if the instrument is valid - not too long, not stereo,
           16-bit or packed }
        if (s3mi.length > SMPMAX) or ((s3mi.flags and 6) <> 0) or
            (s3mi.pack <> 0) then
        begin
            s3mLoadError(SD);
            mError(errInvalidInst, ID_s3mLoadModule);
            s3mLoadModule := errFileRead;
            exit;
        end;

        memcpy(inst^.fileName, s3mi.dosName, 13);   { copy filename }
        memcpy(inst^.iname, s3mi.iname, 28);        { copy inst name }
        inst^.length := s3mi.length;        { copy sample length }
        inst^.loopStart := s3mi.loopStart;  { copy sample loop start }
        inst^.loopEnd := s3mi.loopEnd;      { copy sample loop end }
        inst^.looping := s3mi.flags and 1;  { copy looping status }
        inst^.volume := s3mi.volume;        { copy default volume }
        inst^.c2Rate := s3mi.c2Rate;        { copy C2 playing rate }

        { Make sure that instrument volume is < 63 }
        if inst^.volume > 63 then
            inst^.volume := 63;

        { Check if instrument is used: }
        if ms3m^.instsUsed^[i] = 1 then
        begin
            { Instrument is used - check if there is a sample for this
              instrument - type = 1, signature "SCRS" and length != 0 }
            if (s3mi.itype = 1) and (cmpstr(s3mi.SCRS, 'SCRS') = 0) and
                (inst^.length <> 0) then
            begin
                { Seek to sample position in file: }
                Seek(f, longint(s3mi.samplePtr) * longint(16));
                if IOResult <> 0 then
                begin
                    s3mLoadError(SD);
                    mError(errFileRead, ID_s3mLoadModule);
                    s3mLoadModule := errFileRead;
                    exit;
                end;

                { Read sample to loading buffer: }
                BlockRead(f, smpBuf^, inst^.length, numRead);
                if numRead <> inst^.length then
                begin
                    s3mLoadError(SD);
                    mError(errFileRead, ID_s3mLoadModule);
                    s3mLoadModule := errFileRead;
                    exit;
                end;
            end;

            { Point inst^.sample to NIL, as the instrument is not available
            - only the Sound Device has it }
            inst^.sample := NIL;

            { Add instrument to Sound Device: }
            error := sdev^.AddInstrument(smpBuf, smp8bit, inst^.length,
                inst^.loopStart, inst^.loopEnd, inst^.volume, inst^.looping,
                @inst^.sdInstHandle);
            if error <> OK then
            begin
                s3mLoadError(SD);
                mError(error, ID_s3mLoadModule);
                s3mLoadModule := error;
                exit;
            end;

{$IFDEF REALVUMETERS}
            { if real VU meters are used, prepare VU meter information
              for this instrument: }
            if realVU = 1 then
            begin
                if inst^.looping = 1 then
                    lend := inst^.loopEnd
                else
                    lend := 0;          { no looping - set VU loop end to
                                          zero }

                error := vuPrepare(inst^.sdInstHandle, smpBuf, inst^.length,
                    inst^.loopStart, lend);
                if error <> OK then
                begin
                    s3mLoadError(SD);
                    mError(error, ID_s3mLoadModule);
                    s3mLoadModule := error;
                    exit;
                end;
            end;
{$ENDIF}
        end;
    end;

    { deallocate instrument pointers: }
    error := memFree(instPtrs);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    instPtrs := NIL;

    { deallocate sample loading buffer: }
    error := memFree(smpBuf);
    if error <> OK then
    begin
        s3mLoadError(SD);
        mError(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    smpBuf := NIL;

    Close(f);
    fileOpened := False;

    module^ := ms3m;                    { return module pointer in module^ }

    s3mLoadModule := OK;
end;



{$I+}

END.
