/* filename: CLOCK.C

: T O P A Z for C :Ŀ
                          Version 4.5  05/16/93                              
                                                                             
 Copyright (c) 1988,1994 Software Science Inc. All Rights Reserved Worldwide.
 Unauthorized distribution or disclosure of this source code or modification 
  or removal of this notice  constitutes a breach of the license agreement.  

*/
#include <dos.h>
#include <string.h>
#include <stdlib.h>
#include <clock.h>

// this tells us everything we need to know about each clock
typedef struct ClockRecord {
  int   X, Y, Attr;
  int   Enabled, Military;
  long  StartingTicks;
  int   Format;
} ClockRecord;

typedef unsigned char uchar;
typedef unsigned int  uint;

typedef void (interrupt *ISR)(void);

#define MAXCLOCKS  8
#define HHMMSS     0 // predefined constants for format
#define HHMM       1
#define MMSS       2
#define Trigger    6 // number of ticks between displays, so refresh is
                     //  every 1/2 sec - 1193180 (cycles per second) / 65536 (cycles per tick)
#define TicksPerDay    1573040L // roughly
#define TicksPerSecond 18.2

static ClockRecord _Clock[MAXCLOCKS] = {
  { 70, 1, 112,  TRUE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS },
  { 70, 1, 112, FALSE, TRUE, 0, HHMMSS }
};

static int   ClockNumber = 0;   // user gets ClockNumber + 1
static int   ClocksOn    = FALSE;
static int   Ticks       = 10;  // force clock to appear right away
static uchar ClockISRHandle;
static int   TimerInUse;        // avoids recursion in Clock Interrupt
static long  SnapShot;
static ISR   SaveVector;

#ifdef _MSC_VER
 #pragma optimize("i", on)
 #pragma check_stack(off)
 #pragma check_pointer(off)
#else
 #pragma option -N- -k+
#endif

static char *Time(long Ticks, int ClockNo)
{
  ClockRecord *cr;
  long         TotalSeconds;
  int          hour, minute, second;
  char         temp[11], ampm[4];
  static char  result[16];

  cr = &_Clock[ClockNo];

  if (Ticks < cr->StartingTicks) // we passed midnight
    TotalSeconds = (10L * (Ticks + TicksPerDay - cr->StartingTicks)) / 182L;
  else
    TotalSeconds = (10L * (Ticks - cr->StartingTicks)) / 182L;

  hour = (int) (TotalSeconds / 3600L);
  TotalSeconds -= 3600L * hour;
  minute = (int) (TotalSeconds / 60L);
  second = (int) (TotalSeconds - 60L * minute);

  if(!cr->Military) {
    if(hour > 11) { // it is after noon
      if(hour > 12) // it is after 1 pm
        hour -= 12;
      strcpy(ampm, " pm");
    } else
      strcpy(ampm, " am");
  } else
    *ampm = 0;

  memset(temp, 0, sizeof(temp));

  if(hour < 10) {
    *temp = '0';
    itoa(hour, temp+1, 10);
  } else
    itoa(hour, temp, 10);

  strcat(temp, ":");

  if(minute < 10) {
    temp[3] = '0';
    itoa(minute, temp+4, 10);
  } else
    itoa(minute, temp+3, 10);

  strcat(temp, ":");

  if(second < 10) {
    temp[6] = '0';
    itoa(second, temp + 7, 10);
  } else
    itoa(second, temp + 6, 10);

  switch(cr->Format) {
    case HHMM:
      temp[5] = 0;
      if(second % 2)
        temp[2] = ' ';  // blink the colon
      break;

    case MMSS:
      strncpy(temp, temp + 3, 6);
      break;
  }

  strcpy(result, " ");
  strcat(result, temp);
  strcat(result, ampm);
  strcat(result, " ");

  return result;
}

#ifdef _MSC_VER
 #pragma optimize("i", on)
 #pragma check_stack(off)
 #pragma check_pointer(off)
#else
 #pragma option -N- -k+
#endif

static void ShowAllClocks(void)
{
  ClockRecord *cr;
  char        *ptr;
  int          ClockNo = 0;

  if(Ticks > Trigger) {
    SnapShot = *BiosTimerTicks; // from bios data area
    while(ClockNo < MAXCLOCKS) {
      cr = &_Clock[ClockNo];
      if(cr->Enabled) {
        ptr = Time(SnapShot, ClockNo);
        FastWrite(ptr, cr->Y, cr->X, cr->Attr);
      }
      ClockNo++;
    }
    Ticks = 1;
  }
}

#ifdef _MSC_VER
 #pragma optimize ("e", off)
#endif

static void interrupt PutClock()
{
  _enable();
  Ticks++;
  if (!TimerInUse) {
    TimerInUse = TRUE;
    ShowAllClocks();
    TimerInUse = FALSE;
  }
  SaveVector = (ISR) ChainISRAddress(ClockISRHandle);
  (*SaveVector)();
}

#ifdef _MSC_VER
 #pragma optimize("", on)
 #pragma check_stack(on)
 #pragma check_pointer(on)
#else
 #pragma option -N+ -k+
#endif

void SetClockTo(int x, int y)
{
  _Clock[ClockNumber].X = dBASEOrder ? (y + 1) : x;
  _Clock[ClockNumber].Y = dBASEOrder ? (x + 1) : y;
}

void SetClocksOn(void)
{
  if (!CheckRegisteredUnits("SetClocksOn",REGTZCOMMON+REGTZCLOCK))
    return;

  if(!ClocksOn) {
    TimerInUse     = FALSE;
    ClocksOn       = TRUE;
    ClockISRHandle = AddISRVector(PutClock);
  }
}

#ifndef _MSC_VER
 #pragma warn -eff
#endif

void SetClocksOff(void)
{
  if(ClocksOn) {
    while(TimerInUse) ;
    RemoveISRVector(ClockISRHandle);
    ClocksOn = FALSE;
  }
}

void SetClockColorTo(int foreground, int background)
{
  _Clock[ClockNumber].Attr = (foreground & 0x0F) | (background << 4);
}

void SetClockFormatTo(char *command)
{
  ClockRecord *cr;
  int          More;
  char        *p, temp[80];

  strcpy(temp, command);
  cr = &_Clock[ClockNumber];
  cr->StartingTicks = 0;
  cr->Enabled = *temp ? 1 : 0;
  strupr(temp);
  // look for known clauses
  strcat(temp, " ");

  do {
    More = FALSE;
    if((p = strstr(temp, "HH:MM ")) != NULL) {
      cr->Format = HHMM;
      strcpy(p, p + 6);
      More = TRUE;
    }
    if(((p = strstr(temp, "MM:SS ")) != NULL) && (strstr(temp,"HH:MM:SS ") == NULL)) {
      cr->Format = MMSS;
      strcpy(p, p + 6);
      More = TRUE;
    }
    if((p = strstr(temp, "HH:MM:SS ")) != NULL) {
      cr->Format = HHMMSS;
      strcpy(p, p + 9);
      More = TRUE;
    }
    if((p = strstr(temp, "ELAPSED TIME ")) != NULL) {
      cr->StartingTicks = (long) ((long) (*BiosTimerTicks / TicksPerSecond) * TicksPerSecond);
      strcpy(p, p + 13);
      More = TRUE;
    }
    if((p = strstr(temp, "TOD ")) != NULL) {
      strcpy(p, p + 4);
      More = TRUE;
    }
    if((p = strstr(temp, "MILITARY ")) != NULL) {
      cr->Military = TRUE;
      strcpy(p, p + 9);
      More = TRUE;
    }
    if((p = strstr(temp, "AMPM ")) != NULL) {
      cr->Military = FALSE;
      strcpy(p, p + 5);
      More = TRUE;
    }
    if((p = strstr(temp, "BLINK ")) != NULL) {
      cr->Format |= 0x80;
      strcpy(p, p + 6);
      More = TRUE;
    }
  } while(More);
}

void SelectClock(int x)
{
  int i;

  if(!x) { // find an unused clock
    for(i = 0; i < MAXCLOCKS; ++i)
      if (!_Clock[i].Enabled)
        break;
    if(!_Clock[i].Enabled)
      ClockNumber = i;
  } else
    if (x >= 1 && x <= MAXCLOCKS)
      ClockNumber = --x; // ignore invalid clock numbers
}

int CurrentClock(void)
{
  return ClockNumber + 1;
}

static void ClockExitProc(void)
{
  if(ClocksOn) {
    while(TimerInUse) ;
    RemoveISRVector(ClockISRHandle);
    ClocksOn = FALSE;
  }
}

void TZClockInit(void)
{
  atexit(ClockExitProc);
}
