PAGE 58,132
;==============================================================================
TITLE SIWVID.ASM - Soft-ICE/W Video driver support for the Diamond Viper
;
; This file contains all necessary code to support the Diamond Viper board
; for Soft-ICE/W.  When Soft-ICE/W first starts up it looks in the directory
; where WINICE.EXE is located for a file named SIWVID.386.  If it is found
; Soft-ICE/W automatically loads it and calls it to switch video modes
; between text and graphics modes.
;
; A stub routine is provided for each of the entry points that the video vxd
; might have to provide code for.  Other than initialization all the real
; work is done in two routines, one to save the screen and switch to text mode
; and one to restore the screen.  The following entry points are defined:
;
;  Real mode init   	This is the real mode initialization portion of the
;			VxD called by Windows before switching to protected
;			mode.  At this point video bios calls can be made.
;			The only restriction is that this portion can pass
;			only a single DWORD value to the protected mode
;			portion of the VxD.
;
;  CRITICAL_INIT	This is the first level of protected mode
;			initialization in the VxD.  At this point the video
;			vxd should finish up whatever initialization is 
;			required.
;
;  SIWVID_GetVersion   	This VxD service will be called by Soft-ICE/W to 
;			determine if a video VxD is loaded.  This service does
;			nothing more than return version 1.0 and does not need
;			to be modified.
;
;  SIWVID_Setup		This service is called by Soft-ICE/W immediately after
;			the GetVersion service.  Its only purpose is to pass
;			some pointers that the video VxD might require.  The
;			following values are passed:
;
;			 - A linear address that can be used to access any
;			   physical memory location.
;
;			 - A linear address of an 8x16 rom font that can be
;			   used for 25 line mode.	
;
;			 - A linear address of an 8x8 rom font that can be
;			   used for 44 and 50 line mode.	
;
;			 - A linear address of an 8000 byte save buffer.  
;			   This buffer can be used to save the contents of
;			   video memory and is passed so the VxD does not
;			   have to waste 8000 bytes defining its own save
;			   buffer.
;
;  SIWVID_SaveScreen	This routine is called when Soft-ICE/W is ready to
;			pop up.  This routine must do the following:
;
;			 - Save the current video state for later restoring
;			   (The current video state can be either graphics
;			   or text (DOS BOX)).
;
;			 - Save the 8K of video memory that Soft-ICE/W will
;			   use in text mode.
;
;			 - Switch the video board into the appropriate text
;			   mode (25, 44, or 50 line mode).
;
;  SIWVID_RestoreScreen	This routine is called when Soft-ICE/W is ready to
;			pop down.  This routine must restore the video to the
;			state saved by SaveScreen.
;
;  SIWVID_Debug         This routine is callable from a 16 bit windows or DOS
;			program.  It can be used to debug SIWVID.386 using
;			Soft-ICE/W on a serial terminal or a monochrome 
;			monitor.  A DOS/Windows program is included that calls
;			this entry point.  All the routine does is a call
;			to SaveScreen followed by a call to RestoreScreen.
;			
;
;  The first four entry points above are called once and only once and they
;  are called in the order that they are documented.  The next two routines
;  are called when soft-ice/w needs to pop up but they will not be called
;  prior to the initialization routines.  The SaveScreen and RestoreScreen
;  are called as a pair, that is 1 call to SaveScreen will always be followed
;  by 1 call to RestoreScreen.  The last entry point is used only for 
;  debugging purposes and does not need to be modified.	
;
;
;   (C) Copyright Nu-Mega Technologies Inc. 1994
;
;   Date:	12-Jan-1994
;==============================================================================

.386p

;============================================================================
;   Macros.
;============================================================================

delay	MACRO
	push	ecx
	mov	ecx,2
	loop	$
	pop	ecx
	ENDM

;============================================================================
;   Equates.
;============================================================================

USERSTATE   equ     0			    ;  indicates user state
SICESTATE   equ     1			    ;  indicates debug state
GOSAVE	    equ     0			    ;  indicates save action
GORESTORE   equ     1			    ;  indicates restore action
SCREENADDR  equ     0b8000h		    ;  screen base address
SCREENSIZE  equ     4000		    ;  save of screen area to save

;==============================================================================
;			      I N C L U D E S
;==============================================================================

.xlist

	include vmm.inc


;==============================================================================
;		   V X D   S E R V I C E S   D E C L A R A T I O N
;==============================================================================
SIWVID_Device_ID equ 7a5fh

Create_SIWVID_Service_Table EQU 1   ; SIWVID.386 service table created

Begin_Service_Table SIWVID
	SIWVID_Service SIWVID_GetVersion, LOCAL
	SIWVID_Service SIWVID_Setup, LOCAL
	SIWVID_Service SIWVID_SaveScreen, LOCAL
	SIWVID_Service SIWVID_RestoreScreen, LOCAL
End_Service_Table SIWVID

.list

;==============================================================================
;		 V I R T U A L	 D E V I C E   D E C L A R A T I O N
;==============================================================================

Declare_Virtual_Device SIWVID, 1, 0, SIWVID_Control, SIWVID_Device_ID,,SIWVID_Debug,SIWVID_Debug

VxD_LOCKED_DATA_SEG

db '(C) Copyright Nu-Mega Technologies Inc. 1994',0

;============================================================================
; Values passed from Soft-ICE/W during setup.
;============================================================================

; linear address that maps any physical address

MapPhysical dd	0

; linear address of fonts that soft-ice/w obtains from the rom before
; starting windows.  The 8x16 can be used for 25 line text mode while the
; 8x8 is used for 44 and 50 line text mode.

Font8x16    dd	0
Font8x8     dd	0

; address of 8000 byte save buffer.  This is passed so the VxD doesn't have
; to waste an additional 8000 bytes defining a save buffer.

SaveBufferAddress   dd	0

;============================================================================
; Screen state variables.
;============================================================================

ScreenBase	dd	SCREENADDR	; address of video memory
usercursor	dw	0		; storage for cursor x,y
sicecursor	dw	0		;
vipermode	db	0		; save for viper's sequencer 12h
vipermisc	db	0		; save for viper's miscellaneous
vipercr2	db	0		; save for viper's dac control reg 2
ScreenLines	db	0		; number of lines on screen

;============================================================================
;   Vga Save Areas
;============================================================================

FontSaveArea	dw	0940h dup (?)	;  font save area
Save_Vga	db	251 dup ('V')	;  space for vga registers
Save_Dac	db	64*3 dup (0)	;  save for dac registers

;============================================================================
;   Dac Settings.
;============================================================================

DacRegisterSettings label byte

	db  00h,00h,00h,00h,00h,2ah,00h,2ah,00h,00h,2ah,2ah,2ah,00h,00h,2ah
	db  00h,2ah,2ah,2ah,00h,2ah,2ah,2ah,00h,00h,15h,00h,00h,3fh,00h,2ah
	db  15h,00h,2ah,3fh,2ah,00h,15h,2ah,00h,3fh,2ah,2ah,15h,2ah,2ah,3fh
	db  00h,15h,00h,00h,15h,2ah,00h,3fh,00h,00h,3fh,2ah,2ah,15h,00h,2ah
	db  15h,2ah,2ah,3fh,00h,2ah,3fh,2ah,00h,15h,15h,00h,15h,3fh,00h,3fh
	db  15h,00h,3fh,3fh,2ah,15h,15h,2ah,15h,3Fh,2ah,3fh,15h,2ah,3fh,3fh
	db  15h,00h,00h,15h,00h,2ah,15h,2ah,00h,15h,2ah,2ah,3fh,00h,00h,3fh
	db  00h,2ah,3fh,2ah,00h,3fh,2ah,2ah,15h,00h,15h,15h,00h,3fh,15h,2ah
	db  15h,15h,2ah,3fh,3fh,00h,15h,3fh,00h,3fh,3fh,2ah,15h,3fh,2ah,3fh
	db  15h,15h,00h,15h,15h,2ah,15h,3fh,00h,15h,3fh,2ah,3fh,15h,00h,3fh
	db  15h,2ah,3fh,3fh,00h,3fh,3fh,2ah,15h,15h,15h,15h,15h,3fh,15h,3fh
	db  15h,15h,3fh,3fh,3fh,15h,15h,3fh,15h,3fh,3fh,3fh,15h,3fh,3fh,3fh

;============================================================================
;   Vga Register settings for 400-line color text modes.
;============================================================================

VgaRegisterSettings label byte

	db	01h,03h,00h,02h 			;  sequencer
	db	063h					;  miscellaneous
	db	05fh,04fh,050h,082h,055h,081h,0bfh,01fh ;  crtc
	db	000h,0c7h,0feh,0feh,0feh,0feh,0feh,0feh ;
	db	09ch,08eh,08fh,028h,01fh,096h,0b9h,0a3h ;
	db	0feh					;
	db	000h,001h,002h,003h,004h,005h,014h,007h ;  attribute
	db	038h,039h,03ah,03bh,03ch,03dh,03eh,03fh ;
	db	008h,000h,00fh,000h			;
	db	000h,000h,000h,000h,000h,010h,00eh,000h ;  graphics
	db	0ffh					;

;=============================================================================
;   Register Settings to setup the vga for font operations
;=============================================================================

FontSeq label word			;  sequencer settings

	dw	0100h			;  synchronous reset
	dw	0402h			;  CPU writes only to map 2
	dw	0704h			;  sequential addressing
	dw	0300h			;  clear synchronous reset

FontGraphs label word			;  graphics controller settings

	dw	0204h			; select map 2 for CPU reads
	dw	0005h			; disable odd-even addressing
	dw	0c06h			; map starts at B800:0000

;=============================================================================
;   Register Settings to bring the vga back from font operations
;=============================================================================

FromFontSeq label word			;  sequencer settings

	dw	0100h			;  synchronous reset
	dw	0302h			;  CPU writes to maps 0 and 1
	dw	0304h			;  odd-even addressing
	dw	0300h			;  clear synchronous reset

FromFontGraph label word		;  graphics controller settings

	dw	0004h			; select map 0 for CPU reads
	dw	1005h			; enable odd-even addressing
	dw	0e06h			; map starts at B800:0000

;============================================================================
;   Other data fields.
;=============================================================================

Crtc0e		db	?		;  save for crt register 0eh
Crtc0f		db	?		;  save for crt register 0fh
Crtc09		db	?		;  save for crt register 09h
Crtc0c		db	?		;  save for crt register 0ch
InGraphMode	db	0		;  are we in graphics mode, y/n ?

VxD_LOCKED_DATA_ENDS

VxD_LOCKED_CODE_SEG


;==============================================================================
; 		C O N T R O L   D I S P A T C H   P R O C E D U R E
;==============================================================================

BeginProc SIWVID_Control

	Control_Dispatch Sys_Critical_Init, <SIWVID_Critical_Init>
	clc
	ret

EndProc SIWVID_Control

;==============================================================================
; 		S Y S   C R I T I C A L   I N I T   P R O C E D U R E
;
; This is the only VxD Control message that SIWVID processes.  At this point
; do any protected mode initialization required.  This routine is called
; prior to any services being called.
;   In:  EDX has value passed from real mode init if any
;   Out: carry clear for success, carry set to abort VxD load
;==============================================================================

BeginProc SIWVID_Critical_Init

	clc
	ret

EndProc SIWVID_Critical_Init

;==============================================================================
; 		G E T   V E R S I O N   S E R V I C E
;
; This routine is called once by Soft-ICE/W to determine if a SIW video driver
; is loaded. It does not have to be altered.
;    In:  Nothing.
;    Out: Version in eax.
;==============================================================================

BeginProc SIWVID_GetVersion, SERVICE

	mov	eax,100h		; return version 1.0
	clc				; return success
	ret

EndProc SIWVID_GetVersion

;==============================================================================
; 			S E T U P   S E R V I C E
;
; This routine is called once by Soft-ICE/W (after SYS_CRITICAL_INIT) and
; passes a linear address that can be used to address any physical memory.
; Just add this address to any physical address before accessing memory.
;
;   In:  EDX - Linear address that maps any physical address
;	 ESI - Linear address of 8x16 rom font (for 25 line mode) 
;	 EDI - Linear address of 8x8 rom font (for 44 and 50 line mode)
;	 EBX - Linear address of 8000 byte video memory save buffer
;   Out: Nothing.
;==============================================================================

BeginProc SIWVID_Setup, SERVICE

	mov	MapPhysical,edx 	    ;  save for addressing video memory
	add	ScreenBase,edx		    ;
	mov	Font8x16,esi		    ;
	mov	Font8x8,edi		    ;
	mov	SaveBufferAddress,ebx	    ;
	movzx	esi,word ptr [edx+04a8h]    ;  offset of vga master table
	movzx	eax,word ptr [edx+04a8h+2]  ;  segment of vga master table
	shl	eax,4			    ;  compute linear address
	add	esi,eax 		    ;
	add	esi,edx 		    ;
	movzx	eax,word ptr [esi+2]	    ;  offset of vga register settings
	movzx	esi,word ptr [esi]	    ;  segment of vga register settings
	shl	eax,4			    ;  compute linear address
	add	esi,eax 		    ;
	add	esi,605h		    ;  point to mode 3 vga hires table
	add	esi,edx 		    ;
	mov	ecx,59			    ;  64-5=59, skip first 5 bytes
	lea	edi,VgaRegisterSettings     ;  move them to here
	cld				    ;  (in case of non-compatible vga)
	rep	movsb			    ;
	ret

EndProc SIWVID_Setup


;==============================================================================
; 			S A V E   S C R E E N   S E R V I C E
;
; This routine is called when Soft-ICE/W is ready to pop up.  This routine
; must do the following:
;	- Save the current video state for later restoring (The current video
;	  state can be either graphics or text (DOS BOX))
;	- Save the 8K of video memory that Soft-ICE/W will use in text mode
;	- Switch the video board into the appropriate text mode 
;	  (25, 44, or 50 line mode)
;
;   In:  EDX = Text Mode (25, 44, or 50)
;   Out: Nothing.
;==============================================================================

BeginProc SIWVID_SaveScreen, SERVICE

	call	SaveUserState		;  save user state
	call	RestoreSiceState	;  restore SoftIce state
	ret

EndProc SIWVID_SaveScreen

;==============================================================================
;		R E S T O R E   S C R E E N   S E R V I C E
;
; This routine is called when Soft-ICE/W is ready to pop down.  This routine
; must restore the video to the state saved by SaveScreen.
;   In:  Nothing.
;   Out: Nothing.
;==============================================================================

BeginProc SIWVID_RestoreScreen, SERVICE

	call	SaveSiceState		;  save SoftIce state
	call	RestoreUserState	;  restore user state
	ret

EndProc SIWVID_RestoreScreen

;=============================================================================
; SaveSiceState: Saves SoftIce video state registers & font.
;   In:  Nothing.
;   Out: Nothing.
;=============================================================================

SaveSiceState proc

	mov	al,0ch			;  save our crtc 0ch
	mov	dx,3d4h 		;
	out	dx,al			;
	in	ax,dx			;
	inc	al			;  save our crtc 0dh
	out	dx,al			;
	inc	dx			;
	in	al,dx			;
	mov	sicecursor,ax		;  store as cursor position
	ret

SaveSiceState endp

;=============================================================================
; RestoreSiceState: Restores user's video state registers & font.
;   In:  Nothing.
;   Out: Nothing.
;=============================================================================

RestoreSiceState proc

	mov	bx,sicecursor		;  get saved cursor position
	mov	dx,3d4h 		;
	mov	al,0ch			;  restore crtc 0ch
	mov	ah,bh			;
	out	dx,ax			;
	inc	al			;  restore crtc 0dh
	mov	ah,bl			;
	out	dx,ax			;
	ret

RestoreSiceState endp

;=============================================================================
; SaveUserState: Saves user video state registers & font.
;   In:  DL = screen density (25, 44, or 50 lines/screen)
;   Out: Nothing.
;=============================================================================

SaveUserState proc

	mov	ScreenLines,dl		;  set lines/screen
	call	SaveUserScreen		;  save their screen
	mov	usercursor,ax		;  save user cursor
	ret

SaveUserState endp

;=============================================================================
; RestoreUserState: Restores user's video state registers & font.
;   In:  Nothing.
;   Out: Nothing.
;=============================================================================

RestoreUserState proc

	cld				;  good direction
	mov	ax,usercursor		;  ax - position to be restored
	call	RestoreUserScreen	;  go restore their screen
	jc	short @f		;  exit if carry, don't restore memory
	mov	esi,SaveBufferAddress	;  from user buffer
	mov	edi,ScreenBase		;  to video memory
	mov	ecx,SCREENSIZE		;  load the save/restore length
	cld				;  good direction
	rep	movsw			;  move one row to buffer
@@:
	ret

RestoreUserState endp

;=============================================================================
; Save User Screen: Saves user's video memory and registers.
;   In:  Nothing.
;   Out: ax=user cursor position.
;=============================================================================

SaveUserScreen proc

	mov	Crtc09,0		;  assume no change
	mov	Crtc0c,0		;
	call	IsInGraph		;  see if in graphics mode
	jnc	short @f		;  jump, we are in graphics mode
	mov	dx,3d4h 		;
	mov	al,09h			;  get crtc register 9
	out	dx,al			;
	in	ax,dx			;
	mov	Crtc09,ah		;  save crtc 9
	mov	al,0ch			;  get crtc register 0ch
	out	dx,al			;
	in	ax,dx			;
	mov	Crtc0c,ah		;  save crtc 0ch
	call	SaveFont		;  save the text font
	jmp	short StDontSave	;
@@:
	call	SaveVga 		;  save vga environment
StDontSave:
	mov	esi,ScreenBase		;  add offset in
	mov	edi,SaveBufferAddress	;  point to save buffer area
	mov	ecx,SCREENSIZE		;  size of screen area to be saved
	cld				;  good direction
	rep	movsw			;  save it
	mov	edi,ScreenBase		;
	xor	eax,eax 		;  ax has zeros
	mov	ecx,4000		;  clear the screen area
	rep	stosw			;
	mov	esi,MapPhysical 	;  use virtual address of 0:450
	mov	eax,[esi+450h]		;  return cursor position
	ret

SaveUserScreen endp

;=============================================================================
; Restore User Screen: Restore User's video memory and registers.
;   In:  Nothing.
;   Out: Nothing.
;=============================================================================

RestoreUserScreen proc

	mov	esi,SaveBufferAddress	;  point to screen save area
	mov	edi,ScreenBase		;
	mov	ecx,4000		;  restore the screen
	cld				;  good direction
	rep	movsw			;
	cmp	InGraphMode,0		;  are we in graphics mode ?
	jnz	short SwitchToGraph	;  jump if yes
	cmp	Crtc0c,0		;  need to change video start address?
	jz	short @f		;  no
	mov	ah,Crtc0c		;  get msb start address
	mov	al,0ch			;  crtc port 0ch
	mov	dx,3d4h 		;  program Crtc
	out	dx,ax			;
@@:
	cmp	Crtc09,0		;  need to switch # of screen lines?
	jz	short @f		;  jump if not
	mov	ah,Crtc09		;  else read crt register 9
	mov	al,09h			;
	mov	dx,3d4h 		;
	out	dx,ax			;
	delay				;
	call	RestoreFont		;  restore the text font
	lea	esi,FromFontSeq 	;
	call	ChangeFontMode		;  reinstate char generators
@@:
	stc				;  all done here
	jmp	short BuffDone		;  and go on
SwitchToGraph:
	call	RestoreVga		;  restore the vga environment
BuffDone:
	mov	dx,3d4h 		;  restore cursor registers
	mov	al,0eh			;
	mov	ah,Crtc0e		;
	out	dx,ax			;
	mov	al,0fh			;
	mov	ah,Crtc0f		;
	out	dx,ax			;
	ret

RestoreUserScreen endp

;=============================================================================
;   Save the vga state.
;=============================================================================

SaveVga proc near

	call	ViperGoText		;  put vga in text mode
	call	ViperSaveDac		;  save the dac registers
	call	ViperDoDac		;  load our dac registers
	lea	esi,VgaRegisterSettings ;  point to mode 3 table
	mov	edi,esi 		;  use it for template as well
	call	SetVgaRegisters 	;  set all vga registers
	call	SaveFont		;  save the font
	stc				;  handle this buffer
	ret

SaveVga endp

;=============================================================================
;   Restore VGA state.
;=============================================================================

RestoreVga proc

	call	ViperRestoreDac 	;  restore viper dacs
	call	ViperGoGraph		;  put viper in graphics mode
	clc				;  handle this buffer
	ret

RestoreVga endp

;=============================================================================
;   Write vga registers from a video bytes table in esi.
;=============================================================================

SetVgaRegisters proc

	push	esi			;  save esi
	mov	dx,3c4h 		;  reset the sequencer
	mov	ax,0100h		;
	out	dx,ax			;
	delay				;
	mov	ecx,4			;  4 sequencer registers
	mov	ah,1			;  starting with register 1
	mov	bx,3c5h 		;
	call	SetRegisters		;  go do sequencer
	mov	dx,3c2h 		;  get the miscellaneous
	mov	al,[esi]		;
	inc	esi			;
	inc	edi			;  next byte in template
	push	ax			;  save ax = miscellaneous setting
	or	al,1			;  set misc to color
	out	dx,al			;
	delay				;
	mov	dx,3d4h 		;  do crt controller now
	mov	ax,0011h		;  first open the protection
	out	dx,ax			;
	delay				;
	mov	bx,3d5h 		;
	mov	ecx,25			;  do all 25 Crtc registers
	xor	ah,ah			;  starting at 0
	call	SetRegisters		;  go do crt controller
	mov	dx,3dah 		;  reset attribute flip/flop
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	mov	dx,3c0h 		;  attribute controller base
	mov	bx,dx			;
	mov	ecx,20			;  20 registers total
	xor	ah,ah			;  starting at 0
	call	SetRegisters		;  go do attribute controller
	mov	dx,3ceh 		;  graphics controller
	mov	bx,3cfh 		;
	mov	ecx,9			;  do 9 graphics controller registers
	xor	ah,ah			;  starting at 0
	call	SetRegisters		;  go do graphics controller
	mov	dx,3dah 		;  reset attribute flip/flop
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	mov	dx,3c0h 		;  turn palette back on
	mov	al,20h			;
	out	dx,al			;
	delay				;
	mov	dx,3dah 		;  reset attribute flip/flop
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	pop	ax			;  restore miscellaneous setting
	mov	dx,3c2h 		;  and write it out
	out	dx,al			;
	delay				;
	pop	esi			;  restore base address
	ret

SetVgaRegisters endp

;=============================================================================
; SetRegisters: Set an array of vga registers from a memory table.
;
;   In:  ah = starting register number
;	 bx = data port for this vga controller
;	 dx = address port for this vga controller
;	 cx = number of registers to be written
;	 ds:si = points to register value table
;	 es:di = template of which registers to be written (0feh=don't write)
;=============================================================================

SetRegisters proc

	push	eax
SetLoop:
	mov	al,ah			;  get the index number
	out	dx,al			;  issue index value
	delay				;
	xchg	dx,bx			;  get the data register
	lodsb				;  get next byte from table
	cmp	byte ptr es:[edi],0feh	;  should we do it?
	je	@f			;  skip if not
	out	dx,al			;  else send the data
	delay				;
	jmp	short SetGoon		;
@@:
	cmp	dx,3c0h 		;  is this the attribute controller ?
	jne	@f			;  skip if not
	mov	dx,3dah 		;  else reset the attribute controller
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	mov	dx,3c0h 		;  dx has 3c0 again
SetGoon:
	xchg	dx,bx			;  get the address register back
	inc	ah			;  increment the index number
	inc	edi			;  next in the template
	loop	SetLoop 		;  do them all
	pop	eax			;  restore ax
	ret

SetRegisters endp

;=============================================================================
;   Save vga registers into es:edi.
;=============================================================================

GetVgaRegisters proc

	mov	dx,3c4h 		;  reset the sequencer
	mov	ax,0100h		;
	out	dx,ax			;
	delay				;
	mov	ecx,4			;  4 sequencer registers
	mov	ah,1			;  starting with register 1
	mov	bx,3c5h 		;
	call	GetRegisters		;  save sequencer
	mov	dx,3cch 		;  read the miscellaneous
	in	al,dx			;
	delay				;
	stosb				;  save it
	or	al,01h			;  set miscellaneous to color side
	mov	dx,3c2h 		;
	out	dx,al			;
	delay				;
	mov	dx,3d4h 		;  save the crt controller
	mov	bx,3d5h 		;
	mov	ecx,25			;  25 registers to do
	xor	ah,ah			;  starting at register 0
	call	GetRegisters		;  go save crt controller
	mov	dx,3dah 		;  reset the attribute flip/flop
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	mov	dx,3c0h 		;  point to attribute controller
	mov	bx,3c1h 		;
	mov	ecx,20			;  20 registers to save
	xor	ah,ah			;  start at register 0
	call	GetRegisters		;  save attribute controller
	mov	dx,3ceh 		;  point to graphics controller
	mov	bx,3cfh 		;
	mov	ecx,9			;  9 registers to save
	xor	ah,ah			;
	call	GetRegisters		;
	mov	dx,3c7h 		;  read the first dac registers
	mov	al,0			;  starting at register 0
	out	dx,al			;
	delay
	mov	dx,3c9h 		;
	mov	ecx,0c0h		;  register count * 3
@@:
	in	al,dx			;  read the register
	delay				;
	stosb				;  and save it
	loop	@b			;
	ret

GetVgaRegisters endp

;=============================================================================
; GetRegisters: Save an array of vga registers to a memory area.
;
;   In:  ah = starting register number
;	 bx = data port for this vga controller
;	 dx = address port for this vga controller
;	 cx = number of registers to be written
;	 es:di = points to register save area
;=============================================================================

GetRegisters proc

	push	ax			;  save ax
GetLoop:
	mov	al,ah			;  issue index
	out	dx,al			;
	delay				;
	xchg	bx,dx			;  dx points to data port
	in	al,dx			;  read the data
	delay				;
	stosb				;  save it
	xchg	bx,dx			;  dx points to address port
	cmp	dx,3c0h 		;  attribute controller ?
	jne	@f			;  skip if not
	mov	dx,3dah 		;  else reset attribute flip/flop
	in	al,dx			;
	mov	dx,3bah 		;
	in	al,dx			;
	mov	dx,3c0h 		;  and restore dx
@@:
	inc	ah			;  increment register number
	loop	GetLoop 		;  repeat for all registers
	pop	ax			;  restore ax
	ret

GetRegisters endp

;=============================================================================
;   Save font.
;=============================================================================

SaveFont proc

	lea	esi,FontSeq	    ;  point to font
	call	ChangeFontMode	    ;  setup vga for font operations
	mov	ah,4dh		    ;  assume ega 25
	cmp	ScreenLines,25	    ;  are we in 25 line mode?
	jnz	short @f	    ;  jump if not
	mov	ah,4fh		    ;  else set for 25 lines/screen
	jmp	short SFGoon	    ;  and go on
@@:
	mov	ah,47h		    ;  assume vga 50 or ega 43
	cmp	Crtc09,47h	    ;  is user screen 43 line mode
	jz	short SFGoon	    ;  yes leave it that way
	cmp	ScreenLines,50	    ;  50 lines mode?
	je	short SFGoon	    ;  use it
	mov	ah,48h		    ;  else make it 44 lines
SFGoon:
	mov	al,09h		    ;  reprogram register 9
	mov	dx,3d4h 	    ;
	out	dx,ax		    ;
	delay			    ;
	lea	edi,FontSaveArea    ;  font save area
	mov	esi,ScreenBase	    ;
	mov	ecx,0800h	    ;  move the first 4K
	cld				;  good direction
	rep	movsw		    ;  move it out
	mov	esi,ScreenBase	    ;  get the offset of the font
	add	esi,32*0bah	    ;  get to the graphics chars
	mov	ecx,(20*32)/2	    ;  save 20 of them
	rep	movsw		    ;  save them
	mov	ebx,4		    ;  assume 8X8 font
	mov	esi,Font8x8	    ;  use it
	cmp	ScreenLines,25	    ;  are we in 25 line mode?
	ja	@f		    ;  no so use them (else use larger font)
	mov	esi,Font8x16	    ;  get the offset of the font
	mov	ebx,8		    ;  else this is a lowres font
@@:
	mov	edi,ScreenBase	    ;  start at the beginning
	push	esi		    ;  save the offset again
	mov	ecx,128 	    ;  move 4K in
@@:
	push	ecx		    ;
	mov	ecx,ebx 	    ;
	rep	movsw		    ;  do it
	mov	ecx,16		    ;
	sub	ecx,ebx 	    ;  do the remainder
	xor	eax,eax 	    ;
	rep	stosw		    ;
	pop	ecx		    ;
	loop	@b		    ;
	pop	esi		    ;  restore font offset
	mov	edi,ScreenBase	    ;  edi points to screen
	add	edi,32*0bah	    ;  this is where the graphics chars start
	mov	eax,0bah	    ;  this is the char we want to get to
	mul	ebx		    ;  multiply by bx
	shl	eax,1		    ;  make it bytes
	add	esi,eax 	    ;  get to that char
	mov	ecx,20		    ;  now do the graphic chars
@@:
	push	ecx		    ;
	mov	ecx,ebx 	    ;
	cld				;  good direction
	rep	movsw		    ;  do it
	mov	ecx,16		    ;
	sub	ecx,ebx 	    ;  do the remainder
	xor	eax,eax 	    ;
	rep	stosw		    ;
	pop	ecx		    ;
	loop	@b		    ;
	lea	esi,FromFontSeq     ;
	call	ChangeFontMode	    ;  go back to normal mode
	ret

SaveFont endp

;=============================================================================
;   Restore font.
;=============================================================================

RestoreFont proc

	lea	esi,FontSeq	     ;	point to font
	call	ChangeFontMode	     ;	set vga to operate on font
	mov	edi,ScreenBase	     ;	base the screen
	lea	esi,FontSaveArea     ;	base font area
	mov	ecx,0800h	     ;	move all 4K
	cld				;  good direction
	rep	movsw		     ;	move it out
	mov	edi,ScreenBase	     ;	base the screen
	add	edi,32*0bah	     ;	point to graphics chars
	mov	ecx,(32*20)/2	     ;	do 20 of them
	rep	movsw		     ;	move them
	ret

RestoreFont endp

;=============================================================================
;   Setup card to allow font restore.
;=============================================================================

ChangeFontMode proc

	mov	dx,3c4h 	    ;  Sequencer port address
	mov	ecx,4		    ;
@@:
	lodsw			    ;  ah=value, AL=register number
	out	dx,ax		    ;  program the sequencer register
	loop	@b		    ;
	mov	dl,0ceh 	    ;  dx=3ceh (Graphics Controller port)
	mov	ecx,3		    ;
@@:
	lodsw			    ;  program the Graphics Controller
	out	dx,ax		    ;
	loop	@b		    ;
	ret

ChangeFontMode endp

;=============================================================================
;   Are we in graphic mode, yes/no.
;=============================================================================

IsInGraph proc

	mov	dx,3d4h 		;  talk to the crtc
	mov	al,0eh			;  cursor location high
	out	dx,al			;
	delay				;
	inc	dx			;
	in	al,dx			;  read it and save it
	delay				;
	mov	Crtc0e,al		;
	mov	ah,al			;
	dec	dx			;
	mov	al,0fh			;  cursor location high
	out	dx,al			;
	delay				;
	inc	dx			;
	in	al,dx			;  read it and save it
	delay				;
	mov	crtc0f,al		;
	call	ViperIsGraph		;  P9000 on ?
	jz	IgNot			;  jump if not
	mov	InGraphMode,1		;  yes we are in graphics mode
	clc				;
	jmp	short IgOut		;  and exit
IgNot:
	mov	InGraphMode,0		;  no we are not in graphics mode
	stc				;
IgOut:
	ret

IsInGraph endp

;=============================================================================
;   See if the viper is in graphics mode. NZ if P9000 on, Z if off.
;=============================================================================

ViperIsGraph proc near

	mov	dx,3c4h 		;  sequencer index
	in	al,dx			;
	delay				;
	push	ax			;  save sequencer index
	call	ViperUnlock		;  unlock the viper
	mov	dl,0c4h 		;  sequencer again
	mov	ax,12h			;  enable vga, disable p9000
	out	dx,al			;  read 5186 miscellaneous
	delay				;
	in	ax,dx			;
	push	ax			;  save it
	call	ViperLock		;  lock it again
	pop	ax			;  restore it
	test	al,10h			;  = 1 if p9000 is on
	pop	ax			;  restore 3c4 index
	mov	dx,3c4h 		;
	out	dx,al			;
	ret

ViperIsGraph endp

;=============================================================================
;   Put the Diamond Viper in text mode.
;=============================================================================

ViperGoText proc near

	mov	dx,3c4h 		;  sequencer index
	in	al,dx			;  save it
	delay				;
	push	ax			;
	mov	dl,0cch 		;  save the miscellaneous
	in	al,dx			;
	delay				;
	mov	vipermisc,al		;
	mov	dx,83c9h		;  save command register 2
	in	al,dx			;
	delay				;
	mov	vipercr2,al		;
	and	al,not 3		;  disable graphics cursor
	out	dx,al			;
	delay				;
	call	ViperUnlock		;  unlock the viper
	mov	dx,3c4h 		;  sequencer again
	mov	ax,12h			;  enable vga, disable p9000
	out	dx,al			;
	delay				;
	inc	dl			;
	in	al,dx			;  get the value
	delay				;
	mov	vipermode,al		;  save it in our data area
	mov	al,88h			;
	out	dx,al			;
	delay				;
	call	ViperLock		;  lock it again
	pop	ax			;  restore 3c4 index
	mov	dx,3c4h 		;
	out	dx,al			;
	ret

ViperGoText endp

;=============================================================================
;   Put the Diamond Viper in graphics mode.
;=============================================================================

ViperGoGraph proc near

	mov	dx,3c4h 		;  sequencer index
	in	al,dx			;  save it
	push	ax			;
	call	ViperUnlock		;  unlock the viper
	mov	dx,3c4h 		;  sequencer again
	mov	ax,12h			;  enable vga, disable p9000
	out	dx,al			;
	delay				;
	inc	dl			;
	mov	al,vipermode		;  restore original viper state
	out	dx,al			;
	delay				;
	call	ViperLock		;  lock it again
	pop	ax			;  restore 3c4 index
	mov	dx,3c4h 		;
	out	dx,al			;
	delay				;
	mov	dl,0c2h 		;  restore miscellaneous
	mov	al,vipermisc		;
	out	dx,al			;
	delay				;
	mov	dx,83c9h		;  save command register 2
	mov	al,vipercr2		;
	out	dx,al			;
	delay				;
	ret

ViperGoGraph endp

;=============================================================================
;   Unlock the Viper board.
;=============================================================================

ViperUnlock proc near

	mov	dx,3c4h 		;  sequencer index
	mov	al,8			;  set bit 3 for 5186 unlock
	out	dx,al			;
	delay				;
	mov	dl,0ah			;  unlock 5186 vga chip
	in	al,dx			;
	delay				;
	out	dx,al			;
	delay				;
	out	dx,al			;
	delay				;
	in	al,dx			;
	delay				;
	and	al,not 20h		;
	out	dx,al			;
	delay				;
	ret

ViperUnlock endp

;=============================================================================
;   Lock the Viper board.
;=============================================================================

ViperLock proc near

	mov	dx,3c4h 		;
	mov	al,8			;  set bit 3 for 5186 lock
	out	dx,al			;
	delay				;
	mov	dl,0ah			;
	in	al,dx			;
	delay				;
	or	al,20h			;
	out	dx,al			;
	delay				;
	ret

ViperLock endp

;=============================================================================
;   Save the first 64 dac registers.
;=============================================================================

ViperSaveDac proc near

	lea	edi,Save_Vga		;  point to save area
	mov	dx,3c7h 		;  read dac 0
	mov	bl,0			;
	mov	al,0			;
	out	dx,al			;
	delay				;
	mov	dl,0c9h 		;
@@:
	in	al,dx			;  red
	delay				;
	mov	[edi],al		;
	inc	edi			;
	in	al,dx			;  green
	delay				;
	mov	[edi],al		;
	inc	edi			;
	in	al,dx			;  blue
	delay				;
	mov	[edi],al		;
	inc	edi			;
	inc	bl			;  next register
	cmp	bl,64			;
	jl	@b			;  repeat for all registers
	ret

ViperSaveDac endp


;=============================================================================
;   Set the first 64 dac registers.
;=============================================================================

ViperDoDac proc near

	lea	esi,DacRegisterSettings ;  point to new settings
	mov	bl,0			;
VDDTop:
	mov	al,bl			;  register number
	mov	dx,3c8h 		;
	out	dx,al			;
	delay				;
	inc	dx			;
	mov	al,[esi]		;  red
	or	al,al			;
	jz	@f			;
	or	al,80h			;  make it 8-bit
@@:
	out	dx,al			;
	delay				;
	inc	esi			;
	mov	al,[esi]		;  green
	or	al,al			;
	jz	@f			;
	or	al,80h			;
@@:
	out	dx,al			;
	delay				;
	inc	esi			;
	mov	al,[esi]		;  blue
	or	al,al			;
	jz	@f			;
	or	al,80h			;
@@:
	out	dx,al			;
	delay				;
	inc	esi			;
	inc	bl			;  next register
	cmp	bl,64			;
	jl	VDDTop			;  repeat for all registers
	ret

ViperDoDac endp


;=============================================================================
;   Restore the first 64 dac registers.
;=============================================================================

ViperRestoreDac proc near

	lea	esi,Save_Vga		;  point to save area
	mov	dx,3c8h 		;  dac 0
	mov	al,0			;
	mov	bl,0			;
	out	dx,al			;
	delay				;
	inc	dx			;
@@:
	mov	al,[esi]		;  red
	out	dx,al			;
	delay				;
	inc	esi			;
	mov	al,[esi]		;  green
	out	dx,al			;
	delay				;
	inc	esi			;
	mov	al,[esi]		;  blue
	out	dx,al			;
	delay				;
	inc	esi			;
	inc	bl			;  next register
	cmp	bl,64			;
	jl	@b			;  repeat for all registers
	ret

ViperRestoreDac endp


;==============================================================================
; 		  D E B U G G I N G   E N T R Y   P O I N T
;
; This routine is called by the DOS or Windows program TESTVID.EXE and can be
; used to debug your SIWVID.386.  This routine does not need to be modified.
;==============================================================================

BeginProc SIWVID_Debug

	mov	edx,25
	call	$SIWVID_SaveScreen
	call	$SIWVID_RestoreScreen
	ret

EndProc SIWVID_Debug


VxD_LOCKED_CODE_ENDS


;==============================================================================
; 			R E A L   M O D E   I N I T   
;
; Do any real mode mode initialization required.  A single DWORD value can
; be passed from here to the protected mode portion of the VxD.
;
; INPUT:   CS=DS=ES	code/data segment	
;	   SI		environment segment
;	   AX		VMM version number
;	   BX		flags
;	   EDX          reference data from int 2fh response or zero
;
; OUTPUT   AX		exit code 0-success
;				  1-error don't load this VxD
;				  2-fatal error - abort loading windows
;	   BX		ptr to list of pages to exclude (0 if none)	
;	   SI		ptr to instance data items (0 if none)
;	   EDX		value that can be passed to protected mode portion
;==============================================================================

VxD_REAL_INIT_SEG

BeginProc SIWVID_init

	xor	ax,ax			; return success
	xor	si,si			; NO instance data
	xor	bx,bx			; NO excluded pages
	ret

EndProc SIWVID_init


VxD_REAL_INIT_ENDS

	END SIWVID_init
