#include <linux/errno.h>
#include <linux/kernel.h>
#include <asm/segment.h>
#include <linux/mm.h>		/* defines GFP_KERNEL */
#include <linux/string.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/malloc.h>

/*#define DEBUG*/
/*#define DEBUG2*/
/*#define NORECOVER */

struct module *module_list = NULL;
int freeing_modules;		/* true if some modules are marked for deletion */

struct module *find_module( const char *name);
int get_mod_name( char *user_name, char *buf);
static void dorelease(void);
int free_modules( void);
static struct kernel_sym* ks;
static struct mod_composite * km;
static void * bootloading=0;
static int memory_end;
static int dma_segment=0;
static int dma_seg_size=0;
int dma_release_at=0;  /* Release unused ram after boot after this many mod loads.*/
extern int dma_save_size;
extern int dma_reserve_size;
static int mod_verbose = 0;

#define DO_ALIGN(addr) PAGE_ALIGN(addr)

extern int symbol_table_size;
extern struct mod_sym_entry symbol_table[];
extern release_area(unsigned,unsigned);
#ifdef NORECOVER
#define release_area(a,b)
#endif

asmlinkage unsigned long
module_init(unsigned long mem_start, unsigned int dma_size)
{
	int i, m=0;
	printk("SLS modularized kernel\n");
#ifdef FOLLOW_KERNEL
	ks = (struct kernel_sym*)PAGE_ALIGN(mem_start+sizeof(long));
#else
	ks = (struct kernel_sym*)(1024*2048);
#endif
	if (strncmp(ks->name,MOD_SIGNATURE, strlen(MOD_SIGNATURE)))
	{
		ks=NULL;
		km=NULL;
		return(mem_start);
	}
	ks->name[0] = '\0';  /* Delete sig for warm boots. */
	mem_start=(unsigned int)ks;
	km = (struct mod_composite *) (mem_start+sizeof(struct kernel_sym));
	for (i=0; (i>=0)&&(i<ks->value)&&(ks->value<99); i++)
	{	
#ifdef DEBUG2
		printk("module %s: length=%d\n", km[i].k.name, km[i].m.bootflag);
#endif
		m += DO_ALIGN(km[i].m.bootflag+sizeof(int));
	}
	mem_start = (PAGE_ALIGN(((int)ks)+MOD_HEADER_SIZE+m));
	if (dma_size)	/* save a pool of memory for drivers with requests > 4096. */
	{	dma_seg_size=dma_size;
		dma_segment=mem_start;
		mem_start+=dma_size;
	}
	return(mem_start);
}

/*
 * Allocate space for a module.
 */
asmlinkage int
sys_create_module(char *module_name, unsigned long size)
{
	int npages;
	void* addr;
	int len;
	char name[MOD_MAX_NAME];
	char *savename;
	struct module *mp;
	int error;

	if (!suser())
		return -EPERM;
	if (module_name == NULL || size == 0)
		return -EINVAL;
	if ((error = get_mod_name(module_name, name)) != 0)
		return error;
	if (find_module(name) != NULL) {
		return -EEXIST;
	}
	len = strlen(name) + 1;
	if ((savename = (char*) kmalloc(len, GFP_KERNEL)) == NULL)
		return -ENOMEM;
	memcpy(savename, name, len);
	if ((mp = (struct module*) kmalloc(sizeof *mp, GFP_KERNEL)) == NULL) {
		kfree(savename);
		return -ENOMEM;
	}
	mp->bootflag=0;
	npages = (size + sizeof (int) + 4095) / 4096;
	if (bootloading)
	{	addr=bootloading;
		mp->bootflag=1;
	}
	else if ((addr = vmalloc(npages * 4096)) == 0) {
			kfree_s(mp, sizeof *mp);
			kfree(savename);
			return -ENOMEM;
		}
	mp->name = savename;
	mp->size = npages;
	mp->addr = addr;
	mp->state = MOD_UNINITIALIZED;
	* (int *) addr = 0;		/* set use count to zero */
	mp->cleanup = NULL;
	mp->next = module_list;
	mp->syms = NULL;
	mp->numsyms = 0;
	module_list = mp;
#ifdef DEBUG
	printk("module `%s' (%lu pages @ 0x%08lx) created\n",
		mp->name, (unsigned long) mp->size, (unsigned long) mp->addr);
#endif
	return (int) addr;
}

static struct module* find_mod_user(char *name)
{	/* Find the the first module using mod name */
	struct module *mpl = module_list;
	char *cp;
	int l = strlen(name);
	while (mpl)
	{	
		cp = mpl->dependancies;
		while (cp)
		{	if ((!strncmp(cp,name,l)) && ((cp[l]==' ') || (!cp[l])))
				return mpl;
			if ((cp=strchr(cp,' ')))
				cp++;
		}
		mpl=mpl->next;
	}
	return NULL;
}

static char *find_mod_deps(char *deps)
{	/* Check if all dependant modules are loaded */
	char *cp = deps;
	int found, l;
	while (cp)
	{	struct module *mpl = module_list;
		found=0;
		while (mpl && !found)
		{	
			l=strlen(mpl->name);
			if ((!strncmp(cp,mpl->name,l)) && ((cp[l]==' ') || (!cp[l])))
			{	found=1;
				break;
			}
			mpl=mpl->next;
		}
		if (!found)
			return cp;
		if ((cp=strchr(cp,' ')))
			cp++;
	}
	return 0;
}


/*
 * Initialize a module.
 */
asmlinkage int
sys_init_module(char *module_name, char *code, unsigned codesize,
		struct mod_routines *routines)
{
	struct module *mp;
	char *cp, name[MOD_MAX_NAME];
	int error, rc, clear;
	struct mod_routines rt;

	if (!suser())
		return -EPERM;
	/*
	 * First reclaim any memory from dead modules that where not
	 * freed when deleted. Should I think be done by timers when
	 * the module was deleted - Jon.
	 */
	free_modules();

	if (bootloading)
		strcpy(name,module_name);
	else if ((error = get_mod_name(module_name, name)) != 0)
		return error;
#ifdef DEBUG
	printk( "initializing module `%s', %d (0x%x) bytes\n",
		name, codesize, codesize);
#endif
	if (bootloading) memcpy(&rt, routines, sizeof rt);
	else memcpy_fromfs(&rt, routines, sizeof rt);
	if ((mp = find_module(name)) == NULL)
		return -ENOENT;
	if ((codesize + sizeof (int) + 4095) / 4096 > mp->size)
		return -EINVAL;
	clear=mp->size*PAGE_SIZE;
	if (!bootloading) 
		memcpy_fromfs((char *)mp->addr + sizeof (int), code, codesize);
	memset((char *)mp->addr + sizeof (int) + codesize, 0,
		clear - (codesize + sizeof (int)));
	if ((cp=find_mod_deps(rt.dependancies)))
	{
#ifdef DEBUG2
		printk("module %s missing dependacies: %s\n", name, cp);
#endif
		return -EINVAL;
	}
#ifdef DEBUG
	printk( "  init entry @ 0x%08lx, cleanup entry @ 0x%08lx\n",
		(unsigned long) rt.init, (unsigned long) rt.cleanup);
#endif
	if (rt.setup && (rt.args[0]>0))
		rt.setup("",rt.args);
	mp->cleanup = rt.cleanup;
	rc = (*rt.init)(0,memory_end);
	if (rc < 0)
		return -EBUSY;
	mp->state = MOD_RUNNING;

	mp->syms = rt.stable;
	mp->numsyms = rt.stable_size;
	mp->dependancies = rt.dependancies;
	if (!bootloading) 
		dorelease();
	return 0;
}

int module_setup(char *name, char *str, int *ints)	/* call module setup func. */
{
	int i;
	char *cp, nam[30];
	strncpy(nam,name,sizeof(nam));
	nam[29]=0;
	if ((cp=strchr(nam,'=')))
		*cp=0;
	for (i=0; i<ks->value; i++)
		if (!strcmp(km[i].k.name,nam))
			if (km[i].m.setup && (ints[0]>0))
			{
#ifdef DEBUG2
				printk("module_setup called for %s\n", str);
#endif
				km[i].m.setup(str,ints);
				return 0;
			}
	return 1;
}

void module_ignore(char *name)	/* delete modules from boottime list. */
{
	int i;
	char *nam = name;
	while (*name)
	{
		if ((nam=strchr(name,',')))
			nam++;
		else
			nam=name+strlen(name);
		for (i=0; i<ks->value; i++)
		{	int l = strlen(km[i].k.name);
			if ((!strncmp(km[i].k.name,name,l)) && ((name[l] == ',') || (!name[l])))
			{	km[i].m.init=0;
#ifdef DEBUG
				printk("excluding module %s\n", name);
#endif
				break;
			}
		}
		name=nam;
	}
}

void module_accept(char *name)	/* delete all but selected modules. */
{
	int i;
	char *nam;
	for (i=0; i<ks->value; i++)
	{	int l = strlen(km[i].k.name);
		nam=name;
		while (nam && *nam)
		{
			if ((!strncmp(km[i].k.name,nam,l)) && ((nam[l] == ',') || (!nam[l])))
				break;
			if (!(nam=strchr(nam,',')))
				break;
			else
				nam++;
		}
		if (!nam || !*nam)
		{
			km[i].m.init=0;
#ifdef DEBUG
			printk("deleting module %s\n", km[i].k.name);
#endif
		}
	}
}

asmlinkage int
sys_delete_module(char *module_name)
{
	struct module *mp, *mpl;
	char name[MOD_MAX_NAME];
	int error;

	if (!suser())
		return -EPERM;
	if (module_name != NULL) {
		if ((error = get_mod_name(module_name, name)) != 0)
			return error;
#ifdef DEBUG
		printk("deleting module %s\n", name);
#endif
		if ((mp = find_module(name)) == NULL)
			return -ENOENT;
		if (!bootloading && ((mpl=find_mod_user(name))))
		{	printk("delete of module %s failed: used by %s\n", name, mpl->name);
			return -EPERM;
		}
		mp->numsyms=0;
		mp->syms=NULL;
		if (mp->state == MOD_RUNNING)
			(*mp->cleanup)();
		mp->state = MOD_DELETED;
	}
	free_modules();
	return 0;
}

static void dorelease(void)
{
	if (dma_segment && !dma_release_at)
	{	
		if (!dma_seg_size) {
#ifdef DEBUG2
			printk("releasing %d unused pages of dma memory\n", dma_seg_size);
#endif
			release_area(dma_segment, dma_segment+dma_seg_size);
		}
	}
	if (dma_release_at)   /* How many post boot modules to load before freeing dma */
		dma_release_at--;
}

void module_startup(int mem_start, int mem_end)
{
	int i;
	unsigned int m=(unsigned int)ks;
	memory_end=mem_end;
	m+=MOD_HEADER_SIZE;
	if (!ks) return;
	for (i=0; i<ks->value; i++)
	{	int modsize=DO_ALIGN(km[i].m.bootflag+sizeof(int));
		bootloading=(void*)m;
		if (!km[i].m.init)
			release_area(m,m+modsize);
		else
			if (sys_create_module(km[i].k.name, km[i].m.bootflag)>0)
			{
				if (mod_verbose) printk("{%s", km[i].k.name);

				if (sys_init_module(km[i].k.name, (char *)bootloading+sizeof(int),
					km[i].k.value, &(km[i].m))<0)
				{
					if (mod_verbose) printk("{");
					sys_delete_module(km[i].k.name);
				}
				else
					if (mod_verbose) printk("}");
			}
		m+=modsize;
	}
	dorelease();
	bootloading=NULL;
	/* Now this directory page(s) could/should be freed. */
}

/*
 * Copy the kernel symbol table to user space.  If the argument is null,
 * just return the size of the table.
 */
asmlinkage int
sys_get_kernel_syms(struct kernel_sym *table)
{
	int i, sum=0, index=0, size = symbol_table_size;
	unsigned short inds[sizeof(table->name)/sizeof(unsigned short)];
	struct mod_sym_entry *from=symbol_table;
	struct kernel_sym *to;
	struct kernel_sym sym;
	struct module *mp = module_list;

	to = table;
	while (from)
	{
		sum += size;
		if (table != NULL) {
			i = verify_area(VERIFY_WRITE, to, size * sizeof *table);
			if (i)
				return i;
			for (i = size ; --i >= 0 ; ) {
				sym.value = from->addr;
				strncpy(sym.name, from->name, sizeof sym.name);
				memcpy_tofs(to, &sym, sizeof sym);
				from++, to++;
			}
			inds[index++]=size;
		}
		if (!mp) break;
		while (1)
		{	if (mp->syms)
			{	from=mp->syms;
				size=mp->numsyms;
				mp=mp->next;
				break;
			}
			else
				if (mp->next)
					mp=mp->next;
				else
					return(sum);
		}
	}
	return sum;
}


/*
 * Copy the name of a module from user space.
 */
int
get_mod_name(char *user_name, char *buf)
{
	int i;

	i = 0;
	for (i = 0 ; (buf[i] = get_fs_byte(user_name + i)) != '\0' ; ) {
		if (++i >= MOD_MAX_NAME)
			return -E2BIG;
	}
	return 0;
}


/*
 * Look for a module by name, ignoring modules marked for deletion.
 */
struct module *
find_module( const char *name)
{
	struct module *mp;

	for (mp = module_list ; mp ; mp = mp->next) {
		if (mp->state == MOD_DELETED)
			continue;
		if (!strcmp(mp->name, name))
			break;
	}
	return mp;
}


/*
 * Try to free modules which have been marked for deletion.  Returns nonzero
 * if a module was actually freed.
 */
int
free_modules( void)
{
	struct module *mp;
	struct module **mpp;
	int did_deletion;

	did_deletion = 0;
	freeing_modules = 0;
	mpp = &module_list;
	while ((mp = *mpp) != NULL) {
		if (mp->state != MOD_DELETED) {
			mpp = &mp->next;
		} else if (GET_USE_COUNT(mp) != 0) {
			freeing_modules = 1;
			mpp = &mp->next;
		} else {	/* delete it */
			*mpp = mp->next;
			if (mp->bootflag) {
#ifdef DEBUG2
				printk("releasing module %s (%lx,%lx)\n", mp->name, mp->size, mp->addr);
#endif
				release_area((unsigned)mp->addr, 
					(unsigned)mp->addr+mp->size*PAGE_SIZE);
			} else
				vfree(mp->addr);
			kfree(mp->name);
			kfree_s(mp, sizeof *mp);
			did_deletion = 1;
		}
	}
	return did_deletion;
}


/*
 * Called by the /proc file system to return a current list of modules.
 */
int get_module_list(char *buf)
{
	char *p;
	char *q;
	int i;
	struct module *mp;
	char size[45];

	p = buf;
	for (mp = module_list ; mp ; mp = mp->next) {
		if (p - buf > 4096 - 100)
			break;			/* avoid overflowing buffer */
		q = mp->name;
		i = 20;
		while (*q) {
			*p++ = *q++;
			i--;
		}
		sprintf(size, "%d(%lx)", mp->size, mp->addr);
		i -= strlen(size);
		if (i <= 0)
			i = 1;
		while (--i >= 0)
			*p++ = ' ';
		q = size;
		while (*q)
			*p++ = *q++;
		if (mp->dependancies)
		{
			*p++ = ' ';
			*p++ = '[';
			q = mp->dependancies;
			while (*q)
				*p++ = *q++;
			*p++ = ']';
		}
		if (mp->state == MOD_UNINITIALIZED)
			q = "  (uninitialized)";
		else if (mp->state == MOD_RUNNING)
			q = "";
		else if (mp->state == MOD_DELETED)
			q = "  (deleted)";
		else
			q = "  (bad state)";
		while (*q)
			*p++ = *q++;
		*p++ = '\n';
	}
	return p - buf;
}

#define align_buffer(a,n) (((a)+(n)-1)  & ~((n)-1))

void *dmamalloc(unsigned int size, unsigned long *memory_start, int align)
{
#ifdef DEBUG2
	printk("dmallocing %x: %x (%xk)\n", *memory_start,size,size/1024);
#endif
	if (*memory_start)
	{	unsigned long addr, d=0;
		if (align)
			d = (align_buffer(*memory_start, align) - *memory_start);
		addr = *memory_start+d;
		(*memory_start) += size+d;
		return((void*)addr);
	}
#ifndef NORECOVER
	if (size>PAGE_SIZE)
#endif
		if (dma_segment && (dma_seg_size>=size))
		{	int n, d=0;
			if (align)
				d = (align_buffer(dma_segment, align)-dma_segment);
			n = dma_segment+d;
			dma_segment+=size+d;
			dma_seg_size-=size+d;
#ifdef DEBUG2
			printk("allocating %d dma bytes\n", size);
#endif
			return((void*)n);
		}
		else
		{	printk("malloc: out of dma memory (only %x of %x available)\n",dma_seg_size,size);
			return NULL;
		}
	return(kmalloc(size,GFP_KERNEL));
}

void *dmalloc(unsigned int size, unsigned long *memory_start)
{
	return dmamalloc(size, memory_start, 0);
}

void *dcalloc(unsigned int size, unsigned long *memory_start)
{
	void *p = dmalloc(size,memory_start);
	if (p) memset(p,0,size);
	return(p);
}

void dfree(void * ptr, unsigned int size, unsigned long *memory_start)
{	/* This needs work. */
#ifdef DEBUG2
	printk("dfreeint %x: %x (%xk)\n", *memory_start,size,size/1024);
#endif
#ifndef NORECOVER
	if (memory_start && *memory_start)
	{
		unsigned long n = (unsigned long)ptr;
		if ((n+size) == *memory_start)  /* Can not handle holes */
			(*memory_start) -= size;
	}
	else
		if (size<4096)
			kfree(ptr);
		else if (dma_segment)
			release_area((unsigned)ptr,size+(unsigned)ptr);
#endif
}

void dmafree(void * ptr, unsigned int size, unsigned long *memory_start)
{
	return dfree(ptr, size, memory_start);
}

int generic_register( register_item* item, register_item** list )
{	
	register_item* f = *list;
	if (!item)
		return(-EINVAL);
	if (!f)
	{	*list = item;
		item->next = NULL;
		return(0);
	}
	for (; f; f=f->next )	/* sanity check */
		if (!strcmp(f->name,item->name))
		{	
#ifdef DEBUG2
			printk("error: %s already registered\n", item->name);
#endif
			return(-EINVAL);
		}
	for (f = *list; f; f=f->next )
		if (!f->next)
		{	f->next=item;
			item->next = NULL;
			return(0);
		}
	return(-EINVAL);
}

int generic_unregister( char *name, register_item** list )
{
	register_item* fl = NULL, *f = *list;
	for (; f; fl=f, f=f->next )
		if (!strcmp(f->name,name))
		{	if (fl) fl->next=f->next;
			else *list=f->next;
			f->next = NULL;
			return(0);
		}
	printk("error: could not unregister %s\n", name);
	return(-EINVAL);
}


int do_verify_area(int type, const void * addr, unsigned long size)
{
	return do_verify_area(type, addr, size);
}

void mod_setup(char *str, int* ints)
{
	if ((ints[0] > 0) && (ints[1]>=0))
		dma_reserve_size=ints[1];
	if ((ints[0] > 1) && (ints[2]>=0))
		dma_release_at=ints[2];
	if (str && (!strcmp(str,"debug")))
		mod_verbose=1;
}
