/*
 * Dynamic loading of modules into the kernel.
 */

#ifndef _LINUX_MODULE_H
#define _LINUX_MODULE_H

#include <linux/../../tools/version.h>
#include <linux/linkage.h>
#define MODULE_ERROR(km) (km?km:0xffffffff)
#define MODULE_OK(km) (km?km:0)
#define FOLLOW_KERNEL 1     /* define so as to start right after kernel instead of at 2Meg mark. */
#define MOD_HEADER_SIZE	(2*4096)	/* Reserve 2 pages (enough for about 60 modules) */

/* values of module.state */
#define MOD_UNINITIALIZED 0
#define MOD_RUNNING 1
#define MOD_DELETED 2

/* maximum length of module name */
#define MOD_MAX_NAME 64

/* maximum length of symbol name */
#define SYM_MAX_NAME 60


struct mod_sym_entry {
	unsigned long addr;
	char *name;
};

struct module {
	struct module *next;
	char *name;
	int size;			/* size of module in pages */
	void* addr;			/* address of module */
	int state;
	void (*cleanup)(void);		/* cleanup routine */
	int	bootflag;		/* 1 if loading after boot. */
	struct mod_sym_entry *syms;
	int numsyms;
	char* dependancies;		/* list of modules using this one */
};


struct mod_routines {
					/* initialization routine */
	long (*init)(unsigned long, unsigned long);
	void (*cleanup)(void);		/* cleanup routine */
	void (*setup)(char *str, int args[11] );
	struct mod_sym_entry *stable;		/* symbol table (for layered modules) */
	int stable_size;
	char* dependancies;		/* list of modules using this one */
	int args[11];			/* args to init */
	unsigned long	bootflag;	/* 1 if loading after boot. */
};

struct kernel_sym {
	unsigned long value;		/* value of symbol */
	char name[SYM_MAX_NAME];	/* name of symbol */
};

struct mod_composite {			/* Used for bootloadable modules. */
	struct kernel_sym k;
	struct mod_routines m;
};

extern struct module *module_list;


/*
 * The first word of the module contains the use count.
 */
#define GET_USE_COUNT(module)	(* (int *) (module)->addr)
/*
 * define the count variable, and usage macros.
 */

extern int mod_use_count_;

#define MOD_INC_USE_COUNT      mod_use_count_++
#define MOD_DEC_USE_COUNT      mod_use_count_--
#define MOD_IN_USE	       (mod_use_count_ != 0)
#define MOD_SIGNATURE		"Load Modules"

/* Device memory allocation and free routines */
void *dmalloc(unsigned int, unsigned long *);
void *dcalloc(unsigned int, unsigned long *);
void dfree(void *, unsigned int, unsigned long *);
/* DMA memory is special.  It must be identity mapped and contiguous. */
void *dmamalloc(unsigned int, unsigned long *, int);
void dmafree(void *, unsigned int, unsigned long *);

	/* registration is getting to be so common, we abstract it. */
typedef struct resitem { char *name; struct resitem* next; } register_item; 
asmlinkage int generic_register( register_item* item, register_item** list );
asmlinkage int generic_unregister( char* name, register_item** list );

#endif
