/*
 *  linux/fs/proc/dev.c
 *
 *  Copyright (C) 1993 Peter MacDonald 
 *
 *  proc dev directory handling functions
 */
#include <linux/autoconf.h>

#include <asm/segment.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/module.h>
char procdev_version[] = UTS_RELEASE;

/* forward references */
static int proc_readdevdir(struct inode *, struct file *,
			   struct dirent *, int);
int proc_lookupdev(struct inode *,const char *,int,struct inode **);

struct file_operations *get_blkfops(unsigned int);
struct file_operations *get_chrfops(unsigned int);

extern struct inode_operations proc_dev_inode_operations;

struct file_operations proc_dev_operations = {
	NULL,			/* lseek - default */
	NULL,			/* read - bad */
	NULL,			/* write - bad */
	proc_readdevdir,	/* readdir */
	NULL,			/* select - default */
	NULL,			/* ioctl - default */
	NULL,			/* mmap */
	NULL,			/* no special open code */
	NULL,			/* no special release code */
	NULL			/* can't fsync */
};

static struct proc_dir_entry dev_dir[] = {
	{ 1,2,".." },
	{ 18,1,"." },
	{ 0,0 }
};

unsigned long procdev_init( unsigned long kmem )
{
	printk("loading procdev module\n");
	if (proc_dev_inode_operations.default_file_ops ||
	    proc_dev_inode_operations.lookup)
	{	printk("error: module already loaded?\n");
		return(MODULE_ERROR(kmem));
	}
	proc_dev_inode_operations.default_file_ops = &proc_dev_operations;
	proc_dev_inode_operations.lookup = proc_lookupdev;
	return(MODULE_OK(kmem));
}

void procdev_cleanup(void)
{
	if (MOD_IN_USE)
		printk("procdev: device busy release delayed\n");
	proc_dev_inode_operations.default_file_ops = NULL;
	proc_dev_inode_operations.lookup = NULL;
}

int proc_lookupdev(struct inode * dir,const char * name, int len,
	struct inode ** result)
{
#define doreturn(stat) {status=stat; goto proc_lookup_exit; }
	unsigned int ino;
	int i = 0, j=-1, k=0, n = 0, status=0;
	char buf[40];
	struct inode finode;

	MOD_INC_USE_COUNT;
	finode.i_rdev = -1;

	*result = NULL;
	if (!dir)
		doreturn(-ENOENT);
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		doreturn(-ENOENT);
	}
	if (!proc_match(len,name,dev_dir) && !proc_match(len,name,dev_dir+ ++i))
	{	i=2;
		n=-1;
		dev_dir[2].name=buf;
		dev_dir[2].low_ino=160;
		for (j=MAX_BLKDEV+MAX_CHRDEV-1; j >= 0; j--)
		{	struct file_operations *fops = 
				(j<MAX_BLKDEV?get_blkfops(j):get_chrfops(j-MAX_BLKDEV));
			if (j==(MAX_BLKDEV+TTY_MAJOR)) continue;
			k=-1;
			if (fops && fops->ioctl) 
			while (1) 
			{	
				buf[0] = ++k;
				if (k>255) break;
				if ((n = fops->ioctl(&finode,0,PROC_DEV_NAME,(int)buf)) >= 0)
				{	dev_dir[2].low_ino++;
					dev_dir[2].name=buf;
					dev_dir[2].namelen=strlen(buf);
				}
				if (-ENODEV != n) 
					if ((n < 0) || proc_match(len,name,dev_dir+2))
						break;
			} 
			if (n>=0) break;
		}
	}
	if (n < 0) {
		iput(dir);
		doreturn(-ENOENT);
	}
	ino = dev_dir[i].low_ino;
	if (!(*result = iget(dir->i_sb,ino))) {
		iput(dir);
		doreturn(-ENOENT);
	}
	if (i==2)
	{	(*result)->i_mode = ((j>=MAX_BLKDEV?S_IFCHR:S_IFBLK)|S_IRWXU);	
		(*result)->i_rdev = (k|((j>=MAX_BLKDEV?j-MAX_BLKDEV:j)<<8));
	}
	iput(dir);
	proc_lookup_exit:
#ifdef DEBUG
	printk("proc_lookupdev:  name=%s, return=%d, inode=%d, minor=%d\n", name, status, dev_dir[2].low_ino, k);
#endif
	MOD_DEC_USE_COUNT;
	return(status);
#undef doreturn
}

static int proc_readdevdir(struct inode * inode, struct file * filp,
	struct dirent * dirent, int count)
{
#define doreturn(stat) {status=stat; goto proc_read_exit; }
	struct proc_dir_entry * de = NULL;
	unsigned int ino;
	int i,j, k, n=-1, status=0;
	static int lastminor=-1;  /* Ok, this is bad, but where to store it? */
	static int lastmajor=MAX_BLKDEV+MAX_CHRDEV;
	static int lastino=160;
	char buf[40];
	struct inode finode;
	MOD_INC_USE_COUNT;
	finode.i_rdev = -1;

	if (!inode || !S_ISDIR(inode->i_mode))
		doreturn(-EBADF);
	ino = inode->i_ino;
	if (((unsigned) filp->f_pos) > 1)
	{	i=2;
		dev_dir[2].name=buf;
		dev_dir[2].low_ino=0;
		for (j=lastmajor; j >= 0; j--)
		{	struct file_operations *fops;
			int major=(j<MAX_BLKDEV?j:j-MAX_BLKDEV);
			if (j==(MAX_BLKDEV+TTY_MAJOR)) continue;
			if (j>=(MAX_BLKDEV+MAX_CHRDEV))
			{	lastminor=-1;
				continue;
			}
			fops = (j<MAX_BLKDEV?get_blkfops(major):get_chrfops(major));
			if (fops && fops->ioctl)
			{	for (k=lastminor+1, buf[0]=k; k<256; k++, buf[0]=k)
					if ((n = fops->ioctl(&finode,0,PROC_DEV_NAME,(int)buf)) >= 0)
					{	dev_dir[2].low_ino=lastino+1;
						dev_dir[2].namelen=strlen(buf);
						lastminor=k;
						break;
					}
					else if (n==-E2BIG)
					{	lastminor=-1;
						break;
					}
				if (k==256) lastminor=-1;
				lastmajor=j;
				if (n>=0) break;
			}
		}
		if (n>=0) 
		{	de = dev_dir + 2;
			lastino=dev_dir[2].low_ino;
		}
		if (j<0) 
		{	lastmajor=MAX_BLKDEV+MAX_CHRDEV;
			lastminor=-1;
			lastino=160;
		}
	}
	else
		de = dev_dir + filp->f_pos;
	if (de) {
		filp->f_pos++;
		i = de->namelen;
		ino = de->low_ino;
		put_fs_long(ino, &dirent->d_ino);
		put_fs_word(i,&dirent->d_reclen);
		put_fs_byte(0,i+dirent->d_name);
		j = i;
		while (i--)
			put_fs_byte(de->name[i], i+dirent->d_name);
		doreturn(j);
	}
	proc_read_exit:
#ifdef DEBUG
	printk("proc_readdevdir: name=%s, return=%d, inode=%d, minor=%d, f_pos=%d\n",
		buf, status, dev_dir[2].low_ino, k, filp->f_pos);
#endif
	MOD_DEC_USE_COUNT;
	return(status);
#undef doreturn
}
