/*
 *	isdn.c - Linux device driver for Diehl ISDN cards
 *
 *	Copyright (c) 1993
 *		Tilo Schuerer <tilo@cs.tu-berlin.de>
 *		Michael Riepe <riepe@ifwsn4.ifw.uni-hannover.de>
 *		Henrik Hempelmann <marsu@palumbia.in-berlin.de>
 *
 *	Parts of this code were derived from Diehl's original MS-DOS loader
 *	which is Copyright (c) Diehl Elektronik GmbH 1988-1991.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/tty.h>
#include <linux/signal.h>
#include <linux/malloc.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/module.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/irq.h>

#include <linux/diehl.h>
#include <linux/isdn.h>

char isdn_version[] = UTS_RELEASE;

#define ENABLE_DEBUG 

int d_level = DEBUG_ALL;				/* debug level */

int mal_buf=0;
int ind_buf=0;

/* PROTOTYPES */

/* ------------------------ ISDN-init-function ------------------------ */

unsigned long isdn_init(unsigned long kmem_start);
static void isdn_reset_channels(struct isdn_dev *dev);

/* ----------------- driver interface functions ----------------------- */

static int isdn_open(struct inode *inode, struct file *file);
static void isdn_release(struct inode *inode, struct file *file);
static int isdn_read(struct inode *inode, struct file *file, char *buffer, int count);
static int isdn_write(struct inode *inode, struct file *file, char *buffer, int count);
static int isdn_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int isdn_lseek(struct inode *inode, struct file *file, off_t offset, int origin);
static int isdn_select(struct inode *inode, struct file *file, int sel_type, select_table *wait);

struct file_operations isdn_fops = {
	isdn_lseek,		/* isdn_lseek	*/
	isdn_read,		/* isdn_read    */
	isdn_write,		/* isdn_write   */
	NULL, 			/* isdn_readdir */
	isdn_select,		/* isdn_select  */
	isdn_ioctl, 		/* isdn_ioctl   */
	NULL,			/* isdn_mmap    */
	isdn_open,		/* isdn_open    */
	isdn_release,		/* isdn_release */
	NULL			/* isdn_fsync	*/
};

/* ---- functions for connect and disconnect D3- and B2-channel ------ */

static int isdn_D3_assign(struct isdn_dev *dev, int minor);
static int isdn_D3_call(struct isdn_dev *dev, int minor);
static int isdn_D3_hangup(struct isdn_dev *dev, int minor);
static int isdn_D3_remove(struct isdn_dev *dev, int minor);

static int isdn_B2_assign(struct isdn_dev *dev, int minor);
static int isdn_B2_establish(struct isdn_dev *dev, int minor);
static int isdn_B2_release(struct isdn_dev *dev, int minor);
static int isdn_B2_remove(struct isdn_dev *dev, int minor);

/* ---- functions for reading and writing data on the B2-channel ----- */

static int isdn_B2_read(struct isdn_dev *dev, int minor, char *buffer, int count);
static int isdn_B2_write(struct isdn_dev *dev, int minor, char *buffer, int count);

/* ------------- function for installing ISDN-servers ---------------- */

static int isdn_listen_req(struct isdn_dev *dev, int minor, unsigned long timeout);
static int isdn_ind_req(struct isdn_dev *dev, int minor, unsigned long timeout);
static int isdn_accept_call(struct isdn_dev *dev, int minor);
static int isdn_reject_call(struct isdn_dev *dev, int minor);

/* ------------- various support functions ----------------*/

static int isdn_wait_ind(struct isdn_dev *dev, struct channel *chan, int ind, int wtime, struct isdn_buf **pbuf);
static int remove_ind(struct isdn_dev *dev, struct channel *chan, int Id);
static void clear_parameter(struct channel *chan);


/* ------------------ driver interface code -------------------------- */

static int
isdn_open(struct inode *inode, struct file *file)
/* do (almost) nothing, isdn-connection will be opened by ioctl's */
{
	struct isdn_dev *dev;
	int minor;

	PRINTD(DEBUG_FUNC,"isdn_open()\n");

	minor = MINOR(inode->i_rdev);
	if ((minor >> 5) >= NR_ISDN) {
		return -ENODEV;
	}
	dev = &isdn_devices[minor >> 5]; minor &= 31;
	if (!dev->present || minor >= dev->num_channels) {
		return -ENODEV;
	}
#if 0							/* We allow multiple open on one channel */
	if (dev->channels[minor].inuse) {		/* channel already in use */
		return -EBUSY;
	}
#endif
	dev->channels[minor].inuse++;
	return 0;
}

static void
isdn_release(struct inode *inode, struct file *file)
/* close filepointer for isdn-device */
{
	struct isdn_dev *dev;
	int minor;

	PRINTD(DEBUG_FUNC,"isdn_release()\n");

	minor = MINOR(inode->i_rdev);
	dev = &isdn_devices[minor >> 5]; minor &= 31;
	if (!dev->running) {
		/* nothing to do */
		return;
	}

	if (dev->channels[minor].inuse <= 1)	/* last open filepointer ?*/

	{			
		/* hangup (or try, at least ;-) */
		if (dev->channels[minor].B2_conn) 
			isdn_B2_release(dev, minor);
		if (dev->channels[minor].B2_Id) 
			isdn_B2_remove(dev, minor); 

		if (dev->channels[minor].D3_conn) 
			isdn_D3_hangup(dev, minor);
		if (dev->channels[minor].D3_Id) 
			isdn_D3_remove(dev, minor); 

		dev->channels[minor].inuse = 0;

	}
	else	/* keep connection open */
		dev->channels[minor].inuse--;
}

static int
isdn_read(struct inode * inode, struct file * file, 
char * buffer, int count)
{
	struct isdn_dev *dev;
	int minor;

	PRINTD(DEBUG_FUNC,"isdn_read()\n");

	minor = MINOR(inode->i_rdev);
	dev = &isdn_devices[minor >> 5]; minor &= 31;
	if (!dev->running) {
		PRINTD(DEBUG_ERR,"isdn_read(): No OS downloaded to ISDN-card!\n");
		return -EINVAL;
	}
	if (dev->channels[minor].B2_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_read(): Can't read, call not connected!\n");
		return -ENOTCONN;
	}
	return isdn_B2_read(dev, minor, buffer, count); 
}

static int
isdn_write(struct inode * inode, struct file * file, 
char * buffer, int count)
{
	struct isdn_dev *dev;
	int minor;

	PRINTD(DEBUG_FUNC,"w "); 

	minor = MINOR(inode->i_rdev);
	dev = &isdn_devices[minor >> 5]; minor &= 31;
	if (!dev->running) {
		PRINTD(DEBUG_ERR,"isdn_write(): No OS downloaded to ISDN-card!\n");
		return -EINVAL;
	}
	if (dev->channels[minor].B2_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_write(): Can't write, call not connected!\n");
		return -ENOTCONN;
	}
	return isdn_B2_write(dev, minor, buffer, count); 
}

static int
isdn_select(struct inode *inode, struct file *file, int sel_type, select_table *wait)
/* handle select call */
{
	struct channel *chan;
	struct isdn_buf *buf;
	struct isdn_dev *dev;
	int minor; 

	PRINTD(DEBUG_FUNC,"isdn_select()\n");

	minor = MINOR(inode->i_rdev);
	PRINTD(DEBUG_STAT,"minor: %d\n",minor);

	dev = &isdn_devices[minor >> 5]; minor &= 31;
	if (!dev->present) {
		return -ENODEV;
	}
	if (!dev->running) {
		PRINTD(DEBUG_ERR,"isdn_select(): No OS downloaded to ISDN-card!\n");
		return -EINVAL;
	}
	if (dev->channels[minor].B2_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_select(): Can't select, call not connected!\n");
		return -ENOTCONN;
	}

	if (sel_type == SEL_IN)   /* ready for read? */
	{
		chan = &dev->channels[minor];

		if (chan->backlog) {
			/* process indications */

			buf=chan->backlog;
			do{		/* look for DATA indications in backlogbuffer */
				if ((buf->rri_id == chan->B2_Id) && (buf->rri_code == LL_DATA)) {
					return 1; /* ready with data*/
				}
				else
					buf=buf->next;
			} while (buf != chan->backlog);
		}
		/* wait */
		select_wait(&chan->ind_wait,wait);
		return 0;
	}
	return 1; /* ready with data */
}


static int
isdn_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct isdn_load_req kbuf;
	struct isdn_dev *dev;
	int minor; 
	int ret;
	int i;

	PRINTD(DEBUG_FUNC,"isdn_ioctl()\n");

	minor = MINOR(inode->i_rdev);
	PRINTD(DEBUG_STAT,"minor: %d\n",minor);

	dev = &isdn_devices[minor >> 5]; minor &= 31;
	
	if (!dev->present) {
		return -ENODEV;
	}

	switch (cmd)
	{
#ifdef ENABLE_DEBUG
		case SET_DEBUG:
			d_level |= arg;
			printk("debug_level: %02x\n",d_level);
			break;

		case CLEAR_DEBUG:
			d_level &= ~arg;
			printk("debug_level: %02x\n",d_level);
			break;
#endif

		case ISDN_LOAD:
			PRINTD(DEBUG_FYI,"isdn_ioctl(): Loading OS to ISDN-card!\n");

			/* some parameter checks */
			if (!arg) {
				return -EINVAL;
			}
			ret = verify_area(VERIFY_READ, (void*)arg, sizeof(struct isdn_load_req));
			if (ret) {
				return ret;
			}

			/* copy data from user space */
			memcpy_fromfs(&kbuf, (struct isdn_load_req*)arg, sizeof(kbuf));

			if (!kbuf.data) {
				return -EFAULT;
			}
			ret = verify_area(VERIFY_READ, (void*)kbuf.data, kbuf.len);
			if (ret) {
				return ret;
			}
			ret = (*dev->loados)(dev, kbuf.data, kbuf.len);
			isdn_reset_channels(dev);
			return ret;

		case D3_CONN_REQ:   
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (dev->channels[minor].D3_Id==0)
				if ((ret = isdn_D3_assign(dev, minor)) < 0) {
					return ret;
				}

			remove_ind(dev,&dev->channels[minor],dev->channels[minor].D3_Id);		/* remove old indications */
			clear_parameter(&dev->channels[minor]);			/* clear indication parameters */
			
			return isdn_D3_call(dev, minor);

		case D3_DISC_REQ:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			
			return isdn_D3_hangup(dev, minor);

		case B2_CONN_REQ:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (dev->channels[minor].B2_Id==0)	/* make a new entity ? */
				if ((ret = isdn_B2_assign(dev, minor)) < 0) {
					return ret;
				}
				
			remove_ind(dev,&dev->channels[minor],dev->channels[minor].B2_Id);		 /* remove old indications */
			
			return isdn_B2_establish(dev, minor);

		case B2_DISC_REQ:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			return isdn_B2_release(dev, minor);

		case D3_LISTEN_REQ:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (dev->channels[minor].D3_Id==0)
				if ((ret = isdn_D3_assign(dev, minor)) < 0) {
					return ret;
				}

			remove_ind(dev,&dev->channels[minor],dev->channels[minor].D3_Id);		/* remove old indications */
			clear_parameter(&dev->channels[minor]);			/* clear indication parameters */

			return isdn_listen_req(dev, minor, arg);


		case D3_IND_REQ:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			
			if (dev->channels[minor].D3_Id==0)
				if ((ret = isdn_D3_assign(dev, minor)) < 0) {
					return ret;
				}

			remove_ind(dev,&dev->channels[minor],dev->channels[minor].D3_Id);		/* remove old indications */
			clear_parameter(&dev->channels[minor]);			/* clear indication parameters */

			return isdn_ind_req(dev, minor, arg);
			
		case D3_ACCEPT_CALL:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			
			return isdn_accept_call(dev, minor);

		case D3_REJECT_CALL:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}

			return isdn_reject_call(dev, minor);

		case NUMBER:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (!arg) {
				return -EFAULT;
			}

			dev->channels[minor].numlen = 0;
			for (i = 0; i < MAXNUMLEN; ++i) {
				if ((dev->channels[minor].number[i] = get_fs_byte((char*)arg + i)) == '\0')
				{
					dev->channels[minor].numlen = i;
					break;
				}
			}
			if (dev->channels[minor].numlen == 0) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): Invalid Number\n");
				return -EINVAL;
			}
			PRINTD(DEBUG_FYI,"ioctl(): number to dial: %s\n",dev->channels[minor].number);
			break;
		case SET_OAD:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (!arg) {
				return -EFAULT;
			}

			dev->channels[minor].oad = (u_char)arg;
			if (dev->channels[minor].oad > 9) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): Invalid OAD\n");
				return -EINVAL;
			}
			PRINTD(DEBUG_FYI,"ioctl(): OAD: %d\n",dev->channels[minor].oad);
			break;
		case GET_PARAMETER:
			if (!dev->running) {
				PRINTD(DEBUG_ERR,"isdn_ioctl(): No OS downloaded to ISDN-card!\n");
				return -EINVAL;
			}
			if (!arg) {
				return -EFAULT;
			}

			ret = verify_area(VERIFY_WRITE, (void*)arg, sizeof(struct info_data ));
			if (ret) {
				return ret;
			}

			memcpy_tofs((struct info_data*)arg,&dev->channels[minor].info, sizeof(dev->channels[minor].info));
			break;

		default:	
			PRINTD(DEBUG_ERR,"isdn_ioctl(): Unknown parameter!: %02x\n",cmd);
			return -ENOSYS;
	}
	return 0;
}

static int
isdn_lseek(struct inode *inode, struct file *file, off_t offset, int origin)
{
	/* since this is not a regular file, we can not seek */
	return -ESPIPE;
}

/* ------------------- ISDN-init-function ----------------------------- */

void isdn_cleanup(void)
{
	struct isdn_dev *dev;
	unregister_chrdev(MAJOR_NR,"isdn");
	for (dev = isdn_devices; dev < isdn_devices + NR_ISDN; ++dev)
		dfree( dev->channels,  dev->num_channels * sizeof(struct channel), 0);
}

unsigned long
isdn_init(unsigned long kmem_start)
{
	struct isdn_dev *dev;
	int n;

	PRINTD(DEBUG_FYI,"isdn_init(): Installing ISDN-Driver at major device %d\n", MAJOR_NR);

	if ((n=register_chrdev(MAJOR_NR,"isdn",&isdn_fops)))
	{
		printk("isdn_init(%d): Cannot register to major device %d\n", n, MAJOR_NR);
		return MODULE_ERROR(kmem_start);
	}

	/* initialize all devices */

	for (dev = isdn_devices; dev < isdn_devices + NR_ISDN; ++dev) {

		/* do presence test & basic initialization */
		if ((*dev->init)(dev) != 0) {
			PRINTD(DEBUG_ERR,"isdn_init(): failed to initialize device %d\n", dev - isdn_devices);
			continue;	/* ignore this one */
		}

		/* allocate memory for channels */
		dev->channels = (struct channel*)dmalloc( dev->num_channels * sizeof(struct channel),
			&kmem_start);

		isdn_reset_channels(dev);
	}


	return MODULE_OK(kmem_start);
}

static void
isdn_reset_channels(struct isdn_dev *dev)
/* initialize all channels for given device */
{
	struct channel *chan;

	if (!dev->channels) {
		return;
	}

	/* initialize channels */
	for (chan = dev->channels; chan < dev->channels + dev->num_channels; ++chan) {
		chan->ind_wait = NULL;
		chan->backlog = NULL;		/* possible mem-leak when called at runtime (ioctl:load) ?!? */
		chan->backlog_size = 0;
		chan->inuse = 0;
		chan->D3_conn = 0;
		chan->B2_conn = 0;
		chan->D3_Id = 0;
		chan->B2_Id = 0;
		chan->oad   = DEF_OAD;	/* default OAD */
		chan->numlen = 0;
		chan->number[0] = '\0';
		clear_parameter(chan);
	}
}

/*****************************************************************************/
/*                                                                           */
/*   The function isdn_D3_assign() and isdn_D3_call handles the              */
/*   ioctl()-D3_CONN_REQ                                                     */
/*                                                                           */
/*****************************************************************************/

static int
isdn_D3_assign(struct isdn_dev *dev, int minor)
{
	struct isdn_buf *buf;
	struct channel *chan;
	int i, ret;

	PRINTD(DEBUG_FUNC,"isdn_D3_assign()\n");

	chan = &dev->channels[minor];
	if (chan->D3_Id != 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_assign(): D3-entity already assigned!\n");
		return -EISCONN;
	}

	if ((buf = (struct isdn_buf*)kmalloc(sizeof(*buf) + 16, GFP_BUFFER)) == NULL) {
		return -ENOBUFS;
	}
	buf->buf_len = sizeof(*buf) + 16;
	buf->data = (u_char*)(buf + 1);

	i = 0;
	buf->data[i++] = CAI;            /* information element "call identity" */
	buf->data[i++] = 1;                   /* lenght of information element  */
	buf->data[i++] = 0x09;                    /* use transparent mode first */

	buf->data[i++] = CHI;   /* information element "channel identification" */
	buf->data[i++] = 1;                   /* lenght of information element  */
	buf->data[i++] = 0x80 + ((minor % N_SUB_CHAN)+1); /* use Bx-channel as party */

	buf->data[i++] = OAD;	     /* information element "originaton adress" */
	buf->data[i++] = 2;                   /* lenght of information element  */
	buf->data[i++] = 0xff;                                   /* adress type */
	buf->data[i++] = 0x30 + chan->oad;	                   /* my number */

	buf->data[i++] = 0x96;                  /* shift to codeset 6 (locking) */

	buf->data[i++] = SIN;         /* information element "service indicator */
	buf->data[i++] = 2;                   /* lenght of information element  */
	buf->data[i++] = 0x01;                                 /* use telephony */
	buf->data[i++] = 0x01;                                  /* ISDN 3.1 KHz */

	buf->data_len = i;                        /* total length of parameters */
	buf->rri_id = DSIG_ID;            /* set D-channel signaling entity ID  */
	buf->rri_code = ASSIGN_SIG;                           /* ASSIGN-Request */

	/* pass buffer to the device */
	ret = (*dev->send)(dev, buf);

	/* check result */
	if (ret == 0) {
		if (buf->rri_code == ASSIGN_OK) {
			PRINTD(DEBUG_FYI,"isdn_D3_assign(): ASSIGN_SIG-request ok, got D3_Id %02x\n",buf->rri_id);
			chan->D3_Id = buf->rri_id;
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_D3_assign(): ASSIGN_SIG request failed (err = %02x)\n",buf->rri_code);
			ret = -EIO;
		}
	}

	/* free request buffer */
	kfree_s(buf, buf->buf_len);
	return ret;
}

static int
isdn_D3_call(struct isdn_dev *dev, int minor)
{
	struct isdn_buf *buf;
	struct channel *chan;
	int i, ret;

	PRINTD(DEBUG_FUNC,"isdn_D3_call()\n");

	chan = &dev->channels[minor];
	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_call(): No D3-entity assigned!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_D3_call(): Already connected\n");
		return -EISCONN;
	}
	if (chan->numlen == 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_call(): No Number to dial!\n");
		return -EDESTADDRREQ;
	}
	
	if ((buf = (struct isdn_buf*)kmalloc(sizeof(*buf) + 4 + MAXNUMLEN, GFP_BUFFER)) == NULL) {
		return -ENOBUFS;
	}

	buf->buf_len = sizeof(*buf) + 4 + MAXNUMLEN;
	buf->data = (u_char*)(buf + 1);

	i = 0;
	buf->data[i++] = CPN;         /* information element "called party number" */
	buf->data[i++] = chan->numlen + 1;       /* lenght of information element  */
	buf->data[i++] = 0xff;                               /* use default coding */
	memcpy(&buf->data[i], chan->number, chan->numlen); /* copy number to sendbuffer */
	i += chan->numlen;

	buf->data_len = i;
	buf->rri_id = chan->D3_Id;
	buf->rri_code = CALL_REQ;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, buf);

	/* check result */
	if (ret == 0) {
		if (buf->rri_code == OK) {
			struct isdn_buf *indbuf;

			PRINTD(DEBUG_FYI,"isdn_D3_call(): CALL_REQ request ok\n");

			/* wait for CALL_CON indication */

			ret = isdn_wait_ind(dev, chan, CALL_CON, 10000, &indbuf);
			if (ret == 0) {
				PRINTD(DEBUG_FYI,"isdn_D3_call(): got CALL_CON indication - connected\n");
				kfree_s(indbuf, indbuf->buf_len);	/* free indication buffer */
				chan->D3_conn = 1;
			}
			else
			{
				PRINTD(DEBUG_ERR,"isdn_D3_call(): wait CALL_CON failed (err = %02x)\n",ret);
				return ret;
			}

		}
		else {
			PRINTD(DEBUG_ERR,"isdn_D3_call(): CALL_REQ request failed (err = %02x)\n",buf->rri_code);
			ret = -EIO;
		}
	}
	
	/* free request buffer */
	kfree_s(buf, buf->buf_len);
	return ret;
}

/*****************************************************************************/
/*                                                                           */
/*   The function isdn_D3_hangup() handles the ioctl()-D3_DISC_REQ           */
/*                                                                           */
/*****************************************************************************/

static int
isdn_D3_hangup(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	PRINTD(DEBUG_FUNC,"isdn_D3_hangup()\n");

	chan = &dev->channels[minor];
	if (chan->D3_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_hangup(): D3-Layer already disconnected!\n");
		return -ENOTCONN;
	}
	if (chan->B2_conn != 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_hangup(): Disconnect B-Channel first!\n");
		return -EBUSY;
	}

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = HANGUP;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);

	/* check result */
	if (ret == 0) {
		if (buf.rri_code == OK) {
			struct isdn_buf *indbuf;

			PRINTD(DEBUG_FYI,"isdn_D3_hangup(): request ok\n");

			/* wait for HANGUP indication */

			ret = isdn_wait_ind(dev, chan, HANGUP, 1000, &indbuf);
			if (ret == 0) {
				PRINTD(DEBUG_FYI,"isdn_D3_hangup(): got HANGUP indication - disconnected\n");
				kfree_s(indbuf, indbuf->buf_len);	/* free indication buffer */
				chan->D3_conn = 0;
			}
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_D3_hangup(): HANGUP request failed (err = %02x)\n",buf.rri_code);
			ret = -EIO;
		}
	}

	return ret;
}

static int
isdn_D3_remove(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	PRINTD(DEBUG_FUNC,"isdn_D3_remove()\n");

	chan = &dev->channels[minor];
	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_D3_remove(): D3-Entity already removed!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_D3_remove(): Disconnect D-Channel first!\n");
		return -EBUSY;
	}

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = REMOVE_SIG;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);

	/* check result */
	if (ret == 0) {
		if (buf.rri_code == OK) {
			PRINTD(DEBUG_FYI,"isdn_D3_remove(): REMOVE ok\n");
			remove_ind(dev,chan,chan->D3_Id);		 /* remove old indications */
			chan->D3_Id = 0;
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_D3_remove(): REMOVE_SIG request failed (err = %02x)\n",buf.rri_code);
			ret = -EIO;
		}
	}

	return ret;
}

/*****************************************************************************/
/*                                                                           */
/*   The function isdn_B2_assign() handles the ioctl()-B2_CONN_REQ          */
/*                                                                           */
/*****************************************************************************/

static int
isdn_B2_assign(struct isdn_dev *dev, int minor)
{
	struct isdn_buf *buf;
	struct channel *chan;
	int i, ret;

	PRINTD(DEBUG_FUNC,"isdn_B2_assign()\n");

	chan = &dev->channels[minor];
	if (chan->D3_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_B2_assign(): Connect D3-Layer first!\n");
		return -EINVAL;
	}
	if (chan->B2_Id != 0) {
		PRINTD(DEBUG_ERR,"isdn_B2_assign(): B-Channel already assigned!\n");
		return -EISCONN;
	}

	if ((buf = (struct isdn_buf*)kmalloc(sizeof(*buf) + 8, GFP_BUFFER)) == NULL) {
		return -ENOBUFS;
	}

	buf->buf_len = sizeof(*buf) + 8;
	buf->data = (u_char*)(buf + 1);

	i = 0;
	buf->data[i++] = CAI;            /* information element "call identity" */
	buf->data[i++] = 1;                   /* lenght of information element  */
	buf->data[i++] = chan->D3_Id;      /* corresponding D3-signaling entity */

	buf->data[i++] = LLC;  /* information element "low layer compatibility" */
	buf->data[i++] = 3;                   /* lenght of information element  */
	buf->data[i++] = 0;                                         /* reserved */
	buf->data[i++] = 0;                                         /* reserved */
	buf->data[i++] = 0xc1;                 /* transparent with HDLC-framing */

	buf->data_len = i;                        /* total length of parameters */
	buf->rri_id = BLLC_ID;            /* Id for B-channel link level access */
	buf->rri_code = ASSIGN_LL;                            /* ASSIGN-Request */

	/* pass buffer to the device */
	ret = (*dev->send)(dev, buf);

	/* check result */
	if (ret == 0) {
		if (buf->rri_code == ASSIGN_OK) {
			PRINTD(DEBUG_FYI,"isdn_B2_assign(): ASSIGN_LL-request ok, got B2_Id %02x\n",buf->rri_id);
			chan->B2_Id = buf->rri_id;
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_B2_assign(): ASSIGN_LL request failed (err = %02x)\n",buf->rri_code);
			ret = -EIO;
		}
	}

	/* free request buffer */
	kfree_s(buf, buf->buf_len);
	remove_ind(dev,chan,chan->B2_Id);		 /* remove old indications */
	return ret;
}

static int
isdn_B2_establish(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	PRINTD(DEBUG_FUNC,"isdn_B2_establish()\n");

	chan = &dev->channels[minor];
	if (chan->B2_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_B2_establish(): No B2-entity assigned!\n");
		return -EINVAL;
	}

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->B2_Id;
	buf.rri_code = LL_ESTABLISH;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);

	if (ret == 0) {
		if (buf.rri_code == OK) {
			struct isdn_buf *indbuf;

			PRINTD(DEBUG_FYI,"isdn_B2_establish(): request ok\n");

			/* now wait for LL_ESTABLISH indication */

			ret = isdn_wait_ind(dev, chan, LL_ESTABLISH, 3000, &indbuf);
			if (ret == 0) {
				PRINTD(DEBUG_FYI,"isdn_B2_establish(): connection established\n");
				kfree_s(indbuf, indbuf->buf_len);	/* free indication buffer */
				chan->B2_conn = 1;
			}

		}
		else {
			PRINTD(DEBUG_ERR,"isdn_B2_establish(): LL_ESTABLISH request failed (err = %02x)\n",buf.rri_code);
			ret = -EIO;
		}
	}

	return ret;
}

/*****************************************************************************/
/*                                                                           */
/*   The function isdn_B2_release() handles the ioctl()-B2_DISC_REQ          */
/*                                                                           */
/*****************************************************************************/

static int
isdn_B2_release(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	PRINTD(DEBUG_FUNC,"isdn_B2_release()\n");

	chan = &dev->channels[minor];
	if (chan->B2_conn == 0) {
		PRINTD(DEBUG_ERR,"isdn_B2_release(): B2-Layer already disconnected!\n");
		return -ENOTCONN;
	}

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->B2_Id;
	buf.rri_code = LL_RELEASE;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);

	/* check result */
	if (ret == 0) {
		if (buf.rri_code == OK) {
			struct isdn_buf *indbuf;

			PRINTD(DEBUG_FYI,"isdn_B2_release(): request ok\n");

			/* now wait for LL_RELEASE indication */

			ret = isdn_wait_ind(dev, chan, LL_RELEASE, 1000, &indbuf);

			if (ret == 0) {
				PRINTD(DEBUG_FYI,"isdn_B2_release(): connection released\n");
				kfree_s(indbuf, indbuf->buf_len);	/* free indication buffer */
				chan->B2_conn = 0;
			}
			else
				PRINTD(DEBUG_ERR,"isdn_B2_release(): waiting for LL_RELEASE err-:%x\n",-ret);
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_B2_release(): LL_RELEASE request failed (err = %02x)\n",buf.rri_code);
			ret = -EIO;
		}
	}

	return ret;
}

static int
isdn_B2_remove(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	PRINTD(DEBUG_FUNC,"isdn_B2_remove()\n");

	chan = &dev->channels[minor];
	if (chan->B2_Id == 0) {
		PRINTD(0,"isdn_B2_remove(): B2-Entity already removed!\n");
		return -EINVAL;
	}
	if (chan->B2_conn) {
		PRINTD(0,"isdn_B2_remove(): release B2-Layer first!\n");
		return -EBUSY;
	}

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->B2_Id;
	buf.rri_code = REMOVE_LL;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);

	/* check result */
	if (ret == 0) {
		if (buf.rri_code == OK) {
			PRINTD(DEBUG_FYI,"isdn_B2_remove(): REMOVE ok\n");
			remove_ind(dev,chan,chan->B2_Id);		 /* remove old indications */
			chan->B2_Id = 0;
		}
		else {
			PRINTD(DEBUG_ERR,"isdn_B2_remove(): REMOVE_LL request failed (err = %02x)\n",buf.rri_code);
			ret = -EIO;
		}
	}

	return ret;
}

/*****************************************************************************/
/*                                                                           */
/* The function isdn_B2_read() handles the read-operation on the ISDN-device */
/*                                                                           */
/*****************************************************************************/

static int
isdn_B2_read(struct isdn_dev *dev, int minor, char *buffer, int count)
{
	struct isdn_buf *buf;
	struct channel *chan;
	int bytes_read;
	u_long flags;
	int ret;

	chan = &dev->channels[minor];

	for (bytes_read = 0; bytes_read < count; ) {

		/* wait for LL_DATA indications */
		/* we should not time out here, should we? */

		ret = isdn_wait_ind(dev, chan, LL_DATA, 10, &buf);
		if (ret) {
			if (ret == -EINTR && bytes_read) {
				/* interrupted - return what we read so far */
				return bytes_read;
			}
			return ret;
		}

		if (buf->data_len) {
			int len = buf->data_len;

			if (bytes_read + len > count) {	/* we got too much... */
				len = count - bytes_read;
				memcpy_tofs(buffer + bytes_read, buf->data, len);

				/* put the rest back into the queue */

#if 0
				memcpy(buf->data, buf->data + len, buf->data_len -= len);
#else
				/* dirty trick to avoid copying :-) (from BSD's mbuf code) */
				buf->data += len;
				buf->data_len -= len;
#endif
				save_flags(flags);
				cli();
				if (chan->backlog) {	/* queue back in */
					buf->next = chan->backlog;
					buf->prev = chan->backlog->prev;
					buf->prev->next = buf;
					buf->next->prev = buf;
				}
				else {
					buf->next = buf;
					buf->prev = buf;
				}
				chan->backlog = buf;
				chan->backlog_size++;
				restore_flags(flags);

				/* all done */
				return count;
			}
			memcpy_tofs(buffer + bytes_read, buf->data, len);
			bytes_read += len;
		}
		kfree_s(buf, buf->buf_len);
	}

	return bytes_read;
}

/*****************************************************************************/
/*                                                                           */
/*The function isdn_B2_write() handles the write-operation on the ISDN-device*/
/*                                                                           */
/*****************************************************************************/

static int
isdn_B2_write(struct isdn_dev *dev, int minor, char *buffer, int count)
{
	struct isdn_buf *buf;
	struct channel *chan;
	int bytes_written;

	chan = &dev->channels[minor];

	if ((buf = (struct isdn_buf*)kmalloc(sizeof(*buf) + 270, GFP_BUFFER)) == NULL) {
		return -ENOBUFS;
	}

	buf->buf_len = sizeof(*buf) + 270;
	buf->data = (u_char*)(buf + 1);

	/* --------- send all packets in LL_DATA-requests to card ------------------ */

	for (bytes_written = 0; bytes_written < count; ) {
		int len = count - bytes_written;
		int ret;

		if (chan->B2_conn == 0) {
			PRINTD(DEBUG_ERR,"isdn_B2_write(): Can't write, call not connected!\n");
			kfree_s(buf, buf->buf_len);
			return -ENOTCONN;
		}
		if (len > 269) {
			len = 269;		/* (*dev->send)() adds one byte! */
		}

		memcpy_fromfs(buf->data, buffer + bytes_written, len);

		buf->data_len = len;
		buf->rri_id = chan->B2_Id;
		buf->rri_code = LL_DATA;

		/* pass buffer to the device */
		ret = (*dev->send)(dev, buf);

		/* check result */
		if (ret) {
			kfree_s(buf, buf->buf_len);
			return ret;
		}
		if (buf->rri_code != OK) {
			PRINTD(DEBUG_ERR,"isdn_B2_write(): LL_DATA request failed (err = %02x)\n",buf->rri_code);
			kfree_s(buf, buf->buf_len);
			return -EIO;
		}
		bytes_written += len;
	}

	/* PRINTD(0,"isdn_B2_write(): Successfully sent %d bytes!\n", bytes_written); */

	kfree_s(buf, buf->buf_len);
	return bytes_written;
}

/*****************************************************************************/
/*                                                                           */
/*         The function isdn_listen_req() handles the LISTEN_REQ-ioctl()     */
/*                                                                           */
/*****************************************************************************/

static int
isdn_listen_req(struct isdn_dev *dev, int minor, unsigned long timeout)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct isdn_buf *indbuf;
	struct channel *chan;
	int ret;

	chan = &dev->channels[minor];

	PRINTD(DEBUG_FUNC,"isdn_listen_request()\n");

	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_listen_req(): No D3-entity assigned!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_listen_req(): Already connected\n");
		return -EISCONN;
	}

	/*-------------- send an LISTEN_REQ-request to the card -------------------*/

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = LISTEN_REQ;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);
	if (ret) {
		return ret;
	}
	if (buf.rri_code != OK) {
		PRINTD(DEBUG_ERR,"isdn_listen_req(): LISTEN_REQ request failed (err = %02x)\n",buf.rri_code);
		return -EIO;
	}

	/* ------ now wait until >timeout< jiffies for any incoming call ---------- */

	ret = isdn_wait_ind(dev, chan, CALL_IND, timeout, &indbuf);
	if (ret) {
		return ret;
	}
	/* free indication buffer */
	kfree_s(indbuf, indbuf->buf_len);

	chan->D3_conn = 1;
	
	return ret;
}

static int
isdn_ind_req(struct isdn_dev *dev, int minor, unsigned long timeout)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct isdn_buf *indbuf;
	struct channel *chan;
	int ret;

	chan = &dev->channels[minor];
	PRINTD(DEBUG_FUNC,"isdn_ind_req()\n");

	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_ind_req(): No D3-entity assigned!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_ind_req(): Already connected\n");
		return -EISCONN;
	}

	/*-------------- send an INDICATE_REQ-request to the card -------------------*/

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = INDICATE_REQ;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);
	if (ret) {
		return ret;
	}
	if (buf.rri_code != OK) {
		PRINTD(DEBUG_ERR,"isdn_ind_req(): INDICATE_REQ request failed (err = %02x)\n",buf.rri_code);
		return -EIO;
	}
	PRINTD(DEBUG_FYI,"isdn_ind_req(): INDICATE_REQ ok \n");

	/* ------ now wait until >timeout< jiffies for any incoming call ---------- */

	ret = isdn_wait_ind(dev, chan, INDICATE_IND, timeout, &indbuf);
	if (ret) {
		return ret;
	}
	/* free indication buffer */
	kfree_s(indbuf, indbuf->buf_len);

	return ret;
}

static int
isdn_accept_call(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct isdn_buf *indbuf;
	struct channel *chan;
	int ret;

	chan = &dev->channels[minor];

	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_accept_call(): No D3-entity assigned!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_accept_call(): Already connected\n");
		return -EISCONN;
	}

	/*-------------- send an CALL_RES-request to the card -------------------*/

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = CALL_RES;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);
	if (ret) {
		return ret;
	}
	if (buf.rri_code != OK) {
		PRINTD(DEBUG_ERR,"isdn_accept_call(): CALL_RES request failed (err = %02x)\n",buf.rri_code);
		return -EIO;
	}

	/* ------ now wait until >timeout< jiffies for any incoming call ---------- */

	ret = isdn_wait_ind(dev, chan, CALL_IND, 1000, &indbuf);
	if (ret) {
		return ret;
	}

	/* free indication buffer */
	kfree_s(indbuf, indbuf->buf_len);

	chan->D3_conn = 1;	/* Connected ! */

	return ret;
}

static int isdn_reject_call(struct isdn_dev *dev, int minor)
{
	struct isdn_buf buf;	/* no parameters == no need for kmalloc() */
	struct channel *chan;
	int ret;

	chan = &dev->channels[minor];

	if (chan->D3_Id == 0) {
		PRINTD(DEBUG_ERR,"isdn_reject_call(): No D3-entity assigned!\n");
		return -EINVAL;
	}
	if (chan->D3_conn) {
		PRINTD(DEBUG_ERR,"isdn_reject_call(): Already connected\n");
		return -EISCONN;
	}

	/*-------------- send an REJECT-request to the card -------------------*/

	buf.buf_len = sizeof(buf);
	buf.data = (u_char*)(&buf + 1);

	buf.data_len = 0;
	buf.rri_id = chan->D3_Id;
	buf.rri_code = REJECT;

	/* pass buffer to the device */
	ret = (*dev->send)(dev, &buf);
	if (ret) {
		return ret;
	}
	if (buf.rri_code != OK) {
		PRINTD(DEBUG_ERR,"isdn_reject_call(): REJECT request failed (err = %02x)\n",buf.rri_code);
		return -EIO;
	}
	return ret;

}

static int
isdn_wait_ind(struct isdn_dev *dev, struct channel *chan, int ind, int wtime, struct isdn_buf **pbuf)
{
	struct isdn_buf *buf;
	u_long flags;

	if (wtime) {
		current->timeout = jiffies + wtime;
	}
	do {

		if (chan->backlog) {

			/* process indications */

			save_flags(flags);
			cli();
			while ((buf = chan->backlog) != NULL) {

				/* remove buffer from queue */
				if (buf->next == buf) {
					chan->backlog = NULL;
				}
				else {
					chan->backlog = buf->next;
					buf->prev->next = buf->next;
					buf->next->prev = buf->prev;
				}
				chan->backlog_size--;

				/* is this what we're looking for? */
				if (buf->rri_code == ind) {		/* gotcha! */

					current->timeout = 0;	/* do not time out */
					restore_flags(flags);

					/* pass buffer to caller */

					*pbuf = buf;
					return 0;
				}

				/* do default processing */
				if (buf->rri_id == chan->D3_Id) {
					switch (buf->rri_code) {
						case HANGUP:	/* our peer doesn't like us any more ;-) */

							current->timeout = 0;	/* do not time out */
							restore_flags(flags);
							kfree_s(buf, buf->buf_len);

							PRINTD(DEBUG_FYI,"isdn_wait_ind(): peer sent HANGUP\n");

							chan->D3_conn=0;
							return -ECONNRESET;

						default:
							/*
							 * TODO: handle other indications
							 */
							break;
					}
				}
				else {	/* B2 indication */
					switch (buf->rri_code) {
						case LL_RELEASE:	/* our peer doesn't like us any more ;-) */

							current->timeout = 0;	/* do not time out */
							restore_flags(flags);
							kfree_s(buf, buf->buf_len);

							PRINTD(DEBUG_FYI,"isdn_wait_ind(): peer sent LL_RELEASE\n");

							chan->B2_conn=0;
							return -ECONNRESET;

						default:
							/*
							 * TODO: handle other indications
							 */
							break;
					}
				}

				/* free buffer */
				kfree_s(buf, buf->buf_len);
			}
			restore_flags(flags);
		}

		/* there are no (more) indications - go to bed... */

		interruptible_sleep_on(&chan->ind_wait);
		if (current->signal & ~current->blocked) {
			/* interrupted system call */
			return -EINTR;
		}
	}
	while (!wtime || current->timeout);

	/* wait timed out */

	PRINTD(DEBUG_ERR,"isdn_wait_ind(): timed out\n");
	return -ETIMEDOUT;
}

static void
clear_parameter(struct channel *chan)
{
	chan->info.bc.length=0;
	chan->info.cai.length=0;
	chan->info.cau.length=0;
	chan->info.chi.length=0;
	chan->info.cif.length=0;
	chan->info.cl.length=0;
	chan->info.cpn.length=0;
	chan->info.cps.length=0;
	chan->info.date.length=0;
	chan->info.dsa.length=0;
	chan->info.esccau.length=0;
	chan->info.escstat.length=0;
	chan->info.hlc.length=0;
	chan->info.llc.length=0;
	chan->info.oad.length=0;
	chan->info.osa.length=0;
	chan->info.sin.length=0;
	chan->info.uui.length=0;
}

static int
remove_ind(struct isdn_dev *dev, struct channel *chan, int Id)
{
	struct isdn_buf *buf,*new_buf;
	int term=FALSE;
	u_long flags;

	
	PRINTD(DEBUG_FUNC,"isdn: remove_ind()\n");

		if (chan->backlog) {

			/* process indications */

			save_flags(flags);
			cli();
			buf=chan->backlog;

			do {
				/* is this what we're looking for? */
				if (buf->rri_id == Id) {		/* gotcha! */
					/* remove buffer from queue */
					if (buf->next == buf) {   /* only one */
						chan->backlog = NULL;
						kfree_s(buf, buf->buf_len);
						chan->backlog_size--;
						term=TRUE;
					}
					else
					{
						if (buf == chan->backlog)
						{
							chan->backlog=buf->next;
						}
						else
							if (buf->next == chan->backlog)
								term=TRUE;
						new_buf = buf->next;
						buf->prev->next = buf->next;
						buf->next->prev = buf->prev;
						kfree_s(buf, buf->buf_len);
						chan->backlog_size--;
						buf = new_buf;
					}
				}
				else 
				{
					buf = buf->next;
					if (buf == chan->backlog)
						term=TRUE;
				}

			} while (!term);
			restore_flags(flags);

		}
	return 0;
}
