/* /linux/drivers/block/lzw.c
 *
 * Written by Jean-Marc Verbavatz 12 Dec 1993 <jmv@receptor.mgh.harvard.edu>
 *
 * 15 Feb 1994:
 *	String repeat feature added (RLE).
 */

/* LZW Compression and Decompression:
	Functions:
		RLZW(si, so, len)	Decompression
		code *si; compressed data
		byte *so; buffer for uncompressed data
		int  len; expected length of uncompressed data
	
		WLZW(si, so, len, maxlen)	Coding
		byte *si; raw data
		code *so; buffer for compressed data
		int  len; length of raw data
		int maxlen; maximum size of compressed data
		(so buffer must be slightly longer than maxlen)
   	Return code:
		 0 = Worthless CPU waste (No compression)
		-1 = General error
		-2 = Logical error
		-3 = Expand error
		>0 = OK/total length
*/
#include <linux/string.h>

#define LZSIZE 4096	/* Should stay < 32768 */ 
#define CLEAR 256	/* Clear Code */
#define REPEAT 257
#define START 258

typedef unsigned short code;
typedef unsigned char byte;

static code Prefix, Prefix0, Index;
static int code_len, new_entry;		/* For repeated strings */
static int bits, off, size;
static code *scode;
static byte *sbyte;
static unsigned int pcode, pbyte, length;

code *PTable, *NTable;
byte *CTable;

/* Write Next Code */
static int WCode(code wcode)
{
int todo;

pcode+=bits;
/* Quickly check output buffer size */
if((pcode>>3) > length) return 0; /* Insufficient compression achieved */

if((todo = bits+off-16)>=0) {
	*scode++ |= wcode>>todo;
	*scode = wcode<<(16-todo);
	off = todo;
} else {
	*scode |= wcode<<(-todo);
	off += bits;
}
return 1;
}

/* Read Next Code */
static short RCode(void)
{
code rcode; /* 15 bits maximum, with LZSIZE=32768; never negative */
int todo;

if((todo = bits+off-16)>=0) {
	rcode = (*scode++)<<todo;
	rcode |= (*scode)>>(16-todo);
	off = todo;
} else {
	rcode = (*scode)>>(-todo);
	off += bits;
}
rcode&=size-1;
return (short) rcode;
}

/* Initialization (R/W) */
static void InitTable(void)
{
bits = 8;
size = 256;
pbyte = pcode = off = 0;
Index = START;
memset(PTable, -1, LZSIZE*sizeof(code));

}

/* Initialization for decompression */
static void RInitTable(void)
{
int i;
byte *p;

InitTable();
for(i = 0, p = CTable; i < CLEAR; i++)
	*p++ = i;
}

/* Lookup Table */
static code LookUp(byte Car)
{
code pi;

pi = PTable[Prefix];
while(((short)pi) != -1) {
	if(CTable[pi] == Car) return pi;
	pi = NTable[pi];
}
return -1; /* No Hit */
}

/* byte is added to table and becomes prefix */
static int WAddPrefix(void)
{
code pi;

pi = PTable[Prefix];
PTable[Prefix] = Index;		/* Next entry */
NTable[Index] = pi;
CTable[Index++] = *sbyte;

if(Index == LZSIZE) {		/* Table full */
	if(!WCode(CLEAR)) return 0; 
	Index = START;
	memset(PTable, -1, LZSIZE*sizeof(code));
	bits = 8;
	size = 256;
} else if(Index > size) {
	bits++;
	size <<= 1;
}
return 1;
}

static int Expand(code val)
{
code p,q,r;

if(val > Index)
	return -2;
q = -1;
PTable[Index] = Prefix;
do {
	p = val;
	while((p = PTable[(r = p)]) != q); /* Get previous */
	*sbyte++ = CTable[r];
	if(++pbyte >= length) return 0; /* Done; skip useless stuff */
	if(((short)q) == -1 && new_entry) {
		if(Index == LZSIZE) return -3;
		CTable[Index++] = CTable[r];
		new_entry = 0;
	}
	q = r;
} while(q != val);
Prefix = val;
return 1;
}

int check_repeat(int len)
{
int n = 0;
char *s, *s0;

s = sbyte;		/* Current string */
s0 = s - code_len;	/* Reference string */

while(code_len <= len && (n+1) < size) {
	if(memcmp(s0, s, code_len)) break;
	len -= code_len;	/* Number of bytes left */
	n++;			/* Number of repeats */
	s += code_len;
}
return n;
}

/* Write Lzw (compression) */
int WLZW(byte *si, code *so, int len, int maxlen)
{
code val;
int Repeat;

scode = so;
*scode = 0;
sbyte = si;
length = maxlen;
InitTable();

Prefix = Prefix0 = *sbyte;
code_len = 0;
while(++pbyte<len) {
	code_len++;	/* Length of current string */
	if(*(++sbyte) == Prefix0 && code_len > 1) { /* at leat 2 bytes */
/* Check for string repeat. If positive, we are going to write 3 codes.
   Therefore, we need to filter small repeats */
		Repeat = check_repeat(len-pbyte); /* Could do better */
		if(Repeat > 3) {	/* At least 4 times (1+>3) */
			WCode(REPEAT);
			WCode(Repeat);
			if(!WCode(Prefix)) return 0;
			pbyte += code_len*Repeat; /* Total length of repeat */
			if(pbyte >= len) break;
			sbyte += code_len*Repeat; /* position of next byte */
		}
	}
	if(((short)(val=LookUp(*sbyte)))==-1) {	/* Break in sequence */
		if(!WCode(Prefix)) return 0;	/* Write to buffer */
		if(!WAddPrefix()) return 0;	/* Car is new prefix */
		Prefix = Prefix0 = *sbyte;
		code_len = 0;
	} else Prefix = val;
}
if(!WCode(Prefix)) return 0;

len = pcode/sizeof(code)/8;
if(pcode&(sizeof(code)*8-1)) len++;
len *= sizeof(code);
if(len > maxlen) return 0;
return len;
}

/* CLEAR Code Read */
static int RClear(void)
{

bits = 8;
size = 256;
Prefix = RCode();
*sbyte++ = Prefix;
if(++pbyte >= length) return 0;
new_entry = 1;
Index = START;
bits++;
size = 512;
return 1;
}

/* Read Lzw (decompression) */
int RLZW(code *si, byte *so, int len)
{
code val;
int n;

sbyte = so;
scode = si;
length = len;
RInitTable();

RClear();
for(;;) {
	val = RCode();
	if(val == CLEAR) {
		if(!RClear()) break;
		continue;
	}
	if(val == REPEAT) {
		len = RCode();	/* # of repeats */
		val = RCode();	/* code to expand */
		do
			n = Expand(val);
		while(--len && n > 0);
	} else {
		n = Expand(val);
		if(Index >= size) {
			bits++;
			size <<= 1;
		}
		new_entry = 1;	/* Add new entry at next code */
	}
	if(n < 0) return n;
	if(n == 0) break;
} 
return 1;
}

int iLZW(void *wrk_mem)
{
void *mem0;

mem0 = wrk_mem;
PTable = (code *)wrk_mem;
wrk_mem += LZSIZE*sizeof(code);
NTable = (code *)wrk_mem;
wrk_mem += LZSIZE*sizeof(code);
CTable = (byte *)wrk_mem;
wrk_mem += LZSIZE*sizeof(byte);

return (int)(wrk_mem-mem0);
}
