/* /linux/drivers/block/double/dbtools.c
 *
 * Written by Jean-Marc Verbavatz <jmv@receptor.mgh.harvard.edu>
 * 8 Jan 1994
 *
 * Copyright 1994 by Jean-Marc Verbavatz. Redistribution of this file is
 * permitted under the GNU Public Licence
 */

#include <linux/config.h>
#include <linux/malloc.h>
#include <linux/sched.h>
#include <linux/locks.h>
#include <linux/string.h>
#include <linux/errno.h>
#include "blk.h"
#include <linux/double.h>

void add_request(struct blk_dev_struct *, struct request *);
struct request *get_request_wait(int, int);

char *db_buffer;
long *db_mapbuf;
static struct db_header db_header;
struct buffer_head **db_bh;

int dbtools_debug(void)
{
	if(memcmp(db_buffer+DB_MAXSZ, "$$$$", 4)) return 5;
	return 0;
}

/* Load bitmap for block n into memory */
int get_bitmap(struct dble_device *db, long n)
{
	if(db->bitmap_pos==n) return 1;		/* already loaded */
	if(db->bitmap_mod)		        /* update */
		if(!db_rw_iblock(db, WRITE, db->bitmap_pos+db->addr_bitmap, db->bitmap))
			return 0;
		else db->bitmap_mod = 0;
	/* read new one */
	if(db_rw_iblock(db, READ, n+db->addr_bitmap, db->bitmap)) {
		db->bitmap_pos = n;
		return 1;
	}
	return 0;
}


/* Load BAT for cluster n into memory */
int get_BAT(struct dble_device *db, long n)
{
	long i;


	i = n/(db->isize/db->BATsize);
	if(db->BAT_pos!=i) {
		if(db->BAT_mod) /* Flush */
			if(!db_rw_iblock(db, WRITE, db->BAT_pos+db->addr_BAT, db->BAT_block))
				return 0;
			else db->BAT_mod = 0;
/* Load appropriate BAT block into memory */
		if(!db_rw_iblock(db, READ, i+db->addr_BAT, db->BAT_block))
			return 0;
		db->BAT_pos = i;
	}
	return 1;
}

/* The driver has its own db_getblk, db_bmap functions as
 * the kernel functions getblk and bmap  cannot be called reliably
 * from within a driver.
 */

/* get real block number for regular file block. This information is
 * stored in blocks 1 to addr_bitmap and is updated by the
 * set_fd ioctl call, when the device is mounted.
 */
long db_bmap(struct  dble_device *db, long block)
{
	int n;

	if(block<(n=db->isize/sizeof(long))) return db->bmap0[block];
	else {
		if(!db_rw_iblock(db, READ, block/n, (u_char *) db_mapbuf))
			return 0;
		return db_mapbuf[block%n];
	}
}

/* Add request to physical device queue */
void ll_rw_dble(int rw, struct buffer_head *bh)
{
struct request *req;
int dev = bh->b_dev;

if(rw == WRITE && is_read_only(dev)) {
	printk("DouBle: can't write to read-only device\n");
	bh->b_dirt = bh->b_uptodate = 0;
	return;
}
lock_buffer(bh);
cli();
req = get_request_wait(((rw == READ) ? NR_REQUEST : NR_REQUEST*2/3), dev);
sti();
req->cmd = rw;
req->errors = 0;
req->sector = bh->b_blocknr * (bh->b_size>>9);
req->nr_sectors = req->current_nr_sectors = bh->b_size>>9;
req->buffer = bh->b_data;
req->bh = req->bhtail = bh;
add_request(blk_dev+MAJOR(dev), req);
}
 
/* make buffer n available */
void move_queue(struct buffer_head *bh, int n)
{
	int i;

	for( i = n; i > 0; i--)
		db_bh[i] = db_bh[i-1];
	*db_bh = bh;
	if(bh->b_dirt == 1) wait_on_buffer(bh);
/* Make sure it's not still in use */
	if(bh->b_count != 0) { 
		printk("DouBle: too few bh buffers\n");
		wait_on_buffer(bh);
		if(bh->b_count != 0)
			panic("db_getblk: logical error count=%d\n", bh->b_count);
	}
	bh->b_count = 1;
}


/* get internal block buffer for physical block */
struct buffer_head *db_getblk(dev_t dev, long block, int size)
{
	struct buffer_head *bh;
	int i;

/* Check to see if block is already in a buffer */
	for(i = 0; i<DB_NR_BH; i++) {
		bh = db_bh[i];
		if(bh->b_dev == dev && bh->b_blocknr == block) { /* Yes */
			dble_stat.ihit++;
			move_queue(bh, i);
			return bh;
		}
	}
/* Not in cache; get free buffer */
	dble_stat.inohit++;
	bh = db_bh[DB_NR_BH-1];
	if(bh->b_dirt==2) { /* needs to be written */
		bh->b_dirt = 1;
		ll_rw_dble(WRITE, bh);
		dble_stat.wphysical++;
	}
	move_queue(bh, DB_NR_BH - 1);

	lock_buffer(bh);
/* Clean bh structure */
	bh->b_dev = dev;
	bh->b_blocknr = block;
	bh->b_uptodate = 0;
	bh->b_req = 0;
	bh->b_dirt = 0;
	bh->b_size = size;
	bh->b_lock = 0;
	bh->b_wait = NULL;
	bh->b_reqnext = NULL;
	unlock_buffer(bh);
	return bh;
}

/* Read/Write block  */
int db_rw_iblock(struct dble_device *db, int cmd, long n, u_char *s)
{
	int size, len, blksize, offset;
	long block, rblock;
	struct buffer_head *bh;

	if(n<0 || n>=db->iblocks) {
		printk("db_rw_iblock: %ld/%ld non existent block\n", n, db->iblocks);
		return 0;
	}
	if(!(blksize=db->db_inode->i_blksize))
		blksize=BLOCK_SIZE;
	if(blksize<=db->isize) {
		block=n*(db->isize/blksize);
		offset=0;
	} else {
		block=n/(blksize/db->isize);
		offset=(n%(blksize/db->isize))*db->isize;
	}
	len=db->isize;
	while(len>0) {
		if(db->db_flags & DB_BMAP) {
			rblock = db_bmap(db, block);
			if(!rblock) {
				printk("db_rw_iblock: NULL block %ld %ld\n", n, block);
				return 0;
			}
		} else rblock = block;
		bh = db_getblk(db->db_device, rblock, blksize);

		if(!bh->b_uptodate && ((cmd==READ) || offset || (len<blksize))) {
			dble_stat.rphysical++;
			ll_rw_dble(READ, bh);
			if(db->debug >= 6) printk("DB_RW ll_rw READ\n");
			wait_on_buffer(bh);
			if(!bh->b_uptodate) {
				bh->b_count --;
				printk("db_rw_iblock: failure\n");
				return 0;
			}
		}
		size = blksize - offset;
		if(size>len) size=len;
		if(cmd==READ) {
			memcpy(s, bh->b_data+offset, size);
		} else
		/* Write only if data changed */
			if(!bh->b_uptodate || memcmp(bh->b_data+offset, s, size)) {
				memcpy(bh->b_data+offset, s, size);
				bh->b_dirt=2;
			}
		bh->b_count --;
		s+=size;
		len-=size;
		offset=0;
		block++;
		if(db->debug >= 5) printk("DB_RW DONE\n");
	}
	return 1;
}
	
/* find free block in bitmap */
u_long get_newbit(db)
struct dble_device *db;
{
	int j,k,l;
	u_long i,b, b0;

/* There would be three ways to start:
	- Back from the beginning as in versions 0.0-0.1: Worst choice.
	- From current bitmap, whichever it is.
	- From last bitmap used for writing.
Last choice might seem more logical, but as of 2/15/94 (for version 0.2),
I'll opt for the second choice because:
	- It is easier to implement.
	- 3rd approach is probably not significantly better.
	- That might help keep blocks together; Writing is often READING,
	then WRITING (at least at the block level).
 */
	if(db->bitmap_pos >= 0)
		b = db->bitmap_pos;
	else {
		b = 0;
		if(!get_bitmap(db, b)) return 0;
	}
	i = (b*db->isize)<<3;
	for(b0 = b; ;) {
		for(j = 0; j < db->isize; j++) if((k = db->bitmap[j]) != 0) {
			for(l = 0x80; l; l >>= 1, i++)
				if(k&l) {
					k &= ~l;
					db->bitmap[j] = k;
					db->bitmap_mod = 1;
					i += j<<3;
					return i;
				}
		}
		i += j<<3;
		if(i >= db->iblocks) { /* go back to beginning */
			b = -1;
			i = 0;
		}
		if(++b == b0) { /* Wrapped; not block free */
			printk("get_new_bit: out of space\n");
/* The driver does not recover well from such errors
 * I am seeking ways to recollect disk space in such emergency,
 * but this does not seem to be trivial
 */
			return 0;
		}
		if(!get_bitmap(db, b)) return 0;
	}
}

/* clear free bit (block is being used) */
int clr_bit(struct dble_device *db, u_long n)
{
	u_long i;

	i = n/(db->isize<<3);
	n = n%(db->isize<<3);
	if(!get_bitmap(db, i)) return 0;
	db->bitmap[n>>3] &= ~(0x80>>(n&7));
	db->bitmap_mod = 1;
	return 1;
}

/* set free bit (block has been released) */
int set_bit(struct dble_device *db, u_long n)
{
	u_long i;

	i = n/(db->isize<<3);
	n = n%(db->isize<<3);
	if(!get_bitmap(db, i)) return 0;
	db->bitmap[n>>3] |= (0x80>>(n&7));
	db->bitmap_mod = 1;
	return 1;
}

/* return value of bit n */
int get_bit(struct dble_device *db, u_long n)
{
	u_long i;

	i = n/(db->isize<<3);
	n = n%(db->isize<<3);
	if(!get_bitmap(db, i)) return -1;
	return (db->bitmap[n>>3] & (0x80>>(n&7)));
}

/* Read one cluster */
int read_oblock(struct dble_device *db, long n, u_char *s)
{
	u_char *t;
	u_long *p;
	struct BAT *b;
	int i;

	if(n<0 || n>=db->oblocks) {
		printk("read_oblock: %ld/%ld non existent cluster\n", n, db->oblocks);
		return 0;
	}
	memset(db_buffer+DB_MAXSZ, '$', 4); /* For debugging */
	dble_stat.roblock++;

	if(!get_BAT(db, n)) return 0;
	b = (struct BAT *)(db->BAT_block+db->BATsize*(n%(db->isize/db->BATsize)));
	if(b->type == -1) {         /* Unused (empty) */
		memset(s, 0, db->osize);
		return 1;
	}

/* Fetch compressed data in s */
	for(i = 0, t = db_buffer, p = &(b->iblock); *p && i<db->ratio;
	    i++, p++, t += db->isize) {
		if(!db_rw_iblock(db, READ, *p, t)) return 0;
	}

/* decompress data */
	switch(b->type) {
		case 0: memcpy(s, db_buffer, db->osize);
			break;
		case 1:	/* LZW */
			RLZW((code *)db_buffer, s, db->osize);
			break;
		case 10: /* Algorithm used for predictor */
			rpred(db_buffer, s, db->osize);
			break;
		case 212: /* LZRW2 12 bits */
			rlzrw2(db_buffer, s, db->osize);
			break;
		case 312: /* LZRW3A */
			rlzrw3a(db_buffer, s, db->osize);
			break;
		default: printk("Error compression (R)\n");
			return 0;
	}
	if(dbtools_debug()) {
		printk("read_oblock memory corruption\n");
		return 0;
	}

	return 1;
}

/* write one cluster */
int write_oblock(struct dble_device *db, long n, u_char *s)
{
	int i,l;
	u_long *p;
	struct BAT *b;

	if(n < 0 || n >= db->oblocks) {
		printk("write_oblock: %ld/%ld non existent cluster\n", n, db->oblocks);
		return 0;
	}
	memset(db_buffer+DB_MAXSZ, '$', 4); /* For debugging */
	dble_stat.woblock++;

	if(!get_BAT(db, n)) return 0;
	b = (struct BAT *)(db->BAT_block+db->BATsize*(n%(db->isize/db->BATsize)));

/* Compress data */
	l = db->code;
	i = db->osize - db->isize;
	switch(l) {
		case 0: i = db->osize; 
			break;
		case 1: /* LZW */
			i=WLZW(s, (code *)db_buffer, db->osize, i);
			if(i<0) {
				printk("Error WLZW %d\n", i);
				return 0;
			}
			if(i > 0) s = db_buffer;
			break;
		case 10: /* predictor's algorithm */
			wpred(s, db->osize, db_buffer, &i);
			if(i>0) s = db_buffer;
			break;
		case 212: /* LZRW2 */
			wlzrw2(s, db->osize, db_buffer, &i);
			if(i > 0) s = db_buffer;
			break;
		case 312: /* LZRW3A */
			wlzrw3a(s, db->osize, db_buffer, &i);
			if(i > 0) s = db_buffer;
			break;
		default: printk("Error compression (W)\n");
			return 0;
	}

	if(i == 0) {	/* No effective compression was achieved */
		l = 0;
		i = db->osize;
	}
	if(b->type != l) {	/* Cluster has compression method */
		b->type = l;
		db->BAT_mod = 1;
	}
	l = i/db->isize;	/* Number of blocks */
	if(i%db->isize) l++;
	for(p=&(b->iblock), i=0; i<l; i++,p++) { /* Write one block */
		if(*p==0)	 /* Allocate new block */
			if(!(*p=get_newbit(db))) { /* Out of space */
				b->type = -1;
				db->BAT_mod = 1;
				db->code = 312;
				for(p=&(b->iblock), i = 0; i< db->ratio; i++, p++) {
					set_bit(db, *p);
					*p = 0;
				}
				return 0;
			}
			else db->BAT_mod = 1;
		if(!db_rw_iblock(db, WRITE, *p, s)) return 0;
		s+=db->isize;
	}
	for(;i<db->ratio;i++,p++) {
		if(*p) { /* Fewer blocks used than before */
			if(!set_bit(db, *p)) return 0; /* release block */
			*p=0;
			db->BAT_mod = 1;
		}
		else break;
	}
	if(dbtools_debug()) {
		printk("write_oblock memory corruption\n");
		return 0;
	}

	return 1;
}

/* Launch write requests */
void db_write_buffers(void)
{
int i;
for(i = DB_NR_BH; i-->0;) {
	if(db_bh[i]->b_dirt==2) { /* need to write */
		db_bh[i]->b_dirt = 1;
		ll_rw_dble(WRITE, db_bh[i]);
		dble_stat.wphysical++;
	}
}
}

/* Flush all internal buffers */
int db_flush(struct dble_device *db)
{
	int i, n, result = 0;

/* Flush bitmap */
	if(db->bitmap_mod)
		if(!db_rw_iblock(db, WRITE, db->bitmap_pos+db->addr_bitmap, db->bitmap))
			result = -1;
		else db->bitmap_mod = 0;

/* Flush BAT */
	if(db->BAT_mod)
		if(!db_rw_iblock(db, WRITE, db->BAT_pos+db->addr_BAT, db->BAT_block))
			result = -1;
		else db->BAT_mod = 0;

/* Flush buffers */
	db_write_buffers();
	for(i = DB_NR_BH; i-->0;)
		if(db_bh[i]->b_dirt) wait_on_buffer(db_bh[i]);
	for(i = DB_NR_BH, n = 0; i-->0;)
		if(db_bh[i]->b_count) n++;
	if(n>0) printk("db_flush: %d buffers still in use\n", n);

	return result;
}

/* Read header from disk */
int db_rheader(struct dble_device *db)
{
	long block;
	int blksize;
	struct buffer_head *bh;

	if(db->db_flags & DB_BMAP)	/* Regular file, get real address */
		block = bmap(db->db_inode, 0);
	else block = 0;
	if(!(blksize = db->db_inode->i_blksize))
		blksize = BLOCK_SIZE;
	bh = getblk(db->db_device, block, blksize);
	if(!bh->b_uptodate) ll_rw_block(READ, 1, &bh);

	wait_on_buffer(bh);
	memcpy(&db_header, bh->b_data, sizeof(struct db_header));
	brelse(bh);

	if(db_header.Magic != DB_MAGIC) {
		printk("DouBle: wrong magic number\n");
		return 0;
	}
	db->isize = db_header.isize;
	db->osize = db_header.osize;
	if(db->osize>DB_MAXSZ) {
		printk("DouBle: size %d > MAX (%d)\n", db->osize, DB_MAXSZ);
		return 0;
	}
	db->iblocks = db_header.iblocks;
	db->oblocks = db_header.oblocks;
	db->addr_bitmap = db_header.addr_bitmap;
	db->addr_BAT = db_header.addr_BAT;
	db->addr_blocks = db_header.addr_blocks;
	db->BATsize = db_header.BATsize;
	db->ratio = db_header.ratio;
	db->code = db_header.code;
	db->bitmap_pos = -1;
	db->bitmap_mod = 0;
	db->BAT_pos = -1;
	db->BAT_mod = 0;
	db->debug = 0;

	db->bitmap = kmalloc(db->isize, GFP_KERNEL);
	db->BAT_block = kmalloc(db->isize, GFP_KERNEL);

	if(db->db_flags & DB_BMAP) {
		db->bmap0 = (long *) kmalloc(db->isize, GFP_KERNEL);
		if(db->bmap0 == NULL) {
			printk("DouBle: kmalloc returned NULL\n");
			return 0;
		}
		for(blksize = 0; blksize < db->isize/sizeof(long); blksize++)
			db->bmap0[blksize] = bmap(db->db_inode, (long)blksize);
	} else db->bmap0 = NULL;
	if(db->bitmap == NULL || db->BAT_block == NULL) {
		if(db->bitmap) kfree_s(db->bitmap, db->isize);
		if(db->BAT_block) kfree_s(db->BAT_block, db->isize);
		if(db->bmap0) kfree_s(db->bmap0, db->isize);
		printk("DouBle: kmalloc returned NULL\n");
		return 0;
	}

	return 1;
}
