#include <linux/config.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/ddi.h>
#include <linux/module.h>
#include "ax25.h"
#include "inet/inet.h"
#include "dev.h"
#include "skbuff.h"
#include "sock.h"
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/interrupt.h>

#include "inet/ip.h"
#include "inet/arp.h"

#ifdef CONFIG_AX25

#ifdef CONFIG_NETROM
#include "netrom.h"
#endif

char ax25_version[] = UTS_RELEASE;

/**********************************************************************************************************************\
*														       *
*						Handlers for the socket list.					       *
*														       *
\**********************************************************************************************************************/

static ax25_socket *volatile ax25_socket_list=NULL;

/* ax25 -> ascii */
#ifdef DEBUGGING
static char *ax2asc(ax25_address *a)
{
	static char buf[8];
	int ct=0;
	while(ct<7)
	{
		buf[ct]=a->ax25_call[ct]>>1;
		ct++;
	}
	buf[ct]=0;
	return buf;
}
#endif

/* Compare two ax.25 addresses */

static int ax25cmp(ax25_address *a, ax25_address *b)
{
	int ct=0;
	while(ct<6)
	{
		if((a->ax25_call[ct]&0xFE)!=(b->ax25_call[ct]&0xFE))	/* Clean off repeater bits */
			return 1;
		ct++;
	}
 	if((a->ax25_call[ct]&0x1E)==(b->ax25_call[ct]&0x1E))	/* SSID without control bit */
 		return 0;
 	return 2;			/* Partial match */
}

/*
 *	Note: Sockets may not be removed _during_ an interrupt or inet_bh
 *	handler using this technique. They can be added although we do not
 *	use this facility.
 */
 
static void ax25_remove_socket(ax25_socket *sk)
{
	ax25_socket *s;
	
	cli();
	s=ax25_socket_list;
	if(s==sk)
	{
		ax25_socket_list=s->next;
		sti();
		return;
	}
	while(s && s->next)
	{
		if(s->next==sk)
		{
			s->next=sk->next;
			sti();
			return;
		}
		s=s->next;
	}
	sti();
}

static void ax25_insert_socket(ax25_socket *sk)
{
	cli();
	sk->next=ax25_socket_list;
	ax25_socket_list=sk;
	sti();
}

static ax25_socket *ax25_find_listener(ax25_address *addr, int type)
{
	ax25_socket *s;
	s=ax25_socket_list;
	while(s)
	{
		if(ax25cmp(&s->ax25_source_addr,addr)==0 && s->type==type && s->state==TCP_LISTEN)
		{
			return(s);
		}
		s=s->next;
	}
	return(NULL);
}

static ax25_socket *ax25_find_socket(ax25_address *my_addr, ax25_address *dest_addr, int type)
{
	ax25_socket *s;
	s=ax25_socket_list;
	while(s)
	{
		if(ax25cmp(&s->ax25_source_addr,my_addr)==0 && ax25cmp(&s->ax25_dest_addr,dest_addr)==0 && s->type==type)
		{
			return(s);
		}
		s=s->next;
	}
	return(NULL);
}

/*
 *	This is only called from user mode. Thus it protects itself against
 *	interrupt users but doesn't worry about being called during work.
 *	Once it is removed from the queue no interrupt or bottom half will
 *	touch it and we are (fairly 8-) ) safe.
 */
 
void ax25_destroy_socket(ax25_socket *sk)	/* Not static as its used by the timer */
{
	struct sk_buff *skb;
	
	KILL_TIMER(sk);
	
	ax25_remove_socket(sk);
	ax25_init_vars(sk);			/* Flush the retransmit queue and send queue */
	
	while((skb=skb_dequeue(&sk->rqueue))!=NULL)
	{
		if(skb->sk!=sk)			/* A pending connection */
		{
			skb->sk->dead=1;	/* Queue the unaccepted socket for death */
			SET_TIMER(skb->sk);
			skb->sk->ax25_state=DISCONNECTED;
		}
		kfree_skb(skb,FREE_READ);
	}
	
	kfree_s(sk,sizeof(*sk));
}

/*******************************************************************************************************************\
*														    *
*		Routing rules for AX.25: Basically iterate over the active interfaces 				    *
*														    *
\*******************************************************************************************************************/

struct device *ax25rtr_get_dev(ax25_address *addr)
{
	struct device *dev;
	
	for(dev=dev_base;dev;dev=dev->next)
	{
		if((dev->flags & IFF_UP) && dev->hard_header_len==17 && dev->type == 3) /* Active kiss ax25 mode */
		{
			if(ax25cmp(addr,(ax25_address *)dev->dev_addr)==0)
				return(dev);
		}
	}
	return NULL;
}

/*******************************************************************************************************************\
*													            *
*	      Handling for system calls applied via the various interfaces to an AX25 socket object		    *
*														    *
\*******************************************************************************************************************/
 
static int ax25_fcntl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
	ax25_socket *sk;
	
	sk=(ax25_socket *)sock->data;
	
	if(sk==NULL)
	{
		printk("AX.25:fcntl:passed sock->data=NULL\n");
		return(0);
	}
	
	switch(cmd)
	{
		/* FIXME : TIOCINQ */
		default:
			return(-EINVAL);
	}
}

static int ax25_setsockopt(struct socket *sock, int level, int optname,
	char *optval, int optlen)
{
	ax25_socket *sk;
	int err,opt;
	
	sk=(ax25_socket *)sock->data;
	
	if(sk==NULL)
	{
		printk("AX25:setsockopt:passed sock->data=NULL\n");
		return 0;
	}
	
	if(level==SOL_SOCKET)
		return isock_setsockopt(sk,level,optname,optval,optlen);
	if(level!=SOL_AX25)
		return(-EOPNOTSUPP);
	if(optval==NULL)
		return(-EINVAL);
	err=verify_area(VERIFY_READ,optval,sizeof(int));
	if(err)
		return err;
	opt=get_fs_long((unsigned long *)optval);
	
	switch(optname)
	{
		case AX25_WINDOW:
			if(opt<1||opt>7)
				return -EINVAL;
			sk->window=opt;
			return 0;
		default:
			return(-ENOPROTOOPT);
	}
}

static int ax25_getsockopt(struct socket *sock, int level, int optname,
	char *optval, int *optlen)
{
	ax25_socket *sk;
	int val=0;
	int err;
	
	sk=(ax25_socket *)sock->data;
	if(sk==NULL)
	{
		printk("AX25:getsockopt:passed NULL sock->data.\n");
		return 0;
	}

	if(level==SOL_SOCKET)
		return isock_getsockopt(sk,level,optname,optval,optlen);
	
	if(level!=SOL_AX25)
		return(-EOPNOTSUPP);
	
	switch(optname)
	{
		case AX25_WINDOW:
			val=sk->window;
			break;
		default:
			return(-ENOPROTOOPT);
	}
	err=verify_area(VERIFY_WRITE,optlen,sizeof(int));
	if(err)
		return err;
	put_fs_long(sizeof(int),(unsigned long *)optlen);
	err=verify_area(VERIFY_WRITE,optval,sizeof(int));
	put_fs_long(val,(unsigned long *)optval);
	return(0);
}

static int ax25_listen(struct socket *sock, int backlog)
{
	ax25_socket *sk;
	
	sk=(ax25_socket *)sock->data;
	if(sk->type==SOCK_SEQPACKET && sk->state!=TCP_LISTEN)
	{
		sk->max_ack_backlog=backlog;
		sk->state=TCP_LISTEN;
		return 0;
	}
	return -EOPNOTSUPP;
}

static void def_callback1(struct sock *sk)
{
	if(!sk->dead)
		wake_up(sk->sleep);
}

static void def_callback2(struct sock *sk, int len)
{
	if(!sk->dead)
		wake_up(sk->sleep);
}

static int ax25_create(struct socket *sock, int protocol)
{
	ax25_socket *sk;
	sk=(ax25_socket *)kmalloc(sizeof(*sk),GFP_ATOMIC);
	if(sk==NULL)
		return(-ENOMEM);
	sk->type=sock->type;
	switch(sock->type)
	{
		case SOCK_DGRAM:
			break;
		case SOCK_SEQPACKET:
			sk->wfront=NULL;
			sk->ax25_retxqi=0;
			sk->ax25_lastrxnr=0;
			ax25_init_vars(sk);
			break;
		default:
			kfree_s((void *)sk,sizeof(*sk));
			return(-ESOCKTNOSUPPORT);
	}
	sk->rmem_alloc=0;
	sk->dead=0;
	sk->next=NULL;
	sk->broadcast=0;
	sk->rcvbuf=SK_RMEM_MAX;
	sk->sndbuf=SK_WMEM_MAX;
	sk->wmem_alloc=0;
	sk->rmem_alloc=0;
	sk->inuse=0;
	sk->dead=0;
	sk->debug=0;
	sk->prot=NULL;	/* So we use default free mechanisms */
	sk->broadcast=0;
	sk->err=0;
	sk->rqueue=NULL;
	sk->wback=NULL;
	sk->wfront=NULL;
	sk->send_head=NULL;
	sk->back_log=NULL;
	sk->state=TCP_CLOSE;
	sk->window=4;

	sk->state_change=def_callback1;
	sk->data_ready=def_callback2;
	sk->write_space=def_callback1;
	sk->error_report=def_callback1;

	memset(&sk->ax25_dest_addr,'\0',sizeof(sk->ax25_dest_addr));
	memset(&sk->ax25_source_addr,'\0',sizeof(sk->ax25_source_addr));
	sk->mtu=AX25_MTU;	/* 256 */

	if(sock!=NULL)
	{
		sock->data=(void *)sk;
		sk->sleep=sock->wait;
	}
	
	sk->zapped=1;
	return(0);
}

static ax25_socket *ax25_make_new(struct sock *osk)
{
	ax25_socket *sk;
	sk=(ax25_socket *)kmalloc(sizeof(*sk),GFP_ATOMIC);
	if(sk==NULL)
		return(NULL);
	sk->type=osk->type;
	switch(osk->type)
	{
		case SOCK_DGRAM:
			break;
		case SOCK_SEQPACKET:
			sk->wfront=NULL;
			sk->ax25_retxqi=0;
			sk->ax25_lastrxnr=0;
			ax25_init_vars(sk);
			break;
		default:
			kfree_s((void *)sk,sizeof(*sk));
			return NULL;
	}
	sk->rmem_alloc=0;
	sk->dead=0;
	sk->next=NULL;
	sk->broadcast=0;
	sk->rcvbuf=osk->rcvbuf;
	sk->sndbuf=osk->sndbuf;
	sk->wmem_alloc=0;
	sk->rmem_alloc=0;
	sk->inuse=0;
	sk->dead=0;
	sk->prot=NULL;	/* So we use default free mechanisms */
	sk->broadcast=0;
	sk->err=0;
	sk->rqueue=NULL;
	sk->wback=NULL;
	sk->wfront=NULL;
	sk->send_head=NULL;
	sk->back_log=NULL;
	sk->debug=osk->debug;
	sk->state=TCP_ESTABLISHED;
	sk->window=4;

	/* These two are vital! */
	sk->ax25_retxqi=0;
	sk->ax25_lastrxnr=0;

	sk->state_change=def_callback1;
	sk->data_ready=def_callback2;
	sk->write_space=def_callback1;
	sk->error_report=def_callback1;
	
	memset(&sk->ax25_dest_addr,'\0',sizeof(sk->ax25_dest_addr));
	memcpy(&sk->ax25_source_addr,&osk->ax25_source_addr,sizeof(sk->ax25_source_addr));
	sk->mtu=osk->mtu;
	sk->sleep=osk->sleep;
	sk->zapped=osk->zapped;
	return(sk);
}

static int ax25_dup(struct socket *newsock,struct socket *oldsock)
{
	return(ax25_create(newsock,oldsock->type));
}

static int ax25_release(struct socket *sock, struct socket *peer)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	if(sk==NULL)
		return(0);
	sk->state_change(sk);
	if(sk->state==TCP_ESTABLISHED && sk->type==SOCK_SEQPACKET)
	{
		sk->ax25_state=DISC_SENT;	
		ax25_write_internal(sk,DISC,POLLON,C_COMMAND);
		SET_TIMER(sk);
		sk->dead=1;
	}		
	else
		ax25_destroy_socket(sk);
	sock->data=NULL;	
	return(0);
}
		
static int ax25_bind(struct socket *sock, struct sockaddr *uaddr,int addr_len)
{
	ax25_socket *sk;
	int err;
	struct sockaddr_ax25 addr;
	struct device *dev;
	
	sk=(ax25_socket *)sock->data;
	if(sk==NULL)
	{
		printk("AX25:bind:sock->data=NULL\n");
		return 0;
	}
	
	if(sk->zapped==0)
		return(-EIO);
		
	err=verify_area(VERIFY_READ,uaddr,addr_len);
	if(err)
		return err;
	if(addr_len!=sizeof(addr))
		return -EINVAL;
	memcpy_fromfs(&addr,uaddr,addr_len);
#ifdef DONTDO	
	if(ax25_find_socket(&addr.sax25_call,sk->type)!=NULL)
	{
		if(sk->debug)
			printk("AX25: bind failed: in use.\n");
		return(-EADDRINUSE);	   
	}
#endif	
	memcpy(&sk->ax25_source_addr,&addr.sax25_call,sizeof(sk->ax25_source_addr));

	if((dev=ax25rtr_get_dev(&sk->ax25_source_addr))==NULL)
	{
		if(sk->debug)
			printk("AX25 bind failed: no device.\n");
		return(-EADDRNOTAVAIL);
	}
	ax25_insert_socket(sk);
	sk->zapped=0;
	if(sk->debug)
		printk("AX25: socket is bound.\n");
	return(0);
}

static int ax25_connect(struct socket *sock, struct sockaddr *uaddr,
	int addr_len, int flags)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	struct sockaddr_ax25 addr;
	int err;
	
	if(sk==NULL)
	{
		printk("AX25:connect:sock->data=NULL!\n");
		return 0;
	}
	
	if(sk->state==TCP_ESTABLISHED && sock->state == SS_CONNECTING)
	{
		sock->state=SS_CONNECTED;
		return 0;	/* Connect completed during a ERESTARTSYS event */
	}
	
	if(sk->state==TCP_ESTABLISHED && sk->type == SOCK_SEQPACKET)
		return -EISCONN;	/* No reconnect on a seqpacket socket */
		
	sk->state = TCP_CLOSE;	
	sock->state = SS_UNCONNECTED;
	
	if(addr_len!=sizeof(addr))
		return(-EINVAL);
	err=verify_area(VERIFY_READ,uaddr,addr_len);
	if(err)
		return err;
	memcpy_fromfs(&addr,uaddr,sizeof(addr));
	
	if(sk->ax25_source_addr.ax25_call[0]==0)	/* Must bind first - no autobinding in this */
		return -EINVAL;
		
	if(sk->type==SOCK_SEQPACKET && ax25_find_socket(&sk->ax25_source_addr,&addr.sax25_call,sk->type)!=NULL)
		return -EBUSY;				/* Already such a connection */
		
	memcpy(&sk->ax25_dest_addr,&addr.sax25_call,sizeof(sk->ax25_dest_addr));
	
	/* First the easy one */
	
	if(sk->type!=SOCK_SEQPACKET)
	{
		sock->state = SS_CONNECTED;
		sk->state=TCP_ESTABLISHED;
		return(0);
	}
	

	/* Move to connecting socket, ax.25 lapb WAIT_UA.. */	
	sock->state = SS_CONNECTING;
	sk->state=TCP_SYN_SENT;
	sk->ax25_state=WAIT_UA;
	ax25_write_internal (sk, SABM, POLLOFF, C_COMMAND);
	SET_TIMER(sk);		/* Start going SABM SABM until a UA or a give up and DM */
	
	/* Now the loop */
	
	if(sk->state!=TCP_ESTABLISHED && (flags & O_NONBLOCK))
		return -EINPROGRESS;
		
	cli();	/* To avoid races on the sleep */
	/* A DM or timeout will go to closed, a UA will go to ABM */
	while(sk->state==TCP_SYN_SENT)
	{
		interruptible_sleep_on(sk->sleep);
		if (current->signal & ~current->blocked)
		{
			sti();
			return -ERESTARTSYS;
		}
	}
	if(sk->state!=TCP_ESTABLISHED)	/* Not in ABM, not in WAIT_UA -> failed */
	{
		sti();
		sock->state=SS_UNCONNECTED;
		return -sk->err;	/* Always set at this point */
	}
	
	sock->state=SS_CONNECTED;
	sti();
	
	return 0;
}
	
static int ax25_socketpair(struct socket *sock1, struct socket *sock2)
{
	return(-EOPNOTSUPP);
}

static int ax25_accept(struct socket *sock, struct socket *newsock, int flags)
{
	ax25_socket *sk;
	ax25_socket *newsk;
	struct sk_buff *skb;

	if(newsock->data)
		kfree_s(newsock->data,sizeof(ax25_socket));
	newsock->data=NULL;
	
	sk=(ax25_socket *)sock->data;

	if(sk->type!=SOCK_SEQPACKET)
		return -EOPNOTSUPP;
	
	if(sk->state!=TCP_LISTEN)
		return -EINVAL;
		
	/* The write queue this time is holding sockets ready to use
	   hooked into the SABM we saved */
	
	do
	{
		cli();
		skb=skb_dequeue(&sk->rqueue);
		if(skb==NULL)
		{
			if(flags&O_NONBLOCK)
			{
				sti();
				return 0;
			}
			interruptible_sleep_on(sk->sleep);
			if(current->signal & ~current->blocked) 
			{
				sti();
				return -ERESTARTSYS;
			}
		}
	}
	while(skb==NULL);
	newsk=skb->sk;
	newsk->pair=NULL;
	sti();
	/* Now attach up the new socket */
	sk->ack_backlog--;
	newsock->data=newsk;
	return(0);			
}

static int ax25_getname(struct socket *sock, struct sockaddr *uaddr,
	int *uaddr_len, int peer)
{
	ax25_address *addr;
	struct sockaddr_ax25 sax;
	ax25_socket *sk;
	int len;
	int err;
	
	sk=(ax25_socket *)sock->data;
	
	err = verify_area(VERIFY_WRITE,uaddr_len,sizeof(long));
	if(err)
		return err;
		
	len = get_fs_long(uaddr_len);
	
	err = verify_area(VERIFY_WRITE, uaddr, len);
	if(err)
		return err;
	
	if(len<sizeof(struct sockaddr_ax25))
		return -EINVAL;
		
	if(peer)
	{
		if(sk->state!=TCP_ESTABLISHED)
			return -ENOTCONN;
		addr=&sk->ax25_dest_addr;
	}
	else
		addr=&sk->ax25_source_addr;
		
	sax.sax25_family = AF_AX25;
	memcpy(&sax.sax25_call,addr,sizeof(sax.sax25_call));
	memcpy_tofs(uaddr,&sax,sizeof(sax));
	put_fs_long(len,uaddr_len);
	return(0);
}

 
int ax25_rcv(struct sk_buff *skb, struct device *dev, struct packet_type *ptype)
{
	unsigned char *data=(unsigned char *)(skb+1);
	unsigned char *dest;
	ax25_socket *sk;
	int type=0;
	int sz=3;
	
	
	if((*data&0x0F)!=0)
	{
		kfree_skb(skb,FREE_READ);	/* Not a KISS data frame */
		return(0);
	}
	data++;
	if(ax25cmp((ax25_address *)data,(ax25_address *)dev->dev_addr)!=0 &&
	   ax25cmp((ax25_address *)data,(ax25_address *)dev->broadcast)!=0 
#ifdef CONFIG_NETROM	   
	   && ax25cmp((ax25_address *)data,(ax25_address *)netrom_dev.broadcast)!=0
#endif	   
	)
	{
		kfree_skb(skb,FREE_READ);	/* Someone elses packet (very likely - KISS is almost always
						   promiscuous */
		return(0);
	}
	
	/* Now check for source/digipeat requests. We don't cope with them but we check ! */
	
	if(data[6]&LAPB_C)
		type=C_COMMAND;
		
	dest=data+7;
	sz+=7;
	
	if(dest[6]&LAPB_C)
		type=C_RESPONSE;
	
	while(!(dest[6]&LAPB_E))	/* Walk the digi chain until we see the final digi (or the source)*/
	{
		if(!(dest[6]&AX25_REPEATED))	/* It is for us but hasn't got there yet! */
		{
			kfree_skb(skb,FREE_READ);
			return(0);
		}
		dest+=7;
		sz+=7;
	}
	
	sz+=7;
	
	/* Ok point dest at the LAPB header */
	dest+=7;
	
	if((*dest&~0x10)==LAPB_UI)	/* UI frame - bypass LAPB processing */
	{
		dest++;
		/* Now we are pointing at the pid byte */
		switch(*dest++)
		{
			case PID_IP:
				skb->h.raw=dest;
				ip_rcv(skb,dev,ptype);	/* Note ptype here is the wrong one, fix me later */
				break;
			case PID_ARP:
				skb->h.raw=dest;
				arp_rcv(skb,dev,ptype);	/* Ditto ptype wrong here */
				break;
			case 0:
			case PID_AX25:
				skb->h.raw=dest;
				/* Now find a suitable dgram socket */
				sk=ax25_find_socket((ax25_address *)data,(ax25_address *)(data+7),SOCK_DGRAM);
				if(sk!=NULL)
				{
					if(sk->rmem_alloc>=sk->rcvbuf)
						kfree_skb(skb,FREE_READ);
					else
					{
						skb_queue_tail(&sk->rqueue,skb);
						skb->sk=sk;
						sk->rmem_alloc+=skb->mem_len;
						if(!sk->dead)
							sk->data_ready(sk,skb->len-2);
					}
				}
				else
				{
					kfree_skb(skb,FREE_READ);
				}
				break;	
#ifdef CONFIG_NETROM				
			case PID_NETROM:
				/* UI netrom frame. Kick at the router */
				skb->h.raw=dest;
				netrom_uiframe(skb);
				kfree_skb(skb,FREE_READ);
				break;
#endif

			default:
				
				kfree_skb(skb,FREE_READ);	/* Will scan SOCK_AX25 RAW sockets */
				break;
		}
		return(0);
	}
	/* LAPB protocol data - not yet finished (or much past started!) */
	sk=ax25_find_socket((ax25_address *)data,(ax25_address *)(data+7),SOCK_SEQPACKET);

	if(sk)
	{
		skb->h.raw=dest;
		skb->sk=sk;
		/* Process the frame. If it is queued up internally it returns one otherwise we 
		   free it immediately. This routine itself wakes the user context layers so we
		   do no further work */
		if(ax25_process_rx_frame(sk,skb,type)==0)
			kfree_skb(skb,FREE_READ);
		return(0);
	}	
	else
	{
		ax25_socket *make;	

		if((dest[0]&0xEF)!=SABM_CONTROL)
		{
			kfree_skb(skb,FREE_READ);
			return(0);
		}

		sk=ax25_find_listener((ax25_address *)(data),SOCK_SEQPACKET);
		if(sk==NULL || sk->ack_backlog == sk->max_ack_backlog || (make=ax25_make_new(sk))==NULL)
		{
			kfree_skb(skb,FREE_READ);
			return(0);
		}
		memcpy(&make->ax25_dest_addr,data+7,sizeof(make->ax25_dest_addr));
		skb->sk=make;
		make->state=TCP_ESTABLISHED;
		ax25_init_vars(make);
		make->ax25_state=ABM;
		ax25_write_internal(make,UA,POLLON,C_RESPONSE);
		sk->ack_backlog++;
		make->pair=sk;
		ax25_insert_socket(make);
		skb_queue_head(&sk->rqueue,skb);
		if(!sk->dead)
			sk->data_ready(sk,skb->len-2);
		return 0;
	}
}

static int ax25_sendto(struct socket *sock, void *ubuf, int len, int noblock,
	unsigned flags, struct sockaddr *usip, int addr_len)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	struct sockaddr_ax25 *usax=(struct sockaddr_ax25 *)usip;
	int err;
	struct sockaddr_ax25 sax;
	struct sk_buff *skb;
	struct device *dev;
	unsigned char *asmptr;
	int size;
	
	if(sk->err)
	{
		err=sk->err;
		sk->err=0;
		return -err;
	}

	if(flags)
		return -EINVAL;
	if(len<0)
		return -EINVAL;
	if(len == 0)
		return 0;
		
		
	if(usax)
	{
		if(addr_len <sizeof(sax))
			return(-EINVAL);
		err=verify_area(VERIFY_READ,usax,sizeof(sax));
		if(err)
			return(err);
		memcpy_fromfs(&sax,usax,sizeof(sax));
		if(sk->type==SOCK_SEQPACKET && memcmp(&sk->ax25_dest_addr,&sax.sax25_call,sizeof(sax.sax25_call))!=0)
			return -EISCONN;
		if(sax.sax25_family != AF_AX25)
			return -EINVAL;
	}
	else
	{
		if(sk->state!=TCP_ESTABLISHED)
			return -ENOTCONN;
		sax.sax25_family=AF_AX25;
		memcpy(&sax.sax25_call,&sk->ax25_dest_addr,sizeof(sax.sax25_call));
	}
	
	err=verify_area(VERIFY_READ,ubuf,len);
	if(err)
		return err;
		
	
	if(sk->debug)
		printk("AX.25: sendto: Addresses built.\n");
	/* Build a packet */
	
	if(sk->debug)
		printk("AX.25: sendto: building packet.\n");
	err=verify_area(VERIFY_READ,ubuf,len);
	if(err)
		return err;

	size=sizeof(struct sk_buff)+2+len+15;	/* 2 bytes for PID and (U)I frame byte: 15 for KISS data, 2*calls */	
	
	
	cli();
	while(size+sk->wmem_alloc>sk->sndbuf)
	{
		if(sk->debug)
			printk("Didnt fit size=%d used=%d buf=%d\n",size,sk->wmem_alloc,sk->sndbuf);
		if(noblock)
		{
			sti();
			return -EAGAIN;
		}
		if(sk->err)
		{
			sti();
			err=sk->err;
			sk->err=0;
			return -err;
		}	
		interruptible_sleep_on(sk->sleep);
		if (current->signal & ~current->blocked) 
		{
			sti();
			return(-ERESTARTSYS);
		}
	}					

	sk->wmem_alloc+=size;
	sti();
	if(sk->debug)	
		printk("AX.25: sendto: allocating buffer (%d)\n",size-sizeof(struct sk_buff));

	skb=alloc_skb(size,GFP_KERNEL);
	if(skb==NULL)
	{
		sk->wmem_alloc-=size;
		sk->write_space(sk);
		return -ENOMEM;
	}		

	skb->sk=sk;
	skb->free=1;
	skb->arp=1;
	skb->len=size-sizeof(struct sk_buff);
	
	
	asmptr=(unsigned char *)(skb+1);
	if(sk->debug)
		printk("Building AX.25 Header.\n");
	/* Build an AX.25 header */
	*asmptr++=0;	/* KISS data */
	memcpy(asmptr,&sax.sax25_call,sizeof(sax.sax25_call));
	asmptr[6]&=~LAPB_E;
	asmptr[6]|=LAPB_C;
	asmptr+=7;
	memcpy(asmptr,&sk->ax25_source_addr,sizeof(sk->ax25_source_addr));
	asmptr[6]|=LAPB_E;
	asmptr[6]&=~AX25_REPEATED;
	asmptr+=7;
	*asmptr++=LAPB_UI;	/* Datagram - will get replaced for I frames */
	*asmptr++=PID_AX25;	/* Raw AX.25 */
		
	if(sk->debug)
		printk("AX.25: Appending user data.\n");
	/* User data follows immediately after the AX.25 data */
	memcpy_fromfs(asmptr,ubuf,len);
	if(sk->debug)
		printk("AX.25: Transmitting buffer\n");
	if(sk->type==SOCK_SEQPACKET)
	{
		/* Connected mode sockets go via the LAPB machine */
		if(sk->state!=TCP_ESTABLISHED)
		{
			kfree_skb(skb,FREE_WRITE);
			return -ENOTCONN;
		}
		ax25_output(sk,skb);	/* Shove it onto the queue and kick */
		return len;
	}
	else
	{
		/* Datagram frames go straight out of the door as UI */
		/* Find out where this has to go */
		if((dev=ax25rtr_get_dev(&sk->ax25_source_addr))==NULL)
		{
			kfree_skb(skb,FREE_WRITE);
			return -ENETUNREACH;
		}
		dev->queue_xmit(skb,dev,SOPRI_NORMAL);
		return len;
	}
}

static int ax25_send(struct socket *sock, void *ubuf, int size, int noblock, unsigned flags)
{
	return ax25_sendto(sock,ubuf,size,noblock,flags,NULL,0);
}

static int ax25_write(struct socket *sock, char *ubuf, int size, int noblock)
{
	return ax25_send(sock,ubuf,size,noblock,0);
}

static int ax25_recvfrom(struct socket *sock, void *ubuf, int size, int noblock,
		   unsigned flags, struct sockaddr *sip, int *addr_len)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	struct sockaddr_ax25 *sax=(struct sockaddr_ax25 *)sip;
	ax25_address *ap;
	/* FILL ME IN */
	int copied = 0;
	struct sk_buff *skb;
	int er;
	
	if(sk->err)
	{
		er= -sk->err;
		sk->err=0;
		return er;
	}
	
	if(size==0)
		return 0;
	if(size<0)
		return -EINVAL;
	if(addr_len)
	{
		er=verify_area(VERIFY_WRITE,addr_len,sizeof(*addr_len));
		if(er)
			return er;
		put_fs_long(sizeof(*sax),addr_len);
	}
	if(sax)
	{
		er=verify_area(VERIFY_WRITE,sax,sizeof(*sax));
		if(er)
			return er;
	}
	er=verify_area(VERIFY_WRITE,ubuf,size);
	if(er)
		return er;
	/* This works for seqpacket too. The receiver has ordered the queue for us! We do one quick check first though */
	if(sk->type==SOCK_SEQPACKET && sk->state!=TCP_ESTABLISHED)
		return -ENOTCONN;
	/* Now we can treat all alike */
	skb=skb_recv_datagram(sk,flags,noblock,&er);
	if(skb==NULL)
		return er;
	copied=(size<skb->len)?size:skb->len;
	skb_copy_datagram(skb,sk->type==SOCK_SEQPACKET?2:0,ubuf,copied);
	
	if(sax)
	{
		struct sockaddr_ax25 addr;
		unsigned char *dp=(unsigned char *)(skb+1);
		ap=(ax25_address *)&dp[8];	/* Fixme: Messy */
		addr.sax25_family=AF_AX25;
		memcpy(&addr.sax25_call,ap,sizeof(addr.sax25_call));
		memcpy_tofs(sax,&addr,sizeof(*sax));
	}
	skb_free_datagram(skb);
	return(copied);
}		

static int ax25_recv(struct socket *sock, void *ubuf, int size , int noblock,
	unsigned flags)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	if(sk->zapped)
		return -ENOTCONN;
	return ax25_recvfrom(sock,ubuf,size,noblock,flags,NULL, NULL);
}

static int ax25_read(struct socket *sock, char *ubuf, int size, int noblock)
{
	return ax25_recv(sock,ubuf,size,noblock,0);
}


static int ax25_shutdown(struct socket *sk,int how)
{
	/* FIXME - generate DM and RNR states */
	return -EOPNOTSUPP;
}

static int ax25_select(struct socket *sock , int sel_type, select_table *wait)
{
	ax25_socket *sk=(ax25_socket *)sock->data;
	/* FIXME: datagram_select covers seqpacket for now (doesnt do pre-connect stuff right) */
	return datagram_select(sk,sel_type,wait);
}

static int ax25_ioctl(struct socket *sock,unsigned int cmd, unsigned long arg)
{
	
	switch(cmd)
	{
		default:
			return -EINVAL;
	}
	/*NOTREACHED*/
	return(0);
}

static int ax25_fioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	int minor;
	minor=MINOR(inode->i_rdev);
	if(minor!=0)
		return(-ENODEV);
	return ax25_ioctl(NULL,cmd,arg);
}
	
static struct file_operations ax25_fops = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	ax25_fioctl,	/* We have ioctl */
	NULL,
	NULL,
	NULL
};


static struct proto_ops ax25_proto_ops = {
	AF_AX25,
	
	ax25_create,
	ax25_dup,
	ax25_release,
	ax25_bind,
	ax25_connect,
	ax25_socketpair,
	ax25_accept,
	ax25_getname,
	ax25_read,
	ax25_write,
	ax25_select,
	ax25_ioctl,
	ax25_listen,
	ax25_send,
	ax25_recv,
	ax25_sendto,
	ax25_recvfrom,
	ax25_shutdown,
	ax25_setsockopt,
	ax25_getsockopt,
	ax25_fcntl,
};

static struct packet_type ax25_packet_type = {
  NET16(ETH_P_AX25),
  0,
  ax25_rcv,
  NULL,
  NULL
};

unsigned long ax25_init(unsigned long mem_start)
{
	if(register_chrdev(AF_AX25_MAJOR,"af_ax25", &ax25_fops) < 0) {
		printk("%s: cannot register major device %d!\n",
			"af_ax25", AF_AX25_MAJOR);
		return MODULE_ERROR(mem_start);
	}
	
	(void) sock_register(ax25_proto_ops.family, &ax25_proto_ops);
	dev_add_pack(&ax25_packet_type);
	
	printk("GW4PTS AX.25 for Linux 0.06 ALPHA\n");
	printk("Portions (c) Copyright 1984 University Of British Columbia\n");
	printk("Portions (c) Copyright 1990 The Regents of the University Of California\n");
	return MODULE_OK(mem_start);
	
}

void ax25_cleanup(void)
{
	dev_remove_pack(&ax25_packet_type);
	sock_unregister(ax25_proto_ops.family, &ax25_proto_ops);
	unregister_chrdev(AF_AX25_MAJOR,"af_ax25");
}

/* Called by ddi.c on kernel start up */

void ax25_proto_init(struct ddi_proto *pro)
{
	ax25_init(0);
}

#endif
