/* VGAlib version 1.2 - (c) 1993 Tommy Frandsen 		   */
/*								   */
/* This library is free software; you can redistribute it and/or   */
/* modify it without any restrictions. This library is distributed */
/* in the hope that it will be useful, but without any warranty.   */

/* Multi-chipset support Copyright 1993 Harm Hanemaayer */

#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/kd.h>
#include <sys/stat.h>
#include <sys/vt.h>
#include <linux/mm.h>
#undef free
#include <stdlib.h>
#include "vga.h"
#include "vgamacros.h"
#include "chipsets.h"
#include "config.h"

/* #define char unsigned char */
/* This is very silly, I don't know if removing it breaks anything */
/* I don't have the guts to try it. */


/* Delay in microseconds after a mode is set (screen is blanked during this */
/* time), allows video signals to stabilize */
#define MODESWITCHDELAY 100000


#define GRAPH_BASE 0xA0000
#define GRAPH_SIZE 0x10000
#define FONT_BASE  0xA0000
#define FONT_SIZE  0x2000

#define MAX_REGS 100

#define SETSIG(sa, sig, fun) \
	sa.sa_handler = fun; \
	sa.sa_flags = 0; \
	sa.sa_mask = 0; \
	sigaction(sig, &sa, NULL);

#define SETSIG2(sa, sig, fun, oldsa) \
	sa.sa_handler = fun; \
	sa.sa_flags = 0; \
	sa.sa_mask = 0; \
	sigaction(sig, &sa, &oldsa);


/* variables used to shift between monchrome and color emulation */
int CRT_I;			/* current CRT index register address */
int CRT_D;			/* current CRT data register address */
int IS1_R;			/* current input status register address */
static int color_text;		/* true if color text emulation */

static struct info infotable[] = {
	{ 80, 25, 16, 160 },		/* 0 */
	{ 320, 200, 16, 40 },
	{ 640, 200, 16, 80 },
	{ 640, 350, 16, 80 },
	{ 640, 480, 16, 80 },		/* 4 */
	{ 320, 200, 256, 320 },
	{ 320, 240, 256, 80 },
	{ 320, 400, 256, 80 },
	{ 360, 480, 256, 90 },		/* 8 */
	{ 640, 480, 2, 80 },

	{ 640, 480, 256, 640 },		/* 10 */
	{ 800, 600, 256, 800 },
	{ 1024, 768, 256, 1024 },

	{ 320, 200, 32768, 640 },	/* 13 */
	{ 320, 200, 65536, 640 },
	{ 320, 200, 65536 * 256, 960 },

	{ 640, 480, 32768, 1280 },	/* 16 */
	{ 640, 480, 65536, 1280 },
	{ 640, 480, 65536 * 256, 640 * 3 },
	
	{ 800, 600, 32768, 1600 },	/* 19 */
	{ 800, 600, 65536, 1600 },
	{ 800, 600, 65536 * 256, 800 * 3 },

	{ 1024, 768, 32768, 2048 },	/* 22 */
	{ 1024, 768, 65536, 2048 },
	{ 1024, 768, 65536 * 256, 1024 * 3 },
	
	{ 1280, 1024, 256, 1280 },	/* 25 */
	{ 1280, 1024, 32768, 2560 },
	{ 1280, 1024, 65536, 2560 },
	{ 1280, 1024, 65536 * 256, 1280 * 3 },

	{ 720, 350, 16, 90 },		/* 29 */
	{ 720, 480, 16, 90 }
};


/* default palette values */
static unsigned char default_red[256]   
             = { 0, 0, 0, 0,42,42,42,42,21,21,21,21,63,63,63,63,
		 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
		 0,16,31,47,63,63,63,63,63,63,63,63,63,47,31,16,
		 0, 0, 0, 0, 0, 0, 0, 0,31,39,47,55,63,63,63,63,
		63,63,63,63,63,55,47,39,31,31,31,31,31,31,31,31,
		45,49,54,58,63,63,63,63,63,63,63,63,63,58,54,49,
		45,45,45,45,45,45,45,45, 0, 7,14,21,28,28,28,28,
		28,28,28,28,28,21,14, 7, 0, 0, 0, 0, 0, 0, 0, 0,
		14,17,21,24,28,28,28,28,28,28,28,28,28,24,21,17,
		14,14,14,14,14,14,14,14,20,22,24,26,28,28,28,28,
		28,28,28,28,28,26,24,22,20,20,20,20,20,20,20,20,
		 0, 4, 8,12,16,16,16,16,16,16,16,16,16,12, 8, 4,
		 0, 0, 0, 0, 0, 0, 0, 0, 8,10,12,14,16,16,16,16,
		16,16,16,16,16,14,12,10, 8, 8, 8, 8, 8, 8, 8, 8,
		11,12,13,15,16,16,16,16,16,16,16,16,16,15,13,12,
		11,11,11,11,11,11,11,11, 0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char default_green[256] 
	     = { 0, 0,42,42, 0, 0,21,42,21,21,63,63,21,21,63,63,
		 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
		 0, 0, 0, 0, 0, 0, 0, 0, 0,16,31,47,63,63,63,63,
		63,63,63,63,63,47,31,16,31,31,31,31,31,31,31,31,
		31,39,47,55,63,63,63,63,63,63,63,63,63,55,47,39,
		45,45,45,45,45,45,45,45,45,49,54,58,63,63,63,63,
		63,63,63,63,63,58,54,49, 0, 0, 0, 0, 0, 0, 0, 0,
		 0, 7,14,21,29,28,28,28,28,28,28,28,28,21,14, 7,
		14,14,14,14,14,14,14,14,14,17,21,24,28,28,28,28,
		28,28,28,28,28,24,21,17,20,20,20,20,20,20,20,20,
		20,22,24,26,28,28,28,28,28,28,28,28,28,26,24,22,
		 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 8,12,16,16,16,16,
		16,16,16,16,16,12, 8, 4, 8, 8, 8, 8, 8, 8, 8, 8,
		 8,10,12,14,16,16,16,16,16,16,16,16,16,14,12,10,
		11,11,11,11,11,11,11,11,11,12,13,15,16,16,16,16,
		16,16,16,16,16,15,13,12, 0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char default_blue[256]  
             = { 0,42, 0,42, 0,42, 0,42,21,63,21,63,21,63,21,63,
		 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
		63,63,63,63,63,47,31,16, 0, 0, 0, 0, 0, 0, 0, 0,
		 0,16,31,47,63,63,63,63,63,63,63,63,63,55,47,39,
		31,31,31,31,31,31,31,31,31,39,47,55,63,63,63,63,
		63,63,63,63,63,58,54,49,45,45,45,45,45,45,45,45,
		45,49,54,58,63,63,63,63,28,28,28,28,28,21,14, 7,
		 0, 0, 0, 0, 0, 0, 0, 0, 0, 7,14,21,28,28,28,28,
		28,28,28,28,28,24,21,17,14,14,14,14,14,14,14,14,
		14,17,21,24,28,28,28,28,28,28,28,28,28,26,24,22,
		20,20,20,20,20,20,20,20,20,22,24,26,28,28,28,28,
		16,16,16,16,16,12, 8, 4, 0, 0, 0, 0, 0, 0, 0, 0,
		 0, 4, 8,12,16,16,16,16,16,16,16,16,16,14,12,10,
		 8, 8, 8, 8, 8, 8, 8, 8, 8,10,12,14,16,16,16,16,
		16,16,16,16,16,15,13,12,11,11,11,11,11,11,11,11,
		11,12,13,15,16,16,16,16, 0, 0, 0, 0, 0, 0, 0, 0};


/* used to decompose color value into bits (for fast scanline drawing) */
union bits {
    struct {
        unsigned char bit3;
        unsigned char bit2;
        unsigned char bit1;
        unsigned char bit0;
    } b;
    unsigned int i;
};

/* color decompositions */
static union bits color16[16] = {{0,0,0,0},
			         {0,0,0,1},
			      	 {0,0,1,0},
			         {0,0,1,1},
			         {0,1,0,0},
			         {0,1,0,1},
			         {0,1,1,0},
			         {0,1,1,1},
			         {1,0,0,0},
			         {1,0,0,1},
			         {1,0,1,0},
			         {1,0,1,1},
			         {1,1,0,0},
			         {1,1,0,1},
			         {1,1,1,0},
			         {1,1,1,1}};

/* display plane buffers (for fast scanline drawing) */
static unsigned char plane0[256];
static unsigned char plane1[256];
static unsigned char plane2[256];
static unsigned char plane3[256];


static unsigned char text_regs[MAX_REGS];   /* VGA registers for saved text mode */   

/* saved text mode palette values */ 
static unsigned char text_red[256];  
static unsigned char text_green[256];  
static unsigned char text_blue[256];  

/* saved graphics mode palette values */ 
static unsigned char graph_red[256];  
static unsigned char graph_green[256];  
static unsigned char graph_blue[256];  

static int         prv_mode  = TEXT;    /* previous video mode      */
static int         flip_mode = TEXT;    /* flipped video mode       */

int __vga_cur_mode  = TEXT;		/* current video mode       */
struct info __vga_cur_info;		/* current video parameters */
int __vga_cur_color;			/* current color            */

#define cur_mode __vga_cur_mode
#define cur_info __vga_cur_info
#define cur_color __vga_cur_color

static int initialized = 0;   /* flag: initialize() called ?  */
static int flip        = 0;   /* flag: executing vga_flip() ? */

/* svgalib additions: */

static int chipset = UNDEFINED;
int __driver_report = 1;	/* report driver used after chipset detection */
int __videomemoryused = -1;
int __svgalib_critical = 0;	/* indicates blitter is busy */
static int currentpage;
static int currentlogicalwidth;
static int currentdisplaystart;

static int   tty0_fd;   /* /dev/tty0 file descriptor 		     */
static int   mem_fd;    /* /dev/mem file descriptor		     */
static FILE* console;   /* console input stream			     */
unsigned char* graph_mem;  /* dummy buffer for mmapping grahics memory */

static unsigned char* graph_buf = NULL;  /* saves graphics data during flip */

static unsigned char font_buf1[FONT_SIZE];  /* saved font data - plane 2 */
static unsigned char font_buf2[FONT_SIZE];  /* saved font data - plane 3 */

static struct termios text_termio;  /* text mode termio parameters     */
static struct termios graph_termio; /* graphics mode termio parameters */

int flipchar = '\x1b';   /* flip character - initially  ESCAPE */


/* Chipset specific functions */

int (**chipsetfunctions)() = vga_chipsetfunctions;

#define chipset_saveregs chipsetfunctions[CHIPSET_SAVEREGS]
#define chipset_setregs chipsetfunctions[CHIPSET_SETREGS]
#define chipset_unlock chipsetfunctions[CHIPSET_UNLOCK]
#define chipset_test chipsetfunctions[CHIPSET_TEST]
#define chipset_setpage chipsetfunctions[CHIPSET_SETPAGE]
#define chipset_setmode chipsetfunctions[CHIPSET_SETMODE]
#define chipset_modeavailable chipsetfunctions[CHIPSET_MODEAVAILABLE]
#define chipset_getmodeinfo chipsetfunctions[CHIPSET_GETMODEINFO]

int (**chipsetfunctionslist[])() = {
	NULL,
	vga_chipsetfunctions,
	#ifdef INCLUDE_ET4000_DRIVER
	et4000_chipsetfunctions,
	#else
	NULL,
	#endif
	#ifdef INCLUDE_CIRRUS_DRIVER
	cirrus_chipsetfunctions,
	#else
	NULL,
	#endif
	#ifdef INCLUDE_TVGA_DRIVER
	tvga8900_chipsetfunctions,
	#else
	NULL,
	#endif
	#ifdef INCLUDE_OAK_DRIVER
	oak_chipsetfunctions,
	#else
	NULL,
	#endif
};


/* Chipset drivers */

/* vgadrv	Standard VGA (also used by drivers below) */
/* et4000	Tseng ET4000 (from original vgalib) */
/* cirrus	Cirrus Logic GD542x */
/* tvga8900     Trident TVGA 8900/9000 (derived from tvgalib) */
/* oak		Oak Technologies 037/067/077 */


static void set_graphtermio()
{
        /* set graphics mode termio parameters */
        ioctl(0, TCSETSW, &graph_termio);
}


static void set_texttermio()
{
    /* restore text mode termio parameters */
    ioctl(0, TCSETSW, &text_termio);
}


static void disable_interrupt()
{
    struct termios cur_termio;

    ioctl(0, TCGETS, &cur_termio);
    cur_termio.c_lflag &= ~ISIG;
    ioctl(0, TCSETSW, &cur_termio);
}


static void enable_interrupt()
{
    struct termios cur_termio;

    ioctl(0, TCGETS, &cur_termio);
    cur_termio.c_lflag |= ISIG;
    ioctl(0, TCSETSW, &cur_termio);
}


static void waitvtactive() {
	struct stat chkbuf;
	struct vt_stat vtstat;
	int major, minor;
	int fd;

	/* get console number we are running in */
	fd = dup(0);
	fstat(fd, &chkbuf);
	major = chkbuf.st_rdev >> 8;
	minor = chkbuf.st_rdev & 0xff;	/* console number */

	if (major != 4 || minor >= 64) {
		printf("Not running in graphics-capable virtual console.\n");
		exit(-1);
	}
	close(fd);

	fd = dup(0);
	ioctl(fd, VT_WAITACTIVE, minor);
	close(fd);
}


static int get_perm()
{
	/* get I/O permissions for VGA registers */

	if (iopl(3)) {
		printf("VGAlib: Cannot get I/O permissions.\n");
		exit(-1);
	}
   
/*
    if (ioperm(CRT_IC, 1, 1)) {
	printf("VGAlib: can't get I/O permissions \n");
	exit (-1);
    }
    ioperm(CRT_IM,  1, 1);
    ioperm(ATT_IW, 1, 1);
    ioperm(GRA_I,  1, 1);
    ioperm(SEQ_I,  1, 1);
    ioperm(PEL_IW, 1, 1);
    ioperm(PEL_IR, 1, 1);
    ioperm(CRT_DC,  1, 1);
    ioperm(CRT_DM,  1, 1);
    ioperm(ATT_R,  1, 1);
    ioperm(GRA_D,  1, 1);
    ioperm(SEQ_D,  1, 1);
    ioperm(MIS_R,  1, 1);
    ioperm(MIS_W,  1, 1);
    ioperm(IS1_RC,  1, 1);
    ioperm(IS1_RM,  1, 1);
    ioperm(PEL_D,  1, 1);
*/
    /* ET4000 registers */
/*  ioperm(0x3bf,  1, 1);
    ioperm(0x3cc,  1, 1);
    ioperm(0x3d8,  1, 1);
    ioperm(0x3b8,  1, 1);
    ioperm(0x3c3,  1, 1);
    ioperm(0x3cd,  1, 1);
*/    
    /* Sierra Hicolor registers */
/*  ioperm(0x3c6, 1, 1);
*/

    /* color or monochrome text emulation? */
    color_text = port_in(MIS_R)&0x01;

    /* chose registers for color/monochrome emulation */
    if (color_text) {
	CRT_I = CRT_IC;
	CRT_D = CRT_DC;
	IS1_R = IS1_RC;
    } else {
	CRT_I = CRT_IM;
	CRT_D = CRT_DM;
	IS1_R = IS1_RM;
    }
}


static void delay() {
	int i;
	for (i = 0; i < 10; i++);
}

static int save_regs(unsigned char regs[])
{
    int i;

    /* save VGA registers */
    for (i = 0; i < CRT_C; i++) {
	 port_out(i, CRT_I); 
	 regs[CRT+i] = port_in(CRT_D); 
    }
    for (i = 0; i < ATT_C; i++) {
      	 port_in(IS1_R);
      	 delay();
         port_out(i, ATT_IW); 
         delay();
         regs[ATT+i] = port_in(ATT_R); 
         delay();
    }
    for (i = 0; i < GRA_C; i++) {
       	 port_out(i, GRA_I); 
       	 regs[GRA+i] = port_in(GRA_D); 
    }
    for (i = 0; i < SEQ_C; i++) {
       	 port_out(i, SEQ_I); 
       	 regs[SEQ+i] = port_in(SEQ_D); 
    }
    regs[MIS] = port_in(MIS_R); 

	chipset_saveregs(regs);		/* save chipset-specific registers */

}


int vga_setregs(unsigned char regs[])
{
    int i;

    /* disable video */
/*    port_in(IS1_R); */
/*    port_out(0x00, ATT_IW); */
  
    /* update misc output register */
    port_out(regs[MIS], MIS_W);         

    /* synchronous reset on */
    port_out(0x00,SEQ_I); 
    port_out(0x01,SEQ_D);	        
  
    /* write sequencer registers */
    for (i = 1; i < SEQ_C; i++) {       
	port_out(i, SEQ_I); 
	port_out(regs[SEQ+i], SEQ_D); 
    }

    /* synchronous reset off */
    port_out(0x00, SEQ_I); 
    port_out(0x03, SEQ_D);	        
  
    /* deprotect CRT registers 0-7 */
    port_out(0x11, CRT_I);		  
    port_out(port_in(CRT_D)&0x7F, CRT_D);   
  
    /* write CRT registers */
    for (i = 0; i < CRT_C; i++) {       
	port_out(i, CRT_I); 
	port_out(regs[CRT+i], CRT_D); 
    }

    /* write graphics controller registers */
    for (i = 0; i < GRA_C; i++) {       
	port_out(i, GRA_I); 
	port_out(regs[GRA+i], GRA_D); 
    }
     
    /* write attribute controller registers */
    for (i = 0; i < ATT_C; i++) {       
	port_in(IS1_R);   /* reset flip-flop */
	delay();
	port_out(i, ATT_IW);
	delay();
	port_out(regs[ATT+i], ATT_IW);
	delay();
    }
}

/* We invoke the old interrupt handler after setting text mode */

static struct sigaction old_SIGINT_handler;
static struct sigaction old_SIGFPE_handler;
static struct sigaction old_SIGSEGV_handler;

static void restoretextmode() {
	/* handle unexpected interrupts - restore text mode and exit */
	if (cur_mode != TEXT) 
		vga_setmode(TEXT);
 	set_texttermio();
}    

static void SIGINT_handler( int v ) {
	restoretextmode();
	printf("Interrupted (ctrl-c pressed)\n");
	sigaction(SIGINT, &old_SIGINT_handler, NULL);
	raise(SIGINT);
}

static void SIGFPE_handler( int v ) {
	restoretextmode();
	printf("Floating point exception or division by zero\n");
	sigaction(SIGFPE, &old_SIGFPE_handler, NULL);
	raise(SIGFPE);
}

static void SIGSEGV_handler( int v ) {
	restoretextmode();
	printf("Segment violation\n");
	sigaction(SIGSEGV, &old_SIGSEGV_handler, NULL);
	raise(SIGSEGV);
}

int getchipset() {
	if (chipset == UNDEFINED) {
		unsigned char *s;
		get_perm();
		#ifdef INCLUDE_ET4000_DRIVER
		if (et4000_chipsetfunctions[CHIPSET_TEST]())
			chipset = ET4000;
		else
		#endif
		#ifdef INCLUDE_TVGA_DRIVER
		if (tvga8900_chipsetfunctions[CHIPSET_TEST]())
			chipset = TVGA8900;
		else
		#endif
		#ifdef INCLUDE_CIRRUS_DRIVER
		/* The Cirrus detection is not very clean. */
		if (cirrus_chipsetfunctions[CHIPSET_TEST]())
			chipset = CIRRUS;
		else
		#endif
		#ifdef INCLUDE_OAK_DRIVER
		if (oak_chipsetfunctions[CHIPSET_TEST]())
			chipset = OAK;
		else
		#endif
		/* else */
			{
			chipset = VGA;
			chipsetfunctions = vga_chipsetfunctions;
			if (__driver_report)
				printf("Using VGA driver.\n");
			}
	}
	return chipset;
}


void vga_setchipset( int c ) {
	chipset = c;
	#ifdef DEBUG
	printf("Setting chipset\n");
	#endif
	if (chipset != UNDEFINED) {
		get_perm();
		chipsetfunctionslist[c][CHIPSET_INIT](0, 0, 0);
	}
}


void vga_setchipsetandfeatures( int c, int par1, int par2 ) {
	chipset = c;
	#ifdef DEBUG
	printf("Forcing chipset and features\n");
	#endif
	get_perm();
	chipsetfunctionslist[c][CHIPSET_INIT](1, par1, par2);
	#ifdef DEBUG
	printf("Finished forcing chipset and features\n");
	#endif
}


/* Virtual console switching */

static int forbidvtrelease = 0;
static int forbidvtacquire = 0;
static void takevtcontrol();

static void releasevt_signal( int n ) {
	struct sigaction siga;
	#ifdef DEBUG
	printf("Release request.\n");
	#endif
	forbidvtacquire = 1;
	SETSIG(siga, SIGUSR1, releasevt_signal);
	if (forbidvtrelease) {	
		forbidvtacquire = 0;
		ioctl(tty0_fd, VT_RELDISP, 0);
		return;
	}
	vga_flip();
	ioctl(tty0_fd, VT_RELDISP, 1);
	#ifdef DEBUG
	printf("Finished release.\n");
	#endif
	forbidvtacquire = 0;

	/* suspend program until switched to again */
	#ifdef DEBUG
	printf("Suspended.\n");
	#endif
	waitvtactive();
	#ifdef DEBUG
	printf("Waked.\n");
	#endif
} 

static void acquirevt_signal( int n ) {
	struct sigaction siga;
	#ifdef DEBUG
	printf("Acquisition request.\n");
	#endif
	forbidvtrelease = 1;
	SETSIG(siga, SIGUSR2, acquirevt_signal);
	if (forbidvtacquire) {
		forbidvtrelease = 0;
		return;
	}
	vga_flip();
	ioctl(tty0_fd, VT_RELDISP, 2);
	#ifdef DEBUG
	printf("Finished acquisition.\n");
	#endif
	forbidvtrelease = 0;
}

void takevtcontrol() {
    	struct vt_mode vtmode;
    	struct sigaction siga;

    	ioctl(tty0_fd, VT_GETMODE, &vtmode);
    	vtmode.mode = VT_PROCESS;	/* handle VT changes */
    	vtmode.relsig = SIGUSR1;	/* I didn't find SIGUSR1/2 anywhere */
    	vtmode.acqsig = SIGUSR2;	/* in the kernel sources, so I guess */
    					/* they are free */
    	ioctl(tty0_fd, VT_SETMODE, &vtmode);

	SETSIG(siga, SIGUSR1, releasevt_signal);
    	SETSIG(siga, SIGUSR2, acquirevt_signal);
}


static void initialize( int mode )
{
    int  i, j;
    struct sigaction siga;

	waitvtactive();

    /* save text mode termio parameters */
    ioctl(0, TCGETS, &text_termio);

    graph_termio = text_termio;

    /* change termio parameters to allow our own I/O processing */
    graph_termio.c_iflag &= ~(BRKINT|PARMRK|INPCK|IUCLC|IXON|IXOFF);
    graph_termio.c_iflag |=  (IGNBRK|IGNPAR);

    graph_termio.c_oflag &= ~(ONOCR);

    graph_termio.c_lflag &= ~(ICANON|ECHO|ECHOE|ECHOK|ECHONL|NOFLSH);
    graph_termio.c_lflag |= (ISIG);   /* enable interrupt */

    graph_termio.c_cc[VMIN] = 1;
    graph_termio.c_cc[VTIME] = 0;
    graph_termio.c_cc[VSUSP] = 0;   /* disable suspend */

    disable_interrupt();

	getchipset();		/* make sure a chipset has been selected */
	chipset_unlock();
	
    /* open /dev/tty0 - current virtual console */
    if ((tty0_fd = open("/dev/tty0", O_RDONLY) ) < 0) {
	printf("VGAlib: can't open /dev/tty0 \n");
	exit (-1);
    }
    console = fdopen(tty0_fd, "r"); 

    /* disable text output to console */
    ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);
    
    takevtcontrol();	/* HH: Take control over VT */

    /* open /dev/mem */
    if ((mem_fd = open("/dev/mem", O_RDWR) ) < 0) {
	printf("VGAlib: can't open /dev/mem \n");
	exit (-1);
    }


    /* mmap graphics memory */

    /* valloc allocates aligned on page boundary */
    if ((graph_mem = valloc(GRAPH_SIZE)) == NULL) {
	printf("VGAlib: allocation error \n");
	exit (-1);
    }

    /* this wastes 64K; mmap can probably do better by now. */
    graph_mem = (unsigned char *)mmap(
	(caddr_t)graph_mem, 
	GRAPH_SIZE,
	PROT_READ|PROT_WRITE,
	MAP_SHARED|MAP_FIXED,
	mem_fd, 
	GRAPH_BASE
    );
    if ((long)graph_mem < 0) {
	printf("VGAlib: mmap error \n");
	exit (-1);
    }

    /* disable video */
    port_in(IS1_R);	        
    port_out(0x00, ATT_IW);     
  
    save_regs(text_regs);

    /* save text mode palette - first select palette index 0 */
    port_out(0, PEL_IR);

    /* read RGB components - index is autoincremented */
    for (i = 0; i < 256; i++) {
    	delay();
	text_red[i] = port_in(PEL_D);
	delay();
	text_green[i] = port_in(PEL_D);
	delay();
	text_blue[i] = port_in(PEL_D);
    }

    /* shift to color emulation */
    CRT_I = CRT_IC;
    CRT_D = CRT_DC;
    IS1_R = IS1_RC;
    port_out(port_in(MIS_R)|0x01, MIS_W); 

    /* save font data - first select a 16 color graphics mode */
    chipsetfunctions[CHIPSET_SETMODE](G640x480x16);

    /* save font data in plane 2 */
    port_out(0x04, GRA_I); 
    port_out(0x02, GRA_D); 
    memcpy(font_buf1, graph_mem, FONT_SIZE);

    /* save font data in plane 3 */
    port_out(0x04, GRA_I); 
    port_out(0x03, GRA_D); 
    memcpy(font_buf2, graph_mem, FONT_SIZE);

    initialized = 1;

	/* do our own interrupt handling */
	SETSIG2(siga, SIGINT, SIGINT_handler, old_SIGINT_handler);

	/* Restore textmode after segment violation and math exception */
	/* I wonder why vgalib didn't do this */
    	SETSIG2(siga, SIGSEGV, SIGSEGV_handler, old_SIGSEGV_handler);
    	SETSIG2(siga, SIGFPE, SIGFPE_handler, old_SIGFPE_handler);

	if (mode == G320x200x256)
		__videomemoryused = 65536;
	else
	if (STDVGAMODE(mode))
		__videomemoryused = 256 * 1024;
	else {
		vga_modeinfo *modeinfo;
		modeinfo = vga_getmodeinfo(mode);
		__videomemoryused = modeinfo->linewidth * modeinfo->height;
	}
}


int vga_dumpregs()
{
    unsigned char regs[MAX_REGS];
    int i;

    get_perm();

    save_regs(regs);

    if (getchipset() == ET4000)
	printf("static unsigned char regs[70] = {\n  ");
    else
	printf("static unsigned char regs[60] = {\n  ");

    for (i = 0; i < 12; i++) 
        printf("0x%02X,", regs[CRT+i]);
    printf("\n  "); 
    for (i = 12; i < CRT_C; i++) 
        printf("0x%02X,", regs[CRT+i]);
    printf("\n  "); 
    for (i = 0; i < 12; i++) 
        printf("0x%02X,", regs[ATT+i]);
    printf("\n  "); 
    for (i = 12; i < ATT_C; i++) 
        printf("0x%02X,", regs[ATT+i]);
    printf("\n  "); 
    for (i = 0; i < GRA_C; i++) 
        printf("0x%02X,", regs[GRA+i]);
    printf("\n  "); 
    for (i = 0; i < SEQ_C; i++) 
        printf("0x%02X,", regs[SEQ+i]);
    printf("\n  "); 
    printf("0x%02X", regs[MIS]);

    if (getchipset() == ET4000) {
	printf(",\n  "); 
	for (i = 0; i < 9; i++) 
            printf("0x%02X,", regs[EXT+i]);
	printf("0x%02X", regs[EXT+9]);
    }

    printf("\n};\n"); 
}


static int setmode( int mode ) {
	if (chipset_setmode(mode))
		return 1;	/* cannot set mode */
	return 0;	
}


int vga_setmode(int mode) 
{
    int i;

    if (!initialized)
        initialize(mode);

    disable_interrupt();

    prv_mode = cur_mode;
    cur_mode = mode;

    /* disable video */
    port_in(IS1_R); 		
    port_out(0x00, ATT_IW);	

    if (getchipset() == ET4000 && prv_mode == G1024x768x256)
	chipset_setmode(G640x480x256);

    if (mode == TEXT) {
	if (SVGAMODE(prv_mode))
		vga_setpage(0);

	chipset_setregs(text_regs);	/* restore old extended regs */

        /* restore font data - first select a 16 color graphics mode */
        chipsetfunctions[CHIPSET_SETMODE](G640x480x16);

	/* disable Set/Reset Register */
    	port_out(0x01, GRA_I ); 
    	port_out(0x00, GRA_D );   

        /* restore font data in plane 2 - necessary for all VGA's */
    	port_out(0x02, SEQ_I ); 
    	port_out(0x04, SEQ_D );   
	memcpy(graph_mem, font_buf1, FONT_SIZE);

        /* restore font data in plane 3 - necessary for Trident VGA's */
    	port_out(0x02, SEQ_I ); 
    	port_out(0x08, SEQ_D );   
	memcpy(graph_mem, font_buf2, FONT_SIZE);

        /* change register adresses if monochrome text mode */
        if (!color_text) {
            CRT_I = CRT_IM;
            CRT_D = CRT_DM;
            IS1_R = IS1_RM;
            port_out(port_in(MIS_R)&0xFE, MIS_W); 
        }

        /* restore saved palette */
        for(i = 0; i < 256; i++)
            vga_setpalette(
                i, text_red[i], text_green[i], text_blue[i]
            );


	/* restore text mode VGA registers */
    	vga_setregs(text_regs);

	/* memset(graph_mem, 0, 132 * 60 * 2); */

    	if (!flip)
		/* enable text output - restores the screen contents */ 
	      	ioctl(tty0_fd, KDSETMODE, KD_TEXT);

	usleep(MODESWITCHDELAY);	/* wait for signal to stabilize */
	
        /* enable video */
        port_in(IS1_R); 
        port_out(0x20, ATT_IW); 
  
  	if (!flip)
	        /* restore text mode termio */
        	set_texttermio();

    } else {
    	if (!flip)
		/* disable text output */
        	ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);

        /* shift to color emulation */
        CRT_I = CRT_IC;
        CRT_D = CRT_DC;
        IS1_R = IS1_RC;
        port_out(port_in(MIS_R)|0x01, MIS_W); 

	setmode(mode);
        cur_info.xdim = infotable[mode].xdim;
        cur_info.ydim = infotable[mode].ydim;
        cur_info.colors = infotable[mode].colors;
        cur_info.xbytes = infotable[mode].xbytes;

        if (!flip) {
            	/* set default palette */
            	if (cur_info.colors <= 256)
	        	for (i = 0; i < 256; i++)
        	        	vga_setpalette(
                	    	i, default_red[i], default_green[i],
                	    	default_blue[i]);
 
            	/* clear screen (sets current color to 15) */
            	currentpage = -1;
            	vga_clear();
        }

        currentpage = -1;
        currentlogicalwidth = cur_info.xbytes;
        currentdisplaystart = 0;

	usleep(MODESWITCHDELAY);	/* wait for signal to stabilize */

        /* enable video */
        port_in(IS1_R); 
        port_out(0x20, ATT_IW); 
  
  	if (!flip)
	        /* set graphics mode termio */
        	set_graphtermio();
        else
        	enable_interrupt();

    }

    return 0;  
}

void vga_gettextfont( void *font ) {
	memcpy(font, font_buf1, FONT_SIZE);
}

void vga_puttextfont( void *font ) {
	memcpy(font_buf1, font, FONT_SIZE);
	memcpy(font_buf2, font, FONT_SIZE);
}

void vga_gettextmoderegs( void *regs ) {
	memcpy(regs, text_regs, MAX_REGS);
}

void vga_settextmoderegs( void *regs ) {
	memcpy(text_regs, regs, MAX_REGS);
}

int vga_getcurrentmode() {
	return cur_mode;
}

int vga_getcurrentchipset() {
	return getchipset();
}

void vga_disabledriverreport() {
	__driver_report = 0;
}

static vga_modeinfo modeinfo;

vga_modeinfo *vga_getmodeinfo( int mode ) {
	getchipset();
	modeinfo.width = infotable[mode].xdim;
	modeinfo.height = infotable[mode].ydim;
	modeinfo.bytesperpixel = infotable[mode].xbytes / modeinfo.width;
	modeinfo.colors = infotable[mode].colors;
	modeinfo.linewidth = infotable[mode].xbytes;
	if (modeinfo.bytesperpixel == 0) /* 16-color or planar 256-color */
		vga_chipsetfunctions[CHIPSET_GETMODEINFO](mode, &modeinfo);
	else
		chipset_getmodeinfo(mode, &modeinfo);
			/* get chipset specific info */
	return &modeinfo;
}

int vga_hasmode(int mode)
{
	getchipset();
	return chipset_modeavailable(mode);
}

int vga_setrgbcolor( int r, int g, int b ) {
	switch (infotable[cur_mode].colors) {
	case 32768 :
		cur_color =
			(b >> 3) +
			((g >> 3) << 5) +
			((r >> 3) << 10);
		break;
	case 65536 :
		cur_color =
			(b >> 3) +
			((g >> 2) << 5) +
			((r >> 3) << 11);
		break;
	case 65536 * 256 :
		cur_color = b + (g << 8) + (r << 16);
		break;
	}
	return cur_color;
}


int vga_setcolor(int color)
{
    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G720x350x16:
        case G720x480x16:
            /* update set/reset register */
	    port_out(0x00, GRA_I ); 
	    port_out(color, GRA_D );   
            break;
        case G640x480x2:
	    if (color != 0)
		color = 15;
            /* update set/reset register */
	    port_out(0x00, GRA_I ); 
	    port_out(color, GRA_D );   
            break;
	case G320x200x256:
	case G320x240x256:
	case G320x400x256:
	case G360x480x256:
	case G640x480x256:
	case G800x600x256:
	case G1024x768x256:
        case G640x480x32K:
        case G640x480x64K:
        case G800x600x32K:
        case G800x600x64K:
        case G1024x768x32K:
        case G1024x768x64K:
        case G640x480x16M:
        case G800x600x16M:
        case G1024x768x16M:
        case G320x200x32K:
        case G320x200x64K:
        case G320x200x16M:
	    cur_color = color;
            break;
    }

    return 0;
}


int vga_screenoff()
{
    /* turn off screen for faster VGA memory acces */
    port_out(0x01, SEQ_I); 		  
    port_out(port_in(SEQ_D)|0x20, SEQ_D);   

    return 0;
}


int vga_screenon()
{
    /* turn screen back on */
    port_out(0x01, SEQ_I); 		  
    port_out(port_in(SEQ_D)&0xDF, SEQ_D);   

    return 0;
}


int vga_getxdim()
{
    return cur_info.xdim;
}


int vga_getydim()
{
    return cur_info.ydim;
}


int vga_getcolors()
{
    return cur_info.colors;
}

int vga_claimvideomemory( int m ) {
	vga_modeinfo *modeinfo;
	int cardmemory;
	modeinfo = vga_getmodeinfo(cur_mode);
	if (m < __videomemoryused)
		return 0;
	__videomemoryused = m;
	cardmemory = (modeinfo->maxpixels * modeinfo->bytesperpixel + 2) &
		0xffff0000;
	/* maxpixels * bytesperpixel can be 2 less than video memory in */
	/* 3 byte-per-pixel modes; assume memory is multiple of 64K */
	if (__videomemoryused > cardmemory)
		return -1;
	return 0;
} 

static int savepalette() {
	int i, j;
    	/* save graphics mode palette - first select palette index 0 */
    	port_out(0, PEL_IR); 

    	/* read RGB components - index is autoincremented */
    	for (i = 0; i < 256; i++) {
            for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_red[i] = port_in(PEL_D);
    	    for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_green[i] = port_in(PEL_D);
    	    for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_blue[i] = port_in(PEL_D);
        }
}

static int restorepalette() {
	int i;
        /* restore saved palette */
        for (i = 0; i < 256; i++)
        	vga_setpalette(i, graph_red[i], graph_green[i], graph_blue[i]);
}

static int saved_page;
static int saved_logicalwidth;
static int saved_displaystart;

static void savestate() {
	int i;

	vga_screenoff();

	savepalette();

	saved_page = currentpage;
	saved_logicalwidth = currentlogicalwidth;
	saved_displaystart = currentdisplaystart;

	if (cur_mode == G320x200x256 && __videomemoryused <= 65536) {
		/* 320x200x256 is a special case; only 64K is addressable */
		/* (unless more has been claimed, in which case we assume */
		/* SVGA bank-switching) */
        	if ((graph_buf = malloc(GRAPH_SIZE)) == NULL) {
        		printf("Cannot allocate memory for VGA state\n");
        		vga_setmode(TEXT);
			exit (-1);
	        }
	        memcpy(graph_buf, graph_mem, GRAPH_SIZE);
	}
	else
	if (cur_mode > TEXT && STDVGAMODE(cur_mode) && cur_mode != G320x200x256) {
		/* for planar VGA modes, save the full 256K */
		vga_chipsetfunctions[CHIPSET_SETMODE](G640x480x16);
        	if ((graph_buf = malloc(4*GRAPH_SIZE)) == NULL) {
        		printf("Cannot allocate memory for VGA state\n");
        		vga_setmode(TEXT);
			exit (-1);
	        }

	        for (i = 0; i < 4; i++) {
        		/* save plane i */
	            	port_out(0x04, GRA_I); 
           		port_out(i, GRA_D); 
            		memcpy(graph_buf + i*GRAPH_SIZE, graph_mem, GRAPH_SIZE);
		}
	}
	else {	/* SVGA, and SVGA 320x200x256 if videomemoryused > 65536 */
        	int size;
        	int page;
		size = __videomemoryused;

		#ifdef DEBUG
		printf("Saving %dK of video memory.\n", (size + 2) / 1024);
		#endif
        	if ((graph_buf = malloc(size)) == NULL) {
        		printf("Cannot allocate memory for SVGA state.\n");
        		vga_setmode(TEXT);
			exit(-1);
	        }
	        page = 0;
	        while (size >= 65536) {
	        	vga_setpage(page);
	        	memcpy(graph_buf + page * 65536, graph_mem, 65536);
	        	page++;
	        	size -= 65536;
	        }
	        if (size > 0) {
	        	vga_setpage(page);
	        	memcpy(graph_buf + page * 65536, graph_mem, size);
	        }
	}
}

static void restorestate() {
	int i;

	vga_screenoff();

	restorepalette();

	if (cur_mode == G320x200x256 && __videomemoryused <= 65536) {
		memcpy(graph_mem, graph_buf, 65536);
	}
	else
	if (cur_mode > TEXT && STDVGAMODE(cur_mode) && cur_mode != G320x200x256) {
		int setresetreg, planereg;
		/* disable Set/Reset Register */
    		port_out(0x01, GRA_I); 
    		setresetreg = inb(GRA_D);
	    	port_out(0x00, GRA_D);   
	    	outb(SEQ_I, 0x02);
	    	planereg = inb(SEQ_D);

	        for (i = 0; i < 4; i++) {
        		/* restore plane i */
	    		port_out(0x02, SEQ_I ); 
    			port_out(1<<i, SEQ_D );   
            		memcpy(graph_mem, graph_buf + i * GRAPH_SIZE,
	            		GRAPH_SIZE);
		}
		outb(GRA_I, 0x01);
		outb(GRA_D, setresetreg);
		outb(SEQ_I, 0x02);
		outb(SEQ_D, planereg);
	}
	else {
        	vga_modeinfo *modeinfo;
        	int size;
        	int page;
		size = __videomemoryused;

		#ifdef DEBUG
		printf("Restoring %dK of video memory.\n", (size + 2) / 1024);
		#endif
	        page = 0;
	        while (size >= 65536) {
	        	vga_setpage(page);
	        	memcpy(graph_mem, graph_buf + page * 65536, 65536);
	        	size -= 65536;
	        	page++;
	        }
	        if (size > 0) {
	        	vga_setpage(page);
	        	memcpy(graph_mem, graph_buf + page * 65536, size);
	        }
	}
	
	if (saved_logicalwidth != cur_info.xbytes)
		vga_setlogicalwidth(saved_logicalwidth);
	if (saved_page != 0)
		vga_setpage(saved_page);
	if (saved_displaystart != 0)
		vga_setdisplaystart(saved_displaystart);

	vga_screenon();

	free(graph_buf);
}


int vga_getch()
{
	char c;
	int oldmode;

	if (cur_mode == TEXT)
	return -1;

	read(tty0_fd, &c, 1);
/*
	do {
		read(tty0_fd, &c, 1);

		if (c == flipchar) {
			vga_flip();
			set_graphtermio();
			printf("Graphics application suspended.\n");
			read(tty0_fd, &c, 1);
			vga_flip();
			read(tty0_fd, &c, 1);
		}
	} while (c == 0);
*/
	return c;
}

int vga_flip() {
	if (cur_mode != TEXT) {		/* save state and go to textmode */
		savestate();
		flip_mode = cur_mode;
		flip = 1;
		vga_setmode(TEXT);
		flip = 0;
		return 0;
	}
	else {				/* restore graphics mode and state */
		flip = 1;
		vga_setmode(flip_mode);
		flip = 0;
		restorestate();
		return 0;
	}
}


int vga_setflipchar(int c)
{
    flipchar = c;

    return 0;
}


#if 0		/* old vga_flip from VGAlib v1.2 */
int vga_flip()
{
    int i, j;

    if (cur_mode == TEXT && flip_mode == TEXT)
	return -1;

    flip = 1;

    disable_interrupt();

    /* disable video */
    port_in(IS1_R); 		
    port_out(0x00, ATT_IW);	
  
    if (cur_mode == TEXT) {
	/* disable text output */
        ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);

        /* restore all four planes - first select a 16 color graphics mode */
        chipsetfunctions[CHIPSET_SETMODE](G640x480x16);

	/* disable Set/Reset Register */
    	port_out(0x01, GRA_I ); 
    	port_out(0x00, GRA_D );   

        for(i = 0; i < 4; i++) {
            /* restore plane i */
    	    port_out(0x02, SEQ_I ); 
    	    port_out(1<<i, SEQ_D );   
            memcpy(graph_mem, graph_buf + i*GRAPH_SIZE, GRAPH_SIZE);
	}

        free(graph_buf);

        /* restore saved palette */
        for(i = 0; i < 256; i++)
            vga_setpalette(
                i, graph_red[i], graph_green[i], graph_blue[i]
            );

        chipsetfunctions[CHIPSET_SETMODE](flip_mode);

	flip_mode = TEXT;
    } else {
    
    	/* temporary hack */
	chipset_setregs(text_regs);	/* restore old extended regs */

	/* save all four planes - first select a 16 color graphics mode */
        chipsetfunctions[CHIPSET_SETMODE](G640x480x16);

	/* allocate memory for saved graphics data - only if necessary */
        if ((graph_buf = malloc(4*GRAPH_SIZE)) == NULL) {
            printf("vga_flip: allocation error \n");
            exit (-1);
        }

        for(i = 0; i < 4; i++) {
            /* save plane i */
            port_out(0x04, GRA_I); 
            port_out(   i, GRA_D); 
            memcpy(graph_buf + i*GRAPH_SIZE, graph_mem, GRAPH_SIZE);
	}

    	/* save graphics mode palette - first select palette index 0 */
    	port_out(0, PEL_IR); 

    	/* read RGB components - index is autoincremented */
    	for(i = 0; i < 256; i++) {
            for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_red[i] = port_in(PEL_D);
    	    for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_green[i] = port_in(PEL_D);
    	    for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */ 
            graph_blue[i] = port_in(PEL_D);
        }

        flip_mode = cur_mode;

	chipsetfunctions[CHIPSET_SETMODE](TEXT);
    }

    flip = 0;

    return 0;
}
#endif

void vga_setpage( int p ) {
	if (p == currentpage)
		return;
	chipset_setpage(p);
	currentpage = p;
}

void vga_waitretrace() {
	while (!(inb(0x3da) & 8));
	while (inb(0x3da) & 8);
}

void vga_setlogicalwidth( int w ) {
	chipsetfunctions[CHIPSET_SETLOGICALWIDTH](w);
	currentlogicalwidth = w;
}

void vga_setdisplaystart( int a ) {
	chipsetfunctions[CHIPSET_SETDISPLAYSTART](a);
	currentdisplaystart = a;
}

void vga_bitblt( int srcaddr, int destaddr, int w, int h, int pitch ) {
	chipsetfunctions[CHIPSET_BITBLT](srcaddr, destaddr, w, h, pitch);
}

void vga_imageblt( void *srcaddr, int destaddr, int w, int h, int pitch ) {
	chipsetfunctions[CHIPSET_IMAGEBLT](srcaddr, destaddr, w, h, pitch);
}

void vga_fillblt( int destaddr, int w, int h, int pitch, int c ) {
	chipsetfunctions[CHIPSET_FILLBLT](destaddr, w, h, pitch, c);
}

void vga_hlinelistblt( int ymin, int n, int *xmin, int *xmax, int pitch,
int c ) {
	chipsetfunctions[CHIPSET_HLINELISTBLT](ymin, n, xmin, xmax, pitch, c);
}