/* VGAlib version 1.2 - (c) 1993 Tommy Frandsen 		   */
/*								   */
/* This library is free software; you can redistribute it and/or   */
/* modify it without any restrictions. This library is distributed */
/* in the hope that it will be useful, but without any warranty.   */

/* Cirrus support Copyright 1993 Harm Hanemaayer */


#include <stdio.h>	/* for printf */
#include "vga.h"
#include "vgamacros.h"
#include "chipsets.h"


/*
	Cirrus registers addition:	
			4 extra CRT regs 
		    	3 extra Graphics regs 
			26 extra Sequencer regs (last one is not changed)
			Hicolor DAC register

	For SVGA modes, if a 5426 is present, bit 5 of Graphics register 0x0b
	is set to enable 16K bank granularity (the default is 4K). This way,
	2Mb of video memory is addressable.

	Currently the default (lowest) dot clocks (and frequencies) are used
	for each mode. All Cirrus cards support higher clocks for most modes;
	up to 75 MHz, which is not that high. But 90 MHz is available
	on 5426's at least; it is only used for 1024x768x16bit interlaced
	with my BIOS, but it is not documented, so I don't know for sure if
	it is available	for any other mode. In the following table, dot clock
	is in Mbytes/sec.

			Default		BIOS max.	Hypothetical max.
			Dot	Freq	Dot 	Freq	Dot	Freq
	640x480x256	25 MHz	60 Hz	31.5	72		90
	640x480x32k	50	60	63	72		90
	640x480x16M	75	60	75	60	90	72
	800x600x256	36	56	50	72		90
	800x600x32k	72	56	72	56	90	68
	800x600x16M	96.4	50
	800x600x16M*					90	90i
	1024x768x256	45	87i	75	70	90	84
	1024x768x32k	90	87i	90	87i	90	87i
	1280x1024x256	75	87i	75	87i
	1150x910x256					90	~60
	* - very hypothetical

	Note that the 90 MHz dot clock would make 640x480x16M and 800x600x32k
	flicker free (i.e. really usable), although it slows down the card
	a lot.

	The Hicolor DAC register seems have the following values:

	0x00	256 color
	0xf0	32K color
	0xe1	64K color
	0xe5	16M color 

	The dot clock must be multiplied by the number of bytes per pixel.
*/


#define CIRRUS_HIGHDOTCLOCK 1
/*
	0 : no 800x600x16M NI
	1 : 800x600x16M NI at 49.7 Hz (dotclock 96.1 MHz)
		Horizontal Sync: 31.5 KHz
		This is a bit flickery of course, and needs some
		adjusting on a fixed frequency monitor.
	2 : 800x600x16M NI at 56 Hz (dotclock 108.3 MHz)
		Horizontal Sync: 35.5 KHz
		Most cards probably cannot handle this dot clock (my card
		manages up to 100 MHz).

	800x600x16M interlaced at 90Hz (dot clock 90 MHz) would be easier on
	the hardware (and the eye, probably), but is hard to define.
*/

static int cirrus_init(int, int, int);

static int cirrus_memory;
static int cirrus_chiptype;

enum { CLGD5420 = 0, CLGD5422, CLGD5424, CLGD5426, CLGD5428 };


static unsigned char cirrus_g640x480x256_regs[95] = {
	/* CRTC */
	0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
	0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x50,0x60,0xE7,0x04,0xAB,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0xe3,	/* use standard VGA dot clock 25 MHz */
	/* extra CRT registers: */
	0x00,0x00,0x00,
	0x22,	/* bit 4 is high bit of scanline length */
	/* extra graphics registers: */
	0x00,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x01,	/* (0x07) indicates number of colors */
	0x80,
	0x01,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	0x4a,	/* (0x0e) dot clock */
	0x30,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x2b,	/* (0x1e) dot clock */
	0x1c,	/* Internal memory clock. Not changed. */
	0x00	/* Hicolor DAC */
};

static unsigned char cirrus_g640x480x32k_regs[95] = {
	/* CRTC */
	0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
	0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,
	0xa0, /* (0x13) bytes in scanline (was 0x50) */
	0x60,0xE7,0x04,0xAB,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x07,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0xEF,	/* use VLCK3 (was 0xE3: VGA 25Mhz) */	
	/* extra CRT registers: */
	0x00,0x00,0x00,
	0x22, /* bit 0: ?; bit 2: ? (frame); bit 6: wrap at 256K */
	/* extra graphics registers: */
	0x20,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x03,	/* (0x07) indicates number of colors */
	0x80,
	0x0d,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	0x65,	/* (0x0e) dot clock (bytes): 49.87 MHz */
	0x70,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x3a,	/* (0x1e) dot clock */
	0x1c,
	0xf0	/* Hicolor DAC */
};

static unsigned char cirrus_g640x480x16m_regs[95] = {
	/* CRTC */
	0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
	0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,
	0xf0, /* (0x13) bytes in scanline / 8 */
	0x60,0xE7,0x04,0xAB,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,
	0x00,	/* 0x40 for other modes */
	0x05,
	0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0xEF,	/* use VLCK3 (was 0xE3: VGA 25Mhz) */	
	/* extra CRT registers: */
	0x00,0x00,0x00,
	0x22, /* bit 0: ?; bit 2: ? (frame); bit 6: wrap at 256K */
	/* extra graphics registers: */
	0x20,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x05,	/* (0x07) indicates number of colors */
	0x80,
	0x0d,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	0x3a,	/* (0x0e) dot clock (bytes): 75 MHz*/
	0x70,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,
	0xdd,	/* 0xd8 for other modes */
	0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x16,	/* (0x1e) dot clock */
	0x1c,
	0xe5	/* Hicolor DAC */
};

static unsigned char cirrus_g800x600x256_regs[95] = {
	/* CRTC */
	0x7b,0x63,0x64,0x9e,0x69,0x92,0x6f,0xf0,0x00,0x60,0x00,0x00,
	0x00,0x00,0x02,0xbc,0x58,0x8a,0x57,0x64,0x00,0x58,0x6f,0xe3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0x2f,
	/* extra CRT registers: */
	0xff,
	0x00,	/* interlaced: CR0 / 2? */
	0x00,	/* sync overflow bits; bit 0 set for interlaced */
	0x22,	/* note: bit 4 is high bit of logical scanline length */
	/* extra graphics registers: */
	0x00,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,0x01,0x80,
	0x05,	/* (0x09) monitor type */
	0x19,0x4a,0x5b,0x45,
	0x7e,	/* (0x0e) dot clock: 36 MHz */
	0x30,0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x33,	/* (0x1e) dot clock */
	0x1c,
	0x00	/* Hicolor DAC */
};

static unsigned char cirrus_g800x600x32k_regs[95] = {
	/* CRTC */
	0x7b,0x63,0x64,0x9e,0x69,0x92,0x6f,0xf0,0x00,0x60,0x00,0x00,
	0x00,0x00,0x02,0xbc,0x58,0x8a,0x57,
	0xc8,	/* bytes in scanline / 8 */
	0x00,0x58,0x6f,0xe3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x07,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0xef,
	/* extra CRT registers: */
	0xff,
	0x00,	/* interlaced: CR0 / 2? */
	0x00,	/* sync overflow bits; bit 0 set for interlaced */
	0x22,	/* note: bit 4 is high bit of logical scanline length */
	/* extra graphics registers: */
	0x10,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,0x03,0x80,
	0x0d,	/* (0x09) monitor type */
	0x19,0x4a,0x5b,0x45,
	0x7e,	/* (0x0e) dot clock: 72 MHz */
	0x30,0x00,0x00,
	0x00,0x00,0x00,0x00,0xdd,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x32,	/* (0x1e) dot clock */
	0x1c,
	0xf0	/* Hicolor DAC */
};

static unsigned char cirrus_g800x600x16m_regs[95] = {
	/* CRTC */
	0x7b,0x63,0x64,0x9e,0x69,0x92,0x6f,0xf0,0x00,0x60,0x00,0x00,
	0x00,0x00,0x02,0xbc,0x58,0x8a,0x57,
	0x2c, /* (0x13) bytes in scanline / 8 */
	0x00,0x58,0x6f,0xe3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,
	0x40,
	0x05,
	0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0xEF,	/* use VLCK3 */	
	/* extra CRT registers: */
	0x00,0x00,0x00,
	0x32, /* bit 4: high bit of logical scanline length (set) */
	/* extra graphics registers: */
	0x20,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x05,	/* (0x07) indicates number of colors */
	0x80,
	0x0d,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	(CIRRUS_HIGHDOTCLOCK == 1 ? 0x5e : 0x52),
		/* (0x0e) dot clock (bytes): 96.1 MHz / 108.3 MHz */
	0xb0,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,
	0xdd,
	0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	(CIRRUS_HIGHDOTCLOCK == 1 ? 0x1c : 0x16),
		/* (0x1e) dot clock: 96.1 MHz / 108.3 MHz */
	0x1c,
	0xe5	/* Hicolor DAC */
};

/* Interlaced */
static unsigned char cirrus_g1024x768x256_regs[95] = {
	/* CRTC */
	0x99,0x7F,0x80,0x9c,0x83,0x19,0x96,0x1f,0x00,
	0x40,
	0x00,0x00,
	0x00,0x00,0x03,0x80,0x80,0x84,0x7F,0x80,0x00,0x80,0x96,0xE3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0x2f,
	/* extra CRT registers: */
	0xff,0x4a,0x01,0x22,
	/* extra graphics registers: */
	0x10,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,0x01,0x80,0x04,0x19,0x4a,0x5b,0x45,
	0x55, 	/* (0x0e) dot clock: 45 MHz */
	0x30,0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x36,	/* (0x1e) dot clock */
	0x1c,
	0x00	/* Hicolor DAC */
};

/* Interlaced */
static unsigned char cirrus_g1024x768x32k_regs[95] = {
	/* CRTC */
	0x99,0x7F,0x80,0x9c,0x83,0x19,0x96,0x1f,0x00,
	0x40,
	0x00,0x00,
	0x00,0x00,0x03,0x80,0x80,0x84,0x7F,0x00,0x00,0x80,0x96,0xE3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x07,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0x2f,
	/* extra CRT registers: */
	0xff,0x4a,0x01,0x32,
	/* extra graphics registers: */
	0x00,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,0x07,0x80,0x0d,0x19,0x4a,0x5b,0x45,
	0x55, 	/* (0x0e) dot clock: 45 MHz */
	0xb0,0x00,0x00,
	0x00,0x00,0x00,0x00,0xdf,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x36,	/* (0x1e) dot clock */
	0x1c,
	0xf0	/* Hicolor DAC */
};

static unsigned char cirrus_g320x200x32k_regs[95] = {
	/* CRTC */
	0x2d,0x27,0x27,0x91,0x2a,0x90,0xbf,0x1f,0x00,0xc0,0x00,0x00,
	0x00,0x00,0x01,0x90,0x9c,0x8e,0x8f,0x50,0x00,0x97,0xb8,0xe3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x40,0x07,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0x63,
	/* extra CRT registers: */
	0xff,0x00,0x00,0x22,
	/* extra graphics registers: */
	0x00,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x03,	/* (0x07) indicates number of colors */
	0x80,
	0x01,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	0x7e,	/* (0x0e) dot clock */
	0x30,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x33,	/* (0x1e) dot clock */
	0x1c,
	0xf0	/* Hicolor DAC */
};

static unsigned char cirrus_g320x200x16m_regs[95] = {
	/* CRTC */
	0x2d,0x27,0x27,0x91,0x2a,0x90,0xbf,0x1f,0x00,0xc0,0x00,0x00,
	0x00,0x00,0x01,0x90,0x9c,0x8e,0x8f,0x78,0x00,0x97,0xb8,0xe3,
	/* ATC */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
	0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
	/* Graphics */
	0x00,0x00,0x00,0x00,0x00,0x00,0x05,0x0F,0xFF,
	/* Sequencer */
	0x03,0x01,0x0F,0x00,0x0E,
	/* misc. output */
	0x6f,
	/* extra CRT registers: */
	0xff,0x00,0x00,0x22,
	/* extra graphics registers: */
	0x00,0x00,0x00,
	/* extra sequencer registers: */
	0x0f,0x12,
	0x05,	/* (0x07) indicates number of colors */
	0x80,
	0x01,	/* (0x09) bits 2-4: monitor type */
	0x19,0x4a,0x5b,0x45,
	0x1d,	/* (0x0e) dot clock */
	0x30,	/* (0x0f) bits 3-4: active video memory?, bit 7: use 2Mb */
	0x00,0x00,
	0x00,0x00,0x00,0x00,0xd8,0x00,0x00,0x01,0x00,0x2b,0x2f,0x30,
	0x16,	/* (0x1e) dot clock */
	0x1c,
	0xe5	/* Hicolor DAC */
};



/* Mode table */

static unsigned char *cirrus_moderegs[] = {
	cirrus_g640x480x256_regs, cirrus_g800x600x256_regs,
	cirrus_g1024x768x256_regs,
	cirrus_g320x200x32k_regs, cirrus_g320x200x32k_regs,
	cirrus_g320x200x16m_regs,
	cirrus_g640x480x32k_regs, cirrus_g640x480x32k_regs,
	cirrus_g640x480x16m_regs,
	cirrus_g800x600x32k_regs, cirrus_g800x600x32k_regs,
	cirrus_g800x600x16m_regs,
	cirrus_g1024x768x32k_regs, cirrus_g1024x768x32k_regs
};

 
/* Fill in chipset specific mode information */

static int cirrus_getmodeinfo( int mode, vga_modeinfo *modeinfo ) {
	modeinfo->maxpixels = cirrus_memory * 1024 / modeinfo->bytesperpixel;
	modeinfo->maxlogicalwidth = 4088;
	if (mode != G320x200x256) {
		modeinfo->startaddressrange = 0x1fffff;
		if (cirrus_chiptype >= CLGD5426 &&
		__videomemoryused <= (cirrus_memory - 16384)) {
			modeinfo->haveblit = HAVE_FILLBLIT | HAVE_BITBLIT |
				HAVE_IMAGEBLIT;
			/* 16K is used for 8x8 fill patterns */
			modeinfo->maxpixels = (cirrus_memory * 1024 - 16384)
				/ modeinfo->bytesperpixel;
		}
		else
			modeinfo->haveblit = 0;
	}
	else {
		modeinfo->startaddressrange = 0;
		modeinfo->maxpixels = 65536;
		modeinfo->haveblit = 0;
	}
}


/* Read and save chipset-specific registers */

static int cirrus_saveregs( unsigned char regs[] ) {
    	int i;

/*	#ifdef DEBUG
	printf("Saving Cirrus extended registers.\n");
	#endif
*/	
	/* save extended CRTC registers */
	for (i = 0; i < 4; i++) {
		port_out(0x18 + i, CRT_I);
		regs[EXT + i] = port_in(CRT_D);
	}
	
	/* save extended graphics registers */
	for (i = 0; i < 3; i++) {
		port_out(0x09 + i, GRA_I);
		regs[EXT + i + 4] = port_in(GRA_D);
	}
	
    	/* save extended sequencer registers */
    	/* used to be 27, but last register indicates internal memory clock */
    	/* (you can now take benefit from a higher speed set with a DOS */
    	/* driver disk program) */
    	for (i = 0; i < 26; i++) {
	 	port_out(0x05 + i, SEQ_I); 
	 	regs[EXT + i + 7] = port_in(SEQ_D); 
    	}

    	/* save Hicolor DAC register */
    	inb(0x3c8);
    	inb(0x3c6); inb(0x3c6); inb(0x3c6); inb(0x3c6);
    	regs[EXT + 34] = inb(0x3c6);
}


static void writehicolordac( unsigned char c ) {
    	inb(0x3c8);
    	inb(0x3c6); inb(0x3c6); inb(0x3c6); inb(0x3c6);
    	outb(0x3c6, c);
    	inb(0x3c8);
}


/* Set chipset-specific registers */

static int cirrus_setregs( unsigned char regs[] )
{
    	int i;
    	
/*	#ifdef DEBUG
	printf("Setting Cirrus extended registers.\n");
 	#endif
*/ 
 	/* write extended CRTC registers */
	for (i = 0; i < 4; i++) {
		port_out(0x18 + i, CRT_IC);
		port_out(regs[EXT + i], CRT_DC);
	}
	
	/* write extended graphics registers */
	for (i = 0; i < 3; i++) {
		port_out(0x09 + i, GRA_I);
		port_out(regs[EXT + i + 4], GRA_D);
	}
   	
    	/* write extended sequencer registers */
    	for (i = 0; i < 26; i++) {
    		port_out(0x05 + i, SEQ_I); 
    		port_out(regs[EXT + i + 7], SEQ_D); 
	}

    	/* write Hicolor DAC register */
    	writehicolordac(regs[EXT + 34]);
}


/* Return nonzero if mode is available */

static int cirrus_modeavailable( int mode ) {
	if (STDVGAMODE(mode))
		return 1;	/* standard modes are always available */
	switch (mode) {
	case G640x480x256 :
	case G800x600x256 :
	case G320x200x32K :
	case G320x200x64K :
	case G320x200x16M :
		return (cirrus_memory >= 512);
	case G1024x768x256 :
	case G640x480x32K :
	case G640x480x64K :
	case G640x480x16M :
	case G800x600x32K :
	case G800x600x64K :
		return (cirrus_memory >= 1024);
	case G1024x768x32K :
	case G1024x768x64K :
		return (cirrus_memory == 2048);
	case G800x600x16M :
		return (cirrus_memory == 2048 && CIRRUS_HIGHDOTCLOCK);
	default :
		return 0;
	}
}


/* Set a mode */

static int cirrus_setmode( int mode ) {
	if (!cirrus_modeavailable(mode))
		return 1;	/* mode not available */
	if (STDVGAMODE(mode))
		/* Let the standard VGA driver set standard VGA modes */
		vga_chipsetfunctions[CHIPSET_SETMODE](mode);
	else {
		unsigned char *regs = cirrus_moderegs[mode - G640x480x256];
		vga_setregs(regs);	/* set standard regs */
		cirrus_setregs(regs);	/* set extended regs */
		if (cirrus_memory == 2048) {
			outb(0x3c4, 0x0f);	/* enable 2M */
			outb(0x3c5, inb(0x3c5) | 0x80);
			outb(0x3ce, 0x0b);
			outb(0x3cf, 0x20);	/* set 16K bank granularity */
		}
		switch (mode) {
			/* For 64K color modes, we set the 32K registers to */
			/* save space. Now set 64K color mode. */
			case G640x480x64K :
			case G800x600x64K :
			case G1024x768x64K :
			case G320x200x64K :
				writehicolordac(0xe1);	/* 64K color mode */
				break;
		}
	}
}


/* Unlock chipset-specific registers */

static int cirrus_unlock() {
	int vgaIOBase, temp;
	
				/* Are we Mono or Color? */
       	vgaIOBase = (inb(0x3CC) & 0x01) ? 0x3D0 : 0x3B0;

       	outb(0x3C4,0x06);
       	outb(0x3C5,0x12);	/* unlock cirrus special */

				/* Put the Vert. Retrace End Reg in temp */

       	outb(vgaIOBase + 4, 0x11); temp = inb(vgaIOBase + 5);

				/* Put it back with PR bit set to 0 */
				/* This unprotects the 0-7 CRTC regs so */
				/* they can be modified, i.e. we can set */
				/* the timing. */

       	outb(vgaIOBase + 5, temp & 0x7F);
}


/* Relock chipset-specific registers */
/* (currently not used) */

static int cirrus_lock() {
       outb(0x3C4,0x06);
       outb(0x3C5,0x0F);	 /* relock cirrus special */
}


/* Indentify chipset, initialize and return non-zero if detected */

static int cirrus_test() {
	int oldlockreg;
	int lockreg;
	
	outb(0x3c4, 0x06);
	oldlockreg = inb(0x3c5);

	cirrus_unlock();

	  /* If it's a Cirrus at all, we should be */
	  /* able to read back the lock register */
	  
	  outb(0x3C4,0x06);
	  lockreg = inb(0x3C5);
	  
	  /* Ok, if it's not 0x12, we're not a Cirrus542X. */
	if (lockreg != 0x12) {
		outb(0x3c4, 0x06);
		outb(0x3c5, oldlockreg);
		return 0;
	}
	
	/* The above check seems to be weak, so we also check the chip ID. */
	
	outb(CRT_I, 0x27);
	switch (inb(CRT_D) >> 2) {
		case 0x22 : 
		case 0x23 :
		case 0x24 :
		case 0x25 :
		case 0x26 :
			break;
		default :
			outb(0x3c4, 0x06);
			outb(0x3c5, oldlockreg);
			return 0;
	}

	cirrus_init(0, 0, 0);
	return 1;
}


/* Bank switching function -- set 64K page number */

static int cirrus_setpage( unsigned page ) {
	if (cirrus_memory == 2048)
		/* Cirrus banking register has been set to 16K granularity */
		outw(0x3ce, (page << 10) + 0x09);
	else
		/* default 4K granularity */
		outw(0x3ce, (page << 12) + 0x09);
}


/* Set display start address (not for 16 color modes) */
/* Cirrus supports any address in video memory (up to 2Mb) */

static int cirrus_setdisplaystart( int address ) {
	outw(0x3d4, 0x0d + ((address >> 2) & 0x00ff) * 256);	/* sa2-sa9 */
	outw(0x3d4, 0x0c + ((address >> 2) & 0xff00));		/* sa10-sa17 */
	inb(0x3da);			/* set ATC to addressing mode */
	outb(0x3c0, 0x13 + 0x20);	/* select ATC reg 0x13 */
	/* Cirrus specific bits 0,1 and 18,19,20: */
	outb(0x3c0, (inb(0x3c1) & 0xf0) | (address & 3));
		/* write sa0-1 to bits 0-1; other cards use bits 1-2 */
	outb(0x3d4, 0x1b);
	outb(0x3d5, (inb(0x3d5) & 0xf2) 
		| ((address & 0x40000) >> 18)	/* sa18: write to bit 0 */
		| ((address & 0x80000) >> 17)	/* sa19: write to bit 2 */
		| ((address & 0x100000) >> 17)); /* sa20: write to bit 3 */
}


/* Set logical scanline length (usually multiple of 8) */
/* Cirrus supports multiples of 8, up to 4088 */

static int cirrus_setlogicalwidth( int width ) { 
	outw(0x3d4, 0x13 + (width >> 3) * 256);	/* lw3-lw11 */
	outb(0x3d4, 0x1b);
	outb(0x3d5, (inb(0x3d5) & 0xef) | ((width & 0x800) >> 7));
		/* write lw12 to bit 4 of Sequencer reg. 0x1b */
}


/* Function table (exported) */

static int cirrus_bitblt( int, int, int, int, int );
static int cirrus_imageblt( unsigned *, int, int, int, int );
static int cirrus_fillblt( int, int, int, int, int );
static int cirrus_hlinelistblt( int, int, int *, int *, int, int );
static int cirrus_bltwait();

int (*cirrus_chipsetfunctions[])() = {
	cirrus_saveregs,
	cirrus_setregs,
	cirrus_unlock,
	cirrus_lock,
	cirrus_test,
	cirrus_init,
	cirrus_setpage,
	cirrus_setmode,
	cirrus_modeavailable,
	cirrus_setdisplaystart,
	cirrus_setlogicalwidth,
	cirrus_getmodeinfo,
	cirrus_bitblt,
	cirrus_imageblt,
	cirrus_fillblt,
	cirrus_hlinelistblt,
	cirrus_bltwait
};


/* Initialize chipset (called after detection) */

static int cirrus_typenumber[] = { 5420, 5422, 5424, 5426, 5428 };

static int cirrus_init( int force, int par1, int par2 ) {
	unsigned char v;
	if (force) {
		cirrus_memory = par1;
		cirrus_chiptype = par2;
	}
	else {
		outb(0x3c4,0x0a);	/* read memory register */
		v = inb(0x3c5);
		cirrus_memory = 256 << ((v >> 3) & 3);
		outb(CRT_I, 0x27);
		cirrus_chiptype = inb(CRT_D) >> 2;
		switch (cirrus_chiptype) {
			case 0x22 : cirrus_chiptype = CLGD5420; break;
			case 0x23 : cirrus_chiptype = CLGD5422; break;
			case 0x24 : cirrus_chiptype = CLGD5426; break;
			case 0x25 : cirrus_chiptype = CLGD5424; break;
			case 0x26 : cirrus_chiptype = CLGD5428; break;
			default :
				printf("Unknown Cirrus chip %2x. Assuming "
					"GD5426 compatibility.\n",
					cirrus_chiptype);
				cirrus_chiptype = CLGD5426;
				break;
		}
	}
	if (__driver_report) {
		printf("Using Cirrus Logic GD542x driver (%d, %dK).\n",
			cirrus_typenumber[cirrus_chiptype], cirrus_memory);
	}
	chipsetfunctions = cirrus_chipsetfunctions;
}



/* 	Some information on the accelerated features of the 5426,
	derived from beta XFree86 driver source. This may have severe errors.

	port	index

	Addresses have 21 bits (2Mb of memory).
	0x3ce,	0x28	bits 0-7 of the destination address
	0x3ce,  0x29	bits 8-15
	0x3ce,	0x2a	bits 16-20

	0x3ce,	0x2c	bits 0-7 of the source address
	0x3ce,	0x2d	bits 8-15
	0x3ce,	0x2e	bits 16-20

	Maximum pitch is 4095.
	0x3ce,	0x24	bits 0-7 of the destination pitch (screen width)
	0x3ce,	0x25	bits 8-11
	
	0x3ce,	0x26	bits 0-7 of the source pitch (screen width)
	0x3ce,	0x27	bits 8-11
	
	Maximum width is 2047.
	0x3ce,	0x20	bits 0-7 of the box width - 1
	0x3ce,	0x21	bits 8-10
	
	Maximum height is 1023.
	0x3ce,	0x22	bits 0-7 of the box height - 1
	0x3ce,	0x23	bits 8-9

	0x3ce,	0x30	bit 0: direction (0 = down, 1 = up)
			bit 2: source (0 = video memory, 1 = system memory)
			bit 6: 8x8 pattern copy

		0x32	operation (0x0d = Copy)

		0x31	bit 1: start operation
			bit 0: busy

	When doing blits from system memory to video memory, pixel data
	must apparently be written to 0xa0004 in dwords (each to the same
	address). This is handy because the chip handles line transitions
	and alignment automatically.

	The pattern copy requires an 8x8 pattern (64 pixels) at the source
	address in video memory, and fills a box with specified size and 
	destination address with the pattern. This is in fact the way to do
	solid fills.

	The chip can also handle transparencies, and has a monochrome 
	adressing mode where each byte corresponds to 8 pixels, which should
	be very fast and convenient for two color graphics (like text).
	Unfortunately, I don't know how to do that yet. 
*/


static int cirrus_bltwait() {
	while (inb(0x3cf) & 2);		/* wait until finished */
}

static int cirrus_bitblt( int srcaddr, int destaddr, int w, int h, 
int pitch ) {
	w--;
	outw(0x3ce, 0x20 + (w << 8));		/* bits 0-7 of width */
	outw(0x3ce, 0x21 + (w & 0xff00));	/* bits 8-10 */

	h--;
	outw(0x3ce, 0x22 + (h << 8));		/* bits 0-7 of height */
	outw(0x3ce, 0x23 + (h & 0xff00));	/* bits 8-9 */

	outw(0x3ce, 0x24 + (pitch << 8));	/* bits 0-7 of dest. pitch */
	outw(0x3ce, 0x25 + (pitch & 0xff00));	/* bits 8-11 */
	outw(0x3ce, 0x26 + (pitch << 8));	/* source pitch */
	outw(0x3ce, 0x27 + (pitch & 0xff00));

	outw(0x3ce, 0x28 + (destaddr << 8));	/* bits 0-7 of dest. address */
	outw(0x3ce, 0x29 + (destaddr & 0xff00));	  /* bits 8-15 */
	outw(0x3ce, 0x2a + ((destaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	outw(0x3ce, 0x2c + (srcaddr << 8));	/* bits 0-7 of src address */
	outw(0x3ce, 0x2d + (srcaddr & 0xff00));		 /* bits 8-15 */
	outw(0x3ce, 0x2e + ((srcaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	outw(0x3ce, 0x30 + 0);		/* normal blit */
	outw(0x3ce, 0x32 + 0x0d00);	/* operation: copy */
	outw(0x3ce, 0x31 + 0x0200);	/* start operation */

	cirrus_bltwait();
}


static int cirrus_imageblt( unsigned *srcaddr, int destaddr, int w, int h, 
int pitch ) {
	int count;
	unsigned *base;
	
	w--;
	outw(0x3ce, 0x20 + (w << 8));		/* bits 0-7 of width */
	outw(0x3ce, 0x21 + (w & 0xff00));	/* bits 8-10 */

	h--;
	outw(0x3ce, 0x22 + (h << 8));		/* bits 0-7 of height */
	outw(0x3ce, 0x23 + (h & 0xff00));	/* bits 8-9 */

	outw(0x3ce, 0x24 + (pitch << 8));	/* bits 0-7 of dest. pitch */
	outw(0x3ce, 0x25 + (pitch & 0xff00));	/* bits 8-11 */

	/* source pitch not required */

	outw(0x3ce, 0x28 + (destaddr << 8));	/* bits 0-7 of dest. address */
	outw(0x3ce, 0x29 + (destaddr & 0xff00));	  /* bits 8-15 */
	outw(0x3ce, 0x2a + ((destaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	/* blit source address not required */

	outw(0x3ce, 0x30 + 0x0400);	/* from system memory */
	outw(0x3ce, 0x32 + 0x0d00);	/* operation: copy */
	outw(0x3ce, 0x31 + 0x0200);	/* start operation */
	
	w++;
	h++;
	count = (w * h + 3) / 4;	/* #longwords */
	/* This may not be quite right for the last few bytes */

	base = (unsigned *)(graph_mem + 4);
	
#define USE_ASM	

	{
	unsigned *srcp;
	unsigned *endp;
	srcp = srcaddr;
	endp = srcaddr + count;
	#ifdef USE_ASM
	while (srcp + 16 <= endp) {
		int t;
		asm(
			"movl (%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 4(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 8(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 12(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 16(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 20(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 24(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 28(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 32(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 36(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 40(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 44(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 48(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 52(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 56(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			"movl 60(%1),%0\n\t"
			"movl %0,(%2)\n\t"
			: : "r" (t), "r" (srcp), "r" (base)
		);
		srcp += 16;	/* add 16 * 4 to pointer */
	}
	#endif
	while (srcp < endp) {
		*base = *srcp;
		srcp++;
	}
	}

	cirrus_bltwait();
}

static int patterntable = -1;

static int initializepatterns() {
	int i;
	/* write 256 8x8 patterns above claimed video memory */
	if (vga_claimvideomemory(__videomemoryused + 256 * 64)) {
		printf("svgalib: Cannot allocate video memory for Cirrus fill patterns.\n");
		return 1;
	}
	patterntable = __videomemoryused;
	for (i = 0; i < 256; i++) {
		int srcaddr = patterntable + i * 64;
		cirrus_setpage(srcaddr >> 16);
		memset(graph_mem + (srcaddr & 0xffff), i, 64);
	}
	return 0;
}

static int cirrus_fillblt( int destaddr, int w, int h, int pitch, int c ) {
	int srcaddr;

	if (patterntable == -1)
		if (initializepatterns())
			return -1;
	
	srcaddr = patterntable + c * 64;

	outw(0x3ce, 0x2c + (srcaddr << 8));	/* bits 0-7 of src address */
	outw(0x3ce, 0x2d + (srcaddr & 0xff00));		 /* bits 8-15 */
	outw(0x3ce, 0x2e + ((srcaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	w--;
	outw(0x3ce, 0x20 + (w << 8));		/* bits 0-7 of width */
	outw(0x3ce, 0x21 + (w & 0xff00));	/* bits 8-10 */

	h--;
	outw(0x3ce, 0x22 + (h << 8));		/* bits 0-7 of height */
	outw(0x3ce, 0x23 + (h & 0xff00));	/* bits 8-9 */

	outw(0x3ce, 0x24 + (pitch << 8));	/* bits 0-7 of dest. pitch */
	outw(0x3ce, 0x25 + (pitch & 0xff00));	/* bits 8-11 */

	outw(0x3ce, 0x28 + (destaddr << 8));	/* bits 0-7 of dest. address */
	outw(0x3ce, 0x29 + (destaddr & 0xff00));	  /* bits 8-15 */
	outw(0x3ce, 0x2a + ((destaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	outw(0x3ce, 0x30 + 0x4000);	/* pattern fill */
	outw(0x3ce, 0x32 + 0x0d00);	/* operation: copy */
	outw(0x3ce, 0x31 + 0x0200);	/* start operation */

	cirrus_bltwait();
}

static int cirrus_hlinelistblt( int ymin, int n, int *xmin, int *xmax,
int pitch, int c ) {
	int srcaddr, i;

	if (patterntable == -1)
		if (initializepatterns())
			return -1;

	srcaddr = patterntable + c * 64;

	/* registers that won't change */

	outw(0x3ce, 0x2c + (srcaddr << 8));	/* bits 0-7 of src address */
	outw(0x3ce, 0x2d + (srcaddr & 0xff00));		 /* bits 8-15 */
	outw(0x3ce, 0x2e + ((srcaddr & 0x1f0000) >> 8)); /* bits 16-20 */

	outw(0x3ce, 0x24 + (pitch << 8));	/* bits 0-7 of dest. pitch */
	outw(0x3ce, 0x25 + (pitch & 0xff00));	/* bits 8-11 */

	outw(0x3ce, 0x30 + 0x4000);	/* pattern fill */
	outw(0x3ce, 0x32 + 0x0d00);	/* operation: copy */

	for (i = 0; i < n; i++) {
		int w;
		int destaddr;
		w = xmax[i] - xmin[i] - 1;	/* discard rightmost pixel */
		if (w < 0)			/* (and -1 for chip) */
			continue;

		while (inb(0x3cf) & 2);		/* wait for blitter */

		outw(0x3ce, 0x20 + (w << 8));		/* bits 0-7 of width */
		outw(0x3ce, 0x21 + (w & 0x0700));	/* bits 8-10 */

		/* set height to 1 */
		outw(0x3ce, 0x22 + (0 << 8));		/* bits 0-7 of height */
		outw(0x3ce, 0x23 + (0 & 0x0300));	/* bits 8-9 */

		destaddr = (ymin + i) * pitch + xmin[i];

		outw(0x3ce, 0x28 + (destaddr << 8));
		outw(0x3ce, 0x29 + (destaddr & 0xff00));
		outw(0x3ce, 0x2a + ((destaddr & 0x1f0000) >> 8));

		outw(0x3ce, 0x31 + 0x0200);	/* start operation */
		__svgalib_critical = 1;
	}
	cirrus_bltwait();
	__svgalib_critical = 0;
}
