/*
 * truncate.c		- File truncation for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 */

#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

static void trunc_direct (struct ext2_inode * inode)
{
	int i;
#define DIRECT_BLOCK ((inode->i_size + 1023) >> 10)

	for (i = DIRECT_BLOCK; i < EXT2_NDIR_BLOCKS; i++)
	{
		unmark_block (inode->i_block[i]);
		inode->i_block[i] = 0;
	}
}

static void trunc_indirect (int dev, struct ext2_inode * inode, int offset,
			    unsigned long * p)
{
	int i;
	unsigned int dirty = 0;
	unsigned long * ind = (unsigned long *) (blkbuf + block_size);
#define INDIRECT_BLOCK (DIRECT_BLOCK - offset)

	if (!*p)
		return;
	read_block (dev, p, (char *) ind);
	for (i = INDIRECT_BLOCK; i < addr_per_block; i++)
	{
		if (!ind[i])
			continue;
		dirty = 1;
		unmark_block (ind[i]);
		ind[i] = 0;
	}
	for (i = 0; i < addr_per_block; i++)
		if (ind[i++])
			break;
	if (i >= addr_per_block)
	{
		unmark_block (*p);
		*p = 0;
	}
	else
		if (dirty)
			write_block(dev, *p, (char *) ind);
}

static void trunc_dindirect (int dev, struct ext2_inode * inode, int offset,
			     unsigned long * p)
{
	int i;
	unsigned int dirty = 0;
	unsigned long * dind = (unsigned long *) (blkbuf + block_size * 2);
#define DINDIRECT_BLOCK ((DIRECT_BLOCK - offset) >> 8)

	if (!*p)
		return;
	read_block (dev, p, (char *) dind);
	for (i = DINDIRECT_BLOCK; i < addr_per_block; i++)
	{
		if (!dind[i])
			continue;
		trunc_indirect (dev, inode, offset + (i << 8), dind + i);
		dirty = 1;
	}
	for (i = 0; i < addr_per_block; i++)
		if (dind[i])
			break;
	if (i >= addr_per_block)
	{
		unmark_block (*p);
		*p = 0;
	}
	else
		if (dirty)
			write_block (dev, *p, (char *) dind);
}

static void trunc_tindirect (int dev, struct ext2_inode * inode)
{
	int i;
	unsigned int dirty = 0;
	unsigned long * tind = (unsigned long *) (blkbuf + block_size * 3);
#define TINDIRECT_BLOCK	((DIRECT_BLOCK - (addr_per_block * addr_per_block + \
			addr_per_block + EXT2_NDIR_BLOCKS)) >> 16)

	if (!inode->i_block[EXT2_TIND_BLOCK])
		return;
	read_block (dev, &inode->i_block[EXT2_TIND_BLOCK], (char *) tind);
	for (i = TINDIRECT_BLOCK; i < addr_per_block; i++)
	{
		trunc_dindirect(dev, inode, EXT2_NDIR_BLOCKS + addr_per_block +
				addr_per_block * addr_per_block + (i << 16),
				tind + i);
		dirty = 1;
	}
	for (i = 0; i < addr_per_block; i++)
		if (tind[i])
			break;
	if (i >= addr_per_block)
	{
		unmark_block (inode->i_block[EXT2_TIND_BLOCK]);
		inode->i_block[EXT2_TIND_BLOCK] = 0;
	}
	else
		if (dirty)
			write_block (dev, inode->i_block[EXT2_TIND_BLOCK],
				     (char *) tind);
}

void truncate_file (int dev, struct ext2_inode * inode)
{
	trunc_direct(inode);
	trunc_indirect(dev, inode, EXT2_NDIR_BLOCKS, inode->i_block + 
		       EXT2_IND_BLOCK);
	trunc_dindirect(dev, inode, EXT2_NDIR_BLOCKS + addr_per_block,
			inode->i_block + EXT2_DIND_BLOCK);
	trunc_tindirect(dev, inode);
}
