/*
 * mksuper.c		- Re-create a super block for an ext2 fs
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/11/24	- Creation from mke2fs and tables2
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <linux/ext2_fs.h>

#include "e2fsprogs.h"

#define INODE_SIZE		(sizeof (struct ext2_inode))

char * program_name = "mksuper";
char * output_file = NULL;

static char super_block_buffer[EXT2_MIN_BLOCK_SIZE];

struct ext2_super_block * Super = (struct ext2_super_block *) super_block_buffer;

long block_size = 1024;
long frag_size = 1024;

unsigned long blocks;
unsigned long blocks_per_group = 8192;
unsigned long frags_per_block;
unsigned long inode_ratio = 4096;
unsigned long reserved_ratio = 5;

static volatile void fatal_error (const char * fmt_string, int retcode)
{
	fprintf (stderr, fmt_string, program_name);
	exit (retcode);
}

#define usage() fatal_error ("usage: %s [-i bytes-per-inode]\n" \
			     "       [-b block-size] [-f fragment-size]\n" \
			     "       [-g blocks-per-group]\n" \
			     "       [-m reserved-blocks-ratio]\n" \
			     "       [-o file] blocks\n", \
                             EXIT_USAGE)

static volatile void die (const char * str, int retcode)
{
	char blkbuf[4096];

	strcpy (blkbuf, "%s: ");
	strcat (blkbuf, str);
	strcat (blkbuf, "\n");
	fatal_error (blkbuf, retcode);
}

static void setup_tables (void)
{
	int log_size = 0;
	int size;
	unsigned long last_group_blocks;
	unsigned long min_blocks;
	unsigned long group_desc_count;
	unsigned long group_desc_size;
	unsigned long desc_blocks;
	unsigned char ok = 0;

	memset ((char *) Super, 0, EXT2_MIN_BLOCK_SIZE);
	MTIME = 0;
	WTIME = time (NULL);
	MAGIC = EXT2_SUPER_MAGIC;
	STATE = EXT2_VALID_FS;
	MNTCNT = 0;
	MAXMNTCNT = EXT2_DFL_MAX_MNT_COUNT;

	size = block_size >> (EXT2_MIN_BLOCK_LOG_SIZE + 1);
	while (size)
	{
		log_size++;
		size >>= 1;
	}
	BLOCKSIZE = log_size;

	size = frag_size >> (EXT2_MIN_FRAG_LOG_SIZE + 1);
	log_size = 0;
	while (size)
	{
		log_size++;
		size >>= 1;
	}
	FRAGSIZE = log_size;

	BLOCKS = blocks / (block_size / EXT2_MIN_BLOCK_SIZE);
	FRAGSPERGROUP = blocks_per_group * (block_size / frag_size);
	BLOCKSPERGROUP = blocks_per_group;

	do
	{
		RBLOCKS = (BLOCKS * reserved_ratio) / 100;
		group_desc_count = (((BLOCKS - NORM_FIRSTBLOCK) *
				     frags_per_block) / BLOCKSPERGROUP);
		if (group_desc_count * BLOCKSPERGROUP <
		    (BLOCKS - NORM_FIRSTBLOCK) * frags_per_block)
			group_desc_count++;
		if (group_desc_count % EXT2_DESC_PER_BLOCK (Super))
			desc_blocks = (group_desc_count /
				       EXT2_DESC_PER_BLOCK (Super)) + 1;
		else
			desc_blocks = (group_desc_count /
				       EXT2_DESC_PER_BLOCK (Super));
		group_desc_size = desc_blocks * block_size;

/* some magic nrs: 1 inode / 4096 bytes (now use the inode ratio) */
		INODES = (blocks * EXT2_MIN_BLOCK_SIZE) / inode_ratio;

/* Round the inodes count to fully use each group descriptor */
		if (INODES % group_desc_count)
			INODESPERGROUP = (INODES / group_desc_count) + 1;
		else
			INODESPERGROUP = INODES / group_desc_count;

/* Round the inodes count to fully use each block in each descriptor */
		if (INODESPERGROUP % EXT2_INODES_PER_BLOCK (Super))
			INODESPERGROUP = ((INODESPERGROUP /
					   EXT2_INODES_PER_BLOCK (Super)) +
					  1) * EXT2_INODES_PER_BLOCK (Super);
		INODES = INODESPERGROUP * group_desc_count;

	/*
	 * Now check that the lat group is big enough
	 * The last group must contain enough space:
	 * - 1 block for the super block backup
	 * - N block for the descriptors backup
	 * - 1 block for the blocks bitmap
	 * - 1 block for the inodes bitmap
	 * - N blocks for the inode table
	 * - a reasonable number of data blocks (randomly set to 50)
	 */
		last_group_blocks = ((BLOCKS - NORM_FIRSTBLOCK) %
				     blocks_per_group);
		min_blocks = (1 + desc_blocks + 1 + 1 + 50 +
			      (INODESPERGROUP * sizeof (struct ext2_inode)) /
			      block_size);
		if (last_group_blocks > 0 && last_group_blocks < min_blocks)
		{
			printf ("Warning: %lu block%s %s not allocated\n",
				last_group_blocks,
				last_group_blocks != 1 ? "s" : "",
				last_group_blocks != 1 ? "were" : "was");
			BLOCKS -= last_group_blocks;
		}
		else
			ok = 1;
	} while (!ok);

	FIRSTBLOCK = NORM_FIRSTBLOCK;

	printf ("%lu inodes\n", INODES);
	printf ("%lu blocks\n", BLOCKS);
	printf ("%lu blocks reserved for the super user\n", RBLOCKS);
	printf ("First data block=%lu (%lu)\n", FIRSTBLOCK, NORM_FIRSTBLOCK);
	printf ("Block size=%lu (log=%lu)\n",
		(unsigned long) EXT2_MIN_BLOCK_SIZE << BLOCKSIZE, BLOCKSIZE);
	printf ("Fragment size=%lu (log=%lu)\n",
		(unsigned long) EXT2_MIN_FRAG_SIZE << FRAGSIZE, FRAGSIZE);
	printf ("%lu blocks group%s\n",
		group_desc_count, (group_desc_count > 1) ? "s" : "");
	printf ("%lu blocks per group\n", BLOCKSPERGROUP);
	printf ("%lu fragments per group\n", FRAGSPERGROUP);
	printf ("%lu inodes per group\n", INODESPERGROUP);
}

static int check_size (long size)
{
	long tmp = size;

	while (tmp != 0)
	{
		if ((tmp & 1) && (tmp & ~1))
		{
			printf ("%ld is not a power of two\n", size);
			return 0;
		}
		tmp >>= 1;
        }
	return 1;
}

void main (int argc, char ** argv)
{
	char c;
	char * tmp;
	int dev;

	fprintf (stderr, "mksuper %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	if (argc && *argv)
		program_name = *argv;
	if (INODE_SIZE * (EXT2_MIN_BLOCK_SIZE / INODE_SIZE) != EXT2_MIN_BLOCK_SIZE)
		die("bad inode size", EXIT_ERROR);
	while ((c = getopt (argc, argv, "b:f:g:i:m:")) != EOF)
		switch (c)
		{
			case 'b':
				block_size = strtol (optarg, &tmp, 0);
				if (*tmp)
				{
					printf ("bad block size: %s\n",
						optarg);
					usage ();
				}
				if (!check_size (block_size) ||
				    block_size < EXT2_MIN_BLOCK_SIZE ||
				    block_size > EXT2_MAX_BLOCK_SIZE)
					die ("bad block size", EXIT_USAGE);
#if 0
				if (block_size != 1024)
					die ("Only 1024 bytes blocks are currently supported",
					     EXIT_USAGE);
#endif
				break;
			case 'f':
				frag_size = strtol (optarg, &tmp, 0);
				if (*tmp)
				{
					printf ("bad fragment size: %s\n",
						optarg);
					usage ();
				}
				if (!check_size (frag_size) ||
				    frag_size < EXT2_MIN_FRAG_SIZE ||
				    frag_size > EXT2_MAX_FRAG_SIZE)
					die ("bad fragment size", EXIT_USAGE);
				printf ("Fragments are not supported yet"
					", ignoring -f %ld\n", frag_size);
#if 0
				frag_size = 1024;
#endif
				break;
			case 'g':
				blocks_per_group = strtol (optarg, &tmp, 0);
				if (*tmp || (blocks_per_group % 8) != 0)
				{
					printf ("bad blocks per group: %s\n",
						optarg);
					usage ();
				}
			case 'i':
				inode_ratio = strtol (optarg, &tmp, 0);
				if (*tmp || inode_ratio < 1024)
				{
					printf ("bad inode ratio : %s\n",
						optarg);
					usage ();
				}
				break;
			case 'm':
				reserved_ratio = strtol (optarg, &tmp, 0);
				if (*tmp || reserved_ratio > 50)
				{
					printf ("bad reserved block ratio : %s\n", optarg);
					usage ();
				}
				break;
			case 'o':
				output_file = optarg;
				break;
			default:
				usage ();
		}
	if (optind == argc - 1)
		blocks = strtol (argv[optind], &tmp, 0);
	else
		usage ();
	if (*tmp)
	{
		printf ("bad block count : %s\n", argv[optind + 1]);
		usage ();
	}
	frag_size = block_size;
	if (blocks < 10)
		die("number of blocks must be greater than 10.", EXIT_USAGE);
	frags_per_block = block_size / frag_size;
	if (output_file == NULL)
		output_file = "superblock";
	dev = open (output_file, O_CREAT | O_WRONLY, 0644);
	if (dev < 0)
		die("unable to open output file", EXIT_USAGE);
	setup_tables ();
	if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
	    EXT2_MIN_BLOCK_SIZE)
		die ("unable to write super-block", EXIT_ERROR);
	close (dev);
	exit (EXIT_OK);
}
