/*
 * mke2fs.c		- Second extended file system creator
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation
 */

/*
 * Old history:
 * 24.11.91  -	time began. Used the fsck sources to get started.
 *
 * 25.11.91  -	corrected some bugs. Added support for ".badblocks"
 *		The algorithm for ".badblocks" is a bit weird, but
 *		it should work. Oh, well.
 *
 * 25.01.92  -  Added the -l option for getting the list of bad blocks
 *              out of a named file. (Dave Rivers, rivers@ponds.uucp)
 *
 * 28.02.92  -	added %-information when using -c.
 *
 * 28.06.92  -	modifications by Remy Card (card@masi.ibp.fr)
 *		Used the mkfs sources to create mkefs.c
 *
 * 14.07.92  -	cleaned up the code
 *
 * 19.07.92  -	now use the same bit op than mkfs instead of an internal linked
 *		list of marked numbers
 *
 * 26.07.92  -	mkefs is now able to produce triple indirection for the
 *		badblocks file
 *
 * 11.08.92  -	changes by Wayne Davison (davison@borland.com)
 *		- Unified some of the buffers to make it use less memory at
 *		  runtime.
 *		- Changed some ints to unsigned longs when referring to block
 *		  numbers (there's still more to "fix" in this area).
 *		- Fixed an off-by-one error in the alarm_intr() routine.
 *		- Removed a redundant lseek call in the block test code.
 *
 * 16.08.92  -  added the -i argument to specify the ratio of inodes
 *		use getopt() to get arguments
 *
 * 19.08.92  -	converted mkefs to mke2fs to create a new extended file system
 *
 * 27.08.92  -	added creation of a lost+found directory used during checks
 *
 * 30.08.92  -  added the -b and -f options which allow the user to specify
 *		blocks and fragments sizes.  mke2fs should now be able to
 *		create file systems with bigger block sizes.
 *
 * 20.12.92  -  lots of change to make mkefs understand the new (final ?)
 *		structure of the new extended file system (with blocks
 *		groups and bitmaps)
 *
 * 16.01.93  -	added -g blocks_per_group
 *		replaced some 'block_size * 8' by GROUPSBERBLOCK
 *		corrected the i_blocks field of / and /lost+found
 *
 * 24.01.93  -	removed the -g option which seems incompatible with
 *		super block/descriptors recovery (if the blocks count per
 *		group is contained in the super block, how can e2fsck access
 *		backups in other groups if the super block is corrupted ?)
 *		add backups of the super block and of the group descriptors
 *
 * 01.03.93  -	added the new used_dirs_count field computation
 *
 * 13.03.93  -	added again -g blocks_per_group
 *		made -t an alias to -c for compatibility with e2fsck
 *		made mke2fs use much less memory
 *		display the bad blocks list in verbose mode
 *
 * 15.04.93  -	fixed a bug in the free blocks count in case the last group
 *		is exactly 8 MB long
 *		check that the device is not mounted (wish I had included this
 *		before destroying my root fs during the tests :-{ )
 *		
 * Usage:  mke2fs [-c] [-i bytes-per-inode]
 *		  [-b block-size] [-f fragment-size]
 *		  [-g blocks_per_group] [-v]
 *		  [-m reserved_blocks_ratio]
 *		  device [size-in-blocks]
 *         mke2fs [-l filename ] [-i bytes-per-inode]
 *		  [-b block-size] [-f fragment-size]
 *		  [-g blocks_per_group] [-v]
 *		  [-m reserved_blocks_ratio]
 *		  device [size-in-blocks]
 *
 *	-b to specify the size of a block
 *	-c for readablility checking (SLOW!)
 *	-f to specify the size of a fragment
 *	-g to specify the number of blocks per group
 *	-i to specify the number of inodes
 *      -l for getting a list of bad blocks from a file.
 *	-m to specify the number of blocks reserved for the super user
 *	-t for readablility checking (SLOW!)
 *	-v for verbose mode
 *
 * The device may be a block device or a image of one, but this isn't
 * enforced (but it's not much fun on a character device :-). 
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include <termios.h>
#include <time.h>
#include <sys/stat.h>
#include <getopt.h>
#include <mntent.h>

#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "mkfunc.h"
#include "mkvar.h"

char blkbuf[EXT2_MAX_BLOCK_SIZE * TEST_BUFFER_BLOCKS];

#undef EXT2_INODES_PER_BLOCK
#undef EXT2_ADDR_PER_BLOCK
#undef EXT2_DESC_PER_BLOCK
#define EXT2_INODES_PER_BLOCK	(block_size / sizeof (struct ext2_inode))
#define EXT2_ADDR_PER_BLOCK (block_size / sizeof (unsigned long))
#define EXT2_DESC_PER_BLOCK (block_size / sizeof (struct ext2_group_desc))

#define UPPER(size, n)		((size + ((n) - 1)) / (n))
#define INODE_SIZE		(sizeof (struct ext2_inode))
#define INODE_BLOCKS		UPPER (INODES, EXT2_INODES_PER_BLOCK)
#define INODE_BUFFER_SIZE	(INODE_BLOCKS * block_size)

#define BITS_PER_BLOCK		(block_size << 3)

char * program_name = "mke2fs";
char * device_name = NULL;
int inode_ratio = 4096;
int reserved_ratio = 5;
long block_size = 1024;
long frag_size = 1024;
int frags_per_block;
int blocks_per_group = 8192;
int dev = -1;
long blocks = 0;
int check = 0;
int badblocks = 0;
int verbose = 0;

int changed = 0;

#define ROOT_INO_STRING		"\002\000\000\000"
#define LPF_INO_STRING		"\013\000\000\000"

#define ROOT_RECLEN_STRING	"\014\000"
#define LPF_RECLEN_STRING	"\024\000"

#define DOT_NAMELEN_STRING	"\001\000"
#define DOTDOT_NAMELEN_STRING	"\002\000"
#define LPF_NAMELEN_STRING	"\012\000"

static char root_block[EXT2_MIN_BLOCK_SIZE] =
ROOT_INO_STRING ROOT_RECLEN_STRING DOT_NAMELEN_STRING    ".\0\0\0"
ROOT_INO_STRING ROOT_RECLEN_STRING DOTDOT_NAMELEN_STRING "..\0\0"
LPF_INO_STRING  LPF_RECLEN_STRING   LPF_NAMELEN_STRING    "lost+found\0\0";

char * inode_buffer = NULL;

#define Inode	(((struct ext2_inode *) inode_buffer) - 1)

static char super_block_buffer[EXT2_MIN_BLOCK_SIZE];

struct ext2_super_block * Super = (struct ext2_super_block *) super_block_buffer;

char * bad_map   = NULL;
char * block_map = NULL;
char * inode_map = NULL;

unsigned long group_desc_count;
unsigned long group_desc_size;
unsigned long desc_blocks;
struct ext2_group_desc * group_desc = NULL;
int inode_blocks_per_group;

static volatile void fatal_error (const char * fmt_string, int retcode)
{
	fprintf (stderr, fmt_string, program_name, device_name);
	exit (retcode);
}

#define usage()	fatal_error ("usage: %s [-c | -t | -l filename] [-i bytes-per-inode]\n" \
			     "       [-b block-size] [-f fragment-size]\n" \
			     "       [-g blocks-per-group]\n" \
			     "       [-m reserved-blocks-ratio] [-v]\n" \
			     "       /dev/name [blocks]\n", \
			     EXIT_USAGE)

volatile void die (const char * str, int retcode)
{
	strcpy (blkbuf, "%s: ");
	strcat (blkbuf, str);
	strcat (blkbuf, "\n");
	fatal_error (blkbuf, retcode);
}

void write_block (unsigned long blk, char * buffer)
{
	if (lseek (dev, blk * block_size, SEEK_SET) != blk * block_size)
		die ("seek failed in write_block", EXIT_ERROR);
	if (write(dev, buffer, block_size) != block_size)
	{
		printf ("blk = %lu\n", blk);
		die ("write failed in write_block", EXIT_ERROR);
	}
}

static int get_free_block (void)
{
	static unsigned long blk = 0;

	if (!blk)
		blk = FIRSTBLOCK;
	while (blk < BLOCKS && block_in_use (blk))
		blk++;
	if (blk >= BLOCKS)
		die ("not enough good blocks", EXIT_ERROR);
	mark_block (blk);
	return blk++;
}

static unsigned long next_bad (unsigned long block)
{

	if (!block)
		block = FIRSTBLOCK - 1;
	while (++block < BLOCKS)
		if (block_is_bad (block))
		{
			printf ("%lu ", block);
			return block;
		}
	return 0;
}

static void make_bad_inode (void)
{
	struct ext2_inode * inode = Inode + EXT2_BAD_INO;
	int i, j, k;
	unsigned long block;
	int ind = 0, dind = 0, tind = 0;
	unsigned long * ind_block  = (unsigned long *) (blkbuf + block_size);
	unsigned long * dind_block = (unsigned long *) (blkbuf + block_size * 2);
	unsigned long * tind_block = (unsigned long *) (blkbuf + block_size * 3);

	mark_inode (EXT2_BAD_INO);
	inode->i_links_count = 0;
	inode->i_atime = time(NULL);
	inode->i_ctime = inode->i_atime;
	inode->i_mtime = inode->i_atime;
	inode->i_mode = 0;
	inode->i_size = badblocks * block_size;
	inode->i_blocks = badblocks * frags_per_block;
	if (!badblocks)
		return;
	if (verbose)
		printf ("Recording bad blocks : ");
	block = next_bad (0);
	for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
	{
		inode->i_block[i] = block;
		if (!(block = next_bad (block)))
			goto end_bad;
	}
	inode->i_block[EXT2_IND_BLOCK] = ind = get_free_block ();
	memset (ind_block, 0, block_size);
	for (i = 0; i < EXT2_ADDR_PER_BLOCK; i++)
	{
		ind_block[i] = block;
		if (!(block = next_bad (block)))
			goto end_bad;
	}
	inode->i_block[EXT2_DIND_BLOCK] = dind = get_free_block ();
	memset (dind_block, 0, block_size);
	for (i = 0; i < EXT2_ADDR_PER_BLOCK; i++)
	{
		write_block (ind, (char *) ind_block);
		dind_block[i] = ind = get_free_block();
		memset (ind_block, 0, block_size);
		for (j = 0; j < EXT2_ADDR_PER_BLOCK; j++)
		{
			ind_block[j] = block;
			if (!(block = next_bad (block)))
				goto end_bad;
		}
	}
	inode->i_block[EXT2_TIND_BLOCK] = tind = get_free_block ();
	memset (tind_block, 0, block_size);
	for (i = 0; i < EXT2_ADDR_PER_BLOCK; i++)
	{
		write_block (dind, (char *) dind_block);
		tind_block[i] = dind = get_free_block();
		memset (dind_block, 0, block_size);
		for (j = 0; j < EXT2_ADDR_PER_BLOCK; j++)
		{
			write_block (ind, (char *) ind_block);
			dind_block[j] = ind = get_free_block();
			memset (ind_block, 0 , block_size);
			for (k = 0; k < EXT2_ADDR_PER_BLOCK; k++)
			{
				ind_block[k] = block;
				if (!(block = next_bad (block)))
					goto end_bad;
			}
		}
	}
	die ("too many bad blocks", EXIT_ERROR);
end_bad:
	printf ("\n");
	if (ind)
		write_block (ind, (char *) ind_block);
	if (dind)
		write_block (dind, (char *) dind_block);
	if (tind)
		write_block (tind, (char *) tind_block);
}

/*
 * Create the lost+found directory
 *
 * Note: the direct blocks are pre-allocated for this directory so e2fsck
 * won't have to allocate space when moving inodes to this directory
 */
static unsigned long make_lpf_inode (void)
{
	unsigned long ino = EXT2_FIRST_INO;
	struct ext2_inode * inode;
	char * lpf_block = blkbuf;
	struct ext2_dir_entry * de;
	int i;

	while (ino <= INODESPERGROUP && inode_in_use (ino))
		ino++;
	if (ino > INODESPERGROUP)
		return 0;	/* Should never happen */
	if (verbose)
		printf ("Making lost+found (inode %lu): blocks", ino);
	mark_inode (ino);
	group_desc[0].bg_used_dirs_count++;
	inode = Inode + ino;
	inode->i_links_count = 2;
	inode->i_atime = time (NULL);
	inode->i_ctime = inode->i_atime;
	inode->i_mtime = inode->i_atime;
	inode->i_dtime = 0;
	inode->i_version = 1;
	inode->i_mode = S_IFDIR + 0755;
	inode->i_block[0] = get_free_block ();
	if (verbose)
		printf (" %lu", inode->i_block[0]);
	de = (struct ext2_dir_entry *) lpf_block;
	de->inode = ino;
	de->rec_len = EXT2_DIR_REC_LEN (1);
	de->name_len = 1;
	strcpy (de->name, ".");
	de = (struct ext2_dir_entry *) (lpf_block + EXT2_DIR_REC_LEN (1));
	de->inode = EXT2_ROOT_INO;
	de->rec_len = block_size - EXT2_DIR_REC_LEN (1);
	de->name_len = 2;
	strcpy (de->name, "..");
	write_block (inode->i_block[0], lpf_block);
	de = (struct ext2_dir_entry *) lpf_block;
	de->inode = 0;
	de->rec_len = block_size;
	de->name_len = 0;
	for (i = 1; i < EXT2_NDIR_BLOCKS; i++)
	{
		inode->i_block[i] = get_free_block ();
		write_block (inode->i_block[i], lpf_block);
		if (verbose)
			printf (" %lu", inode->i_block[i]);
	}
	if (verbose)
		printf ("\n");
	inode->i_blocks = (block_size * EXT2_NDIR_BLOCKS) / 512;
	inode->i_size = block_size * EXT2_NDIR_BLOCKS;
	return ino;
}

static void make_root_inode (unsigned long lpf_ino)
{
	struct ext2_inode * inode = Inode + EXT2_ROOT_INO;
	struct ext2_dir_entry * de;

	if (verbose)
		printf ("Making root inode (inode %d): ", EXT2_ROOT_INO);
	mark_inode (EXT2_ROOT_INO);
	group_desc[0].bg_used_dirs_count++;
	inode->i_block[0] = get_free_block();
	if (verbose)
		printf (" block %lu\n", inode->i_block[0]);
	inode->i_links_count = 3;
	inode->i_atime = time (NULL);
	inode->i_ctime = inode->i_atime;
	inode->i_mtime = inode->i_atime;
	inode->i_dtime = 0;
	inode->i_size = block_size;
	inode->i_blocks = block_size / 512;
	inode->i_mode = S_IFDIR + 0755;
	inode->i_version = 1;
	if (lpf_ino)
	{
		de = (struct ext2_dir_entry *) (root_block +
						EXT2_DIR_REC_LEN (1) +
						EXT2_DIR_REC_LEN (2));
		de->rec_len = block_size - (EXT2_DIR_REC_LEN (1) +
					    EXT2_DIR_REC_LEN (2));
	}
	else
	{
		de = (struct ext2_dir_entry *) (root_block +
						EXT2_DIR_REC_LEN (1));
		de->rec_len = block_size - EXT2_DIR_REC_LEN (1);
	}
	write_block (inode->i_block[0], root_block);
}

static int check_size (long size)
{
	long tmp = size;

	while (tmp != 0)
	{
		if ((tmp & 1) && (tmp & ~1))
		{
			printf ("%ld is not a power of two\n", size);
			return 0;
		}
		tmp >>= 1;
	}
	return 1;
}

static void check_mount (const char * device_name)
{
	FILE * f;
	struct mntent * mnt;

	if ((f = setmntent (MOUNTED, "r")) == NULL)
		return;
	while ((mnt = getmntent (f)) != NULL)
		if (strcmp (device_name, mnt->mnt_fsname) == 0)
			die ("%s contains a mounted file system.", EXIT_USAGE);
	endmntent (f);
}

void main (int argc, char ** argv)
{
	char c;
	char * tmp;
	struct stat statbuf;
	char * listfile = NULL;
	unsigned long lpf_ino;

	fprintf (stderr, "mke2fs %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	if (argc && *argv)
		program_name = *argv;
	if (INODE_SIZE * EXT2_INODES_PER_BLOCK != EXT2_MIN_BLOCK_SIZE)
		die("bad inode size", EXIT_ERROR);
	while ((c = getopt (argc, argv, "b:cf:g:i:l:m:tv")) != EOF)
		switch (c)
		{
			case 'b':
				block_size = strtol (optarg, &tmp, 0);
				if (*tmp)
				{
					printf ("bad block size: %s\n",
						optarg);
					usage ();
				}
				if (!check_size (block_size) ||
				    block_size < EXT2_MIN_BLOCK_SIZE ||
				    block_size > EXT2_MAX_BLOCK_SIZE)
					die ("bad block size", EXIT_USAGE);
#if 0
				if (block_size != 1024)
					die ("Only 1024 bytes blocks are currently supported",
					     EXIT_USAGE);
#endif
				break;
			case 'c':
			case 't':
				check = 1;
				break;
			case 'f':
				frag_size = strtol (optarg, &tmp, 0);
				if (*tmp)
				{
					printf ("bad fragment size: %s\n",
						optarg);
					usage ();
				}
				if (!check_size (frag_size) ||
				    frag_size < EXT2_MIN_FRAG_SIZE ||
				    frag_size > EXT2_MAX_FRAG_SIZE)
					die ("bad fragment size", EXIT_USAGE);
				printf ("Fragments are not supported yet"
					", ignoring -f %ld\n", frag_size);
#if 0
				frag_size = 1024;
#endif
				break;
			case 'g':
				blocks_per_group = strtol (optarg, &tmp, 0);
				if (*tmp || (blocks_per_group % 8) != 0)
				{
					printf ("bad blocks per group: %s\n",
						optarg);
					usage ();
				}
			case 'i':
				inode_ratio = strtol (optarg, &tmp, 0);
				if (*tmp || inode_ratio < 1024)
				{
					printf ("bad inode ratio : %s\n",
						optarg);
					usage ();
				}
				break;
			case 'l':
				listfile = optarg;
				break;
			case 'm':
				reserved_ratio = strtol (optarg, &tmp, 0);
				if (*tmp || reserved_ratio > 50)
				{
					printf ("bad reserved block ratio : %s\n", optarg);
					usage ();
				}
				break;
			case 'v':
				verbose = 1;
				break;
			default:
				usage ();
		}
	device_name = argv [optind];
	if (optind == argc - 2)
		blocks = strtol (argv[optind + 1], &tmp, 0);
	else if (optind == argc - 1)
	{
		blocks = count_blocks (device_name);
		tmp = "";
	}
	else
		usage ();
	if (*tmp)
	{
		printf ("bad block count : %s\n", argv[optind + 1]);
		usage ();
	}
	frag_size = block_size;
	if (check && listfile)
		die("-c and -l are incompatible\n", EXIT_USAGE);
	if (blocks < 10)
		die("number of blocks must be greater than 10.", EXIT_USAGE);
	frags_per_block = block_size / frag_size;
	check_mount (device_name);
	dev = open (device_name, O_RDWR);
	if (dev < 0)
		die("unable to open %s", EXIT_USAGE);
	if (fstat (dev, &statbuf) < 0)
		die ("unable to stat %s", EXIT_ERROR);
	if (!S_ISBLK (statbuf.st_mode))
		check = 0;
	else if (statbuf.st_rdev == 0x0300 || statbuf.st_rdev == 0x0340)
		die ("will not try to make filesystem on '%s'", EXIT_USAGE);
	setup_tables ();
	if (check)
		check_blocks (dev);
        else if (listfile)
                get_list_blocks (listfile);
	make_group_desc ();
	lpf_ino = make_lpf_inode ();
	make_root_inode (lpf_ino);
	make_bad_inode ();
	count_free ();
	write_tables (dev);
	exit (EXIT_OK);
}
