/*
 * dirdump.c		- Dumps an ext 2 fs directory
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/01/04	- Creation
 */

#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <unistd.h>

#include <sys/stat.h>

#include <linux/ext2_fs.h>

#include "e2fsprogs.h"

static volatile void usage (void)
{
	printf ("Usage: dirdump [-ax] directory\n");
	exit (EXIT_USAGE);
}

void main (int argc, char **argv)
{
	struct dir_entry_header
	{
		unsigned long inode;
		unsigned short rec_len;
		unsigned short name_len;
	} deh;
	char name [256];
	struct stat st;
	int dir;
	long offset = 0;
	int i, n;
	int ascii = 0, hexa = 0;
	char c;

	fprintf (stderr, "dirdump %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	while ((c = getopt (argc, argv, "ax")) != EOF)
		switch (c)
		{
			case 'a':
				ascii = 1;
				break;
			case 'x':
				hexa = 1;
				break;
			default:
				usage ();
		}
	if (optind != argc - 1)
		usage ();
	if (stat (argv[optind], &st) == -1)
	{
		perror ("stat");
		exit (EXIT_ERROR);
	}
	if (!S_ISDIR (st.st_mode))
	{
		fprintf (stderr, "%s is not a directory\n", argv[optind]);
		exit (EXIT_USAGE);
	}
	if ((dir = open (argv[optind], O_RDONLY)) < 0)
	{
		perror ("open directory");
		exit (EXIT_ERROR);
	}
	if (!ascii && !hexa)
		ascii = 1;
	n = read (dir, (char *) &deh, sizeof (deh));
	while (n != 0)
	{
		if (n == -1)
		{
			perror ("reading header");
			exit (EXIT_ERROR);
		}
		if (n < sizeof (deh))
		{
			printf ("Error reading header, read = %d\n", n);
			exit (EXIT_ERROR);
		}
		printf ("%4ld (%04lx): inode = %5ld, name_len = %3d"
			", rec_len = %4d, name = ",
			offset, offset, deh.inode, deh.name_len, deh.rec_len);
		if (deh.inode)
		{
			n = read (dir, name, deh.name_len);
			if (n == -1)
			{
				printf ("\n");
				perror ("reading name");
				exit (EXIT_ERROR);
			}
			if (n < deh.name_len)
			{
				printf ("\nError reading name, read = %d\n", n);
				exit (EXIT_ERROR);
			}
			name[deh.name_len] = 0;
			if (ascii)
				for (i = 0; i < deh.name_len; i++)
					putchar (name[i]);
			if (hexa)
			{
				if (ascii)
					printf (" (");
				for (i = 0; i < deh.name_len - 1; i++)
					printf ("0x%x,", name[i]);
				printf ("0x%02x", name[deh.name_len - 1]);
				if (ascii)
					printf (")");
			}
			printf ("\n");
		}
		else
			printf ("** Unused **\n");
		if (deh.rec_len %4 != 0)
		{
			printf ("Record length %%4 != 0, skipping block\n");
			offset = ((offset / st.st_blksize) + 1) * st.st_blksize;
		}
		else if (deh.rec_len < deh.name_len + 8)
		{
			printf ("Record length < Name length + 8, skipping block\n");
			offset = ((offset / st.st_blksize) + 1) * st.st_blksize;
		}
		else if (deh.rec_len == 0)
		{
			printf ("Record length == 0, skipping block\n");
			offset = ((offset / st.st_blksize) + 1) * st.st_blksize;
		}
		else if (((offset + deh.rec_len - 1) / st.st_blksize) > (offset / st.st_blksize))
		{
			printf ("Directory entry across two blocks, skipping\n");
			offset = ((offset / st.st_blksize) + 1) * st.st_blksize;
		}
		else
			offset += deh.rec_len;
		if (lseek (dir, offset, SEEK_SET) < 0)
		{
			perror ("lseek");
			exit (EXIT_ERROR);
		}
		n = read (dir, (char *) &deh, sizeof (deh));
	}
	exit (EXIT_OK);
}
