// This is an example of an extended procedure DLL built with Open Data
// Services. The functions within the DLL can be invoked by using the extended
//	stored procedures support in SQL Server.  To register the functions 
// and allow all users to use them run the ISQL script INSTXP.SQL.
//
// For further information on Open Data Services refer to the Microsoft Open 
// Data Services Programmer's Reference.
//
//  The extended procedures implemented in this DLL are:
//
//  XP_PROCLIST	Returns all this DLL's extended stored procedures and 
//					their usuage.
//
//	 XP_DISKLIST	Returns a row for each defined drive containing its name 
//					and the amount of disk space available.
//
//  XP_DISKFREE   Returns the amount of space available on a given drive.
//             The value is placed into the defined return parameter of
//             the stored procedure.
//
//  XP_SCAN_XBASE	Reads an xBase file and sends it to the client as if it 
//					were a SQL Server query result set (the equivalent of a 
//					'SELECT * FROM tablename' SQL statement).
//
//  XP_ECHO		Used to demo the handling of output parameters in
//					extended procedures.
//

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <windows.h>
#include <srv.h>

// Miscellaneous defines
//
#define  XP_NOERROR		0
#define  XP_ERROR		1

#define  MAXNAME       31	// Maximum extended procedure name length
#define  MAXLEN        80	// Maximum string length
#define  COMMAND_PARAM	1	// Command Parameter
#define  OUTPUT_PARAM	2	// Command Parameter

// Extended procedure error codes
//
#define  SRV_MAXERROR           20000
#define  XP_PROCLIST_ERROR      SRV_MAXERROR + 1
#define  CMDSHELL_ERROR         SRV_MAXERROR + 2
#define  DISKLIST_ERROR      	  SRV_MAXERROR + 3
#define  SCAN_XBASE_ERROR       SRV_MAXERROR + 4

// Standard error macro for reporting API errors
//
#define SETERROR( api, retstring)               \
    sprintf(retstring,"%s: Error %d from %s on line %d\n",  \
        __FILE__, GetLastError(), api, __LINE__);
 
// SCAN_XBASE defines
//
#define XBASE_HDR_SIZE  		32
#define XBASE_MAX_COLUMNS 		128
#define XBASE_BUF_SIZE    		2048

// Stored Procedure information structure.
//
typedef struct xp_info {
    DBCHAR name[MAXNAME];   	// Extended procedure name
    DBCHAR usage[MAXLEN];   		// Usage string
} XP_INFO;

// Array of Extended Stored Procedures supported by this DLL.
//
XP_INFO Xps[] = 
{
    "xp_proclist",					// Procedure name
    "usage: xp_proclist",		 	// Procedure usage string

    "xp_disklist",
    "usage: xp_disklist",

    "xp_diskfree",
    "usage: xp_diskfree <[@drive =] drive letter> [,] <[@space =] free space>",

    "xp_scan_xbase",
    "usage: xp_scan_xbase <xbase file name>",

    "xp_echo",
    "usage: xp_echo @p1=<input> , @p2=<output-param> output",

};
#define Xpcnumber sizeof(Xps) / sizeof(XP_INFO)

//
// PROCLIST
//    Returns the usage for all defined stored procedures
//
// Parameters:
//    srvproc - the handle to the client connection that got the SRV_CONNECT.
//
// Returns:
//    XP_NOERROR
//    XP_ERROR
//
// Side Effects:
//    Returns a result set to client
//
RETCODE xp_proclist(srvproc)
SRV_PROC *srvproc;
{
    int paramnum;
    DBCHAR colname1[MAXNAME];
    DBCHAR colname2[MAXNAME];
    int i;

    // Get number of parameters
    //
    paramnum = srv_rpcparams(srvproc);

    // Check number of parameters
    //
    if (paramnum != -1) {
        // Send error message and return
        //
        srv_sendmsg(srvproc, SRV_MSG_ERROR, XP_PROCLIST_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error executing extended stored procedure: Invalid Parameter",
                    SRV_NULLTERM);
	 	  // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	 	  // result set of an Extended Stored Procedure.
	 	  //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }

    sprintf(colname1, "spname");
    srv_describe(srvproc, 1, colname1, SRV_NULLTERM, SRVCHAR, MAXNAME,
                 SRVCHAR, 0, NULL);

    sprintf(colname2, "spusage");
    srv_describe(srvproc, 2, colname2, SRV_NULLTERM, SRVCHAR, MAXLEN, SRVCHAR,
                 0, NULL);

    // Return each XP handler as a row
    //
    for (i = 0; i < Xpcnumber; i++) {
        srv_setcoldata(srvproc, 1, Xps[i].name);
        srv_setcollen(srvproc, 1, strlen(Xps[i].name));

        srv_setcoldata(srvproc, 2, Xps[i].usage);
        srv_setcollen(srvproc, 2, strlen(Xps[i].usage));

        srv_sendrow(srvproc);
    }
	 // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	 // result set of an Extended Stored Procedure.
	 //
    srv_senddone(srvproc, (SRV_DONE_COUNT | SRV_DONE_MORE), 0, i);

    return(XP_NOERROR);
}

//
// XP_ECHO
//	Used to demo the handling of output parameters in extended procedures.
//	The first parameter should be the input parameter and the second
//	parameter should be the output.  The input parameter is placed in the
//	output parameter and returned.
//
// Parameters:
//    srvproc - the handle to the client connection that got the SRV_CONNECT.
//
// Returns:
//    XP_NOERROR
//    XP_ERROR
//
//

RETCODE xp_echo(srvproc)
SRV_PROC *srvproc;
{
    int paramnum;

    // Check number of parameters
    //
    if ((paramnum = srv_rpcparams(srvproc)) != 2) {
        // Send error message and return
        //
        srv_sendmsg(srvproc, SRV_MSG_ERROR, CMDSHELL_ERROR, SRV_INFO, (DBTINYINT)0,
		    NULL, 0, 0, "Error executing extended stored procedure: Invalid # of Parameters",
                    SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }

    // Set output parameter to input parameter.
    //

    srv_paramset(srvproc, 2, srv_paramdata(srvproc,1), srv_paramlen(srvproc,1));

    srv_senddone(srvproc, SRV_DONE_MORE, 0, 0);

    return(XP_NOERROR);

}

//
// XP_DISKLIST
//     Returns a row for each defined drive containing its name and the
//     amount of disk space available.
//
// Parameters:
//    srvproc - the handle to the client connection that got the SRV_CONNECT.
//
// Returns:
//    XP_NOERROR
//    XP_ERROR
//
// Side Effects:
//     Returns a result set to client
//
RETCODE xp_disklist(srvproc)
SRV_PROC *srvproc;
{
    int paramnum;
    DBCHAR colname1[MAXNAME];
    DBCHAR colname2[MAXNAME];
    DBCHAR drivename;
    DBCHAR rootname[16];
    int drivenum;
    int secPerCluster;
    int bytesPerSector;
    int freeClusters;
    int totalClusters;
    int drivenums;
    int space_remaining;
    int i = 0;

    // Get number of parameters
    //
    paramnum = srv_rpcparams(srvproc);

    // Check number of parameters
    //
    if (paramnum != -1) {
        // Send error message and return
        //
        srv_sendmsg(srvproc, SRV_MSG_ERROR, DISKLIST_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error executing extended stored procedure: Invalid Parameter",
                    SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }
    sprintf(colname1, "drive");
    srv_describe(srvproc, 1, colname1, SRV_NULLTERM, SRVCHAR, 1, SRVCHAR, 1,
                 (BYTE *)&drivename);

    sprintf(colname2, "bytes free");
    srv_describe(srvproc, 2, colname2, SRV_NULLTERM, SRVINT4, 4, SRVINT4, 4,
                 (BYTE *)&space_remaining);

    drivenums = GetLogicalDrives();

    drivenums >>= 2;        //Ignore drives A and B
    for (drivename = 'C', drivenum = 3; drivename <= 'Z';
         drivename++, drivenum++) {

        if (drivenums & 1) {
            i++;
            sprintf(rootname, "%c:\\", drivename);
            GetDiskFreeSpace(rootname, &secPerCluster, &bytesPerSector,
                             &freeClusters, &totalClusters);
            space_remaining = secPerCluster * freeClusters * bytesPerSector;

            srv_sendrow(srvproc);
        }
        drivenums >>= 1;
    }
    // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
    // result set of an Extended Stored Procedure.
    //
    srv_senddone(srvproc, (SRV_DONE_COUNT | SRV_DONE_MORE), 0, i);
    return(XP_NOERROR);
}


//
// XP_DISKFREE
//    Returns the amount of space available on a given drive. The value
//     is placed into the defined return parameter of the stored procedure.
//
// Parameters:
//    srvproc - the handle to the client connection that got the SRV_CONNECT.
//
// Returns:
//    XP_NOERROR
//    XP_ERROR
//
// Side Effects:
//    Returns messages and/or a result set to client. Returns a value in the
//     defined return parameter.
//
RETCODE xp_diskfree(srvproc)
SRV_PROC *srvproc;
{
    DBCHAR *drive;
    DBCHAR colname1[MAXNAME];
    DBCHAR colname2[MAXNAME];
    DBINT paramlength;
    DBCHAR rootname[16];
    DBCHAR bErrorMsg[MAXLEN];

    int drivenum;
    int drivenums;
    int secPerCluster;
    int bytesPerSector;
    int freeClusters;
    int totalClusters;
    int space_remaining = -1;

    // Check number of parameters
    //
    if (srv_rpcparams(srvproc) > 0) {

	// Allocation local storage for drive name.
	//
	paramlength = srv_paramlen(srvproc, 1);
	drive = (DBCHAR *)malloc(paramlength);
	if (!drive) {

	    SETERROR("Malloc", bErrorMsg);
	    srv_sendmsg(srvproc, SRV_MSG_ERROR, CMDSHELL_ERROR, SRV_INFO, (DBTINYINT)0,
		       NULL, 0, 0, bErrorMsg, SRV_NULLTERM);
	    // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	    // result set of an Extended Stored Procedure.
	    //
	    srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	    return(XP_ERROR);
	}

	// Fetch drive letter.
	srv_bmove(srv_paramdata(srvproc, 1), drive, paramlength);
	drive = strupr(drive);

    } else

	// Default drive is C.
	drive = "C";

    drivenums = GetLogicalDrives();
    drivenum = drive[0] - 'A' + 1;

    drivenums >>= drivenum - 1; //Ignore drives A and B
    if (drivenums & 0x01) {

        sprintf(rootname, "%c:\\", drive[0]);
        GetDiskFreeSpace(rootname, &secPerCluster, &bytesPerSector,
                         &freeClusters, &totalClusters);

        space_remaining = secPerCluster * freeClusters * bytesPerSector;

    }

    // Process return values
    //
    if (srv_paramstatus(srvproc, 2) < 0 ) {
	// Setup column heading
	sprintf(colname1, "drive");
	sprintf(colname2, "space");
	srv_describe(srvproc, 1, colname1, SRV_NULLTERM, SRVCHAR, 1, SRVCHAR, 1,
		     (BYTE *)drive);
	srv_describe(srvproc, 2, colname2, SRV_NULLTERM,
		     SRVINT4, sizeof(space_remaining), SRVINT4,
		     sizeof(space_remaining), &space_remaining);

	srv_sendrow(srvproc);
	srv_senddone(srvproc, (SRV_DONE_COUNT | SRV_DONE_MORE), 0, 1);

   }
    else {
	srv_paramset(srvproc, 2, (BYTE *)&space_remaining, 4);
	srv_senddone(srvproc, SRV_DONE_MORE, 0, 0);
    }

    return(XP_NOERROR);
}



//
// XP_SCAN_XBASE
//    Reads an xBase file and sends it to the client as if it were a SQL
//    Server query result set (the equivalent of a 'SELECT * FROM
//    tablename' SQL statement).
//
// Parameters:
//    srvproc - the handle to the client connection that got the SRV_CONNECT.
//
// Returns:
//    XP_NOERROR
//    XP_ERROR
//
// Side Effects:
//    Returns messages and/or a result set to client
//
RETCODE xp_scan_xbase(srvproc)
SRV_PROC *srvproc;
{
    int paramnum;
    DBINT paramtype;
    DBCHAR *filename;
    FILE *xbasefile;
    size_t count;
    char buffer[XBASE_BUF_SIZE];
    short numrecords;
    short headerlength;
    short recordlength;
    short lengthlist[XBASE_MAX_COLUMNS];
    int i;
    short j;
    short position;
    short numcolumns;

    // Get number of parameters
    //
    paramnum = srv_rpcparams(srvproc);

    // Check number of parameters
    //
    if (paramnum != 1) {
        // Send error message and return
        //
        srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error executing extended stored procedure: Invalid Parameter",
                    SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }

    // Check parameters for correct type
    //
    paramtype = srv_paramtype(srvproc, paramnum);
    if (paramtype != SRVVARCHAR) {
        // Send error message and return
        //
        srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0,
                    "Error executing extended stored procedure: Invalid Parameter Type",
                    SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }
    filename = srv_paramdata(srvproc, 1);

    // now read the database header info
    //
    if ((xbasefile = fopen(filename, "r")) == NULL) {
        srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error reading xBase file", SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
	return(XP_ERROR);
    }
    count = fread(buffer, XBASE_HDR_SIZE, 1, xbasefile);

    if (count == 0) {
        srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error reading xBase file", SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
        fclose(xbasefile);
	return(XP_ERROR);
    }
    numrecords = *((short *)&buffer[4]);
    headerlength = *((short *)&buffer[8]);
    recordlength = *((short *)&buffer[10]);
    numcolumns = (headerlength - 32 - 1) / 32;

    // now get the column header information
    //
    for (j = 0; j < numcolumns; j++) {
        count = fread(buffer,
        XBASE_HDR_SIZE,
        1,
        xbasefile);
        if (count == 0) {
            srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                        NULL, 0, 0, "Error reading xBase file", SRV_NULLTERM);
	         // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	         // result set of an Extended Stored Procedure.
	         //
            srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
            fclose(xbasefile);
	    return(XP_ERROR);
        }

        // we need to NULL terminate the column name (if it is a
        // full 11 characters int)
        //
        buffer[11] = '\0';

        // now find our the column length for this data buffer
        //
        lengthlist[j] = (short)buffer[16];

        // now 'describe' this column
        //
        srv_describe(srvproc, j + 1,// column number
                buffer,     // pointer to column name
                SRV_NULLTERM,// column name is NULL terminated
                SRVCHAR,    // datatype is char (xBase numbers are ASCII
                lengthlist[j],// column length
                SRVCHAR,    // destination datatype is also char
                lengthlist[j],// destination column length
                NULL);      // pointer to where the data will be

    }
    // now read the one byte 'column header seperator'
    //
    count = fread(buffer, 1, 1, xbasefile);
    if (count == 0) {
        srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                    NULL, 0, 0, "Error reading xBase file", SRV_NULLTERM);
	     // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	     // result set of an Extended Stored Procedure.
	     //
        srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
        fclose(xbasefile);
	return(XP_ERROR);
    }
    for (i = 0; i < numrecords; i++) {
        count = fread(buffer, recordlength, 1, xbasefile);
        if (count == 0 && !feof(xbasefile)) {
            srv_sendmsg(srvproc, SRV_MSG_ERROR, SCAN_XBASE_ERROR, SRV_INFO, (DBTINYINT)0,
                        NULL, 0, 0, "Error reading xBase file", SRV_NULLTERM);
	         // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	         // result set of an Extended Stored Procedure.
	         //
            srv_senddone(srvproc, (SRV_DONE_ERROR | SRV_DONE_MORE), 0, 0);
            fclose(xbasefile);
	    return(XP_ERROR);
        }

        // check to see if this is a deleted row
        //
        if (buffer[0] == '*')
            break;

        // Now set the length and data pointers for each column
        for (j = 0, position = 1; j < numcolumns; j++) {
            srv_setcollen(srvproc, j + 1, lengthlist[j]);
            srv_setcoldata(srvproc, j + 1, &buffer[position]);
            position += lengthlist[j];
        }

        // send the row to the client.
        //
        srv_sendrow(srvproc);
    }
	 // A SRV_DONE_MORE instead of a SRV_DONE_FINAL must complete the
	 // result set of an Extended Stored Procedure.
	 //
    srv_senddone(srvproc, SRV_DONE_COUNT | SRV_DONE_MORE, 0, i);
    fclose(xbasefile);
    return(XP_NOERROR);
}
