(*******************************************************************)
(*  Avatar level 1 Console driver.  Unit for providing a program   *)
(*  with proper Avatar levels 0, 0+, and 1 emulations.             *)
(*  Copyright (c) 1991 - 93 Gregory P. Smith                       *)
(*  All Rights Reserved                                            *)
(*-----------------------------------------------------------------*)
(*  Last Update:   April  4, 1993      v 1.50                      *)
(*=====================================*****************************)
(* Background sound facilities for use *)
(* with PAvatar 1.50 and greater.      *)
(* July '92: Now supports AdLib sound  *)
(***************************************)
Unit PAvtSND;
{$I PAVTVER.INC -- Checks compiler version and defines }
{$F-,O-  This unit can't be overlayed!! }

INTERFACE

Uses Dos, Crt,
     PAvtIO;

Procedure StuffSound(frq,dur:integer);                 { Hz, ticks }
Procedure Flush_Sound;                                 { flush sound queue }
Function  Sounds_Left: integer;                        { snds left in queue }
Procedure Sound_Finish;                                { wait for empty buf }
Procedure ANSISound(note:byte; oct:byte; ticks:byte);  { Used by ANSI music }
Procedure AvtSound(note:byte; oct:shortint; dur:byte); { AVT style sound }
Procedure Set_Sound_Backg(backg:boolean);              { In background? }
Procedure Sound_Exit;                                  { ExitProc }
Procedure FMSound(freq: word);
Procedure FMNoSound;
Procedure Sound2(freq: word);                          { AdLib supporting }
Procedure NoSound2;                                    { Crt like functions }
Procedure SoundBell;                                   { ^G bell sound }

const
  MaxSoundBuf = 80;      { Max number of notes in queue }

var
  OldInt1C : pointer;    { Old Timer Interrupt }
  OldEP : pointer;       { Old ExitProc }
  AdLib_Found : boolean; { AdLib compatible card present? }
  Use_AdLib : boolean;   { Use the AdLib card for sounds? }

IMPLEMENTATION

type
  Queue_Record = record
    freq  : integer;    { frequency in Hz }
    ticks : integer;    { ticks (18.2ths of a second) remaining for sound }
  end;
  TenBit = 0..$3FF;
  FM_Port = object                        { AdLib FM music channel }
              {------------------ Variables -----------------------}
              Addr,
              Data : TenBit;
              {------------------ Procedures ----------------------}
              Procedure WriteToReg(reg, dat: byte); { write to reg }
              Function Status : byte;                { port status }
              Function CardDetected : boolean;  { FM card present? }
              Procedure Reset;                    { Reset the card }
            end; { FM_Port }


const
  scount : integer = 0; { time left for current sound }
  sound_on : boolean = False; { using background sound? }

var
  Sound_Queue : Array[1..MaxSoundBuf] of Queue_Record; { Circular Buffer }
  Sound_Ptr : byte; { current pointer in Queue }
  Sound_End : byte; { insertion place in Queue }

{$F+}
{ $1C timer interrupt chain for background sound support }
procedure Timer_Sound; interrupt;
{$F-}
const
  InTone : boolean = True;
begin
  With Sound_Queue[Sound_Ptr] do
   begin
     if scount <= 1 then
      begin
        scount := 0; { done with tone }
        if InTone then
         begin
           InTone := False;
           NoSound2;
         end;
        if Sound_Ptr <> Sound_End then
         begin
           if freq <> 0 then
            begin
              Sound2(freq);
              InTone := True;
            end;
           scount := ticks;
           inc(Sound_Ptr);
           if Sound_Ptr > MaxSoundBuf then Sound_Ptr := 1;
         end;
      end { scount = 1 }
     else dec(scount); { one less tick left on sound }
   end; { with }
  Inline($9C / $FF / $1E / OldInt1C); { pushf, call far OldInt1C }
end;

{ Add a sound to the queue, waiting until there is room }
Procedure StuffSound(frq,dur:integer); { Hz, ticks }
const
  lastfreq : integer = -1;
  lastqnum : byte = 1;
begin
  if frq < 20 then frq := 20;
  if dur < 1 then exit;
  if Sound_Ptr = Sound_End then { if all sounds played }
   lastfreq := -1;
  if ((Sound_Stat and 2) = 0) then exit; { Sound is off }
  if not Sound_On then { if non-background sound }
   begin
     Sound2(frq);
     (* Delay(dur*50); *)
     DelayTicks(dur);
     NoSound2;
     exit;
   end;
  if In_DV AND (not (Use_AdLib and AdLib_Found)) then
   begin
     DV_Sound(frq,dur); { DESQview is in background }
     exit;
   end;
  if frq = lastfreq then  { if same as last note, make them run together }
   begin
     if lastqnum = Sound_Ptr then
      begin
        inline($FA); { CLI }
        inc(scount,dur);                   { add to current sound }
        inline($FB); { STI }
      end
     else
      inc(Sound_Queue[lastqnum].ticks,dur);  { add to queue }
     exit;
   end;
  while (Sound_End = pred(Sound_Ptr)) or ((Sound_End = MaxSoundBuf) and
         (Sound_Ptr = 1)) do { wait for room } ;
  with Sound_Queue[Sound_End] do
   begin
     freq := frq;
     ticks := dur;
   end;
  lastfreq := frq;
  lastqnum := Sound_End;
  inc(Sound_End);
  if Sound_End > MaxSoundBuf then Sound_End := 1; { wrap around }
end;

{ flushes the sound queue and halts currently playing sound; won't under DV }
procedure Flush_Sound;
begin
  Sound_End := Sound_Ptr;  { flush queue }
  scount := 0;  { end current sound }
  NoSound2;     { silence sound device }
end;

{ return the number of sounds left in the queue; 0 under DV }
function Sounds_Left: integer;
begin
  if Sound_End >= Sound_Ptr then
    Sounds_Left := Sound_End - Sound_Ptr
  else { queue wrapped around the array }
    Sounds_Left := (MaxSoundBuf - Sound_Ptr) + Sound_End;
end;

{ wait until the sound buffer is empty; returns immediately under DV }
procedure Sound_Finish;
begin
  while Sounds_Left > 0 do
    IdlePause;  { give up time until sounds finish }
end;

{$F+}
procedure Sound_Exit;
{$F-}
begin
  if not In_DV then
   begin
     SetIntVec($1C,OldInt1C); { restore timer }
     NoSound2; { turn off sound. }
     ExitProc := OldEP; { restore exit proc }
     Sound_On := False; { bg sound is off }
   end;
end;

procedure Set_Sound_Backg(backg:boolean);
begin
  if backg then { set sound to background mode }
   begin
     Sound_On := True;
     if not In_DV then { use interrupt outside of DESQview }
      begin
        FillChar(Sound_Queue,SizeOf(Sound_Queue),1);
        Sound_End := 1;
        Sound_Ptr := 1;
        if ExitProc = @Sound_Exit then exit; { never reinitalize }
        OldEP := ExitProc;           { get old exit proc }
        GetIntVec($1C,OldInt1C);     { get old timer int }
        ExitProc := @Sound_Exit;     { set exit proc }
        SetIntVec($1C,@Timer_Sound); { set sound timer int }
      end;
   end
  else { foreground }
   begin
     if Sound_On and (not In_DV) then
      begin
        SetIntVec($1C,OldInt1C); { restore timer }
        NoSound;
        ExitProc := OldEP; { restore exit proc }
      end;
     Sound_On := False;
   end;
end; { Set_Sound_Backg }

procedure ANSISound(note:byte; oct:byte; ticks:byte);
Const { Avatar note #, normalized octave, # of timer ticks }
  Notes               : Array[1..96] Of Word =
  { C    C#,D-  D    D#,E-  E     F    F#,G-  G    G#,A-  A    A#,B-  B  }
  (0033, 0035, 0037, 0039, 0041, 0044, 0046, 0049, 0052, 0055, 0058, 0062,
   0065, 0069, 0073, 0078, 0082, 0087, 0093, 0098, 0104, 0110, 0117, 0123,
   0131, 0139, 0147, 0156, 0165, 0175, 0185, 0196, 0208, 0220, 0233, 0247,
   0262, 0277, 0294, 0311, 0330, 0349, 0370, 0392, 0415, 0440, 0466, 0494,
   0523, 0554, 0587, 0622, 0659, 0698, 0740, 0784, 0831, 0880, 0932, 0987,
   1047, 1109, 1175, 1245, 1329, 1397, 1480, 1568, 1661, 1760, 1865, 1976,
   2093, 2217, 2349, 2489, 2637, 2794, 2960, 3136, 3322, 3520, 3729, 3951,
   4186, 4435, 4699, 4978, 5274, 5588, 5920, 6272, 6645, 7040, 7459, 7902);
  Trans : Array[0..13] of byte = (10,11,12,1,1,2,3,4,5,6,6,7,8,9);
var
  tone : byte; { tone index in the Notes array }
begin
  if note < 14 then
   begin
     tone := trans[note];
     tone := tone + (oct * 12);
     if oct < 0 then oct := 0
      else if oct > 7 then oct := 7;
     if (note = 3) and (oct < 7) then inc(tone,12); { B# is high C }
     StuffSound(Notes[tone],ticks);
   end
  else
   StuffSound(32767,ticks); { inaudible = musical pause }
end;

procedure AvtSound(note:byte; oct:shortint; dur:byte);
begin
  ANSISound(note, oct+3, dur * 2); { dur * 2 is approx 1/10 seconds }
end;

{------------------------- FM_Port Implementation --------------------------}

const
  FWaveCTL  = $01;
  FTimer1   = $02;
  FTimerCTL = $04;

Procedure FM_Port.WriteToReg(reg, dat:byte);
var
  misc_count : byte;
begin
  Port[Addr] := reg;         { select the register }
  for misc_count := 1 to 6 do
   reg := Port[Addr];        { read Reg port 6x to delay 3.3 us }
  Port[Data] := dat;         { write the data }
  for misc_count := 1 to 35 do
   dat := Port[Addr];        { read Reg port 35x to delay 23 us }
end;

Function FM_Port.Status : byte;
begin
  Status := Port[Addr];
end;

Function FM_Port.CardDetected : boolean;
var
  tmp, tmp2 : byte;
begin
  WriteToReg(FTimerCTL, $60); { write 60h to reg 4 to reset timers }
  WriteToReg(FTimerCTL, $80); { enable interrupts }
  tmp := Status;
  WriteToReg(FTimer1, $FF);   { write FFh to reg 2 (timer 1) }
  WriteToReg(FTimerCTL, $21); { start timer 1 }
  Delay(1);                   { delay at least 80us (1000us here) }
  tmp2 := Status;
  WriteToReg(FTimerCTL, $60); { write 60h to reg 4 to reset timers }
  WriteToReg(FTimerCTL, $80); { enable interrupts }
  CardDetected := ((tmp AND $E0) = $00) AND ((tmp2 AND $E0) = $C0);
end;

Procedure FM_Port.Reset;
var
  reg : byte;
begin
  WriteToReg(FWaveCTL, $20); { Enable waveform changes  }
  for reg := $02 to $F5 do
   WriteToReg(reg, 0);
  WriteToReg(FWaveCTL, $00); { Disable waveform changes }
end;

const
  AdLib : FM_Port = (Addr: $388; Data: $389);

Procedure FMSound(freq: word);
var
  fnum : word;
begin
  if not AdLib_Found then exit;
  if freq < 1 then freq := 1; { use freqs < 25 with CAUTION... }
  if freq > 6243 then freq := 6243;
  case freq of
    1..64      : fnum := $0400 or round(freq * 10.48576); { octave 1 sounds }
    65..128    : fnum := $0800 or round(freq * 5.24288);  { octave 2 sounds }
    129..256   : fnum := $0C00 or round(freq * 2.62144);  { octave 3 sounds }
    257..512   : fnum := $1000 or round(freq * 1.31072);  { octave 4 sounds }
    513..1024  : fnum := $1400 or round(freq * 0.65536);  { octave 5 sounds }
    1025..2048 : fnum := $1800 or round(freq * 0.32768);  { octave 6 sounds }
    2049..6243 : fnum := $1C00 or round(freq * 0.16384);  { octave 7 sounds }
  end;
  with AdLib do
   begin
     WriteToReg($32, $01); { set modulator's multiple to 1 }
     WriteToReg($52, $10); { set modulator's level to about 40 dB }
     WriteToReg($72, $f0); { modualtor Attack: Quick, Decay: Long }
     WriteToReg($92, $77); { modulator sustain & release: medium }
     WriteToReg($35, $01); { set carrier's multiple to 1 }
     WriteToReg($55, $00); { set carrier to max. volume (aprox. 47 dB) }
     WriteToReg($75, $f0); { carrier attack: quick, decay: long }
     WriteToReg($95, $77); { carrier sustain & release: medium }
     WriteToReg($a8, lo(fnum));         { send lo multiplier }
     WriteToReg($b8, $20 or hi(fnum));  { Key ON, octave and hi multiplier }
   end;
end;

Procedure FMNoSound;
begin
  if not AdLib_Found then exit;
  AdLib.WriteToReg($b8, $00);  { Key OFF }
end;

Procedure Sound2(freq: word);
begin
  if freq <> 32767 then  { 32767 is our silent (pause) frequency }
   if AdLib_Found AND Use_AdLib then FMSound(freq)
    else Sound(freq);
end;

Procedure NoSound2;
begin
  if AdLib_Found AND Use_AdLib then FMNoSound
   else NoSound;
end;

Procedure SoundBell;
begin
  if (Sound_Stat and 1) = 1 then begin  { bell enabled }
    AvtSound(5, 0, 1);   { 3rd octave C for 1/10th second }
    ANSISound(14, 0, 1); { pause for a tick }
  end;
end;

BEGIN
  AdLib_Found := AdLib.CardDetected;
  Use_AdLib := True;
  if AdLib_Found then AdLib.Reset;
END. { unit }
