(*******************************************************************)
(*  Avatar level 1 Console driver.  Unit for providing a program   *)
(*  with proper Avatar levels 0, 0+, and 1 emulations.             *)
(*  Copyright (c) 1991 - 93 Gregory P. Smith                       *)
(*  All Rights Reserved                                            *)
(*-----------------------------------------------------------------*)
(*  Last Update:   March 10, 1993                                  *)
(*.................................................................*)
(*  Special thanks goes to Bo Bendtsen of Albertslund, Denmark     *)
(*  for the ANSI music routines!  Also for finding a bug in the    *)
(*  partial clear screen handling.  Thanks to Tom Thayer for       *)
(*  finding a bug in the cursor positioning when less that two     *)
(*  parameters were passed.                                        *)
(*===============================================================***)
(* Include file containing code to be linked in with the PAvatar *)
(* unit to provide ANSI-BBS/ANSI/VT52 support.                   *)
(*****************************************************************)
type
  TabSet = Set of 1..132;                           { TAB (^I) stops }
{$IFDEF VT102}
  VT100Scrol = record  { VT100 Scrolling Window Definitions }
    top,
    bottom : byte;
    relative : boolean;
  end;
{$ENDIF}

const
  DefaultTabSet = [9,17,25,33,41,49,57,65,73,81,89,97,105,113,121,129];
  HTabs : TabSet = DefaultTabSet;
{$IFDEF VT102}
  RelativePos : boolean = False; { cursor pos relative to Scrolling region? }
  InQMarkCmd : boolean = False; { are we in a Esc[?... type command? }
{$ENDIF}

var
{$IFDEF VT52}
  EightBitControl,    { ANSI/VT52 Eight bit control codes active? }
{$ENDIF}
  Reverse : boolean;  { are ANSI attributes (not blink & bold) reversed? }
{$IFDEF ANSI_MUSIC}
  In_Ciffer : boolean;                { calc music number }
  Octave, Tempo, Duration, MusicMode, { ansi music controls }
{$ENDIF}
  CurPrm : byte;               { current parameter }
  Saved : Array[1..3] of byte; { (1,1,7) = (x,y,a) }

{-- forward references to the ANSI procedures -----------------------}

{$F+}
Procedure ANSI_Waiting(ch:char); forward;
Procedure ANSI_Bracket(ch:char); forward;
{$IFNDEF VT102}
Procedure ANSI_EatParams(ch:char); forward;
{$ENDIF}
Procedure ANSI_InParam(ch:char); forward;
{$IFDEF ANSI_MUSIC}
Procedure ANSI_InMusic(ch:char); forward;
{$ENDIF}
Procedure Sec_Params(ch:char); forward;
{$IFNDEF OVERLAY}
{$F-}
{$ENDIF}
Procedure ANSI_Command(ch:char); forward;

{-- Misc. Support Code ----------------------------------------------}

Procedure GotoTab;
var
  t : byte;
begin
  with ActWin do
   begin
     if cx < Width then begin
       for t := succ(cx) to Width do
        if t in HTabs then
         begin
           SetXY(t,cy);
           exit;
         end;
     end;
     { if we reach here, no tabs were found; Do nothing }
   end;
end;

{-- code for ANSI interpreter ---------------------------------------}

Procedure ANSI_Waiting(ch:char);
{ employs a cut down TTY for use in ActWin as well as the first
  stage of ansi interpretation. The AVT/1 parser will usually handle
  this, but it is here so that it works standalone.                  }
begin
  if ch = #27 then
   begin
     AVTInterp := ANSI_Bracket;
     exit;
   end;
{$IFDEF VT52}
  if EightBitControl and (ch >= #$80) and (ch <= #$9F) then
   begin
     ANSI_Bracket(chr(ord(ch) - $40));       { treat as Esc <ch-$40> }
     exit;
   end;
{$ENDIF}
{$IFDEF TTY_HOOK}
  if not TTYCharH(ch) then exit;         { Can TTY continue process? }
{$ENDIF}
  with ActWin do
   if ch in [^G,^H,^I,^J,^L,^M] then
    begin
      case ch of
        ^M : begin
               SetXY(1,cy); { CR }
               Insert := False; { insert off for ANSI & VT series }
             end;
        ^J : begin { LF }
               if cy = Depth then
                Scroll(1,1,1,Width,Depth,1)
               else
                begin
                  SetXY(cx,succ(cy));
                  if Insert then Scroll(2, 1, cy, Width, Depth, 1);
                end;
             end; { ^J }
        ^H : begin { Destructive BS }
               SetXY(pred(cx),cy);
               if Insert then Scroll(3, cx, cy, Width, cy, 1)
                else
                 if Dest_BS then WriteAT(cx,cy,attr,' ');
             end;
        ^I : GotoTab; { Move to next set tab stop }
        ^L : begin
               Clear(7,' '); { Clear Screen }
               SetXY(1,1);
             end;
        ^G : SoundBell; { bell }
      end; { case ch of }
    end { if ch in ^G..^M }
   else { if }
    begin
      if (ch = #0) and IgnoreNULL then exit;  { Ignore the null character }
      if Insert then Scroll(4, cx, cy, Width, cy, 1); { insert char }
      WriteAT(cx,cy,attr,ch);
      if cx = Width then
       begin
         if cy = Depth then
          Scroll(1,1,1,Width,Depth,1);
         SetXY(1,succ(cy));
       end
      else
       SetXY(succ(cx),cy);
    end; { if -- else }
end; { ANSI_Waiting (aka ansi_tty) }

Procedure ANSI_Bracket(ch:char);
var
  b : byte;

 procedure specexit;
 begin
   if fallback and awake then AvtInterp := AVT1_Waiting
    else AvtInterp := ANSI_Waiting;
   ANSI_Waiting(ch); { low level TTY of char }
   exit;
 end;

begin
  with ActWin do
   case ch of
     '[' : begin { Common ANSI sequence }
             CurPrm := 1;
             AVTInterp := ANSI_InParam;
             FillChar(PrmArr,SizeOf(PrmArr),0);
             exit;
           end; { [ }
     'A' : begin { cursor up }
             SetXY(cx,pred(cy));
           end;
     'B' : begin { cursor down }
             SetXY(cx,succ(cy));
           end;
     'C' : begin { cursor right }
             SetXY(succ(cx),cy);
           end;
     'D' : begin
{$IFDEF VT52}
             if VT52On then { cursor left }
              SetXY(pred(cx),cy)
             else
              begin { index down w/scroll }
{$ENDIF}
                if succ(cy) > Depth then
                 Scroll(1,1,1,Width,Depth,1)
                else
                 SetXY(cx,succ(cy));
{$IFDEF VT52} end; {$ENDIF}
           end; { D }
     'E' : begin
{$IFDEF VT52}
             if VT52On then { clear screen }
              Clear(7,' ')
             else
              begin { CR/LF (basically) }
{$ENDIF}
                if cy = Depth then
                 Scroll(1,1,1,Width,Depth,1)
                else
                 SetXY(1,succ(cy));
{$IFDEF VT52} end; {$ENDIF}
           end; { E }
     'H' : begin
{$IFDEF VT52}
             if VT52On then { home cursor }
              SetXY(1,1)
             else { set horizontal tab at cp }
{$ENDIF}
              HTabs := HTabs + [cx];
           end; { H }
     'I' : begin
{$IFDEF VT52}
             if VT52On then { Reverse LF }
              begin
                if cy = 1 then
                 Scroll(2,1,1,Width,Depth,1)
                else
                 SetXY(cx,pred(cy));
              end
             else
{$ENDIF}
              GotoTab;
           end; { I }
{$IFDEF VT52}
     'J' : begin { VT52 ClrEos }
             if VT52On then FillArea(1,cy,Width,Depth,attr,#0)
              else specexit;
           end;
     'K' : begin { VT52 ClrEol }
             if VT52On then FillArea(cx,cy,Width,cy,attr,#0)
              else specexit;
           end;
     'L' : begin { VT52 Insert Line }
             if VT52On then Scroll(2,1,succ(cy),Width,Depth,1)
              else specexit;
           end;
{$ENDIF}
     'M' : begin
{$IFDEF VT52}
             if VT52On then
              Scroll(1,1,cy,Width,Depth,1) { DelLine }
             else
              begin
{$ENDIF}
                if cy = 1 then
                 Scroll(2,1,1,Width,Depth,1)
                else
                 SetXY(cx,pred(cy));
              end;
{$IFDEF VT52}
           end;
     'N' : begin { VT52 DelChar }
             if VT52On then Scroll(3,cx,cy,Width,cy,1)
              else specexit;
           end;
     'O' : begin { VT52 Insert Off }
             if VT52On then Insert := False
              else specexit;
           end;
     'Y',' ' : begin { move cursor row+31,col+31 and 8 bit control toggle }
             if VT52On then
              begin
                AVTInterp := Sec_Params;
                FillChar(PrmArr,SizeOf(PrmArr),0);
                CurPrm := 1;
                PrmArr[10] := ord(ch);
                exit;
              end
             else specexit;
           end;
{$ENDIF}
     'Z' : begin { request ID }
{$IFDEF VT52}
             if VT52On then
              Query_Hook(#27+'/Z') { VT100 emulating a VT52 }
             else
{$ENDIF}
              Query_Hook(QueryReply);
           end;
{$IFDEF VT52}
     'F' : begin
             if not VT52On then specexit;
             { VT52 Select Graphics Character Set }
           end;
     'G' : begin
             if not VT52On then specexit;
             { VT52: Select Text Character Set }
           end;
{$ENDIF}
     '@' : Insert := True;    { Insert mode on }
     '>' : { NumLock On  } ;
     '=' : { NumLock Off } ;
     '7' : begin { save cursor & attributes }
             Saved[1] := cx;
             Saved[2] := cy;
             Saved[3] := attr;
           end;
     '8' : begin { restore cursor & attributes }
             SetXY(Saved[1],Saved[2]);
             attr := Saved[3];
           end;
    else specexit; { else ch }
   end; { case ch }
  if fallback and awake then AvtInterp := AVT1_Waiting
   else AvtInterp := ANSI_Waiting;
end; { ANSI_Bracket }

{$IFNDEF VT102}
Procedure ANSI_EatParams(ch:char);
begin
  if ch in ['0'..'9',';'] then exit; { eat params }
  if fallback and awake then AVTInterp := AVT1_Waiting { return to AVT/1 con }
   else AVTInterp := ANSI_Waiting;
end;
{$ENDIF}

Procedure ANSI_InParam(ch:char);
begin
  case ch of
    '0'..'9' : PrmArr[CurPrm] := PrmArr[CurPrm]*10 + (ord(Ch) - ord('0'));
    ';' : if CurPrm < NumPrms then inc(CurPrm);
{$IFNDEF VT102}
    '?' : AvtInterp := ANSI_EatParams; { ignore re/set mode commands }
{$ELSE}
    '?' : InQMarkCmd := True;  { this is an extended Esc[?... command }
{$ENDIF}
{$IFDEF ANSI_MUSIC}
    'M' : if ANSI_MUSIC and (CurPrm = 1) then
           begin
             Fillchar(PrmArr, sizeof(PrmArr) ,0);
             PrmArr[1]:= ord('M'); { need M when translating }
             CurPrm:=2;
             AvtInterp:=Ansi_InMusic;
           end
          else ANSI_Command(ch);
{$ENDIF}
   else { case }
    ANSI_Command(ch);
  end; { case ch }
end; { ANSI_InParam }

{$IFDEF ANSI_MUSIC}
{*********************** ANSI Music Section ****************************}
Procedure ANSI_PlayMusic;
type
  MusicArrType = Array[1..NumPrms] of char;
const
  TicksPerMin = 1092; { timer ticks per minute }
  msPerTick = 55;     { miliseconds per timer tick }
var
  fr,du,cy,de,va : Longint;
  x,y,i : integer;
  si,tone:shortint;
  tmps : string[5];
  MArr : MusicArrType absolute PrmArr;
begin
  if fallback and awake then AVTInterp := AVT1_Waiting { return to AVT/1 con }
   else AVTInterp := ANSI_Waiting;

  i:=1;
  while i<CurPrm do
   begin
    case MArr[i] of
      'M'      : begin
                  Case MArr[i+1] of
                   'F' : {Set_sound_backg(false)};
                   'B' : {Set_sound_backg(true)};
                   'N' : MusicMode:=1;
                   'L' : MusicMode:=2;
                   'S' : MusicMode:=3;
                  end;
                  inc(i);
                 end;
      { A=0 A#=1 B=2 B#=3 C=4 C#=5 D=6 D#=7 E=8 E#=9 F=10 F#=11 G=12 G#=13 }
      'A'..'G' : begin
                   tone:=(Ord(MArr[i])-Ord('A'))*2;
                   x:=0;
                   if MArr[i+1]='+' then
                   begin
                     inc(tone);
                     if tone>13 then begin tone:=0; x:=1; end;
                     inc(i);
                   end
                   else if MArr[i+1]='-' then
                   begin
                     dec(tone);
                     If tone<0 Then begin tone:=13; x:=-1; end;
                     inc(i);
                   end;
                   si:= octave + x;
                   du := Duration;
                   If MArr[i+1] in ['0'..'9'] then
                   begin
                     x:=i+1;
                     while MArr[x+1] in ['0'..'9'] Do Inc(x);
                     y:=x-i;
                     Move(MArr[i+1], tmps[1], x-i);
                     tmps[0] := chr(x-i);
                     Val(tmps, du, x);
                     if du > 64 then du := 64;
                     if du <= 0 then du := 1;
                     inc(i,y);
                   end;
                   { calculate length of time for the note to play.  }
                   { ticks/min / beats/min / quarter note per beat / }
                   { note type }
                   y:=(TicksPerMin Div (Tempo Div 4)) Div du;
                   if y = 0 then y := 1; { always play a note }

                   while MArr[i+1]='.' Do
                   Begin;
                     y:=y*15 Div 10;
                     Inc(i);
                   End;

                   if (Sound_Stat and $02) = 2 then begin { music enabled }
                     Case MusicMode of
                      1: Begin { regular note, 7/8 play time }
                          (* AvtSound(tone,si,y*70 Div 8);
                           AvtSound(14,0,y*10 Div 8); *)
                           ANSISound(tone,si,y*7 DIV 8 );  { note }
                           ANSISound(14,0,y DIV 8); { short delay }
                         End;
                      2: ANSISound(tone,si,y); { legato, full time }
                      3: Begin { sticatto, 6/8 play time }
                           y:=y*3 Div 4;
                           If y=0 Then
                           Begin
                             ANSISound(tone,si,1);
                             ANSISound(14,si,1)
                           End
                           Else Begin
                             ANSISound(tone,si,y*3 Div 4);
                             ANSISound(14,0,y Div 4);
                           End;
                         End;
                     End;
                   end; { if Sound_Stat.. }

                 end;
      'N'      : ; { Note lav en konverter fra 0-34 til octave, tone }
      'O'      : begin
                   Octave:=Ord(MArr[i+1]);
                   if Octave>6 then Octave:=6;
                   inc(i)
                 end;
      'L'      : begin
                   Duration:=Ord(MArr[i+1]);
                   if Duration>64 then Duration:=64;
                   if Duration=0 then Duration:=1;
                   inc(i);
                 end;
      'P'      : Begin
                   x:=Ord(MArr[i+1]);
                   if x>64 then x:=64;
                   ANSISound(14,0,(((TicksPerMin Div (Tempo Div 4)) Div Duration)*x));
                   inc(i);
                 End;
      'T'      : begin
                   Tempo:=Ord(MArr[i+1]);
                   if Tempo<32 then Tempo:=32;
                   inc(i);
                 end;
      '>'      : if Octave<6 then inc(Octave);
      '<'      : if Octave>0 then dec(Octave);
      ';',
      '0'..'9' : begin
                   if MArr[i+1] in [';','0'..'9'] then
                   begin
                     fr:=0;du:=0;cy:=0;de:=0;va:=0;
                     if MArr[i]=';' then inc(i)
                     else while (MArr[i]<>';') and (i<CurPrm) Do
                     begin fr:=fr*10 + (ord(MArr[i]) - ord('0')); inc(i); end;
                     if fr<0 Then fr:=0; if fr>10000 Then fr:=10000;
                     inc(i);
                     while (MArr[i]<>';') and (i<CurPrm) Do
                     begin du:=du*10 + (ord(MArr[i]) - ord('0')); inc(i); end;
                     if du<0 then du:=0; if du>10000 then du:=10000;
                     inc(i);
                     while (MArr[i]<>';') and (i<CurPrm) Do
                     begin cy:=cy*10 + (ord(MArr[i]) - ord('0')); inc(i); end;
                     if cy<0 then cy:=0; if cy>10000 then cy:=10000;
                     inc(i);
                     while (MArr[i]<>';') and (i<CurPrm) Do
                     begin de:=de*10 + (ord(MArr[i]) - ord('0')); inc(i); end;
                     if de<0 then de:=0; if de>65535 then de:=65535;
                     inc(i);
                     si:=1;
                     if MArr[i]='-' then begin si:=-1; inc(i); end;
                     while (MArr[i]<>' ') and (i<CurPrm) Do
                     begin va:=va*10 + (ord(MArr[i]) - ord('0')); inc(i); end;
                     va:=va * si;
                     if va<-1000 then va:=-1000; if va>1000 then va:=1000;
                     if (Sound_Stat and $02) = 2 then begin
                       for x:=1 to cy do
                       begin
                         StuffSound(fr,du div msPerTick);
                         Delay(de);
                         inc(fr,va);
                       end;
                     end; { if Sound_Stat.. }
                   end;
                 end;

    end;
    inc(i);
   end;
end;

Procedure ANSI_InMusic(ch:char);
begin    { #14 for end-of-music command, others are for compatibility }
  if not (ch in [#14,#27,#13,#10]) and (CurPrm < NumPrms) Then
   begin
     if (ch in ['0'..'9']) and not (chr(PrmArr[CurPrm-1]) in [';',' ','0'..'9','A'..'G','+','#','-']) then
      begin
        In_Ciffer:=True;   { Converts numbers to a byte used by _PlayMusic }
        PrmArr[CurPrm] := PrmArr[CurPrm]*10 + (ord(Ch)-ord('0'));
      end
     else begin
       if In_Ciffer then begin In_Ciffer:=False; Inc(CurPrm); end;
       PrmArr[CurPrm]:= ord(upcase(ch));
       inc(CurPrm);
     end
   end
  else
    ANSI_PlayMusic;
end; { ANSI_InMusic }
{$ENDIF}

Procedure Sec_Params(ch:char);
begin { process VT52 & extended ANSI commands }
  case chr(PrmArr[10]) of
{$IFDEF VT52}
    'Y' : begin { VT52 Set Cursor Position }
            PrmArr[CurPrm] := ord(ch)-31;
            if CurPrm >= 2 then
             begin
               ActWin.SetXY(PrmArr[2],PrmArr[1]);
               if fallback and awake then AvtInterp := AVT1_Waiting
                else AvtInterp := ANSI_Waiting;
               CurPrm := 1;
             end
            else inc(CurPrm);
          end;
{$ENDIF}
    ' ' : begin
            case ch of
{$IFDEF VT52}
              'F' : EightBitControl := False;
              'G' : EightBitControl := True;
{$ENDIF}
              '@' : With ActWin do { scroll left }
                     Scroll(3,1,1,Width,Depth,PrmArr[1]);
              'A' : With ActWin do { scroll right }
                     Scroll(4,1,1,Width,Depth,PrmArr[1]);
            end;
          end; { ' ' }
{$IFDEF VT102}
    '#' : begin { line height & screen alignment tests }
            if ch = '8' then
             With ActWin do
              Clear(attr,'E'); { fill screen with Es }
          end;
{$ENDIF}
  end; { case }
end; { Sec_Param }

Procedure ANSI_Command(ch:char);
var
  qs  : Array[1..2] of String[2];
  i   : integer absolute qs; { never needed at the same time as qs }
  a   : byte;    { " }
begin
  if fallback and awake then AVTInterp := AVT1_Waiting { return to AVT/1 con }
   else AVTInterp := ANSI_Waiting;
   case ch of
     { Note: the order of commands is optimized for execution speed }
     'm' :                 {sgr}
       begin
         a := ActWin.attr;
         for i := 1 to CurPrm do begin
{$IFNDEF DOS_ANSI_REV}
           if Reverse then  { swap RGB bytes only for reverse }
             a := (((a shr 4) or (a shl 4)) and $77) or (a and $88);
{$ENDIF}
           Case PrmArr[i] Of
             0 : begin                   { reset attributes }
                   a := $07;
{$IFNDEF DOS_ANSI_REV} Reverse := False; {$ENDIF}
                 end;
             1 : a := a and $FF or $08;  { bold (intensity bit) on }
             5 : a := a or $80;          { blinking on }
             30..37 : case PrmArr[i] of
                        30 : a := a and $F8{ or $00};
                        31 : a := a and $F8 or $04;
                        32 : a := a and $F8 or $02;
                        33 : a := a and $F8 or $06;
                        34 : a := a and $F8 or $01;
                        35 : a := a and $F8 or $05;
                        36 : a := a and $F8 or $03;
                        37 : a := a and $F8 or $07;
                      end;
             40..47 : case PrmArr[i] of
                        40 : a := a and $8F{ or $00};
                        41 : a := a and $8F or $40;
                        42 : a := a and $8F or $20;
                        43 : a := a and $8F or $60;
                        44 : a := a and $8F or $10;
                        45 : a := a and $8F or $50;
                        46 : a := a and $8F or $30;
                        47 : a := a and $8F or $70;
                      end;
             21, 22, 2  : a := a and $F7{ or $00};  { turn off bold }
{$IFDEF DOS_ANSI_REV} { DOS/VT-100 Style of reverse attributes }
             7  : begin
                    a := (a and $88) or $70; { black on white }
                  end;
             27 : begin
                    a := (a and $88) or $07; { white on black }
                  end;
{$ELSE}               { VT-200 style reverse attributes }
             7  : Reverse := True;
             27 : Reverse := False;
{$ENDIF}
             24 : a := a and $F8 or $04; { turn off underline }
             25 : a := a and $7F; { turn off blinking }
             4  : a := a and $F8 or $01;  { underline (on mono cards) }
             8  : a := a and $f0 or ((a and $f0) shr 4); { invisible }
           end; { case }
{$IFNDEF DOS_ANSI_REV}
           { fixup for reverse mode }
           if Reverse then
             a := (((a shr 4) or (a shl 4)) and $77) or (a and $88);
{$ENDIF}
         end; { for loop }
         ActWin.attr := a; { set new attribute }
       end;
     'C' : begin   {cuf}
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              SetXY(cx + PrmArr[1], cy);
           end;
     'D' : begin   {cub}
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              SetXY(cx - PrmArr[1], cy);
           end;
     'A' : begin   {cup}
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              SetXY(cx,cy - PrmArr[1]);
           end;
     'B' : begin   {cud}
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              SetXY(cx,cy + PrmArr[1]);
           end;
     'H','f' : begin
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             if PrmArr[2] = 0 then PrmArr[2] := 1;
             with ActWin do  {cup,hvp}
               SetXY(PrmArr[2],PrmArr[1]);
           end;
     'J' : with ActWin do       {EID}
            begin
              if ANSI_BBS and (CurPrm = 1) then PrmArr[1] := 2; { BBS defualt }
              case PrmArr[1] of
                2 : begin
                      Clear(attr,' ');
                      if ANSI_BBS then SetXY(1,1); { BBS ANSI moves home }
                    end;
                0 : begin  { ClrEos }
                      if cy < depth then { to EOS }
                        FillArea(1,succ(cy),Width,Depth,attr,#0);
                      FillArea(cx,cy,Width,cy,attr,#0); { to EOL }
                    end;
                1 : begin  { ClrBos }
                      if cy > 1 then
                        FillArea(1,1,Width,pred(cy),attr,#0);
                      FillArea(1,cy,cx,cy,attr,#0); { from BOL }
                    end;
              end; { case }
            end;
     'K' : with ActWin do     {eil}
            case PrmArr[1] Of
              0 : FillArea(cx,cy,Width,cy,attr,#0); { DelEol }
              1 : FillArea(1,cy,cx,cy,attr,#0); { DelBol }
              2 : FillArea(1,cy,Width,cy,attr,#0); { ClrLine }
            end; { case PrmArr }
     'L' : with ActWin do   {il }
            if PrmArr[1] > 0 then Scroll(2,1,cy,Width,Depth,PrmArr[1]);
     'M' : with ActWin do   {d_l}
            if PrmArr[1] > 0 then Scroll(1,1,cy,Width,Depth,PrmArr[1]);
     'P' : begin  {dc }
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              Scroll(3,cx,cy,Width,cy,PrmArr[1]);
           end;
     '@' : begin  {ic }
             if PrmArr[1] = 0 then PrmArr[1] := 1;
             with ActWin do
              Scroll(4,cx,cy,Width,cy,PrmArr[1]);
           end;
     's' : begin           {scp}
             Saved[1] := ActWin.cx;
             Saved[2] := ActWin.cy;
           end;
     'u' : ActWin.SetXY(Saved[1],Saved[2]); {rcp}
     'n' : begin           {dsr}
             case PrmArr[1] of
               5 : begin
                     Query_Hook(#27+'[0n'); { VT100 ready status }
                   end;
               6 : begin
                     Str(ActWin.cx,qs[1]);
                     Str(ActWin.cy,qs[2]);
                     if (PrmArr[1] = 6) then
                      Query_Hook(#27+'['+qs[1]+';'+qs[2]+'R'); { return cp }
                   end;
             end; { case }
           end;
{$IFDEF VT102}
     'r' : With ActWin do begin { set top & bottom scrolling margins }
             if CurPrm = 1 then begin { reset margins }
               y1 := 1;
               y2 := ScrnLines;
             end else begin
               y1 := PrmArr[1]; { top margin }
               if CurPrm > 2 then y2 := PrmArr[2]; { bottom margin }
             end;
             if RelativePos then
               Depth := succ(y2 - y1); { update depth variable }
           end;
     'g' : begin { tabs clear }
             case PrmArr[1] of
               0 : HTabs := HTabs - [ActWin.cx]; { remove this tab stop }
               3 : HTabs := []; { clear tabs, empty set }
             end;
           end;
     'c' : begin
             Query_Hook(#27+'[?6c'); { device attributes }
           end;
     'l',                  {reset mode}
     'h' : begin           {set mode}
             if not InQMarkCmd then
               case PrmArr[1] of
                 4 : begin   { Insert mode on/off }
                       ActWin.Insert := (ch = 'h');
                     end;
                 20 : begin  { Newline mode (CR => CR or CR => CR/LF) }
                        CRtoCRLF := (ch = 'h');
                      end;
               end { case }
             else { Esc[? Ps; Ps h style }
               case PrmArr[1] of
                 2 : VT52On := (ch = 'l');  { VT52 or VT102 mode }
               { 3 : if ch = 'h' switch to 132 columns, else 80 columns }
                 5 : Reverse := (ch = 'h'); { reverse video }
                 6 : begin { relative / absolute cursor positioning }
                       RelativePos := (ch = 'h');
                       With ActWin do
                        if RelativePos then Depth := succ(y2-y1)
                         else Depth := ScrnLines;
                     end;
                 7 : begin { EOL wrapping mode }
                       if ch = 'h' then ActWin.Wrap := None { wrap at EOL }
                        else ActWin.Wrap := NoWrap;
                     end;
                 25 : begin
                        if ch = 'h' then Cursorh(NormCursor) { cursor mode }
                         else Cursorh(HiddenCursor);
                      end;
               end; { case }
           end;
{$ENDIF}
  {  'p' : keyboard reassignment can be dangerous }
   end { case ch } ;
end; { ANSI_Command }

{-- ANSI Init section ------------------------------------------------}

Procedure ANSI_Reset;
begin
  FillChar(PrmArr,SizeOf(PrmArr),0);
  Saved[1] := 1;  Saved[2] := 1;
  Saved[3] := 7;
  CurPrm := 1;
  Reverse := False;
{$IFDEF VT52}
  EightBitControl := False;
{$ENDIF}
{$IFDEF ANSI_MUSIC}
  Tempo:=120;
  Octave:=4;
  Duration:=4;
  In_Ciffer:=false;
  MusicMode := 1;
{$ENDIF}
end;

(* END. of include file *)
