/*
 * findqth.c
 *
 * This program takes a city and state from the command line, searches
 * the sam database for matches, and displays matching records in a
 * 1-line per record format.
 *
 * 
 * The main purpose is to demostrate the API for SAM, and use this
 * as a starting point for fancier extracts.
 * findqth depends on the API to the sam db provided by SAMAPI.EXE.
 *
 * to compile with Turbo-C
 *  tcc findqth.c samlib.c
 *
 * Other C compilers can be used, but a "pack structs" option may be
 * required.
 *
 * Copyright 1991 by Tom Thompson, Athens, AL and by RT Systems
 *                   [n4yos]
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "samapi.h"			/* samapi interface spec */

	
/*
 * functions in samlib.c
 */

int LocateSam(void);
int CallSam(int cmd, void far *cmdbuf, void far *rspbuf);
int SetCountrySam(int country);

/*
 * decls for functions contained here
 */

void display_info(datarec_t *d);
long getnumrecs(void);
long getdictcode(char *str, int dno);


int canada = 0;		/* set to 1 if /v */

/***********************
 * main
 ***********************
 */

main(int argc, char *argv[])
{
	int err, nocity, x;
	long citycode, statecode, numrecs, numfound, rn;
	cmdgetrecs_t sam_in;
	rspdatarec_t sam_out;

	if (argc < 3)
	{
		printf("usage: FINDQTH [/v] city state\n");
		printf("  use * for city to match any city\n");
		printf("  /v to select Canada database\n");
		printf("  If city name contains spaces, enclose the name in quotes.  Example\n");
		printf("      FINDQTH \"NEW YORK\" NY\n");
		exit(1);
	}
	if (LocateSam())
	{
		printf("*** SAMAPI not loaded\n");
		exit(1);
	}

	x = 1;
	if (!stricmp(argv[1], "/v"))
	{
		if (SetCountrySam(1))		/* 0 is USA, 1 is Canada */
		{
			printf("Canadian Database not installed\n");
			exit(1);
		}
		canada = 1;
		x++;
	}
	if (!strcmp(argv[x], "*"))
		nocity = 1;
	else
	{
		nocity = 0;
		citycode = getdictcode(argv[x], DnoCity);
		if (citycode == -1)
		{
			printf("City %s not in database\n", argv[1]);
			exit(1);
		}
	}

	statecode = getdictcode(argv[x+1], DnoState);
	if (statecode == -1)
	{
		printf("State %s not in database\n", argv[x+1]);
		exit(1);
	}

	numrecs = getnumrecs();
	numfound = 0;
	printf("Searching %ld records for %s, %s, please wait\n",
								numrecs, argv[x], argv[x+1]);
	printf("Esc quits\n");
	for (rn = 0; rn < numrecs; rn ++)
	{
		if (!((int) rn & 0xff))
		{
			if (kbhit() && 27 == getch())
				exit(1);
		}
		sam_in.packflags = 1+2+4+8+16+32+64;	/* don't unpack */
		sam_in.index = rn;				/* record to get */
		

		err = CallSam(SamGetRecordByCall, &sam_in, &sam_out);
		if (err != 0)
		{
			printf("*** Unexpected SAMAPI error %d\n", err);
			exit(1);
		}
		if ( (nocity || citycode == sam_out.d.CityCode)
					&& statecode == sam_out.d.StateCode)
		{
			/* found a match, get again unpacked and display */
			sam_in.packflags = 0;		/* do unpack */
			err = CallSam(SamGetRecordByCall, &sam_in, &sam_out);
			if (err != 0)
			{
				printf("*** Unexpected SAMAPI error %d\n", err);
				exit(1);
			}
			display_info(&sam_out.d);
			numfound++;
		}
	}
	printf("%ld matches in %ld records\n", numfound, numrecs);
	return 0;
}

/******************
 * display_info
 ******************
 *
 * displays some of info in callsign data record in a 1 line format.
 * Since we searched by city/state these are not displayed.
 * The display format is:
 * call class last   first  m  address_truncated  zip
 * For canada, the format is:
 * call last   first  m  address_truncated  postal code
 *
 * returns: nothing
 */

void display_info(datarec_t *d)
{
	/* note, only first 29 characters of 35 character address displayed */
	if (!canada)
		printf("%s %s %-20.20s %-11.11s %s %-29.29s %s\n",
					d->Call, d->Class,
					d->LastName, d->FirstName, d->MidInitial,
					d->Address, d->Zip);
	else
		printf("%s %-20.20s %-11.11s %s %-29.29s %s\n",
					d->Call, 
					d->LastName, d->FirstName, d->MidInitial,
					d->Address, d->Zip);
}

/******************
 * getnumrecs
 ******************
 *
 * calls samapi to get number of records in database
 *
 * returns: number of records in sam db
 */

long getnumrecs(void)
{
	chdr_t in;			/* api command block */
	rspnumrecs_t out;	/* api response buffer */

	CallSam(SamGetNumRecs, &in, &out);
	return out.numrecs;
}


/******************
 * getdictcode
 ******************
 *
 * This calls sam api to get dictionary code for strings
 * The reason for this is because it is sooo slow to have each
 * data record fully unpacked.  We can retrieve data records
 * with codes (instead of strings) for names, address, city, etc.
 * and compare these codes to what we learned here.
 *
 * input: string and dictionary (field) number (see samapi.c)
 *
 * returns: dictionary code or -1 if string not in dictionary
 */

long getdictcode(char *str, int dno)
{
	cmdgetdictcode_t in;	/* api command buffer */
	rspgetdictcode_t out;	/* api response buffer */
	int err;

	/*
	 * build the command block and call sam api
	 */

	in.dno = dno;
	in.string[35] = 0;
	strncpy(in.string, str, 35);

	err = CallSam(SamGetDictCode, &in, &out);

	/*
	 * check for unusual error
	 */

	if (err && err != SerrNotFound)
	{
		printf("Unexpected error %d in getdictcode\n", err);
		exit(1);
	}

	/*
	 * return code or -1 if not found
	 */

	return err ? -1 : out.dictcode;
}
