/*
**
** LMUCID.CMD
**
**   IBM LAN NetView Management Utilities
**   5622-153 (C) Copyright IBM Corp. 1991, 1993. All rights reserved.
**
**   US Government Users Restricted Rights -
**   Use, duplication or disclosure restricted by
**   GSA ADP Schedule Contract with IBM Corp.
**   Licensed Materials - Property of IBM
**
**
** Command syntax:
**
**     LMUCID 
**         drive         path 
**
**     ? Ĵ
**       Ŀ  
**                                        
**      /S:SOURCE  
**            /R:SRESPONSE Ĵ
**            /L1:LOG Ĵ
**            /T:TARGET 
**
**  Where:
**    ?         Help information
**    SOURCE    Drive and path to the LMU2 install files.  Default is
**              path to the LMUCID program.
**    SRESPONSE Drive, path and name of Specific Response File
**    LOG       Drive, path and name of Log File
**    TARGET    Drive to install to.  Path must be \LMU2.  Default drive
**              is the boot drive.
**
** Response file format:
**
**   ; LMUCID keywords are LMUINST and LMUCUST, or LMUFIX, the LMU2
**   ; install, customization and update program names.  The keywords are
**   ; followed by command line parameters (thus, the line looks just like
**   ; a command line):
**   ;
**   LMUINST IBM MANAGING GUI
**   ;
**   LMUCUST MANAGED MANAGING FAULT_MANAGER ADMINISTRATOR
**   ;
**   ;
**   ; If no response file is specified, the default for LMUINST is '',
**   ; do not install code locally, and for LMUCUST is MANAGED, customize
**   ; for Managed System only.
**   ;
**   ; LMUFIX has no command line parameters, and overrides any entries
**   ; or defaults for LMUINST or LMUCUST.  LMUFIX must be either the
**   ; only keyword in the response file, or the first entry if LMUINST
**   ; or LMUCUST are desired (the order of execution will still be
**   ; LMUINST, LMUCUST, LMUFIX).
**   ;
*/

pgm = 'LMUCID V1.06'

'@ECHO OFF'
signal on syntax
trace 'Off'

ARG p.1 p.2 p.3 p.4 p.5 p.6            /* get command line parms       */
PARSE ARG parms                        /* get parm string              */

PARSE UPPER source . . s1              /* get fully qualified source   */
source = substr(s1,1,lastpos('\',s1))  /* extract source path          */

if (RxFuncquery('lmu2rx') <> 0) then   /* make sure lmu2rx is avail    */
  rcx = rxfuncadd('lmu2rx',source||'lmu2rx.dll','LMU2REXX')

call copyright                         /* print copyright              */

call lmu2rx                            /* get boot drive               */
call rxfuncdrop 'lmu2rx'

/***********************************************************************/
/* Initialize variables                                                */
/***********************************************************************/

sresponse = ''                         /* no specific response file    */
log = 'con:'                           /* log to the console           */
target = LMU_BOOT_DRIVE                /* target is boot drive         */

lmuinst = ''                           /* default is do not install    */
lmucust = 'MANAGED'                    /* and managed only             */
lmufix  = ''                           /* and not update               */

call validate                          /* validate parameters          */
custsrc = source                       /* set initial LMUCUST path     */

call loginit                           /* initialize log               */

if (sresponse <> '') then              /* if a Specific Resp File      */
  call readrsp sresponse

if (lmuinst<>'') then do               /* if LMUINST to be run         */

  call RxFuncAdd 'SysDriveInfo', 'RexxUtil', 'SysDriveInfo' /*      @05*/
  dinfo = SysDriveInfo(target)         /*                           @05*/
  parse var dinfo . freespace .        /* get free space            @05*/
  if (freespace < 8000000) then do     /* if not enough             @05*/
    call lineout log, 'LMUCID: 8MB req,' freespace 'available' /*   @05*/
    rc = '12 08'x                      /* insufficient disk space   @05*/
    signal error                       /*                           @05*/
  end                                  /*                           @05*/

  'call' source||'LMUINST' lmuinst '/T'||target '1>>'||log '2>&1'
  if (rc<>0) then do                   /* if it failed                 */
    rc = '16 04'x                      /* unexpected condition         */
    signal error
  end
  custsrc = target||'\LMU2\'           /* execute LMUCUST locally      */
end

if (lmucust<>'') then do               /* if LMUCUST to be run         */
  custsrc||'LMUCUST' lmucust '/T'||target '1>>'||log '2>&1'
  if (rc<>0) then do                   /* if it failed                 */
    rc = '16 04'x                      /* unexpected condition         */
    signal error
  end
end

if (lmufix<>'') then do                /* if LMUFIX to be run          */
  'call' source||'LMUFIX' '/T'||target '1>>'||log '2>&1'
  if (rc<>0) then do                   /* if it failed                 */
    rc = '16 04'x                      /* unexpected condition         */
    signal error
  end
end

exit c2d('FE 00'x)                     /* exit, reboot                 */

/***********************************************************************/
/* Print copyright                                                     */
/***********************************************************************/

copyright: procedure

  say 'IBM LAN NetView Management Utilities'
  say '5622-153 (C) Copyright IBM Corp. 1991, 1993. All rights reserved.'
  say ''
  say 'US Government Users Restricted Rights -'
  say 'Use, duplication or disclosure restricted by'
  say 'GSA ADP Schedule Contract with IBM Corp.'
  say ''
  return

/***********************************************************************/
/* Validate parameters                                                 */
/***********************************************************************/

validate:

if (p.1 = '?') then signal help        /* if help requested            */

DO I = 1 to 6 while (p.I<>'')          /* for each parameter           */
  arg = substr(p.I,1,3)                /* get the first 3 chars        */
  select
    when (arg = '/S:') then            /* if SOURCE                    */
      do                               /*                           @02*/
        source = substr(p.I,4)
        if (right(source,1)<>'\') then /* doesn't end in \?         @02*/
          source = source||'\'         /* add a \                   @02*/
      end                              /*                           @02*/
    when (arg = '/R:') then            /* if SRESPONSE                 */
      sresponse = substr(p.I,4)
    when (substr(p.I,1,4)= '/L1:') then/* if LOG                       */
      log = substr(p.I,5)
    when (arg = '/T:') then            /* if TARGET                    */
      target = substr(p.I,4,1)||':'    /* make target drive         @01*/
    otherwise do                       /* invalid parameter            */
      say 'LMUCID: "'||p.I||'" is an unrecognized parameter.'
      rc = '16 00'x                    /* invalid program invocation   */
      signal error
    end
  end
END I

return

/***********************************************************************/
/* Initialize log file                                                 */
/***********************************************************************/

loginit:

call lineout log, ''
call lineout log, copies('*',72)
call lineout log, 'LMU C/I/D Install/Update' date() time()
call lineout log, 'Parameters: ' parms
call lineout log, copies('*',72)
call lineout log, ''
call lineout log                       /* close log file               */
return

/***********************************************************************/
/* Read response file                                                  */
/***********************************************************************/

readrsp:

ARG filename

lmucust = ''                           /* reset lmucust default     @05*/

if (stream(filename,'C','QUERY EXISTS')='') then /* if no file         */
  DO
    call lineout log, 'LMUCID:' filename 'does not exist'
    rc = '08 00'x                      /* data resource not found      */
    signal error
  END

resp = stream(filename,'C','OPEN READ') /* open read only           @05*/
if (left(resp,5) <> 'READY') then      /* if open failed            @05*/
  DO                                   /*                           @05*/
    call lineout log, 'LMUCID:' filename 'open failed,' resp /*     @05*/
    rc = '12 00'x                      /* I/O error                 @05*/
    signal error                       /*                           @05*/
  END                                  /*                           @05*/

SIGNAL ON NOTREADY                     /*                           @05*/

DO while(lines(filename))              /* do while lines to read       */
  line = linein(filename)              /* get line                     */
  line = strip(line,'L')               /* strip leading spaces         */
  line = strip(line,'L',X2C('9'))      /* strip leading tabs           */
  if ((line='') | (substr(line,1,1)='*') | (substr(line,1,1)=':')) then
    iterate
  parse upper var line command parms   /* cvt to upper case & get cmd  */
  select                               /* switch on command         @03*/
    when (command = 'LMUINST') then
      lmuinst = parms
    when (command = 'LMUCUST') then
      lmucust = parms
    when (command = 'LMUFIX') then
      DO
        lmufix = 'TRUE'
        lmuinst = ''
        lmucust = ''
      END
    otherwise                          /* keyword is invalid           */
      DO
        call lineout log, 'LMUCID: unrecognized response keyword: '||command
        rc = '16 00'x
        signal error
      END
  end                                  /* end select                   */
END                                    /* end lines to read            */

SIGNAL OFF NOTREADY                    /*                           @05*/

call stream filename,'C','CLOSE'       /*                           @05*/

if ((lmuinst = '') & (lmucust = '') & (lmufix = '')) then
  DO                                   /*                           @05*/
    call lineout log, 'LMUCID:' filename 'has no keywords' /*       @05*/
    rc = '16 04'x                      /* unexpected condition      @05*/
    signal error                       /*                           @05*/
  END                                  /*                           @05*/

return

/***********************************************************************/
/* NOTREADY condition                                               @05*/
/***********************************************************************/

notready:                              /*                           @05*/
  call lineout log, 'LMUCID: I/O failure on' condition('D') /*      @05*/
  rc = '12 00'x                        /* I/O error                 @05*/
  signal error                         /*                           @05*/

/***********************************************************************/
/* Error exit                                                          */
/***********************************************************************/

error:

  call lineout log, 'LMUCID: rc =' c2x(rc)
  exit c2d(rc)

/***********************************************************************/
/* Program error -- log and get out                                    */
/***********************************************************************/

syntax:

  trace 'Off'
  if (rc = 43) then do                 /* routine not found            */
    errline = sourceline(sigl)         /* get error line               */
    PARSE UPPER VAR errline errline    /* convert to upper case        */
    errsub = 'LMU2RX'
    if (pos(errsub, errline)=0) then do /* if not LMU2RX               */
      errsub = 'AUEREXGA'
      if (pos(errsub, errline)=0) then /* if not AUEREXGA              */
        errsub =''                     /* should not occur             */
      end

    say 'LMUCID: Unable to access LMU' errsub||'.DLL'
    end

  else do                              /* real syntax error            */

    say 'LMUCID: Fatal program error at line' sigl||', REXX RC =' rc
    end

  exit c2d('16 04'x)

/***********************************************************************/
/* Help text                                                           */
/***********************************************************************/

help:

say '                                                                      '
say '     LMUCID         '
say '         drive         path                               '
say '                                                                      '
say '     ? Ĵ        '
say '       Ŀ                            '
say '                                                                  '
say '      /S:SOURCE                            '
say '            /R:SRESPONSE Ĵ                               '
say '            /L1:LOG Ĵ                               '
say '            /T:TARGET                                '
say '                                                                      '
say '  Where:                                                              '
say '    ?         Help information                                        '
say '    SOURCE    Drive and path to the LMU install files.  Default is  '
say '              path to the LMUCID program.                             '
say '    SRESPONSE Drive, path and name of Specific Response File          '
say '    LOG       Drive, path and name of Log File                        '
say '    TARGET    Drive to install to.  Path must be \LMU2.  Default drive'
say '              is the boot drive.                                      '

exit c2d('00 00'x)
