/*****************************************************************************\
*                                                                             *
* shell.h -  SHELL.DLL functions, types, and definitions		      *
*                                                                             *
* Copyright (c) 1993-1994, Microsoft Corp.	All rights reserved	      *
*                                                                             *
\*****************************************************************************/
//

#ifndef _INC_SHELL
#define _INC_SHELL


//
// All pure 32-bit only control classes (i.e., not-thunked) will have
// "32" at the end of their class names.
//
#ifndef WC_SUFFIX
#ifdef WIN32
#define WC_SUFFIX "32"
#else
#define WC_SUFFIX
#endif
#endif


// For historical reasons, these definitions are kept in a separate file.

#ifndef _INC_SHELLAPI
#include <shellapi.h>
#endif

#ifndef RC_INVOKED
#pragma pack(1)         /* Assume byte packing throughout */
#endif /* !RC_INVOKED */

#ifdef __cplusplus
extern "C" {            /* Assume C declarations for C++ */
#endif  /* __cplusplus */

//====== Ranges for control message IDs
// (making each control's messages unique makes validation and
// debugging easier).
//
#define LVM_FIRST       0x1000      // ListView messages
#define TV_FIRST        0x1100      // TreeView messages
#define HDM_FIRST       0x1200      // Header messages

//====== Ranges for WM_NOTIFY codes ==================================

// Note that these are defined to be unsigned to avoid compiler warnings
// since NMHDR.code is declared as UINT.

#define NM_FIRST        (0U-  0U)
#define NM_LAST         (0U- 99U)

#define LVN_FIRST       (0U-100U)
#define LVN_LAST        (0U-199U)

// PSN_FIRST - PSN_LAST defined in prsht.h (0U-200U) - (0U-299U)

#define HDN_FIRST       (0U-300U)
#define HDN_LAST        (0U-399U)

#define TVN_FIRST       (0U-400U)
#define TVN_LAST        (0U-499U)

#define RDN_FIRST       (0U-500U)
#define RDN_LAST        (0U-519U)

#define TTN_FIRST	(0U-520U)
#define TTN_LAST	(0U-549U)

#define EXN_FIRST       (0U-1000U)  // shell explorer/browser
#define EXN_LAST        (0U-1199U)

#define MAXPATHLEN      260

//====== IMAGE APIS ==================================================

#define CLR_NONE    0xFFFFFFFFL
#define CLR_DEFAULT 0xFF000000L

struct _IMAGELIST;
typedef struct _IMAGELIST NEAR* HIMAGELIST;

    // fMask is TRUE to create a masked (two-bitmap) image list.
    // cInitial is initial size of bitmap, cGrow is size to increase by.
HIMAGELIST WINAPI ImageList_Create(int cx, int cy, BOOL fMask, int cInitial, int cGrow);
BOOL WINAPI ImageList_Destroy(HIMAGELIST himl);

    // loads one resource, figures out cy and cInitial from bmp,
    // and makes mask from a specified color (unless CLR_NONE given)
HIMAGELIST WINAPI ImageList_LoadBitmap(HINSTANCE hi, LPCSTR lpbmp, int cx, int cGrow, COLORREF crMask);

    // Number of images in list.
int WINAPI ImageList_GetImageCount(HIMAGELIST himl);
    // x and y dimensions of an icon
BOOL WINAPI ImageList_GetIconSize(HIMAGELIST himl, int FAR *cx, int FAR *cy);

    // Add, Remove, replace images.  Add returns index of Added item, or -1 if error.
int  WINAPI ImageList_Add(HIMAGELIST himl, HBITMAP hbmImage, HBITMAP hbmMask);
int  WINAPI ImageList_AddMasked(HIMAGELIST himl, HBITMAP hbmImage, COLORREF crMask);
BOOL WINAPI ImageList_Remove(HIMAGELIST himl, int i);
BOOL WINAPI ImageList_Replace(HIMAGELIST himl, int i, HBITMAP hbmImage, HBITMAP hbmMask);

    // Add or replace given an icon handle.  hicon is not saved; can be destroyed
    // after call.  Add returns index of added item, else -1.
int   WINAPI ImageList_AddIcon(HIMAGELIST himl, HICON hicon);
BOOL  WINAPI ImageList_ReplaceIcon(HIMAGELIST himl, int i, HICON hicon);
HICON WINAPI ImageList_ExtractIcon(HINSTANCE hAppInst, HIMAGELIST himl, int i);
int   WINAPI ImageList_AddFromImageList(HIMAGELIST himlDest, HIMAGELIST himlSrc, int iSrc);

#define ILD_NORMAL      0x0000          // use current bkcolor
#define ILD_TRANSPARENT 0x0001          // override bkcolor setting
#define ILD_SELECTED    0x0002		// draw as selected
#define ILD_FOCUS       0x0004		// draw as focused (selection)
#define ILD_ASICON      (ILD_TRANSPARENT)
    // Draw item i into hdcDst at x, y.  Flags define the drawing style.
BOOL WINAPI ImageList_Draw(HIMAGELIST himl, int i, HDC hdcDst, int x, int y, UINT fStyle);

    // Get bounding rectangle of image in image bitmaps.
BOOL WINAPI ImageList_GetImageRect(HIMAGELIST himl, int i, RECT FAR* prcImage);

    // Set & get background color for speedy drawing.  Set to CLR_NONE for true
    // cursor-style AND/XOR transparency.
COLORREF WINAPI ImageList_SetBkColor(HIMAGELIST himl, COLORREF clrBk);
COLORREF WINAPI ImageList_GetBkColor(HIMAGELIST himl);

typedef struct _IMAGEINFO
{
    HBITMAP hbmImage;
    HBITMAP hbmMask;
    int cPlanes;
    int cBitsPerPixel;
    RECT rcImage;
} IMAGEINFO;

    // Get information about an image for direct manipulation.  rcImage is
    // part of hbmImage and hbmMask that make up the image.  hbmMask is
    // NULL if not a masked imagelist.
BOOL WINAPI ImageList_GetImageInfo(HIMAGELIST himl, int i, IMAGEINFO FAR* pImageInfo);

    // Start tracking an image i, starting at x, y, offset by dx/dyHotspot.
BOOL WINAPI ImageList_StartDrag(HIMAGELIST himl, int i, int x, int y, int dxHotspot, int dyHotspot);
    // End a dragging operation.
HIMAGELIST WINAPI ImageList_EndDrag();
    // Called on mouse move to move dragging image.
BOOL WINAPI ImageList_DragMove(int x, int y);

    // Hide or show drag image
BOOL WINAPI ImageList_DragShow(BOOL fShow);

    // Create new IMAGELIST that is the merge of two existing images.
HIMAGELIST WINAPI ImageList_Merge(HIMAGELIST himl1, int i1, HIMAGELIST himl2, int i2, int dx, int dy);

    // make a dithered copy of a source image into a destionation
void FAR PASCAL ImageList_CopyDitherImage(HIMAGELIST himlDst, WORD iDst,
			int xDst, int yDst, HIMAGELIST himlSrc, int iSrc);


//================ HEADER APIS =============================================
//
// Class name: SysHeader (WC_HEADER)
//
// The SysHeader control provides for column and row headers much like those
// found in MSMail and Excel.  Header items appear as text on a gray
// background. Items can behave as pushbuttons, in which case they have a
// raised face.
//
// SysHeaders support changing width or height of items using the mouse.
// These controls do not support a keyboard interface, so they do not accept
// the input focus.
//
// There are notifications that allow applications to determine when an item
// has been clicked or double clicked, width change has occured, drag tracking
// is occuring, etc.
//
//////////////////////////////////////////////////////////////////////////////

#define WC_HEADER       "SysHeader" WC_SUFFIX

// Header control styles
#define HDS_HORZ            0x00000000  // Horizontal header
#define HDS_VERT            0x00000001  // Vertical header (Not currently supported)
#define HDS_BUTTONS         0x00000002  // Items act as buttons
#define HDS_DIVIDERTRACK    0x00000004  // Allow width tracking

// The HD_ITEM structure describes a header item.
// The first field contains a combination of HDI_* flags
// indicating which fields contain valid data.
//
typedef struct _HD_ITEM
{
    UINT    mask;
    int     cxy;            // width (HDS_HORZ) or height (HDS_VERT)
    LPSTR   pszText;
    HBITMAP hbm;            // Bitmap to use (implies HDF_BITMAP)
    int     cchTextMax;     // Valid only for GetItem: contains size of buffer
    int     fmt;            // HDF_* value
    LPARAM  lParam;
} HD_ITEM;

// HD_ITEM mask flags
#define HDI_WIDTH       0x0001
#define HDI_HEIGHT      HDI_WIDTH
#define HDI_TEXT        0x0002
#define HDI_FORMAT      0x0004
#define HDI_LPARAM      0x0008
#define HDI_BITMAP      0x0010

// HD_ITEM fmt field values
// First justification values
#define HDF_LEFT        0
#define HDF_RIGHT       1
#define HDF_CENTER      2
#define HDF_JUSTIFYMASK 0x0003

// Now other formating options
#define HDF_OWNERDRAW   0x8000
#define HDF_STRING      0x4000
#define HDF_BITMAP      0x2000

// Define Ownerdraw type for Header controls BUGBUG: should be in windows.h?
#define ODT_HEADER      100

// Returns number of items in header.
//
// int Header_GetItemCount(HWND hwndHD);
//
#define HDM_GETITEMCOUNT    (HDM_FIRST + 0)
#define Header_GetItemCount(hwndHD) \
    (int)SendMessage((hwndHD), HDM_GETITEMCOUNT, 0, 0L)

// Insert header item at specified index.  Item is inserted at end if
// i is greater than or equal to the number of items in the header.
// Returns the index of the inserted item.
//
// int Header_InsertItem(HWND hwndHD, int i, const HD_ITEM FAR* pitem);
//
#define HDM_INSERTITEM      (HDM_FIRST + 1)
#define Header_InsertItem(hwndHD, i, phdi) \
    (int)SendMessage((hwndHD), HDM_INSERTITEM, (WPARAM)(int)(i), (LPARAM)(const HD_ITEM FAR*)(phdi))

// Delete a header item at the specified index.
//
// BOOL Header_DeleteItem(HWND hwndHD, int i);
//
#define HDM_DELETEITEM      (HDM_FIRST + 2)
#define Header_DeleteItem(hwndHD, i) \
    (BOOL)SendMessage((hwndHD), HDM_DELETEITEM, (WPARAM)(int)(i), 0L)

// Get header item at index i.  The mask field of the pointed-to
// HD_ITEM structure indicates which fields will be set by this
// function; other fields are not changed.
//
// The cchTextMax field of *pitem contains the maximum
// length of the returned string.
//
// BOOL Header_GetItem(HWND hwndHD, int i, HD_ITEM FAR* phdi);
//
#define HDM_GETITEM         (HDM_FIRST + 3)
#define Header_GetItem(hwndHD, i, phdi) \
    (BOOL)SendMessage((hwndHD), HDM_GETITEM, (WPARAM)(int)(i), (LPARAM)(HD_ITEM FAR*)(phdi))

// Set header item at index i.  The mask field of the pointed-to
// HD_ITEM structure indicates which header item attributes will
// be changed by this call; other fields of *pitem that do not
// correspond to pitem->mask are ignored.
//
// The cchTextMax of *pitem is ignored.
//
// BOOL Header_SetItem(HWND hwndHD, int i, const HD_ITEM FAR* phdi);
//
#define HDM_SETITEM         (HDM_FIRST + 4)
#define Header_SetItem(hwndHD, i, phdi) \
    (BOOL)SendMessage((hwndHD), HDM_SETITEM, (WPARAM)(int)(i), (LPARAM)(const HD_ITEM FAR*)(phdi))

// Calculate size and position of header within a rectangle.
// Results are returned in a WINDOWPOS structure you supply,
// and the layout rectangle is adjusted to exclude the leftover area.
//
typedef struct _HD_LAYOUT
{
    RECT FAR* prc;
    WINDOWPOS FAR* pwpos;
} HD_LAYOUT;

// BOOL Header_Layout(HWND hwndHD, HD_LAYOUT FAR* playout);
//
#define HDM_LAYOUT          (HDM_FIRST + 5)
#define Header_Layout(hwndHD, playout) \
    (BOOL)SendMessage((hwndHD), HDM_LAYOUT, 0, (LPARAM)(HD_LAYOUT FAR*)(playout))

// Header Notifications
//
// All header notifications are via the WM_NOTIFY message.
// lParam of WM_NOTIFY points to a HD_NOTIFY structure for
// all of the following notifications.

// *pitem contains item being changed.  pitem->mask indicates
// which fields are valid (others have indeterminate state)
//
#define HDN_ITEMCHANGING    (HDN_FIRST-0)
#define HDN_ITEMCHANGED     (HDN_FIRST-1)

// Item has been clicked or doubleclicked (HDS_BUTTONS only)
// iButton contains button id: 0=left, 1=right, 2=middle.
//
#define HDN_ITEMCLICK       (HDN_FIRST-2)
#define HDN_ITEMDBLCLICK    (HDN_FIRST-3)

// Divider area has been clicked or doubleclicked (HDS_DIVIDERTRACK only)
// iButton contains button id: 0=left, 1=right, 2=middle.
//
#define HDN_DIVIDERCLICK    (HDN_FIRST-4)
#define HDN_DIVIDERDBLCLICK (HDN_FIRST-5)

// Begin/end divider tracking (HDS_DIVIDERTRACK only)
// Return TRUE from HDN_BEGINTRACK notification to prevent tracking.
//
#define HDN_BEGINTRACK      (HDN_FIRST-6)
#define HDN_ENDTRACK        (HDN_FIRST-7)

// HDN_DRAG: cxy field contains new height/width, which may be < 0.
// Changing this value will affect the tracked height/width (allowing
// for gridding, pinning, etc).
//
// Return TRUE to cancel tracking.
//
#define HDN_TRACK           (HDN_FIRST-8)

typedef struct _HD_NOTIFY
{
    NMHDR   hdr;
    int     iItem;
    int     iButton;        // *CLICK notifications: 0=left, 1=right, 2=middle
    HD_ITEM FAR* pitem;     // May be NULL
} HD_NOTIFY;

//================ LISTVIEW APIS ===========================================
//
// Class name: SysListView (WC_LISTVIEW)
//
// The SysListView control provides for a group of items which are displayed
// as a name and/or an associated icon and associated sub-items, in one of
// several organizations, depending on current style settings:
//  * The Icon Format (LVS_ICON)
//      The control arranges standard-sized icons on an invisible grid
//      with their text caption below the icon. The user can drag icons to
//      rearrange them freely, even overlapping each other.
//  * The Small Icon Format (LVS_SMALLICON)
//      The control arranges half-sized icons on an invisible columnar grid
//      like a multi-column owner-draw listbox, with the caption of each
//      item to the icon's right.  The user can still rearrange items
//      freely to taste.  Converting from LVS_ICON to LVS_SMALLICON and back
//      will attempt to preserve approximate relative positions of
//      repositioned items.
//  * The List Format (LVS_LIST)
//      The control enforces a multi-column list of small-icon items with
//      each item's caption to the right.  No free rearranging is possible.
//  * The Report Format (LVS_REPORT)
//      The control enforces a single-column list of small-icon items with
//      each item's caption to the right, and further columns used for item-
//      specific sub-item text.  The columns are capped with a SysHeader
//      bar (unless specified) which allows the user to change the relative
//      widths of each sub-item column.
//
// The icons and small-icons presented may be assigned as indices into
// an ImageList of the appropriate size.  These ImageLists (either custom
// lists or copies of the system lists) are assigned to the control by the
// owner at initialization time or at any later time.
//
// Text and icon values may be "late-bound," or assigned by a callback
// routine as required by the control.  For example, if it would be slow to
// compute the correct icon or caption for an item, the item can be assigned
// special values which indicate that they should be computed only as the
// items become visible (say, for a long list of items being scrolled into
// view).
//
// Each item has a state, which can be (nearly) any combination of the
// following attributes, mostly managed automatically by the control:
//  * Selected (LVIS_SELECTED)
//      The item appears selected.  The appearance of selected items
//      depends on whether the control has the focus, and the selection
//      system colors.
//  * Focused (LVIS_FOCUSED)
//      One item at a time may be focused.  The item is surrounded with a
//      standard focus-rectangle.
//  * Marked (LVIS_MARKED)
//      REVIEW: Call this "Checked"?
//  * Disabled (LVIS_DISABLED)
//      The item is drawn with the standard disabled style and coloring.
//  * Hidden (LVIS_HIDDEN)
//      The item is not visible nor does it respond to user interaction.
//  * Drop-Highlighted (LVIS_DROPHILITED)
//      The item appears marked when the user drags an object over it, if
//      it can accept the object as a drop-target.
//  * Link/Alias/Shortcut (LVIS_LINK)
//      The item's text has a standard link-indicator appended (Foo >>).
//
// There are notifications that allow applications to determine when an item
// has been clicked or double clicked, caption text changes have occured,
// drag tracking is occuring, widths of columns have changed, etc.
//
//////////////////////////////////////////////////////////////////////////////

#define WC_LISTVIEW         "SysListView" WC_SUFFIX

BOOL WINAPI SetWindowStyle(HWND hwnd, DWORD style, DWORD mask, BOOL fExtended);

// ListView styles

// view type styles
#define LVS_ICON            0x00000000L
#define LVS_REPORT          0x00000001L
#define LVS_SMALLICON       0x00000002L
#define LVS_LIST            0x00000003L
#define LVS_TYPEMASK        0x00000003L

// shared styles
#define LVS_SINGLESEL       0x00000004L
#define LVS_ALWAYSSEL       0x00000008L
#define LVS_SORTASCENDING   0x00000010L
#define LVS_SORTDESCENDING  0x00000020L
#define LVS_SHAREIMAGELISTS 0x00000040L

#define LVS_NOLABELWRAP     0x00000080L
#define LVS_AUTOARRANGE     0x00000100L
#define LVS_EDITLABELS      0x00000200L
#define LVS_ALIGNTOP        0x00000000L
#define LVS_ALIGNBOTTOM     0x00000400L
#define LVS_ALIGNLEFT       0x00000800L
#define LVS_ALIGNRIGHT      0x00000c00L
#define LVS_ALIGNMASK       0x00000c00L

#define LVS_COLUMNBUTTONS   0x00001000L // with LVS_REPORT only
#define LVS_NOSCROLL        0x00002000L
#define LVS_NOCOLUMNHEADER  0x00004000L // with LVS_REPORT only

// COLORREF ListView_GetBkColor(HWND hwnd);
#define LVM_GETBKCOLOR      (LVM_FIRST + 0)
#define ListView_GetBkColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), LVM_GETBKCOLOR, 0, 0L)

// BOOL ListView_SetBkColor(HWND hwnd, COLORREF clrBk);
#define LVM_SETBKCOLOR      (LVM_FIRST + 1)
#define ListView_SetBkColor(hwnd, clrBk) \
    (BOOL)SendMessage((hwnd), LVM_SETBKCOLOR, 0, (LPARAM)(COLORREF)(clrBk))

// HIMAGELIST ListView_GetImageList(HWND hwnd, BOOL fSmallImages);
#define LVM_GETIMAGELIST    (LVM_FIRST + 2)
#define ListView_GetImageList(hwnd, fSmallImages) \
    (HIMAGELIST)SendMessage((hwnd), LVM_GETIMAGELIST, (WPARAM)(BOOL)(fSmallImages), 0L)

// BOOL ListView_SetImageList(HWND hwnd, HIMAGELIST himl, BOOL fSmallImages);
#define LVM_SETIMAGELIST    (LVM_FIRST + 3)
#define ListView_SetImageList(hwnd, himl, fSmallImages) \
    (BOOL)SendMessage((hwnd), LVM_SETIMAGELIST, (WPARAM)(BOOL)(fSmallImages), (LPARAM)(UINT)(HIMAGELIST)(himl))

// int ListView_GetItemCount(HWND hwnd);
#define LVM_GETITEMCOUNT    (LVM_FIRST + 4)
#define ListView_GetItemCount(hwnd) \
    (int)SendMessage((hwnd), LVM_GETITEMCOUNT, 0, 0L)

// ListView Item structure

#define LVIF_TEXT       0x0001  // ListView mask flags
#define LVIF_IMAGE      0x0002
#define LVIF_SMALLIMAGE 0x0004
#define LVIF_PARAM      0x0008
#define LVIF_STATE      0x0010


typedef struct _LV_ITEM
{
    UINT mask;
    int iItem;
    int iSubItem;
    UINT state;
    UINT stateMask;
    LPSTR pszText;
    int cchTextMax;
    int iImage;
    int iSmallImage;
    LPARAM lParam;
} LV_ITEM;

    // Values used to cause text/image GETDISPINFO callbacks
#define LPSTR_TEXTCALLBACK      ((LPSTR)-1L)
#define I_IMAGECALLBACK         (-1)

// State flags
#define LVIS_FOCUSED        0x0001
#define LVIS_SELECTED       0x0002
#define LVIS_MARKED         0x0004
#define LVIS_DROPHILITED    0x0008

#define LVIS_DISABLED       0x0100
#define LVIS_HIDDEN         0x0200
#define LVIS_LINK           0x0400


// BOOL ListView_GetItem(HWND hwnd, LV_ITEM FAR* pitem);
#define LVM_GETITEM         (LVM_FIRST + 5)
#define ListView_GetItem(hwnd, pitem) \
    (BOOL)SendMessage((hwnd), LVM_GETITEM, 0, (LPARAM)(LV_ITEM FAR*)(pitem))

// Sets items and subitems.
//
// BOOL ListView_SetItem(HWND hwnd, const LV_ITEM FAR* pitem);
#define LVM_SETITEM         (LVM_FIRST + 6)
#define ListView_SetItem(hwnd, pitem) \
    (BOOL)SendMessage((hwnd), LVM_SETITEM, 0, (LPARAM)(const LV_ITEM FAR*)(pitem))

// int ListView_InsertItem(HWND hwnd, const LV_ITEM FAR* pitem);
#define LVM_INSERTITEM         (LVM_FIRST + 7)
#define ListView_InsertItem(hwnd, pitem)   \
    (int)SendMessage((hwnd), LVM_INSERTITEM, 0, (LPARAM)(const LV_ITEM FAR*)(pitem))

// Deletes the specified item along with all its subitems.
//
// BOOL ListView_DeleteItem(HWND hwnd, int i);
#define LVM_DELETEITEM      (LVM_FIRST + 8)
#define ListView_DeleteItem(hwnd, i) \
    (BOOL)SendMessage((hwnd), LVM_DELETEITEM, (WPARAM)(int)(i), 0L)

// BOOL ListView_DeleteAllItems(HWND hwnd);
#define LVM_DELETEALLITEMS  (LVM_FIRST + 9)
#define ListView_DeleteAllItems(hwnd) \
    (BOOL)SendMessage((hwnd), LVM_DELETEALLITEMS, 0, 0L);

// UINT ListView_GetCallbackMask(HWND hwnd);
#define LVM_GETCALLBACKMASK (LVM_FIRST + 10)
#define ListView_GetCallbackMask(hwnd) \
    (BOOL)SendMessage((hwnd), LVM_GETCALLBACKMASK, 0, 0)

// BOOL ListView_SetCallbackMask(HWND hwnd, UINT mask);
#define LVM_SETCALLBACKMASK (LVM_FIRST + 11)
#define ListView_SetCallbackMask(hwnd, mask) \
    (BOOL)SendMessage((hwnd), LVM_SETCALLBACKMASK, (WPARAM)(UINT)(mask), 0)

// ListView_GetNextItem flags (can be used in combination)
#define LVNI_ALL        0x0000
#define LVNI_FOCUSED    0x0001  // return only focused item
#define LVNI_SELECTED   0x0002  // return only selected items
#define LVNI_MARKED     0x0004  // return only marked items
#define LVNI_DROPHILITED 0x0008 // return only drophilited items
#define LVNI_HIDDEN     0x0010  // return only hidden items
#define LVNI_PREVIOUS   0x0020  // Go backwards

#define LVNI_ABOVE      0x0100  // return item geometrically above
#define LVNI_BELOW      0x0200  // "" below
#define LVNI_TOLEFT     0x0400  // "" to left
#define LVNI_TORIGHT    0x0800  // "" to right (NOTE: these four are
                                //              mutually exclusive, but
                                //              can be used with other LVNI's)

// int ListView_GetNextItem(HWND hwnd, int i, UINT flags);
#define LVM_GETNEXTITEM     (LVM_FIRST + 12)
#define ListView_GetNextItem(hwnd, i, flags) \
    (int)SendMessage((hwnd), LVM_GETNEXTITEM, (WPARAM)(int)(i), MAKELPARAM((flags), 0))

// ListView_FindInfo definitions
#define LVFI_PARAM      0x0001
#define LVFI_STRING     0x0002
#define LVFI_SUBSTRING  0x0004
#define LVFI_PARTIAL    0x0008
#define LVFI_NOCASE     0x0010

typedef struct _LV_FINDINFO
{
    UINT flags;
    LPCSTR psz;
    LPARAM lParam;
} LV_FINDINFO;

// int ListView_FindItem(HWND hwnd, int iStart, const LV_FINDINFO FAR* plvfi);
#define LVM_FINDITEM        (LVM_FIRST + 13)
#define ListView_FindItem(hwnd, iStart, plvfi) \
    (int)SendMessage((hwnd), LVM_FINDITEM, (WPARAM)(int)(iStart), (LPARAM)(const LV_FINDINFO FAR*)(plvfi))

#define LVIR_BOUNDS     0
#define LVIR_ICON       1
#define LVIR_LABEL      2

    // Rectangle bounding all or part of item, based on LVIR_* code.  Rect is returned in view coords
    // BOOL ListView_GetItemRect(HWND hwndLV, int i, RECT FAR* prc, int code);
#define LVM_GETITEMRECT     (LVM_FIRST + 14)
#define ListView_GetItemRect(hwnd, i, prc, code) \
    ((prc)->left = (code), (BOOL)SendMessage((hwnd), LVM_GETITEMRECT, (WPARAM)(int)(i), (LPARAM)(RECT FAR*)(prc)))

    // Move top-left corner of item to (x, y), specified in view rect relative coords
    // (icon and small view only)

// BOOL ListView_SetItemPosition(HWND hwndLV, int i, int x, int y);
#define LVM_SETITEMPOSITION (LVM_FIRST + 15)
#define ListView_SetItemPosition(hwndLV, i, x, y) \
    (BOOL)SendMessage((hwndLV), LVM_SETITEMPOSITION, (WPARAM)(int)(i), MAKELPARAM((x), (y)))

// BOOL ListView_GetItemPosition(HWND hwndLV, int i, POINT FAR* ppt);
#define LVM_GETITEMPOSITION (LVM_FIRST + 16)
#define ListView_GetItemPosition(hwndLV, i, ppt) \
    (BOOL)SendMessage((hwndLV), LVM_GETITEMPOSITION, (WPARAM)(int)(i), (LPARAM)(POINT FAR*)(ppt))

    // Get column width of string
    // int ListView_GetStringWidth(HWND hwndLV, LPCSTR psz);
#define LVM_GETSTRINGWIDTH  (LVM_FIRST + 17)
#define ListView_GetStringWidth(hwndLV, psz) \
    (int)SendMessage((hwndLV), LVM_GETSTRINGWIDTH, 0, (LPARAM)(LPCSTR)(psz))

    // Hit test item.  Returns item at (x,y), or -1 if not on an item.
    // Combination of LVHT_ values *pflags, indicating where the cursor
    // is relative to edges of ListView window (above, below, right, left)
    // or whether (x, y) is over icon, label, or inside window but not on item.
    // int ListView_HitTest(HWND hwndLV, LV_HITTESTINFO FAR* pinfo);

    // ItemHitTest flag values
#define LVHT_NOWHERE        0x0001
#define LVHT_ONITEMICON     0x0002
#define LVHT_ONITEMLABEL    0x0004
#define LVHT_ONITEM         (LVHT_ONITEMICON | LVHT_ONITEMLABEL)

#define LVHT_ABOVE          0x0008
#define LVHT_BELOW          0x0010
#define LVHT_TORIGHT        0x0020
#define LVHT_TOLEFT         0x0040

typedef struct _LV_HITTESTINFO
{
    POINT pt;	    // in
    UINT flags;	    // out
} LV_HITTESTINFO;

    // int ListView_HitTest(HWND hwndLV, LV_HITTESTINFO FAR* pinfo);
#define LVM_HITTEST     (LVM_FIRST + 18)
#define ListView_HitTest(hwndLV, pinfo) \
    (int)SendMessage((hwndLV), LVM_HITTEST, 0, (LPARAM)(LV_HITTESTINFO FAR*)(pinfo))

    // Return view rectangle, relative to window
    // BOOL ListView_GetViewRect(HWND hwndLV, RECT FAR* prcVis);
    // Scroll an item into view if not wholly or partially visible
    // BOOL ListView_EnsureVisible(HWND hwndLV, int i, BOOL fPartialOK);
#define LVM_ENSUREVISIBLE   (LVM_FIRST + 19)
#define ListView_EnsureVisible(hwndLV, i, fPartialOK) \
    (BOOL)SendMessage((hwndLV), LVM_ENSUREVISIBLE, (WPARAM)(int)(i), MAKELPARAM((fPartialOK), 0))

    // Scroll listview -- offsets origin of view rectangle by dx, dy
    // BOOL ListView_Scroll(HWND hwndLV, int dx, int dy);
#define LVM_SCROLL          (LVM_FIRST + 20)
#define ListView_Scroll(hwndLV, dx, dy) \
    (BOOL)SendMessage((hwndLV), LVM_SCROLL, 0, MAKELPARAM((dx), (dy)))

    // Force eventual redraw of range of items (redraw doesn't occur
    // until WM_PAINT processed -- call UpdateWindow() after to redraw right away)
    // BOOL ListView_RedrawItems(HWND hwndLV, int iFirst, int iLast);
#define LVM_REDRAWITEMS     (LVM_FIRST + 21)
#define ListView_RedrawItems(hwndLV, iFirst, iLast) \
    (BOOL)SendMessage((hwndLV), LVM_REDRAWITEMS, 0, MAKELPARAM((iFirst), (iLast)))

    // Arrange style
#define LVA_DEFAULT         0x0000
#define LVA_ALIGNLEFT       0x0001
#define LVA_ALIGNTOP        0x0002
#define LVA_ALIGNRIGHT      0x0003
#define LVA_ALIGNBOTTOM     0x0004
#define LVA_SNAPTOGRID      0x0005

#define LVA_SORTASCENDING   0x0100  // can be used in combo with above LVA_* values
#define LVA_SORTDESCENDING  0x0200  // ""

    // Arrange icons according to LVA_* code
    // BOOL ListView_Arrange(HWND hwndLV, UINT code);
#define LVM_ARRANGE         (LVM_FIRST + 22)
#define ListView_Arrange(hwndLV, code) \
    (BOOL)SendMessage((hwndLV), LVM_ARRANGE, (WPARAM)(UINT)(code), 0L)

    // Begin editing the label of a control.  Implicitly selects and focuses
    // item.  Send WM_CANCELMODE to cancel.
    // HWND ListView_EditLabel(HWND hwndLV, int i);
#define LVM_EDITLABEL       (LVM_FIRST + 23)
#define ListView_EditLabel(hwndLV, i) \
    (HWND)SendMessage((hwndLV), LVM_EDITLABEL, (WPARAM)(int)(i), 0L)

    // Return edit control being used for editing.  Subclass OK, but
    // don't destroy.  Will be destroyed when editing is finished.
    //HWND ListView_GetEditControl(HWND hwndLV);
#define LVM_GETEDITCONTROL  (LVM_FIRST + 24)
#define ListView_GetEditControl(hwndLV) \
    (HWND)SendMessage((hwndLV), LVM_GETEDITCONTROL, 0, 0L)

typedef struct _LV_COLUMN
{
    UINT mask;
    int fmt;
    int cx;
    LPSTR pszText;
    int cchTextMax;
    int iSubItem;       // subitem to display
} LV_COLUMN;

// LV_COLUMN mask values
#define LVCF_FMT        0x0001
#define LVCF_WIDTH      0x0002
#define LVCF_TEXT       0x0004
#define LVCF_SUBITEM    0x0008


// Column format codes
#define LVCFMT_LEFT     0
#define LVCFMT_RIGHT    1
#define LVCFMT_CENTER   2

// Set/Query column info
// BOOL ListView_GetColumn(HWND hwndLV, int iCol, LV_COLUMN FAR* pcol);
#define LVM_GETCOLUMN       (LVM_FIRST + 25)
#define ListView_GetColumn(hwnd, iCol, pcol) \
    (BOOL)SendMessage((hwnd), LVM_GETCOLUMN, (WPARAM)(int)(iCol), (LPARAM)(LV_COLUMN FAR*)(pcol))

// BOOL ListView_SetColumn(HWND hwndLV, int iCol, LV_COLUMN FAR* pcol);
#define LVM_SETCOLUMN       (LVM_FIRST + 26)
#define ListView_SetColumn(hwnd, iCol, pcol) \
    (BOOL)SendMessage((hwnd), LVM_SETCOLUMN, (WPARAM)(int)(iCol), (LPARAM)(const LV_COLUMN FAR*)(pcol))

// insert/delete report view column
// int ListView_InsertColumn(HWND hwndLV, int iCol, const LV_COLUMN FAR* pcol);
#define LVM_INSERTCOLUMN    (LVM_FIRST + 27)
#define ListView_InsertColumn(hwnd, iCol, pcol) \
    (int)SendMessage((hwnd), LVM_INSERTCOLUMN, (WPARAM)(int)(iCol), (LPARAM)(const LV_COLUMN FAR*)(pcol))

// BOOL ListView_DeleteColumn(HWND hwndLV, int iCol);
#define LVM_DELETECOLUMN    (LVM_FIRST + 28)
#define ListView_DeleteColumn(hwnd, iCol) \
    (BOOL)SendMessage((hwnd), LVM_DELETECOLUMN, (WPARAM)(int)(iCol), 0);

#define LVM_GETCOLUMNWIDTH  (LVM_FIRST + 29)
#define ListView_GetColumnWidth(hwnd, iCol) \
    (int)SendMessage((hwnd), LVM_GETCOLUMNWIDTH, (WPARAM)(int)(iCol), 0);

#define LVM_SETCOLUMNWIDTH  (LVM_FIRST + 30)
#define ListView_SetColumnWidth(hwnd, iCol, cx) \
    (BOOL)SendMessage((hwnd), LVM_SETCOLUMNWIDTH, (WPARAM)(int)(iCol), MAKELPARAM((cx), 0))


// Drag the selected items in a ListView.  Other than hwndLV (the ListView
// window), the parameters are the same as those for ShellDragObjects.
//
typedef struct _LV_DRAGITEMINFO
{
    HWND    hwndRoot;
    HWND    hwndFrom;
    UINT    fmt;
    LPARAM  lParam;
    POINT   ptOffset;
} LV_DRAGITEMINFO;

// LRESULT ListView_DragItem(HWND hwndLV, int i, const LV_DRAGITEMINFO FAR* pinfo);
#define LVM_DRAGITEM        (LVM_FIRST + 33)
#define ListView_DragItem(hwnd, i, pinfo) \
    SendMessage((hwnd), LVM_DRAGITEM, (WPARAM)(int)(i), (LPARAM)(const LV_DRAGITEMINFO FAR*)(pinfo))

// BOOL ListView_GetViewRect(HWND hwndLV, RECT FAR* prc);
#define LVM_GETVIEWRECT     (LVM_FIRST + 34)
#define ListView_GetViewRect(hwnd, prc) \
    (BOOL)SendMessage((hwnd), LVM_GETVIEWRECT, 0, (LPARAM)(RECT FAR*)(prc))


// get/set text and textbk color for text drawing.  these override
// the standard window/windowtext settings.  they do NOT override
// when drawing selected text.
// COLORREF ListView_GetTextColor(HWND hwnd);
#define LVM_GETTEXTCOLOR      (LVM_FIRST + 35)
#define ListView_GetTextColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), LVM_GETTEXTCOLOR, 0, 0L)

// BOOL ListView_SetTextColor(HWND hwnd, COLORREF clrText);
#define LVM_SETTEXTCOLOR      (LVM_FIRST + 36)
#define ListView_SetTextColor(hwnd, clrText) \
    (BOOL)SendMessage((hwnd), LVM_SETTEXTCOLOR, 0, (LPARAM)(COLORREF)(clrText))

// COLORREF ListView_GetTextBkColor(HWND hwnd);
#define LVM_GETTEXTBKCOLOR      (LVM_FIRST + 37)
#define ListView_GetTextBkColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), LVM_GETTEXTBKCOLOR, 0, 0L)

// BOOL ListView_SetTextBkColor(HWND hwnd, COLORREF clrTextBk);
#define LVM_SETTEXTBKCOLOR      (LVM_FIRST + 38)
#define ListView_SetTextBkColor(hwnd, clrTextBk) \
    (BOOL)SendMessage((hwnd), LVM_SETTEXTBKCOLOR, 0, (LPARAM)(COLORREF)(clrTextBk))

// messages for getting the index of the first visible item
#define LVM_GETTOPINDEX         (LVM_FIRST + 39)
#define ListView_GetTopIndex(hwndLV, ppt) \
    (int)SendMessage((hwndLV), LVM_GETTOPINDEX, 0, 0)

// Message for getting the count of items per page
#define LVM_GETCOUNTPERPAGE     (LVM_FIRST + 40)
#define ListView_GetCountPerPage(hwndLV, ppt) \
    (int)SendMessage((hwndLV), LVM_GETCOUNTPERPAGE, 0, 0)

// Message for getting the listview origin, which is needed for SetItemPos...
#define LVM_GETORIGIN           (LVM_FIRST + 41)
#define ListView_GetOrigin(hwndLV, ppt) \
    (BOOL)SendMessage((hwndLV), LVM_GETORIGIN, (WPARAM)0, (LPARAM)(POINT FAR*)(ppt))

// Message for getting the count of items per page
#define LVM_UPDATE     (LVM_FIRST + 42)
#define ListView_Update(hwndLV, i) \
    (BOOL)SendMessage((hwndLV), LVM_UPDATE, (WPARAM)i, 0L)

// set and item's state.  this macro will return VOID.  but the
// message returns BOOL success.
#define LVM_SETITEMSTATE                (LVM_FIRST + 43)
#define ListView_SetItemState(hwndLV, i, data, mask) \
{ LV_ITEM lvi;\
  lvi.stateMask = mask;\
  lvi.state = data;\
  SendMessage((hwndLV), LVM_SETITEMSTATE, (WPARAM)i, (LPARAM)(LV_ITEM FAR *)&lvi);\
}

// get the item's state
#define LVM_GETITEMSTATE                (LVM_FIRST + 44)
#define ListView_GetItemState(hwndLV, i, mask) \
   (UINT)SendMessage((hwndLV), LVM_GETITEMSTATE, (WPARAM)i, (LPARAM)mask)

// get the item  text.
// if you want the int return value of how the buff size, you call it yourself.
#define LVM_GETITEMTEXT                 (LVM_FIRST + 45)
#define ListView_GetItemText(hwndLV, i, iSubItem_, pszText_, cchTextMax_) \
{ LV_ITEM lvi;\
  lvi.iSubItem = iSubItem_;\
  lvi.cchTextMax = cchTextMax_;\
  lvi.pszText = pszText_;\
  SendMessage((hwndLV), LVM_GETITEMTEXT, (WPARAM)i, (LPARAM)(LV_ITEM FAR *)&lvi);\
}

// get the item  text.
// if you want the int return value (BOOL) success do it yourself
#define LVM_SETITEMTEXT                 (LVM_FIRST + 46)
#define ListView_SetItemText(hwndLV, i, iSubItem_, pszText_) \
{ LV_ITEM lvi;\
  lvi.iSubItem = iSubItem_;\
  lvi.pszText = pszText_;\
  SendMessage((hwndLV), LVM_SETITEMTEXT, (WPARAM)i, (LPARAM)(LV_ITEM FAR *)&lvi);\
}


// ListView notification codes

// Structure used by all ListView control notifications.
// Not all fields supply useful info for all notifications:
// iItem will be -1 and others 0 if not used.
// Some return a BOOL, too.
//

typedef struct _NM_LISTVIEW
{
    NMHDR   hdr;
    int     iItem;
    int     iSubItem;
    UINT    uNewState;      // Combination of LVIS_* (if uChanged & LVIF_STATE)
    UINT    uOldState;      // Combination of LVIS_*
    UINT    uChanged;       // Combination of LVIF_* indicating what changed
    POINT   ptAction;       // Only valid for LVN_BEGINDRAG and LVN_BEGINRDRAG
} NM_LISTVIEW;

#define LVN_ITEMCHANGING        (LVN_FIRST-0)  // item changing.  Return FALSE to disallow
#define LVN_ITEMCHANGED         (LVN_FIRST-1)  // item changed.
#define LVN_INSERTITEM          (LVN_FIRST-2)
#define LVN_DELETEITEM          (LVN_FIRST-3)
#define LVN_DELETEALLITEMS      (LVN_FIRST-4)
#define LVN_BEGINLABELEDIT      (LVN_FIRST-5)  // start of label editing
#define LVN_ENDLABELEDIT        (LVN_FIRST-6)  // end of label editing
                                        // (iItem == -1 if cancel)

//(LVN_FIRST-7) not used


#define LVN_COLUMNCLICK         (LVN_FIRST-8)  // column identified by iItem
                                        // was clicked

#define LVN_BEGINDRAG           (LVN_FIRST-9)  // Start of drag operation requested
                                        // (return FALSE if the app handles it)
#define LVN_ENDDRAG             (LVN_FIRST-10)  // End of dragging operation.
#define LVN_BEGINRDRAG          (LVN_FIRST-11)  // Start of button 2 dragging
#define LVN_ENDRDRAG            (LVN_FIRST-12)  // End of button 2 drag (not used yet)

#ifdef PW2
#define LVN_PEN                 (LVN_FIRST-20)  // pen notifications
#endif //PW2

// LVN_DISPINFO notification

#define LVN_GETDISPINFO         (LVN_FIRST-50)
#define LVN_SETDISPINFO         (LVN_FIRST-51)

typedef struct _LV_DISPINFO
{
    NMHDR hdr;
    LV_ITEM item;
} LV_DISPINFO;

// LVN_KEYDOWN notification
#define LVN_KEYDOWN	(LVN_FIRST-55)

typedef struct _LV_KEYDOWN
{
    NMHDR hdr;
    WORD wVKey;
    UINT flags;
} LV_KEYDOWN;

#define FI_SetPathPtr(pfi, psz)     /* Internal */ \
// ====== TREEVIEW APIs =================================================
//
// Class name: SysTreeView (WC_TREEVIEW)
//
// The SysTreeView control provides for a group of items which are
// displayed in a hierarchical organization.  Each item may contain
// independent "sub-item" entries which are displayed below and indented
// from the parent item.
//
// Operation of this control is similar to the SysListView control above,
// except that sub-items are distinct entries, not supporting text elements
// belonging to the owning object (which is the case for the Report View
// mode of the SysListView).
//
// There are notifications that allow applications to determine when an item
// has been clicked or double clicked, caption text changes have occured,
// drag tracking is occuring, widths of columns have changed, node items
// are expanded, etc.
//
// NOTE: All "messages" below are documented as APIs; eventually these
// will be changed to window messages, and corresponding macros will be
// written that have the same signature as the APIs shown below.
//

#define WC_TREEVIEW     "SysTreeView" WC_SUFFIX

#define TVS_HASBUTTONS      0x0001L
#define TVS_HASIMAGES       0x0002L
#define TVS_HASLINES        0x0004L
#define TVS_MULTIPLESEL     0x0008L
#define TVS_LINESATROOT     0x0020L
#define TVS_EDITLABELS      0x0040L

#define TV_ADDITEM          (TV_FIRST +  0)
#define TV_ENABLEITEM       (TV_FIRST +  2)
#define TV_EXPANDCHILDREN   (TV_FIRST +  3)
#define TV_GETCOUNT         (TV_FIRST +  7)
#define TV_GETFIRSTVISIBLE  (TV_FIRST +  8)
#define TV_GETINDENT        (TV_FIRST +  9)
#define TV_GETIMAGELIST     (TV_FIRST + 10)
#define TV_GETITEM          (TV_FIRST + 11)
#define TV_GETITEMINFO      (TV_FIRST + 12)
#define TV_EDITLABEL        (TV_FIRST + 13)
#define TV_GETITEMTEXTLEN   (TV_FIRST + 14)
#define TV_GETSELECT        (TV_FIRST + 15)
#define TV_GETSELECTCOUNT   (TV_FIRST + 16)
#define TV_GETVISIBLECOUNT  (TV_FIRST + 17)
#define TV_REMOVEALL        (TV_FIRST + 18)
#define TV_REMOVEITEM       (TV_FIRST + 19)
#define TV_SETCARET         (TV_FIRST + 22)
#define TV_SETFIRSTVISIBLE  (TV_FIRST + 23)
#define TV_SETIMAGELIST     (TV_FIRST + 24)
#define TV_SETINDENT        (TV_FIRST + 25)
#define TV_SETITEMINFO      (TV_FIRST + 26)
#define TV_SETSELECT        (TV_FIRST + 28)
#define TV_SHOWBUTTONS      (TV_FIRST + 29)
#define TV_SHOWLINES        (TV_FIRST + 30)
#define TV_LINESATROOT      (TV_FIRST + 31)

#define TV_HITTEST          (TV_FIRST + 32)
#define TV_CREATEDRAGIMAGE  (TV_FIRST + 33)
#define TV_SORTCHILDREN     (TV_FIRST + 34)
#define TV_GETEDITCONTROL   (TV_FIRST + 35)
#define TV_ENSUREVISIBLE    (TV_FIRST + 36)

#define TVHT_NOWHERE        0x0001
#define TVHT_ONITEMICON     0x0002
#define TVHT_ONITEMLABEL    0x0004
#define TVHT_ONITEM         (TVHT_ONITEMICON | TVHT_ONITEMLABEL)
#define TVHT_ONITEMINDENT   0x0008
#define TVHT_ONITEMBUTTON   0x0010
#define TVHT_ONITEMRIGHT    0x0020

#define TVHT_ABOVE          0x0100
#define TVHT_BELOW          0x0200
#define TVHT_TORIGHT        0x0400
#define TVHT_TOLEFT         0x0800

#define TVSS_DROPHILITE     0x0010
#define TVSS_ENSUREHVISIBLE 0x0001
#define TVSS_ENSUREVVISIBLE 0x0002
#define TVSS_ENSUREVISIBLE  (TVSS_ENSUREHVISIBLE | TVSS_ENSUREVVISIBLE)
#define TVSS_ENSURESHOWING  0x0004

#define F_HASNOKIDS		0
#define F_HASKIDS		1
#define F_HASKIDSCALLBACK	2


struct _TREEITEM;
typedef struct _TREEITEM FAR* HTREEITEM;

typedef struct _TVHITTEST
{
    POINT       ptHit;
    HTREEITEM   hItem;
} TVHITTEST, FAR *LPTVHITTEST;

#define TVA_ROOT  ((HTREEITEM) 0xFFFF0000)
#define TVA_FIRST ((HTREEITEM) 0xFFFF0001)
#define TVA_LAST  ((HTREEITEM) 0xFFFF0002)
#define TVA_SORT  ((HTREEITEM) 0xFFFF0003)

#define TVIM_HANDLE     0x0001
#define TVIM_RECT       0x0002
#define TVIM_TEXT       0x0004
#define TVIM_DATA       0x0008
#define TVIM_KIDS       0x0010
#define TVIM_NORMALIMAGE      0x0020
#define TVIM_SELECTEDIMAGE    0x0040
#define TVIM_DISABLEDIMAGE    0x0080
#define TVIM_ALLIMAGES  (TVIM_NORMALIMAGE | TVIM_SELECTEDIMAGE |TVIM_DISABLEDIMAGE)

typedef struct _TVITEMINFO
{
    UINT        wMask;		// Set of TVIM_* flags
    HTREEITEM   hItem;		// The item to be changed
    WORD      iNormalImage;  // The index of the image for this item
    WORD      iSelectedImage; // the index of the selected imagex
    WORD      iDisabledImage; // the index of the disabled image
    LPSTR       lpText;		// The text for this item
    WORD       cText;		// The length of the lpText buffer
    WORD       fHasKids;	// F_HASKIDS, F_HASNOKIDS, F_HASKIDSCALLBACK
    DWORD       dwData;		// App defined data
    RECT        rcItem;		// The rect of the item (returned only)
} TVITEMINFO, FAR *LPTVITEMINFO;

typedef struct _TREEADDSTRUCT
{
    HTREEITEM   hParent;
    HTREEITEM   hInsertAfter;
    TVITEMINFO  info;
} TREEADDSTRUCT, FAR *LPTREEADDSTRUCT;

typedef struct _CREATETREESTRUCT
{
    HIMAGELIST  hImageList;
} CREATETREESTRUCT, FAR *LPCREATETREESTRUCT;

typedef struct tagNM_TREEVIEW
{
    NMHDR       hdr;
    BYTE        wMask;          // BUGBUG: make UINT
    BYTE        wFlags;         // BUGBUG: make UINT
    TVITEMINFO  infoOld;
    TVITEMINFO  infoNew;
    POINT       ptDrag;
} NM_TREEVIEW, FAR *LPNM_TREEVIEW;

typedef struct _TV_KEYDOWN
{
    NMHDR hdr;
    WORD wVKey;
    //UINT flags;
} TV_KEYDOWN;

#define TVN_SELCHANGING     (TVN_FIRST-1)
#define TVN_SELCHANGED      (TVN_FIRST-2)

#define TVC_BYMOUSE         0x0001
#define TVC_BYKEYBOARD      0x0002

#define TVN_TEXTCHANGING    (TVN_FIRST-3)
#define TVN_TEXTCHANGED     (TVN_FIRST-4)

#define TVN_GETDISPINFO     (TVN_FIRST-5)
#define TVN_SETDISPINFO     (TVN_FIRST-6)

#define TVN_ITEMEXPANDING   (TVN_FIRST-7)
#define TVN_ITEMEXPANDED    (TVN_FIRST-8)

#define TVE_COLLAPSE        0x0001
#define TVE_EXPAND          0x0002
#define TVE_TOGGLE          0x0003
#define TVE_ACTION          (TVE_COLLAPSE|TVE_EXPAND|TVE_TOGGLE)
#define TVE_FIRSTEXPAND     0x0004
#define TVE_COLLAPSERESET   0x0008

#define TVN_BEGINDRAG       (TVN_FIRST-9)
#define TVN_BEGINRDRAG      (TVN_FIRST-10)

#define TVN_REMOVEITEM      (TVN_FIRST-11)

#define TVN_BEGINLABELEDIT  (TVN_FIRST-12)
#define TVN_ENDLABELEDIT    (TVN_FIRST-13)

#define TVN_KEYDOWN             (TVN_FIRST-14)

#define TVGI_ROOT               0x0000
#define TVGI_NEXT               0x0001
#define TVGI_PREVIOUS           0x0002
#define TVGI_PARENT             0x0004
#define TVGI_CHILD              0x0008
#define TVGI_NEXTVISIBLE        0x0010
#define TVGI_PREVIOUSVISIBLE    0x0020

//============================================================================
//
// Class name: SysTabControl (WC_TABCONTROL)
//
//////////////////////////////////////////////////////////////////////////////

#define WC_TABCONTROL         "SysTabControl" WC_SUFFIX // window class name

// window styles to control tab control behavior

#define TCS_SORTNONE	    0x00000000L
#define TCS_SORTASCENDING   0x00000010L
#define TCS_SORTDESCENDING  0x00000020L

#define TCS_SHAREIMAGELISTS 0x00000040L

#define TCS_TABS		0x00000000L  // default
#define TCS_BUTTONS		0x00000100L
#define TCS_SINGLELINE		0x00000000L  // default
#define TCS_MULTILINE		0x00000200L
#define TCS_RIGHTJUSTIFY	0x00000000L  // default
#define TCS_FIXEDWIDTH		0x00000400L
#define TCS_RAGGEDRIGHT		0x00000800L
#define TCS_FOCUSONBUTTONDOWN   0x00001000L

#define TCS_ALIGNTOP        0x00000000L
#define TCS_ALIGNBOTTOM     0x00000100L
#define TCS_ALIGNRIGHT      0x00000200L
#define TCS_ALIGNLEFT       0x00000300L
#define TCS_ALIGNMASK       0x00000300L

#define TCM_FIRST	    0x1300	    // Tab Control messages


// COLORREF TabCtrl_GetBkColor(HWND hwnd);
#define TCM_GETBKCOLOR      (TCM_FIRST + 0)
#define TabCtrl_GetBkColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), TCM_GETBKCOLOR, 0, 0L)

// BOOL TabCtrl_SetBkColor(HWND hwnd, COLORREF clrBk);
#define TCM_SETBKCOLOR      (TCM_FIRST + 1)
#define TabCtrl_SetBkColor(hwnd, clrBk) \
    (BOOL)SendMessage((hwnd), TCM_SETBKCOLOR, 0, (LPARAM)(COLORREF)(clrBk))

// HIMAGELIST TabCtrl_GetImageList(HWND hwnd);
#define TCM_GETIMAGELIST    (TCM_FIRST + 2)
#define TabCtrl_GetImageList(hwnd) \
    (HIMAGELIST)SendMessage((hwnd), TCM_GETIMAGELIST, 0, 0L)

// this returns the old image list (null if no previous)
// BOOL TabCtrl_SetImageList(HWND hwnd, HIMAGELIST himl);
#define TCM_SETIMAGELIST    (TCM_FIRST + 3)
#define TabCtrl_SetImageList(hwnd, himl) \
    (HIMAGELIST)SendMessage((hwnd), TCM_SETIMAGELIST, 0, (LPARAM)(UINT)(HIMAGELIST)(himl))

// int TabCtrl_GetItemCount(HWND hwnd);
#define TCM_GETITEMCOUNT    (TCM_FIRST + 4)
#define TabCtrl_GetItemCount(hwnd) \
    (int)SendMessage((hwnd), TCM_GETITEMCOUNT, 0, 0L)

// TabView Item structure

#define TCIF_TEXT       0x0001  // TabView mask flags
#define TCIF_IMAGE      0x0002
#define TCIF_PARAM      0x0008
#define TCIF_STATE      0x0010


typedef struct _TC_ITEMHEADER
{
    UINT mask;		// TCIF_ bits
    UINT state;		// TCIS_ bits
    UINT stateMask;
    LPSTR pszText;
    int cchTextMax;
    int iImage;
} TC_ITEMHEADER;

typedef struct _TC_ITEM
{
    // This block must be identical to TC_TEIMHEADER
    UINT mask;		// TCIF_ bits
    UINT state;		// TCIS_ bits
    UINT stateMask;
    LPSTR pszText;
    int cchTextMax;
    int iImage;

    LPARAM lParam;
} TC_ITEM;

// State flags
#define TCIS_FOCUSED        0x0001
#define TCIS_SELECTED       0x0002
#define TCIS_MARKED         0x0004
#define TCIS_DROPHILITED    0x0008

#define TCIS_DISABLED       0x0100
#define TCIS_HIDDEN         0x0200


// BOOL TabCtrl_GetItem(HWND hwnd, int iItem, TC_ITEM FAR* pitem);
#define TCM_GETITEM         (TCM_FIRST + 5)
#define TabCtrl_GetItem(hwnd, iItem, pitem) \
    (BOOL)SendMessage((hwnd), TCM_GETITEM, (WPARAM)(int)iItem, (LPARAM)(TC_ITEM FAR*)(pitem))

// BOOL TabCtrl_SetItem(HWND hwnd, int iItem, TC_ITEM FAR* pitem);
#define TCM_SETITEM         (TCM_FIRST + 6)
#define TabCtrl_SetItem(hwnd, iItem, pitem) \
    (BOOL)SendMessage((hwnd), TCM_SETITEM, (WPARAM)(int)iItem, (LPARAM)(TC_ITEM FAR*)(pitem))

// int TabCtrl_InsertItem(HWND hwnd, int iItem, const TC_ITEM FAR* pitem);
#define TCM_INSERTITEM         (TCM_FIRST + 7)
#define TabCtrl_InsertItem(hwnd, iItem, pitem)   \
    (int)SendMessage((hwnd), TCM_INSERTITEM, (WPARAM)(int)iItem, (LPARAM)(const TC_ITEM FAR*)(pitem))

// Deletes the specified item along with all its subitems.
//
// BOOL TabCtrl_DeleteItem(HWND hwnd, int i);
#define TCM_DELETEITEM      (TCM_FIRST + 8)
#define TabCtrl_DeleteItem(hwnd, i) \
    (BOOL)SendMessage((hwnd), TCM_DELETEITEM, (WPARAM)(int)(i), 0L)

// BOOL TabCtrl_DeleteAllItems(HWND hwnd);
#define TCM_DELETEALLITEMS  (TCM_FIRST + 9)
#define TabCtrl_DeleteAllItems(hwnd) \
    (BOOL)SendMessage((hwnd), TCM_DELETEALLITEMS, 0, 0L);

    // Rectangle bounding all or part of item, based on code.  Rect is returned in view coords
    // BOOL TabCtrl_GetItemRect(HWND hwndTC, int i, RECT FAR* prc);
#define TCM_GETITEMRECT     (TCM_FIRST + 10)
#define TabCtrl_GetItemRect(hwnd, i, prc) \
    (BOOL)SendMessage((hwnd), TCM_GETITEMRECT, (WPARAM)(int)(i), (LPARAM)(RECT FAR*)(prc))

    // Rectangle bounding all or part of item, based on code.  Rect is returned in view coords
    // BOOL TabCtrl_GetCurSel(HWND hwndTC);
#define TCM_GETCURSEL     (TCM_FIRST + 11)
#define TabCtrl_GetCurSel(hwnd) \
    (int)SendMessage((hwnd), TCM_GETCURSEL, 0, 0)

#define TCM_SETCURSEL     (TCM_FIRST + 12)
#define TabCtrl_SetCurSel(hwnd, i) \
    (int)SendMessage((hwnd), TCM_SETCURSEL, (WPARAM)i, 0)

    // ItemHitTest flag values
#define TCHT_NOWHERE        0x0001
#define TCHT_ONITEMICON     0x0002
#define TCHT_ONITEMLABEL    0x0004
#define TCHT_ONITEM         (TCHT_ONITEMICON | TCHT_ONITEMLABEL)

 // int TabCtrl_HitTest(HWND hwndTC, TC_HITTESTINFO FAR* pinfo);
#define TCM_HITTEST     (TCM_FIRST + 13)
#define TabCtrl_HitTest(hwndTC, pinfo) \
    (int)SendMessage((hwndTC), TCM_HITTEST, 0, \
    (LPARAM)(TC_HITTESTINFO FAR*)(pinfo))

// Set the size of extra byte (abExtra[]) for each item.
#define TCM_SETITEMEXTRA    (TCM_FIRST + 14)
#define TabCtrl_SetItemExtra(hwndTC, cb) \
    (BOOL)SendMessage((hwndTC), TCM_SETITEMEXTRA, (WPARAM)(cb), 0L)

// get/set text and textbk color for text drawing.  these override
// the standard window/windowtext settings.  they do NOT override
// when drawing selected text.
// COLORREF TabCtrl_GetTextColor(HWND hwnd);
#define TCM_GETTEXTCOLOR      (TCM_FIRST + 35)
#define TabCtrl_GetTextColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), TCM_GETTEXTCOLOR, 0, 0L)

// BOOL TabCtrl_SetTextColor(HWND hwnd, COLORREF clrText);
#define TCM_SETTEXTCOLOR      (TCM_FIRST + 36)
#define TabCtrl_SetTextColor(hwnd, clrText) \
    (BOOL)SendMessage((hwnd), TCM_SETTEXTCOLOR, 0, (LPARAM)(COLORREF)(clrText))

// COLORREF TabCtrl_GetTextBkColor(HWND hwnd);
#define TCM_GETTEXTBKCOLOR      (TCM_FIRST + 37)
#define TabCtrl_GetTextColor(hwnd)  \
    (COLORREF)SendMessage((hwnd), TCM_GETTEXTCOLOR, 0, 0L)

// BOOL TabCtrl_SetTextBkColor(HWND hwnd, COLORREF clrTextBk);
#define TCM_SETTEXTBKCOLOR      (TCM_FIRST + 38)
#define TabCtrl_SetTextBkColor(hwnd, clrTextBk) \
    (BOOL)SendMessage((hwnd), TCM_SETTEXTBKCOLOR, 0, (LPARAM)(COLORREF)(clrTextBk))

#define TCM_ADJUSTRECT	(TCM_FIRST + 40)
#define TabCtrl_AdjustRect(hwnd, bLarger, prc) \
    (void)SendMessage(hwnd, TCM_ADJUSTRECT, (WPARAM)(BOOL)bLarger, (LPARAM)(RECT FAR *)prc)

#define TCM_SETITEMWIDTH	(TCM_FIRST + 41)
#define TabCtrl_SetItemWidth(hwnd, i) \
    (void)SendMessage((hwnd), TCM_SETITEMWIDTH, i, 0L)

#define TCM_REMOVEIMAGE         (TCM_FIRST + 42)
#define TabCtrl_RemoveImage(hwnd, i) \
        (void)SendMessage((hwnd), TCM_REMOVEIMAGE, i, 0L)

#define TCM_SETPADDING          (TCM_FIRST + 43)
#define TabCtrl_SetPadding(hwnd,  cx, cy) \
        (void)SendMessage((hwnd), TCM_SETPADDING, 0, MAKELPARAM(cx, cy));

#define TCM_GETROWCOUNT         (TCM_FIRST + 44)
#define TabCtrl_GetRowCount(hwnd) \
        (int)SendMessage((hwnd), TCM_GETROWCOUNT, 0, 0L)

// TabView notification codes

// selection has changed
#define TCN_SELCHANGE	    1

// selection changing away from current tab
// return:  FALSE to continue, or TRUE to not change
#define TCN_SELCHANGING	    2	



#define WM_DRAGSHOW	    0x4435  // another hacked message

typedef struct _WM_SETSTYLEINFO
{
    DWORD styleNew;
    DWORD styleOld;
    DWORD mask;
} WM_SETSTYLEINFO;

//REVIEW: We need to add an "int cbHdr" field to NMHDR!

LRESULT WINAPI SendNotify(HWND hwndTo, HWND hwndFrom, int code, NMHDR FAR* pnmhdr);

/* LRESULT Cls_OnNotify(HWND hwnd, int idFrom, NMHDR FAR* pnmhdr); */
#define HANDLE_WM_NOTIFY(hwnd, wParam, lParam, fn) \
    (fn)((hwnd), (int)(wParam), (NMHDR FAR*)(lParam))
#define FORWARD_WM_NOTIFY(hwnd, idFrom, pnmhdr, fn) \
    (void)(fn)((hwnd), WM_NOTIFY, (WPARAM)(int)(id), (LPARAM)(NMHDR FAR*)(pnmhdr))

// Generic WM_NOTIFY notification codes


#define NM_OUTOFMEMORY          (NM_FIRST-1)

#define NM_CLICK                (NM_FIRST-2)
#define NM_DBLCLK               (NM_FIRST-3)
#define NM_RETURN               (NM_FIRST-4)
#define NM_RCLICK               (NM_FIRST-5)
#define NM_RDBLCLK              (NM_FIRST-6)
#define NM_SETFOCUS             (NM_FIRST-7)
#define NM_KILLFOCUS            (NM_FIRST-8)

/* util.c */

BOOL WINAPI Shell_Initialize(void);     // app must call this before using any shell services
BOOL WINAPI Shell_Terminate(void);      // app must call this on exit to clean up shell services


//===================================================================
// Hotkey management API's

// Set the pending hotkey for the given app,
// The next top level window to be created by the given app will be given
// this hotkey.
BOOL WINAPI Hotkey_SetPending(LPCSTR lpszPath, WORD wHotkey);

// Check the list of pending hotkeys and change the first occurence
// of lpszFrom to lpszTo
BOOL WINAPI Hotkey_ChangePending(LPCSTR lpszFrom, LPCSTR lpszTo);

// Delete all pending hotkeys.
void WINAPI Hotkey_DeleteAllPending(void);

// Set the hotkey for the given instance of an app.
BOOL WINAPI Hotkey_SetByInstance(HINSTANCE hinst, WORD wHotkey);

// Delete a pending instance.
BOOL WINAPI Hotkey_DeletePending(LPCSTR lpszPath);

// Get a pending hotkey given a path.
WORD WINAPI Hotkey_GetPending(LPCSTR lpszPath);

//===================================================================
// Penshell API's

#ifdef PW2
#define PW2F_OBJECTINKING	0x0002	// allow inking on objects if set

UINT WINAPI PenShell_GetFlags(VOID);
UINT WINAPI PenShell_SetFlags(UINT fFlags);
#endif //PW2



HINSTANCE WINAPI ShellExecuteEx(HWND hwnd, LPCSTR lpVerb, LPCSTR lpFile,
        LPCSTR lpParameters, LPCSTR lpDirectory, LPCSTR lpClass,
        int nShow);

// Hint flags for ShellFileClass
#define SFC_UNKNOWN         0x0000
#define SFC_FILE            0x0001
#define SFC_DIRECTORY       0x0002
#define SFC_SYSTEM          0x0004
#define SFC_TYPEMASK        (SFC_FILE|SFC_DIRECTORY|SFC_SYSTEM)
#define SFC_EXIST           0x4000  // We already know that this one exists
#define SFC_FULLYQUALIFIED  0x8000  // lpFile is fully qualified path.

int WINAPI ShellFileClass(LPCSTR lpFile, UINT wFlags,
        LPSTR lpszClassName, UINT cbClassName);
#define EIRESID(x) (-1 * (int)(x))
UINT WINAPI ExtractIconEx(LPCSTR lpszFile, int nIconIndex,
        HICON FAR *phiconLarge, HICON FAR *phiconSmall, UINT nIcons);

// Drag/Drop API
typedef struct _DRAGINFO {
    UINT uSize;			/* init with sizeof(DRAGINFO) */
    POINT pt;
    BOOL fNC;
    DWORD dwEffect;
    DWORD grfKeyState;
    HWND hwndSource;
    LPSTR lpFileList;
} DRAGINFO, FAR* LPDRAGINFO;

BOOL WINAPI DragQueryInfo(HDROP, LPDRAGINFO);	/* get extra info about a drop */

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#ifndef RC_INVOKED
#pragma pack()
#endif  /* !RC_INVOKED */

#endif  // !_INC_SHELL
