/*  $Id$
 *  
 *  File	verify.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Verify utility
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	05/10/93  (Created)
 *  		30/11/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"

char *		UTILITY_NAME = "Verify utility";
char *		UTILITY_VERSION = "1.2.0";

void		helpUtility(FILE *fd);

bool		VERIFY_HEADER = TRUE;
bool		VERIFY_MOVES = TRUE;
bool		VERIFY_NAMES = FALSE;
bool		VERIFY_INDEX = FALSE;


/*------------------------------------------------------------
 *  Verify utility
 *------------------------------------------------------------*/

void
verifyUtility(Option opt)
{ Game g = newGame();
  CBase cb = opt->database;
  long from = opt->from;
  long to = (opt->to < getNoGamesCBase(cb) ? opt->to : getNoGamesCBase(cb));
  long count;
  long n;
  long inc;
  unsigned long nextIndex = 0L;		/* Next index expected */
  unsigned long bytesWasted = 0L;	/* Bytes wasted */
  unsigned long gamesDisplaced = 0L;	/* Games with incorrect order */

  reportCBase(cb, stderr);

  n = to-from+1;
  if (n < 0)
    return;
  inc = (n < 50L ? 1 : n / 50);
  
  for (n=from, count=0; n<=to; n++, count++)
  { if (inc == count)
    { fprintf(stderr, ".");
      count = 0;
    }

    environmentError(cb, g, n);
    initialiseGame(g, n, cb);
    if (foundError())
    { reportError(stderr);
      continue;
    }
    
    if (VERIFY_INDEX == TRUE)
    { unsigned long index;

      index = getIndexGameCBase(cb, n);
      if (foundError())
      { reportError(stderr);
	continue;
      }
	
      if (index < nextIndex)
      { fprintf(stderr, "\nGame %6ld: Game is displaced (%ld, %ld)",
		n, nextIndex, index);
	gamesDisplaced++;
      }
      if (index > nextIndex)
      { fprintf(stderr, "\nGame %6ld: %ld bytes wasted (%ld, %ld)",
		n, (index-nextIndex), nextIndex, index);
	bytesWasted += (index - nextIndex);
      }
      if (index >= nextIndex)
	nextIndex = index + bytesGame(g);
    }

    if (VERIFY_HEADER == TRUE)
    { checkHeaderGame(g);
      if (foundError())
      { reportError(stderr);
	continue;
      }
    }

    if (VERIFY_MOVES == TRUE)
    { getFirstMoveGame(g);
      if (foundError())
      { reportError(stderr);
	continue;
      }
    }

    if (VERIFY_NAMES == TRUE)
    { char name[MAX_NAME_SIZE+1];

      if (getWhiteGame(g) == NULL || getBlackGame(g) == NULL)
      { fprintf(stderr, "Game %6ld: Could not extract player names\n", n);
	fprintf(stderr, "             -- Players: %s\n", getPlayersGame(g));
	continue;
      }
      
      stringChessSymbol(getPlayersGame(g), name);
      if (foundError())
      { reportError(stderr);
        continue;
      }
      
      stringChessSymbol(getSourceGame(g), name);
      if (foundError())
      { reportError(stderr);
	continue;
      }
    }
  }
  fprintf(stderr, "\n");

  if (VERIFY_INDEX)
  { fprintf(stderr, "Games displaced:  %6ld\n", gamesDisplaced);
    fprintf(stderr, "Bytes wasted:     %6ld\n", bytesWasted);
    if (gamesDisplaced || bytesWasted)
      fprintf(stderr, "Use ``cuclean'' to correct the above\n");
  }

  freeGame(g);
}


/*------------------------------------------------------------
 *  Main
 *------------------------------------------------------------*/

int
main(int argc, char *argv[])
{ int i;
  Option options = newOption();

  initChessSymbols(NULL);

  for (i=1; i<argc; i++)
  {
    if (strhead(argv[i], "-verify"))
    { VERIFY_MOVES = TRUE;
      VERIFY_HEADER = TRUE;
      continue;
    }

    if (strhead(argv[i], "-noverify"))
    { VERIFY_MOVES = FALSE;
      VERIFY_HEADER = FALSE;
      continue;
    }

    if (strhead(argv[i], "-moves"))
    { VERIFY_MOVES = TRUE;
      continue;
    }

    if (strhead(argv[i], "-nomoves"))
    { VERIFY_MOVES = FALSE;
      continue;
    }

    if (strhead(argv[i], "-names"))
    { VERIFY_NAMES = TRUE;
      continue;
    }

    if (strhead(argv[i], "-nonames"))
    { VERIFY_NAMES = FALSE;
      continue;
    }

    if (strhead(argv[i], "-index"))
    { VERIFY_INDEX = TRUE;
      continue;
    }

    if (strhead(argv[i], "-noindex"))
    { VERIFY_INDEX = FALSE;
      continue;
    }

    if (strhead(argv[i], "-"))
    { int n;

      n = genericOption(options, argv, argc, i);
      if (n == 0)
      { fprintf(stderr, "Fatal: Unknown command %s\n", argv[i]);
	fprintf(stderr, "Do ``%s -help'' or see the documentation\n", argv[0]);
	exit(1);
      }
      i = n;
      continue;
    }

    setCurrentCBase(argv[i], "-database", argc, i);
    options->database = CurrentBase;
    verifyUtility(options);
    freeCBase(options->database);
    options->database = (CBase) NULL;
  }

  if (options->database)
  { verifyUtility(options);
    freeCBase(options->database);
  }

  return 0;
}


/*------------------------------------------------------------
 *  Help
 *------------------------------------------------------------*/

void
helpUtility(FILE *fd)
{ helpCBUFF(fd);
  fprintf(fd, "%s options:\n", UTILITY_NAME);
  fprintf(fd, "-index        Verifies index entries for gaps\n");
  fprintf(fd, "-moves        Verifies moves (default)\n");
  fprintf(fd, "-names        Verifies names of players and source\n");
  fprintf(fd, "-noindex      Does not verify index entries for gaps (default)\n");
  fprintf(fd, "-nomoves      Does not verify moves\n");
  fprintf(fd, "-nonames      Does not verify names of players and source (default)\n");
  fprintf(fd, "-noverify     Does not verify header and moves\n");
  fprintf(fd, "-verify       Verifies header and moves (default)\n");
}
