/*  $Id$
 *  
 *  File	symbols.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Handling of chess symbols and other special characters
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	19/06/93  (Created)
 *  		09/12/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


/*------------------------------------------------------------
 *  Definitions
 *------------------------------------------------------------*/

char *		SymbolNames[LAST_SYMBOL];
char *		AsciiSymbols[LAST_SYMBOL];

static char *	quotedString(char *);


static int	ChessSymbol[] =
           /*      0              1              2              3
		   4              5              6              7 */
/* 0200 */ { CEDILLE_C,     UMLAUT_u,      ACUTE_e,       CIRCUMFLEX_a,
             UMLAUT_a,      GRAVE_a,       DOT_a,         CEDILLE_c,
/* 0210 */   CIRCUMFLEX_e,  UMLAUT_e,	   GRAVE_e,       UMLAUT_i,
	     CIRCUMFLEX_i,  GRAVE_i,       UMLAUT_A,      DOT_A,
/* 0220 */   ACUTE_E,	    SCANDI_ae,     SCANDI_AE,	  CIRCUMFLEX_o,
	     UMLAUT_o,      GRAVE_o,	   CIRCUMFLEX_u,  GRAVE_u,
/* 0230 */   UMLAUT_y,      UMLAUT_O,      UMLAUT_U,      0   ,
	     0   ,	    0   ,	   GERMAN_S,	  0   ,
/* 0240 */   ACUTE_a,	    ACUTE_i,	   ACUTE_o,	  ACUTE_u,
	     TILDE_n,	    TILDE_N,	   0   ,	  0   ,
/* 0250 */   0   ,	    0   ,	   0   ,	  0   ,
	     QUEEN_SYMBOL,  0   ,	   0   ,	  0   ,
/* 0260 */   ETC,           KING_SYMBOL,   QUEEN_SYMBOL,  KNIGHT_SYMBOL,
	     BISHOP_SYMBOL, ROOK_SYMBOL,   PAWN_SYMBOL,   QUARTER,
/* 0270 */   HALF,          THREE_QUARTERS,0   ,          WHITE_ADVANTAGE,
	     WHITE_BETTER,WHITE_COMPENSATION, UNCLEAR,    BLACK_COMPENSATION,
/* 0300 */   BLACK_BETTER, BLACK_ADVANTAGE, BISHOP_PAIR,  OPPOSITE_BISHOPS,
	     SAME_BISHOPS,  WORSE_IS,      BETTER_IS,     BETTER_IS,
/* 0310 */   SPACE_ADVANTAGE, ONLY_MOVE,   WITH_IDEA,     QUEENS_SIDE,
	     KINGS_SIDE,    KINGS_SIDE,    QUEENS_SIDE,   WITH_INITIATIVE,
/* 0320 */   WITH_ATTACK,   WITH_COUNTERPLAY, DIAGONAL,   CENTRE,
	     LINE_FILE,     ENDGAME,       WEAK_POINT,    ZUGZWANG,
/* 0330 */   TIME_TROUBLE,  PAWN_EVALUATION, PAWN_CONNECTION, DOUBLE_PAWN,
	     PASSED_PAWN,   0   ,          0   ,          DEVELOPMENT_ADVANTAGE,
/* 0340 */   0   , GERMAN_S, 0   ,   0   ,   0   ,   0   ,   0   ,   0   ,
/* 0350 */   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,
/* 0360 */   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,   0   ,
/* 0370 */   0   ,   0   ,   0   ,   0   ,   0   ,   0   , NEXT_LINE,0   
};


/*------------------------------------------------------------
 *  Initialisation
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node initChessSymbols
@deftypefun void initChessSymbols (char *@var{symbolsFile})
This function should be called before any other processing.
It initialises the mapping of chess symbols in ChessBase format
to those in ASCII format.  The argument is the name of ``symbols''
file.  When @var{symbolsFile} is NULL, then the standard file
@file{ascii.sym} will be used.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
initChessSymbols(char *symbolsFile)
{ int i;

  if (symbolsFile == NULL)
    symbolsFile = "ascii.sym";

  for (i=0; i<LAST_SYMBOL; i++)
    SymbolNames[i] = AsciiSymbols[i] = 0   ;
  
  SymbolNames[ETC] = "Etc";
  SymbolNames[KING_SYMBOL] = "KingSymbol";
  SymbolNames[QUEEN_SYMBOL] = "QueenSymbol";
  SymbolNames[ROOK_SYMBOL] = "RookSymbol";
  SymbolNames[BISHOP_SYMBOL] = "BishopSymbol";
  SymbolNames[KNIGHT_SYMBOL] = "KnightSymbol";
  SymbolNames[PAWN_SYMBOL] = "PawnSymbol";
  SymbolNames[QUARTER] = "Quarter";
  SymbolNames[HALF] = "Half";
  SymbolNames[THREE_QUARTERS] = "ThreeQuarters";
  SymbolNames[WHITE_BETTER] = "WhiteBetter";
  SymbolNames[WHITE_ADVANTAGE] = "WhiteAdvantage";
  SymbolNames[WHITE_WINNING] = "WhiteWinning";
  SymbolNames[BLACK_BETTER] = "BlackBetter";
  SymbolNames[BLACK_ADVANTAGE] = "BlackAdvantage";
  SymbolNames[BLACK_WINNING] = "BlackWinning";
  SymbolNames[WHITE_COMPENSATION] = "WhiteCompensation";
  SymbolNames[BLACK_COMPENSATION] = "BlackCompensation";
  SymbolNames[UNCLEAR] = "Unclear";
  SymbolNames[BISHOP_PAIR] = "BishopPair";
  SymbolNames[OPPOSITE_BISHOPS] = "OppositeBishops";
  SymbolNames[SAME_BISHOPS] = "SameBishops";
  SymbolNames[DOUBLE_PAWN] = "DoublePawn";
  SymbolNames[PASSED_PAWN] = "PassedPawn";
  SymbolNames[ISOLATED_PAWN] = "IsolatedPawn";
  SymbolNames[CONNECTED_PAWN] = "ConnectedPawn";
  SymbolNames[WORSE_IS] = "WorseIs";
  SymbolNames[BETTER_IS] = "BetterIs";
  SymbolNames[SPACE_ADVANTAGE] = "SpaceAdvantage";
  SymbolNames[ONLY_MOVE] = "OnlyMove";
  SymbolNames[WITH_IDEA] = "WithIdea";
  SymbolNames[QUEENS_SIDE] = "QueensSide";
  SymbolNames[KINGS_SIDE] = "KingsSide";
  SymbolNames[WITH] = "With";
  SymbolNames[WITHOUT] = "Without";
  SymbolNames[WITH_INITIATIVE] = "WithInitiative";
  SymbolNames[WITH_ATTACK] = "WithAttack";
  SymbolNames[WITH_COUNTERPLAY] = "WithCounterplay";
  SymbolNames[DIAGONAL] = "Diagonal";
  SymbolNames[CENTRE] = "Centre";
  SymbolNames[LINE_FILE] = "LineFile";
  SymbolNames[ENDGAME] = "Endgame";
  SymbolNames[ZUGZWANG] = "Zugzwang";
  SymbolNames[TIME_TROUBLE] = "TimeTrouble";
  SymbolNames[DEVELOPMENT_ADVANTAGE] = "DevelopmentAdvantage";
  SymbolNames[WHITE_WINS] = "WhiteWins";
  SymbolNames[BLACK_WINS] = "BlackWins";
  SymbolNames[DRAW] = "Draw";
  SymbolNames[EQUALITY] = "Equality";
  SymbolNames[NOVELTY] = "Novelty";
  SymbolNames[COMPENSATION] = "Compensation";
  SymbolNames[WEAK_POINT] = "WeakPoint";
  SymbolNames[PAWN_EVALUATION] = "PawnEvaluation";
  SymbolNames[PAWN_CONNECTION] = "PawnConnection";
  SymbolNames[UMLAUT_u] = "Umlaut_u";
  SymbolNames[UMLAUT_a] = "Umlaut_a";
  SymbolNames[UMLAUT_U] = "Umlaut_U";
  SymbolNames[GERMAN_S] = "German_S";
  SymbolNames[NEXT_LINE] = "NextLine";
  SymbolNames[EDITORIAL_COMMENT] = "EditorialComment";
  SymbolNames[GOOD_MOVE] = "GoodMove";
  SymbolNames[BAD_MOVE] = "BadMove";
  SymbolNames[INTERESTING_MOVE] = "InterestingMove";
  SymbolNames[DUBIOUS_MOVE] = "DubiousMove";
  SymbolNames[BRILLIANT_MOVE] = "BrilliantMove";
  SymbolNames[BLUNDER] = "Blunder";
  SymbolNames[MATE] = "Mate";
  SymbolNames[CEDILLE_C] = "Cedille_C";
  SymbolNames[ACUTE_e] = "Acute_e";
  SymbolNames[CIRCUMFLEX_a] = "Circumflex_a";
  SymbolNames[GRAVE_a] = "Grave_a";
  SymbolNames[DOT_a] = "Dot_a";
  SymbolNames[CEDILLE_c] = "Cedille_c";
  SymbolNames[CIRCUMFLEX_e] = "Circumflex_e";
  SymbolNames[UMLAUT_e] = "Umlaut_e";
  SymbolNames[GRAVE_e] = "Grave_e";
  SymbolNames[UMLAUT_i] = "Umlaut_i";
  SymbolNames[CIRCUMFLEX_i] = "Circumflex_i";
  SymbolNames[GRAVE_i] = "Grave_i";
  SymbolNames[UMLAUT_A] = "Umlaut_A";
  SymbolNames[DOT_A] = "Dot_A";
  SymbolNames[ACUTE_E] = "Acute_E";
  SymbolNames[SCANDI_ae] = "Scandi_ae";
  SymbolNames[SCANDI_AE] = "Scandi_AE";
  SymbolNames[CIRCUMFLEX_o] = "Circumflex_o";
  SymbolNames[GRAVE_o] = "Grave_o";
  SymbolNames[CIRCUMFLEX_u] = "Circumflex_u";
  SymbolNames[GRAVE_u] = "Grave_u";
  SymbolNames[UMLAUT_y] = "Umlaut_y";
  SymbolNames[UMLAUT_O] = "Umlaut_O";
  SymbolNames[UMLAUT_o] = "Umlaut_o";
  SymbolNames[ACUTE_a] = "Acute_a";
  SymbolNames[ACUTE_i] = "Acute_i";
  SymbolNames[ACUTE_o] = "Acute_o";
  SymbolNames[ACUTE_u] = "Acute_u";
  SymbolNames[TILDE_n] = "Tilde_n";
  SymbolNames[TILDE_N] = "Tilde_N";
  SymbolNames[CASTLING_SHORT] = "CastlingShort";
  SymbolNames[CASTLING_LONG] = "CastlingLong";
  SymbolNames[CAPTURE_SYMBOL] = "CaptureSymbol";
  SymbolNames[INCHECK_SYMBOL] = "IncheckSymbol";
  SymbolNames[MOVE_HYPHEN] = "MoveHyphen";
  SymbolNames[PROMOTION_SYMBOL] = "PromotionSymbol";
  SymbolNames[WHITE_MOVE_SYMBOL] = "WhiteMoveSymbol";
  SymbolNames[BLACK_MOVE_SYMBOL] = "BlackMoveSymbol";
  SymbolNames[START_MAJOR_ALTERNATIVE] = "StartMajorAlternative";
  SymbolNames[END_MAJOR_ALTERNATIVE] = "EndMajorAlternative";
  SymbolNames[START_ALTERNATIVE] = "StartAlternative";
  SymbolNames[END_ALTERNATIVE] = "EndAlternative";
  SymbolNames[START_COMMENT] = "StartComment";
  SymbolNames[END_COMMENT] = "EndComment";
  SymbolNames[START_SUB_ALTERNATIVE] = "StartSubAlternative";
  SymbolNames[START_DIAGRAM] = "StartDiagram";
  SymbolNames[END_DIAGRAM] = "EndDiagram";
  SymbolNames[START_DIAGRAM_RANK] = "StartDiagramRank";
  SymbolNames[END_DIAGRAM_RANK] = "EndDiagramRank";
  SymbolNames[DIAGRAM_WK] = "DiagramWK";
  SymbolNames[DIAGRAM_WQ] = "DiagramWQ";
  SymbolNames[DIAGRAM_WR] = "DiagramWR";
  SymbolNames[DIAGRAM_WB] = "DiagramWB";
  SymbolNames[DIAGRAM_WN] = "DiagramWN";
  SymbolNames[DIAGRAM_WP] = "DiagramWP";
  SymbolNames[DIAGRAM_BK] = "DiagramBK";
  SymbolNames[DIAGRAM_BQ] = "DiagramBQ";
  SymbolNames[DIAGRAM_BR] = "DiagramBR";
  SymbolNames[DIAGRAM_BB] = "DiagramBB";
  SymbolNames[DIAGRAM_BN] = "DiagramBN";
  SymbolNames[DIAGRAM_BP] = "DiagramBP";
  SymbolNames[DIAGRAM_WHITE_SQUARE] = "DiagramWhiteSquare";
  SymbolNames[DIAGRAM_BLACK_SQUARE] = "DiagramBlackSquare";

  loadChessSymbols(symbolsFile);
}


/*------------------------------------------------------------
 *  Loading
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node loadChessSymbols
@deftypefun void loadChessSymbols (char *@var{fileName})
Reads the chess symbols found in @var{fileName} and defines
their mappings to ASCII.  This function is normally called
by @code{initChessSymbols} or by the @code{-symbols} generic
option.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
loadChessSymbols(char *fileName)
{ char line[MAX_LINE_SIZE+1];
  char *s;
  int n;
  FILE *fd;
  
  fd = fopenCbuffFile(fileName, "r");
  if (fd == NULL)
  { fprintf(stderr, "Fatal: could not open ``%s''\n", fileName);
    exit(1);
  }

  while (fgets(line, MAX_LINE_SIZE, fd))
  { if (!isupper(line[0]))
      continue;			
    s = strchr(line, ':');	
    if (s == NULL)
    { fprintf(stderr, "Warning: no ``:'' seen on line\n\t%s\n", s);
      continue;
    }
    *s++ = '\0';
    for (n=0; n<LAST_SYMBOL; n++)
    { if (SymbolNames[n] && strcmp(line, SymbolNames[n]) == 0)
      { AsciiSymbols[n] = allocCharp(quotedString(s));
	break;
      }
    }
    if (n >= LAST_SYMBOL)
      fprintf(stderr, "Warning: %s not a known chess symbol name\n", line);
  }
  fclose(fd);
}


/*------------------------------------------------------------
 *  Printing
 *------------------------------------------------------------*/

void
printChessSymbols(FILE *fd)
{ int n;

  for (n=0; n<LAST_SYMBOL; n++)
    if (SymbolNames[n])
      fprintf(fd, "%-30s: %s\n", SymbolNames[n], AsciiSymbols[n]);
}


/*------------------------------------------------------------
 *  Principle Functions
 *------------------------------------------------------------*/

char *
chessSymbol(int symbol)
{ if (symbol < LAST_SYMBOL && AsciiSymbols[symbol])
    return AsciiSymbols[symbol];
  setIntError(symbol);
  setError(ERR_UNKNOWN_SYMBOL);
  return "";
}


char *
mapChessSymbol(char c)
{ int value;

  value = ChessSymbol[c & 0177];

  if (value == 0)
  { setCharError(c);
    setError(ERR_UNKNOWN_CHARACTER);
    reportError(stderr);
    return "?";
  }

  return chessSymbol((int) value);
}


char *
stringChessSymbol(char *s, char *t)
{ char *result;

  for (result=t; *s; s++)
  { if (*s >= ' ' && *s < 0177)
    { *result++ = *s;
      continue;
    }
    switch (*s)
    { case 0177:			/* Used be some */
	strcpy(result, chessSymbol(WITH_IDEA));
	result = strchr(result, '\0');
	continue;

      case REQUEST_DIAGRAM:		/* ^D in comment field */
	continue;

      case 035:				/* Frequently appears in Hubner */
	strcpy(result, chessSymbol(UMLAUT_u));
	result = strchr(result, '\0');
	continue;

      case 037:				/* As in Vaxjo */
	strcpy(result, chessSymbol(UMLAUT_o));
	result = strchr(result, '\0');
	continue;

      default:
	strcpy(result, mapChessSymbol(*s));
	result = strchr(result, '\0');
    }
  }
  *result = '\0';
  return t;
}


/*------------------------------------------------------------
 *  Private functions
 *------------------------------------------------------------*/

static char *
quotedString(char *s)
{ char quote;
  char *t, *u;

  while (*s && isspace(*s))
    s++;
  if (*s == '\0')
    return s;
  quote = *s++;
  for (t=s, u=s; *t; t++)
  { if (*t == quote)
      break;
    if (*t == '\\')
    { switch (*++t)
      { case 'n':	*u++ = '\n';  continue;
        case 't':	*u++ = '\t';  continue;
	default:	*u++ = *t;    continue;
      }
    }
    *u++ = *t;
  }
  *u = '\0';

  return s;
}
