/*  $Id$
 *  
 *  File	string.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	String manipulations
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  University of Amsterdam
 *  
 *  History	14/11/93  (Created)
 *  		09/12/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


/*------------------------------------------------------------
 *  Player names
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node normalisePlayer
@deftypefun {char *} normalisePlayer (char *@var{p})
Normalises the name of a chess player.  Normalisation means that all
special characters are replaced by the defined chess symbols
(@code{-symbols}) option, and that a number of heuristics are
applied.  Currently the heuristics are that name prefixes are uppercased
(e.g. @code{"van Wely"} becomes @code{"Van Wely"}, dots and spaces at
the end of a name are removed, and semicolons and underscores are
replaced by spaces.  The normalised name is returned.  This is a statically
allocated string.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

char *
normalisePlayer(char *p)
{ static unsigned char name[MAX_NAME_SIZE+1];
  unsigned char tmp[MAX_NAME_SIZE+1];

  strcpy(tmp, p);

  	/* Strip white space. */
  while (isspace(tmp[0]))
    strcpy(tmp, &tmp[1]);
  while (isspace(tmp[strlen(tmp)-1]))
    tmp[strlen(tmp)-1] = '\0';

  stringChessSymbol(tmp, name);

  if (islower(name[0]))
  {      if (headStringP(name, "van ")) name[0] = 'V';
    else if (headStringP(name, "von ")) name[0] = 'V';
    else if (headStringP(name, "ten ")) name[0] = 'T';
    else if (headStringP(name, "te "))  name[0] = 'T';
    else if (headStringP(name, "di "))  name[0] = 'D';
    else if (headStringP(name, "d' "))  name[0] = 'D';
    else if (headStringP(name, "de "))  name[0] = 'D';
    else if (headStringP(name, "le "))  name[0] = 'L';
    else if (headStringP(name, "den ")) name[0] = 'D';
  }

  if (tailStringP(name, "."))    name[strlen(name)-1] = '\0';
  if (tailStringP(name, ",x"))   name[strlen(name)-2] = '\0';
  if (tailStringP(name, ",#"))   name[strlen(name)-2] = '\0';
  replaceString(name, "_;", " ,");
  chopString(name, "([*");

  return name;
}



static void	loadSpacePlaces(void);


char **		SpacePlaces = NULL;	/* List of places with spaces */

static void
loadSpacePlaces()
{ FILE *fd;
  int count;
  char place[MAX_NAME_SIZE+1];

  if (SpacePlaces != NULL)
    return;

  fd = fopenCbuffFile("places.def", "r");
  if (foundError())
  { reportError(stderr);
    exit(1);
  }

  for (count=0; fgets(place, MAX_NAME_SIZE, fd); count++)
    ;
  fclose(fd);

  SpacePlaces = alloc(sizeof(char *) * (count+1));

  fd = fopenCbuffFile("places.def", "r");
  for (count=0; fgets(place, MAX_NAME_SIZE, fd); count++)
  { stripString(place);
    SpacePlaces[count] = allocCharp(place);
  }
  fclose(fd);

  SpacePlaces[count] = NULL;
}



/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node normalisePlace
@deftypefun {char *} normalisePlace (char *@var{source})
Returns the place (i.e. the city where a game was played) from the
@var{source} field of a game.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

char *
normalisePlace(char *source)
{ static char name[MAX_NAME_SIZE+1];
  char tmp[MAX_NAME_SIZE+1];
  char *s, *t;

  strcpy(tmp, source);

  	/* Strip white space. */
  while (isspace(tmp[0]))
    strcpy(tmp, &tmp[1]);
  while (isspace(tmp[strlen(tmp)-1]))
    tmp[strlen(tmp)-1] = '\0';

  stringChessSymbol(tmp, name);

  replaceString(name, "_;-:", " ,  ");

					/* Bundesliga 2 */
  if (name[0] == '2')
  { if (headStringP(name, "2 BL"))         { return "BRD tt"; }
    if (headStringP(name, "2. BL"))        { return "BRD tt"; }
    if (headStringP(name, "2.BL"))         { return "BRD tt"; }
    if (headStringP(name, "2BL"))          { return "BRD tt"; }
  }

					/* Bundesliga */
  if (name[0] == 'B')
  { if (headStringP(name, "BL "))          { return "BRD tt"; }
    if (headStringP(name, "BL8"))          { return "BRD tt"; }
    if (headStringP(name, "Bundesliga"))   { return "BRD tt"; }
  }
  if (headStringP(name, "Germany Bundesliga")) { return "BRD tt"; }

					/* Wijk aan Zee */
  if (headStringP(name, "Wijk "))        { return "Wijk aan Zee"; }

  if (strchr(name, ' '))
  { loadSpacePlaces();
    if (SpacePlaces)
    { int i;

      for (i=0; SpacePlaces[i]; i++)
	if (headStringP(name, SpacePlaces[i]))
	  return SpacePlaces[i];
    }
  }

  chopString(name, "([*<0123456789 ");

  return name;
}


/*------------------------------------------------------------
 *  String utilities
 *------------------------------------------------------------*/

bool
headStringP(char *s, char *head)
{ if (strncmp(s, head, strlen(head)) == 0)
    return TRUE;
  return FALSE;
}


bool
tailStringP(char *s, char *tail)
{ if (strlen(s) >= strlen(tail))
    if (strcmp(s+strlen(s)-strlen(tail), tail) == 0)
      return TRUE;
  return FALSE;
}


void
replaceString(char *s, char *in, char *out)
{ char *t, *u, *v;

  for (t=s; *t; t++)
  { for (u=in,v=out; *u; u++,v++)
      if (*t == *u)
	*t = *v;
  }
}


void
chopString(char *s, char *specials)
{ char *t, *u;

  for (t=s; *t; t++)
  { for (u=specials; *u; u++)
    { if (*t == *u)
      { *t = '\0';
	stripString(s);
	return;
      }
    }
  }
}


void
stripString(char *s)
{ char *t;

  for (t=s; *t; t++)
    if (!isspace(*t))
      break;
  if (t != s)
    strcpy(s, t);
  t = s + strlen(s) - 1;
  while (isspace(*t) && t >= s)
    *t-- = '\0';
}
