/*  $Id$
 *  
 *  File	stats.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Statistics utility
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	15/10/93  (Created)
 *  		30/11/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"

char *		UTILITY_NAME = "Statistics utility";
char *		UTILITY_VERSION = "1.2.0";

void		helpUtility(FILE *fd);

static void	statisticsUtility(Option);
static void	statisticsGame(Game);

static void	resetStatistics(void);
static void	reportStatistics(FILE *);


/*------------------------------------------------------------
 *  Definitions
 *------------------------------------------------------------*/

long		NoGames;		/* Total number of games */
long		NoWhiteWins;		/* Games won by White */
long		NoBlackWins;		/* Games won by Black */
long		NoDraws;		/* Games drawn */
long		NoLines;		/* Games with no known result */

long		NoYear[2100];		/* Games per year */


/*------------------------------------------------------------
 *  Statistics utility
 *------------------------------------------------------------*/

static void
statisticsUtility(Option opt)
{ Game g = newGame();
  CBase cb = opt->database;
  long from = opt->from;
  long to = (opt->to < getNoGamesCBase(cb) ? opt->to : getNoGamesCBase(cb));
  long n;

  reportCBase(cb, stderr);

  resetStatistics();
  
  for (n=from; n<=to; n++)
  { environmentError(cb, g, n);
    initialiseGame(g, n, cb);
    if (foundError())
    { reportError(stderr);
      continue;
    }
    statisticsGame(g);
    if (foundError())
      reportError(stderr);
  }

  reportStatistics(opt->output);

  freeGame(g);
}


/*------------------------------------------------------------
 *  Statistics of a single game
 *------------------------------------------------------------*/

static void
statisticsGame(Game g)
{ NoGames++;

  switch (getResultGame(g))
  { case WHITE_WINS:	NoWhiteWins++; break;
    case DRAW:		NoDraws++; break;
    case BLACK_WINS:	NoBlackWins++; break;
    default:		NoLines++;
  }
  NoYear[getYearGame(g)]++;
}


/*------------------------------------------------------------
 *  Functions
 *------------------------------------------------------------*/

static float	pct(long, long);


static float
pct(long n, long m)
{ return 100.0 * ((float) n / (float) m);
}


static void
resetStatistics()
{ NoWhiteWins = 0;
  NoBlackWins = 0;
  NoDraws = 0;
  NoLines = 0;
  NoGames = 0;
}


static void
reportStatistics(FILE *fd)
{ int year;

  fprintf(fd, "\n");
  fprintf(fd, "Total games     %6ld\n", NoGames);
  fprintf(fd, "======================\n");
  fprintf(fd, "Won by White    %6ld  (%4.1f%%)\n",
	  NoWhiteWins, pct(NoWhiteWins,NoGames));
  fprintf(fd, "Won by Black    %6ld  (%4.1f%%)\n",
	  NoBlackWins, pct(NoBlackWins,NoGames));
  fprintf(fd, "Drawn           %6ld  (%4.1f%%)\n",
	  NoDraws, pct(NoDraws, NoGames));
  fprintf(fd, "Lines           %6ld  (%4.1f%%)\n",
	  NoLines, pct(NoLines, NoGames));

  fprintf(fd, "\n");
  fprintf(fd, "Distribution by year\n");
  fprintf(fd, "====================\n");
  if (NoYear[0] == 0)
    fprintf(fd, "Unknown %6ld  (%4.1f%%)\n",
	    NoYear[0], pct(NoYear[0], NoGames));
  for (year=1; year<2100; year++)
    if (NoYear[year])
      fprintf(fd, "%4d     %6ld  (%4.1f%%)\n",
	      year, NoYear[year], pct(NoYear[year], NoGames));
}


/*------------------------------------------------------------
 *  Main
 *------------------------------------------------------------*/

int
main(int argc, char *argv[])
{ int i;
  Option options = newOption();

  initChessSymbols(NULL);

  for (i=1; i<argc; i++)
  {
    if (strhead(argv[i], "-"))
    { int n;

      n = genericOption(options, argv, argc, i);
      if (n == 0)
      { fprintf(stderr, "Fatal: Unknown command %s\n", argv[i]);
	fprintf(stderr, "Do ``%s -help'' or see the documentation\n", argv[0]);
	exit(1);
      }
      i = n;
      continue;
    }

    setCurrentCBase(argv[i], "-database", argc, i);
    options->database = CurrentBase;
    statisticsUtility(options);
    freeCBase(options->database);
    options->database = (CBase) NULL;
  }

  if (options->database)
  { statisticsUtility(options);
    freeCBase(options->database);
  }

  exit(0);
}


/*------------------------------------------------------------
 *  Help
 *------------------------------------------------------------*/

void
helpUtility(FILE *fd)
{ helpCBUFF(fd);
  fprintf(fd, "%s options:\n", UTILITY_NAME);
}
