/*  $Id$
 *  
 *  File	pgn.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Portable Game Notation (PGN) utility
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	15/10/93  (Created)
 *  		05/12/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"

char *		UTILITY_NAME = "PGN utility";
char *		UTILITY_VERSION = "1.2.0";

void		helpUtility(FILE *fd);

void		pgnGame(Game, FILE *, int);


bool		PGN_COMPLETE = FALSE;
char *		PGN_EVENT = (char *) NULL;
char *		PGN_PLACE = (char *) NULL;
char *		PGN_DATE = (char *) NULL;


/*------------------------------------------------------------
 *  PGN utility
 *------------------------------------------------------------*/

void
pgnUtility(Option opt)
{ Game g = newGame();
  CBase cb = opt->database;
  long from = opt->from;
  long to = (opt->to < getNoGamesCBase(cb) ? opt->to : getNoGamesCBase(cb));
  long n;

  reportCBase(cb, stderr);

  n = to-from+1;
  if (n < 0)
    return;
  
  for (n=from; n<=to; n++)
  { environmentError(cb, g, n);
    initialiseGame(g, n, cb);
    if (foundError())
    { reportError(stderr);
      continue;
    }
    pgnGame(g, opt->output, opt->notation);
    if (foundError())
      reportError(stderr);
  }

  freeGame(g);
}


/*------------------------------------------------------------
 *  Portable Game Notation
 *------------------------------------------------------------*/

void
pgnGame(Game g, FILE *fd, int notation)
{ TextBuffer tb;
  Position pos;

  if (!fullGameP(g))
  { fprintf(stderr, "Don't know how to print partial game (%ld) in PGN\n",
	    g->number);
    return;
  }

  tb = newTextBuffer(fd, 75);

  formatTextBuffer(tb, "[Event \"%s\"]\n",
		   (PGN_EVENT ? PGN_EVENT : "?"));
  formatTextBuffer(tb, "[Site \"%s\"]\n",
		   (PGN_PLACE ? PGN_PLACE
		              : getPlaceGame(g)));

  if (PGN_DATE)
    formatTextBuffer(tb, "[Date \"%s\"]\n", PGN_DATE);
  else
  { if (getYearGame(g))
      formatTextBuffer(tb, "[Date \"%d.??.??\"]\n", getYearGame(g));
    else
      stringTextBuffer(tb, "[Date \"????.??.??\"]\n");
  }

  if (getRoundGame(g))
    formatTextBuffer(tb, "[Round \"%d\"]\n", getRoundGame(g));
  else
    stringTextBuffer(tb, "[Round \"?\"]\n");

  if (getWhiteGame(g))
    formatTextBuffer(tb, "[White \"%s\"]\n", getWhiteGame(g));
  else
    stringTextBuffer(tb, "[White \"?\"]\n");

  if (getBlackGame(g))
    formatTextBuffer(tb, "[Black \"%s\"]\n", getBlackGame(g));
  else
    stringTextBuffer(tb, "[Black \"?\"]\n");

  if (getFinalResultGame(g) != NO_RESULT)
    formatTextBuffer(tb, "[Result \"%s\"]\n",
		     chessSymbol(getFinalResultGame(g)));
  else
    stringTextBuffer(tb, "[Result \"*\"]\n");

  if (PGN_COMPLETE)
  { if (getEloWhiteGame(g))
      formatTextBuffer(tb, "[WhiteElo \"%4d\"]\n", getEloWhiteGame(g));
    if (getEloBlackGame(g))
      formatTextBuffer(tb, "[BlackElo \"%4d\"]\n", getEloBlackGame(g));
    if (getEcoGame(g))
      formatTextBuffer(tb, "[ECO \"%s\"]\n", getEcoGame(g));
  }

  stringTextBuffer(tb, "\n");

  pos = newPosition();

  pgnMovesGame(g, getFirstMoveGame(g), pos, 0, notation, tb);

  if (getFinalResultGame(g) != NO_RESULT)
    stringSpaceTextBuffer(tb, chessSymbol(getResultGame(g)));
  else
    stringSpaceTextBuffer(tb, "*");

  stringNewlineTextBuffer(tb, "\n\n");
  freeTextBuffer(tb);
}


/*------------------------------------------------------------
 *  Main
 *------------------------------------------------------------*/

int
main(int argc, char *argv[])
{ int i;
  Option options = newOption();

  initChessSymbols(NULL);
  loadChessSymbols("pgn.sym");

  for (i=1; i<argc; i++)
  {
    if (strhead(argv[i], "-complete"))
    { PGN_COMPLETE = TRUE;
      continue;
    }

    if (strhead(argv[i], "-date"))
    { i++;
      PGN_DATE = stringArgument(argv[i], "-date", argc, i);
      continue;
    }

    if (strhead(argv[i], "-event"))
    { i++;
      PGN_EVENT = stringArgument(argv[i], "-event", argc, i);
      continue;
    }

    if (strhead(argv[i], "-place"))
    { i++;
      PGN_PLACE = stringArgument(argv[i], "-place", argc, i);
      continue;
    }

    if (strhead(argv[i], "-"))
    { int n;

      n = genericOption(options, argv, argc, i);
      if (n == 0)
      { fprintf(stderr, "Fatal: Unknown command %s\n", argv[i]);
	fprintf(stderr, "Do ``%s -help'' or see the documentation\n", argv[0]);
	exit(1);
      }
      i = n;
      continue;
    }

    setCurrentCBase(argv[i], "-database", argc, i);
    options->database = CurrentBase;
    pgnUtility(options);
    freeCBase(options->database);
    options->database = (CBase) NULL;
  }

  if (options->database)
  { pgnUtility(options);
    freeCBase(options->database);
  }

  exit(0);
}


/*------------------------------------------------------------
 *  Help
 *------------------------------------------------------------*/

void
helpUtility(FILE *fd)
{ helpCBUFF(fd);
  fprintf(fd, "%s options:\n", UTILITY_NAME);
  fprintf(fd, "-complete     Generate as many of the PGN fields as possible\n");
  fprintf(fd, "-date s       String s is the date for all games\n");
  fprintf(fd, "-event s      String s is the event for all games\n");
  fprintf(fd, "-place s      String s is the place for all games\n");
}
