/*  $Id$
 *  
 *  File	list.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	List utility
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	22/09/93  (Created)
 *  		30/11/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


char *		UTILITY_NAME = "List utility";
char *		UTILITY_VERSION = "1.2.0";

void		helpUtility(FILE *fd);

static void	setDefaultListUtility(char *name, int value);
static void	defaultListUtility(FILE *fd);


/*------------------------------------------------------------
 *  Formats
 *------------------------------------------------------------*/

struct field
{ char *	name;			/* Name of the field */
  int		width;			/* Width in characters */
  char *	format;			/* Style to print */
  int		padding;		/* Number of padding characters */
  bool		print;			/* Print or not? */
} ListFields[] =
{ { "number",	   6, "%%%dld",       1, TRUE },
  { "players",    25, "%%-%d.%ds",    1, TRUE },
  { "white",      15, "%%-%d.%ds",    1, FALSE },
  { "elowhite",    4, "%%-%d.%dd",    1, FALSE },
  { "black",      15, "%%-%d.%ds",    1, FALSE },
  { "eloblack",    4, "%%-%d.%dd",    1, FALSE },
  { "source",     25, "%%-%d.%ds",    1, TRUE },
  { "place",      15, "%%-%d.%ds",    1, FALSE },
  { "event",      15, "%%-%d.%ds",    1, FALSE },
  { "round",       2, "%%%dd",        1, FALSE },
  { "annotator",  15, "%%-%d.%ds",    1, FALSE },
  { "eco",         6, "%%-%d.%ds",    1, FALSE },
  { "ecomajor",    3, "%%-%d.%ds",    1, FALSE },
  { "year",        4, "%%-%d.%dd",    1, TRUE },
  { "result",      4, "%%-%d.%ds",    1, TRUE },
  { "moves",       3, "%%%dd",        1, TRUE },
  { "deleted",     1, "%%c",          0, TRUE },
  { "marked",      1, "%%c",          0, TRUE },
  { "position",    1, "%%c",          0, TRUE },
  { "comments",    1, "%%c",          0, TRUE },
  { "variations",  1, "%%c",          0, TRUE },
  { "flags2",      1, "%%c",          0, TRUE },
  { (char *) NULL, 0, (char *) NULL,  0, 0 }
};


/*------------------------------------------------------------
 *  List utility
 *------------------------------------------------------------*/

void
listUtility(Option opt)
{ Game g = newGame();
  CBase cb = opt->database;
  FILE *fd = opt->output;
  long from = opt->from;
  long to = (opt->to < getNoGamesCBase(cb) ? opt->to : getNoGamesCBase(cb));
  long n;

  reportCBase(cb, stderr);

  n = to-from+1;
  if (n < 0)
    return;
  
  for (n=from; n<=to; n++)
  { environmentError(cb, g, n);
    initialiseGame(g, n, cb);
    if (foundError())
    { reportError(stderr);
      continue;
    }
    if (checkHeaderGame(g) == FALSE)
    { reportError(stderr);
      continue;
    }
    listGame(g, fd);
    if (foundError())
      reportError(stderr);
  }

  freeGame(g);
}


/*------------------------------------------------------------
 *  Listing game header
 *------------------------------------------------------------*/

void
listGame(Game g, FILE *fd)
{ int i;
  struct field *f;
  char format[MAX_NAME_SIZE+1];
  char entry[MAX_NAME_SIZE+1];

  for (f=ListFields; f->name; f++)
  { if (f->print == FALSE)
      continue;
    sprintf(format, f->format, f->width, f->width);
    if (strcmp(f->name, "number") == 0)
      fprintf(fd, format, getNumberGame(g)); else
    if (strcmp(f->name, "players") == 0)
      fprintf(fd, format, stringChessSymbol(getPlayersGame(g),entry)); else
    if (strcmp(f->name, "white") == 0)
      fprintf(fd, format, getWhiteGame(g)); else
    if (strcmp(f->name, "elowhite") == 0)
      fprintf(fd, format, getEloWhiteGame(g)); else
    if (strcmp(f->name, "black") == 0)
      fprintf(fd, format, getBlackGame(g)); else
    if (strcmp(f->name, "eloblack") == 0)
      fprintf(fd, format, getEloBlackGame(g)); else
    if (strcmp(f->name, "source") == 0)
      fprintf(fd, format, stringChessSymbol(getSourceGame(g),entry)); else
    if (strcmp(f->name, "place") == 0)
      fprintf(fd, format, getPlaceGame(g)); else
    if (strcmp(f->name, "round") == 0)
      fprintf(fd, format, getRoundGame(g)); else
    if (strcmp(f->name, "event") == 0)
      fprintf(fd, format, getEventGame(g)); else
    if (strcmp(f->name, "annotator") == 0)
      fprintf(fd, format, getAnnotatorGame(g)); else
    if (strcmp(f->name, "eco") == 0)
      fprintf(fd, format, (getEcoGame(g) || "")); else
    if (strcmp(f->name, "ecomajor") == 0)
      fprintf(fd, format, (getEcoMajorGame(g) || "")); else
    if (strcmp(f->name, "year") == 0)
      fprintf(fd, format, getYearGame(g)); else
    if (strcmp(f->name, "result") == 0)
    { Result r;

      r = getResultGame(g);
      fprintf(fd, format, (r == NO_RESULT ? "No" : chessSymbol(r)));
    } else
    if (strcmp(f->name, "moves") == 0)
      fprintf(fd, format, getMovesGame(g)); else
    if (strcmp(f->name, "deleted") == 0)
      fprintf(fd, format, deletedGameP(g) ? 'D' : ' '); else
    if (strcmp(f->name, "marked") == 0)
      fprintf(fd, format, markedGameP(g) ? 'M' : ' '); else
    if (strcmp(f->name, "position") == 0)
      fprintf(fd, format, fullGameP(g) ? ' ' : 'P'); else
    if (strcmp(f->name, "comments") == 0)
      fprintf(fd, format, containsCommentsGameP(g) ? 'C' : ' '); else
    if (strcmp(f->name, "variations") == 0)
      fprintf(fd, format, containsVariationsGameP(g) ? 'V' : ' '); else
    if (strcmp(f->name, "flags2") == 0)
      fprintf(fd, format, flags2Bit6GameP(g) ? '1' : '0'); else
    { fprintf(stderr, "Internal error: Field named %s (listGame)\n", f->name);
      exit(1);
    }
    for (i=0; i<f->padding; i++)
      fprintf(fd, " ");
  }
  fprintf(fd, "\n");
}


/*------------------------------------------------------------
 *  Main
 *------------------------------------------------------------*/

int
main(int argc, char *argv[])
{ int i;
  Option options = newOption();

  initChessSymbols(NULL);

  for (i=1; i<argc; i++)
  {
    if (strhead(argv[i], "-annotator"))	/* -annotator */
    { ++i;
      setDefaultListUtility("annotator", intArgument(argv[i],"-annotator",argc,i));
      continue;
    }

    if (strhead(argv[i], "-b"))		/* -black */
    { ++i;
      setDefaultListUtility("black", intArgument(argv[i],"-black",argc,i));
      continue;
    }

    if (strhead(argv[i], "-comments"))	/* -comments */
    { ++i;
      setDefaultListUtility("comments", intArgument(argv[i],"-comments",argc,i));
      continue;
    }

    if (strhead(argv[i], "-def"))	/* -defaults */
    { FILE *fd;

      ++i;
      fd = fileArgument(argv[i], "-defaults", argc, i, "r");
      defaultListUtility(fd);
      fclose(fd);
      continue;
    }

    if (strhead(argv[i], "-del"))	/* -deleted */
    { ++i;
      setDefaultListUtility("deleted", intArgument(argv[i],"-deleted",argc,i));
      continue;
    }

    if (strhead(argv[i], "-eco"))	/* -eco */
    { ++i;
      setDefaultListUtility("eco", intArgument(argv[i],"-eco",argc,i));
      continue;
    }

    if (strhead(argv[i], "-ecom"))	/* -ecomajor */
    { ++i;
      setDefaultListUtility("ecomajor", intArgument(argv[i],"-ecomajor",argc,i));
      continue;
    }

    if (strhead(argv[i], "-elob"))	/* -eloblack */
    { ++i;
      setDefaultListUtility("eloblack", intArgument(argv[i],"-eloblack",argc,i));
      continue;
    }

    if (strhead(argv[i], "-elow"))	/* -elowhite */
    { ++i;
      setDefaultListUtility("elowhite", intArgument(argv[i],"-elowhite",argc,i));
      continue;
    }

    if (strhead(argv[i], "-ev"))	/* -event */
    { ++i;
      setDefaultListUtility("event", intArgument(argv[i],"-event",argc,i));
      continue;
    }

    if (strhead(argv[i], "-fl"))	/* -flags2 */
    { ++i;
      setDefaultListUtility("flags2", intArgument(argv[i],"-flags2",argc,i));
      continue;
    }

    if (strhead(argv[i], "-ma"))	/* -marked */
    { ++i;
      setDefaultListUtility("marked", intArgument(argv[i],"-marked",argc,i));
      continue;
    }

    if (strhead(argv[i], "-mo"))	/* -moves */
    { ++i;
      setDefaultListUtility("moves", intArgument(argv[i],"-moves",argc,i));
      continue;
    }

    if (strhead(argv[i], "-n"))		/* -number */
    { ++i;
      setDefaultListUtility("number", intArgument(argv[i],"-number",argc,i));
      continue;
    }

    if (strhead(argv[i], "-o"))		/* -output */
    { ++i;
      if (options->output != stdout)
      	fclose(options->output);
      options->output = fileArgument(argv[i], "-output", argc, i, "w");
      continue;
    }

    if (strhead(argv[i], "-plac"))	/* -place */
    { ++i;
      setDefaultListUtility("place", intArgument(argv[i],"-place",argc,i));
      continue;
    }

    if (strhead(argv[i], "-play"))	/* -players */
    { ++i;
      setDefaultListUtility("players", intArgument(argv[i],"-players",argc,i));
      continue;
    }

    if (strhead(argv[i], "-po"))	/* -position */
    { ++i;
      setDefaultListUtility("position", intArgument(argv[i],"-position",argc,i));
      continue;
    }

    if (strhead(argv[i], "-re"))	/* -result */
    { ++i;
      setDefaultListUtility("result", intArgument(argv[i],"-result",argc,i));
      continue;
    }

    if (strhead(argv[i], "-ro"))	/* -round */
    { ++i;
      setDefaultListUtility("round", intArgument(argv[i],"-round",argc,i));
      continue;
    }

    if (strhead(argv[i], "-so"))	/* -source */
    { ++i;
      setDefaultListUtility("source", intArgument(argv[i],"-source",argc,i));
      continue;
    }

    if (strhead(argv[i], "-va"))	/* -variations */
    { ++i;
      setDefaultListUtility("variations", intArgument(argv[i],"-variations",argc,i));
      continue;
    }

    if (strhead(argv[i], "-w"))		/* -white */
    { ++i;
      setDefaultListUtility("white", intArgument(argv[i],"-white",argc,i));
      continue;
    }

    if (strhead(argv[i], "-y"))		/* -year */
    { ++i;
      setDefaultListUtility("year", intArgument(argv[i],"-year",argc,i));
      continue;
    }

    if (strhead(argv[i], "-"))
    { int n;

      n = genericOption(options, argv, argc, i);
      if (n == 0)
      { fprintf(stderr, "Fatal: Unknown command %s\n", argv[i]);
	fprintf(stderr, "Do ``%s -help'' or see the documentation\n", argv[0]);
	exit(1);
      }
      i = n;
      continue;
    }

    setCurrentCBase(argv[i], "-database", argc, i);
    options->database = CurrentBase;
    listUtility(options);
    freeCBase(options->database);
    options->database = (CBase) NULL;
  }

  if (options->database)
  { listUtility(options);
    freeCBase(options->database);
  }

  exit(0);
}


/*------------------------------------------------------------
 *  Defaults
 *------------------------------------------------------------*/

static void
setDefaultListUtility(char *name, int value)
{ struct field *f;

  if (strcmp(name, "*") == 0)
  { for (f=ListFields; f->name; f++)
    { if (value)
      { f->width = value;
	f->print = TRUE;
      } else
	f->print = FALSE;
    }
    return;
  }

  for (f=ListFields; f->name; f++)
  { if (strcmp(f->name, name) == 0)
    { if (value)
      { f->width = value;
	f->print = TRUE;
      } else
	f->print = FALSE;
      return;
    }
  }
  fprintf(stderr, "Warning: No default called list.%s\n", name);
}


static void
defaultListUtility(FILE *fd)
{ char line[MAX_LINE_SIZE+1];

  while (fgets(line, MAX_LINE_SIZE, fd))
  { if (strhead(line, "list."))
    { char name[MAX_NAME_SIZE+1];
      int value;

      sscanf(line, "list.%s %d\n", name, &value);
      setDefaultListUtility(name, value);
    }
  }
}


/*------------------------------------------------------------
 *  Help
 *------------------------------------------------------------*/

void
helpUtility(FILE *fd)
{ helpCBUFF(fd);
  fprintf(fd, "List utility options (argument 0 turns off a field)\n");
  fprintf(fd, "-annotator n  Annotator of the game (15; not printed)\n");
  fprintf(fd, "-black n      Name of Black (15; not printed)\n");
  fprintf(fd, "-comments n   If game contains comments (1; prints 'C')\n");
  fprintf(fd, "-deleted n    If game is deleted (1; prints 'D')\n");
  fprintf(fd, "-eco n        ECO code for the game (6; not printed)\n");
  fprintf(fd, "-ecomajor n   ECO major (3; not printed)\n");
  fprintf(fd, "-eloblack n   Rating of Black (4; not printed)\n");
  fprintf(fd, "-elowhite n   Rating of White (4; not printed)\n");
  fprintf(fd, "-event n      Event in which game was played (not implemented)\n");
  fprintf(fd, "-flags2 n     If game contains flag (1; prints '1' or '0')\n");
  fprintf(fd, "-marked n     If game is marked (1; prints 'M')\n");
  fprintf(fd, "-moves n      Number of moves in game (3 default)\n");
  fprintf(fd, "-number n     Game number (6 default)\n");
  fprintf(fd, "-place n      Place where game was played (not implemented)\n");
  fprintf(fd, "-players n    Player names (25 default)\n");
  fprintf(fd, "-position n   If game starts from a position (1; prints 'P')\n");
  fprintf(fd, "-result n     Result of the game (4 default)\n");
  fprintf(fd, "-round n      Round games was played (2; not printed)\n");
  fprintf(fd, "-source n     Source of the game (25 default)\n");
  fprintf(fd, "-variations n If game contains variations (1; prints 'V')\n");
  fprintf(fd, "-white n      Name of White (15; not printed)\n");
  fprintf(fd, "-year n       Year game was played (4 default)\n");
}
