/*  $Id$
 *  
 *  File	cbase.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  		Horst Aurisch, aurisch@informatik.uni-bonn.de
 *  Purpose	Manipulation of entire ChessBase databases
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	08/06/93  (Created)
 *  		24/11/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


/*------------------------------------------------------------
 *  Prototypes
 *------------------------------------------------------------*/

static CBase	allocCBase(char *);
static void	checkModeCBase(CBase, int);
static long	checkBoundsCBase(CBase, long);


/*------------------------------------------------------------
 *  Initialisation
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node newCBase
@deftypefun CBase newCBase (char *@var{name}, char *@var{mode})
Allocates a new database and returns it.  @var{name} is the name of 
the database (without the @file{.cbf} extension).  The @var{mode}
argument is similar to the second of @code{fopen}.  @var{mode} can be:
@example
"r"   @r{Open for reading}
"a"   @r{Open for appending}
"c"   @r{Open for writing, create first}
@end example
The @code{"c"} mode requires that a database of the given @var{name}
does not exist, @code{"a"} can be used for both existing and new databases.

The constant @code{NULL} is returned when the database cannot be opened or
on another error.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

CBase
newCBase(char *baseName, char *mode)
{ FILE *cbi;
  FILE *cbf;
  CBase cb;

  if (strcmp(mode, "r") == 0)
  { cbi = fopenExtension(baseName, "cbi", "rb", TRUE);
    cbf = fopenExtension(baseName, "cbf", "rb", TRUE);
    if (foundError())
    { reportError(stderr);
      if (cbi) fclose(cbi);
      if (cbf) fclose(cbf);
      return NULL;
    }
    cb = allocCBase(baseName);
    cb->cbi = cbi;
    cb->cbf = cbf;
    cb->mode = READ_MODE;
  } else
  if (strcmp(mode, "c") == 0)
  { cbi = fopenExtension(baseName, "cbi", "wb", FALSE);
    cbf = fopenExtension(baseName, "cbf", "wb", FALSE);
    if (foundError())
    { reportError(stderr);
      if (cbi) fclose(cbi);
      if (cbf) fclose(cbf);
      return NULL;
    }
    cb = allocCBase(baseName);
    cb->cbi = cbi;
    cb->cbf = cbf;
    cb->mode = WRITE_MODE;
  } else
  if (strcmp(mode, "a") == 0)
  { cbi = fopenExtension(baseName, "cbi", "r+b", TRUE);
    cbf = fopenExtension(baseName, "cbf", "r+b", TRUE);
    if (foundError())
    { reportError(stderr);
      if (cbi) fclose(cbi);
      if (cbf) fclose(cbf);
      return NULL;
    }
    cb = allocCBase(baseName);
    cb->cbi = cbi;
    cb->cbf = cbf;
    cb->mode = WRITE_MODE;
    seekIndexFileCBase(cb, 0L);
    cb->noGames = readLongIndexFileCBase(cb) - 1;
    seekIndexFileCBase(cb, (cb->noGames+1)*sizeof(unsigned long));
    cb->position = readLongIndexFileCBase(cb) - cb->noGames - 1;
  } else
  { fprintf(stderr, "Internal error: Opening database %s; mode (%s)?\n",
	    baseName, mode);
    exit(1);
  }

  if (cb->mode == READ_MODE)
  { long i;
    unsigned long pos;

    cb->noGames = readLongIndexFileCBase(cb) - 1;
    cb->index = (unsigned long *) alloc(cb->noGames*sizeof(unsigned long));
    if (cb->index == NULL)
    { fprintf(stderr, "Could not allocate index for %s.cbi\n", baseName);
      freeCBase(cb);
      return NULL;
    }
    for (i=1; i<=cb->noGames; i++)
    { pos = readLong(cbi);
      cb->index[i-1] = pos - (i+1);
    }
    cb->position = readLong(cbi) - cb->noGames - 1;
  }

  return cb;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node freeCBase
@deftypefun void freeCBase (CBase @var{cb})
Reclaims the memory associated with the database @var{cb}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
freeCBase(CBase cb)
{ if (cb->cbf) fclose(cb->cbf);
  if (cb->cbi) fclose(cb->cbi);
  if (cb->index) unalloc(cb->index);
  unallocCharp(cb->name);
  unalloc(cb);
}


/*------------------------------------------------------------
 *  Functions
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node getNoGamesCBase
@deftypefun long getNoGamesCBase (CBase @var{cb})
Returns the number of games in the database @var{cb}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

long
getNoGamesCBase(CBase cb)
{ return cb->noGames;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node exportGameCBase
@deftypefun void exportGameCBase (CBase @var{dst}, CBase @var{src}, long @var{n})
Exports (appends) game @var{n} from database @var{src} to the output database @var{dst}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
exportGameCBase(CBase dst, CBase src, long n)
{ CbGame cg;
  char *buf;
  long bytes;

  checkModeCBase(dst, WRITE_MODE);
  checkModeCBase(src, READ_MODE);

  if (src->index[n-1] & PHYSICALLY_DELETED)
    return;

  cg = newCbGame();
  if ((bytes=readCbGame(cg,src,n)) == (long) NULL)
  { reportError(stderr);
    return;
  }
  buf = (char *) alloc(bytes);
					/* Read game from source */
  seekGameFileCBase(src, src->index[n-1]);
  readGameFileCBase(src, buf, bytes);
					/* Write number of games */
  seekIndexFileCBase(dst, 0L);
  writeLongIndexFileCBase(dst, dst->noGames+1+1);
					/* Write index in destination */
  seekIndexFileCBase(dst, (dst->noGames+1) * sizeof(long));
  writeLongIndexFileCBase(dst, dst->position+dst->noGames+1+1);
					/* Write game in destination */
  seekGameFileCBase(dst, dst->position);
  writeGameFileCBase(dst, buf, bytes);
  dst->position += bytes;
  dst->noGames++;
					/* Write last position */
  writeLongIndexFileCBase(dst, dst->position+dst->noGames+1);
  freeCbGame(cg);
  unalloc(buf);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node exportManyCBase
@deftypefun void exportManyCBase (CBase @var{dst}, CBase @var{src}, long @var{from}, long @var{to})
Exports (appends) games from @var{from} through @var{to} from database
@var{src} to the output database @var{dst}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
exportManyCBase(CBase dst, CBase src, long from, long to)
{ CbGame cg;
  char *buf;
  long bytes;
  long n;
  long count, inc;

  checkModeCBase(dst, WRITE_MODE);
  checkModeCBase(src, READ_MODE);

  from = checkBoundsCBase(src, from);
  to = checkBoundsCBase(src, to);

  cg = newCbGame();

  fprintf(stderr, "Writing games %ld - %ld to database %s\n",
	  from, to, dst->name);

  n = to-from+1;
  inc = (n < 50L ? 1 : n/50L);
  
  seekIndexFileCBase(dst, (dst->noGames+1) * sizeof(long));

  for (n=from, count=0; n && n<=to; n++, count++)
  { if (count == inc)
    { fprintf(stderr, ".");
      count = 0;
    }
    if (src->index[n-1] & PHYSICALLY_DELETED)
      continue;
    if ((bytes=loadDataCbGame(cg,src,n,&buf)) == (long) NULL)
    { reportError(stderr);
      fprintf(stderr, "; Game not written to database\n");
      continue;
    }
					/* Write index in destination */
    writeLongIndexFileCBase(dst, dst->position+dst->noGames+1+1);
					/* Write game in destination */
    seekGameFileCBase(dst, dst->position);
    writeGameFileCBase(dst, buf, bytes);
    dst->position += bytes;
    dst->noGames++;
  }
					/* Write last position, assumes
					 * file pointer is correct.
					 */
  writeLongIndexFileCBase(dst, dst->position+dst->noGames+1);
					/* Write number of games */
  seekIndexFileCBase(dst, 0L);
  writeLongIndexFileCBase(dst, dst->noGames+1);
  freeCbGame(cg);

  fprintf(stderr, "\n");
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node reportCBase
@deftypefun void reportCBase (CBase @var{cb}, FILE *@var{fd})
Prints the name of the database @var{cb} on stream @var{fd}.
This is useful for informing which database the utility is
operating on.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
reportCBase(CBase cb, FILE *fd)
{ fprintf(fd, "ChessBase database %s (%ld games)\n",
	  cb->name, cb->noGames);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node deleteGameCBase
@deftypefun void deleteGameCBase (CBase @var{cb}, long @var{n})
Marks game @var{n} in the database @var{cb} for deletion.  The
game will be physically deleted when the game from @var{cb}
is exported to another database (for example with
@code{exportedManyCBase}).
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
deleteGameCBase(CBase cb, long n)
{ checkModeCBase(cb, READ_MODE);
  if (n >= 1 && n <= cb->noGames)
    cb->index[n-1] |= PHYSICALLY_DELETED;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node deletedGameCBaseP
@deftypefun bool deletedGameCBaseP (CBase @var{cb}, long @var{n})
Succeeds if game @var{n} in database @var{cb} has been marked for
deletion with @code{deleteGameCBase}.  Note that this is different
from @code{deleteGameP} (which indicates that the user has marked
the game for deletion).
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
deletedGameCBaseP(CBase cb, long n)
{ checkModeCBase(cb, READ_MODE);
  if (n >= 1 && n <= cb->noGames)
  { if (cb->index[n-1] & PHYSICALLY_DELETED)
      return TRUE;
  }
  return FALSE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node getIndexGameCBase
@deftypefun {unsigned long} getIndexGameCBase (CBase @var{cb}, long @var{n})
Returns the index of game @var{n} in database @var{cb}.  The index
is the position where the game starts in the @code{.cbf} file.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

unsigned long
getIndexGameCBase(CBase cb, long n)
{ checkModeCBase(cb, READ_MODE);
  if (n >= 1 && n <= cb->noGames)
    return cb->index[n-1] & ~PHYSICALLY_DELETED;
  setError(ERR_GAME_NUMBER_OUT_OF_RANGE);
  return (unsigned long) NULL;
}


/*------------------------------------------------------------
 *  Seeking, reading and writing
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node seekGameCBase
@deftypefun {unsigned long} seekGameCBase (CBase @var{cb}, long @var{n})
Positions the seek pointer of the given database @var{cb} such
that game number @var{n} can be read next.  The absolute index
to the start of the game is returned.  @code{NULL} is returned
when the game number is out of range or the game is marked
for physical deletion.  Note that @code{NULL} is also a valid
index position.
This function is mostly used internally.  
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

unsigned long
seekGameCBase(CBase cb, long n)
{ checkModeCBase(cb, READ_MODE);
  if (n >= 1 && n <= cb->noGames)
  { unsigned long pos;

    pos = cb->index[n-1] & ~PHYSICALLY_DELETED;
    seekGameFileCBase(cb, pos);
    return pos;
  }
  setError(ERR_GAME_NUMBER_OUT_OF_RANGE);
  return (unsigned long) NULL;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node seekIndexFileCBase
@deftypefun bool seekIndexFileCBase (CBase @var{cb}, long @var{n})
Seeks into the index file of database @var{cb} such that the
seek pointer is positioned at @var{n}.  This functions avoids
redundant seeks (which are very time consuming).  Note that
@code{writeIndexFileCBase} and @code{readIndexFileCBase} must be
used for reading and writing the index file.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
seekIndexFileCBase(CBase cb, long n)
{ if (cb->ptrCBI == n)
    return TRUE;
  if (fseek(cb->cbi, n, SEEK_SET) == -1)
    return FALSE;
  return TRUE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node writeIndexFileCBase
@deftypefun bool writeIndexFileCBase (CBase @var{cb}, void *@var{ptr}, size_t @var{size})
Writes a buffer starting at @var{ptr} of @var{size} bytes to the
index file of database @var{cb}.  This function must be used for
writing to a database.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
writeIndexFileCBase(CBase cb, void *ptr, size_t size)
{ if (size != fwrite(ptr, 1, size, cb->cbi))
  { setError(ERR_CANNOT_SEEK);
    cb->ptrCBI = ftell(cb->cbi);
    return FALSE;
  }
  cb->ptrCBI += (long) size;
  return TRUE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node readIndexFileCBase
@deftypefun bool readIndexFileCBase (CBase @var{cb}, void *@var{ptr}, size_t @var{size})
Reads a buffer starting at @var{ptr} of @var{size} bytes from the
index file of database @var{cb}.  This function must be used for
reading from a database.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
readIndexFileCBase(CBase cb, void *ptr, size_t size)
{ if (size != fread(ptr, 1, size, cb->cbi))
  { setError(ERR_CANNOT_SEEK);
    cb->ptrCBI = ftell(cb->cbi);
    return FALSE;
  }
  cb->ptrCBI += (long) size;
  return TRUE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node writeLongIndexFileCBase
@deftypefun void writeLongIndexFileCBase (CBase @var{cb}, long @var{n})
Writes @var{n} (four bytes) to the index file of database @var{cb}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
writeLongIndexFileCBase(CBase cb, long n)
{ writeLong(n, cb->cbi);
  cb->ptrCBI += sizeof(long);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node readLongIndexFileCBase
@deftypefun long readLongIndexFileCBase (CBase @var{cb})
Reads a long (four bytes) from the index file of database @var{cb}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

long
readLongIndexFileCBase(CBase cb)
{ cb->ptrCBI += sizeof(long);
  return readLong(cb->cbi);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node seekGameFileCBase
@deftypefun bool seekGameFileCBase (CBase @var{cb}, long @var{n})
Seeks into the game file of database @var{cb} such that the
seek pointer is positioned at @var{n}.  This functions avoids
redundant seeks (which are very time consuming).  Note that
@code{writeGameFileCBase} and @code{readGameFileCBase} must be
used for reading and writing the game file.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
seekGameFileCBase(CBase cb, long n)
{ if (cb->ptrCBF == n)
    return TRUE;
  if (fseek(cb->cbf, n, SEEK_SET) == -1)
    return FALSE;
  return TRUE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node writeGameFileCBase
@deftypefun bool writeGameFileCBase (CBase @var{cb}, void *@var{ptr}, size_t @var{size})
Writes a buffer starting at @var{ptr} of @var{size} bytes to the
game file of database @var{cb}.  This function must be used for
writing to a database.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
writeGameFileCBase(CBase cb, void *ptr, size_t size)
{ if (size != fwrite(ptr, 1, size, cb->cbf))
  { setError(ERR_CANNOT_SEEK);
    cb->ptrCBF = ftell(cb->cbf);
    return FALSE;
  }
  cb->ptrCBF += (long) size;
  return TRUE;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node readGameFileCBase
@deftypefun bool readGameFileCBase (CBase @var{cb}, void *@var{ptr}, size_t @var{size})
Reads a buffer starting at @var{ptr} of @var{size} bytes from the
game file of database @var{cb}.  This function must be used for
reading from a database.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
readGameFileCBase(CBase cb, void *ptr, size_t size)
{ if (size != fread(ptr, 1, size, cb->cbf))
  { setError(ERR_CANNOT_SEEK);
    cb->ptrCBF = ftell(cb->cbf);
    return FALSE;
  }
  cb->ptrCBF += (long) size;
  return TRUE;
}


/*------------------------------------------------------------
 *  Private functions
 *------------------------------------------------------------*/

static CBase
allocCBase(char *name)
{ CBase cb;

  cb = alloc(sizeof(struct cbase));
  cb->name = allocCharp(name);
  cb->noGames = 0;
  cb->index = NULL;
  cb->cbf = NULL;
  cb->cbi = NULL;
  cb->position = 0L;
  cb->ptrCBI = 0L;
  cb->ptrCBF = 0L;

  return cb;
}


static void
checkModeCBase(CBase cb, int mode)
{ if (cb->mode != mode)
  { switch (mode)
    { case READ_MODE:
	fprintf(stderr, "Internal error: Database %s not opened for reading\n",
		cb->name);
	exit(1);
	return;
      case WRITE_MODE:
	fprintf(stderr, "Internal error: Database %s not opened for writing\n",
		cb->name);
	exit(1);
	return;
      default:
	fprintf(stderr, "Internal error: checkModeCBase mode = %d\n", mode);
	exit(1);
    }
  }
}


static long
checkBoundsCBase(CBase cb, long n)
{ if (n < 1)
    return 1;
  if (n > cb->noGames)
    return cb->noGames;
  return n;
}
