;*****************************************************************************
;*
;*    RMAN.LSP
;*
;*    RenderMan commands for AutoCAD
;*
;*    Designed and implemented by Bill Adkison and Larry Knott in April of 1989
;*
;*    NOTE: This module requires the presence of ASCOMMON.LSP.
;*
;*****************************************************************************
;*
;*    Global variables:
;*
;*    G:SVER   -  INT, AutoShade version. (v1.1 = 11, v2.0 = 20)
;*    G:SHEV   -  STR, SHADE path if valid, else nil.
;*    G:R11    -  BOOL, T if AutoCAD r11, nil otherwise.
;*    G:MODE   -  LIST, saved system variables and values.
;*    G:RMAN   -  BOOL, prompt for RenderMan attributes.
;*    G:SCAL   -  REAL, scale factor for block insertion.
;*    G:SETU   -  LIST, setup block names.
;*
;*    Temporary files used:
;*
;*    tempfile.$$$   -  Utility temp file
;*    tempfile.$$a   -  Surface Property/Finish names
;*    tempfile.$$[n] -  Shader parameters
;*
;*****************************************************************************

(vmon)
(prompt "\nLoading rman.lsp...")

;-----------------------------------------------------------------------------
;  LiSt RenderMan Blocks
;
;  (#LSRMB bool) -> nil
;
;  List currently defined RenderMan blocks from #SURF0 and G:SURF.
;-----------------------------------------------------------------------------

(defun #LSRMB  (

   bit1        ;  List RenderMan Setup blocks?
   /
   int1        ;  Current line number
   int2        ;  Current item
   int3        ;  String length
   item1       ;  Current block information.
   file1       ;  tempfile.$$a - List of SPB names
   )

(setq int1  3
      int2  0)
(if bit1
   (cond
      (G:SETU
         (textscr)
         (princ "\nDefined setup blocks:\n  Name\n--------\n")
         (while (and (setq item1 (nth int2 G:SETU)) (#1PAGE int1))
            (princ item1)
            (terpri)
            (setq int1  (1+ int1)
                  int2  (1+ int2))))
      (T (princ "\nNo setup blocks defined."))))
(setq int2  1)
(if (#SURF0)
   (progn
      (textscr)
      (princ "\nDefined surface blocks:\n  Name    ACI\n-------- -----\n")
      (setq file1 (open (strcat
                  (cond (G:SHEV) (""))    ; SHADE dir or current
                  "tempfile.$$a") "r"))
      (foreach item1 (reverse (#SURF0))
         (#1PAGE int1)
         (princ (substr (strcat (read-line file1) "             ")
               1 (- 13 (strlen (itoa item1)))))
         (princ item1)
         (terpri)
         (setq int1  (1+ int1)
               int2  (1+ int2)))
      (setq file1 (close file1)))
   (princ "\nNo surface blocks defined.")))

;-----------------------------------------------------------------------------
;  GeT SHaDer
;
;  (#GTSHD int str bool) -> sym
;
;  Prompt the user for a shader name, and prompt for all parameters for that
;  shader either from the current default, from shaders.txt, or from a preset.
;-----------------------------------------------------------------------------

(defun #GTSHD  (

   int1        ;  Enumerated shader type (1=light,2=disp,3=surf,4=atmo)
   var1        ;  Current shader name attrib variable. (LIST)
   bit1        ;  Explicit shader name to get. (STR)
   /
   str1        ;  Current shader name
   str2        ;  Current shader preset name
   str3        ;  Shader type
   str4        ;  New shader name
   str5        ;  New shader preset name
   str6        ;  Prompt string
   list1       ;  New shader parameters
   file1       ;  Preset file.
   bit2        ;  Read shaders.txt
   bit3        ;  Prompt for parameters
   )

(setq str1  (car (eval var1))
      str2  (cadr (eval var1))
      str3  (nth int1 '("" "Light" "Displacement" "Surface" "Atmosphere")))
(if bit1
   (setq str4 (if (eq bit1 str1)       ;  "project" or "decal" already.
            ""
            bit1))
   (progn
      (setq str6  (strcat "\n?/" str3 " shader name"
               (if (and (= int1 4) (/= str1 "")) " or . for none" "")
               (if (= str1 "") ": " (strcat " <" str1 ">: "))))
      (while (= (setq str4 (getstring str6)) "?")
         (#LSSHD int1))))
(cond
   ((and (= int1 4) (= str4 "."))      ;  Reset Atmosphere to none.
      (set  var1 '("" ""))
      (close (open (strcat (cond (G:SHEV) ("")) "tempfile.$$4") "w")))
   (T (if (/= str4 "")
         (setq bit2 T                  ;  New shader name, read shaders.txt
               str5 (if (= str4 str1) str2 "")) ;  Reset preset name.
         (setq bit3 T                  ;  Same shader, parameters yes!
               str4 str1               ;  Same shader name.
               str5 str2))             ;  Default the preset name.
      (setq file1 (strcat (substr str4 1 8) ".sp" (itoa int1))
            file1 (cond
               ((findfile (strcat "./" file1))) ;  In current dir?
               ((if G:SHEV                      ;  In SHADE dir?
                  (findfile (strcat G:SHEV file1))))
               (T nil)))))             ;  Preset file not found
(if file1                              ;  Preset file found.
   (progn
      (setq str6 (strcat "\n?/Preset name for \"" str4 "\""
               (if (/= str5 "")
                  (strcat " or . for none <" str5 ">: ")
                  ": ")))
      (while (= (setq str5 (getstring str6)) "?")
         (#LSPST int1 file1))
      (setq str5 (substr str5 1 28))   ;  Truncate preset name to 28 characters.
      (cond
         ((= str5 ".")                 ;  Use NO preset
            (setq str5 ""
                  bit3 T))             ;  Prompt for parameters.
         ((and (= str5 "") (= str2 ""));  No preset requested.
            (setq bit3 T))             ;  Prompt for parameters.
         (T (if (= str5 "")            ;  Same preset requested.
               (setq str5 str2))
            (setq bit3                 ;  T if not found.
               (#GTPST (list (+ int1 20) str5 str4) file1 var1))
            (if (not bit3)             ;  Requested preset found!
               (setq bit2 nil
                     bit3 nil))))))    ;  Don't prompt for anything else!
(cond
   (bit2                               ;  Need shader definition & Parameters!
      (if (#GTSDF (list int1 str4) str3 var1) ;  New shader not found
         (setq bit2 nil
               bit3 nil)
         (setq bit1 (#GTPRM int1))))   ;  Shader defined.   (re-use bit1)
   (bit3                               ;  Need parameter values
      (set  var1 (list str4 str5))     ;  Shader name, preset name
      (setq bit1 (#GTPRM int1))))      ;  Get parameter values    (re-use bit1)
(if (and bit1 (or bit2 bit3))          ;  Do we have parameters?
   (if (setq str5 (#WTPST str4 int1))  ;  In either case...
      (set  var1 (list str4 str5))))
T)                                     ;  Return value

;-----------------------------------------------------------------------------
;  GeT PreSeT definition
;
;  (#GTPST list str sym) -> list/str
;
;  Find the requested preset definition, if found update attribibute variables
;  else post message and return nil.
;-----------------------------------------------------------------------------

(defun #GTPST  (

   list1       ;  Enumerated type, preset name, shader name
   file1       ;  Full preset file name
   var1        ;  Current shader name attrib variable. (LIST)
   /
   bit1        ;  Preset found.
   file2       ;  Shader parameter file name
   list2       ;  Current read-line
   list3       ;  New parameter list
   )

(setq list3 '())
(setq file1 (open file1 "r"))
(princ "\nSearching preset file...")
(while (and (not bit1) (setq list2 (read-line file1)))
   (setq list2 (read list2))
   (if (equal list2 list1)
      (progn
         (setq file2 (open (strcat     ;  Ready parm file
                     (cond (G:SHEV) (""))
                     "tempfile.$$"
                     (itoa (rem (car list1) 10))) "w"))
         (while (and (setq list2 (read-line file1))
                     (read list2))     ;  Not a blank line
            (write-line list2 file2))  ;  Copy preset definition to parm file
         (setq file2 (close file2)
               bit1  T))))
(princ "done.")
(setq file1 (close file1))
(cond
   (bit1
      (set  var1 (reverse (cdr list1)))
      nil)                             ;  Return value
   (T (princ (strcat
      "\n*** Preset \"" (cadr list1) "\" not defined in preset file."))
      T)))                             ;  Return value

;-----------------------------------------------------------------------------
;  GeT Shader DeFault from shaders.txt
;
;  (#GTSDF list str sym) -> list/str
;
;  Return nil if search is successful, else return T.
;-----------------------------------------------------------------------------

(defun #GTSDF  (

   list1       ;  Enumerated type, shader name
   str1        ;  Shader type
   var1        ;  Current shader name attrib variable. (LIST)
   /
   file1       ;  shaders.txt
   file2       ;  Current shader parameter file
   list2       ;  Temporary
   list3       ;  Parameters read from shaders.txt
   bit1        ;  Requested shader found in shaders.txt
   bit2        ;  shaders.txt not found
   )

(setq list3 '())
(cond
   ((setq file1 (cond
               ((findfile "./shaders.txt"))  ;  In current dir?
               ((if G:SHEV                   ;  In SHADE dir?
                  (findfile (strcat G:SHEV "shaders.txt"))))
               (T nil)))                     ;  Not found
      (setq file1 (open file1 "r"))
      (princ "\nSearching shader file...")
      (while (and (not bit1) (setq list2 (read-line file1)))
         (setq list2 (read list2))
         (if (equal list2 list1)       ;  We found a match !
            (progn
               (setq file2 (open (strcat  ;  ONLY if found, reset parameters!
                           (cond (G:SHEV) (""))
                           "tempfile.$$"
                           (itoa (car list1))) "w"))
               (while (and (setq list2 (read-line file1))
                           (/= list2 "")) ;  We're on the same shader definition
                     (write-line list2 file2))  ;  No added blank lines
               (setq bit1 T            ;  A match WAS found.
                     file2 (close file2)))))
      (princ "done.")
      (setq file1 (close file1)))
   (T (setq bit2 T)))
(cond
   (bit1                               ;  Shader was found
      (set  var1 (list (cadr list1) ""))
      nil)                             ;  Return value
   (T (if bit2
         (princ "\n*** Shader file not present.")
         (princ (strcat "\n*** " str1 " shader \"" (cadr list1)
                        "\" not defined in shader file.")))
      T)))                             ;  Return value

;-----------------------------------------------------------------------------
;  GeT PaRaMeters
;
;  (#GTPRM int) -> T/nil
;
;  Use prameter file to prompt for new shader parameter values.  Returns T if
;  there are parameters, else nil.
;-----------------------------------------------------------------------------

(defun #GTPRM  (

   int1        ;  Enumerated shader type (1=light,2=disp,3=surf,4=atmo)
   /
   bit1        ;  There ARE parameters
   file1       ;  Shader parameter file
   file2       ;  Temporary file
   item1       ;  New value
   list1       ;  Current parameter list
   str1        ;  Parameter type
   str2        ;  Prompt
   str3        ;  Current read-line from ...
   )

(setq file1 (open (strcat              ;  Open current shader's parameter file
            (cond (G:SHEV) (""))       ;  to prompt with
            "tempfile.$$"
            (itoa int1)) "r")
      file2 (open (strcat              ;  Open temporary file to write to
            (cond (G:SHEV) (""))
            "tempfile.$$$") "w"))
(while (and (setq str3 (read-line file1)) ;  There is more ...
            (setq list1 (read str3)))     ;  and its not a blank
   (setq bit1  T
         str1  (nth (- (car list1) 11) '("scalar" "color" "point" "string"))
         str2  (if (/= (caddr list1) "")
                  (strcat " <"         ;  Express points in current UCS
                     (if (and (= str1 "point") (zerop (getvar "WORLDUCS")))
                        (#PTSTR (trans (#STRPT (caddr list1)) 0 1) ",")
                        (caddr list1))
                     ">: ")
                  ": ")
         str2  (strcat "\nEnter " str1 " \"" (cadr list1) "\"" str2))
   (cond
      ((= str1 "scalar")
         (if (setq item1 (getdist str2))
            (setq item1 (rtos item1 2 6))))
      ((= str1 "string")
         (setq item1 (#2UNIX (getstring str2))))
      ((= str1 "point")                ;  Convert entered points to WCS
         (if (setq item1 (#GTP3D 0 nil str2))
            (setq item1 (#PTSTR (trans item1 1 0) ","))))
      ((= str1 "color")
         (if (setq item1 (#GT0-1 (substr str2 2 (- (strlen str2) 3)) nil 1))
            (setq item1 (#PTSTR item1 ",")))))
   (if (and item1 (/= item1 ""))
      (setq list1 (list (car list1) (cadr list1) item1)))
   (prin1 list1 file2)
   (princ "\n" file2))
(setq file1 (close file1)
      file2 (close file2))
(if bit1
   (progn
      (setq file1 (open (strcat        ;  Open shader parameter file to write to
                  (cond (G:SHEV) (""))
                  "tempfile.$$"
                  (itoa int1)) "w")
            file2 (open (strcat        ;  Open temporary file to read from
                  (cond (G:SHEV) (""))
                  "tempfile.$$$") "r"))
      (while (setq str1 (read-line file2))
         (write-line str1 file1))
      (setq file1 (close file1)
            file2 (close file2))
      T)))                             ;  Return T if parameters, else nil

;-----------------------------------------------------------------------------
;  GeT Preset NaMe
;
;  (#GTPNM int str str) -> str/nil
;
;  A preset name will be requested until either it's null or not defined in the
;  preset file already.
;-----------------------------------------------------------------------------

(defun #GTPNM  (

   int1        ;  Shader enumerated type
   str1        ;  Shader name
   str3        ;  Preset file name
   /
   bit1        ;  Preset prompting complete
   bit2        ;  Requested name duplicated
   str2        ;  Preset name
   list1       ;  Temporary list
   list2       ;  List to search for: (int str str)
   file1       ;  Preset file
   )

(setq str3  (cond
         ((findfile (strcat "./" str3)))  ;  In current dir?
         ((if G:SHEV                      ;  In SHADE dir?
            (findfile (strcat G:SHEV str3))))
         (T nil)))                  ;  Should always be found
(while (not bit1)
   (setq file1 (if str3 (open str3 "r"))  ;  Should always be found
         bit2  nil
         str2  (getstring "\nName of preset to save: ")
         list2 (list (+ int1 20) str2 str1))
   (if (equal str2 "")
      (setq bit1 T                  ;  User wants none.
            bit2 T)                 ;  Don't return one.
      (if file1                     ;  Check for duplicates
         (progn
            (while (and (not bit2) (setq list1 (read-line file1)))
               (setq list1 (read list1))
               (if (equal list1 list2) ;   Duplicate.
                  (setq bit2 T)))
            (if bit2                   ;  Duplicated name
               (progn
                  (initget "Yes No")
                  (if (equal (getkword (strcat "\n*** Preset \"" str2
                           "\" is already defined, replace it?  Yes/<No>: "))
                        "Yes")
                     (progn
                        (setq bit1  T        ;  We're done
                              bit2  nil      ;  Use the preset name
                              file1 (close file1))
                        (#RPLPS list2 str3)))   ;  Full preset file name
                     (setq bit2  nil))
               (setq bit1 T))
            (if file1 (setq file1 (close file1))))
         (setq bit1 T))             ;  Got name and no preset file exists
   ))
(if (not bit2) str2))

;-----------------------------------------------------------------------------
;  RePLace PreSet
;
;  (#RPLPS str file) -> nil/str
;
;-----------------------------------------------------------------------------

(defun #RPLPS  (

   list1       ;  Preset list
   str3        ;  Full preset file name
   /
   file1       ;  Shader parm file
   file2       ;  Temporary file
   str1        ;  Temporary file name
   list2       ;  Current read-line
   int1        ;  Number of relevant lines read
   )

(setq str1  (strcat
            (cond (G:SHEV) (""))       ;  SHADE dir or current
            "tempfile.$$$")
      file1 (open str3 "r")
      file2 (open str1 "w")            ;  tempfile.$$$
      int1  0)
(while (setq list2 (read-line file1))  ;  Not EOF
   (if (equal (read list2) list1)      ;  Found the definition to replace, now
      (while (and (setq list2 (read-line file1))   ;  read until a blank line
                  (read list2))        ;  is found
            T)
      (progn
         (setq int1 (1+ int1))         ;  Count # lines to save
         (write-line list2 file2))))   ;  Copy to tempfile.$$$
(setq file1 (close file1)
      file2 (close file2)
      file1 (open str3 "w")            ;  Dump contents of preset file
      file2 (open str1 "r"))           ;  Temporary file
(repeat int1                              ;  Copy contents of temp file
   (write-line (read-line file2) file1))  ;  to Preset file
(setq file1 (close file1)
      file2 (close file2)))

;-----------------------------------------------------------------------------
;  WriTe PreSeT
;
;  (#WTPST str int) -> nil/str
;
;  If the user wants a preset written, and supplies a non-nil name for the
;  preset, the current shader has it's preset file appended with the new entry.
;-----------------------------------------------------------------------------

(defun #WTPST  (

   str1        ;  Shader name
   int1        ;  Enumerated shader type (1=light,2=disp,3=surf,4=atmo)
   /
   str2        ;  Preset name to save
   str3        ;  Preset file name
   file1       ;  Preset file
   file2       ;  Parameter file
   )

(initget "Yes No")
(setq str2 (getkword "\nWrite current values to preset file?  Yes/<No>: ")
      str3 (strcat (substr str1 1 8) ".sp" (itoa int1)))
(if (and (= str2 "Yes")
         (setq str2 (#GTPNM int1 str1 str3)))
  (progn
      (setq file1 (open (strcat
                  (cond (G:SHEV) (""))
                  str3) "a")           ;  Append new Preset definition
            file2 (open (strcat
                  (cond (G:SHEV) (""))
                  "tempfile.$$" (itoa int1)) "r"))
      (prin1 (list (+ 20 int1) str2 str1) file1)
      (princ "\n" file1)
      (while (setq str1 (read-line file2))   ;  Read parameters from temp file
         (write-line str1 file1))      ;  Append parameters to preset file
      (princ "\n" file1)               ;  Add blank line at end
      (setq file1 (close file1)
            file2 (close file2))
      str2)))                          ;  Return preset name or nil

;-----------------------------------------------------------------------------
;  LiSt PreSeTs
;
;  (#LSPST int str) -> nil
;
;  List all of the preset names in the preset file.
;-----------------------------------------------------------------------------

(defun #LSPST (

   int1        ;  Shader enumerated type
   str1        ;  Full preset file name
   /
   int2        ;  Line count
   list1       ;  Current read-line
   file1       ;  Preset file
   )

(setq int1  (+ int1 20)
      int2  2
      file1 (open str1 "r"))
(textscr)
(princ (strcat "\nPresets listed in " str1 ":"
               "\n-------------------"))
(repeat (strlen str1)
   (princ "-"))
(terpri)
(while (and (#1PAGE int2) (setq list1 (read-line file1)))
   (setq list1 (read list1))
   (if (= (car list1) int1)
      (progn
         (setq int2 (1+ int2))
         (princ (cadr list1))
         (terpri))))
(setq file1 (close file1)))

;-----------------------------------------------------------------------------
;  GeT Texture Coordinates using AutoCAD's "Insert" method.
;
;  (#GTTCI list) -> list
;
;  Use AutoCAD's insert prompt sequence to define texture coordinates for
;  primatives.
;-----------------------------------------------------------------------------

(defun #GTTCI  (

   /
   pt1         ;  Texture origin
   real1       ;  X-scale factor ("s")
   real2       ;  Y-scale factor ("t")
   ang1        ;  Rotation angle
   list1       ;  List of tcoords
   list2       ;  Temporary list
   )

(setq pt1   (cond ((getpoint "\nTexture origin <0,0>: ")) ('(0.0 0.0 0.0)))
      pt1   (list (car pt1) (cadr pt1))
      list1 (mapcar
         '(lambda (list2)              ;  Process each point
            (mapcar '- list2 pt1))
         '((0 0) (1 0) (0 1) (1 1))))  ;  Return list modified
(initget 2)                            ;  n /= 0
(setq real1 (cond ((getreal "\nX scale factor <1>: ")) (1)))
(initget 2)                            ;  n /= 0
(setq real2 (cond ((getreal "\nY scale factor <1>: ")) (1))
      list1 (mapcar
         '(lambda (list2)              ;  Process each point
            (mapcar '/ list2 (list real1 real2)))
         list1)                        ;  Return list modified
      ang1  (cond ((getangle "\nRotation angle <0>: ")) (0))
      real1 (cos ang1)
      real2 (sin ang1))
(mapcar
   '(lambda (list2)
      (list
         (+ (* (car list2) real1)      (* (cadr list2) real2))
         (+ (* (- (car list2)) real2)  (* (cadr list2) real1))))
   list1))

;-----------------------------------------------------------------------------
;  GeT Texture Coordinates Numerically
;
;  (#GTTCN 'point) -> 'point
;
;  Given texture coordinate variable name, reassign variable to input point,
;  else leave unchanged.  Returns new value.
;-----------------------------------------------------------------------------

(defun #GTTCN  (

   var1        ;
   int1        ;
   /
   item1       ;
   )

(if (setq item1 (getpoint (strcat
      "\nTexture coordinate " int1 " <" (#PTSTR (eval var1) ",") ">: ")))
   (set var1 (list (car item1) (cadr item1)))
   (eval var1)))

;-----------------------------------------------------------------------------
;  GeT Texture Coordinates by Picking
;
;  (#GTTCP) -> nil
;
;
;-----------------------------------------------------------------------------

(defun #GTTCP  (

   /
   x1 y1 x2 y2 x3 y3 x4 y4
   x1_2  x1_3  x1_4  x2_3  x2_4  x3_4
   y1_2  y1_3  y1_4  y2_3  y2_4  y3_4
   bit1
   item1
   item2
   )

(and
   (setq item2 (#GTMSH))               ;  We got a mesh...
   (setq item2 (#VP2UV item2))         ;  and computed (U, V)'s (no coincident)
   (if (or  (inters  (car item2)    (cadr item2)
                     (caddr item2)  (cadddr item2) nil)
            (inters  (car item2)    (caddr item2)
                     (cadr item2)   (cadddr item2) nil))
      (if (zerop (setq real3 (#REAL3 item2)))   ;  and division is OK...
         (prompt "\n*** Cannot compute (S, T) from selected (U, V).")
         (setq item2 (#UV2ST real3)    ;  New PIXAR code.
               bit1  T))
      (setq item2 (mapcar '(lambda (item1)
                     (list
                        (#PERDI (car item2) (caddr item2) (cadr item2) item1)
                        (#PERDI (car item2) (cadr item2) (caddr item2) item1)))
                     '((0 0) (1 0) (0 1) (1 1)))
            bit1  T)))
(mapcar 'set
   '(pt1 pt2 pt3 pt4)
   (if bit1
      item2
      '((0 0) (1 0) (0 1) (1 1)))))

;-----------------------------------------------------------------------------
;  PERpendicular DIstance.
;
;  (#PERDI point point point point) -> real
;
;  Given 2 points that define a line, and another point whose perpendicular
;  distance from that line defines one positive unit, the perpendicular
;  distance from that line to the last point in terms of the new units is
;  returned.
;-----------------------------------------------------------------------------

(defun #PERDI  (

   pt1         ;  First endpoint.
   pt2         ;  Second endpoint.
   pt3         ;  Unit point.
   pt4         ;  Point in question.
   )

(/ (* (sin (- (* 2 pi) (- (angle pt1 pt2) (angle pt1 pt4))))
      (distance pt1 pt4))
   (* (sin (- (* 2 pi) (- (angle pt1 pt2) (angle pt1 pt3))))
      (distance pt1 pt3))))

;-----------------------------------------------------------------------------
;  Vertex Points to parametric U V points.
;
;  (#VP2UV entlist) -> list
;
;  Computes parametric mesh points from selected vertex points.
;-----------------------------------------------------------------------------

(defun #VP2UV  (

   list1       ;  Main polyline entity list
   /
   list2       ;  List of VP converted to vertex numbers (index)
   item1       ;  Temporary
   bit1
   int1  int2  int3  int4  str1  str2
   )

(setq pt1 nil pt2 nil pt3 nil pt4 nil) ;  Nullify current tcoords.
(#SVRST '(("OSMODE" . 1)))
(while (or (not pt1) (not (osnap pt1 "end")))
   (setq pt1 (getpoint "\nTexture coordinate 1 <end>: ")))
(while (or (not pt2) (not (osnap pt2 "end")))
   (setq pt2 (getpoint pt1 "\nTexture coordinate 2 <end>: ")))
(grdraw pt1 pt2 -1 1)
(while (or (not pt3) (not (osnap pt3 "end")))
   (setq pt3 (getpoint pt1 "\nTexture coordinate 3 <end>: ")))
(grdraw pt1 pt3 -1 1)
(while (or (not pt4) (not (osnap pt4 "end")))
   (setq pt4 (getpoint pt3 "\nTexture coordinate 4 <end>: ")))
(#SVRST 1)
(grdraw pt2 pt4 -1 1)   (grdraw pt3 pt4 -1 1)
(foreach item1 '(pt1 pt2 pt3 pt4)      ;  Convert all points to WCS
   (set item1 (trans (eval item1) 1 0)))
(setq list2 (list pt1 pt2 pt3 pt4))
(setq bit1  (= (logand (#GTVAL 70 list1) 4) 4)
      int1  (#GTVAL (if bit1 73 71) list1)
      int2  (#GTVAL (if bit1 74 72) list1)
      int3  0
      int4  0
      str1  (strcat "Checking " (itoa (* int1 int2)) " vertices: ")
      str2  '(princ (strcat str1 (itoa int4) "\r"))
      list1 (entget (entnext (#GTVAL -1 list1))))
(terpri)
(while (and (< int3 4) (= (#GTVAL 0 list1) "VERTEX"))
   (cond
      ((if bit1                                 ;  If spline-fit polyline,
            (= (logand (#GTVAL 70 list1) 8) 8)  ;  ignore control vertexs.
            T)
         (eval str2)
         (if (member (#GTVAL 10 list1) list2)
            (setq int3  (1+ int3)
                  list2 (subst int4 (#GTVAL 10 list1) list2)))
         (setq int4 (1+ int4))))
   (setq list1 (entget (entnext (#GTVAL -1 list1)))))
(grdraw pt1 pt2 -1 1)   (grdraw pt1 pt3 -1 1)
(grdraw pt2 pt4 -1 1)   (grdraw pt3 pt4 -1 1)
(if (< int3 4)                         ;  All vertexes found?
   (prompt "\n*** All points do not lie on mesh or some are coincident.")
   (mapcar
      '(lambda (item1)
         (list
            (/ (rem item1 int2)        ;  Row # (starts at 0)
               (float (1- int1)))      ;  % row's traversed
            (/ (/ item1 int2)          ;  Column # (starts at 0)
               (float (1- int2)))))    ;  % column's traversed
         list2)))                      ;  List of vertex #'s matched.

;-----------------------------------------------------------------------------
;  Utility function for #UV2ST
;-----------------------------------------------------------------------------

(defun #REAL3 (

   list1       ;  List of (U, V) points.
   )

(setq x1_4  (car list1)    x1 (car x1_4)  y1 (cadr x1_4)
      x2_4  (cadr list1)   x2 (car x2_4)  y2 (cadr x2_4)
      y1_4  (caddr list1)  x3 (car y1_4)  y3 (cadr y1_4)
      y2_4  (cadddr list1) x4 (car y2_4)  y4 (cadr y2_4)
      list1 nil)

(setq x1_2  (* x1 x2)      y1_2  (* y1 y2)
      x1_3  (* x1 x3)      y1_3  (* y1 y3)
      x1_4  (* x1 x4)      y1_4  (* y1 y4)
      x2_3  (* x2 x3)      y2_3  (* y2 y3)
      x2_4  (* x2 x4)      y2_4  (* y2 y4)
      x3_4  (* x3 x4)      y3_4  (* y3 y4))

(- (+ (* x1_3 y1_2)  (* x2_4 y1_2)  (* x2_3 y1_3)  (* x1_4 y1_3)
      (* x1_2 y2_3)  (* x3_4 y2_3)  (* x1_2 y1_4)  (* x3_4 y1_4)
      (* x2_3 y2_4)  (* x1_4 y2_4)  (* x1_3 y3_4)  (* x2_4 y3_4))
   (+ (* x2_3 y1_2)  (* x1_4 y1_2)  (* x1_2 y1_3)  (* x3_4 y1_3)
      (* x1_3 y2_3)  (* x2_4 y2_3)  (* x1_3 y1_4)  (* x2_4 y1_4)
      (* x1_2 y2_4)  (* x3_4 y2_4)  (* x2_3 y3_4)  (* x1_4 y3_4))))

;-----------------------------------------------------------------------------
;  parametric U V points to texture coordinates S T.
;
;  (#UV2ST entlist) -> list
;
;  Computes texture coordinates from supplied parametric U V points.
;-----------------------------------------------------------------------------

(defun #UV2ST  (

   real3       ;  Magic number from #REAL3
   /
   real1       ;
   real2       ;
   pt1         ;  (s1 t1)
   pt2         ;  (s2 t2)
   pt3         ;  (s3 t3)
   pt4         ;  (s4 t4)
   )

(setq real1 (- (+ (* x2_3 y1_2)  (* x1_2 y1_3)  (* x1_3 y2_3))
               (+ (* x1_3 y1_2)  (* x2_3 y1_3)  (* x1_2 y2_3)))
      real2 (- (+ (* x3_4 y1_3)  (* x1_3 y1_4)  (* x1_4 y3_4))
               (+ (* x1_4 y1_3)  (* x3_4 y1_4)  (* x1_3 y3_4)))
      pt1   (- (- (/ real1 real3)) (/ real2 real3))               ;  s1
      real2 (- (+ (* x1_4 y1_2)  (* x2_4 y1_4)  (* x1_2 y2_4))
               (+ (* x2_4 y1_2)  (* x1_2 y1_4)  (* x1_4 y2_4)))
      pt1   (list pt1 (- (- (/ real1 real3)) (/ real2 real3))))   ;  (s1 t1)

(setq real1 (- (+ (* x1   y1_2)  (* x2_3 y1_2)  (* x1_2 y1_3)
                  (* x3   y1_3)  (* x2   y2_3)  (* x1_3 y2_3))
               (+ (* x2   y1_2)  (* x1_3 y1_2)  (* x1   y1_3)
                  (* x2_3 y1_3)  (* x1_2 y2_3)  (* x3   y2_3)))
      real2 (- (+ (* x1   y1_3)  (* x3_4 y1_3)  (* x1_3 y1_4)
                  (* x4   y1_4)  (* x3   y3_4)  (* x1_4 y3_4))
               (+ (* x3   y1_3)  (* x1_4 y1_3)  (* x1   y1_4)
                  (* x3_4 y1_4)  (* x1_3 y3_4)  (* x4   y3_4)))
      pt2   (- (- (/ real1 real3)) (/ real2 real3))               ;  s2
      real2 (- (+ (* x2   y1_2)  (* x1_4 y1_2)  (* x1   y1_4)
                  (* x2_4 y1_4)  (* x1_2 y2_4)  (* x4   y2_4))
               (+ (* x1   y1_2)  (* x2_4 y1_2)  (* x1_2 y1_4)
                  (* x4   y1_4)  (* x2   y2_4)  (* x1_4 y2_4)))
      pt2   (list pt2 (- (- (/ real1 real3)) (/ real2 real3))))   ;  (s2 t2)

(setq real1 (- (+ (* x1_3 y1)    (* x1_2 y2)    (* x2_3 y1_2)
                  (* x2_3 y3)    (* x1_2 y1_3)  (* x1_3 y2_3))
               (+ (* x1_2 y1)    (* x2_3 y2)    (* x1_3 y1_2)
                  (* x1_3 y3)    (* x2_3 y1_3)  (* x1_2 y2_3)))
      real2 (- (+ (* x1_4 y1)    (* x1_3 y3)    (* x3_4 y1_3)
                  (* x3_4 y4)    (* x1_3 y1_4)  (* x1_4 y3_4))
               (+ (* x1_3 y1)    (* x3_4 y3)    (* x1_4 y1_3)
                  (* x1_4 y4)    (* x3_4 y1_4)  (* x1_3 y3_4)))
      pt3   (- (- (/ real1 real3)) (/ real2 real3))               ;  s3
      real2 (- (+ (* x1_2 y1)    (* x2_4 y2)    (* x1_4 y1_2)
                  (* x1_4 y4)    (* x2_4 y1_4)  (* x1_2 y2_4))
               (+ (* x1_4 y1)    (* x1_2 y2)    (* x2_4 y1_2)
                  (* x2_4 y4)    (* x1_2 y1_4)  (* x1_4 y2_4)))
      pt3   (list pt3 (- (- (/ real1 real3)) (/ real2 real3))))   ;  (s3 t3)

(setq real1 (- (+ (* x2   y1)    (* x1_3 y1)    (* x1_2 y2)
                  (* x3   y2)    (* x1   y1_2)  (* x2_3 y1_2)
                  (* x1   y3)    (* x2_3 y3)    (* x1_2 y1_3)
                  (* x3   y1_3)  (* x2   y2_3)  (* x1_3 y2_3))
               (+ (* x1_2 y1)    (* x3   y1)    (* x2   y1_2)
                  (* x1_3 y3)    (* x1_2 y2_3)  (* x1   y2)
                  (* x2_3 y2)    (* x1_3 y1_2)  (* x2   y3)
                  (* x1   y1_3)  (* x2_3 y1_3)  (* x3   y2_3)))
      real2 (- (+ (* x3   y1)    (* x1_4 y1)    (* x1_3 y3)
                  (* x4   y3)    (* x1   y1_3)  (* x3_4 y1_3)
                  (* x1   y4)    (* x3_4 y4)    (* x1_3 y1_4)
                  (* x4   y1_4)  (* x3   y3_4)  (* x1_4 y3_4))
               (+ (* x4   y1)    (* x3   y1_3)  (* x1_4 y4)
                  (* x1_3 y3_4)  (* x1_3 y1)    (* x1   y3)
                  (* x3_4 y3)    (* x1_4 y1_3)  (* x3   y4)
                  (* x1   y1_4)  (* x3_4 y1_4)  (* x4   y3_4)))
      pt4   (- (- (/ real1 real3)) (/ real2 real3))               ;  s4
      real2 (- (+ (* x1_2 y1)    (* x4   y1)    (* x1   y2)
                  (* x2_4 y2)    (* x2   y1_2)  (* x1_4 y1_2)
                  (* x2   y4)    (* x1_4 y4)    (* x1   y1_4)
                  (* x2_4 y1_4)  (* x1_2 y2_4)  (* x4   y2_4))
               (+ (* x2   y1)    (* x1_4 y1)    (* x1_2 y2)
                  (* x4   y2)    (* x1   y1_2)  (* x2_4 y1_2)
                  (* x1   y4)    (* x2_4 y4)    (* x1_2 y1_4)
                  (* x4   y1_4)  (* x2   y2_4)  (* x1_4 y2_4)))
      pt4   (list pt4 (- (- (/ real1 real3)) (/ real2 real3))))   ;  (s4 t4)
(list pt1 pt2 pt3 pt4))

;-----------------------------------------------------------------------------
;  GeT MeSH for texture mapping.
;
;  (#GTMSH) -> entlist/T
;
;  If a Polyline mesh is selected, its entity list is returned, else T is
;  returned.
;-----------------------------------------------------------------------------

(defun #GTMSH  (

   /
   bit1        ;  T if no entity selected, else valid polyline elist
   list1       ;  (entsel) list
   list2       ;  Current elist
   )

(while (not bit1)
   (if (setq list1 (entsel "\nSelect polygon mesh: "))
      (if (eq (#GTVAL 0 (setq list2 (entget (car list1)))) "POLYLINE")
         (if (= (logand (#GTVAL 70 list2) 16) 16)
            (setq bit1 list2)
            (prompt "*** Selected object is not a polygon mesh."))
         (prompt (strcat "*** " (#GTVAL 0 list2) ", not a valid entity.")))
      (setq bit1 T)))
(if (listp bit1)                       ;  Return value.
   bit1))

;-----------------------------------------------------------------------------
;  GeT Surface Property Options
;
;  (#GTSPO) -> nil
;
;  GeT Surface Property Options for RM_SDB.DWG
;-----------------------------------------------------------------------------

(defun #GTSPO  (

   /
   item1       ;
   item2       ;
   real1       ;
   real2       ;
   real3       ;
   str1        ;  Temporary
   bit1        ;  Valid Rate
   )

(initget "Air Color Displacement Opaque Project Rate SMooth SUrface Tcoord")
(setq item1 (getkword
   "\nAir/Color/Displacement/Opaque/Project/Rate/SMooth/SUrface/Tcoord: "))
(cond
   ((= item1 "Air")
      '(#GTSHD 4 'att12 nil))
   ((= item1 "Color")
      '(setq att3
            (cond
               ((#GT0-1 (strcat "-" (itoa att2) "/RGB color") att3 0))
               (att3))))
   ((= item1 "Displacement")
      '(progn
         (#GTSHD 2 'att8 nil)
         (cond
            ((/= (car att8) "nulldisp")
               (initget 4)             ;  n >= 0
               (setq att10
                     (cond
                        ((getreal (strcat "\nMaximum displacement <"
                                    (rtos att10 2 6) ">: ")))
                        (att10))))
            (T (setq att10 0.0)))      ;  Dbounds = 0.0 with nulldisp
      T))
   ((= item1 "Opaque")
      '(setq att4 (cond
               ((#GT0-1 "Opacity" att4 nil))
               (att4))))
   ((= item1 "Rate")
      '(progn
         (setq bit1 nil)
         (while (not bit1)
            (setq item2 (getreal (strcat
                           "\nShading rate <"
                           (if (minusp att7)
                              "setup default"
                              (rtos att7 2 2))
                           ">: ")))
            (if item2                  ;  New value
               (if (or  (minusp item2) ;  Scene default or valid range ...
                        (and (>= item2 0.25) (<= item2 1e+36)))
                  (setq bit1  T
                        att7  (cond
                           ((minusp item2)   -1)
                           (T                item2)))
                  (prompt
                     "\n*** Value must be negative or in range 0.25 to 1e+36."))
               (setq bit1 T)))         ;  Keep old value.
      T))
   ((= item1 "SMooth")
      '(progn
         (initget "Yes No")
         (setq item2
               (getkword
                  (strcat
                     "\nSmooth arbitrary polygon meshes?  "
                     (if (zerop att11)
                        "Yes/<No>: "
                        "No/<Yes>: ")))
               att11
                  (cond
                     ((= item2 "No")   0)
                     ((= item2 "Yes")  1)
                     (T                att11)))))
   ((= item1 "SUrface")
      '(#GTSHD 3 'att5 nil))
   ((= item1 "Tcoord")
      '(progn
         (if (and (/= (car att5) "decal")
                  (progn
                     (initget "Yes No")
                     (equal (getkword (strcat
                        "\nReplace current surface shader \"" (car att5)
                        "\" with \"decal\"?  Yes/<No>: "))
                     "Yes")))
            (#GTSHD 3 'att5 "decal"))
         (initget "Insert Pick Numeric")
         (setq item2 (getkword "\nInsert/Pick/<Numeric>: "))
         (cond
            ((= item2 "Insert")
               (setq item2 (#GTTCI)
                     pt1 (car item2)
                     pt2 (cadr item2)
                     pt3 (caddr item2)
                     pt4 (cadddr item2)))
            ((= item2 "Pick")
               (#GTTCP))
            (T (#SVRST '(("BLIPMODE" . 0)))
               (#GTTCN 'pt1 "1")
               (#GTTCN 'pt2 "2")
               (#GTTCN 'pt3 "3")
               (#GTTCN 'pt4 "4")
               (#SVRST 1)))
         T))
   ((= item1 "Project")
      '(progn
         (if (or  (equal (car att5) "project")
                  (progn
                     (initget "Yes No")
                     (equal (getkword (strcat
                        "\nReplace current surface shader \"" (car att5)
                        "\" with \"project\"?  Yes/<No>: "))
                     "Yes")))
            (#GTSHD 3 'att5 "project"))
         T))))

;-----------------------------------------------------------------------------
;  RenderMan surface PROPerty.
;
;  (C:RMPROP) -> nil
;
;  Main RenderMan Surface Property block definition.
;-----------------------------------------------------------------------------

(defun C:RMPROP   ( /   #ERROR inspt att1 att2 att3 att4 att5 att6 att7
                        att8 att9 att10 att11 att12 att13 pt1 pt2 pt3 pt4)

(#HEAD (#SVINS))
(setq att1 (#GTDNM "\nSurface property name"))
(while (not att2)
   (setq att2 (#GTDCO)))
(#SBDFL)
(while (eval (#GTSPO)) T)
(initget 1)                            ;  No null
(setq inspt (getpoint "\nSurface property location: "))
(#INSDB)
(#ADSDB att1 att2 nil)
(#TAIL))

;-----------------------------------------------------------------------------
;  RenderMan ATTACH.
;
;  (C:RMATTACH) -> nil
;
;  Attach Surface blocks to entities or layers.
;-----------------------------------------------------------------------------

(defun C:RMATTACH (

   /
   #ERROR
   bit1
   int1
   str1
   item1
   str2
   )

(#HEAD (#SVINS))
(setq int1 0)
(while (not bit1)
   (setq str1 (strcase (getstring "\nSurface property name/<?>: ")))
   (cond
      ((or (= str1 "") (= str1 "?"))
         (#LSRMB nil))
      ((setq int1 (#GTSBI str1 T))
         (setq bit1 T))
      (T (prompt (strcat "\n*** Surface property \"" str1 "\" not defined.")))))
(initget "Layer Select")
(setq item1 (getkword "\nAttach by: Layer/<Select>: "))
(cond
   ((or (null item1) (= item1 "Select"))
      (prompt (strcat "\nSelect entities to attach \"" str1 "\" to: "))
      (setq sset1 (ssget))
      (if sset1
         (progn
            (#SVRST '(("HIGHLIGHT" . 0)))
            (command ".chprop" sset1 "" "c" int1 "")
            (#SVRST 1))
         (prompt "\nNo entities selected.")))
   (T (setq str2  (getvar "clayer")
            sset1 (cond
               ((getstring (strcat  "\nLayer name to apply \"" str1
                                    "\" to <" str2 ">: ")))
               (str2)))
      (while (and (/= sset1 "") (not (tblsearch "LAYER" sset1)))
         (prompt (strcat "\nLayer \"" sset1 "\" not found."))
         (setq sset1 (cond
                  ((getstring (strcat  "\nLayer name to apply \"" str1
                                       "\" to <" str2 ">: ")))
                  (str2))))
      (command ".layer" "c" int1 sset1 "")))
(#TAIL))

;-----------------------------------------------------------------------------
;  LiSt SHaDers defined in shaders.txt
;
;  (#LSSHD int/nil) -> nil
;
;  If the shader type is supplied, show defined shaders, else prompt for the
;  type, then show the shaders defined.
;-----------------------------------------------------------------------------

(defun #LSSHD  (

   int1        ;  Enumerated shader type.
   /
   str1        ;
   file1       ;
   int2        ;
   item1       ;
   list1       ;
   )

(cond
   ((setq file1
            (cond
               ((findfile "./shaders.txt"))  ;  In current dir?
               ((if G:SHEV                   ;  In SHADE dir?
                  (findfile (strcat G:SHEV "shaders.txt"))))
               (T nil)))                     ;  Not found
      (cond
         (int1
            (setq str1 (nth int1
                     '("" "Light" "Displacement" "Surface" "Atmosphere"))))
         (T (initget "Atmosphere Displacement Light Surface")
            (setq str1
                  (getkword "\nAtmosphere/Displacement/Light/<Surface>: "))
            (if (not str1)
               (setq str1 "Surface"))
            (setq int1
                  (length (member str1
                     '("Atmosphere" "Surface" "Displacement" "Light"))))))
      (setq file1 (open file1 "r")
            int2  1)
      (textscr)
      (terpri)
      (princ (strcat str1 " shaders listed in shader file:"
                     "\n--------------------------------------------\n"))
      (while (and (#1PAGE int2) (setq list1 (read-line file1)))
         (setq list1 (read list1))
         (if (= (car list1) int1)
            (progn
               (setq int2 (1+ int2))
               (princ (substr (strcat (cadr list1) "               ") 1 16))
               (while (and (setq list1 (read-line file1))
                           (setq list1 (read list1)))
                  (princ (cadr list1))
                  (princ " "))
               (terpri))
            (if (zerop (rem int2 15))
               (setq int2 1))))
      (setq file1 (close file1)))
   (T (princ "\n*** Shader file not present."))))

;-----------------------------------------------------------------------------
;  RenderMan LIST
;
;  (C:RMLIST) -> nil
;
;  General function to list RenderMan information defined or available.
;-----------------------------------------------------------------------------

(defun C:RMLIST   (

   /
   #ERROR         ;
   int1           ;  ACI
   item1          ;  Option
   str1           ;  Various strings
   str2           ;  Surface name
   )

(#HEAD (#SVINS))
(initget "Colors Shaders Entity Layer Defined")
(setq item1 (getkword "\nColors/Shaders/Entity/Layer/<Defined>: ")
      int1  nil)
(cond
   ((= item1 "Colors")
      (#LSCLR))
   ((= item1 "Shaders")
      (#LSSHD nil))
   ((= item1 "Entity")    ;  MAKE THIS LOOP UNTIL A NULL SELECTION?
      (setq int1 (#GTECL "\nSelect an entity: ")))
   ((= item1 "Layer")
      (setq str1  nil
            str2  (strcat "\nLayer name <" (getvar "clayer") ">: "))
      (while (not str1)
         (if (= (setq str1 (getstring str2)) "")
            (setq str1 (getvar "clayer")))
            (if (not (tblsearch "LAYER" str1))
               (setq str1
                     (prompt (strcat "\nCannot find layer " str1 ".")))))
      (setq int1 (abs (#GTVAL 62 (tblsearch "layer" str1)))))
   (T (#LSRMB T)))
(if int1
   (prompt
      (cond
         ((setq str2 (#GTSBI int1 T))
            (strcat  "\nSurface block \"" str2
                     "\" is attached (using ACI-" (itoa int1) ")."))
         (int1  "\nNo surface block attached."))))
(#TAIL))

;-----------------------------------------------------------------------------
;  GeT CroP value Numerically
;
;  (#GTCPN str real real) -> real
;
;  Get a single crop fraction value.
;-----------------------------------------------------------------------------

(defun #GTCPN  (

   str1        ;  Prompt.
   real1       ;  Minimum value.
   real2       ;  Default value.
   /
   real3       ;  New value
   str2        ;  Error message
   )

(if (member str1 '("Left" "Bottom"))
   (setq var2 >=
         var1 <
         str2 ">= 0.0000 and <")
   (setq var2 >
         var1 <=
         str2 (strcat "> " (rtos real1 2 4) " and <=")))
(while (or
      (var1
         (setq real3 (cond
                  ((getreal (strcat "\n" str1 " <" (rtos real2 2 4) ">: ")))
                  (real2)))
         real1)
      (var2 real3 1.0))
   (prompt (strcat "*** Value must be " str2 " 1.0000 .")))
real3)

;-----------------------------------------------------------------------------
;  GeT CroP value Visually
;
;  (#GTCPV) -> list
;-----------------------------------------------------------------------------

(defun #GTCPV  (

   /
   bit1        ;
   pt1         ;
   pt2         ;
   real1       ;
   real2       ;
   real3       ;
   real4       ;
   )

(if (= (logand 1 (getvar "VIEWMODE")) 1)
   (progn                              ;  We're in perspective, so we can't
      (setq bit1 T)                    ;  be in Paper Space.
      (prompt "\nPlease wait...")
      (#SVRST '(("UCSICON" . 0) ("GRIDMODE" . 0)))
      (command ".mslide" (strcat (getvar "dwgprefix") "rmancrop")
               ".dview" "" "off" ""
               ".ucs" "v"
               ".vslide" (strcat (getvar "dwgprefix") "rmancrop"))))
(initget 1)                            ;  No null
(setq pt1   (getpoint "\nFirst corner: "))
(initget 33)                           ;  No null, dashed lines
(setq pt2   (getcorner pt1 "\nOther corner: ")
      pt1   (trans pt1 1 2)
      pt2   (trans pt2 1 2)
      real1 (min (car pt1) (car pt2))
      real2 (max (car pt1) (car pt2))
      real3 (min (cadr pt1) (cadr pt2))
      real4 (max (cadr pt1) (cadr pt2))
      pt1   (trans (getvar "vsmin") 1 2)
      pt2   (trans (getvar "vsmax") 1 2))
(if bit1
   (progn
      (command ".zoom" "p" "ucs" "p")
      (#SVRST 2)))
(list
   (/ (- real1 (car pt1))  (- (car pt2)   (car pt1)))
   (/ (- real2 (car pt1))  (- (car pt2)   (car pt1)))
   (/ (- real3 (cadr pt1)) (- (cadr pt2)  (cadr pt1)))
   (/ (- real4 (cadr pt1)) (- (cadr pt2)  (cadr pt1)))))

;-----------------------------------------------------------------------------
;  GeT renderman Setup Options 2
;
;  (#GTSO2) -> : nil
;
;  Prompt for all RenderMan Setup block values.
;-----------------------------------------------------------------------------

(defun #GTSO2  (

   /
   item1       ;
   item2       ;
   bit1        ;  Valid Rate
   )

(setq item1 T)
(while item1
   (initget "Air Bucket Crop Destination Exposure FIlter FOrmat Merge Rate Samples")
   (setq item1 (getkword "\nAir/Bucket/Crop/Destination/Exposure/FIlter/FOrmat/Merge/Rate/Samples: "))
   (cond
      ((= item1 "Air")
         (#GTSHD 4 'att13 nil))
      ((= item1 "Bucket")
         (initget 6)                   ;  n > 0
         (setq real1 (cond
                  ((getint
                     (strcat "\nBucket X size <" (itoa (car att5)) ">: ")))
                  ((car att5))))
         (initget 6)                   ;  n > 0
         (setq real2 (cond
                  ((getint
                     (strcat "\nBucket Y size <" (itoa (cadr att5)) ">: ")))
                  ((cadr att5)))
               att5  (list real1 real2)))
      ((= item1 "Crop")
         (initget "Visual Numeric")
         (setq item2 (getkword "\nVisual/<Numeric>: "))
         (if (= item2 "Visual")
            (setq att12 (#GTCPV))
            (setq pt1 (#GTCPN "Left"   0.0   (car att12))
                  pt2 (#GTCPN "Right"  pt1   (cadr att12))
                  pt3 (#GTCPN "Bottom" 0.0   (caddr att12))
                  pt4 (#GTCPN "Top"    pt3   (cadddr att12))
                  att12 (list pt1 pt2 pt3 pt4))))
      ((= item1 "Destination")                ;  att11
         (princ "\nCurrent destination: ")
         (princ (strcase att11))
         (initget "FIle TIff Cpostscript Postscript VGa TGa TArga VIsta FRamebuffer")
         (setq item2 (getkword
            "\nFIle/TIff/Cpostscript/Postscript/VGa/TGa/TArga/VIsta/FRamebuffer: "))
         (if item2
            (setq att11 (strcase item2 T))))
      ((= item1 "Exposure")
         (initget 6)                   ;  n > 0
         (setq real1 (getreal (strcat
            "\nOutput gain factor <" (rtos att10 2 6) ">: ")))
         (if real1 (setq att10 real1))
         (initget 6)                   ;  n > 0
         (setq real1 (getreal (strcat
            "\nGamma correction <" (rtos att7 2 6) ">: ")))
         (if real1 (setq att7 real1)))
      ((= item1 "FIlter")
         (initget "Box Gaussian")
         (setq item2 (cond
                  ((getkword (strcat "\nPixel filter type?  "
                        (if (= (car att4) "box")
                           "Gaussian/<Box"
                           "Box/<Gaussian")
                        ">: ")))
                  ((car att4)))
               real1 (cond
                  ((progn
                     (initget 6)       ;  n > 0
                     (getreal (strcat "\nFilter X size <"
                           (rtos (cadr att4) 2 2)
                           ">: "))))
                  ((cadr att4)))
               real2 (cond
                  ((progn
                     (initget 6)       ;  n > 0
                     (getreal (strcat "\nFilter Y size <"
                           (rtos (caddr att4) 2 2)
                           ">: "))))
                  ((caddr att4)))
               att4  (list (strcase item2 T) real1 real2)))
      ((= item1 "FOrmat")
         (initget "Ascii Binary")
         (setq item2 (getkword
                  (strcat "\nRIB file format?  "
                          (if (= att6 "ascii")
                             "Binary/<Ascii"
                             "Ascii/<Binary")
                          ">: ")))
         (if item2 (setq att6 (strcase item2 T))))
      ((= item1 "Merge")
         (initget "Yes No")
         (setq item2 (getkword
                  (strcat "\nMerge framebuffer?  "
                          (if (zerop att8) "Yes/<No" "No/<Yes")
                          ">: "))
               att8  (cond
                  ((= item2 "No")   0)
                  ((= item2 "Yes")  1)
                  (T                att8))))
      ((= item1 "Rate")
         (setq bit1 nil)
         (while (not bit1)
            (setq real1
                     (getreal (strcat "\nShading rate <" (rtos att2 2 2) ">: ")))
            (if real1
               (if (and (>= real1 1.0) (<= real1 1e+36))
                  (setq bit1  T
                        att2  real1)
                  (prompt "\n*** Value must be in range 1.0 to 1e+36."))
               (setq bit1 T))))        ;  Keep old value.
      ((= item1 "Samples")
         (initget 6)                   ;  n > 0
         (setq real1 (cond
                  ((getreal (strcat  "\nNumber of X pixel samples <"
                                    (rtos (car att3) 2 2)
                                    ">: ")))
                  ((car att3))))
         (initget 6)                   ;  n > 0
         (setq real2 (cond
                  ((getreal (strcat  "\nNumber of Y pixel samples <"
                                    (rtos (cadr att3) 2 2)
                                    ">: ")))
                  ((cadr att3)))
               att3  (list real1 real2))))))

;-----------------------------------------------------------------------------
;  GeT renderman Setup Options 1
;
;  (#GTSO1) -> : nil
;
;  Prompt for initial RenderMan Setup block values.
;-----------------------------------------------------------------------------

(defun #GTSO1  (

   /
   item1       ;
   )

(setq att4  '("box" 1.0 1.0)     ;  Pixel filter.
      att6  "ascii"              ;  Format.
      att7  1.0                  ;  Gamma.
      att8  0                    ;  Merge framebuffer.
      att9  ""                   ;  Search path.
      att10 1.0                  ;  Gain.
      att11 "framebuffer"        ;  Destination
      att12 '(0.0 1.0 0.0 1.0)   ;  Crop.
      att13 '("nullatmo" ""))    ;  Atmosphere shader.
(close (open (strcat (cond (G:SHEV) ("")) "tempfile.$$4") "w"))
(initget "Fast Good Renderman")
(setq item1 (getkword "\nFast/Good/<RenderMan setup options>: "))
(mapcar  'set '(att2 att3 att5)
         (cond ((= item1 "Fast") '(1.0e+4  (1.0 1.0) (24 24) ))
               ((= item1 "Good") '(1.0     (4.0 4.0) (8 8)   ))
               (T                '(10.0    (2.0 2.0) (12 12) )))))

;-----------------------------------------------------------------------------
;  RenderMan SETUP.
;
;  (C:RMSETUP) -> nil
;
;  Main RenderMan Setup block definition.
;-----------------------------------------------------------------------------

(defun C:RMSETUP  ( /   #ERROR inspt att1 att2 att3 att4 att5 att6 att7
                        att8 att9 att10 att11 att12 att13 att14 real1 real2 int1
                        int2 item1 item2)

(#HEAD (#SVINS))
(initget 1)                            ;  No null
(while (member (setq att1 (#GTSTR "\nRenderMan setup name")) G:SETU)
   (prompt (strcat
         "\n*** RenderMan setup name \"" att1
         "\" is in use.\nTry a different name, or execute RMSCAN to reset.")))
(#GTSO1)
(#GTSO2)
(initget 1)                            ;  No null
(setq inspt (getpoint "\nRenderMan setup location: "))
(command ".insert"
         "rm_rcb"
         inspt
         (eval G:SCAL)
         ""
         "<<0"
         att1                          ;  Setup Name
         (rtos att2 2 6)               ;  Shading Rate
         (#PTSTR att3 " ")             ;  Pixel Samples
         (strcat  (car att4) " "       ;  Pixel Filter
            (rtos (cadr att4 2 6)) " "
            (rtos (caddr att4 2 6)))
         (strcat  (itoa (car att5)) " ";  Bucket Size
                  (itoa (cadr att5)))
         att6                          ;  Format
         (rtos att7 2 6)               ;  Gamma Correction Factor
         att8                          ;  Merge Framebuffer flag
         att9                          ;  Shader/Texture Search Path
         (rtos att10 2 6)              ;  Light intensity gain factor
         att11                         ;  Destination device (or file)
         (#PTSTR att12 " ")            ;  Crop Specification
                                       ;  Atmosphere Shader Instance
         (strcat "(\"" (car att13) "\" \"" (cadr att13) "\")"))
(#SHPRM 4)                             ;  Atmosphere Parameters
(#CLASH)
(#ADRSB att1)
(#TAIL))

;-----------------------------------------------------------------------------
;  STRing to PoinT
;
;  (#STRPT str) -> list
;
;  Convert a point string into a list.
;-----------------------------------------------------------------------------

(defun #STRPT  (

   str1        ;  Point string
   /
   str2        ;  Setvar name
   pt1         ;  Last point
   pt2         ;  Return value
   )

(#SVRST '(("BLIPMODE" . 0)))
(setq str2  (if G:R11 "lastpoint" "lastpt3d")
      pt1   (getvar str2)) ;  Save old value
(command ".setvar" str2 str1)    ;  Must be COMMAND
(#SVRST 1)
(setq pt2 (getvar str2))   ;  Get converted point
(setvar str2 pt1)          ;  Restore old value
pt2)                       ;  Return converted point

;-----------------------------------------------------------------------------
;  GeT Shader PaRameter default values
;
;  (#GTSPR int) -> nil
;
;  Read shader parameter values from attributes into tempfile.$$[n].  If old
;  format (< 10-9-90) fix.
;  list1 scoped from #GTDFL, it's NOT set by #NXATT when sent as parameter.
;-----------------------------------------------------------------------------

(defun #GTSPR  (

   int1        ;  Shader type to fetch
   /
   file1       ;  Shader parameter file handles ...
   file2       ;  tempfile.$$$
   str1        ;  First attrib text
   str2        ;  Character read from tempfile.$$$ (added to str1)
   )

(setq file1 (open (strcat
            (cond (G:SHEV) (""))
            "tempfile.$$"
            (itoa int1)) "w")
      str1  (#GTVAL 1 (entget (entnext (#GTVAL -1 list1)))))
(if (= (substr str1 1 2) "((")         ;))   We have OLD formatted attrib's
   (progn
      (princ ".")                      ;  A tiny note that somethings up
      (#NXATT 'list1)                  ;  Move to next attribute
      (setq file2 (open (strcat        ;  Open temporary file
                  (cond (G:SHEV) (""))
                  "tempfile.$$$") "w"))
      (princ str1 file2)               ;  Dump the first string
      (setq str1 "")                   ;  Reclaim string space.
      (repeat 3                        ;  Dump the remaining parameters
         (princ (#GTVAL 1 (#NXATT 'list1)) file2))
      (setq file2 (close file2)
            file2 (open (strcat
                  (cond (G:SHEV) (""))
                  "tempfile.$$$") "r"))
      (read-char file2)                      ;  Dump first left parenthesis 40
      (while (setq str2 (read-char file2))   ;  While we have a character ...
         (setq str1 (strcat str1 (chr str2)))
         (if (and (= str2 41)          ;  We've found the end of 1 parameter
                  (/= str1 ")"))       ;( definition, and we're not at the end.
            (progn
               (write-line str1 file1) ;  Write this parm to file
               (setq str1 ""))))       ;  and reset parm string
      (setq file2 (close file2)))      ;  We're done, #GTDFL will close file1
   (progn                              ;  Attrib's are up-to date (10-9-90)
      (repeat 4                        ;  Get Surface shader parameters
      (setq str1 (#GTVAL 1 (#NXATT 'list1))
            str1 (read (strcat "(" str1 ")")))
      (foreach item1 str1              ;  And write to parameter file...
         (prin1 item1 file1)
         (princ "\n" file1)))))
(setq file1 (close file1)))

;-----------------------------------------------------------------------------
;  GeT surface block DeFauLts
;
;  (#GTDFL ent) -> nil
;
;  Get default values from a surface block: RM_SDB.  Att variables scoped from
;  calling defun.
;-----------------------------------------------------------------------------

(defun #GTDFL  (

   list1       ;  Main entity list of Surface block
   /
   item1       ;  Temp value/list
   )

(princ "\nGathering information...")
(foreach item1 '(att1 att2 att3 att4 att5)
   (set item1 (#GTVAL 1 (#NXATT 'list1))))
(setq att5  (read att5))
(#GTSPR 3)                             ;  Get Surface shader parameters
(foreach item1 '(att7 att8)
   (set item1 (#GTVAL 1 (#NXATT 'list1))))
(setq att8  (read att8))
(#GTSPR 2)                             ;  Get Displacement shader parameters
(foreach item1 '(att10 att11 att12)
   (set item1 (#GTVAL 1 (#NXATT 'list1))))
(setq att12 (read att12))
(#GTSPR 4)                             ;  Get Atmosphere shader parameters
(setq att14 (#GTVAL 1 (#NXATT 'list1))
      att2  (atoi att2)
      att3  (#STRPT att3)
      att4  (#STRPT att4)
      att7  (atof att7)
      att10 (atof att10)
      att11 (atoi att11)
      list1 (read (strcat "(" att14 ")")))   ;  Re-use "list1" variable.
(foreach item1 '(pt1 pt2 pt3 pt4)      ;  Texture coordinates
   (set item1 (list (car list1) (cadr list1)))
   (setq list1 (cddr list1)))
(princ "done."))

;-----------------------------------------------------------------------------
;  RenderMan COPY
;
;  (C:RMCOPY) -> nil
;
;  Copy Surface block using the values contained in an existing block as the
;  default values of the new block.
;-----------------------------------------------------------------------------

(defun C:RMCOPY   ( /   #ERROR inspt att1 att2 att3 att4 att5 att6 att7
                        att8 att9 att10 att11 att12 att13 att14 pt1 pt2 pt3 pt4
                        list1 list2)

(#HEAD (#SVINS))
(setq list1 (#GTBLK "\nSelect a surface block to copy: "
                  "surface block"
                  '("RM_SDB")
                  T))
(#GTDFL list1)
(setq att1  (#GTDNM "\nNew surface block name")
      att2  (#GTDCO))
(while (eval (#GTSPO)) T)
(initget 1)                            ;  No null
(setq inspt (getpoint "\nSurface block location: "))
(#INSDB)
(#ADSDB att1 att2 nil)
(#TAIL))

;-----------------------------------------------------------------------------
;  RenderMan EDIT
;
;  (C:RMEDIT) -> nil
;
;  Edit an existing Surface block.
;-----------------------------------------------------------------------------

(defun C:RMEDIT   ( /   #ERROR inspt att1 att2 att3 att4 att5 att6 att7 att8
                        att9 att10 att11 att12 att13 att14 pt1 pt2 pt3 pt4
                        list1 list2 list3 item1 item2)

(#HEAD (#SVINS))
(setq list1 (#GTBLK "\nSelect a surface block to edit: "
                  "surface block"
                  '("RM_SDB")
                  T)
      item1 (#GTVAL -1 list1)          ;  Get entity name
      inspt '(0 0 0))
(#GTDFL list1)                         ;  Read in default values
(while (eval (#GTSPO)) T)              ;  Get revised values...
(#SVRST '(("BLIPMODE" . 0)))
(setq list3 (vports))
(if (and G:R11 (zerop (getvar "tilemode")))  ;  We could be in Paper Space...
   (if (eq (getvar "cvport") 1)        ;  We ARE in Paper Space...
      (#SVRST '(("GRIDMODE" . 0) ("UCSFOLLOW" . 0) ("UCSICON" . 0)))
      (progn                           ;  Remove Paper Space viewport...
         (setq list3 (append  (cdr (member (assoc 1 list3) list3))
                              (cdr (member (assoc 1 list3) (reverse list3))))
               list3 (reverse list3))  ;  Save SYS VAR's for each vport to
         (foreach item2 list3          ;  avoid redraws/regens of UCS CMD.
            (#SVRST (cons (cons "CVPORT" (car item2))
               '(("GRIDMODE" . 0)  ("UCSFOLLOW" . 0) ("UCSICON" . 0)))))))
   (foreach item2 list3                ;  We can't be in Paper Space so...
      (#SVRST (cons (cons "CVPORT" (car item2))
         '(("GRIDMODE" . 0)  ("UCSFOLLOW" . 0) ("UCSICON" . 0))))))
(command ".ucs" "e" item1)
(entdel item1)
(#INSDB)
(command ".ucs" "p")
(#TAIL))

;*****************************************************************************

(prompt "loaded.")

;-----------------------------------------------------------------------------
;  Check for ASCOMMON.LSP, G:SCAL set by C:RMSCAN contained in ASCOMMON.LSP
;-----------------------------------------------------------------------------
(if (null G:SCAL) (load "ascommon"))

;-----------------------------------------------------------------------------
;  Enable RenderMan prompting.
;-----------------------------------------------------------------------------
(setq G:RMAN T)

(if (not G:SHEV)
   (prompt (strcat   "\n*** Environment variable \"SHADE\" not set, "
                     "using current directory.")))

(prin1)