;*****************************************************************************
;*
;*    ASHADE.LSP
;*
;*    AutoShade commands for AutoCAD
;*
;*    Designed and implemented by Kelvin R. Throop in May of 1987
;*    6/28/88  CLH --  Modified for Release 10
;*    7/27/88  JSY & KWL -- Added CAMVIEW
;*    4/15/89  Larry Knott & Bill Adkison:
;*                   Added VCAMERA, VLIGHT, FINISH, the "Spotlight" option of
;*                   LIGHT, and the recording of Rel 10 viewing options in the
;*                   new camera:
;*                      - lens
;*                      - viewmode
;*                      - clipping planes
;*                      - view twist
;*                      - view size/aspect ratio
;*                   and the new light fields:
;*                      - intensity
;*                      - RGB color
;*                      - depth map size (RMAN only)
;*                      - cone angle (for spotlites only)
;*                      - cone delta (for spotlites only)
;*                      - beam dist. (for spotlites only)
;*
;*    NOTE: This module requires the presence of ASCOMMON.LSP.
;*
;*****************************************************************************
;*
;*    Global variables:
;*
;*    G:SVER   -  INT, AutoShade version. (v1.1 = 11, v2.0 = 20)
;*    G:SHEV   -  STR, SHADE path if valid, else nil.
;*    G:R11    -  BOOL, T if AutoCAD r11, nil otherwise.
;*    G:MODE   -  LIST, saved system variables and values.
;*    G:RMAN   -  BOOL, prompt for RenderMan attributes.
;*    G:SCAL   -  REAL, scale factor for block insertion.
;*    G:SETU   -  LIST, setup block names.
;*
;*    Temporary files used:
;*
;*    tempfile.$$$   -  Utility temp file
;*    tempfile.$$a   -  Surface Property/Finish names
;*    tempfile.$$[n] -  Shader parameters
;*
;*****************************************************************************

(vmon)
(prompt "\nLoading ashade.lsp...")

;-----------------------------------------------------------------------------
;  SeT TARget point.
;
;  (#SETAR) -> nil
;
;  Creates the current view with DVIEW if it wasn't for C:VCAMERA and C:VLIGHT.
;-----------------------------------------------------------------------------

(defun #SETAR  (

   /
   pt1         ;  VIEWCTR
   pt2         ;  TARGET
   )

(setq pt1 (trans (getvar "viewctr") 1 2)
      pt2 (trans (getvar "target") 1 2))
(if (> (distance (list (car pt1) (cadr pt1))
                 (list (car pt2) (cadr pt2))) 1e-6)
   (progn
      (prompt "\nSetting up view with DVIEW...")
      (command ".dview" "" ""))))

;-----------------------------------------------------------------------------
;  AutoShade Camera specification and insert routines.
;
;  (C:CAMERA) ->
;  (C:VCAMERA) ->
;
;  Front end in specifing a camera.
;-----------------------------------------------------------------------------

(defun C:VCAMERA  ( /   #ERROR)

(#HEAD (#SVINS))
(if (and (= 0 (getvar "tilemode"))
         (= 1 (getvar "cvport")))      ; Exit if in PAPER SPACE
   (prompt "** Command not allowed in Paper space **")
   (#CAMER T))
(#TAIL))

;-----------------------------------------------------------------------------

(defun C:CAMERA   ( /   #ERROR)

(#HEAD (#SVINS))
(#CAMER nil)
(#TAIL))

;-----------------------------------------------------------------------------

(defun #CAMER  (

   bit1        ;  C:VCAMERA?
   /
   bit2        ;  In prespective?
   inspt       ;  Camera point
   ang1        ;
   att1        ;  Name
   att3        ;  Look-at
   att6        ;  Lenslength
   att12       ;  Aspect ratio, viewsize
   )

(graphscr)
(if (= (logand (getvar "viewmode") 1) 1)
   (setq bit2 T))
(if (and bit1 (not bit2))
   (#SETAR))
(setq att1  (#GTSTR "Enter camera name")
      att3  (if bit1
               (getvar "target")
               (#GTP3D 1 nil "\nEnter target point: "))
      inspt (if bit1
               (mapcar '+ att3 (getvar "viewdir"))
               (#GTP3D 1 att3 "\nEnter camera location: "))
      ang1  (* (/ 180 pi) (- (angle inspt att3) (/ pi 2.0)))
      att3  (trans att3 1 0)
      att6  (if bit2
               (getvar "lenslength")
               50.0)                   ;  Force 50.0mm lens default.
      att12 (mapcar '- (trans (getvar "vsmax") 1 2)
                       (trans (getvar "vsmin") 1 2))
      att12 (rtos (/ (car att12) (cadr att12)) 2 6)
      att12 (strcat att12 " " (rtos (getvar "viewsize") 2 6)))
(command
      ".insert"
      "camera"
      inspt
      (eval G:SCAL)
      ""
      (strcat "<<" (rtos ang1 2 6))
      att1
      ""
      (rtos (car att3) 2 6)
      (rtos (cadr att3) 2 6)
      (rtos (caddr att3) 2 6)
      (rtos att6 2 6)
      (rtos (getvar "frontz") 2 6)
      (rtos (getvar "backz") 2 6)
      (rtos (* (/ 180 pi) (getvar "viewtwist")) 2 6)
      (getvar "viewmode")
      ""
      att12)
(#CLASH))

;-----------------------------------------------------------------------------
;  GeT Light Options 1
;
;  (#GTLO1 bool) -> nil
;
;  Get AutoShade options for light insertion.
;-----------------------------------------------------------------------------

(defun #GTLO1  (

   bit1        ;  C:VLIGHT ?
   )

(graphscr)
(if (= (logand (getvar "viewmode") 1) 1)
   (setq bit2 T))
(if (and bit1 (not bit2))
   (#SETAR))
(setq att6  1.0      ;  Intensity
      att7  '(1 1 1) ;  Color
      att8  0        ;  Depth map size 2^(value+6)
      att11 10.0     ;  Cone angle
      att12 0.0      ;  Cone delta angle
      att13 0.0      ;  Beam distribution
      att1  (#GTSTR "Enter light name"))
(initget
   (if G:RMAN
      "Point Directed Spotlight"
      "Point Directed"))
(setq str1
      (getkword
         (if G:RMAN
            "\nPoint source, Directed, or Spotlight <P>: "
            "\nPoint source or Directed <P>: ")))
(cond
   ((or (not str1) (= str1 "Point"))
      (setq str2 "overhead"
            att2 " "
            att3
            (if bit1
               (getvar "target")
               '(0 0 0))))
   (T (if (= str1 "Directed")
         (setq str2 "direct"
               att2 "Parallel")
         (setq str2 "sh_spot"
               att2 "Spot"))
      (setq att3
            (if bit1
               (getvar "target")
               (#GTP3D 1 nil "\nEnter light aim point: ")))))
(setq att14
      (if bit1
         (mapcar '+ att3 (getvar "viewdir"))
         (#GTP3D 1 (if (= att2 " ") nil att3) "\nEnter light location: ")))
(setq att6
      (cond
         ((getdist "\nLight intensity <1.00>: "))
         (1.0)))
(if (= str2 "overhead")
   (setq ang1 0
         att3 '(0 0 0))                ;  Reset after att14 set for VLIGHT.
   (setq ang1 (* (/ 180 pi) (- (angle att14 att3) (/ pi 2)))
         att3 (trans att3 1 0))))

;-----------------------------------------------------------------------------
;  GeT Light Options 2
;
;  (#GTLO2) -> nil
;
;  Get RenderMan options for light insertion.
;-----------------------------------------------------------------------------

(defun #GTLO2  ()

(setq att7
      (cond
         ((#GT0-1 "Light color (RGB)" '(1 1 1) nil))
         ('(1.0 1.0 1.0))))
(initget "Yes No")
(if (= (getkword "\nDoes this light cast a shadow?  Yes/<No>: ") "Yes")
   (progn
      (while (or (< att8 1) (> att8 6))
         (initget 6)             ;  n > 0
         (setq att8
               (cond
                  ((getint "\nDepth map size (1-6) <1>: "))
                  (1)))
         (if (or (< att8 1) (> att8 6))
            (prompt "\n*** Value must be in range 1-6.")))))
(if (= str1 "Spotlight")
   (progn
      (initget 6)                      ;  n > 0
      (setq att11
            (cond
               ((getreal "\nCone angle <10.00>: "))
               (10.0)))
      (initget 4)                      ;  n >= 0
      (setq att12
            (cond
               ((getreal "\nCone delta angle <0.0>: "))
               (0.0)))
      (initget 4)                      ;  n >= 0
      (setq att13
            (cond
               ((getreal "\nBeam distribution <0.0>: "))
               (0.0))))))

;-----------------------------------------------------------------------------
;  INsert LiGhT
;
;  (#INLGT) -> nil
;
;  Insert light routine.
;-----------------------------------------------------------------------------

(defun #INLGT  ()

(command   ".insert"
           str2                        ;  Light block name
           att14                       ;  Light location
           (eval G:SCAL)               ;  Scale
           ""                          ;  *Not used*
           (strcat "<<" (rtos ang1 2 6))  ;  Rotation
           att1                        ;  Light name
           att2                        ;  Light type
           (rtos (car att3) 2 6)       ;  Look-at X
           (rtos (cadr att3) 2 6)      ;  Look-at Y
           (rtos (caddr att3) 2 6)     ;  Look-at Z
           (rtos att6 2 6)             ;  Intensity
           (#PTSTR att7 ",")           ;  Color
           att8                        ;  Depth map size
           "-1"                        ;  Not used
           "-1")                       ;  Not used
(if (= str1 "Spotlight")
   (command    (rtos att11 2 6)        ;  Cone angle
               (rtos att12 2 6)        ;  Cone delta angle
               (rtos att13 2 6)        ;  Beam distribution
               (#PTSTR (trans att14 1 0) ","))))   ;  Light location

;-----------------------------------------------------------------------------
;  Main AutoShade light specification and insert routines.
;
;  (C:LIGHT) ->
;  (C:VLIGHT) ->
;
;  Front end in specifing a light.
;-----------------------------------------------------------------------------

(defun C:VLIGHT   ( /   #ERROR bit1 bit2 str1 str2 ang1 att1 att2 att3 att6
                        att7 att11 att12 att13 att14 )

(#HEAD (#SVINS))
(if (and (= 0 (getvar "tilemode"))
         (= 1 (getvar "cvport")))      ; Exit if in PAPER SPACE
   (prompt "** Command not allowed in Paper space **")
   (progn
      (#GTLO1 T)
      (if G:RMAN  (#GTLO2))            ;  Prompt for RMAN attrib's ?
      (#INLGT)
      (#CLASH)))
(#TAIL))

;-----------------------------------------------------------------------------

(defun C:LIGHT   ( /    #ERROR bit1 bit2 str1 str2 ang1 att1 att2 att3 att6        ;
                        att7 att11 att12 att13 att14 )

(#HEAD (#SVINS))
(#GTLO1 nil)
(if G:RMAN  (#GTLO2))                  ;  Prompt for RMAN attrib's ?
(#INLGT)
(#CLASH)
(#TAIL))

;-----------------------------------------------------------------------------
;  SCENE specification and insert routine.
;
;  (C:SCENE) ->
;
;  One camera and optional lights comprise a scene.
;-----------------------------------------------------------------------------

(defun C:SCENE    ( /   #ERROR str1 str2 str3 item1 list1 list2 inspt
                        real1)

(#HEAD (#SVINS))
(graphscr)
(setq str1  (#GTSTR "Enter scene name")
      str2  (#GTBLK  "\nSelect the camera: "
                     "camera"
                     '("CAMERA")
                     T)
      str2  (#GTVAL 1 (#NXATT 'str2))
      str3  "Lights:"
      list1 T
      list2 '())
(while list1
   (setq list1 (#GTBLK  "\nSelect a light: "
                        "light"
                        '("OVERHEAD" "DIRECT" "SH_SPOT")
                        nil))
   (if list1
      (if (member (setq item1 (#GTVAL 1 (#NXATT 'list1))) list2)
         (prompt " already selected.")
         (setq list2 (append list2 (list item1))
               str3 (strcat str3 " " item1)))))
(setq inspt (#GTP3D 1 nil "\nEnter scene location: ")
      real1 (/ (eval G:SCAL) 8.5))
(#SVRST '(("BLIPMODE" . 0)))
(#INSCN "clapper" inspt real1 str1 (strcat "Camera: " str2) str3)
(#INSCN "shot" inspt real1 "CAMERA" str2 str1)
(foreach item1 list2
   (setq inspt (cons (+ (car inspt) (* 0.8 real1)) (cdr inspt)))
   (#INSCN "shot" inspt real1 "LIGHT" item1 str1))
(prompt (strcat "\nScene " str1 " included."))
(#TAIL))

;-----------------------------------------------------------------------------
;  INsert SCeNe object
;
;  (#INSCN str point real str str str) -> nil
;
;  Inserts either CLAPPER or SHOT block.
;-----------------------------------------------------------------------------

(defun #INSCN  (

   str1        ;  Block name
   inspt       ;  Insertion point
   real1       ;  Scale factor
   att1        ;  Scene name  / Object name
   att2        ;  Camera name / Object type
   att3        ;  Lights      / Scene name
   )

(command    ".insert"
            str1
            inspt
            real1
            ""
            "<<0"
            att1
            att2
            att3)
(#CLASH))

;-----------------------------------------------------------------------------
;  CAMera VIEW
;
;  (C:CAMVIEW) ->
;
;  Restores the view contained an a camera.
;-----------------------------------------------------------------------------

(defun C:CAMVIEW  ( /   #ERROR list1)

(#HEAD (#SVINS))
(if (and (= 0 (getvar "tilemode"))
         (= 1 (getvar "cvport")))      ; Exit if in PAPER SPACE
   (prompt "** Command not allowed in Paper space **")
   (progn
      (setq list1 (#GTBLK "\nSelect the camera: " "camera" '("CAMERA") T))
      (#STVEW list1)))
(#TAIL))

;-----------------------------------------------------------------------------
;  #STVEW
;
;  (#STVEW ename) -> nil
;
;  Gather all information from the selected camera and use DVIEW to set up the
;  view contained in the camera.
;-----------------------------------------------------------------------------

(defun #STVEW  (

   list1       ;
   /
   list2       ;
   list3       ;
   int1        ;
   att1 att2 att3 att4 att5 att6 att7 att8 att9 att10
   real1       ;
   bit1        ;
   bit2        ;
   var1        ;
   pt1         ;
   pt2         ;
   )

(setq int1  (if (zerop (getvar "worldview")) 1 0)  ;  Control DVIEW/TRANS
      att6  0.0
      att7  0.0
      att8  0.0
      att9  "0"
      att10 "1.28 9.0"
      list2 list1)
(setq list3
      '( ("SNAME"      . att1)
         ("LAX"        . att2)
         ("LAY"        . att3)
         ("LAZ"        . att4)
         ("LENS"       . att5)
         ("FRONTCLIPZ" . att6)
         ("BACKCLIPZ"  . att7)
         ("TWIST"      . att8)
         ("VIEWMODES"  . att9)
         ("IMGFN"      . att10)  ;  ASPECT/VIEWSIZE
      ))
(while (#NXATT 'list1)
   (if (setq var1 (#GTVAL (#GTVAL 2 list1) list3))
      (set var1 (#GTVAL 1 list1))))
(setq pt1   (trans (mapcar 'atof (list att2 att3 att4)) 0 int1)
      pt2   (trans (#GTVAL 10 list2) (#GTVAL -1 list2) int1)
      real1 (distance pt1 pt2)
      pt1   (#PTSTR pt1 ",")        ;  6 Decimal places for both !
      pt2   (#PTSTR pt2 ",")
)
(if att5     ;  New Camera block.
   (setq bit1 T
         att9  (atoi att9)
         bit2  (= (logand att9 1) 1)
         att6  (if (= (logand att9 2) 2)
                  (if (= (logand att9 16) 16)
                     att6
                     "eye")
                  "off")
         att7  (if (= (logand att9 4) 4) att7 "off")
         att10 (strcat "(" att10 ")")
         att10 (cadr (read att10)))
   (setq att5 50.0))
(cond
   ((eq pt1 pt2)
      (prompt "\n*** ERROR: Camera and Target points are coincident."))
   (T (command ".dview" "" "of" "po" pt1 pt2)
      (cond
         ((and bit1 bit2)              ;  New camera in perspective
      (command "tw" att8 "cl" "f" att6 "cl" "b" att7 "d" real1 "z" att5 ""))
         (bit1                         ;  New camera in orthographic
      (command "tw" att8 "cl" "f" att6 "cl" "b" att7 ""
               ".zoom" "c" "" att10))
         (T (command "d" real1 ""))))))   ;  Old camera

;-----------------------------------------------------------------------------
;  GeT autoshade Surface Finish Options
;
;  (#GTSFO) -> str
;
;  Surface Finish options are specified as parameters to the "nullsurf" surface
;  shader.
;-----------------------------------------------------------------------------

(defun #GTSFO  (

   /
   real1       ;  Ambient factor
   real2       ;  Diffuse factor
   real3       ;  Specular factor
   real4       ;  Specular exponent
   real5       ;  Temp variable
   var1        ;  Temp symbol
   str1        ;  Prompt string
   file1       ;  Surface parameter file
   item1       ;  Each parameter
   )

(setq real1 0.3
      real2 0.7
      real3 0.0
      real4 10.0)
(mapcar '(lambda (var1 str1 / real5)   ;  Negative values allowed !!!
   (setq real5 (getreal (strcat "\n" str1 " <" (rtos (eval var1) 2 2) ">: ")))
   (if real5 (set var1 real5)))
   '(real1 real2 real3 real4)
   '("Ambient factor" "Diffuse factor" "Specular factor" "Specular exponent"))
(setq real4 (if (<= real4 0.0001)
            10000.0
            (/ 1.0 real4))
      att6  (mapcar '(lambda (str1 var1)
            (list 11 str1 (rtos (eval var1) 2 6)))
            '("Ka" "Kd" "Ks" "roughness")
            '(real1 real2 real3 real4))
      file1 (open (strcat
            (cond (G:SHEV) (""))
            "tempfile.$$3") "w"))
(prin1 (car att6) file1)
(foreach item1 (cdr att6)
   (print item1 file1))
(setq file1 (close file1)))

;-----------------------------------------------------------------------------
;  surface FINISH
;
;  (C:FINISH) -> nil
;
;  Main surface FINISH routine.
;-----------------------------------------------------------------------------

(defun C:FINISH   ( /   #ERROR inspt att1 att2 att3 att4 att5 att6 att7
                        att8 att9 att10 att11 att12 att13 pt1 pt2 pt3 pt4)

(#HEAD (#SVINS))
(setq att1 (#GTDNM "\nSurface finish name"))
(while (not att2)
   (setq att2 (#GTDCO)))
(#SBDFL)
(#GTSFO)
(initget 1)                            ;  No null
(setq inspt (getpoint "\nSurface finish location: "))
(#INSDB)
(#ADSDB att1 att2 nil)
(#TAIL))

;-----------------------------------------------------------------------------
;  ChecK BLocKs
;
;  (#CKBLK) ->
;
;  Check for pre-version 2.0 CAMERA, DIRECT and OVERHEAD block definitions
;  and update if necessary.  #CKBLK is set to nil once executed.
;-----------------------------------------------------------------------------

(defun #CKBLK  (

   /
   item1       ;  List of block name and new attribute tag
   list1       ;  Current entity list
   bit1        ;  Reached the end of current block definition?
   bit2        ;  Current block is up to date
   bit3        ;  Blocks were updated
   )

(#HEAD (#SVINS))
(princ "\nChecking AutoShade blocks...")
(#SVRST '(("REGENMODE" . 0)))
(foreach item1 '(("camera" "LENS") ("overhead" "COLOR") ("direct" "COLOR"))
   (cond
      ((setq bit1    nil               ;  If block IS in drawing
             bit2    nil
             list1   (tblsearch "block" (car item1)))
         (setq list1 (entget (#GTVAL -2 list1)))
         (while (not bit1)             ;  While more attributes ...
            (if (equal (#GTVAL 2 list1) (cadr item1))
               (setq bit2  T           ;  New block ...
                     bit1  T)          ;  and we're done
               (if (setq list1 (entnext (#GTVAL -1 list1)))
                  (setq list1 (entget list1))
                  (setq bit1 T))))     ;  No more attributes
         (cond
            ((not bit2)                ;  Old block so ...
               (setq bit3 T)
               (prompt (strcat "\nUpdating \"" (car item1) "\"..."))
               (command ".insert" (strcat (car item1) "="))
               (command)
               (prompt "done."))))))
(prompt
   (if bit3
      "\nUpdate completed."
      "done."))
(#TAIL))

;*****************************************************************************

(prompt "loaded.")
(setq G:SVER 20)                       ;  (v1.1 = 11, v2.0 = 20)

;-----------------------------------------------------------------------------
;  Check for ASCOMMON.LSP, G:SCAL set by C:RMSCAN contained in ASCOMMON.LSP
;-----------------------------------------------------------------------------
(if (null G:SCAL) (load "ascommon"))

;-----------------------------------------------------------------------------
;  Check for latest versions of CAMERA, DIRECT and OVERHEAD.
;-----------------------------------------------------------------------------
(setq #CKBLK (#CKBLK))

(prin1)