;*****************************************************************************
;*
;*    ASCOMMON.LSP
;*
;*    Common functions for ASHADE.LSP and RMAN.LSP.
;*
;*    Designed and Implemented by Larry Knott;  4/89
;*
;*    NOTE: This module is required for ASHADE.LSP and RMAN.LSP.
;*
;*****************************************************************************
;*
;*    Global variables:
;*
;*    G:SVER   -  INT, AutoShade version. (v1.1 = 11, v2.0 = 20)
;*    G:SHEV   -  STR, SHADE path if valid, else nil.
;*    G:R11    -  BOOL, T if AutoCAD r11, nil otherwise.
;*    G:MODE   -  LIST, saved system variables and values.
;*    G:RMAN   -  BOOL, prompt for RenderMan attributes.
;*    G:SCAL   -  REAL, scale factor for block insertion.
;*    G:SETU   -  LIST, setup block names.
;*
;*    Temporary files used:
;*
;*    tempfile.$$$   -  Utility temp file
;*    tempfile.$$a   -  Surface Property/Finish names
;*    tempfile.$$[n] -  Shader parameters
;*
;*****************************************************************************

(vmon)
(prompt "\nLoading ascommon.lsp...")

;-----------------------------------------------------------------------------
;  2 UNIX
;
;  (#2UNIX str)   -> str
;
;  Replace all instances of "\\" to "/" in a string.
;-----------------------------------------------------------------------------

(defun #2UNIX  (

   str1        ;  Text string to convert
   /
   int1        ;  Number of characters
   int2        ;  Current character
   )

(setq int1 (1- (strlen str1))
      int2 1)
(if (= (substr str1 1 1) "\\")
   (setq str1 (strcat "/" (substr str1 2))))
(repeat int1
   (setq int2 (1+ int2))
   (if (= (substr str1 int2 1) "\\")
      (setq str1 (strcat
               (substr str1 1 (1- int2)) "/" (substr str1 (1+ int2))))))
str1)

;-----------------------------------------------------------------------------
;  Change to Layer ASHade
;
;  (#CLASH) -> nil
;
;  Changes the last entity in the database to layer "ASHADE".
;-----------------------------------------------------------------------------

(defun #CLASH  (

   /
   list1       ;  Entity list of last inserted entity (INSERT)
   )

(setq list1 (entget (entlast)))
(if (/= (#GTVAL 8 list1) "ASHADE")
   (entmod (subst (cons 8 "ASHADE") (assoc 8 list1) list1))))

;-----------------------------------------------------------------------------
;  GeT VALue
;
;  (#GTVAL int elist) -> int/str/real
;
;  Returns the associated group code value.
;-----------------------------------------------------------------------------

(defun #GTVAL  (

   int1        ;  Group code
   list1       ;  List
   )

(cdr (assoc int1 list1)))

;-----------------------------------------------------------------------------
;  Get a Point that is 3D
;
;  (#GTP3D int list str) -> list
;
;  Makes the selected point the last point, so subsequent points can be entered
;  with the "@" modifier.  int is 0/1 for null input, list is point to rubber
;  band from.
;-----------------------------------------------------------------------------

(defun #GTP3D  (

   int1        ;  Null responses
   pt1         ;  Optional point to rubber-band from
   str1        ;  Prompt
   /
   pt2         ;  Selected point
   str2        ;  Setvar name
   )

(initget (+ int1 24))                  ;  3D points, no limits
(setq str2  (if G:R11 "lastpoint" "lastpt3d")
      pt2   (if pt1
               (getpoint pt1 str1)
               (getpoint str1)))
(if pt2
   (setvar str2 pt2)))

;-----------------------------------------------------------------------------
;  PoinT to STRing
;
;  (#PTSTR list str) -> str
;
;  Return string given delimeter and list of reals (1 or more reals.) with 6
;  decimal places of accuracy.
;-----------------------------------------------------------------------------

(defun #PTSTR  (

   pt1         ;  Point list
   str1        ;  Delimiter
   /
   str2        ;  Point string
   item1       ;  Temp
   )

(setq str2 (rtos (car pt1) 2 6))
(foreach item1 (cdr pt1)
   (setq str2 (strcat str2 str1 (rtos item1 2 6)))))

;-----------------------------------------------------------------------------
;  ONE PaGe listed at a time
;
;  (#1PAGE) -> T/nil
;
;  Tests for one pagefull or 15 lines, if T, prompt to continue.  If not a
;  pagefull, or if user wants to continue, return T, else nil.  COUNT from
;  calling function.
;-----------------------------------------------------------------------------

(defun #1PAGE  (

   int1        ;
   /
   str1        ;
   )

(if (zerop (rem int1 15))
   (progn
      (princ "\n-- Press any character when done or RETURN for more --\n\n")
      (setq str1 (grread))
      (if (and (= (car str1) 2)
               (member (cadr str1) '(32 13 10)))
         T))
   T))

;-----------------------------------------------------------------------------
;  LiSt CoLoRs
;
;  (#LSCLR) -> nil
;
;  Color names in colors.txt must be lower case.  Color values in colors.txt
;  must be in range 0 to 1.
;-----------------------------------------------------------------------------

(defun #LSCLR  (

   /
   file1       ;  Colors.txt
   int1        ;
   list1       ;
   )

(cond
   ((setq file1 (findfile "colors.txt"))
      (textscr)
      (setq file1 (open file1 "r")
            int1 1)
      (princ "\nRGB colors listed in color file:")
      (princ "\n--------------------------------\n")
      (while (and (#1PAGE (setq int1 (1+ int1)))
                  (setq list1 (read-line file1)))
         (setq list1 (read list1))
         (princ (substr (strcat (car list1) "                ") 1 16))
         (princ "  ")
         (princ (#PTSTR (cadr list1) " "))
         (terpri))
      (setq file1 (close file1)))
   (T    (princ "\n*** Color file not present."))))

;-----------------------------------------------------------------------------
;  GeT BLocK
;
;  (#GTBLK str str list bool) -> list/nil
;
;  If retry bit is on, loop until one of the requested blocks are selected.
;  If found, print name attribute and return insert entity list, else return
;  nil.
;-----------------------------------------------------------------------------

(defun #GTBLK  (

   str1        ;  Prompt
   item1       ;  Object name
   str2        ;  List of valid block names
   bit1        ;  Retry bit
   /
   bit2        ;  Requested block not selected.
   list1       ;  (entsel) list
   list2       ;  Valid, selected block entity list
   )

(setq bit2 T)
(while bit2
   (setq list2 nil)
   (if (setq list1 (entsel str1))
      (if (and (= (#GTVAL 0 (setq list2 (entget (car list1)))) "INSERT")
               (member (#GTVAL 2 list2) str2))
         (setq bit2 nil)
         (princ (strcat "Selected object is not a " item1 ".")))
      (if bit1
         (princ "No object found.")
         (setq bit2 nil))))
(cond
   (list2
      (princ (#GTVAL 1 (entget (entnext (car list1)))))
      list2)))

;-----------------------------------------------------------------------------
;  GeT CoLoR from colors.txt
;
;  (#GTCLR file) -> list/nil
;
;  If a valid color name (one with an entry in colors.txt) was supplied, its
;  color is returned, else nil.
;-----------------------------------------------------------------------------

(defun #GTCLR  (

   str1        ;  FINDFILE'd name of color file.
   /
   file1       ;  colors.txt file handle.
   str2        ;
   list1       ;
   bit1        ;  Requested color found
   )

(if str1
   (while (not bit1)
      (setq str2 (strcase (getstring "\n?/Color name: ") T))
      (cond
         ((= str2 "")   (setq bit1 T))
         ((= str2 "?")  (#LSCLR))
         (T (setq file1 (open str1 "r"))
            (princ "\nSearching color file...")
            (while (and str2 (setq list1 (read-line file1)))
               (setq list1 (read list1))
               (if (= str2 (car list1))
                  (setq str2 nil)))
            (princ "done.")
            (if str2
                  (princ (strcat "\n*** Color \"" (strcase str2)
                                 "\" not defined in color file."))
                  (setq bit1 T))
            (setq file1 (close file1)))))
   (prompt "\n*** Color file not present."))
(if (not str2)
   (cadr list1)))

;-----------------------------------------------------------------------------
;  GeT point in range 0-1.
;
;  (#GT0-1 str point bool) -> point/nil
;
;  Validate RGB Color triplet or Opacity triplet or any RtPoint. Given a default
;  list of three reals, return new list if all lie between 0 and 1.
;
;  bool has the following meanings:
;
;     nil   :  Values must be in range 0-1 (Light color, SPB Opacity)
;     0     :  Keywords "Use" and "-1" return '(-1 -1 -1) (SPB Color)
;     1     :  Keyword "-1" returns '(-1 -1 -1) (Shader color parameter)
;
;-----------------------------------------------------------------------------

(defun #GT0-1  (

   str1        ;  Prompt string
   pt1         ;  Default value
   bit1        ;  Set allowable input (see above)
   /
   pt2         ;  Selected color
   )

(setq str1  (strcat "\nName/"          ;  Colors.txt is an option
               (if (and bit1 (zerop bit1))
                  "Use ACI"           ;  SPB Color
                  "")                  ;  Light or shader parameter color
               str1                    ;  Prompt string
               (cond                   ;  There IS a default value...
                  (pt1 (strcat " <" (#PTSTR pt1 ",") ">: "))
                  (": "))))            ;  No default value
(while (not pt2)
   (initget                            ;  Set valid keywords
      (cond
         ((not bit1)    "Name")        ;  Light color, SPB Opacity
         ((zerop bit1)  "Name Use -1") ;  SPB Color
         (T             "Name -1")))   ;  Shader color parameter
   (#SVRST '(("BLIPMODE" . 0)))        ;  Don't flip to graphics screen
   (setq pt2 (getpoint str1))
   (#SVRST 1)
   (cond
      ((and bit1 (or (equal pt2 '(-1 -1 -1)) (= pt2 "-1") (= pt2 "Use")))
         (setq pt2 '(-1 -1 -1)))       ;  Take care of keywords
      ((= pt2 "Name")
         (setq pt2 (#GTCLR (findfile "colors.txt"))))
      (pt2
         (if (or (< (apply 'min pt2) 0.0) (> (apply 'max pt2) 1.0))
            (setq pt2 (prompt "\nValues must be in the range 0-1."))
            pt2))                      ;  Return validated triplet
      (T (setq pt2   T                 ;  Use default
               str1  nil)))))          ;  Exit condition for default

;-----------------------------------------------------------------------------
;  GeT STRing
;
;  (#GTSTR str) -> str
;
;  Acquire an 8 character MAX, uppercase string.  Ignores null input and returns
;  new value.  Characters cannot be any of the following: "*+,./:;<=>?[\]|
;-----------------------------------------------------------------------------

(defun #GTSTR  (

   str1        ;  Prompt
   /
   str2        ;  New string
   bit1        ;  Input OK
   int1        ;  Counter
   list1       ;  String converted to list of ASCII #
   )

(while (not bit1)
   (setq str2 (getstring (strcat "\n" str1 ": ")))
   (cond
      ((zerop (ascii str2)) nil)       ;  Ignore null input
      (T (setq str2  (strcase          ;  Trim UPPER-CASE input string
                     (substr str2 1 8))
               int1  1
               bit1  T                 ;  String is OK now...
               list1 '())
         (repeat (strlen str2)         ;  Create list of ASCII #
            (setq list1 (cons (ascii (substr str2 int1 1)) list1)
                  int1  (1+ int1)))
         (foreach int1  list1          ;  Compare each character
            (if (member int1
                  ;  "  *  +  ,  .  /  :  ;  <  =  >  ?  [  \  ]  |
                  '(34 42 43 44 46 47 58 59 60 61 62 63 91 92 93 124))
               (setq bit1  nil)))      ;  Bad character found in input string
         (if (not bit1)
            (princ "*** Invalid name.")))))
str2)

;-----------------------------------------------------------------------------
;  NeXt ATTribute
;
;  (#NXATT 'entlist) -> 'entlist/nil
;
;  Get NeXt ATTribute subroutine.  Given quoted sym name of "INSERT" or "ATTRIB"
;  entity list, reassigns sym to next attribute list or returns nil if next
;  entity is not an attribute.
;-----------------------------------------------------------------------------

(defun #NXATT  (

   var1        ;  Symbol
   /
   item1       ;  Entity list pointed to by symbol
   )

(setq item1 (eval var1)                ;  For memory's sake ...
      item1 (#GTVAL -1 item1)
      item1 (entget (entnext item1)))
(if (= (#GTVAL 0 item1) "ATTRIB")
   (set var1 item1)))

;-----------------------------------------------------------------------------
;  ADd Surface Definition Block
;
;  (#ADSDB str int bool) -> T/nil
;
;  Add unique name to tempfile.$$a and unique ACI to #SURF0.  If bit1 is set,
;  and ACI is used, post error message and return nil.
;-----------------------------------------------------------------------------

(defun #ADSDB  (

   str1        ;  Surface property name
   int1        ;  ACI
   bit1        ;  Check for duplicate ACI
   /
   file1       ;
   )

(cond
   ((and bit1 (setq str2 (#GTSBI int1 T)))
      (prompt (strcat
            "\n*** ERROR: ACI-" (itoa int1) " is referenced by both \"" str2
            "\" and \"" str1 "\".")))
   (T (setq #SURF0 (subst              ;  Add new ACI to list of defined
                  (append (list 'LIST) (list int1) (#SURF0))
                  (last #SURF0)        ;  Eval'ing #SURF0 pages it in so ...
                  #SURF0)              ;  works
            file1 (open (strcat
                  (cond (G:SHEV) ("")) ; SHADE dir or current
                  "tempfile.$$a") "a"))
      (write-line str1 file1)          ;  Add new name
      (setq file1 (close file1))
      T)))                             ;  Return value

;-----------------------------------------------------------------------------
;  ADd Renderman Setup Block
;
;  (#ADRSB str) -> nil
;
;  Add unique name to list of defined, or post error message and return nil.
;-----------------------------------------------------------------------------

(defun #ADRSB  (

   str1        ;  Setup name
   )

(cond
   ((not G:SETU)
      (setq G:SETU (list str1)))
   ((member str1 G:SETU)
      (prompt (strcat
            "\n*** ERROR: Setup name \"" str1 "\" is duplicated.")))
   (T (setq G:SETU (append G:SETU (list str1))))))

;-----------------------------------------------------------------------------
;  GeT Entities CoLor
;
;  (#GTECL str) -> int
;
;  Returns the color of the selected entity.
;-----------------------------------------------------------------------------

(defun #GTECL  (

   str1        ;  Prompt
   /
   list1       ;  Entity list
   )

(while (null (setq list1 (entsel str1)))
   (prompt "No object found."))
(setq list1 (entget (car list1)))
(cond
   ((#GTVAL 62 list1))
   ((#GTVAL 62 (tblsearch "layer" (#GTVAL 8 list1))))))

;-----------------------------------------------------------------------------
;  GeT surface Definition COlor
;
;  (#GTDCO) -> int
;
;  Get color index.  Used by C:FINISH, C:RMPROP and C:RMCOPY.
;-----------------------------------------------------------------------------

(defun #GTDCO  (

   /
   int1        ;  Option keyword and ACI
   str1        ;  Surface name
   )

(while (not int1)
   (initget 6 "Find Select")           ;  n > 0
   (setq int1 (getint "\nAutoCAD color index/Select/<Find>: "))
   (cond
      ((numberp int1)
         (if (> int1 255)
            (setq int1 (prompt "\n  Index must be between 1 and 255."))))
      ((= int1 "Select")
         (setq int1 (#GTECL
               "\n  Select the entity whose color index is to be used: ")))
      (T (while (< 255 (progn
                       (initget 6)     ;  n > 0
                       (setq int1
                       (cond ((getint "\n  Minimum index <1>: "))
                             (1)))))
            (prompt "\n  Index must be between 1 and 255."))
         (while (and (#GTSBI int1 nil) (< int1 256))
            (setq int1 (1+ int1)))
         (if (> int1 255)
            (progn
               (prompt "\n  No available ACI after that index.")
               (setq int1 -1)))))
   (cond
      ((or (not int1) (minusp int1))
         T)
      ((setq str1 (#GTSBI int1 T))
         (setq int1 (prompt (strcat
               "\n*** Color index ACI-" (itoa int1) " is in use by \"" str1
               "\".\nTry a different color, or execute RMSCAN to reset."))))
      (T (prompt (strcat "\n  Using color " (itoa int1) "."))
         int1)))
(if (minusp int1)
   nil
   int1))

;-----------------------------------------------------------------------------
;  GeT Surface Block Information
;
;  (#GTSBI int/str bool) -> int/str/nil
;
;  Given int, search #SURF0 (defun'd list of defined ACI's) for matching ACI and
;  if found and bool set, return its associated name, else return T, if not
;  found return nil.
;
;  Given str, search tempfile.$$a for matching name, and if found and bool set,
;  return its associated ACI, else return T, if not found return nil.
;-----------------------------------------------------------------------------

(defun #GTSBI  (

   item1       ;  ACI or name to check
   bit1        ;  Return associated item?
   /
   bit2        ;  Name not found yet
   file1       ;  tempfile.$$a (List of surface names)
   list1       ;  ACI and remainder found
   int1        ;  Location of name in tempfile.$$a
   str1        ;  Name to return
   )

(cond
   ((numberp item1)                    ;  Check ACI
      (setq list1 (member item1 (#SURF0)))
      (if (and bit1 list1)
         (setq file1 (open (strcat
                     (cond (G:SHEV) ("")) ; SHADE dir or current
                     "tempfile.$$a") "r")
               str1  (repeat (length list1)  ;  Get to name entry..
                        (read-line file1))   ;  and return name
               file1 (close file1)))
      (cond
         ((and bit1 list1)    str1)    ;  Return and Found
         (list1               T)))     ;  Found
   (T (setq int1  0                    ;  Check Surface name
            file1 (open (strcat
                  (cond (G:SHEV) ("")) ; SHADE dir or current
                  "tempfile.$$a") "r"))
      (while (and (not bit2) (setq str1 (read-line file1)))
         (if (= item1 str1)
            (setq bit2 T)              ;  Found!
            (setq int1 (1+ int1))))
      (setq file1 (close file1))
      (cond
         ((and bit1 bit2)              ;  Found and Return.
            (nth int1 (reverse (#SURF0))))
         (bit2                T)))))   ;  Found

;-----------------------------------------------------------------------------
;  GeT surface Definition NaMe
;
;  (#GTDNM str) -> str
;
;  Get unique Surface name.
;-----------------------------------------------------------------------------

(defun #GTDNM  (

   str1        ;  Prompt (including "\n")
   /
   str2        ;  Requested name
   int1        ;  Duplicated ACI
   )

(while (not str2)
   (setq str2 (#GTSTR str1))
   (if (setq int1 (#GTSBI str2 T))
      (setq str2  (prompt (strcat
                  "\n*** " (substr str1 2) " \"" str2
                  "\" is in use (using ACI-" (itoa int1)
                  ").\nTry a different name, or execute RMSCAN to reset.")))))
str2)

;-----------------------------------------------------------------------------
;  SHader PaRaMeters
;
;  (#SHPRM) -> nil
;
;  Supply Shader Parameters in tempfile.$$[n] to attributes.
;-----------------------------------------------------------------------------

(defun #SHPRM  (

   int1        ;  Enumerated shader type (1=light,2=disp,3=surf,4=atmo)
   /
   int2        ;  # Command's made (must be < 5) *OVERFLOW*
   str1        ;  Composite string
   str2        ;  Current parameter read.
   file1       ;  Shader parameter files
   )

(setq str1  ""
      int2  1
      file1 (open (strcat
            (cond (G:SHEV) (""))    ;  SHADE dir or current
            "tempfile.$$"
            (itoa INT1)) "r"))
(while (and (< int2 5)                       ;  Attrib slots are available &
            (setq str2 (read-line file1)))   ;  we have more parameters ...
   (if (< (+ (strlen str1) (strlen str2)) 256)  ;  Is there room?
      (setq str1 (strcat str1 str2))   ;  Add strings
      (progn                           ;  Otherwise supply value, and re-start
         (command str1)                ;  Supply parameters
         (setq int2  (1+ int2)         ;  Bump "command" counter
               str1  str2))))          ;  Reset string.
(setq file1 (close file1))             ;  Close parameter file
(cond                                  ;  Finish up ...
   ((and (= int2 5)                    ;  All attrib slots taken and
         (> (strlen str1) 0))          ;  we have more parameters ...
      (prompt "\n*** Shader parameter overflow."))
   ((> (strlen str1) 0)                ;  We have some parameters left
      (command str1)                   ;  Supply them
      (repeat (- 4 int2)
         (command "")))
   (T (repeat (- 5 int2)
         (command "")))))

;-----------------------------------------------------------------------------
;  INsert Surface Definition Block
;
;  (#INSDB) -> nil
;
;  Insert Surface Property/Finish block ("RM_SDB") routine.
;-----------------------------------------------------------------------------

(defun #INSDB  ()

(command ".insert"
         "rm_sdb"
         inspt
         (eval G:SCAL)
         ""
         "<<0"                         ;  No Rotation
         att1                          ;  Surface Property name
         att2                          ;  ACAD Color Index (int)
         (#PTSTR att3 ",")             ;  RGB Color
         (#PTSTR att4 ",")             ;  Opacity
                                       ;  Surface Shader
         (strcat "(\"" (car att5) "\" \"" (cadr att5) "\")"))
(#SHPRM 3)                             ;  Surface Parameters
(command (rtos att7 2 6)               ;  Shading Rate
                                       ;  Displacement Shader
         (strcat "(\"" (car att8) "\" \"" (cadr att8) "\")"))
(#SHPRM 2)                             ;  Displacement Parameters
(command (rtos att10 2 6)              ;  Displacement Bounds
         att11                         ;  Smooth meshes
                                       ;  Atmosphere Shader
         (strcat "(\"" (car att12) "\" \"" (cadr att12) "\")"))
(#SHPRM 4)                             ;  Atmosphere Parameters
(command (#PTSTR (append pt1 pt2 pt3 pt4) " ")) ;  Texture Coordinate
(#CLASH))

;-----------------------------------------------------------------------------
;  Surface Block DeFauLt
;
;  (#SBDFL) ->
;
;  Sets the default values for all attributes of the Surface block.
;-----------------------------------------------------------------------------

(defun #SBDFL  ( / file1)

;  Initionalize parameter files...
(close (open (strcat (cond (G:SHEV) ("")) "tempfile.$$2") "w"))   ;  Disp
(close (open (strcat (cond (G:SHEV) ("")) "tempfile.$$4") "w"))   ;  Atmo
(setq file1 (open (strcat              ;  Surf
            (cond (G:SHEV) (""))       ;  SHADE dir or current.
            "tempfile.$$3") "w"))
(prin1 (list 11 "Ka" "0.30") file1)    ;  No leading blank line
(print (list 11 "Kd" "0.70") file1)    ;  But add here...
(print (list 11 "Ks" "0.00") file1)
(print (list 11 "roughness" "0.10") file1)
(setq file1    (close file1)
      att3     '(-1.0 -1.0 -1.0)       ;  RGB Color
      att4     '(1.0 1.0 1.0)          ;  Opacity
      att5     '("nullsurf" "")        ;  Surface Shader
      att7     -1.0                    ;  Shading Rate
      att8     '("nulldisp" "")        ;  Displacement Shader
      att10    0                       ;  Displacement Bounds
      att11    0                       ;  Smooth Surface
      att12    '("" "")                ;  Atmosphere Shader
      pt1      '(0 0)                  ;  Texture Coordinates
      pt2      '(1 0)                  ;  ""
      pt3      '(0 1)                  ;  ""
      pt4      '(1 1)                  ;  ""
))

;-----------------------------------------------------------------------------
;  SCAN for surface Property blocks
;
;  (#SCANP) -> nil
;
;  Scan the drawing for Surface Property/Finish blocks and initialize
;  tempfile.$$a with the names, and #SURF0 with the ACI's.
;-----------------------------------------------------------------------------

(defun #SCANP  ()

(defun #SURF0  () (list))              ;  Let's re-defun #SURF0 ...
(setq file1    (close (open (strcat    ;  Dump current contents
               (cond (G:SHEV) (""))    ;  SHADE dir or current
               "tempfile.$$a") "w"))
      int1     0
      int2     0
      sset1    (if (tblsearch "BLOCK" "RM_SDB")
                  (ssget "x" '((0 . "INSERT") (2 . "RM_SDB"))))
      list1    (if sset1
                  (entget (ssname sset1 0)))
      real1    (cond
                  (G:SCAL)             ;  Keep user scale
                  (list1 (#GTVAL 41 list1))))
(if list1
   (repeat (sslength sset1)
      (setq str1 (#GTVAL 1 (#NXATT 'list1))           ;  "" if null ...
            int2 (atoi (#GTVAL 1 (#NXATT 'list1))))   ;  0 if null ...
      (if (equal str1 "") (setq str1 nil))   ;  so set to nil
      (if (zerop int2) (setq int2 nil))      ;  so set to nil
      (if (or (not str1) (not int2))
         (progn
            (prompt "\n*** ERROR: Surface block with missing name and/or ACI.")
            (setq bit1 T)))
      (if (and str1 int2)
         (if (not (#ADSDB str1 int2 T))
            (setq bit1 T)))
      (setq list1 (ssname sset1 (setq int1 (1+ int1)))
            list1 (if list1 (entget list1))))))

;-----------------------------------------------------------------------------
;  SCAN for Setup blocks
;
;  (#SCANP) -> nil
;
;  Scan the drawing for RenderMan Setup blocks and initialize G:SURF with the
;  names.
;-----------------------------------------------------------------------------

(defun #SCANS  ()

(setq G:SETU   nil
      int1     0
      int2     0
      sset1    (if (tblsearch "BLOCK" "RM_RCB")
                  (ssget "x" '((0 . "INSERT") (2 . "RM_RCB"))))
      list1    (if sset1
                  (entget (ssname sset1 0)))
      real1    (cond
                  (real1)
                  (list1 (#GTVAL 41 list1))))
(if list1
   (repeat (sslength sset1)
      (setq str1 (#GTVAL 1 (#NXATT 'list1)))
      (if (equal str1 "") (setq str1 nil))
      (if (not str1)
         (progn
            (prompt "\n*** ERROR: Setup block with missing name.")
            (setq bit1 T))
         (if (not (#ADRSB str1))
            (setq bit1 T)))
      (setq list1 (ssname sset1 (setq int1 (1+ int1)))
            list1 (if list1 (entget list1))))))

;-----------------------------------------------------------------------------
;  RenderMan SCAN
;
;  (C:RMSCAN) ->
;
;  Scan drawing and initialize Surface name (tempfile.$$a), ACI (#SURF0), and
;  RenderMan Setup name (G:SETU) lists for uniqueness verification.  Also sets
;  the default Block Scale Factor (G:SCAL), and RenderMan prompting mode
;  (G:RMAN).
;-----------------------------------------------------------------------------

(defun C:RMSCAN   ( /   #ERROR int1 int2 sset1 list1 str1 real1 bit1)

(#HEAD (#SVINS))
(prompt "\nSearching for AutoShade blocks...")
(#SCANP)                               ;  Scan for Surface blocks
(#SCANS)                               ;  Scan for Setup blocks
(if (not real1)                        ;  No block scale yet
   (foreach item1 '("CAMERA" "OVERHEAD" "DIRECT" "SH_SPOT")
      (if (not real1)
         (if (tblsearch "BLOCK" item1)
            (if (setq sset1 (ssget "x" (list '(0 . "INSERT") (cons 2 item1))))
               (setq real1 (#GTVAL 41 (entget (ssname sset1 0)))
                     sset1 nil))))))   ;  Reclaim selection set
(setq G:SCAL (if real1                 ;  Set the block scale
            real1
            '(setq G:SCAL (/ (getvar "viewsize") 10.0))))
(if G:SETU     (setq G:RMAN T))        ;  If RCB's exist then G:RMAN yes!
(if (not bit1) (prompt "done."))
(setq sset1 nil)                       ;  Reclaim selection set
(#TAIL))

;-----------------------------------------------------------------------------
;  AutoShade SCAle factor for blocks
;
;  (#ASSCA) -> list
;
;  Sets the scale factor for block insertion and echos factor.
;-----------------------------------------------------------------------------

(defun #ASSCA  (

   /
   bit1        ;  Valid block selected?
   list1       ;  Entity list
   str1        ;  Prompt
   real1       ;  Scale
   )

(initget 6 "Select")                   ;  n > 0
(setq str1  (strcat
            "\nSelect/AutoShade block scale factor"
            (if (listp G:SCAL)
               ": "
               (strcat " <" (rtos G:SCAL 2 2) ">: ")))
      real1 (getreal str1))
(cond
   ((= real1 "Select")                 ;  Set scale by selection
      (while (not bit1)
         (setq list1 (entsel "\nSelect AutoShade block with desired scale: "))
         (cond
            (list1                     ;  We picked something ...
               (setq list1 (entget (car list1)))
               (if (= (#GTVAL 0 list1) "INSERT")
                  (cond                ;  What type of INSERT?
                     ((member (#GTVAL 2 list1)
         '("CAMERA" "OVERHEAD" "DIRECT" "SH_SPOT" "RM_SDB" "RM_RCB"))
                        (setq bit1 T)) ;  and it's valid
                     ((member (#GTVAL 2 list1) '("CLAPPER" "SHOT"))
                        (prompt "Not a valid AutoShade block."))
                     (T (prompt "Not an AutoShade block.")))
                  (prompt "Not an AutoShade block.")))
            (T (prompt "No object found."))))
      (setq G:SCAL (princ (#GTVAL 41 list1))))  ;  Get the scale from the block
   (real1                              ;  Set scale
      (setq G:SCAL real1))))

;-----------------------------------------------------------------------------
;  AutoShade PROmpting
;
;  (#ASPRO) -> list/nil
;
;  Changes the mode in effect for RenderMan prompting. (rman)
;-----------------------------------------------------------------------------

(defun #ASPRO  (

   /
   str1        ;
   )

(initget "Yes No")
(setq str1  (getkword (strcat
            "\nPrompt for RenderMan attributes?  "
            (if G:RMAN "No/<Yes" "Yes/<No")
            ">: ")))
(if str1
   (setq G:RMAN (cond
               ((= str1 "Yes")   T)
               ((= str1 "No")    nil)))))

;-----------------------------------------------------------------------------
;  AutoShade RESiZe blocks
;
;  (#ASRES) -> nil
;
;  All AutoShade/RenderMan blocks except CLAPPER and SHOT are rescaled to the
;  current scale factor (G:SCAL).
;-----------------------------------------------------------------------------

(defun #ASRES  (

   /
   sset1       ;  Combined selection set
   sset2       ;  Interim selection set
   real1       ;  Scale factor
   item1       ;  Block name to resize
   int1        ;  Selection set counter
   int2        ;  # blocks to scale
   str1        ;  Partial prompt string
   list1       ;  Entity list
   )

(setq sset1 (ssadd)
      real1 (if (listp G:SCAL)   ;  Don't set the scale here!
               (/ (getvar "viewsize") 10.0)
               G:SCAL))
(princ "\nGathering AutoShade blocks...")
(foreach item1 '("CAMERA" "OVERHEAD" "DIRECT" "SH_SPOT" "RM_SDB" "RM_RCB")
   (if (setq   int1  -1
               sset2 nil
               sset2 (ssget "x" (list '(0 . "INSERT") (cons 2 item1))))
      (repeat (sslength sset2)
         (ssadd (ssname sset2 (setq int1 (1+ int1))) sset1))))
(princ "done.\n")
(setq int1    0
      int2   (sslength sset1)
      str1   (strcat " of " (itoa int2) ".\r")) ;  Stay on same line
(#SVRST '(("BLIPMODE" . 0) ("HIGHLIGHT" . 0)))
(repeat int2                           ;  Resize each block
   (setq list1 (entget (ssname sset1 int1)))
   (princ (strcat "Resizing block " (itoa (setq int1 (1+ int1))) str1))
   (command ".scale"                   ;  ENTMOD no good here: attrib's
               (#GTVAL -1 list1)       ;  Entity name
               ""
               (trans (#GTVAL 10 list1) (#GTVAL -1 list1) 1)
               (/ real1 (#GTVAL 41 list1))))
(#SVRST 2))

;-----------------------------------------------------------------------------
;  Main operating mode defaults.
;-----------------------------------------------------------------------------

(defun C:DEFAULTS ( /   #ERROR str1)

(#HEAD (#SVINS))
(initget "Resize Scale Prompts")
(setq str1 (getkword "\nResize AutoShade blocks/Scale/<Prompts>: "))
(cond
   ((= str1 "Resize")   (#ASRES))
   ((= str1 "Scale")    (#ASSCA))
   (T                   (#ASPRO)))
(#TAIL))

;-----------------------------------------------------------------------------
;  AutoShade and RenderMan ERRor function
;
;  (#ASERR str) -> nil
;
;  Set by #HEAD.
;-----------------------------------------------------------------------------

(defun #ASERR  (

   str1        ;
   )

(cond
   ((= str1 "console break")
      (princ "*Cancel*"))
   ((/= str1 "Function cancelled")  ;  CTRL-C is not an error, all others are.
      (prompt (strcat "\nError: " str1))))
(command)                           ;  DVIEW
(command ".undo" "end" ".undo" "1") ;  Reset drawing environment.
(if (= (type file1) 'FILE)          ;  shaders.txt & colors.txt always
   (setq file1 (close file1)))      ;     referred to as 'FILE1'.
(if (= (type file2) 'FILE)          ;  Parameter and temp files either file1
   (setq file2 (close file2)))      ;     or file2.
(#SVRST G:MODE)                     ;  Restore System Variables.
(setq *error*  #ERROR)              ;  Restore old *error* handler.
(foreach item1 '(                   ;  Unbind ALL variables.

   G:MODE
   ang1 ang2 ang3
   att1 att2 att3 att4 att5 att6 att7 att8 att9 att10 att11 att12 att13 att14
   bit1 bit2 bit3
   int1 int2 int3 int4
   inspt
   item1 item2
   list1 list2 list3
   pt1 pt2 pt3 pt4
   real1 real2 real3 real4 real5 real6
   sset1 sset2
   str1 str2 str3 str4 str5 str6
   var1 var2
   x1 y1 x2 y2 x3 y3 x4 y4
   x1_2  x1_3  x1_4  x2_3  x2_4  x3_4  y1_2  y1_3  y1_4  y2_3  y2_4  y3_4

               )
   (set item1 nil))
(gc)                                ;  Force Garbage Collection.
(princ))                            ;  Exit quietly.

;-----------------------------------------------------------------------------
;  SaVe and ReSTore system variables.
;
;  (#SVRST list/int) -> int/str
;
;  The global G:MODE has the form:
;
;     ((<sysvar name> . <value>) (...) ...)
;
;  If the argument is an integer, POP that many system variable settings.
;  If the argument is a list, save their settings into G:MODE, and set.
;-----------------------------------------------------------------------------

(defun #SVRST  (

   item1       ;  SysVar list to set, or # to POP
   /
   list1       ;
   str1        ;
   )

(cond
   ((numberp item1)                    ;  POP this many settings
      (repeat item1
         (setvar (caar G:MODE) (cdar G:MODE))
         (setq G:MODE (cdr G:MODE))))  ;  Remove from list
   (T (foreach list1 item1             ;  PUSH these vars onto the stack
         (setq str1     (car list1)
               G:MODE   (append (list (cons str1 (getvar str1))) G:MODE))
         (setvar str1 (cdr list1))))))

;-----------------------------------------------------------------------------
;  Set operating modes used whilst accessing our blocks
;
;  (#SVINS) -> list
;
;  This are the system variables that need to be set when inserting blocks with
;  attributes.
;-----------------------------------------------------------------------------

(defun #SVINS  ()

'( ("ATTDIA"    . 0)
   ("ATTMODE"   . 1)
   ("ATTREQ"    . 1)
   ("CMDECHO"   . 0)
   ("EXPERT"    . 1)
   ("FLATLAND"  . 0)
   ("LIMCHECK"  . 0)))

;-----------------------------------------------------------------------------
;  HEAD and TAIL functions for each C:xxxx function.
;
;  (#HEAD) -> nil
;  (#TAIL) ->
;
;  #HEAD sets the AutoShade *error* function, system variables and UNDO GROUP.
;  #TAIL restores the *error* function, system variables and ENDs the UNDO.
;-----------------------------------------------------------------------------

(defun #HEAD   (

   list1       ;  List of SysVars to set
   )
(setq #ERROR   *error*                 ;  Trap old *error* function.
      *error*  #ASERR                  ;  Install AutoShade *error* function.
      G:MODE    '())
(#SVRST list1)
(command ".undo" "group"))             ;  UNDO must be set to ALL.

;-----------------------------------------------------------------------------

(defun #TAIL   ()
(command ".undo" "end")                ;  Make function's effects one command.
(#SVRST G:MODE)                        ;  Restore System Variables.
(setq *error*  #ERROR                  ;  Reset old *error* function.
      G:MODE   '())
(princ))

;*****************************************************************************

(prompt "loaded.")

;-----------------------------------------------------------------------------
;  Make layer "ASHADE" if it doesn't exist.
;-----------------------------------------------------------------------------
(if (not (tblsearch "layer" "ashade"))
   (progn
      (#HEAD '(("CMDECHO" . 0)))
      (command ".layer" "n" "ashade" "")
      (#TAIL)))

;-----------------------------------------------------------------------------
;  Initialize global variables.
;-----------------------------------------------------------------------------
(if (setq str1 (getenv "SHADE"))       ;  If environment variable is set to a
   (if (setq file1 (open (strcat str1 "/tempfile.$$$") "a"))   ;  valid path
      (setq file1    (close file1)     ;  then use it.
            G:SHEV   (#2UNIX (strcat str1 "/")))))
(setq str1  nil
      G:R11 (if (getvar "TILEMODE") T))
(if (null G:SCAL) (C:RMSCAN))

(prin1)
