//
// MODULE NAME:  SPX.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for IPX.
//
//	There are two main entrypoints to this module.	PacketSummarySpx
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailSpx returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/17.	Original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

typedef struct _SPXHDR {
    UCHAR ConnectionControl;		// bitflags, as follows:
#define SPXHDR_CONNCTRL_SYSTEM	0x80	// system packet.
#define SPXHDR_CONNCTRL_ACKREQ	0x40	// acknowlegement requested.
#define SPXHDR_CONNCTRL_ATTN	0x20	// attention.
#define SPXHDR_CONNCTRL_EOM	0x10	// end of message.

    UCHAR DataStreamType;		// type of data in this packet.
#define SPXHDR_DATATYPE_EOCACK	0xfe	// end of connection acknowlegement.

    USHORT SourceConnectionId;		// big-endian.
    USHORT DestConnectionId;		// big-endian.
    USHORT SequenceNumber;		// big-endian.
    USHORT AcknowlegeNumber;		// big-endian.
    USHORT AllocationNumber;		// big-endian.
    UCHAR Buffer [1];			// the data inside this packet.
} SPXHDR, *PSPXHDR;

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketFilterSpx (PSPXHDR Buffer, USHORT BufferLength)
{
    if (BufferLength < sizeof (SPXHDR)-1) {
	return FALSE;			// too short to be NetWare SPX.
    }
    if (CaptureFilter & FILTER_SPX) {
	return TRUE;			// we'll take all SPX packets.
    }

    //
    // We aren't accepting raw SPX packets, but if this SPX packet
    // contains a nice SMB, we'll accept that.
    //

    if (!(Buffer->ConnectionControl & SPXHDR_CONNCTRL_SYSTEM)) {
	return PacketFilterSmb (Buffer->Buffer, BufferLength-(sizeof (SPXHDR)-1));
    }
    return FALSE;			// we're not taking SPX packets.
} // PacketFilterSpx

PUCHAR PacketSummarySpx (PSPXHDR Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_SPX)) {
	return NULL;			// we aren't decoding SPX.
    }
    if (BufferLength < sizeof (SPXHDR)-1) {
	return NULL;			// too short to be NetWare SPX.
    }

    if (!(Buffer->ConnectionControl & SPXHDR_CONNCTRL_SYSTEM)) {
	t = PacketSummarySmb (Buffer->Buffer, BufferLength-(sizeof (SPXHDR)-1));
	if (t != NULL) {
	    return t;
	}
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    sprintf (p, "SPX S=%u D=%u SEQ=%u ACK=%u",
	     Swap (Buffer->SourceConnectionId),
	     Swap (Buffer->DestConnectionId),
	     Swap (Buffer->SequenceNumber),
	     Swap (Buffer->AcknowlegeNumber));
    return p;
} // PacketSummarySpx

VOID SpxTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "SPX:");
} // SpxTrailer

PDETREC PacketDetailSpx (PSPXHDR Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;

    if (!(DisplayFilter & FILTER_SPX)) {
	return NULL;			// we aren't decoding SPX.
    }
    if (BufferLength < sizeof (SPXHDR)-1) {
	return NULL;			// too short to be NetWare SPX.
    }

    AppendHeader (&r, "SPX:  ----- Sequential Packet Exchange header -----");
    sprintf (ScratchBuf, "SPX:  ConnectionControl = 0x%02x", Buffer->ConnectionControl);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:      ...%c .... = End of message",
	     Buffer->ConnectionControl & SPXHDR_CONNCTRL_EOM ? '1' : '0');
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:      ..%c. .... = Attention",
	     Buffer->ConnectionControl & SPXHDR_CONNCTRL_ATTN ? '1' : '0');
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:      .%c.. .... = ACK requested",
	     Buffer->ConnectionControl & SPXHDR_CONNCTRL_ACKREQ ? '1' : '0');
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:      %c... .... = System",
	     Buffer->ConnectionControl & SPXHDR_CONNCTRL_SYSTEM ? '1' : '0');
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "SPX:  Data stream type = 0x%02x", Buffer->DataStreamType);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:  Source Connection ID = 0x%04x, Dest Connection ID = 0x%04x",
	     Buffer->SourceConnectionId, Buffer->DestConnectionId);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SPX:  Sequence = %u, Acknowlegement = %u, Allocation = %u",
	     Swap (Buffer->SequenceNumber),
	     Swap (Buffer->AcknowlegeNumber),
	     Swap (Buffer->AllocationNumber));
    AppendDetRec (&r, ScratchBuf);
    SpxTrailer (&r);

    if (!(Buffer->ConnectionControl & SPXHDR_CONNCTRL_SYSTEM)) {
	if ((q=PacketDetailSmb (Buffer->Buffer, BufferLength-(sizeof (SPXHDR)-1))) != NULL) {
	    JoinDetRec (&r, q);
	    return r;
	}
    }

    sprintf (ScratchBuf, "SPX:  Encapsulated data, size=%u (0x%04x hex) bytes.",
	     BufferLength, BufferLength);
    AppendDetRec (&r, ScratchBuf);
    return r;
} // PacketDetailSpx
