//
// MODULE NAME:  NETBEUI.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for IBM
//	NETBEUI and Steve Jones's JetBEUI protocol developed at Microsoft
//	and later shipped by IBM as their new version of NETBEUI.
//
//	There are two main entrypoints to this module.	PacketSummaryNetbeui
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailNetbeui returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/04/09.	Original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// NETBEUI packet structures.
//

typedef struct _NBI {
    USHORT Length;
    USHORT Signature;			// always 0xefff.
    UCHAR Cmd;
    UCHAR Data1;
    USHORT Data2;
    USHORT Xmit;
    USHORT Resp;
    UCHAR Rsn;
    UCHAR Lsn;
    UCHAR Buffer [1];
} NBI, *PNBI;

typedef struct _NBUI {
    USHORT Length;
    USHORT Signature;
    UCHAR Cmd;
    UCHAR Data1;
    USHORT Data2;
    USHORT Xmit;
    USHORT Resp;
    UCHAR RemoteName [16];
    UCHAR LocalName [16];
    UCHAR Buffer [1];
} NBUI, *PNBUI;

#define NB_ADD_GROUP_NAME_QUERY 0x00
#define NB_ADD_NAME_QUERY	0x01
#define NB_ADD_NAME_RESPONSE	0x0d
#define NB_DATA_ACK		0x14
#define NB_DATA_FIRST_MIDDLE	0x15
#define NB_DATAGRAM		0x08
#define NB_DATAGRAM_BROADCAST	0x09
#define NB_DATA_ONLY_LAST	0x16
#define NB_NAME_IN_CONFLICT	0x02
#define NB_NAME_QUERY		0x0a
#define NB_NAME_RECOGNIZED	0x0e
#define NB_NO_RECEIVE		0x1a
#define NB_RECEIVE_CONTINUE	0x1c
#define NB_RECEIVE_OUTSTANDING	0x1b
#define NB_SESSION_ALIVE	0x1f
#define NB_SESSION_CONFIRM	0x17
#define NB_SESSION_END		0x18
#define NB_SESSION_INITIALIZE	0x19
#define NB_STATUS_QUERY         0x03
#define NB_STATUS_RESPONSE	0x0f
#define NB_TERMINATE_TRACE	0x07	// only remote nodes.
#define NB_TERMINATE_TRACE2	0x13	// all nodes.

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeNetbeui (PVOID Buffer, USHORT BufferLength)
{
    UCHAR *p, cmd, *t;
    PNBUI nbuip;
    PNBI nbip;

    nbip = (PNBI)Buffer;
    nbuip = (PNBUI)Buffer;

    if (nbip->Length > BufferLength) {
	return FALSE;			// partial record not decoded.
    }
    if ((nbip->Length != 14) && (nbip->Length != 44)) {
	return FALSE;			// not a NETBEUI frame.
    }

    if (nbip->Length == 14) {
	if ((nbip->Cmd == NB_DATA_FIRST_MIDDLE) ||
	    (nbip->Cmd == NB_DATA_ONLY_LAST)) {
	    return PacketReqTypeSmb (nbip->Buffer, BufferLength-(sizeof (NBI)-1));
	}
    } else if (nbuip->Length == 44) {
	if ((nbuip->Cmd == NB_DATAGRAM) || (nbuip->Cmd == NB_DATAGRAM_BROADCAST)) {
	    return PacketReqTypeSmb (nbuip->Buffer, BufferLength-(sizeof (NBUI)-1));
	}
    }
    return FALSE;			// this isn't a NETBEUI data frame.
} // PacketReqTypeNetbeui

BOOLEAN PacketFilterNetbeui (PVOID Buffer, USHORT BufferLength)
{
    UCHAR *p, cmd, *t;
    PNBUI nbuip;
    PNBI nbip;

    nbip = (PNBI)Buffer;
    nbuip = (PNBUI)Buffer;

    if (nbip->Length > BufferLength) {
	return FALSE;			// partial record not decoded.
    }
    if ((nbip->Length != 14) && (nbip->Length != 44)) {
	return FALSE;			// not a NETBEUI frame.
    }

    if (CaptureFilter & FILTER_NETBEUI) {
	return TRUE;			// we're accepting NETBEUI.
    }

    if (nbip->Length == 14) {
	if ((nbip->Cmd == NB_DATA_FIRST_MIDDLE) ||
	    (nbip->Cmd == NB_DATA_ONLY_LAST)) {
	    return PacketFilterSmb (nbip->Buffer, BufferLength-(sizeof (NBI)-1));
	}
    } else if (nbuip->Length == 44) {
	if ((nbuip->Cmd == NB_DATAGRAM) || (nbuip->Cmd == NB_DATAGRAM_BROADCAST)) {
	    return PacketFilterSmb (nbuip->Buffer, BufferLength-(sizeof (NBUI)-1));
	}
    }
    return FALSE;			// we're not accepting NETBEUI.
} // PacketFilterNetbeui

PUCHAR NetbeuiCmdName (UCHAR Cmd)
{
    switch (Cmd) {
	case NB_ADD_GROUP_NAME_QUERY:	return "Add Group Name Query";
	case NB_ADD_NAME_QUERY:         return "Add Name Query";
	case NB_ADD_NAME_RESPONSE:	return "Add Name Response";
	case NB_DATA_ACK:		return "Data ACK";
	case NB_DATA_FIRST_MIDDLE:	return "Data First Middle";
	case NB_DATAGRAM:		return "Datagram";
	case NB_DATAGRAM_BROADCAST:	return "Broadcast Datagram";
	case NB_DATA_ONLY_LAST:         return "Data Only Last";
	case NB_NAME_IN_CONFLICT:	return "Name In Conflict";
	case NB_NAME_QUERY:		return "Name Query";
	case NB_NAME_RECOGNIZED:	return "Name Recognized";
	case NB_NO_RECEIVE:		return "No Receive";
	case NB_RECEIVE_CONTINUE:	return "Receive Continue";
	case NB_RECEIVE_OUTSTANDING:	return "Receive Outstanding";
	case NB_SESSION_ALIVE:		return "Session Alive";
	case NB_SESSION_CONFIRM:	return "Session Confirm";
	case NB_SESSION_END:		return "Session End";
	case NB_SESSION_INITIALIZE:	return "Session Initialize";
	case NB_STATUS_QUERY:		return "Status Query";
	case NB_STATUS_RESPONSE:	return "Status Response";
	case NB_TERMINATE_TRACE:	return "Terminate Remote Trace";
	case NB_TERMINATE_TRACE2:	return "Terminate Trace";
	default:			return "Undecoded";
    }
} // NetbeuiCmdName

PUCHAR PacketSummaryNetbeui (PVOID Buffer, USHORT BufferLength)
{
    UCHAR *p, cmd, *t;
    PNBUI nbuip;
    PNBI nbip;

    if (!(DisplayFilter & FILTER_NETBEUI)) {
	return NULL;			// we aren't decoding NETBEUI.
    }

    nbip = (PNBI)Buffer;
    nbuip = (PNBUI)Buffer;

    if (nbip->Length > BufferLength) {
	return NULL;			// partial record not decoded.
    }

    if ((nbip->Length != 14) && (nbip->Length != 44)) {
	return NULL;			// not a NETBEUI frame.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    if (nbip->Length == 14) {

	if ((nbip->Cmd == NB_DATA_FIRST_MIDDLE) ||
	    (nbip->Cmd == NB_DATA_ONLY_LAST)) {
	    t = PacketSummarySmb (nbip->Buffer, BufferLength-(sizeof (NBI)-1));
	    if (t != NULL) {
		free (p);
		return t;
	    }
	}
	sprintf (p, "NET L=%u R=%u %s",
		 nbip->Lsn, nbip->Rsn, NetbeuiCmdName (nbip->Cmd));

    } else {

	if ((nbuip->Cmd == NB_DATAGRAM) || (nbuip->Cmd == NB_DATAGRAM_BROADCAST)) {
	    t = PacketSummarySmb (nbuip->Buffer, BufferLength-(sizeof (NBUI)-1));
	    if (t != NULL) {
		free (p);
		return t;
	    }
	}
	sprintf (p, "NET %s", NetbeuiCmdName (nbuip->Cmd));

    }
    return p;
} // PacketSummaryNetbeui

VOID NetbeuiTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "NET:");
} // NetbeuiTrailer

PDETREC PacketDetailNetbeui (PVOID Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    UCHAR *p, cmd, *t;
    PNBUI nbuip;
    PNBI nbip;

    if (!(DisplayFilter & FILTER_NETBEUI)) {
	return NULL;			// we aren't decoding NETBEUI.
    }

    nbip = (PNBI)Buffer;
    nbuip = (PNBUI)Buffer;

    if (nbip->Length > BufferLength) {
	return NULL;			// partial record not decoded.
    }

    if ((nbip->Length != 14) && (nbip->Length != 44)) {
	return NULL;			// not a NETBEUI frame.
    }

    sprintf (ScratchBuf, "NET:  ----- NETBEUI %s -----",
	     NetbeuiCmdName (nbip->Cmd));
    AppendHeader (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Length = %u, Signature = 0x%04x",
	     nbip->Length, nbip->Signature);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Command = 0x%02x  (%s)",
	     nbip->Cmd, NetbeuiCmdName (nbip->Cmd));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Data1 = 0x%02x", nbip->Data1);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Data2 = 0x%04x", nbip->Data2);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Xmit Corr = 0x%04x", nbip->Xmit);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "NET:  Resp Corr = 0x%04x", nbip->Resp);
    AppendDetRec (&r, ScratchBuf);

    if (nbip->Length == 14) {

	sprintf (ScratchBuf, "NET:  Local Session Number = %u, Remote Session Number = %u",
		 nbip->Lsn, nbip->Rsn);
	AppendDetRec (&r, ScratchBuf);
	NetbeuiTrailer (&r);

	if ((nbip->Cmd == NB_DATA_FIRST_MIDDLE) ||
	    (nbip->Cmd == NB_DATA_ONLY_LAST)) {
	    if ((q=PacketDetailSmb (nbip->Buffer, BufferLength-(sizeof (NBI)-1))) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	}

    } else {

	MakeAsciiz (ScratchBuf1, nbuip->LocalName, 16);
	sprintf (ScratchBuf, "NET:  Local Name = %s", ScratchBuf1);
	AppendDetRec (&r, ScratchBuf);
	MakeAsciiz (ScratchBuf1, nbuip->RemoteName, 16);
	sprintf (ScratchBuf, "NET:  Remote Name = %s", ScratchBuf1);
	AppendDetRec (&r, ScratchBuf);
	NetbeuiTrailer (&r);

	if ((nbuip->Cmd == NB_DATAGRAM) || (nbuip->Cmd == NB_DATAGRAM_BROADCAST)) {
	    if ((q=PacketDetailSmb (nbuip->Buffer, BufferLength-(sizeof (NBUI)-1))) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	}

    }
    return r;
} // PacketDetailNetbeui
