//
// MODULE NAME:  NCP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for IPX.
//
//	There are two main entrypoints to this module.	PacketSummaryNcp
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailNcp returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/17.	Original.
//	S. E. Jones	93/01/02.	#2.0, added Novell packet burst protocol.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

#define DOS_ATTR_READONLY      0x01	// file is read-only.
#define DOS_ATTR_HIDDEN        0x02	// file is excluded from normal searches.
#define DOS_ATTR_SYSTEM        0x04	// file is excluded from normal searches.
#define DOS_ATTR_VOLUME_LABEL  0x08	// this is a volume label entry.
#define DOS_ATTR_DIRECTORY     0x10	// this is a subdirectory.
#define DOS_ATTR_ARCHIVE       0x20	// file has been modified since last archive.
#define DOS_ATTR_SHARABLE      0x80	// file is sharable.

typedef struct _BURSTHDR {		// packet burst header.
    USHORT ReqRspTicket;		// (see REQRSP defines, below).
    USHORT StreamType;			// sys [sak] 0 eob [bsy] abt 0 0.
    USHORT SrcConn;			// source connection ID.
    USHORT DestConn;			// destination connection ID.
    USHORT Seq;                         // sequence number.
    USHORT Delay;			// send delay time.
    USHORT BurstSeq;			// burst sequence number.
    USHORT AckSeq;			// ACK sequence number.
    ULONG BurstLength;			// total length of outgoing burst.
    ULONG BurstOffset;			// offset into burst where pkt's data go.
    USHORT ByteCount;			// bytes in this packet.
} BURSTHDR, *PBURSTHDR;

typedef struct _NCPHDR {
    USHORT ReqRspTicket;		// request or response indicator.
#define REQRSP_CREATE_CONN	0x1111	// creation of a connection.
#define REQRSP_REQUEST		0x2222	// this is a request.
#define REQRSP_RESPONSE         0x3333	// this is a response.
#define REQRSP_DELETE_CONN	0x5555	// deletion of a connection.
#define REQRSP_BURST		0x7777	// packet burst protocol header.
#define REQRSP_POSITIVE_ACK	0x9999	// positive acknowlege.

    UCHAR Seq;				// sequence number.
    UCHAR ConnId;			// connection number.
    UCHAR ProcId;			// process # within this connection.
    UCHAR Unk3;
    UCHAR Cmd;				// command code on requests, 0/ffh->resp.
#define CMD_CREATE_CONNECTION	0x00
#define CMD_LOCK_FILE		0x03
#define CMD_LOCK_ALL_FILES	0x04
#define CMD_UNLOCK_FILE         0x07
#define CMD_GET_STATION_NUMBER	0x13
#define CMD_GET_SERVER_CLOCK	0x14
#define CMD_GETSETDIR		0x16	// plenty of sub-functions.
#define CMD_ACCESS_CONTROL	0x17	// lots of sub-functions.
#define CMD_PROCESS_EXIT	0x18
#define CMD_LOGOUT		0x19
#define CMD_LOCK		0x1a
#define CMD_UNLOCK		0x1e
#define CMD_NEGOTIATE_BUFSIZE	0x21
#define CMD_SETDIRHANDLE	0x3e
#define CMD_FIND		0x3f
#define CMD_FINDUNIQUE		0x40
#define CMD_OPEN_FCB		0x41
#define CMD_CLOSE		0x42
#define CMD_CREATE		0x43
#define CMD_DELETE		0x44
#define CMD_RENAME		0x45
#define CMD_SETFILEINFO         0x46
#define CMD_GET_FILE_SIZE	0x47
#define CMD_READ		0x48
#define CMD_WRITE		0x49
#define CMD_SET_FILE_TIME	0x4b
#define CMD_OPEN		0x4c
#define CMD_CREATE_NEW		0x4d	// create temp/new file.

    UCHAR Buffer [1];			// the rest.
} NCPHDR, *PNCPHDR;

typedef struct _NCPLOCKALLFILES {
    USHORT Timeout;
} NCPLOCKALLFILES, *PNCPLOCKALLFILES;

typedef struct _NCPGETFILESIZE {
    UCHAR Unk1;                         // filler.
    ULONG Handle;
} NCPGETFILESIZE, *PNCPGETFILESIZE;

typedef struct _NCPLOCKFILE {
    UCHAR DirHandle;
    UCHAR LockFlags;
#define NCPLOCKFILE_FLAGS_NONEXCLUSIVE	0x0002	// non-exclusive mode.
#define NCPLOCKFILE_FLAGS_NO_LOGGING	0x0001	// don't log request.
    USHORT Timeout;
    UCHAR NameLen;
    UCHAR NameBuf [1];
} NCPLOCKFILE, *PNCPLOCKFILE;

typedef struct _NCPUNLOCKFILE {
    UCHAR DirHandle;
    UCHAR NameLen;
    UCHAR NameBuf [1];			// variable length name of file to unlock.
} NCPUNLOCKFILE, *PNCPUNLOCKFILE;

typedef struct _NCPSETFILETIME {
    UCHAR Unk1;
    ULONG Handle;			// handle to open file.
    USHORT Unk2;
    ULONG DateTime;			// date/time stamp, unknown format.
} NCPSETFILETIME, *PNCPSETFILETIME;

typedef struct _NCPNEGOTIATE {
    USHORT ReqBufferSize;		// requested buffer size in bytes.
} NCPNEGOTIATE, *PNCPNEGOTIATE;

typedef struct _NCPGETCONNINFO {
    USHORT Unk1;			// unknown junk.
    UCHAR FuncCode;			// always NCPAC_FUNC_GET_CONN_INFO.
    UCHAR Connection;			// connection number.
} NCPGETCONNINFO, *PNCPGETCONNINFO;

typedef struct _NCPLOGIN {
    USHORT Unk1;			// unknown junk.
    UCHAR FuncCode;			// as follows:
#define NCPAC_FUNC_LOGIN	0x00	// for unencrypted login.
#define NCPAC_FUNC_CHECK_VERSION 0x11	// check server version.
#define NCPAC_FUNC_GET_CONN_INFO 0x16	// get connection information.
#define NCPAC_FUNC_GET_TRUSTEE_MAPPING 0x35 // map name to trustee.
#define NCPAC_FUNC_SET_TRUSTEE_MAPPING 0x36 // map trustee to user.
#define NCPAC_FUNC_SEARCH_BINDERY 0x37	// search bindery for object.
#define NCPAC_FUNC_READ_PROPERTIES 0x3d // read object properties.
#define NCPAC_FUNC_CHECK_PSWD	0x3f	// verify object password.
#define NCPAC_FUNC_GET_ACCESS_LEVEL 0x46 // get bindery access level.
    UCHAR UserNameLen;			// length of following username.
    UCHAR UserNameBuf [1];		// username (NOT ASCIIZ).

    //
    // What follows after this is a single byte containing the PASSWORD LENGTH,
    // followed by the PASSWORD string itself (can you believe this?).
    //

} NCPLOGIN, *PNCPLOGIN;

typedef struct _NCPSEARCH_BINDERY {
    USHORT Unk1;			// unknown junk.
    UCHAR FuncCode;			// always NCPAC_FUNC_SEARCH_BINDERY.
    ULONG SearchKey;			// 0xffffffff for find first.
    USHORT ObjectType;			// object type to search for.
    UCHAR NameLen;			// length of name in bytes.
    UCHAR NameBuf [1];			// name to search for (NOT ASCIIZ).
} NCPSEARCH_BINDERY, *PNCPSEARCH_BINDERY;

typedef struct _NCPCHECK_PASSWORD {
    USHORT Unk1;			// unknown junk.
    UCHAR FuncCode;			// always 0x3f.
    USHORT ObjectType;			// for USER, it's 0x0001.
    UCHAR UserNameLen;			// length of following username.
    UCHAR UserNameBuf [1];		// username (NOT ASCIIZ).

    //
    // What follows after this is a single byte containing the PASSWORD LENGTH,
    // followed by the PASSWORD string itself (can you believe this?).
    //

} NCPCHECK_PASSWORD, *PNCPCHECK_PASSWORD;

typedef struct _NCPREAD_PROPERTIES {
    USHORT Unk1;			// unknown junk.
    UCHAR FuncCode;			// always 0x3d.
    USHORT ObjectType;			// for USER, it's 0x0001.
    UCHAR NameLen;			// length of following object name.
    UCHAR NameBuf [1];			// object name (NOT ASCIIZ).

    //
    // What follows after this is a single byte containing the "SEGMENT NUMBER",
    // followed by the PROPERTY NAME LENGTH (byte), followed by the PASSWORD
    // string itself (can you believe this?).
    //

} NCPREAD_PROPERTIES, *PNCPREAD_PROPERTIES;

typedef struct _NCPLOCK {
    UCHAR Unk1;
    ULONG Handle;			// open's handle.
    USHORT Unk2;
    ULONG Position;			// offset to lock/unlock at.
    ULONG Length;			// length of locked range of bytes.
    USHORT Unk3;
} NCPLOCK, *PNCPLOCK;

typedef struct _NCPSETDIRHANDLE {
    UCHAR DirHandle;			// directory handle to set.
    UCHAR NameLen;			// length of directory name, below.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPSETDIRHANDLE, *PNCPSETDIRHANDLE;

typedef struct _NCPFINDUNIQUE {
    USHORT EntryNumber;                 // typically, 0xffff.
    UCHAR DirHandle;			// directory handle.
    UCHAR Attrib;			// bitflags, file attributes.
#define FINDATTR_NORMAL         0x00	// normal files.
#define FINDATTR_HIDDEN         0x02	// hidden files.
#define FINDATTR_SYSTEM         0x04	// system files.
#define FINDATTR_DIRECTORY	0x10	// directories.

    UCHAR NameLen;			// length of name, below.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPFINDUNIQUE, *PNCPFINDUNIQUE;

typedef struct _NCPSETFILEINFO {
    UCHAR DosAttr;			// DOS attribute.
    UCHAR DirHandle;			// directory handle.
    UCHAR Attrib;			// bitflags, file attributes.
#define FINDATTR_NORMAL         0x00	// normal files.
#define FINDATTR_HIDDEN         0x02	// hidden files.
#define FINDATTR_SYSTEM         0x04	// system files.
#define FINDATTR_DIRECTORY	0x10	// directories.

    UCHAR NameLen;			// length of name, below.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPSETFILEINFO, *PNCPSETFILEINFO;

typedef struct _NCPFIND {
    UCHAR Unk1;
    USHORT Unk2;
    USHORT EntryNo;			// 0xffff for first (little-endian).
#define FIND_ENTRY_FIRST	0xffff	// this is the first of a find.
    UCHAR Attrib;
    UCHAR NameLen;			// length of name.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPFIND, *PNCPFIND;

typedef struct _NCPOPEN {
    UCHAR DirHandle;			// directory handle.
    UCHAR DosAttr;
    UCHAR Access;			// desired access flags.
#define ACCESS_EXCLUSIVE	0x10	// desire exclusive access to files.
#define ACCESS_DENY_WRITE	0x08	// deny others to open for write.
#define ACCESS_DENY_READ	0x04	// deny others to open for reading.
#define ACCESS_WRITE		0x02	// need write access ourselves.
#define ACCESS_READ		0x01	// need read access ourselves.
    UCHAR NameLen;			// length of name field in bytes.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPOPEN, *PNCPOPEN;

typedef struct _NCPCREATE {
    UCHAR DirHandle;			// directory handle.
    UCHAR DosAttr;			// attributes.
    UCHAR NameLen;			// length of name field in bytes.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPCREATE, *PNCPCREATE;

typedef struct _NCPCREATE_NEW {
    UCHAR DirHandle;			// directory handle.
    UCHAR DosAttr;			// DOS-based attribute.
    UCHAR NameLen;			// length of name field in bytes.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPCREATE_NEW, *PNCPCREATE_NEW;

typedef struct _NCPDELETE {
    UCHAR DirHandle;			// directory handle.
    UCHAR Unk1;
    UCHAR NameLen;			// length of name field in bytes.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPDELETE, *PNCPDELETE;

typedef struct _NCPRENAME {
    UCHAR DirHandle;			// directory handle.
    UCHAR Unk1;
    UCHAR NameLen1;			// length of first name.
    UCHAR NameBuf1 [1];                 // first name.

    //
    // The following fields are displaced in the packet by the length
    // of the first name; therefore, they cannot be statically defined
    // in this structure here; instead, we use comments for clarity.
    //
    // UCHAR NameLen2;
    // UCHAR NameBuf2 [1];
    //

} NCPRENAME, *PNCPRENAME;

typedef struct _NCPCLOSE {
    UCHAR Unk1;
    ULONG Handle;			// handle returned from OPEN.
} NCPCLOSE, *PNCPCLOSE;

typedef struct _NCPIO {
    UCHAR Unk0;
    ULONG Handle;			// handle returned from OPEN.
    USHORT Unk1;
    ULONG Position;			// offset to read, rel from BOF.
    USHORT BytesToTransfer;		// desired buffer size.
} NCPREAD, *PNCPIO;

typedef struct _NCPGETSETDIR {
    UCHAR Unk1;
    UCHAR FrameLen;			// bytes following on SETCWD only.
    UCHAR Operation;			// different for get/set.
#define GSD_SET                 0x00	// set current directory operation.
#define GSD_GET                 0x01	// get directory operation.
#define GSD_CREATE_PERM_HANDLE	0x12	// create permanent handle.
#define GSD_CREATE_HANDLE	0x13	// create directory handle.
#define GSD_DELETE_HANDLE	0x14	// delete directory handle.
#define GSD_GETVOL		0x15	// get volume information.
#define GSD_MKDIR		0x0a	// create directory.
#define GSD_RMDIR		0x0b	// remove directory.
    UCHAR DirHandle;			// directory handle.
    UCHAR Unk2;                         // only for SET/CREATE_PERM.
    UCHAR NameLen;			// size of name field in bytes.
    UCHAR NameBuf [1];			// name (NOT ASCIIZ).
} NCPGETSETDIR, *PNCPGETSETDIR;

//
// Routines in other modules.
//

//
// Routines in this module.
//

PUCHAR AccessControlFuncName (UCHAR FunctionCode)
{
    UCHAR *t;
    switch (FunctionCode) {
	case NCPAC_FUNC_LOGIN: t = "Login"; break;
	case NCPAC_FUNC_CHECK_VERSION: t = "Check server version"; break;
	case NCPAC_FUNC_CHECK_PSWD: t = "Check object password"; break;
	case NCPAC_FUNC_READ_PROPERTIES: t = "Read object properties"; break;
	case NCPAC_FUNC_SEARCH_BINDERY: t = "Search bindery for object"; break;
	case NCPAC_FUNC_GET_ACCESS_LEVEL: t = "Get bindery access level"; break;
	case NCPAC_FUNC_GET_TRUSTEE_MAPPING: t = "Get trustee mapping"; break;
	case NCPAC_FUNC_SET_TRUSTEE_MAPPING: t = "Map trustee to user"; break;
	case NCPAC_FUNC_GET_CONN_INFO: t = "Get connection information"; break;
	default: t = "Undecoded";
    }
    return t;
} // AccessControlFuncName

VOID DetailDosAttr (PDETREC *r, UCHAR DosAttr)
{
    Append8BitMask (r, "NCP:", DosAttr,
		    "DOS attribute",
		    "Read only",
		    "Hidden",
		    "System",
		    "Volume label",
		    "Directory",
		    "Archive",
		    NULL, NULL);
} // DetailDosAttr

VOID DetailNcpAttr (PDETREC *r, UCHAR NcpAttr)
{
    Append8BitMask (r, "NCP:", NcpAttr,
		    "NetWare attribute",
		    "Read only",
		    "Hidden",
		    "System",
		    "Execute only",
		    "Directory",
		    "Archive",
		    "Sharable",
		    NULL);
} // DetailNcpAttr

VOID DetailAccess (PDETREC *r, UCHAR Access)
{
    Append8BitMask (r, "NCP:", Access,
		    "Desired access",
		    "Read permission",
		    "Write permission",
		    "Deny read",
		    "Deny write",
		    "Exclusive access",
		    NULL,
		    NULL,
		    NULL);
} // DetailAccess

BOOLEAN PacketReqTypeNcp (PNCPHDR Buffer, USHORT BufferLength)
{
    PNCPLOGIN loginp;

    if (BufferLength < sizeof (NCPHDR)-1) {
	return FALSE;			// too short to be NetWare NCP.
    }

    if (Buffer->ReqRspTicket != REQRSP_REQUEST) {
	return TRUE;			// only decode requests.
    }

    switch (Buffer->Cmd) {
	case CMD_SETFILEINFO:
	case CMD_SET_FILE_TIME:
	case CMD_GETSETDIR:
	case CMD_SETDIRHANDLE:
	    break;
	case CMD_CREATE_CONNECTION:
	case CMD_GET_STATION_NUMBER:
	case CMD_GET_SERVER_CLOCK:
	case CMD_NEGOTIATE_BUFSIZE:
	case CMD_LOCK_ALL_FILES:
	case CMD_GET_FILE_SIZE:
	    break;
	case CMD_PROCESS_EXIT:
	    CountFileIoType (FILEIO_DESTROY_PROCESS);
	    break;
	case CMD_LOGOUT:
	    CountFileIoType (FILEIO_LOGOUT);
	    break;
	case CMD_ACCESS_CONTROL:
	    loginp = (PNCPLOGIN)Buffer->Buffer;
	    switch (loginp->FuncCode) {
		case NCPAC_FUNC_CHECK_VERSION:
		case NCPAC_FUNC_GET_ACCESS_LEVEL:
		case NCPAC_FUNC_SEARCH_BINDERY:
		case NCPAC_FUNC_GET_CONN_INFO:
		    break;
		case NCPAC_FUNC_LOGIN:
		case NCPAC_FUNC_CHECK_PSWD:
		case NCPAC_FUNC_READ_PROPERTIES:
		case NCPAC_FUNC_GET_TRUSTEE_MAPPING:
		case NCPAC_FUNC_SET_TRUSTEE_MAPPING:
		    CountFileIoType (FILEIO_LOGIN);
	    }
	    break;
	case CMD_CREATE:
	case CMD_CREATE_NEW:
	    CountFileIoType (FILEIO_CREATE);
	    break;
	case CMD_DELETE:
	    CountFileIoType (FILEIO_DELETE);
	    break;
	case CMD_OPEN:
	case CMD_OPEN_FCB:
	    CountFileIoType (FILEIO_OPEN);
	    break;
	case CMD_LOCK_FILE:
	case CMD_LOCK:
	    CountFileIoType (FILEIO_LOCK);
	    break;
	case CMD_UNLOCK_FILE:
	case CMD_UNLOCK:
	    CountFileIoType (FILEIO_UNLOCK);
	    break;
	case CMD_CLOSE:
	    CountFileIoType (FILEIO_CLOSE);
	    break;
	case CMD_READ:
	    CountFileIoType (FILEIO_READ);
	    break;
	case CMD_WRITE:
	    CountFileIoType (FILEIO_WRITE);
	    break;
	case CMD_RENAME:
	    CountFileIoType (FILEIO_RENAME);
	    break;
	case CMD_FIND:
	case CMD_FINDUNIQUE:
	    CountFileIoType (FILEIO_SEARCH);
    }
    return TRUE;
} // PacketReqTypeNcp

PUCHAR PacketSummaryNcp (PNCPHDR Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;
    PNCPSETFILEINFO sfip;
    PNCPOPEN op;
    PNCPCREATE cp;
    PNCPCREATE_NEW cnp;
    PNCPDELETE dp;
    PNCPRENAME rp;
    PNCPIO iop;
    PNCPGETSETDIR gsp;
    PNCPFINDUNIQUE fup;
    PNCPFIND fp;
    PNCPLOCK lp;
    PNCPLOGIN loginp;
    PNCPCHECK_PASSWORD chkp;
    PNCPSEARCH_BINDERY sbp;
    PNCPLOCKFILE lfp;
    PNCPUNLOCKFILE ulfp;

    if (!(DisplayFilter & FILTER_NCP)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (NCPHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }

    p = malloc (128);
    if (p == NULL) {
	return NULL;
    }

    switch (Buffer->ReqRspTicket) {
	case REQRSP_RESPONSE:
	    if (Buffer->Cmd == 0) {
		strcpy (p, "NCP R OK");
	    } else {
		sprintf (p, "NCP R Failed, status=0x%02x", Buffer->Cmd);
	    }
	    return p;

	case REQRSP_BURST:
	    strcpy (p, "NCP Packet Burst");
	    return p;

	case REQRSP_CREATE_CONN:
	    strcpy (p, "NCP Create Connection");
	    return p;

	case REQRSP_DELETE_CONN:
	    strcpy (p, "NCP Delete Connection");
	    return p;

	case REQRSP_POSITIVE_ACK:
	    strcpy (p, "NCP Positive Acknowlegement");
	    return p;
    }

    //
    // This has to be a command.
    //

    switch (Buffer->Cmd) {
	case CMD_CREATE_CONNECTION:
	    strcpy (p, "NCP C Create connection");
	    break;
	case CMD_PROCESS_EXIT:
	    strcpy (p, "NCP C Process exit");
	    break;
	case CMD_GET_STATION_NUMBER:
	    strcpy (p, "NCP C Get station ID");
	    break;
	case CMD_GET_SERVER_CLOCK:
	    strcpy (p, "NCP C Read server clock");
	    break;
	case CMD_LOGOUT:
	    strcpy (p, "NCP C Logout");
	    break;
	case CMD_NEGOTIATE_BUFSIZE:
	    strcpy (p, "NCP C Negotiate buffer size");
	    break;
	case CMD_LOCK_ALL_FILES:
	    strcpy (p, "NCP C Lock all files");
	    break;
	case CMD_GET_FILE_SIZE:
	    strcpy (p, "NCP C Get file size");
	    break;
	case CMD_ACCESS_CONTROL:
	    loginp = (PNCPLOGIN)Buffer->Buffer;
	    switch (loginp->FuncCode) {
		case NCPAC_FUNC_LOGIN:
		    MakeAsciiz (ScratchBuf, loginp->UserNameBuf, loginp->UserNameLen);
		    sprintf (p, "NCP C Login '%s'", ScratchBuf);
		    break;
		case NCPAC_FUNC_CHECK_PSWD:
		    chkp = (PNCPCHECK_PASSWORD)loginp;
		    MakeAsciiz (ScratchBuf, chkp->UserNameBuf, chkp->UserNameLen);
		    sprintf (p, "NCP C Check password for '%s'", ScratchBuf);
		    break;
		case NCPAC_FUNC_READ_PROPERTIES:
		    chkp = (PNCPCHECK_PASSWORD)loginp;
		    MakeAsciiz (ScratchBuf, chkp->UserNameBuf, chkp->UserNameLen);
		    sprintf (p, "NCP C Read properties for '%s'", ScratchBuf);
		    break;
		case NCPAC_FUNC_CHECK_VERSION:
		    strcpy (p, "NCP C Check server version");
		    break;
		case NCPAC_FUNC_GET_ACCESS_LEVEL:
		    strcpy (p, "NCP C Get bindary access level");
		    break;
		case NCPAC_FUNC_SEARCH_BINDERY:
		    sbp = (PNCPSEARCH_BINDERY)loginp;
		    MakeAsciiz (ScratchBuf, sbp->NameBuf, sbp->NameLen);
		    sprintf (p, "NCP C Search bindery for '%s'", ScratchBuf);
		    break;
		case NCPAC_FUNC_GET_TRUSTEE_MAPPING:
		    chkp = (PNCPCHECK_PASSWORD)loginp;
		    MakeAsciiz (ScratchBuf, chkp->UserNameBuf, chkp->UserNameLen);
		    sprintf (p, "NCP C Map '%s' to trustee", ScratchBuf);
		    break;
		case NCPAC_FUNC_SET_TRUSTEE_MAPPING:
		    strcpy (p, "NCP C Map trustee to user");
		    break;
		case NCPAC_FUNC_GET_CONN_INFO:
		    strcpy (p, "NCP C Get connection information");
		    break;
		default:
		    sprintf (p, "NCP C Access Control (0x%02x)", loginp->FuncCode);
	    }
	    break;
	case CMD_CREATE:
	    cp = (PNCPCREATE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, cp->NameBuf, cp->NameLen);
	    sprintf (p, "NCP C Create '%s'", ScratchBuf);
	    break;
	case CMD_CREATE_NEW:
	    cnp = (PNCPCREATE_NEW)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, cnp->NameBuf, cnp->NameLen);
	    sprintf (p, "NCP C Create new '%s'", ScratchBuf);
	    break;
	case CMD_DELETE:
	    dp = (PNCPDELETE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, dp->NameBuf, dp->NameLen);
	    sprintf (p, "NCP C Delete '%s'", ScratchBuf);
	    break;
	case CMD_OPEN:
	    op = (PNCPOPEN)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, op->NameBuf, op->NameLen);
	    sprintf (p, "NCP C Open '%s'", ScratchBuf);
	    break;
	case CMD_OPEN_FCB:
	    cp = (PNCPCREATE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, cp->NameBuf, cp->NameLen);
	    sprintf (p, "NCP C Open (FCB) '%s'", ScratchBuf);
	    break;
	case CMD_SETFILEINFO:
	    sfip = (PNCPSETFILEINFO)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, sfip->NameBuf, sfip->NameLen);
	    sprintf (p, "NCP C Set file info '%s'", ScratchBuf);
	    break;
	case CMD_LOCK_FILE:
	    lfp = (PNCPLOCKFILE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, lfp->NameBuf, lfp->NameLen);
	    sprintf (p, "NCP C Lock file '%s'", ScratchBuf);
	    break;
	case CMD_UNLOCK_FILE:
	    ulfp = (PNCPUNLOCKFILE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, ulfp->NameBuf, ulfp->NameLen);
	    sprintf (p, "NCP C Unlock file '%s'", ScratchBuf);
	    break;
	case CMD_CLOSE:
	    strcpy (p, "NCP C Close");
	    break;
	case CMD_SET_FILE_TIME:
	    strcpy (p, "NCP C Set file timestamp");
	    break;
	case CMD_LOCK:
	    lp = (PNCPLOCK)Buffer->Buffer;
	    sprintf (p, "NCP C Lock %ld at %ld",
		     SwapDword (lp->Length), SwapDword (lp->Position));
	    break;
	case CMD_UNLOCK:
	    lp = (PNCPLOCK)Buffer->Buffer;
	    sprintf (p, "NCP C Unlock %ld at %ld",
		     SwapDword (lp->Length), SwapDword (lp->Position));
	    break;
	case CMD_READ:
	    iop = (PNCPIO)Buffer->Buffer;
	    sprintf (p, "NCP C Read %u at %lu",
		     Swap (iop->BytesToTransfer), SwapDword (iop->Position));
	    break;
	case CMD_WRITE:
	    iop = (PNCPIO)Buffer->Buffer;
	    sprintf (p, "NCP C Write %u at %lu",
		     Swap (iop->BytesToTransfer), SwapDword (iop->Position));
	    break;
	case CMD_RENAME:
	    rp = (PNCPRENAME)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, rp->NameBuf1, rp->NameLen1);
	    sprintf (p, "NCP C Rename '%s'", ScratchBuf);
	    break;
	case CMD_FIND:
	    fp = (PNCPFIND)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, fp->NameBuf, fp->NameLen);
	    if (fp->EntryNo == FIND_ENTRY_FIRST) {
		sprintf (p, "NCP C Find first '%s'", ScratchBuf);
	    } else {
		sprintf (p, "NCP C Find next '%s' (%u)", ScratchBuf, fp->EntryNo);
	    }
	    break;
	case CMD_GETSETDIR:
	    gsp = (PNCPGETSETDIR)(Buffer->Buffer);
	    switch (gsp->Operation) {
		case GSD_SET: strcpy (p, "NCP C Set current directory"); break;
		case GSD_GET: strcpy (p, "NCP C Get current directory"); break;
		case GSD_CREATE_PERM_HANDLE: strcpy (p, "NCP C Create perm handle"); break;
		case GSD_CREATE_HANDLE: strcpy (p, "NCP C Create handle"); break;
		case GSD_DELETE_HANDLE: strcpy (p, "NCP C Delete handle"); break;
		case GSD_GETVOL: strcpy (p, "NCP C Get volume information"); break;
		case GSD_MKDIR: strcpy (p, "NCP C Create directory"); break;
		case GSD_RMDIR: strcpy (p, "NCP C Delete directory"); break;
		default: sprintf (p, "NCP C Directory control (%02x)", gsp->Operation);
	    }
	    break;
	case CMD_FINDUNIQUE:
	    fup = (PNCPFINDUNIQUE)Buffer->Buffer;
	    MakeAsciiz (ScratchBuf, fup->NameBuf, fup->NameLen);
	    sprintf (p, "NCP C Find unique '%s'", ScratchBuf);
	    break;
	case CMD_SETDIRHANDLE:
	    strcpy (p, "NCP C Set directory handle");
	    break;
	default:
	    sprintf (p, "NCP C Undecoded (0x%02x), size=%u", Buffer->Cmd, BufferLength);
    }
    return p;
} // PacketSummaryNcp

PDETREC PacketDetailNcp (PNCPHDR Buffer, USHORT BufferLength)
{
    UCHAR *t;
    PDETREC r=NULL, q;
    PNCPOPEN o;
    PNCPSETFILEINFO sfip;
    PNCPSETDIRHANDLE sdhp;
    PNCPCREATE crp;
    PNCPCREATE_NEW cnp;
    PNCPDELETE dp;
    PNCPRENAME rp;
    PNCPFIND fp;
    PNCPFINDUNIQUE fup;
    PNCPIO iop;
    PNCPLOCK lp;
    PNCPCLOSE cp;
    PNCPGETSETDIR gsp;
    PNCPLOGIN loginp;
    PNCPCHECK_PASSWORD chkp;
    PNCPNEGOTIATE negp;
    PNCPSEARCH_BINDERY sbp;
    PNCPGETCONNINFO gcip;
    PNCPSETFILETIME sftp;
    PNCPLOCKFILE lfp;
    PNCPUNLOCKFILE ulfp;
    PNCPLOCKALLFILES lafp;
    PNCPGETFILESIZE gfsp;
    PNCPREAD_PROPERTIES rpp;
    PBURSTHDR burst;

    if (!(DisplayFilter & FILTER_NCP)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (NCPHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }

    AppendHeader (&r, "NCP:  ----- NetWare Core Protocol header -----");

    switch (Buffer->ReqRspTicket) {
	case REQRSP_CREATE_CONN: t = "Create connection"; break;
	case REQRSP_REQUEST: t = "Request"; break;
	case REQRSP_RESPONSE: t = "Response"; break;
	case REQRSP_DELETE_CONN: t = "Delete connection"; break;
	case REQRSP_BURST: t = "Packet burst protocol"; break;
	case REQRSP_POSITIVE_ACK: t = "Positive Acknowlegement"; break;
	default: t = "Undecoded";
    }
    sprintf (ScratchBuf, "NCP:  Action = 0x%04x (%s)",
	     Buffer->ReqRspTicket, t);
    AppendDetRec (&r, ScratchBuf);

    if (Buffer->ReqRspTicket == REQRSP_BURST) {
	burst = (PBURSTHDR)Buffer;    // burst = ptr to packet burst header.
	sprintf (ScratchBuf, "BURST:  Stream Type = %04x", burst->StreamType);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Source Connection Id = %04x", burst->SrcConn);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Destination Connection Id = %04x", burst->DestConn);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Sequence Number = %u", burst->Seq);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Delay = %u", burst->Delay);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Burst Sequence Number = %u", burst->BurstSeq);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Burst ACK Number = %u", burst->AckSeq);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Burst Length = %lu", burst->BurstLength);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Burst Offset = %lu", burst->BurstLength);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "BURST:  Byte Count = %u", burst->ByteCount);
	AppendDetRec (&r, ScratchBuf);
	return r;
    }

    sprintf (ScratchBuf,
	     "NCP:  Sequence number = %u, Connection ID = %u, Process ID = %u",
	     Buffer->Seq, Buffer->ConnId, Buffer->ProcId);
    AppendDetRec (&r, ScratchBuf);

    if (Buffer->ReqRspTicket == REQRSP_RESPONSE) {
	if (Buffer->Cmd == 0) {
	    AppendDetRec (&r, "NCP:  Status = OK");
	} else {
	    sprintf (ScratchBuf, "NCP:  Status = 0x%02x", Buffer->Cmd);
	    AppendDetRec (&r, ScratchBuf);
	}
	return r;
    }

    switch (Buffer->Cmd) {
	case CMD_GET_STATION_NUMBER: t = "Get station ID"; break;
	case CMD_LOCK_FILE: t = "Lock file"; break;
	case CMD_UNLOCK_FILE: t = "Unlock file"; break;
	case CMD_LOCK_ALL_FILES: t = "Lock all files"; break;
	case CMD_GET_FILE_SIZE: t = "Get file size"; break;
	case CMD_GET_SERVER_CLOCK: t = "Read server clock"; break;
	case CMD_NEGOTIATE_BUFSIZE: t = "Negotiate buffer size"; break;
	case CMD_CREATE_CONNECTION: t = "Create connection"; break;
	case CMD_PROCESS_EXIT: t = "Process exit"; break;
	case CMD_LOGOUT: t = "Logout"; break;
	case CMD_ACCESS_CONTROL: t = "Access control"; break;
	case CMD_CREATE: t = "Create"; break;
	case CMD_CREATE_NEW: t = "Create new file"; break;
	case CMD_DELETE: t = "Delete"; break;
	case CMD_OPEN: t = "Open"; break;
	case CMD_OPEN_FCB: t = "Open FCB"; break;
	case CMD_CLOSE: t = "Close"; break;
	case CMD_READ: t = "Read"; break;
	case CMD_WRITE: t = "Write"; break;
	case CMD_FIND: t = "Search"; break;
	case CMD_RENAME: t = "Rename"; break;
	case CMD_SETFILEINFO: t = "Set file information"; break;
	case CMD_SET_FILE_TIME: t = "Set file time"; break;
	case CMD_GETSETDIR: t = "Directory control"; break;
	case CMD_FINDUNIQUE: t = "Find unique"; break;
	case CMD_SETDIRHANDLE: t = "Set directory handle"; break;
	case CMD_LOCK: t = "Lock byte range"; break;
	case CMD_UNLOCK: t = "Unlock byte range"; break;
	default: t = "Undecoded";
    }
    sprintf (ScratchBuf, "NCP:  Command = 0x%02x  (%s)", Buffer->Cmd, t);
    AppendDetRec (&r, ScratchBuf);

    switch (Buffer->Cmd) {
	case CMD_CREATE_CONNECTION:		// no operands on create connection.
	case CMD_PROCESS_EXIT:			// no operands on process exit.
	case CMD_LOGOUT:			// no operands on logout.
	case CMD_GET_SERVER_CLOCK:		// no operands on this one.
	case CMD_GET_STATION_NUMBER:		// no operands on this one.
	    break;
	case CMD_LOCK_ALL_FILES:
	    lafp = (PNCPLOCKALLFILES)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Timeout = %u", Swap (lafp->Timeout));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_NEGOTIATE_BUFSIZE:
	    negp = (PNCPNEGOTIATE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Requested buffer size = %u bytes",
		     Swap (negp->ReqBufferSize));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_ACCESS_CONTROL:
	    AppendDetRec (&r, "NCP:");
	    AppendHeader (&r, "ACP:  ----- Access Control Protocol -----");
	    loginp = (PNCPLOGIN)Buffer->Buffer;
	    sprintf (ScratchBuf, "ACP:  Function code = 0x%02x (%s)",
		     loginp->FuncCode,
		     AccessControlFuncName (loginp->FuncCode));
	    AppendDetRec (&r, ScratchBuf);
	    switch (loginp->FuncCode) {
		case NCPAC_FUNC_LOGIN:		// w/no encryption.
		    MakeAsciiz (ScratchBuf1, loginp->UserNameBuf, loginp->UserNameLen);
		    sprintf (ScratchBuf, "ACP:  Username = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1,
				loginp->UserNameBuf+loginp->UserNameLen+1,
				*(UCHAR *)(loginp->UserNameBuf+loginp->UserNameLen));
		    sprintf (ScratchBuf, "ACP:  Password = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_CHECK_PSWD:
		    chkp = (PNCPCHECK_PASSWORD)loginp;
		    sprintf (ScratchBuf, "ACP:  Object type = 0x%04x",
			     Swap (chkp->ObjectType));
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, chkp->UserNameBuf, chkp->UserNameLen);
		    sprintf (ScratchBuf, "ACP:  Username = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1,
				chkp->UserNameBuf+chkp->UserNameLen+1,
				*(UCHAR *)(chkp->UserNameBuf+chkp->UserNameLen));
		    sprintf (ScratchBuf, "ACP:  Password = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_READ_PROPERTIES:
		    rpp = (PNCPREAD_PROPERTIES)loginp;
		    sprintf (ScratchBuf, "ACP:  Object type = 0x%04x",
			     Swap (rpp->ObjectType));
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, rpp->NameBuf, rpp->NameLen);
		    sprintf (ScratchBuf, "ACP:  Object name = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "ACP:  Segment = 0x%02x",
			     *(UCHAR *)(rpp->NameBuf+rpp->NameLen));
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, rpp->NameBuf+rpp->NameLen+2,
				*(UCHAR *)(rpp->NameBuf+rpp->NameLen+1));
		    sprintf (ScratchBuf, "ACP:  Property Name = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_SEARCH_BINDERY:
		    sbp = (PNCPSEARCH_BINDERY)loginp;
		    sprintf (ScratchBuf, "ACP:  Search key = 0x%08lx",
			     SwapDword (sbp->SearchKey));
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "ACP:  Object type = 0x%04x",
			     Swap (sbp->ObjectType));
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, sbp->NameBuf, sbp->NameLen);
		    sprintf (ScratchBuf, "ACP:  Name = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_GET_TRUSTEE_MAPPING:
		    chkp = (PNCPCHECK_PASSWORD)loginp;
		    sprintf (ScratchBuf, "ACP:  Object type = 0x%04x",
			     Swap (chkp->ObjectType));
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, chkp->UserNameBuf, chkp->UserNameLen);
		    sprintf (ScratchBuf, "ACP:  Username = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_SET_TRUSTEE_MAPPING:
		    sbp = (PNCPSEARCH_BINDERY)loginp;
		    sprintf (ScratchBuf, "ACP:  Object ID = 0x%08lx",
			     SwapDword (sbp->SearchKey));
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_GET_CONN_INFO:
		    gcip = (PNCPGETCONNINFO)loginp;
		    sprintf (ScratchBuf, "ACP:  Connection ID = %u", gcip->Connection);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case NCPAC_FUNC_CHECK_VERSION:	// no operands.
		case NCPAC_FUNC_GET_ACCESS_LEVEL: // no operands.
		    break;
	    }
	    break;
	case CMD_CREATE:
	    crp = (PNCPCREATE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", crp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, crp->DosAttr);
	    MakeAsciiz (ScratchBuf1, crp->NameBuf, crp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_OPEN_FCB:
	    crp = (PNCPCREATE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", crp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, crp->DosAttr);
	    MakeAsciiz (ScratchBuf1, crp->NameBuf, crp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_CREATE_NEW:
	    cnp = (PNCPCREATE_NEW)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", cnp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, cnp->DosAttr);
	    MakeAsciiz (ScratchBuf1, cnp->NameBuf, cnp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_DELETE:
	    dp = (PNCPDELETE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", dp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Data byte = 0x%02x", dp->Unk1);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1, dp->NameBuf, dp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_LOCK_FILE:
	    lfp = (PNCPLOCKFILE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", lfp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Lock flags = 0x%02x", lfp->LockFlags);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Timeout = %u", lfp->Timeout);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1, lfp->NameBuf, lfp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_UNLOCK_FILE:
	    ulfp = (PNCPUNLOCKFILE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", ulfp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1, ulfp->NameBuf, ulfp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_OPEN:
	    o = (PNCPOPEN)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", o->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, o->DosAttr);
	    DetailAccess (&r, o->Access);
	    MakeAsciiz (ScratchBuf1, o->NameBuf, o->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_CLOSE:
	    cp = (PNCPCLOSE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Handle = 0x%08lx", SwapDword (cp->Handle));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_SET_FILE_TIME:
	    sftp = (PNCPSETFILETIME)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Handle = 0x%08lx", SwapDword (sftp->Handle));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_GET_FILE_SIZE:
	    gfsp = (PNCPGETFILESIZE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Handle = 0x%08lx", SwapDword (gfsp->Handle));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_LOCK:
	case CMD_UNLOCK:
	    lp = (PNCPLOCK)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Handle = 0x%08lx", SwapDword (lp->Handle));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Position = %ld", SwapDword (lp->Position));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Record size = %ld", SwapDword (lp->Length));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_READ:
	case CMD_WRITE:
	    iop = (PNCPIO)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Handle = 0x%08lx", SwapDword (iop->Handle));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Position = %ld", SwapDword (iop->Position));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Bytes to transfer = %u", Swap (iop->BytesToTransfer));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_RENAME:
	    rp = (PNCPRENAME)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", rp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Data byte = 0x%02x", rp->Unk1);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1, rp->NameBuf1, rp->NameLen1);
	    sprintf (ScratchBuf, "NCP:  Source filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1,
			rp->NameBuf1+rp->NameLen1+2,
			*(UCHAR *)(rp->NameBuf1+rp->NameLen1+1));
	    sprintf (ScratchBuf, "NCP:  Destination filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_FIND:
	    fp = (PNCPFIND)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Data byte = 0x%02x", fp->Unk1);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Unknown word = 0x%04x", Swap (fp->Unk2));
	    AppendDetRec (&r, ScratchBuf);
	    if (fp->EntryNo == FIND_ENTRY_FIRST) {
		AppendDetRec (&r, "NCP:  Entry number = 0xffff (Find First)");
	    } else {
		sprintf (ScratchBuf, "NCP:  Entry number = %u", fp->EntryNo);
		AppendDetRec (&r, ScratchBuf);
	    }
	    DetailNcpAttr (&r, fp->Attrib);
	    MakeAsciiz (ScratchBuf1, fp->NameBuf, fp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_SETFILEINFO:
	    sfip = (PNCPSETFILEINFO)Buffer->Buffer;
	    DetailNcpAttr (&r, sfip->DosAttr);
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", sfip->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, sfip->Attrib);
	    MakeAsciiz (ScratchBuf1, sfip->NameBuf, sfip->NameLen);
	    sprintf (ScratchBuf, "NCP:  Filename = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_GETSETDIR:
	    AppendDetRec (&r, "NCP:");
	    AppendHeader (&r, "DIR:  ----- Directory Control Protocol -----");
	    gsp = (PNCPGETSETDIR)(Buffer->Buffer);
	    switch (gsp->Operation) {
		case GSD_MKDIR:
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, "NCP:  Operation = Create directory");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, gsp->NameBuf, gsp->NameLen);
		    sprintf (ScratchBuf, "DIR:  Pathname = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_RMDIR:
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, "NCP:  Operation = Delete directory");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, gsp->NameBuf, gsp->NameLen);
		    sprintf (ScratchBuf, "DIR:  Pathname = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_SET:
		    sprintf (ScratchBuf, "DIR:  Frame length = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, "NCP: Operation = Set current directory");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, gsp->NameBuf, gsp->NameLen);
		    sprintf (ScratchBuf, "DIR:  Path = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_GET:
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, "DIR:  Operation = Get current directory");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_CREATE_PERM_HANDLE:
		    AppendDetRec (&r, "DIR:  Operation = Create permanent directory handle");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Drive name = '%c:'", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, gsp->NameBuf, gsp->NameLen);
		    sprintf (ScratchBuf, "DIR:  Path = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_DELETE_HANDLE:
		    AppendDetRec (&r, "DIR:  Operation = Delete directory handle");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_CREATE_HANDLE:
		    AppendDetRec (&r, "DIR:  Operation = Create directory handle");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Drive name = '%c:'", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    MakeAsciiz (ScratchBuf1, gsp->NameBuf, gsp->NameLen);
		    sprintf (ScratchBuf, "DIR:  Path = '%s'", ScratchBuf1);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		case GSD_GETVOL:
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    AppendDetRec (&r, "DIR:  Operation = Get volume information");
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
		    break;
		default:
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->FrameLen);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Directory control (0x%02x)", gsp->Operation);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Directory handle = 0x%02x", gsp->DirHandle);
		    AppendDetRec (&r, ScratchBuf);
		    sprintf (ScratchBuf, "DIR:  Data byte = 0x%02x", gsp->Unk2);
		    AppendDetRec (&r, ScratchBuf);
	    }
	    break;
	case CMD_SETDIRHANDLE:
	    sdhp = (PNCPSETDIRHANDLE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", sdhp->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    MakeAsciiz (ScratchBuf1, sdhp->NameBuf, sdhp->NameLen);
	    sprintf (ScratchBuf, "NCP:  Path = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case CMD_FINDUNIQUE:
	    fup = (PNCPFINDUNIQUE)Buffer->Buffer;
	    sprintf (ScratchBuf, "NCP:  Entry number = 0x%02x", fup->EntryNumber);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "NCP:  Directory handle = 0x%02x", fup->DirHandle);
	    AppendDetRec (&r, ScratchBuf);
	    DetailDosAttr (&r, fup->Attrib);
	    MakeAsciiz (ScratchBuf1, fup->NameBuf, fup->NameLen);
	    sprintf (ScratchBuf, "NCP:  Pathname = '%s'", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    break;
    }
    return r;
} // PacketDetailNcp
