//
// MODULE NAME:  IPX.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for IPX.
//
//	There are two main entrypoints to this module.	PacketSummaryIpx
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailIpx returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/17.	Original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

typedef struct _IPXHDR {
    USHORT CheckSum;			// big-endian.
    USHORT Length;			// big-endian total bytes in packet.
    UCHAR TransportCtrl;
    UCHAR PacketType;			// packet type, as follows.
#define IPXHDR_TYPE_UNKNOWN	0x00	// unknown type.
#define IPXHDR_TYPE_RIP         0x01	// routing information type.
#define IPXHDR_TYPE_ECHO	0x02	// echo packet.
#define IPXHDR_TYPE_ERROR	0x03	// error packet.
#define IPXHDR_TYPE_PEP         0x04	// packet exchange pkt (NetWare type).
#define IPXHDR_TYPE_SPX         0x05	// sequenced packet exchange protocol.
#define IPXHDR_TYPE_NCP         0x11	// NetWare Core Protocol.

    UCHAR DestNetwork [4];		// destination network address.
    UCHAR DestNode [6];                 // destination node address.
    USHORT DestSocket;			// destination socket.
    UCHAR SrcNetwork [4];		// source network address.
    UCHAR SrcNode [6];			// source node address.
    USHORT SrcSocket;			// source socket.
    UCHAR Buffer [1];			// the data inside this packet.
} IPXHDR, *PIPXHDR;

//
// Routing Information Protocol (RIP) packet layout.
//

typedef struct _RIP {
    USHORT Operation;
#define RIP_RESPONSE	2		// response to a request.
    UCHAR Network [4];			// object network.
    USHORT HopCount;			// number of hops to get there.
} RIP, *PRIP;

//
// Service Advertizing Packet layout.
//

typedef struct _SAP {
    USHORT ResponseType;		// response type.
#define SAP_TYPE_GENERAL_QUERY	2	// general service query response.
#define SAP_TYPE_NEAREST_QUERY	4	// nearest service query response.

    USHORT ServerType;			// type of server.
#define SAP_TYPE_WILD		0xffff	// any server.
#define SAP_TYPE_UNKNOWN	0	// who knows.
#define SAP_TYPE_PRINT		3	// non-advertizing print server.
#define SAP_TYPE_FILE		4	// file server.
#define SAP_TYPE_JOB		5	// job server.
#define SAP_TYPE_ARCHIVE	9	// archive server.
#define SAP_TYPE_BRIDGE         0x24	// remote bridge server.
#define SAP_TYPE_ADV_PRINT	0x47	// advertizing print server.

    UCHAR ServerName [48];		// ASCIIZ name of this server.
    UCHAR NetAddr [4];			// network address.
    UCHAR NodeAddr [6];                 // node address.
    USHORT SockAddr;			// socket address.
    USHORT HopCount;			// # hops this packet has made (0=local).
#define HOPCOUNT_SHUTDOWN	0x10	// indicates server is shutting down.
} SAP, *PSAP;

//
// Service Query packet.
//

typedef struct _SQP {
    USHORT PacketType;			// type of packet.
#define SQP_TYPE_GENERAL_QUERY	1	// general service query request.
#define SQP_TYPE_NEAREST_QUERY	3	// nearest service query request.

    USHORT ServerType;			// type of server.
#define SQP_TYPE_WILD		0xffff	// any server.
#define SQP_TYPE_UNKNOWN	0	// who knows.
#define SQP_TYPE_PRINT		3	// non-advertizing print server.
#define SQP_TYPE_FILE		4	// file server.
#define SQP_TYPE_JOB		5	// job server.
#define SQP_TYPE_ARCHIVE	9	// archive server.
#define SQP_TYPE_BRIDGE         0x24	// remote bridge server.
#define SQP_TYPE_ADV_PRINT	0x47	// advertizing print server.
} SQP, *PSQP;

static UCHAR SrcSocketName [50];	// work area.
static UCHAR DestSocketName [50];	// work area.

//
// Routines in other modules.
//

extern USHORT NibbleToBinary (UCHAR Value);

//
// Routines in this module.
//

BOOLEAN PacketReqTypeIpx (PIPXHDR Buffer, USHORT BufferLength)
{
    USHORT socket;

    if (BufferLength < sizeof (IPXHDR)-1) {
	return FALSE;			// too short to be NetWare IPX.
    }
    if (Buffer->CheckSum != 0xffff) {
	return FALSE;			// NetWare IPX checksum is always 0xffff.
    }

    //
    // If we don't have our IPX filter on, then at least try for
    // encapsulated protocols.	If any one of them is set (i.e., NCP),
    // then we allow that packet even though raw IPX packets wouldn't
    // be captured.  This is useful to watch file sharing protocol
    // without the transport control packets cluttering up everything.
    //

    switch (Buffer->PacketType) {
	case IPXHDR_TYPE_ECHO:
	case IPXHDR_TYPE_ERROR:
	    return TRUE;
	case IPXHDR_TYPE_UNKNOWN:		// used by NetWare.
	case IPXHDR_TYPE_RIP:			// used by NetWare.
	case IPXHDR_TYPE_NCP:			// used by NetWare.
	case IPXHDR_TYPE_PEP:			// used by NetWare.
	    socket = Swap (Buffer->DestSocket);
	    if (socket > 0x4000) {		// the dest socket is anonymous.
		socket = Swap (Buffer->SrcSocket);
	    }
	    switch (socket) {
		case 0x451:
		    return PacketReqTypeNcp (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1));
		case 0x452:
		case 0x453:
		default:
		    return TRUE;
	    }
	case IPXHDR_TYPE_SPX:
	    return TRUE;
    }
    return FALSE;
} // PacketReqTypeIpx

BOOLEAN PacketCountIpx (PIPXHDR Buffer, USHORT BufferLength)
{
    USHORT socket;

    if (BufferLength < sizeof (IPXHDR)-1) {
	return FALSE;			// too short to be NetWare IPX.
    }
    if (Buffer->CheckSum != 0xffff) {
	return FALSE;			// NetWare IPX checksum is always 0xffff.
    }

    CountProtocolType (INDEX_IPX);

    //
    // If we don't have our IPX filter on, then at least try for
    // encapsulated protocols.	If any one of them is set (i.e., NCP),
    // then we allow that packet even though raw IPX packets wouldn't
    // be captured.  This is useful to watch file sharing protocol
    // without the transport control packets cluttering up everything.
    //

    switch (Buffer->PacketType) {
	case IPXHDR_TYPE_ECHO:
	    CountProtocolType (INDEX_ECHO);
	    return TRUE;
	    break;
	case IPXHDR_TYPE_ERROR:
	    CountProtocolType (INDEX_ERROR);
	    return TRUE;
	    break;
	case IPXHDR_TYPE_UNKNOWN:		// used by NetWare.
	case IPXHDR_TYPE_RIP:			// used by NetWare.
	case IPXHDR_TYPE_NCP:			// used by NetWare.
	case IPXHDR_TYPE_PEP:			// used by NetWare.
	    socket = Swap (Buffer->DestSocket);
	    if (socket > 0x4000) {		// the dest socket is anonymous.
		socket = Swap (Buffer->SrcSocket);
	    }
	    switch (socket) {
		case 0x451:
		    CountProtocolType (INDEX_NCP);
		    return TRUE;
		case 0x452:
		    CountProtocolType (INDEX_SAP_SQP);
		    return TRUE;
		case 0x453:
		    CountProtocolType (INDEX_RIP);
		    return TRUE;
		default:
		    CountProtocolType (INDEX_PEP);
		    return TRUE;
	    }
	case IPXHDR_TYPE_SPX:
	    CountProtocolType (INDEX_SPX);
	    return TRUE;
    }
    return FALSE;
} // PacketCountIpx

BOOLEAN PacketFilterIpx (PIPXHDR Buffer, USHORT BufferLength)
{
    USHORT socket;

    if (BufferLength < sizeof (IPXHDR)-1) {
	return FALSE;			// too short to be NetWare IPX.
    }
    if (Buffer->CheckSum != 0xffff) {
	return FALSE;			// NetWare IPX checksum is always 0xffff.
    }

    //
    // If we don't have our IPX filter on, then at least try for
    // encapsulated protocols.	If any one of them is set (i.e., NCP),
    // then we allow that packet even though raw IPX packets wouldn't
    // be captured.  This is useful to watch file sharing protocol
    // without the transport control packets cluttering up everything.
    //

    if (!(CaptureFilter & FILTER_IPX)) {
	switch (Buffer->PacketType) {
	    case IPXHDR_TYPE_ECHO:
		return (CaptureFilter & FILTER_ECHO) == FILTER_ECHO;
	    case IPXHDR_TYPE_ERROR:
		return (CaptureFilter & FILTER_IPX_ERROR) == FILTER_IPX_ERROR;
	    case IPXHDR_TYPE_UNKNOWN:		    // used by NetWare.
	    case IPXHDR_TYPE_RIP:		    // used by NetWare.
	    case IPXHDR_TYPE_NCP:		    // used by NetWare.
	    case IPXHDR_TYPE_PEP:		    // used by NetWare.
		socket = Swap (Buffer->DestSocket);
		if (socket > 0x4000) {		    // the dest socket is anonymous.
		    socket = Swap (Buffer->SrcSocket);
		}
		switch (socket) {
		    case 0x451:
			return (CaptureFilter & FILTER_NCP) == FILTER_NCP;
		    case 0x452:
			return (CaptureFilter & FILTER_SAP_SQP) == FILTER_SAP_SQP;
		    case 0x453:
			return (CaptureFilter & FILTER_RIP) == FILTER_RIP;
		    default:
			return (CaptureFilter & FILTER_PEP) == FILTER_PEP;
		}
	    case IPXHDR_TYPE_SPX:
		return PacketFilterSpx (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1));
	    default:
		return FALSE;			// unknown packet type.
	}
    }
    return TRUE;
} // PacketFilterIpx

VOID IpxSocketName (USHORT SocketNumber, PUCHAR NameBuffer)
{
    UCHAR *t;

    switch (SocketNumber) {
	case 1:     t = "IDP Routing"; break;
	case 2:     t = "IDP Echo"; break;
	case 3:     t = "IDP Error"; break;
	case 0x451: t = "File Service"; break;
	case 0x452: t = "Service Advertizement"; break;
	case 0x453: t = "Routing Info"; break;
	case 0x455: t = "NetBIOS"; break;
	case 0x456: t = "Diagnostic"; break;
	default: t = "0x%04x";
    }
    sprintf (NameBuffer, t, SocketNumber);
} // SocketName

PUCHAR SapSqpCmd (USHORT ResponseType)
{
    UCHAR *t;
    switch (ResponseType) {
	case SAP_TYPE_GENERAL_QUERY: t = "SAP Find"; break;
	case SAP_TYPE_NEAREST_QUERY: t = "SAP Find nearest"; break;
	case SQP_TYPE_GENERAL_QUERY: t = "SQP Find"; break;
	case SQP_TYPE_NEAREST_QUERY: t = "SQP Find nearest"; break;
	default:		     t = "SAP Advertizement";
    }
    return t;
} // SapSqpCmd

PUCHAR SapSqpServerType (USHORT ServerType)
{
    UCHAR *t;
    switch (ServerType) {
	case SAP_TYPE_WILD: t = "any"; break;
	case SAP_TYPE_UNKNOWN: t = "unknown"; break;
	case SAP_TYPE_PRINT: t = "hidden print"; break;
	case SAP_TYPE_FILE: t = "file"; break;
	case SAP_TYPE_JOB: t = "job"; break;
	case SAP_TYPE_ARCHIVE: t = "archive"; break;
	case SAP_TYPE_BRIDGE: t = "bridge"; break;
	case SAP_TYPE_ADV_PRINT: t = "print"; break;
	default: t = "(undecoded)";
    }
    return t;
} // SapSqpServerType

PUCHAR PacketSummarySap (PSAP Buffer, USHORT BufferLength)
{
    UCHAR *p, *t, *q;
    USHORT socket;

    if (!(DisplayFilter & FILTER_SAP_SQP)) {
	return NULL;			// we aren't decoding SAP.
    }
    if (BufferLength < sizeof (SAP)-1) {
	return NULL;			// too short to be NetWare SAP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    t = SapSqpCmd (Swap (Buffer->ResponseType));
    q = SapSqpServerType (Swap (Buffer->ServerType));
    sprintf (p, "%s %s server", t, q);
    return p;
} // PacketSummarySap

PUCHAR PacketSummaryPep (PIPXHDR Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;
    USHORT socket;
    PRIP rp;

    if (!(DisplayFilter & FILTER_PEP)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (IPXHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    socket = Swap (Buffer->DestSocket);
    if (socket > 0x4000) {		// the dest socket is anonymous.
	socket = Swap (Buffer->SrcSocket);
    }

    switch (socket) {
	case 0x451:
	    free (p);
	    return PacketSummaryNcp (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1));
	case 0x452:
	    free (p);
	    return PacketSummarySap ((PSAP)Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1));
	    break;
	case 0x453:
	    rp = (PRIP)Buffer->Buffer;
	    t = rp->Network;
	    sprintf (p, "RIP net=%02x%02x%02x%02x hops=%u",
		     *(t+0), *(t+1), *(t+2), *(t+3), Swap (rp->HopCount));
	    return p;
	    break;
	case 0x455:
	    t = "PEP NetBIOS";
	    break;
	case 0x456:
	    t = "PEP Diagnostic";
	    break;
	default:
	    free (p);
	    return NULL;
    }
    strcpy (p, t);
    return p;
} // PacketSummaryPep

PUCHAR PacketSummaryIpx (PIPXHDR Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_IPX)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (IPXHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }
    if (Buffer->CheckSum != 0xffff) {
	return NULL;			// NetWare IPX checksum is always 0xffff.
    }

    p = malloc (128);
    if (p == NULL) {
	return NULL;
    }

    IpxSocketName (Swap (Buffer->SrcSocket), SrcSocketName);
    IpxSocketName (Swap (Buffer->DestSocket), DestSocketName);

    switch (Buffer->PacketType) {
	case IPXHDR_TYPE_ECHO:
	    sprintf (p, "ECHO DS=%s, SS=%s", SrcSocketName, DestSocketName);
	    return p;
	    break;
	case IPXHDR_TYPE_ERROR:
	    sprintf (p, "ERROR DS=%s, SS=%s", SrcSocketName, DestSocketName);
	    return p;
	    break;
	case IPXHDR_TYPE_UNKNOWN:		// used by NetWare.
	case IPXHDR_TYPE_RIP:			// used by NetWare.
	case IPXHDR_TYPE_NCP:			// used by NetWare.
	case IPXHDR_TYPE_PEP:			// used by NetWare.
	    t = PacketSummaryPep (Buffer, BufferLength);
	    if (t != NULL) {
		free (p);
		return t;
	    }
	    break;
	case IPXHDR_TYPE_SPX:			// (same as SPP).
	    t = PacketSummarySpx (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1));
	    if (t != NULL) {
		free (p);
		return t;
	    }
	    break;
    }

    //
    // We have to decode at our own level.
    //

    sprintf (p, "IPX type=0x%02x length=%u",
	     Buffer->PacketType, Swap (Buffer->Length));
    return p;
} // PacketSummaryIpx

VOID IpxTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "IPX:");
} // IpxTrailer

PDETREC PacketDetailSap (PSAP Buffer, USHORT BufferLength)
{
    PDETREC r=NULL;
    UCHAR *p, *t;
    USHORT socket, cmd;
    PSQP sqpbuf;

    if (!(DisplayFilter & FILTER_SAP_SQP)) {
	return NULL;			// we aren't decoding SAP.
    }
    if (BufferLength < sizeof (SAP)-1) {
	return NULL;			// too short to be NetWare SAP.
    }

    AppendHeader (&r, "SAP:  ----- Service Advertizement Protocol header -----");

    cmd = Swap (Buffer->ResponseType);
    t = SapSqpCmd (cmd);
    sprintf (ScratchBuf, "SAP:  Response type = 0x%04x (%s)",
	     Buffer->ResponseType, t);
    AppendDetRec (&r, ScratchBuf);

    t = SapSqpServerType (Swap (Buffer->ServerType));
    sprintf (ScratchBuf, "SAP:  Server type = 0x%04x (%s)",
	     Swap (Buffer->ServerType), t);
    AppendDetRec (&r, ScratchBuf);

    switch (cmd) {
	case SAP_TYPE_GENERAL_QUERY:
	case SAP_TYPE_NEAREST_QUERY:
	    sprintf (ScratchBuf, "SAP:  Server name = '%s'", Buffer->ServerName);
	    AppendDetRec (&r, ScratchBuf);
	    p = Buffer->NetAddr;
	    sprintf (ScratchBuf, "SAP:  Network address = [%02x.%02x.%02x.%02x]",
		     *(p+0), *(p+1), *(p+2), *(p+3));
	    AppendDetRec (&r, ScratchBuf);
	    ResolveDlcAddress (ScratchBuf1, Buffer->NodeAddr);
	    sprintf (ScratchBuf, "SAP:  Node address = %s", ScratchBuf1);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "SAP:  Socket = 0x%04x", Buffer->SockAddr);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "SAP:  Hop count = %u", Swap (Buffer->HopCount));
	    AppendDetRec (&r, ScratchBuf);
    }
    return r;
} // PacketDetailSap

PDETREC PacketDetailPep (PIPXHDR Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    UCHAR *t;
    USHORT socket, len, type;
    PSAP psap;
    PRIP rp;

    if (!(DisplayFilter & FILTER_PEP)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (IPXHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }
    socket = Swap (Buffer->DestSocket);
    if (socket > 0x4000) {		// the dest socket is anonymous.
	socket = Swap (Buffer->SrcSocket);
    }

    switch (socket) {
	case 0x451:
	    if ((q=PacketDetailNcp (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1))) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	    break;
	case 0x452:
	    if ((q=PacketDetailSap ((PSAP)Buffer->Buffer, BufferLength-(sizeof (IPXHDR)-1))) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	    break;
	case 0x453:
	    AppendHeader (&r, "RIP:  ----- Routing Information Protocol header -----");
	    rp = (PRIP)Buffer->Buffer;
	    sprintf (ScratchBuf, "RIP:  Operation = 0x%04x", Swap (rp->Operation));
	    AppendDetRec (&r, ScratchBuf);
	    t = rp->Network;
	    sprintf (ScratchBuf, "RIP:  Network = %02x%02x%02x%02x",
		     *(t+0), *(t+1), *(t+2), *(t+3));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "RIP:  Hops = %u", Swap (rp->HopCount));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	case 0x455:
	    AppendHeader (&r, "PEP:  ----- Packet Exchange Protocol header -----");
	    AppendDetRec (&r, "PEP:  Undecoded NetBIOS Frame");
	    break;
	case 0x456:
	    AppendHeader (&r, "PEP:  ----- Packet Exchange Protocol header -----");
	    AppendDetRec (&r, "PEP:  Undecoded Diagnostic Frame");
	    break;
	default:
	    AppendHeader (&r, "PEP:  ----- Packet Exchange Protocol header -----");
	    AppendDetRec (&r, "PEP:  Undecoded PEP Frame Type");
    }
    return r;
} // PacketDetailPep

PDETREC PacketDetailIpx (PIPXHDR Buffer, USHORT BufferLength)
{
    UCHAR *t;
    PDETREC r=NULL, q;
    PRIP rp;

    if (!(DisplayFilter & FILTER_IPX)) {
	return NULL;			// we aren't decoding IPX.
    }
    if (BufferLength < sizeof (IPXHDR)-1) {
	return NULL;			// too short to be NetWare IPX.
    }
    if (Buffer->CheckSum != 0xffff) {
	return NULL;			// NetWare IPX checksum is always 0xffff.
    }

    IpxSocketName (Swap (Buffer->SrcSocket), SrcSocketName);
    IpxSocketName (Swap (Buffer->DestSocket), DestSocketName);

    AppendHeader (&r, "IPX:  ----- Internet Packet Exchange header -----");

    sprintf (ScratchBuf, "IPX:  Checksum = 0x%04x", Swap (Buffer->CheckSum));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IPX:  Length = %u (0x%04x hex) bytes",
	     Swap (Buffer->Length), Swap (Buffer->Length));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IPX:  Transport Control = 0x%02x", Buffer->TransportCtrl);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf,
	     "IPX:          %04x .... = Reserved",
	     NibbleToBinary (Buffer->TransportCtrl >> 4));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf,
	     "IPX:          .... %04x = Hop count",
	     NibbleToBinary (Buffer->TransportCtrl & 0x0f));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IPX:  Packet Type = 0x%02x", Buffer->PacketType);
    AppendDetRec (&r, ScratchBuf);

    //
    // Display destination address.
    //

    t = Buffer->DestNetwork;
    ResolveDlcAddress (ScratchBuf1, Buffer->DestNode);
    sprintf (ScratchBuf,
    "IPX:  Dest:    Net=%02x%02x%02x%02x, Node=%s, Socket=0x%04x (%s)",
	     *(t+0), *(t+1), *(t+2), *(t+3),
	     ScratchBuf1,
	     Swap (Buffer->DestSocket),
	     DestSocketName);
    AppendDetRec (&r, ScratchBuf);

    //
    // Display source address.
    //

    t = Buffer->SrcNetwork;
    ResolveDlcAddress (ScratchBuf1, Buffer->SrcNode);
    sprintf (ScratchBuf,
    "IPX:  Source:  Net=%02x%02x%02x%02x, Node=%s, Socket=0x%04x (%s)",
	     *(t+0), *(t+1), *(t+2), *(t+3),
	     ScratchBuf1,
	     Swap (Buffer->SrcSocket),
	     SrcSocketName);
    AppendDetRec (&r, ScratchBuf);

    IpxTrailer (&r);

    //
    // Now try to decode at the higher layer.
    //

    switch (Buffer->PacketType) {
	case IPXHDR_TYPE_UNKNOWN:		// used by NetWare.
	case IPXHDR_TYPE_NCP:			// used by NetWare.
	case IPXHDR_TYPE_PEP:			// used by NetWare.
	    q = PacketDetailPep (Buffer, BufferLength);
	    if (q != NULL) {
		JoinDetRec (&r, q);
	    }
	    break;
	case IPXHDR_TYPE_SPX:			// used by NetWare/3Com.
	    q = PacketDetailSpx (Buffer->Buffer, BufferLength-(sizeof (IPXHDR)+1));
	    if (q != NULL) {
		JoinDetRec (&r, q);
	    }
	    break;
	case IPXHDR_TYPE_RIP:
	    AppendHeader (&r, "RIP:  ----- Routing Information Protocol header -----");
	    rp = (PRIP)Buffer->Buffer;
	    sprintf (ScratchBuf, "RIP:  Operation = 0x%04x", Swap (rp->Operation));
	    AppendDetRec (&r, ScratchBuf);
	    t = rp->Network;
	    sprintf (ScratchBuf, "RIP:  Network = %02x%02x%02x%02x",
		     *(t+0), *(t+1), *(t+2), *(t+3));
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "RIP:  Hops = %u", Swap (rp->HopCount));
	    AppendDetRec (&r, ScratchBuf);
	    break;
    }
    return r;
} // PacketDetailIpx
