//
// MODULE NAME:  GSLLCTDI.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for
//	General Software's Embedded LAN LLCTDI protocol providers.
//
//	There are two main entrypoints to this module.	PacketSummaryLlcTdi
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailLlcTdi returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/11/24.	Original for 1.9.
//	S. E. Jones	93/02/05.	#2.2, added GS FSP callouts.
//	S. E. Jones	93/02/10.	#2.3, tallied creprc/delprc.
//	S. E. Jones	93/03/16.	#2.5, added GSIPC.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// Formats and protocols.
//

#define TDIBITMASK unsigned short

#define TRANSPORT_ADDRESS_LENGTH	64	// max size of transport node name.

typedef struct _TRANSPORT_ADDRESS {
    TDIBITMASK Flags;				// indicates which components are valid.
    UCHAR AsciizNodeName [TRANSPORT_ADDRESS_LENGTH]; // machine name, etc.
    USHORT Tsap;				// transport service access point.
} TRANSPORT_ADDRESS, *PTRANSPORT_ADDRESS;

#define TRANSPORT_ADDRESS_FLAGS_NAME	0x0001	// node name is specified.
#define TRANSPORT_ADDRESS_FLAGS_TSAP	0x0002	// TSAP is specified.

typedef unsigned short TDIHANDLE, *PTDIHANDLE;

typedef struct _XPORT_HEADER {
    ULONG Signature;
    PVOID SourceContext;		// 32-bit token for sender (connection).
    PVOID DestinationContext;		// 32-bit token for receiver (connection).
    ULONG SequenceNumber;		// number of this connection packet.
    ULONG AckNumber;			// number of connection packet we're ACKing.
    USHORT Flags;			// flags/this packet.
    USHORT Length;			// length of data following header.
    USHORT SystemCommand;		// transport-level command code.
    UCHAR Reserved;			// reserved alignment byte, MBZ.
} XPORT_HEADER, *PXPORT_HEADER;

#define XPORT_HEADER_SIGNATURE	0x41322314L	// magic signature.

#define XPORT_HEADER_FLAGS_DATA         0x0001	// packet contains user data.
#define XPORT_HEADER_FLAGS_ENDPOINT	0x0002	// data goes to endpoint (datagram).
#define XPORT_HEADER_FLAGS_CONNECTION	0x0004	// data goes to connection (sequenced).
#define XPORT_HEADER_FLAGS_MUST_ACK	0x0008	// connection data requires ACK.
#define XPORT_HEADER_FLAGS_ACK		0x0010	// AckNumber is valid.
#define XPORT_HEADER_FLAGS_EOR		0x0020	// end of record.
#define XPORT_HEADER_FLAGS_EXPEDITED	0x0040	// expedited data.

#define XPORT_COMMAND_NULL		0	// no command.
#define XPORT_COMMAND_CONNECT_REQ	1	// connection request.
#define XPORT_COMMAND_CONNECT_ACK	2	// connection acknowlege.
#define XPORT_COMMAND_DISCONNECT_REQ	3	// disconnect request.
#define XPORT_COMMAND_DISCONNECT_ACK	4	// disconnect acknowlege.
#define XPORT_COMMAND_DATAGRAM		5	// datagram sent to endpoint.

typedef struct _XPORT_CONNECT_REQ {	// connection request.
    TRANSPORT_ADDRESS DestAddress;	// address to connect to.
    TRANSPORT_ADDRESS SrcAddress;	// address wanting connection.
} XPORT_CONNECT_REQ, *PXPORT_CONNECT_REQ;

typedef struct _XPORT_CONNECT_ACK {	// connection acknowlegement.
    TRANSPORT_ADDRESS DestAddress;	// address wanting connection.
    TRANSPORT_ADDRESS SrcAddress;	// responding address.
} XPORT_CONNECT_ACK, *PXPORT_CONNECT_ACK;

typedef struct _XPORT_DISCONNECT_REQ {	// disconnect request.
    USHORT Reason;			// reason code.
} XPORT_DISCONNECT_REQ, *PXPORT_DISCONNECT_REQ;

#define XPORT_DISCONNECT_REASON_API	0	// normal disconnection.
#define XPORT_DISCONNECT_REASON_ABORT	1	// remote user aborted.
#define XPORT_DISCONNECT_REASON_TIMEOUT 2	// remote TDI timed-out.
#define XPORT_DISCONNECT_REASON_PROT	3	// remote TDI saw protocol error.

typedef struct _XPORT_DISCONNECT_ACK {	// disconnect acknowlegement.
    USHORT Reason;			// reason code (echoed).
} XPORT_DISCONNECT_ACK, *PXPORT_DISCONNECT_ACK;

typedef struct _XPORT_DATAGRAM {	// datagram (unacknowleged).
    TRANSPORT_ADDRESS DestAddress;	// address to send datagram to.
    TRANSPORT_ADDRESS SrcAddress;	// address datagram is sent from.
} XPORT_DATAGRAM, *PXPORT_DATAGRAM;

typedef struct _XPORT_PACKET {
    XPORT_HEADER XportHeader;
    union {
	XPORT_CONNECT_REQ ConnectReq;
	XPORT_CONNECT_ACK ConnectAck;
	XPORT_DISCONNECT_REQ DisconnectReq;
	XPORT_DISCONNECT_ACK DisconnectAck;
	XPORT_DATAGRAM Datagram;
	UCHAR UserData [1];		// user data array (0-n user bytes).
    } Tdu;				// the transport data unit.
} XPORT_PACKET, *PXPORT_PACKET;

//
// Routines in other modules.
//

extern VOID AppendBitRecord (
    PDETREC *Ptr,
    UCHAR *Prefix,
    UCHAR *BitName,
    USHORT BitMask,
    USHORT Value);

//
// Routines in this module.
//

BOOLEAN PacketReqTypeLlcTdi (PVOID Buffer, USHORT BufferLength)
{
    PXPORT_HEADER pheader;

    pheader = (PXPORT_HEADER)Buffer;

    if (BufferLength < sizeof (XPORT_HEADER)) {
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != XPORT_HEADER_SIGNATURE) {
	return FALSE;
    }
    if (pheader->Flags & XPORT_HEADER_FLAGS_DATA) {
	return PacketReqTypeGsFsp (pheader+sizeof (XPORT_HEADER), pheader->Length);
    } else if (pheader->SystemCommand == XPORT_COMMAND_CONNECT_REQ) {
	CountFileIoType (FILEIO_CREATE_PROCESS);
    } else if (pheader->SystemCommand == XPORT_COMMAND_DISCONNECT_REQ) {
	CountFileIoType (FILEIO_DESTROY_PROCESS);
    }
    return TRUE;		       // found one of ours.
} // PacketReqTypeLlcTdi

BOOLEAN PacketFilterLlcTdi (PVOID Buffer, USHORT BufferLength)
{
    PXPORT_PACKET pxport;
    PXPORT_HEADER pheader;

    pheader = (PXPORT_HEADER)Buffer;
    pxport = (PXPORT_PACKET)Buffer;

    if (BufferLength < sizeof (XPORT_HEADER)) {
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != XPORT_HEADER_SIGNATURE) {
	return FALSE;
    }

    if (CaptureFilter & FILTER_LLCTDI) {
	return TRUE;			// we're accepting LLCTDI.
    }

    return FALSE;
} // PacketFilterLlcTdi

PUCHAR LlcTdiCmdName (USHORT Cmd)
{
    switch (Cmd) {
	case XPORT_COMMAND_NULL:	    return "Null Command";
	case XPORT_COMMAND_CONNECT_REQ:     return "Connection Request";
	case XPORT_COMMAND_CONNECT_ACK:     return "Connection Acknowlegement";
	case XPORT_COMMAND_DISCONNECT_REQ:  return "Disconnect Request";
	case XPORT_COMMAND_DISCONNECT_ACK:  return "Disconnect Acknowlegement";
	case XPORT_COMMAND_DATAGRAM:	    return "Datagram";
	default:			    return "Undecoded";
    }
} // LlcTdiCmdName

PUCHAR PacketSummaryLlcTdi (PVOID Buffer, USHORT BufferLength)
{
    PXPORT_PACKET pxport;
    PXPORT_HEADER pheader;
    UCHAR *p, *t;

    pheader = (PXPORT_HEADER)Buffer;
    pxport = (PXPORT_PACKET)Buffer;

    if (!(DisplayFilter & FILTER_LLCTDI)) {
	return NULL;			// we aren't decoding LLCTDI.
    }

    if (BufferLength < sizeof (XPORT_HEADER)) {
	return NULL;			// partial record not decoded.
    }

    if (pheader->Signature != XPORT_HEADER_SIGNATURE) {
	return NULL;			// this isn't LLCTDI.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    switch (pheader->SystemCommand) {
	case XPORT_COMMAND_DATAGRAM:
	    sprintf (p, "TDI %s %u bytes",
		     LlcTdiCmdName (pheader->SystemCommand),
		     pheader->Length);
	    break;

	case XPORT_COMMAND_NULL:
	    strcpy (p, "TDI Null");	// if not connection data.

	    if (pheader->Flags & XPORT_HEADER_FLAGS_CONNECTION) {
		strcpy (p, "TDI ");

		if (pheader->Flags & XPORT_HEADER_FLAGS_DATA) {
		    if ((t=PacketSummaryGsFsp (
			   ((UCHAR *)Buffer + sizeof (XPORT_HEADER)),
			   pheader->Length)) != NULL) {
			strcpy (p, t);
			free (t);
			return p;
		    }
		    if ((t=PacketSummaryGsIpc (
			   ((UCHAR *)Buffer + sizeof (XPORT_HEADER)),
			   pheader->Length)) != NULL) {
			strcpy (p, t);
			free (t);
			return p;
		    }
		    sprintf (ScratchBuf, "Data (%u bytes)", pheader->Length);
		    strcat (p, ScratchBuf);
		}

		t = p + strlen (p);
		sprintf (t, "Seq=%08lx  Ack=%08lx  ",
			 pheader->SequenceNumber,
			 pheader->AckNumber);

		if (pheader->Flags & XPORT_HEADER_FLAGS_EXPEDITED) {
		    strcat (p, "EXP ");
		}
		if (pheader->Flags & XPORT_HEADER_FLAGS_MUST_ACK) {
		    strcat (p, "ARQ ");
		}
		if (pheader->Flags & XPORT_HEADER_FLAGS_ACK) {
		    strcat (p, "ACK");
		}
	    }
	    break;

	case XPORT_COMMAND_CONNECT_REQ:
	case XPORT_COMMAND_CONNECT_ACK:
	case XPORT_COMMAND_DISCONNECT_REQ:
	case XPORT_COMMAND_DISCONNECT_ACK:
	default:
	    sprintf (p, "TDI %s", LlcTdiCmdName (pheader->SystemCommand));
    }
    return p;
} // PacketSummaryLlcTdi

VOID LlcTdiTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "TDI:");
} // LlcTdiTrailer

VOID AppendTransportAddress (PDETREC *r, PTRANSPORT_ADDRESS p)
{
    USHORT Flags;

    Flags = p->Flags;
    sprintf (ScratchBuf, "TDI:    Flags = 0x%04x", Flags);
    AppendDetRec (r, ScratchBuf);

    if (Flags & TRANSPORT_ADDRESS_FLAGS_NAME) {
	AppendDetRec (r, "TDI:            0x0001 - Node name specified");
    }
    if (Flags & TRANSPORT_ADDRESS_FLAGS_TSAP) {
	AppendDetRec (r, "TDI:            0x0002 - TSAP specified");
    }

    if (Flags & TRANSPORT_ADDRESS_FLAGS_NAME) {
	sprintf (ScratchBuf, "TDI:    Name = '%s'", p->AsciizNodeName);
	AppendDetRec (r, ScratchBuf);
    }
    if (Flags & TRANSPORT_ADDRESS_FLAGS_TSAP) {
	sprintf (ScratchBuf, "TDI:    TSAP = 0x%04x", p->Tsap);
	AppendDetRec (r, ScratchBuf);
    }
    LlcTdiTrailer (r);
} // AppendTransportAddress

PDETREC PacketDetailLlcTdi (PVOID Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    UCHAR *p, cmd, *t;
    USHORT Flags;
    PXPORT_PACKET pxport;
    PXPORT_HEADER pheader;
    PXPORT_DATAGRAM pdg;
    PXPORT_CONNECT_REQ pcr;
    PXPORT_CONNECT_ACK pca;
    PXPORT_DISCONNECT_REQ pdr;
    PXPORT_DISCONNECT_ACK pda;

    pheader = (PXPORT_HEADER)Buffer;
    pxport = (PXPORT_PACKET)Buffer;

    if (!(DisplayFilter & FILTER_LLCTDI)) {
	return NULL;			// we aren't decoding LLCTDI.
    }

    if (BufferLength < sizeof (XPORT_HEADER)) {
	return NULL;			// partial record not decoded.
    }

    if (pheader->Signature != XPORT_HEADER_SIGNATURE) {
	return NULL;			// this isn't LLCTDI.
    }

    sprintf (ScratchBuf, "TDI:  ----- General Software LLC TDI %s -----",
	     LlcTdiCmdName (pheader->SystemCommand));
    AppendHeader (&r, ScratchBuf);

    sprintf (ScratchBuf, "TDI:  System Command = 0x%04x  (%s), Length = %u (0x%04x)",
	     pheader->SystemCommand, LlcTdiCmdName (pheader->SystemCommand),
	     pheader->Length, pheader->Length);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TDI:  Source Context = 0x%08lx   Dest Context = 0x%08lx",
	     pheader->SourceContext, pheader->DestinationContext);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TDI:  Sequence Number = 0x%08lx  ACK Number = 0x%08lx",
	     pheader->SequenceNumber, pheader->AckNumber);
    AppendDetRec (&r, ScratchBuf);

    Flags = pheader->Flags;
    sprintf (ScratchBuf, "TDI:  Transport Flags = 0x%04x", Flags);
    AppendDetRec (&r, ScratchBuf);
    AppendBitRecord (&r, "TDI:", "Data", XPORT_HEADER_FLAGS_DATA, Flags);
    AppendBitRecord (&r, "TDI:", "Endpoint", XPORT_HEADER_FLAGS_ENDPOINT, Flags);
    AppendBitRecord (&r, "TDI:", "Connection", XPORT_HEADER_FLAGS_CONNECTION, Flags);
    AppendBitRecord (&r, "TDI:", "Must Ack", XPORT_HEADER_FLAGS_MUST_ACK, Flags);
    AppendBitRecord (&r, "TDI:", "Ack", XPORT_HEADER_FLAGS_ACK, Flags);
    AppendBitRecord (&r, "TDI:", "EOR (End of Record)", XPORT_HEADER_FLAGS_EOR, Flags);
    AppendBitRecord (&r, "TDI:", "Expedited", XPORT_HEADER_FLAGS_EXPEDITED, Flags);

    LlcTdiTrailer (&r);

    switch (pheader->SystemCommand) {
	case XPORT_COMMAND_DATAGRAM:
	    pdg = (PXPORT_DATAGRAM)(&pxport->Tdu.Datagram);
	    sprintf (ScratchBuf, "TDI:  Source Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pdg->SrcAddress);
	    sprintf (ScratchBuf, "TDI:  Destination Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pdg->DestAddress);
	    break;

	case XPORT_COMMAND_CONNECT_REQ:
	    pcr = (PXPORT_CONNECT_REQ)(&pxport->Tdu.ConnectReq);
	    sprintf (ScratchBuf, "TDI:  Source Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pcr->SrcAddress);
	    sprintf (ScratchBuf, "TDI:  Destination Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pcr->DestAddress);
	    break;

	case XPORT_COMMAND_CONNECT_ACK:
	    pca = (PXPORT_CONNECT_ACK)(&pxport->Tdu.ConnectAck);
	    sprintf (ScratchBuf, "TDI:  Source Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pca->SrcAddress);
	    sprintf (ScratchBuf, "TDI:  Destination Transport Address");
	    AppendDetRec (&r, ScratchBuf);
	    AppendTransportAddress (&r, &pca->DestAddress);
	    break;

	case XPORT_COMMAND_NULL:
	    if ((q=PacketDetailGsFsp (
		       ((UCHAR *)Buffer + sizeof (XPORT_HEADER)),
		       pheader->Length)) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	    if ((q=PacketDetailGsIpc (
		       ((UCHAR *)Buffer + sizeof (XPORT_HEADER)),
		       pheader->Length)) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }

	case XPORT_COMMAND_DISCONNECT_REQ:
	case XPORT_COMMAND_DISCONNECT_ACK:
	default:			// nothing else to print.
	    return r;			// DO NOTHING.

    }
    return r;
} // PacketDetailLlcTdi
