//***	FSP.H -- File Sharing Protocol Definitions.
//
//1.	Functional Description.
//	This include file contains frame formats for the file
//	sharing protocol employed by Embedded LAN.  Some basic
//	rules for fields:
//
//	1. When a request fails, the response is returned but not the
//	data that would ordinarily come with it.
//
//	2. All response fields are invalid if the status field is nonzero.
//
//	3. All string arrays (FSP_FILENAME, etc.) are ASCIIZ (0-byte terminated).
//	Strings have case-sensitivity.
//
//	4. All words & doublewords are little-endian format (byte reversed).
//
//	5. Unsigned short Boolean fields are 0 if FALSE, otherwise TRUE.
//
//2.	Modification History.
//	S. E. Jones	93/01/30.	Original for Embedded LAN.
//
//3.	NOTICE: Copyright (C) 1993 General Software, Inc.

#ifndef _FSP
#define _FSP

#define FSP_FILENAME_LENGTH	64

typedef unsigned char FSP_FILENAME [FSP_FILENAME_LENGTH], **PFSP_FILENAME;

typedef unsigned short FSP_STATUS, *PFSP_STATUS;

typedef unsigned short FSP_HANDLE, *PFSP_HANDLE;

//
// Open protocol.
//

typedef struct _FSP_REQ_OPEN {
    FSP_FILENAME FileName;
    USHORT Flags;
} FSP_REQ_OPEN, *PFSP_REQ_OPEN;

typedef struct _FSP_RSP_OPEN {
    FSP_STATUS Status;
    FSP_HANDLE Handle;
} FSP_RSP_OPEN, *PFSP_RSP_OPEN;

//
// Close protocol.
//

typedef struct _FSP_REQ_CLOSE {
    FSP_HANDLE Handle;
} FSP_REQ_CLOSE, *PFSP_REQ_CLOSE;

typedef struct _FSP_RSP_CLOSE {
    FSP_STATUS Status;
} FSP_RSP_CLOSE, *PFSP_RSP_CLOSE;

//
// Create protocol.
//

typedef struct _FSP_REQ_CREATE {
    FSP_FILENAME FileName;
    USHORT IsDirectory;
    USHORT Flags;
} FSP_REQ_CREATE, *PFSP_REQ_CREATE;

typedef struct _FSP_RSP_CREATE {
    FSP_STATUS Status;
    FSP_HANDLE Handle;
} FSP_RSP_CREATE, *PFSP_RSP_CREATE;

//
// Delete protocol.
//

typedef struct _FSP_REQ_DELETE {
    FSP_FILENAME FileName;
    USHORT IsDirectory;
} FSP_REQ_DELETE, *PFSP_REQ_DELETE;

typedef struct _FSP_RSP_DELETE {
    FSP_STATUS Status;
} FSP_RSP_DELETE, *PFSP_RSP_DELETE;

//
// Read protocol (data is shipped as second message with separate EOR).
//

typedef struct _FSP_REQ_READ {
    FSP_HANDLE Handle;
    ULONG Offset;
    USHORT BytesToRead;
} FSP_REQ_READ, *PFSP_REQ_READ;

typedef struct _FSP_RSP_READ {
    FSP_STATUS Status;
    USHORT BytesRead;
} FSP_RSP_READ, *PFSP_RSP_READ;

//
// Write protocol (data is shipped as second message with separate EOR).
//

typedef struct _FSP_REQ_WRITE {
    FSP_HANDLE Handle;
    ULONG Offset;
    USHORT BytesToWrite;
} FSP_REQ_WRITE, *PFSP_REQ_WRITE;

typedef struct _FSP_RSP_WRITE {
    FSP_STATUS Status;
    USHORT BytesWritten;
} FSP_RSP_WRITE, *PFSP_RSP_WRITE;

//
// Rename protocol.
//

typedef struct _FSP_REQ_RENAME {
    FSP_FILENAME OldFileName;
    FSP_FILENAME NewFileName;
} FSP_REQ_RENAME, *PFSP_REQ_RENAME;

typedef struct _FSP_RSP_RENAME {
    FSP_STATUS Status;
} FSP_RSP_RENAME, *PFSP_RSP_RENAME;

//
// Find First protocol (search data is sent as second message with separate EOR).
//

typedef struct _FSP_REQ_FINDFIRST {
    FSP_FILENAME FileName;
    USHORT SearchAttributeFlags;
} FSP_REQ_FINDFIRST, *PFSP_REQ_FINDFIRST;

typedef struct _FSP_RSP_FINDFIRST {
    FSP_STATUS Status;
    FSP_HANDLE Handle;
} FSP_RSP_FINDFIRST, *PFSP_RSP_FINDFIRST;

//
// Find Next protocol (data is sent as second message with separate EOR).
//

typedef struct _FSP_REQ_FINDNEXT {
    FSP_HANDLE Handle;
} FSP_REQ_FINDNEXT, *PFSP_REQ_FINDNEXT;

typedef struct _FSP_RSP_FINDNEXT {
    FSP_STATUS Status;
} FSP_RSP_FINDNEXT, *PFSP_RSP_FINDNEXT;

//
// Commit protocol.
//

typedef struct _FSP_REQ_COMMIT {
    FSP_HANDLE Handle;
} FSP_REQ_COMMIT, *PFSP_REQ_COMMIT;

typedef struct _FSP_RSP_COMMIT {
    FSP_STATUS Status;
} FSP_RSP_COMMIT, *PFSP_RSP_COMMIT;

//
// File Info protocol.
//

typedef struct _FSP_REQ_FILEINFO {
    FSP_FILENAME FileName;
    USHORT IsSetOperation;
    USHORT NewAttributes;
} FSP_REQ_FILEINFO, *PFSP_REQ_FILEINFO;

typedef struct _FSP_RSP_FILEINFO {
    FSP_STATUS Status;
    USHORT ReturnedAttributes;
} FSP_RSP_FILEINFO, *PFSP_RSP_FILEINFO;

//
// File Time protocol.
//

typedef struct _FSP_REQ_FILETIME {
    FSP_HANDLE Handle;
    USHORT IsSetOperation;
    ULONG NewTimeStamp;
} FSP_REQ_FILETIME, *PFSP_REQ_FILETIME;

typedef struct _FSP_RSP_FILETIME {
    FSP_STATUS Status;
    ULONG ReturnedTimeStamp;
} FSP_RSP_FILETIME, *PFSP_RSP_FILETIME;

//
// File System Information protocol (data is sent as second message with separate EOR).
//

typedef struct _FSP_REQ_FSINFO {
    FSP_HANDLE Handle;
} FSP_REQ_FSINFO, *PFSP_REQ_FSINFO;

typedef struct _FSP_RSP_FSINFO {
    FSP_STATUS Status;
} FSP_RSP_FSINFO, *PFSP_RSP_FSINFO;

//
// Lock protocol.
//

typedef struct _FSP_REQ_LOCK {
    FSP_HANDLE Handle;
    ULONG ByteOffset;
    ULONG Length;
} FSP_REQ_LOCK, *PFSP_REQ_LOCK;

typedef struct _FSP_RSP_LOCK {
    FSP_STATUS Status;
} FSP_RSP_LOCK, *PFSP_RSP_LOCK;

//
// Unlock protocol.
//

typedef struct _FSP_REQ_UNLOCK {
    FSP_HANDLE Handle;
    ULONG ByteOffset;
    ULONG Length;
} FSP_REQ_UNLOCK, *PFSP_REQ_UNLOCK;

typedef struct _FSP_RSP_UNLOCK {
    FSP_STATUS Status;
} FSP_RSP_UNLOCK, *PFSP_RSP_UNLOCK;

//
// Generic request type.
//

typedef struct _FSP_FRAME {
    ULONG Signature;

#define FSP_FRAME_SIGNATURE	0x465350ffL // 'FSP'+0xff.

    USHORT Cmd;

#define FSP_FRAME_REQ_NULL	0	// do nothing.
#define FSP_FRAME_RSP_NULL	1	// do nothing.

#define FSP_FRAME_REQ_OPEN	2	// open object.
#define FSP_FRAME_RSP_OPEN	3	// open object.

#define FSP_FRAME_REQ_CLOSE	4	// close handle.
#define FSP_FRAME_RSP_CLOSE	5	// close handle.

#define FSP_FRAME_REQ_CREATE	6	// create object.
#define FSP_FRAME_RSP_CREATE	7	// create object.

#define FSP_FRAME_REQ_DELETE	8	// delete object.
#define FSP_FRAME_RSP_DELETE	9	// delete object.

#define FSP_FRAME_REQ_READ	10	// read object.
#define FSP_FRAME_RSP_READ	11	// read object.

#define FSP_FRAME_REQ_WRITE	12	// write object.
#define FSP_FRAME_RSP_WRITE	13	// write object.

#define FSP_FRAME_REQ_RENAME	14	// rename object.
#define FSP_FRAME_RSP_RENAME	15	// rename object.

#define FSP_FRAME_REQ_FINDFIRST 16	// find first matching object.
#define FSP_FRAME_RSP_FINDFIRST 17	// find first matching object.

#define FSP_FRAME_REQ_FINDNEXT	18	// find next matching object.
#define FSP_FRAME_RSP_FINDNEXT	19	// find next matching object.

#define FSP_FRAME_REQ_COMMIT	20	// commit object.
#define FSP_FRAME_RSP_COMMIT	21	// commit object.

#define FSP_FRAME_REQ_FILEINFO	22	// get/set file information.
#define FSP_FRAME_RSP_FILEINFO	23	// get/set file information.

#define FSP_FRAME_REQ_FILETIME	24	// get/set file timestamp.
#define FSP_FRAME_RSP_FILETIME	25	// get/set file timestamp.

#define FSP_FRAME_REQ_FSINFO	26	// get file system information.
#define FSP_FRAME_RSP_FSINFO	27	// get file system information.

#define FSP_FRAME_REQ_LOCK	28	// lock object.
#define FSP_FRAME_RSP_LOCK	29	// lock object.

#define FSP_FRAME_REQ_UNLOCK	30	// unlock object.
#define FSP_FRAME_RSP_UNLOCK	31	// unlock object.

    union {
	FSP_REQ_OPEN FspOpenReq;
	FSP_RSP_OPEN FspOpenRsp;

	FSP_REQ_CLOSE FspCloseReq;
	FSP_RSP_CLOSE FspCloseRsp;

	FSP_REQ_CREATE FspCreateReq;
	FSP_RSP_CREATE FspCreateRsp;

	FSP_REQ_DELETE FspDeleteReq;
	FSP_RSP_DELETE FspDeleteRsp;

	FSP_REQ_READ FspReadReq;
	FSP_RSP_READ FspReadRsp;

	FSP_REQ_WRITE FspWriteReq;
	FSP_RSP_WRITE FspWriteRsp;

	FSP_REQ_RENAME FspRenameReq;
	FSP_RSP_RENAME FspRenameRsp;

	FSP_REQ_FINDFIRST FspFindFirstReq;
	FSP_RSP_FINDFIRST FspFindFirstRsp;

	FSP_REQ_FINDNEXT FspFindNextReq;
	FSP_RSP_FINDNEXT FspFindNextRsp;

	FSP_REQ_COMMIT FspCommitReq;
	FSP_RSP_COMMIT FspCommitRsp;

	FSP_REQ_FILEINFO FspFileInfoReq;
	FSP_RSP_FILEINFO FspFileInfoRsp;

	FSP_REQ_FILETIME FspFileTimeReq;
	FSP_RSP_FILETIME FspFileTimeRsp;

	FSP_REQ_FSINFO FspFsInfoReq;
	FSP_RSP_FSINFO FspFsInfoRsp;

	FSP_REQ_LOCK FspLockReq;
	FSP_RSP_LOCK FspLockRsp;

	FSP_REQ_UNLOCK FspUnlockReq;
	FSP_RSP_UNLOCK FspUnlockRsp;
    } Operands;
} FSP_FRAME, *PFSP_FRAME;

#endif // _FSP
