//
// MODULE NAME:  ETHERNET.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for 802.3.
//
//	There are two main entrypoints to this module.	PacketSummaryEthernet
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailEthernet returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/17.	Original.
//	S. E. Jones	93/03/10.	#2.4, added new etype from Steven Ward.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

typedef struct _ETHERTYPE {
    USHORT TypeCode;			// actual Ethertype.
    UCHAR *Description;                 // human-readable string.
} ETHERTYPE, *PETHERTYPE;

typedef struct _ETHERNET_HEADER {
    UCHAR DestMacAddr [6];		// destination MAC address.
    UCHAR SrcMacAddr [6];		// source MAC address.
    USHORT EtherType;			// byte-reversed ethertype.
    UCHAR Buffer [1];			// 1st byte of encapsulated protocol.
} ETHERNET_HEADER, *PETHERNET_HEADER;

ETHERTYPE far EtherTypeTable [] = {
    {0x0601, "XNS Address Translation"},
    {0x0800, "Internet Protocol (IP)"},
    {0x0801, "X.75 Internet"},
    {0x0802, "NBS Internet"},
    {0x0803, "ECMA Internet"},
    {0x0804, "MIT CHAOSnet (CHAOS)"},
    {0x0805, "X.25 Level 3"},
    {0x0806, "ARP Address Resolution Protocol"},
    {0x0807, "XNS Compatability"},
    {0x081C, "Symbolics Private"},
    {0x0888, "Xyplex"},
    {0x0889, "Xyplex"},
    {0x088A, "Xyplex"},
    {0x0900, "Ungermann-Bass Network Debugger"},
    {0x0A00, "Xerox 802.3 PUP"},
    {0x0A01, "PUP 802.3 Address Trans. (PUP-ARP)"},
    {0x0BAD, "Banyan Vines-IP, Banyan Systems"},
    {0x0BAF, "Banyan VINES Echo"},
    {0x1000, "Berkeley Trailer Negotiation"},
#if 0
    {0x1001, "Berkeley IP Encapsulation"},
    {0x1002, "Berkeley IP Encapsulation"},
    {0x1003, "Berkeley IP Encapsulation"},
    {0x1004, "Berkeley IP Encapsulation"},
    {0x1005, "Berkeley IP Encapsulation"},
    {0x1006, "Berkeley IP Encapsulation"},
    {0x1007, "Berkeley IP Encapsulation"},
    {0x1008, "Berkeley IP Encapsulation"},
    {0x1009, "Berkeley IP Encapsulation"},
    {0x100a, "Berkeley IP Encapsulation"},
    {0x100b, "Berkeley IP Encapsulation"},
    {0x100c, "Berkeley IP Encapsulation"},
    {0x100d, "Berkeley IP Encapsulation"},
    {0x100e, "Berkeley IP Encapsulation"},
    {0x100f, "Berkeley IP Encapsulation"},
#endif
    {0x1234, "DCA - Multicast"},
    {0x1600, "VALID system protocol"},
    {0x1989, "'Aviator' dogfight simulator"},
    {0x3C00, "3COM NBP virtual circuit datagram"},
    {0x3C01, "3COM NBP System control datagram"},
    {0x3C02, "3COM NBP Connect request"},
    {0x3C03, "3COM NBP Connect response"},
    {0x3C04, "3COM NBP Connect complete"},
    {0x3C05, "3COM NBP Close request"},
    {0x3C06, "3COM NBP Close response"},
    {0x3C07, "3COM NBP Datagram"},
    {0x3C08, "3COM NBP Datagram broadcast"},
    {0x3C09, "3COM NBP Claim NetBIOS name"},
    {0x3C0A, "3COM NBP Delete NetBIOS name"},
    {0x3C0B, "3COM NBP Remote adapter status req"},
    {0x3C0C, "3COM NBP Remote adapter rsp"},
    {0x3C0D, "3COM NBP Reset"},
    {0x4242, "PCS Basic Block Protocol"},
    {0x4321, "THD - Diddle"},
    {0x5208, "BBN Simnet Private"},
    {0x6000, "DEC unassigned"},
    {0x6001, "DEC MOP Dump/Load Assistance"},
    {0x6002, "DEC MOP Remote Console"},
    {0x6003, "DEC DECnet Phase IV"},
    {0x6004, "DEC LAT"},
    {0x6005, "DEC DECnet Diagnostics"},
    {0x6006, "DEC DECNet Customer Use"},
    {0x6007, "DEC DECNet LAVC, SCA"},
#if 0
    {0x6010, "3Com Proprietary"},
    {0x6011, "3Com Proprietary"},
    {0x6012, "3Com Proprietary"},
    {0x6013, "3Com Proprietary"},
    {0x6014, "3Com Proprietary"},
    {0x7000, "Ungermann-Bass download"},
    {0x7001, "Ungermann-Bass NIU"},
    {0x7002, "Ungermann-Bass diagnostic/loopback"},
    {0x7005, "Ungermann-Bass Bridge Spanning Tree"},
    {0x7007, "OS/9 Microware"},
    {0x7009, "OS/9 Net"},
    {0x7030, "Racal InterLAN"},
    {0x7034, "Cabletron"},
    {0x8003, "Cronus VLN"},
    {0x8004, "Cronus Direct"},
    {0x8005, "HP Probe protocol"},
    {0x8006, "Nestar"},
#endif
    {0x8008, "AT&T"},
    {0x8010, "Excelan (Novell)"},
    {0x8013, "SGI diagnostic type"},
    {0x8014, "SGI network games"},
    {0x8015, "SGI reserved type"},
    {0x8016, "SGI XNS NameServer, bounce server"},
    {0x8019, "Apollo DOMAIN"},
    {0x802E, "Tymshare"},
    {0x802F, "Tigan, Inc."},
    {0x8035, "Reverse Address Resolution Protocol (RARP)"},
    {0x8036, "Aeonic Systems"},
    {0x8037, "IPX (Novell)"},
    {0x8038, "DEC LANBridge"},
    {0x8039, "DEC DSM/DTP"},
    {0x803A, "DEC Argonaut Console"},
    {0x803B, "DEC VAXELN"},
    {0x803C, "DEC NMSV DNA Naming Service"},
    {0x803D, "DEC Ethernet Encryption Protocol"},
    {0x803E, "DEC DTS, DNA Time Service"},
    {0x803F, "DEC LAN Traffic Monitor"},
    {0x8040, "DEC NetBios Emulator"},
    {0x8041, "DEC Local Area System Transport"},
    {0x8042, "DEC unassigned"},
    {0x8044, "Planning Research Corp."},
    {0x8046, "AT&T"},
    {0x8047, "AT&T"},
    {0x8049, "ExperData (France)"},
    {0x805B, "VMTP RFC-1045"},
    {0x805B, "Stanford University"},
    {0x805C, "Stanford V Kernel V6.0"},
#if 0
    {0x805D, "Evans & Sutherland"},
    {0x8060, "Little Machines"},
    {0x8062, "Counterpoint Computers"},
    {0x8065, "Univ of Mass @ Amherst"},
    {0x8066, "Univ of Mass @ Amherst"},
    {0x8067, "Veeco Integrated Automation"},
    {0x8068, "General Dynamics"},
    {0x8069, "AT&T"},
    {0x806A, "Autophon (Switzerland)"},
    {0x806C, "ComDesign"},
    {0x806D, "Compugraphic Corporation"},
    {0x806E, "Landmark Graphics"},
    {0x806f, "Landmark Graphics"},
    {0x8070, "Landmark Graphics"},
    {0x8071, "Landmark Graphics"},
    {0x8072, "Landmark Graphics"},
    {0x8073, "Landmark Graphics"},
    {0x8074, "Landmark Graphics"},
    {0x8075, "Landmark Graphics"},
    {0x8076, "Landmark Graphics"},
    {0x8077, "Landmark Graphics"},
#endif
    {0x807A, "Matra (France)"},
    {0x807B, "Dansk Data Elektronic A/S (Denmark)"},
    {0x807C, "Merit Internodal, Univ of Michigan"},
#if 0
    {0x807D, "VitaLink"},
    {0x807e, "VitaLink"},
    {0x807f, "VitaLink"},
    {0x8080, "VitaLink"},
    {0x8081, "Counterpoint Computers"},
    {0x8082, "Counterpoint Computers"},
    {0x8083, "Counterpoint Computers"},
#endif
    {0x8088, "Xyplex"},
    {0x8089, "Xyplex"},
    {0x808a, "Xyplex"},
    {0x809B, "Kinetics Ethertalk"},
    {0x809C, "Datability"},
    {0x809d, "Datability"},
    {0x809e, "Datability"},
    {0x809F, "Spider Systems, Ltd."},
    {0x80A3, "Nixdorf Computer"},
    {0x80C0, "DCA"},
    {0x80C1, "DCA"},
    {0x80C2, "DCA"},
    {0x80C3, "DCA"},
    {0x80C4, "Banyan"},
    {0x80C5, "Banyan"},
    {0x80C6, "Pacer Software"},
    {0x80C7, "Applitek Corporation"},
#if 0
    {0x80C8, "Intergraph Corporation"},
    {0x80C9, "Intergraph Corporation"},
    {0x80Ca, "Intergraph Corporation"},
    {0x80Cb, "Intergraph Corporation"},
    {0x80Cc, "Intergraph Corporation"},
    {0x80CD, "Harris Corporation"},
    {0x80CE, "Harris Corporation"},
    {0x80CF, "Taylor Instrument"},
    {0x80d0, "Taylor Instrument"},
    {0x80d1, "Taylor Instrument"},
    {0x80d2, "Taylor Instrument"},
    {0x80D3, "Rosemount Corporation"},
    {0x80D4, "Rosemount Corporation"},
#endif
    {0x80D5, "IBM SNA Services over Ethernet"},
    {0x80DD, "Varian Assoc."},
    {0x80DE, "Integrated Solutions Transparent Remote FileSys"},
    {0x80DF, "Integrated Solutions"},
    {0x80E0, "Allen-Bradley"},
    {0x80E1, "Allen-Bradley"},
    {0x80E2, "Allen-Bradley"},
    {0x80E3, "Allen-Bradley"},
#if 0
    {0x80E4, "Datability"},
    {0x80E5, "Datability"},
    {0x80E6, "Datability"},
    {0x80E7, "Datability"},
    {0x80E8, "Datability"},
    {0x80E9, "Datability"},
    {0x80Ea, "Datability"},
    {0x80Eb, "Datability"},
    {0x80Ec, "Datability"},
    {0x80Ed, "Datability"},
    {0x80Ee, "Datability"},
    {0x80Ef, "Datability"},
    {0x80f0, "Datability"},
#endif
    {0x80F2, "Retix"},
    {0x80F3, "Kinetics, AppleTalk ARP (AARP)"},
    {0x80F4, "Kinetics"},
    {0x80F5, "Kinetics"},
    {0x80F7, "Apollo Computer (HP)"},
#if 0
    {0x80FF, "Wellfleet Communications"},
    {0x8100, "Wellfleet Communications"},
    {0x8101, "Wellfleet Communications"},
    {0x8102, "Wellfleet Communications"},
    {0x8103, "Wellfleet Communications"},
#endif
    {0x8107, "Symbolics, Inc."},
    {0x8108, "Symbolics, Inc."},
    {0x8109, "Symbolics, Inc."},
    {0x812B, "Talaris"},
    {0x8130, "Waterloo Microsystems"},
    {0x8131, "VG Laboratory Systems"},
    {0x8132, "3Com Bridge"},
    {0x8133, "3Com Bridge"},
    {0x8134, "3Com Bridge"},
    {0x8135, "3Com Bridge"},
    {0x8136, "3Com Bridge"},
    {0x8137, "Novell NetWare IPX (ECONFIG E option)"},
    {0x8138, "Novell, Inc."},
    {0x8139, "KTI"},
    {0x813a, "KTI"},
    {0x813b, "KTI"},
    {0x813c, "KTI"},
    {0x813d, "KTI"},
    {0x8148, "Logicraft"},
    {0x8149, "NCD"},
    {0x814C, "SNMP over Ethernet (RFC-1089)"},
    {0x814D, "BIIN"},
    {0x814E, "BIIN"},
    {0x8150, "Rational"},
    {0x8151, "Qualcom"},
    {0x8152, "Qualcom"},
    {0x8153, "Qualcom"},
    {0x815C, "CPPL"},
    {0x815D, "CPPL"},
    {0x815E, "CPPL"},
    {0x8164, "Charles River Data Systems"},
    {0x8165, "Charles River Data Systems"},
    {0x8166, "Charles River Data Systems"},
#if 0
    {0x817D, "Protocol Engines, Inc."},
    {0x817e, "Protocol Engines, Inc."},
    {0x817f, "Protocol Engines, Inc."},
    {0x8180, "Protocol Engines, Inc."},
    {0x8181, "Protocol Engines, Inc."},
    {0x8182, "Protocol Engines, Inc."},
    {0x8183, "Protocol Engines, Inc."},
    {0x8184, "Protocol Engines, Inc."},
    {0x8185, "Protocol Engines, Inc."},
    {0x8186, "Protocol Engines, Inc."},
    {0x8187, "Protocol Engines, Inc."},
    {0x8188, "Protocol Engines, Inc."},
    {0x8189, "Protocol Engines, Inc."},
    {0x818a, "Protocol Engines, Inc."},
    {0x818b, "Protocol Engines, Inc."},
    {0x818c, "Protocol Engines, Inc."},
#endif
    {0x81D6, "Lantastic"},
    {0x8888, "HP LanProbe test"},
    {0x9000, "Loopback (Configuration Test Protocol)"},
    {0x9001, "3COM XNS Systems Management"},
    {0x9002, "3COM TCP/IP Systems Management"},
    {0x9003, "3COM loopback detection"},
    {0xAAAA, "DEC VAX 6220 DEBNI"},
    {0xFF00, "BBN VITAL LANBridge cache wakeup"},
    {0, "EOT"}				// 0 signifies end of table.
}; // EtherTypeTable

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeEthernet (PETHERNET_HEADER Buffer, USHORT BufferLength)
{
    if (BufferLength < 14) {
	return FALSE;			// at least we need the header.
    }

    if (Swap (Buffer->EtherType) >= 0x600) {

	switch (Swap (Buffer->EtherType)) {
	    case 0x6001:		// DECNET family.
	    case 0x6002:
	    case 0x6003:
	    case 0x6004:
		break;
	    case 0x0600:		// IPX/SPP/SMB.
		return PacketReqTypeIpx (Buffer->Buffer, BufferLength-14);
	    case 0x0800:		// TCP/IP suite.
		return PacketReqTypeIp (Buffer->Buffer, BufferLength-14);
	    case 0x0806:		// internet ARP request.
	    case 0x8035:		// internet ARP response.
		return PacketReqTypeArp (Buffer->Buffer, BufferLength-14);
	}
	return FALSE;			// Ethertype not recognized.
    }

    //
    // We have either (1) a NetWare packet, or (2) an LLC packet.
    // Here, we try to decode the packet as a NetWare packet first, if
    // the filter for IPX is set.  If not, then we try to decode LLC,
    // if the LLC filter is set.
    //

    if (PacketReqTypeIpx (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    if (PacketReqTypeLlc (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    return FALSE;
} // PacketReqTypeEthernet

BOOLEAN PacketCountEthernet (PETHERNET_HEADER Buffer, USHORT BufferLength)
{
    if (BufferLength < 14) {
	return FALSE;			// at least we need the header.
    }

    if (Swap (Buffer->EtherType) >= 0x600) {

	switch (Swap (Buffer->EtherType)) {
	    case 0x0600:		// IPX/SPP/SMB.
		return PacketCountIpx (Buffer->Buffer, BufferLength-14);

	    case 0x0800:		// IP datagram.
	    case 0x0806:		// ARP request packet.
	    case 0x8035:		// ARP response packet.
		return PacketCountIp (Buffer->Buffer, BufferLength-14);

	    case 0x6001:		// DECNET family.
	    case 0x6002:
	    case 0x6003:
	    case 0x6004:
		CountProtocolType (INDEX_DECNET);
	}
	return FALSE;			// Ethertype not recognized.
    }

    //
    // We have either (1) a NetWare packet, or (2) an LLC packet.
    // Here, we try to decode the packet as a NetWare packet first, if
    // the filter for IPX is set.  If not, then we try to decode LLC,
    // if the LLC filter is set.
    //

    if (PacketCountIpx (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    if (PacketCountLlc (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    return FALSE;
} // PacketCountEthernet

BOOLEAN PacketFilterEthernet (PETHERNET_HEADER Buffer, USHORT BufferLength)
{
    if (BufferLength < 14) {
	return FALSE;			// at least we need the header.
    }

    if (Swap (Buffer->EtherType) >= 0x600) {

	switch (Swap (Buffer->EtherType)) {
	    case 0x0600:		// IPX/SPP/SMB.
		return PacketFilterIpx (Buffer->Buffer, BufferLength-14);

	    case 0x0800:		// TCP/IP suite.
		return PacketFilterIp (Buffer->Buffer, BufferLength-14);

	    case 0x0806:		// ARP request packets.
	    case 0x8035:		// ARP response packets.
		return PacketFilterArp (Buffer->Buffer, BufferLength-14);

	    case 0x6001:		// DECNET family.
	    case 0x6002:
	    case 0x6003:
	    case 0x6004:
		return PacketFilterDecnet (Buffer->Buffer, BufferLength-14);
	}
	return FALSE;			// Ethertype not recognized.
    }

    //
    // We have either (1) a NetWare packet, or (2) an LLC packet.
    // Here, we try to decode the packet as a NetWare packet first, if
    // the filter for IPX is set.  If not, then we try to decode LLC,
    // if the LLC filter is set.
    //

    if (PacketFilterIpx (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    if (PacketFilterLlc (Buffer->Buffer, BufferLength-14)) {
	return TRUE;
    }
    return FALSE;
} // PacketFilterEthernet

PUCHAR PacketSummaryEthernet (PETHERNET_HEADER Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;
    USHORT i, type;

    //
    // Build basic SA/DA Ethernet addresses.
    //

    p = malloc (128);
    if (p == NULL) {
	return NULL;
    }
    if (BufferLength < 14) {
	sprintf (p, "DLC Fragment, size %u bytes", BufferLength);
	return p;
    }

    ResolveDlcAddress (ScratchBuf, Buffer->DestMacAddr);
    ResolveDlcAddress (ScratchBuf1, Buffer->SrcMacAddr);
    sprintf (p, "%s  %s  ", ScratchBuf, ScratchBuf1);

    //
    // If this packet has an ethertype field value is above 0x600 (1536),
    // then it is a valid ethertype that we can switch on right here.
    //

    type = Swap (Buffer->EtherType);
    if (type >= 0x600) {

	switch (type) {
	    case 0x0600:			// IPX/SPP/SMB.
		if ((t=PacketSummaryIpx (Buffer->Buffer, BufferLength-14)) != NULL) {
		    strcat (p, t);
		    free (t);
		    return p;
		}
		break;

	    case 0x0800:			// internet IP.
		if ((t=PacketSummaryIp (Buffer->Buffer, BufferLength-14)) != NULL) {
		    strcat (p, t);
		    free (t);
		    return p;
		}
		break;

	    case 0x0806:			// internet ARP request.
	    case 0x8035:			// internet ARP response.
		if ((t=PacketSummaryArp (Buffer->Buffer, BufferLength-14)) != NULL) {
		    strcat (p, t);
		    free (t);
		    return p;
		}
		break;

	    case 0x6001:
	    case 0x6002:
	    case 0x6003:
	    case 0x6004:
		if ((t=PacketSummaryDecnet (Buffer->Buffer, BufferLength-14)) != NULL) {
		    strcat (p, t);
		    free (t);
		    return p;
		}
		break;

	    default:
		for (i=0; i<0xffff; i++) {
		    if (EtherTypeTable [i].TypeCode == 0) {
			break;		// end of table.
		    } else if (EtherTypeTable [i].TypeCode == type) {
			sprintf (ScratchBuf, "%s", EtherTypeTable [i].Description);
			strcat (p, ScratchBuf);
			return p;
		    }
		}
		sprintf (ScratchBuf, "DLC undecoded EtherType=0x%04x, size=%u bytes",
			type, BufferLength);
		strcat (p, ScratchBuf);
		return p;
	}
	sprintf (ScratchBuf, "DLC unexpected protocol on EtherType=0x%04x, size=%u bytes",
		 type, BufferLength);
	strcat (p, ScratchBuf);
	return p;
    }

    //
    // We have either (1) a NetWare packet, or (2) an LLC packet.
    // Here, we try to decode the packet as a NetWare packet first, if
    // the filter for IPX is set.  If not, then we try to decode LLC,
    // if the LLC filter is set.
    //

    if ((t=PacketSummaryIpx (Buffer->Buffer, BufferLength-14)) != NULL) {
	strcat (p, t);
	free (t);
	return p;
    }
    if ((t=PacketSummaryLlc (Buffer->Buffer, BufferLength-14)) != NULL) {
	strcat (p, t);
	free (t);
	return p;
    }
    sprintf (ScratchBuf, "DLC 802.2, size=%u bytes", BufferLength);
    strcat (p, ScratchBuf);
    return p;
} // PacketSummaryEthernet

VOID DlcTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "DLC:");
} // DlcTrailer

PDETREC PacketDetailEthernet (PETHERNET_HEADER Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    USHORT i, etype;

    if (BufferLength < 14) {
	return NULL;
    }

    //
    // Build basic SA/DA Ethernet addresses.
    //

    etype = Swap (Buffer->EtherType);
    if (etype >= 0x600) {
	AppendHeader (&r, "DLC:  ----- Data Link Control header (Bluebook Ethernet) -----");
    } else if (etype > 1518) {
	AppendHeader (&r, "DLC:  ----- Data Link Control header (Extra Extended 802.3) -----");
    } else if (etype > 1500) {
	AppendHeader (&r, "DLC:  ----- Data Link Control header (Extended 802.3) -----");
    } else {
	AppendHeader (&r, "DLC:  ----- Data Link Control header (802.3) -----");
    }

    sprintf (ScratchBuf, "DLC:  Frame #%u, size is %u (0x%04x hex) bytes.",
	     CurrentCapRec->SerialNo, BufferLength, BufferLength);
    AppendDetRec (&r, ScratchBuf);

    ResolveDlcAddress (ScratchBuf1, Buffer->DestMacAddr);
    sprintf (ScratchBuf, "DLC:  Destination: Station %s", ScratchBuf1);
    AppendDetRec (&r, ScratchBuf);

    ResolveDlcAddress (ScratchBuf1, Buffer->SrcMacAddr);
    sprintf (ScratchBuf, "DLC:  Source:      Station %s", ScratchBuf1);
    AppendDetRec (&r, ScratchBuf);

    if (etype >= 0x600) {
	sprintf (ScratchBuf, "DLC:  Ethertype = 0x%04x", etype);
	AppendDetRec (&r, ScratchBuf);
    }
    DlcTrailer (&r);

    if (etype >= 0x600) {

	switch (etype) {
	    case 0x0600:			// XNS.
		if ((q=PacketDetailIpx (Buffer->Buffer, BufferLength-14)) != NULL) {
		    JoinDetRec (&r, q);
		    return r;
		}
		break;

	    case 0x0800:			// internet IP.
		if ((q=PacketDetailIp (Buffer->Buffer, BufferLength-14)) != NULL) {
		    JoinDetRec (&r, q);
		    return r;
		}
		break;

	    case 0x0806:			// internet ARP request.
	    case 0x8035:			// internet ARP response.
		if ((q=PacketDetailArp (Buffer->Buffer, BufferLength-14)) != NULL) {
		    JoinDetRec (&r, q);
		    return r;
		}
		break;

	    case 0x6001:
	    case 0x6002:
	    case 0x6003:
	    case 0x6004:
		if ((q=PacketDetailDecnet (Buffer->Buffer, BufferLength-14)) != NULL) {
		    JoinDetRec (&r, q);
		    return r;
		}
		AppendDetRec (&r, "DLC:  Undecoded encapsulated protocol");
		return r;

	    default:			// undecoded ethertype.
		for (i=0; i<0xffff; i++) {
		    if (EtherTypeTable [i].TypeCode == 0) {
			break;		// end of table.
		    } else if (EtherTypeTable [i].TypeCode == etype) {
			sprintf (ScratchBuf, "DLC:  %s", EtherTypeTable [i].Description);
			AppendDetRec (&r, ScratchBuf);
			return r;
		    }
		}
		AppendDetRec (&r, "DLC:  Undecoded encapsulated protocol");
		return r;
	}
    }

    //
    // We have an LLC or IPX packet here.  Try to decode at the higher layer.
    //

    if ((q=PacketDetailIpx (Buffer->Buffer, BufferLength-14)) != NULL) {
	JoinDetRec (&r, q);
	return r;
    }
    if (etype <= 0x0600) {
	if ((q=PacketDetailLlc (Buffer->Buffer, BufferLength-14)) != NULL) {
	    JoinDetRec (&r, q);
	    return r;
	}
    }
    return r;
} // PacketDetailEthernet
