//
// MODULE NAME:  DISPLAY.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles the actual display mode for the analyzers.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/04/04.	Original.
//	S. E. Jones	93/02/07.	Moved extra line from summary to detail.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// Stuff used during display.
//

static UCHAR JunkBuffer [256];		// used for formatting.
static PCAPREC CaptureStream;		// ptr to 1st packet in stream.

#define MAX_WINDOWS	3
static DISPLAY_WINDOW Window [MAX_WINDOWS];
static PDISPLAY_WINDOW HexWin;
static PDISPLAY_WINDOW ProtWin;
static PDISPLAY_WINDOW SumWin;
static BOOLEAN refresh;

#define SCROLL_JUMP 2			// # characters per horiz. scroll.

static USHORT SummaryLeftEdge=0;	// horiz. scrolling of summary window.
static USHORT DetailLeftEdge=0;         // horiz. scrolling of detail window.
static USHORT HexLeftEdge=0;		// horiz. scrolling of hex window.

//
// Routines in other modules.
//

//
// Routines in this module.
//

VOID BuildHexData (Win)
    PDISPLAY_WINDOW Win;
{
    USHORT i, curofs, maxofs;
    UCHAR *t, TinyBuf [2];
    PDETREC p, q;

    if ((Win == NULL) || (Win->Win == NULL)) {
	return;
    }
    DeallocateDetRecChain (Win);	// clean-up old text.

    maxofs = CaptureStream->Length;	// maxofs = size of packet in bytes.
    for (curofs=0; curofs<maxofs; curofs+=16) {
	sprintf (JunkBuffer, "%04x  ", curofs);
	t = (CaptureStream->Buffer+curofs);
	sprintf (JunkBuffer+strlen (JunkBuffer),
		 "%02x %02x %02x %02x %02x %02x %02x %02x  ",
		 *(t+0), *(t+1), *(t+2), *(t+3), *(t+4), *(t+5), *(t+6), *(t+7));
	sprintf (JunkBuffer+strlen (JunkBuffer),
		 "%02x %02x %02x %02x %02x %02x %02x %02x | ",
		 *(t+8),*(t+9),*(t+10),*(t+11),*(t+12),*(t+13),*(t+14),*(t+15));
	for (i=0; i<16; i++) {
	    TinyBuf [0] = *(t+i);
	    TinyBuf [1] = 0;
	    if (TinyBuf [0] < 32) {     // special character.
		TinyBuf [0] = '.';
	    }
	    strcat (JunkBuffer, TinyBuf);
	}
	p = AllocateDetRec (JunkBuffer);
	if (p == NULL) {
	    break;
	}
	if (Win->Head == NULL) {
	    Win->Head = p;
	    Win->Top = p;
	} else {
	    q = Win->Head;
	    while (q->Fwdlink != NULL) {
		q = q->Fwdlink;
	    }
	    q->Fwdlink = p;
	}
    }
} // BuildHexData

VOID BuildProtocolData (Win)
    PDISPLAY_WINDOW Win;
{
    if ((Win == NULL) || (Win->Win == NULL)) {
	return;
    }
    DeallocateDetRecChain (Win);	// clean-up old text.

    LastHeaderDetRec = NULL;
    CurrentCapRec = CaptureStream;
    Win->Head = PacketDetailEthernet (CaptureStream->Buffer,
				      CaptureStream->Length);

    Win->Top = Win->Head;
    if (HighestLevelOnly) {		// do "highest level only" decoding.
	if (LastHeaderDetRec != NULL) {
	    Win->Top = LastHeaderDetRec;
	}
    }
} // BuildProtocolData

VOID RefreshWindow (Win)
    PDISPLAY_WINDOW Win;
{
    USHORT i, len, ofs;
    ULONG lasttime=0L, deltasec, mins, hours, secs, timestamp, deltams;
    UCHAR *s;
    PCAPREC p;
    PDETREC d;

    if ((Win == NULL) || (Win->Win == NULL)) {
	return;
    }

    if (Win->Type == DISPLAY_TYPE_SUMMARY) {

	//
	// Backup from current record to first line in the display.
	//

	p = CaptureStream;
	for (i=Win->CurrLine; i>0; i--) {
	    if (p->Baklink != NULL) {
		p = p->Baklink;
	    }
	}

	//
	// Now redraw this window from there.
	//

	if (p != NULL) {
	    lasttime = p->Timestamp;
	    if (p->Baklink != NULL) {
		lasttime = p->Baklink->Timestamp;
	    }
	}

	for (i=0; i<Win->WindowLines; i++) {
	    if ((p == NULL) || (!(p->Flags & CAPREC_FLAGS_CAPTURED))) {
		WinWrite (Win->Win, " ", 0, 1+i, 999);
	    } else {
		if (p->Timestamp < lasttime) {
		    deltasec = 0L;	// handle altered 8254 time.
		    deltams = 0L;
		} else {
		    deltasec = (p->Timestamp-lasttime) / 10000L;
		    deltams = (p->Timestamp-lasttime) % 10000L;
		}
		lasttime = p->Timestamp;

		if (DisplayAbsoluteTimes) {
		    secs = p->Timestamp / 10000L;
		    hours = secs / 3600L;
		    secs %= 3600L;
		    mins = secs / 60L;
		    secs %= 60L;
		    deltams = p->Timestamp % 10000L;
		    sprintf (JunkBuffer, " %5u %2lu:%02lu:%02lu.%04lu  ",
			     p->SerialNo, hours, mins, secs, deltams);
		} else {
		    sprintf (JunkBuffer, "%5u %4lu.%04lu  ",
			     p->SerialNo, deltasec, deltams);
		}
		CurrentCapRec = p;	// for protocol interpreters.

		s = PacketSummaryEthernet (p->Buffer, p->Length);
		if (s != NULL) {
		    strcat (JunkBuffer, s);
		    free (s);
		}

		len = strlen (JunkBuffer);
		if (len >= SummaryLeftEdge) {
		    WinWrite (Win->Win, JunkBuffer+SummaryLeftEdge, 0, 1+i, 999);
		} else {
		    WinWrite (Win->Win, " ", 0, 1+i, 999);
		}
		p = p->Fwdlink;
	    }
	}
	WinHilite (Win->Win, 0, Win->CurrLine+1, 80);  // hilite the current line.

    } else {

	d = Win->Top;
	for (i=0; i<Win->WindowLines; i++) {
	    if (d == NULL) {
		WinWrite (Win->Win, " ", 1, i, 999);
	    } else {

		len = strlen (d->DescBuf);

		if (Win->Type == DISPLAY_TYPE_DETAIL) {
		    ofs = DetailLeftEdge;
		} else {		// it's the hex screen.
		    ofs = HexLeftEdge;
		}

		if (len >= ofs) {
		    WinWrite (Win->Win, d->DescBuf+ofs, 1, i, 999);
		} else {
		    WinWrite (Win->Win, " ", 1, i, 999);
		}

		d = d->Fwdlink;
	    }
	}
    }
} // RefreshWindow

VOID UpSummaryLine ()
{
    if (CaptureStream->Baklink != NULL) {
	CaptureStream = CaptureStream->Baklink;
	BuildHexData (HexWin);
	BuildProtocolData (ProtWin);
	if (SumWin->CurrLine > 0) {
	    WinUnHilite (SumWin->Win, 0, SumWin->CurrLine+1, 80);
	    SumWin->CurrLine--;
	    WinHilite (SumWin->Win, 0, SumWin->CurrLine+1, 80);
	    RefreshWindow (HexWin);
	    RefreshWindow (ProtWin);
	} else {
	    refresh = TRUE;
	}
    }
} // UpSummaryLine

VOID DownSummaryLine ()
{
    if (CaptureStream->Fwdlink != NULL) {
	if (!(CaptureStream->Fwdlink->Flags & CAPREC_FLAGS_CAPTURED)) {
	    return;
	}
	CaptureStream = CaptureStream->Fwdlink;
	BuildHexData (HexWin);
	BuildProtocolData (ProtWin);
	if (SumWin->CurrLine < SumWin->WindowLines-1) {
	    WinUnHilite (SumWin->Win, 0, SumWin->CurrLine+1, 80);
	    SumWin->CurrLine++;
	    WinHilite (SumWin->Win, 0, SumWin->CurrLine+1, 80);
	    RefreshWindow (HexWin);
	    RefreshWindow (ProtWin);
	} else {
	    refresh = TRUE;
	}
    }
} // DownSummaryLine

VOID DisplayData ()
{
    USHORT i, wno, nwindows, currwin=0, height, nextcenter, nlines, extra;
    BOOLEAN done=FALSE, UpdateSummaryTitle = TRUE;
    KEY_CODE k;
    PDETREC d;
    UCHAR *p=" Seq    Arrival Time  Destination   Source        Frame";
    UCHAR *q=" Seq     Delta   Destination   Source        Frame";

    if (CaptureBuffer == NULL) {
	PopupWarning ("Capture buffer is empty.");
	return;
    } else {
	CaptureStream = CaptureBuffer;	// start at beginning of buffer.
    }

    //
    // Here is the top-level display stuff.  This is user kbd-driven.
    //

    nwindows = 0;			// count # windows to display.
    Window [0].Win = NULL;
    Window [1].Win = NULL;
    Window [2].Win = NULL;
    refresh = FALSE;
    SummaryLeftEdge = 0;
    DetailLeftEdge = 0;
    HexLeftEdge = 0;
    ProtWin = NULL;
    HexWin = NULL;
    SumWin = NULL;

    if (DisplayHexData) nwindows++;
    if (DisplayProtocolDetail) nwindows++;
    if (DisplaySummaryLevel) nwindows++;
    if (nwindows == 0) {		// at least display summary data.
	DisplaySummaryLevel = TRUE;
	nwindows = 1;
    }
    height = 25 / nwindows;
    nextcenter = height / 2;
    nlines = height - 2;		// SIMPLE windows.
    wno = 0;
    extra = (nwindows > 1) ? 1 : 0;	// use extra display line in 1st win.

    if (DisplaySummaryLevel) {
	Window [wno].Win = WinCreate ("Capture Summary",
				       40,
				       nextcenter,
				       80,
				       height,
				       ((wno == 0) ? WINDOW_BORDER_SIMPLE_D :
						    WINDOW_BORDER_SIMPLE_S) |
				       WINDOW_BORDER_NOSHADOW,
				       NORMAL_PALETTE);
	WinUnHiliteBorder (Window [wno].Win);
	if (DisplayAbsoluteTimes) {
	    WinWrite (Window [wno].Win, p, 0, 0, 999);
	} else {
	    WinWrite (Window [wno].Win, q, 0, 0, 999);
	}
	Window [wno].WindowLines = nlines-1;
	Window [wno].Type = DISPLAY_TYPE_SUMMARY;
	Window [wno].Head = NULL;
	Window [wno].Top = NULL;
	Window [wno].CurrLine = 0;
	SumWin = &Window [wno];
	RefreshWindow (&Window [wno]);
	wno++;
	nextcenter += (height);
    }
    if (DisplayProtocolDetail) {
	Window [wno].Win = WinCreate ("Protocol Detail",
				      40,
				      nextcenter,
				      80,
				      height+extra,
				      ((wno == 0) ? WINDOW_BORDER_SIMPLE_D :
						   WINDOW_BORDER_SIMPLE_S) |
				      WINDOW_BORDER_NOSHADOW,
				      NORMAL_PALETTE);
	WinUnHiliteBorder (Window [wno].Win);
	Window [wno].WindowLines = nlines+extra;
	Window [wno].Type = DISPLAY_TYPE_DETAIL;
	Window [wno].Head = NULL;
	Window [wno].Top = NULL;
	Window [wno].CurrLine = 0;
	ProtWin = &Window [wno];
	BuildProtocolData (ProtWin);
	RefreshWindow (&Window [wno]);
	wno++;
	nextcenter += (height+extra);
	extra = 0;
    }
    if (DisplayHexData) {
	Window [wno].Win = WinCreate ("Hexadecimal Data",
				      40,
				      nextcenter,
				      80,
				      height+extra,
				      ((wno == 0) ? WINDOW_BORDER_SIMPLE_D :
						   WINDOW_BORDER_SIMPLE_S) |
				      WINDOW_BORDER_NOSHADOW,
				      NORMAL_PALETTE);
	WinUnHiliteBorder (Window [wno].Win);
	Window [wno].WindowLines = nlines+extra;
	Window [wno].Type = DISPLAY_TYPE_HEX;
	Window [wno].Head = NULL;
	Window [wno].Top = NULL;
	Window [wno].CurrLine = 0;
	HexWin = &Window [wno];
	BuildHexData (HexWin);
	RefreshWindow (&Window [wno]);
	wno++;
	nextcenter += height+extra;
    }

    while (!done) {
	if (UpdateSummaryTitle) {
	    UpdateSummaryTitle = FALSE;
	    if (DisplayAbsoluteTimes) {
		WinWrite (SumWin->Win, p, 0, 0, 999);
	    } else {
		WinWrite (SumWin->Win, q, 0, 0, 999);
	    }
	}

	WinHiliteBorder (Window [currwin].Win);
	k = KeyRead ();
	if ((k == I_TAB) && (nwindows > 1)) {
	    WinChangeBorderType (Window [currwin].Win, WINDOW_BORDER_SIMPLE_S);
	    WinUnHiliteBorder (Window [currwin].Win);
	    while (TRUE) {
		if (++currwin == MAX_WINDOWS) {
		    currwin = 0;
		}
		if (Window [currwin].Win != NULL) {
		    break;
		}
	    }
	    WinChangeBorderType (Window [currwin].Win, WINDOW_BORDER_SIMPLE_D);
	    WinHiliteBorder (Window [currwin].Win);
	    continue;			// don't process this key.
	}
	if ((k == I_BACKTAB) && (nwindows > 1)) {
	    WinChangeBorderType (Window [currwin].Win, WINDOW_BORDER_SIMPLE_S);
	    WinUnHiliteBorder (Window [currwin].Win);
	    while (TRUE) {
		if (currwin == 0) {
		    currwin = MAX_WINDOWS-1;
		} else {
		    currwin--;
		}
		if (Window [currwin].Win != NULL) {
		    break;
		}
	    }
	    WinChangeBorderType (Window [currwin].Win, WINDOW_BORDER_SIMPLE_D);
	    WinHiliteBorder (Window [currwin].Win);
	    continue;			// don't process this key.
	}
	if (Window [currwin].Type == DISPLAY_TYPE_SUMMARY) {
	    switch (k) {
		case I_ESC:
		case I_F10:
		    done = TRUE;
		    break;
		case I_LEFT:		// scroll left in 2-char increments.
		    if (SummaryLeftEdge >= SCROLL_JUMP) {
			SummaryLeftEdge -= SCROLL_JUMP;
			p -= SCROLL_JUMP;	// update title in window.
			q -= SCROLL_JUMP;	// update title in window.
			UpdateSummaryTitle = TRUE;
			refresh = TRUE;
		    }
		    break;
		case I_RIGHT:		// scroll right in SCROLL_JUMP-char increments.
		    if (SummaryLeftEdge + SCROLL_JUMP < 40) {
			SummaryLeftEdge += SCROLL_JUMP;
			p += SCROLL_JUMP;	// update title in window.
			q += SCROLL_JUMP;	// update title in window.
			UpdateSummaryTitle = TRUE;
			refresh = TRUE;
		    }
		    break;
		case I_UP:
		case I_F7:
		    UpSummaryLine ();
		    break;
		case I_DOWN:
		case I_ENTER:
		case I_F8:
		    DownSummaryLine ();
		    break;
		case I_HOME:
		    CaptureStream = CaptureBuffer;
		    BuildHexData (HexWin);
		    BuildProtocolData (ProtWin);
		    Window [currwin].CurrLine = 0;
		    refresh = TRUE;
		    break;
		case I_END:
		    while ((CaptureStream->Fwdlink != NULL) &&
			   (CaptureStream->Fwdlink->Flags & CAPREC_FLAGS_CAPTURED)) {
			CaptureStream = CaptureStream->Fwdlink;
		    }
		    BuildHexData (HexWin);
		    BuildProtocolData (ProtWin);
		    Window [currwin].CurrLine = 0;
		    refresh = TRUE;
		    break;
		case I_PGUP:
		    for (i=0; i<nlines-1; i++) {
			if (CaptureStream->Baklink != NULL) {
			    CaptureStream = CaptureStream->Baklink;
			    if (Window [currwin].CurrLine > 0) {
				Window [currwin].CurrLine--;
			    }
			}
		    }
		    BuildHexData (HexWin);
		    BuildProtocolData (ProtWin);
		    refresh = TRUE;
		    break;
		case I_PGDN:
		    for (i=0; i<Window [currwin].WindowLines; i++) {
			if (CaptureStream->Fwdlink == NULL) {
			    break;
			}
			if (!(CaptureStream->Fwdlink->Flags & CAPREC_FLAGS_CAPTURED)) {
			    break;
			}
			CaptureStream = CaptureStream->Fwdlink;
			if (Window [currwin].CurrLine < Window [currwin].WindowLines-1) {
			    Window [currwin].CurrLine++;
			}
		    }
		    BuildHexData (HexWin);
		    BuildProtocolData (ProtWin);
		    refresh = TRUE;
		    break;
	    }
	} else {			// this is a detail or hex display.
	    switch (k) {
		case I_ESC:
		case I_F10:
		    done = TRUE;
		    break;
		case I_LEFT:		// scroll left in SCROLL_JUMP-char increments.
		    if (Window [currwin].Type == DISPLAY_TYPE_HEX) {
			if (HexLeftEdge >= SCROLL_JUMP) {
			    HexLeftEdge -= SCROLL_JUMP;
			}
		    } else {
			if (DetailLeftEdge >= SCROLL_JUMP) {
			    DetailLeftEdge -= SCROLL_JUMP;
			}
		    }
		    RefreshWindow (&Window [currwin]);
		    break;
		case I_RIGHT:		// scroll right in SCROLL_JUMP-char increments.
		    if (Window [currwin].Type == DISPLAY_TYPE_HEX) {
			HexLeftEdge += SCROLL_JUMP;
		    } else {
			DetailLeftEdge += SCROLL_JUMP;
		    }
		    RefreshWindow (&Window [currwin]);
		    break;
		case I_F7:
		    UpSummaryLine ();
		    break;
		case I_F8:
		    DownSummaryLine ();
		    break;
		case I_UP:
		    d = Window [currwin].Head;
		    if (d == NULL) {
			break;
		    }
		    if (d == Window [currwin].Top) {
			break;			// we're already at the top.
		    }
		    while ((d->Fwdlink != NULL) &&
			   (d->Fwdlink != Window [currwin].Top)) {
			d = d->Fwdlink;
		    }
		    Window [currwin].Top = d;
		    RefreshWindow (&Window [currwin]);
		    break;
		case I_DOWN:
		case I_ENTER:
		    d = Window [currwin].Top;
		    if (d == NULL) {
			break;
		    }
		    if (d->Fwdlink != NULL) {
			Window [currwin].Top = d->Fwdlink;
			RefreshWindow (&Window [currwin]);
		    }
		    break;
		case I_HOME:
		    Window [currwin].Top = Window [currwin].Head;
		    RefreshWindow (&Window [currwin]);
		    break;
		case I_END:
		    d = Window [currwin].Top;
		    if (d == NULL) {
			break;
		    }
		    while (d->Fwdlink != NULL) {
			d = d->Fwdlink;
		    }
		    Window [currwin].Top = d;
		    RefreshWindow (&Window [currwin]);
		    break;
		case I_PGUP:			// BUGBUG: difficult.
		    break;
		case I_PGDN:
		    d = Window [currwin].Top;
		    if (d == NULL) {
			break;
		    }
		    for (i=0; i<nlines; i++) {
			if (d->Fwdlink != NULL) {
			    d = d->Fwdlink;
			}
		    }
		    Window [currwin].Top = d;
		    RefreshWindow (&Window [currwin]);
		    break;
	    }
	}
	if (refresh) {
	    refresh = FALSE;
	    for (i=0; i<MAX_WINDOWS; i++) {
		RefreshWindow (&Window [i]);
	    }
	}
    }

    for (i=0; i<MAX_WINDOWS; i++) {
	if (Window [i].Win != NULL) {
	    DeallocateDetRecChain (&Window [i]); // clean-up old text.
	    WinDestroy (Window [i].Win);	 // clean-up window on screen.
	}
    }
} // DisplayData
